var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
define(["require", "exports", '../observable/ArrayObservable', '../util/isArray', '../OuterSubscriber', '../util/subscribeToResult'], function (require, exports, ArrayObservable_1, isArray_1, OuterSubscriber_1, subscribeToResult_1) {
    "use strict";
    var none = {};
    /**
     * Combines multiple Observables to create an Observable whose values are
     * calculated from the latest values of each of its input Observables.
     *
     * <span class="informal">Whenever any input Observable emits a value, it
     * computes a formula using the latest values from all the inputs, then emits
     * the output of that formula.</span>
     *
     * <img src="./img/combineLatest.png" width="100%">
     *
     * `combineLatest` combines the values from this Observable with values from
     * Observables passed as arguments. This is done by subscribing to each
     * Observable, in order, and collecting an array of each of the most recent
     * values any time any of the input Observables emits, then either taking that
     * array and passing it as arguments to an optional `project` function and
     * emitting the return value of that, or just emitting the array of recent
     * values directly if there is no `project` function.
     *
     * @example <caption>Dynamically calculate the Body-Mass Index from an Observable of weight and one for height</caption>
     * var weight = Rx.Observable.of(70, 72, 76, 79, 75);
     * var height = Rx.Observable.of(1.76, 1.77, 1.78);
     * var bmi = weight.combineLatest(height, (w, h) => w / (h * h));
     * bmi.subscribe(x => console.log('BMI is ' + x));
     *
     * @see {@link combineAll}
     * @see {@link merge}
     * @see {@link withLatestFrom}
     *
     * @param {Observable} other An input Observable to combine with the source
     * Observable. More than one input Observables may be given as argument.
     * @param {function} [project] An optional function to project the values from
     * the combined latest values into a new value on the output Observable.
     * @return {Observable} An Observable of projected values from the most recent
     * values from each input Observable, or an array of the most recent values from
     * each input Observable.
     * @method combineLatest
     * @owner Observable
     */
    function combineLatest() {
        var observables = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            observables[_i - 0] = arguments[_i];
        }
        var project = null;
        if (typeof observables[observables.length - 1] === 'function') {
            project = observables.pop();
        }
        // if the first and only other argument besides the resultSelector is an array
        // assume it's been called with `combineLatest([obs1, obs2, obs3], project)`
        if (observables.length === 1 && isArray_1.isArray(observables[0])) {
            observables = observables[0];
        }
        observables.unshift(this);
        return new ArrayObservable_1.ArrayObservable(observables).lift(new CombineLatestOperator(project));
    }
    exports.combineLatest = combineLatest;
    /* tslint:enable:max-line-length */
    var CombineLatestOperator = (function () {
        function CombineLatestOperator(project) {
            this.project = project;
        }
        CombineLatestOperator.prototype.call = function (subscriber, source) {
            return source._subscribe(new CombineLatestSubscriber(subscriber, this.project));
        };
        return CombineLatestOperator;
    }());
    exports.CombineLatestOperator = CombineLatestOperator;
    /**
     * We need this JSDoc comment for affecting ESDoc.
     * @ignore
     * @extends {Ignored}
     */
    var CombineLatestSubscriber = (function (_super) {
        __extends(CombineLatestSubscriber, _super);
        function CombineLatestSubscriber(destination, project) {
            _super.call(this, destination);
            this.project = project;
            this.active = 0;
            this.values = [];
            this.observables = [];
        }
        CombineLatestSubscriber.prototype._next = function (observable) {
            this.values.push(none);
            this.observables.push(observable);
        };
        CombineLatestSubscriber.prototype._complete = function () {
            var observables = this.observables;
            var len = observables.length;
            if (len === 0) {
                this.destination.complete();
            }
            else {
                this.active = len;
                this.toRespond = len;
                for (var i = 0; i < len; i++) {
                    var observable = observables[i];
                    this.add(subscribeToResult_1.subscribeToResult(this, observable, observable, i));
                }
            }
        };
        CombineLatestSubscriber.prototype.notifyComplete = function (unused) {
            if ((this.active -= 1) === 0) {
                this.destination.complete();
            }
        };
        CombineLatestSubscriber.prototype.notifyNext = function (outerValue, innerValue, outerIndex, innerIndex, innerSub) {
            var values = this.values;
            var oldVal = values[outerIndex];
            var toRespond = !this.toRespond
                ? 0
                : oldVal === none ? --this.toRespond : this.toRespond;
            values[outerIndex] = innerValue;
            if (toRespond === 0) {
                if (this.project) {
                    this._tryProject(values);
                }
                else {
                    this.destination.next(values);
                }
            }
        };
        CombineLatestSubscriber.prototype._tryProject = function (values) {
            var result;
            try {
                result = this.project.apply(this, values);
            }
            catch (err) {
                this.destination.error(err);
                return;
            }
            this.destination.next(result);
        };
        return CombineLatestSubscriber;
    }(OuterSubscriber_1.OuterSubscriber));
    exports.CombineLatestSubscriber = CombineLatestSubscriber;
});
//# sourceMappingURL=combineLatest.js.map