define(["require", "exports", './find'], function (require, exports, find_1) {
    "use strict";
    /**
     * Emits only the index of the first value emitted by the source Observable that
     * meets some condition.
     *
     * <span class="informal">It's like {@link find}, but emits the index of the
     * found value, not the value itself.</span>
     *
     * <img src="./img/findIndex.png" width="100%">
     *
     * `findIndex` searches for the first item in the source Observable that matches
     * the specified condition embodied by the `predicate`, and returns the
     * (zero-based) index of the first occurrence in the source. Unlike
     * {@link first}, the `predicate` is required in `findIndex`, and does not emit
     * an error if a valid value is not found.
     *
     * @example <caption>Emit the index of first click that happens on a DIV element</caption>
     * var clicks = Rx.Observable.fromEvent(document, 'click');
     * var result = clicks.findIndex(ev => ev.target.tagName === 'DIV');
     * result.subscribe(x => console.log(x));
     *
     * @see {@link filter}
     * @see {@link find}
     * @see {@link first}
     * @see {@link take}
     *
     * @param {function(value: T, index: number, source: Observable<T>): boolean} predicate
     * A function called with each item to test for condition matching.
     * @param {any} [thisArg] An optional argument to determine the value of `this`
     * in the `predicate` function.
     * @return {Observable} An Observable of the index of the first item that
     * matches the condition.
     * @method find
     * @owner Observable
     */
    function findIndex(predicate, thisArg) {
        return this.lift(new find_1.FindValueOperator(predicate, this, true, thisArg));
    }
    exports.findIndex = findIndex;
});
//# sourceMappingURL=findIndex.js.map