var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
define(["require", "exports", '../OuterSubscriber', '../util/subscribeToResult'], function (require, exports, OuterSubscriber_1, subscribeToResult_1) {
    "use strict";
    /**
     * Emits a value from the source Observable, then ignores subsequent source
     * values for a duration determined by another Observable, then repeats this
     * process.
     *
     * <span class="informal">It's like {@link throttleTime}, but the silencing
     * duration is determined by a second Observable.</span>
     *
     * <img src="./img/throttle.png" width="100%">
     *
     * `throttle` emits the source Observable values on the output Observable
     * when its internal timer is disabled, and ignores source values when the timer
     * is enabled. Initially, the timer is disabled. As soon as the first source
     * value arrives, it is forwarded to the output Observable, and then the timer
     * is enabled by calling the `durationSelector` function with the source value,
     * which returns the "duration" Observable. When the duration Observable emits a
     * value or completes, the timer is disabled, and this process repeats for the
     * next source value.
     *
     * @example <caption>Emit clicks at a rate of at most one click per second</caption>
     * var clicks = Rx.Observable.fromEvent(document, 'click');
     * var result = clicks.throttle(ev => Rx.Observable.interval(1000));
     * result.subscribe(x => console.log(x));
     *
     * @see {@link audit}
     * @see {@link debounce}
     * @see {@link delayWhen}
     * @see {@link sample}
     * @see {@link throttleTime}
     *
     * @param {function(value: T): Observable|Promise} durationSelector A function
     * that receives a value from the source Observable, for computing the silencing
     * duration for each source value, returned as an Observable or a Promise.
     * @return {Observable<T>} An Observable that performs the throttle operation to
     * limit the rate of emissions from the source.
     * @method throttle
     * @owner Observable
     */
    function throttle(durationSelector) {
        return this.lift(new ThrottleOperator(durationSelector));
    }
    exports.throttle = throttle;
    var ThrottleOperator = (function () {
        function ThrottleOperator(durationSelector) {
            this.durationSelector = durationSelector;
        }
        ThrottleOperator.prototype.call = function (subscriber, source) {
            return source._subscribe(new ThrottleSubscriber(subscriber, this.durationSelector));
        };
        return ThrottleOperator;
    }());
    /**
     * We need this JSDoc comment for affecting ESDoc.
     * @ignore
     * @extends {Ignored}
     */
    var ThrottleSubscriber = (function (_super) {
        __extends(ThrottleSubscriber, _super);
        function ThrottleSubscriber(destination, durationSelector) {
            _super.call(this, destination);
            this.destination = destination;
            this.durationSelector = durationSelector;
        }
        ThrottleSubscriber.prototype._next = function (value) {
            if (!this.throttled) {
                this.tryDurationSelector(value);
            }
        };
        ThrottleSubscriber.prototype.tryDurationSelector = function (value) {
            var duration = null;
            try {
                duration = this.durationSelector(value);
            }
            catch (err) {
                this.destination.error(err);
                return;
            }
            this.emitAndThrottle(value, duration);
        };
        ThrottleSubscriber.prototype.emitAndThrottle = function (value, duration) {
            this.add(this.throttled = subscribeToResult_1.subscribeToResult(this, duration));
            this.destination.next(value);
        };
        ThrottleSubscriber.prototype._unsubscribe = function () {
            var throttled = this.throttled;
            if (throttled) {
                this.remove(throttled);
                this.throttled = null;
                throttled.unsubscribe();
            }
        };
        ThrottleSubscriber.prototype.notifyNext = function (outerValue, innerValue, outerIndex, innerIndex, innerSub) {
            this._unsubscribe();
        };
        ThrottleSubscriber.prototype.notifyComplete = function () {
            this._unsubscribe();
        };
        return ThrottleSubscriber;
    }(OuterSubscriber_1.OuterSubscriber));
});
//# sourceMappingURL=throttle.js.map