"use strict";
var chai_1 = require('chai');
var Rx = require('../../dist/cjs/Rx');
var iterator_1 = require('../../dist/cjs/symbol/iterator');
var Observable = Rx.Observable;
/** @test {from} */
describe('Observable.from', function () {
    asDiagram('from([10, 20, 30])')('should create an observable from an array', function () {
        var e1 = Observable.from([10, 20, 30])
            .concatMap(function (x, i) { return Observable.of(x).delay(i === 0 ? 0 : 20, rxTestScheduler); });
        var expected = 'x-y-(z|)';
        expectObservable(e1).toBe(expected, { x: 10, y: 20, z: 30 });
    });
    it('should enumerate an Array', function (done) {
        this.timeout(300);
        var expected = [1, 2, 3];
        var i = 0;
        Observable.from(expected).subscribe(function (x) {
            chai_1.expect(x).to.equal(expected[i++]);
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should handle an ArrayLike', function (done) {
        this.timeout(300);
        var arrayLike = {
            length: 3,
            0: 1,
            1: 2,
            2: 3
        };
        var expected = [1, 2, 3];
        Observable.from(arrayLike).subscribe(function (x) {
            chai_1.expect(x).to.equal(expected.shift());
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should handle an ArrayLike from arguments', function (done) {
        this.timeout(300);
        function makeArrayLike() {
            var args = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                args[_i - 0] = arguments[_i];
            }
            var expected = [1, 2, 3];
            Observable.from(arguments).subscribe(function (x) {
                chai_1.expect(x).to.equal(expected.shift());
            }, function (x) {
                done(new Error('should not be called'));
            }, function () {
                done();
            });
        }
        makeArrayLike(1, 2, 3);
    });
    it('should handle an ArrayLike with a mapFn', function (done) {
        this.timeout(300);
        var arrayLike = {
            length: 3,
            0: 1,
            1: 2,
            2: 3
        };
        var expected = [1, 1, 1];
        var mapFn = function (v, k) { return v - k; };
        Observable.from(arrayLike, mapFn).subscribe(function (x) {
            chai_1.expect(x).to.equal(expected.shift());
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should handle an ArrayLike with a thisArg', function (done) {
        var arrayLike = {
            length: 3,
            0: 1,
            1: 2,
            2: 3
        };
        var expected = [123, 123, 123];
        var mapFn = function (x, y) {
            return this.thing;
        };
        Observable.from(arrayLike, mapFn, { thing: 123 }).subscribe(function (x) {
            chai_1.expect(x).to.equal(expected.shift());
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should handle a promise', function (done) {
        var promise = Promise.resolve('pinky swear');
        Observable.from(promise).subscribe(function (x) {
            chai_1.expect(x).to.equal('pinky swear');
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should handle an "observableque" object', function (done) {
        var observablesque = {};
        observablesque[Symbol.observable] = function () {
            return {
                subscribe: function (observer) {
                    observer.next('test');
                    observer.complete();
                }
            };
        };
        Observable.from(observablesque).subscribe(function (x) {
            chai_1.expect(x).to.equal('test');
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should accept scheduler for observableque object', function () {
        var observablesque = {};
        observablesque[Symbol.observable] = function () {
            return {
                subscribe: function (observer) {
                    observer.next('x');
                    observer.complete();
                }
            };
        };
        var e1 = Observable.from(observablesque, rxTestScheduler);
        var expected = '(x|)';
        expectObservable(e1).toBe(expected);
    });
    it('should handle a string', function (done) {
        var expected = ['a', 'b', 'c'];
        Observable.from('abc').subscribe(function (x) {
            chai_1.expect(x).to.equal(expected.shift());
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should handle any iterable thing', function (done) {
        var iterable = {};
        var iteratorResults = [
            { value: 'one', done: false },
            { value: 'two', done: false },
            { done: true }
        ];
        var expected = ['one', 'two'];
        chai_1.expect(iterator_1.$$iterator).to.equal(Symbol.iterator);
        iterable[Symbol.iterator] = function () {
            return {
                next: function () {
                    return iteratorResults.shift();
                }
            };
        };
        Observable.from(iterable).subscribe(function (x) {
            chai_1.expect(x).to.equal(expected.shift());
        }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should throw for non observable object', function () {
        var r = function () {
            Observable.from({}).subscribe();
        };
        chai_1.expect(r).to.throw();
    });
    it('should handle object has observable symbol', function (done) {
        var value = 'x';
        Observable.from(Observable.of(value)).subscribe(function (x) {
            chai_1.expect(x).to.equal(value);
        }, function (err) {
            done(new Error('should not be called'));
        }, function () {
            done();
        });
    });
    it('should return T for ObservableLike objects', function () {
        type(function () {
            /* tslint:disable:no-unused-variable */
            var o1 = Observable.from([], Rx.Scheduler.asap);
            var o2 = Observable.from(Observable.empty());
            var o3 = Observable.from(new Promise(function (resolve) { return resolve(); }));
            /* tslint:enable:no-unused-variable */
        });
    });
    it('should return T and map for arrays', function () {
        type(function () {
            /* tslint:disable:no-unused-variable */
            var o1 = Observable.from([], function (x) { return x.toString(); }, null, Rx.Scheduler.asap);
            /* tslint:enable:no-unused-variable */
        });
    });
});
//# sourceMappingURL=from-spec.js.map