"use strict";
var chai_1 = require('chai');
var Rx = require('../../dist/cjs/Rx');
var Observable = Rx.Observable;
/** @test {repeat} */
describe('Observable.prototype.repeat', function () {
    asDiagram('repeat(3)')('should resubscribe count number of times', function () {
        var e1 = cold('--a--b--|                ');
        var subs = ['^       !                ',
            '        ^       !        ',
            '                ^       !'];
        var expected = '--a--b----a--b----a--b--|';
        expectObservable(e1.repeat(3)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should resubscribe multiple times', function () {
        var e1 = cold('--a--b--|                        ');
        var subs = ['^       !                        ',
            '        ^       !                ',
            '                ^       !        ',
            '                        ^       !'];
        var expected = '--a--b----a--b----a--b----a--b--|';
        expectObservable(e1.repeat(2).repeat(2)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should complete without emit when count is zero', function () {
        var e1 = cold('--a--b--|');
        var subs = [];
        var expected = '|';
        expectObservable(e1.repeat(0)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should emit source once when count is one', function () {
        var e1 = cold('--a--b--|');
        var subs = '^       !';
        var expected = '--a--b--|';
        expectObservable(e1.repeat(1)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should repeat until gets unsubscribed', function () {
        var e1 = cold('--a--b--|      ');
        var subs = ['^       !      ',
            '        ^     !'];
        var unsub = '              !';
        var expected = '--a--b----a--b-';
        expectObservable(e1.repeat(10), unsub).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should be able to repeat indefinitely until unsubscribed', function () {
        var e1 = cold('--a--b--|                                    ');
        var subs = ['^       !                                    ',
            '        ^       !                            ',
            '                ^       !                    ',
            '                        ^       !            ',
            '                                ^       !    ',
            '                                        ^   !'];
        var unsub = '                                            !';
        var expected = '--a--b----a--b----a--b----a--b----a--b----a--';
        expectObservable(e1.repeat(), unsub).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should not break unsubscription chain when unsubscribed explicitly', function () {
        var e1 = cold('--a--b--|                                    ');
        var subs = ['^       !                                    ',
            '        ^       !                            ',
            '                ^       !                    ',
            '                        ^       !            ',
            '                                ^       !    ',
            '                                        ^   !'];
        var unsub = '                                            !';
        var expected = '--a--b----a--b----a--b----a--b----a--b----a--';
        var result = e1
            .mergeMap(function (x) { return Observable.of(x); })
            .repeat()
            .mergeMap(function (x) { return Observable.of(x); });
        expectObservable(result, unsub).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should consider negative count as repeat indefinitely', function () {
        var e1 = cold('--a--b--|                                    ');
        var subs = ['^       !                                    ',
            '        ^       !                            ',
            '                ^       !                    ',
            '                        ^       !            ',
            '                                ^       !    ',
            '                                        ^   !'];
        var unsub = '                                            !';
        var expected = '--a--b----a--b----a--b----a--b----a--b----a--';
        expectObservable(e1.repeat(-1), unsub).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should not complete when source never completes', function () {
        var e1 = cold('-');
        var e1subs = '^';
        var expected = '-';
        expectObservable(e1.repeat(3)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
    });
    it('should not complete when source does not completes', function () {
        var e1 = cold('-');
        var unsub = '                              !';
        var subs = '^                             !';
        var expected = '-';
        expectObservable(e1.repeat(3), unsub).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should complete immediately when source does not complete without emit but count is zero', function () {
        var e1 = cold('-');
        var subs = [];
        var expected = '|';
        expectObservable(e1.repeat(0)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should complete immediately when source does not complete but count is zero', function () {
        var e1 = cold('--a--b--');
        var subs = [];
        var expected = '|';
        expectObservable(e1.repeat(0)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should emit source once and does not complete when source emits but does not complete', function () {
        var e1 = cold('--a--b--');
        var subs = ['^       '];
        var expected = '--a--b--';
        expectObservable(e1.repeat(3)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should complete when source is empty', function () {
        var e1 = cold('|');
        var e1subs = ['(^!)', '(^!)', '(^!)'];
        var expected = '|';
        expectObservable(e1.repeat(3)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
    });
    it('should complete when source does not emit', function () {
        var e1 = cold('----|        ');
        var subs = ['^   !        ',
            '    ^   !    ',
            '        ^   !'];
        var expected = '------------|';
        expectObservable(e1.repeat(3)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should complete immediately when source does not emit but count is zero', function () {
        var e1 = cold('----|');
        var subs = [];
        var expected = '|';
        expectObservable(e1.repeat(0)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should raise error when source raises error', function () {
        var e1 = cold('--a--b--#');
        var subs = '^       !';
        var expected = '--a--b--#';
        expectObservable(e1.repeat(2)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
    });
    it('should raises error if source throws', function () {
        var e1 = cold('#');
        var e1subs = '(^!)';
        var expected = '#';
        expectObservable(e1.repeat(3)).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
    });
    it('should raises error if source throws when repeating infinitely', function () {
        var e1 = cold('#');
        var e1subs = '(^!)';
        var expected = '#';
        expectObservable(e1.repeat()).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
    });
    it('should terminate repeat and throw if source subscription to _next throws', function () {
        var e1 = Observable.of(1, 2, rxTestScheduler);
        e1.subscribe(function () { throw new Error('error'); });
        chai_1.expect(function () {
            e1.repeat(3);
            rxTestScheduler.flush();
        }).to.throw();
    });
    it('should terminate repeat and throw if source subscription to _complete throws', function () {
        var e1 = Observable.of(1, 2, rxTestScheduler);
        e1.subscribe(function () {
            //noop
        }, function () {
            //noop
        }, function () { throw new Error('error'); });
        chai_1.expect(function () {
            e1.repeat(3);
            rxTestScheduler.flush();
        }).to.throw();
    });
    it('should terminate repeat and throw if source subscription to _next throws when repeating infinitely', function () {
        var e1 = Observable.of(1, 2, rxTestScheduler);
        e1.subscribe(function () { throw new Error('error'); });
        chai_1.expect(function () {
            e1.repeat();
            rxTestScheduler.flush();
        }).to.throw();
    });
    it('should terminate repeat and throw if source subscription to _complete throws when repeating infinitely', function () {
        var e1 = Observable.of(1, 2, rxTestScheduler);
        e1.subscribe(function () {
            //noop
        }, function () {
            //noop
        }, function () { throw new Error('error'); });
        chai_1.expect(function () {
            e1.repeat();
            rxTestScheduler.flush();
        }).to.throw();
    });
    it('should raise error after first emit succeed', function () {
        var repeated = false;
        var e1 = cold('--a--|').map(function (x) {
            if (repeated) {
                throw 'error';
            }
            else {
                repeated = true;
                return x;
            }
        });
        var expected = '--a----#';
        expectObservable(e1.repeat(2)).toBe(expected);
    });
    it('should repeat a synchronous source (multicasted and refCounted) multiple times', function (done) {
        var expected = [1, 2, 3, 1, 2, 3, 1, 2, 3, 1, 2, 3, 1, 2, 3];
        Observable.of(1, 2, 3)
            .multicast(function () { return new Rx.Subject(); })
            .refCount()
            .repeat(5)
            .subscribe(function (x) { chai_1.expect(x).to.equal(expected.shift()); }, function (x) {
            done(new Error('should not be called'));
        }, function () {
            chai_1.expect(expected.length).to.equal(0);
            done();
        });
    });
});
//# sourceMappingURL=repeat-spec.js.map