"use strict";
var Rx = require('../../dist/cjs/Rx');
var Observable = Rx.Observable;
/** @test {windowTime} */
describe('Observable.prototype.windowTime', function () {
    asDiagram('windowTime(50, 100)')('should emit windows given windowTimeSpan ' +
        'and windowCreationInterval', function () {
        var source = hot('--1--2--^-a--b--c--d--e---f--g--h-|');
        var subs = '^                         !';
        //  100 frames            0---------1---------2-----|
        //  50                     ----|
        //  50                               ----|
        //  50                                         ----|
        var expected = 'x---------y---------z-----|';
        var x = cold('--a--(b|)                  ');
        var y = cold('-d--e|           ');
        var z = cold('-g--h| ');
        var values = { x: x, y: y, z: z };
        var result = source.windowTime(50, 100, rxTestScheduler);
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should emit windows given windowTimeSpan', function () {
        var source = hot('--1--2--^--a--b--c--d--e--f--g--h--|');
        var subs = '^                          !';
        var timeSpan = time('----------|');
        //  100 frames            0---------1---------2------|
        var expected = 'x---------y---------z------|';
        var x = cold('---a--b--c|                 ');
        var y = cold('--d--e--f-|       ');
        var z = cold('-g--h--|');
        var values = { x: x, y: y, z: z };
        var result = source.windowTime(timeSpan, null, rxTestScheduler);
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should emit windows given windowTimeSpan and windowCreationInterval', function () {
        var source = hot('--1--2--^--a--b--c--d--e--f--g--h--|');
        var subs = '^                          !';
        var timeSpan = time('-----|');
        var interval = time('----------|');
        //  100 frames            0---------1---------2------|
        //  50                     ----|
        //  50                               ----|
        //  50                                         ----|
        var expected = 'x---------y---------z------|';
        var x = cold('---a-|                      ');
        var y = cold('--d--(e|)         ');
        var z = cold('-g--h|  ');
        var values = { x: x, y: y, z: z };
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should return a single empty window if source is empty', function () {
        var source = cold('|');
        var subs = '(^!)';
        var expected = '(w|)';
        var w = cold('|');
        var expectedValues = { w: w };
        var timeSpan = time('-----|');
        var interval = time('----------|');
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result).toBe(expected, expectedValues);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should split a Just source into a single window identical to source', function () {
        var source = cold('(a|)');
        var subs = '(^!)';
        var expected = '(w|)';
        var w = cold('(a|)');
        var expectedValues = { w: w };
        var timeSpan = time('-----|');
        var interval = time('----------|');
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result).toBe(expected, expectedValues);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should be able to split a never Observable into timely empty windows', function () {
        var source = hot('^----------');
        var subs = '^         !';
        var expected = 'a--b--c--d-';
        var timeSpan = time('---|');
        var interval = time('---|');
        var a = cold('---|       ');
        var b = cold('---|    ');
        var c = cold('---| ');
        var d = cold('--');
        var unsub = '          !';
        var expectedValues = { a: a, b: b, c: c, d: d };
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result, unsub).toBe(expected, expectedValues);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should emit an error-only window if outer is a simple throw-Observable', function () {
        var source = cold('#');
        var subs = '(^!)';
        var expected = '(w#)';
        var w = cold('#');
        var expectedValues = { w: w };
        var timeSpan = time('-----|');
        var interval = time('----------|');
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result).toBe(expected, expectedValues);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should handle source Observable which eventually emits an error', function () {
        var source = hot('--1--2--^--a--b--c--d--e--f--g--h--#');
        var subs = '^                          !';
        var timeSpan = time('-----|');
        var interval = time('----------|');
        //  100 frames            0---------1---------2------|
        //  50                     ----|
        //  50                               ----|
        //  50                                         ----|
        var expected = 'x---------y---------z------#';
        var x = cold('---a-|                      ');
        var y = cold('--d--(e|)         ');
        var z = cold('-g--h|  ');
        var values = { x: x, y: y, z: z };
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should emit windows given windowTimeSpan and windowCreationInterval, ' +
        'but outer is unsubscribed early', function () {
        var source = hot('--1--2--^--a--b--c--d--e--f--g--h--|');
        var subs = '^          !                ';
        var timeSpan = time('-----|');
        var interval = time('----------|');
        //  100 frames            0---------1---------2------|
        //  50                     ----|
        //  50                               ----|
        //  50                                         ----|
        var expected = 'x---------y-                ';
        var x = cold('---a-|                      ');
        var y = cold('--                ');
        var unsub = '           !                ';
        var values = { x: x, y: y };
        var result = source.windowTime(timeSpan, interval, rxTestScheduler);
        expectObservable(result, unsub).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(subs);
    });
    it('should not break unsubscription chains when result is unsubscribed explicitly', function () {
        var source = hot('--1--2--^--a--b--c--d--e--f--g--h--|');
        var sourcesubs = '^             !             ';
        var timeSpan = time('-----|');
        var interval = time('----------|');
        //  100 frames            0---------1---------2------|
        //  50                     ----|
        //  50                               ----|
        //  50                                         ----|
        var expected = 'x---------y----             ';
        var x = cold('---a-|                      ');
        var y = cold('--d--             ');
        var unsub = '              !             ';
        var values = { x: x, y: y };
        var result = source
            .mergeMap(function (x) { return Observable.of(x); })
            .windowTime(timeSpan, interval, rxTestScheduler)
            .mergeMap(function (x) { return Observable.of(x); });
        expectObservable(result, unsub).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(sourcesubs);
    });
});
//# sourceMappingURL=windowTime-spec.js.map