"use strict";
var chai_1 = require('chai');
var Rx = require('../../dist/cjs/Rx');
var Observable = Rx.Observable;
/** @test {windowToggle} */
describe('Observable.prototype.windowToggle', function () {
    asDiagram('windowToggle')('should emit windows governed by openings and closings', function () {
        var source = hot('--1--2--^-a--b--c--d--e--f--g--h-|');
        var subs = '^                        !';
        var e2 = cold('----w--------w--------w--|');
        var e2subs = '^                        !';
        var e3 = cold('-----|                ');
        //                                     -----(c|)
        //                                              -----(c|)
        var e3subs = ['    ^    !                ',
            '             ^    !       ',
            '                      ^  !'];
        var expected = '----x--------y--------z--|';
        var x = cold('-b--c|                ');
        var y = cold('-e--f|       ');
        var z = cold('-h-|');
        var values = { x: x, y: y, z: z };
        var result = source.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
        expectSubscriptions(e3.subscriptions).toBe(e3subs);
    });
    it('should emit windows that are opened by an observable from the first argument ' +
        'and closed by an observable returned by the function in the second argument', function () {
        var e1 = hot('--1--2--^--a--b--c--d--e--f--g--h--|');
        var e1subs = '^                          !';
        var e2 = cold('--------x-------x-------x--|');
        var e2subs = '^                          !';
        var e3 = cold('----------(x|)      ');
        //                                    ----------(x|)
        //                                            ----------(x|)
        var e3subs = ['        ^         !         ',
            '                ^         ! ',
            '                        ^  !'];
        var expected = '--------x-------y-------z--|';
        var x = cold('-c--d--e--(f|)      ');
        var y = cold('--f--g--h-| ');
        var z = cold('---|');
        var values = { x: x, y: y, z: z };
        var source = e1.windowToggle(e2, function (value) {
            chai_1.expect(value).to.equal('x');
            return e3;
        });
        expectObservable(source).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
        expectSubscriptions(e3.subscriptions).toBe(e3subs);
    });
    it('should emit windows using constying cold closings', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|      ');
        var e1subs = '^                                  !      ';
        var e2 = cold('--x-----------y--------z---|              ');
        var e2subs = '^                          !              ';
        var close = [
            cold('---------------s--|                     '),
            cold('----(s|)                    '),
            cold('---------------(s|)')];
        var closeSubs = ['  ^              !                        ',
            '              ^   !                       ',
            '                       ^           !      '];
        var expected = '--x-----------y--------z-----------|      ';
        var x = cold('--b---c---d---e|                        ');
        var y = cold('--e-|                       ');
        var z = cold('-g---h------|      ');
        var values = { x: x, y: y, z: z };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return close[i++]; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
        expectSubscriptions(close[0].subscriptions).toBe(closeSubs[0]);
        expectSubscriptions(close[1].subscriptions).toBe(closeSubs[1]);
        expectSubscriptions(close[2].subscriptions).toBe(closeSubs[2]);
    });
    it('should emit windows using constying hot closings', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|   ');
        var e1subs = '^                                  !   ';
        var e2 = cold('--x-----------y--------z---|           ');
        var e2subs = '^                          !           ';
        var closings = [
            { obs: hot('-1--^----------------s-|                   '),
                sub: '  ^              !                     ' },
            { obs: hot('-----3----4-------(s|)                 '),
                sub: '              ^   !                    ' },
            { obs: hot('-------3----4-------5----------------s|'),
                sub: '                       ^           !   ' }]; // eslint-disable-line key-spacing
        var expected = '--x-----------y--------z-----------|   ';
        var x = cold('--b---c---d---e|                     ');
        var y = cold('--e-|                    ');
        var z = cold('-g---h------|   ');
        var values = { x: x, y: y, z: z };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return closings[i++].obs; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
        expectSubscriptions(closings[0].obs.subscriptions).toBe(closings[0].sub);
        expectSubscriptions(closings[1].obs.subscriptions).toBe(closings[1].sub);
        expectSubscriptions(closings[2].obs.subscriptions).toBe(closings[2].sub);
    });
    it('should emit windows using constying empty delayed closings', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|   ');
        var e1subs = '^                                  !   ';
        var e2 = cold('--x-----------y--------z---|           ');
        var e2subs = '^                          !           ';
        var close = [cold('---------------|                     '),
            cold('----|                    '),
            cold('---------------|')];
        var expected = '--x-----------y--------z-----------|   ';
        var x = cold('--b---c---d---e|                     ');
        var y = cold('--e-|                    ');
        var z = cold('-g---h------|   ');
        var values = { x: x, y: y, z: z };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return close[i++]; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should emit windows using constying cold closings, outer unsubscribed early', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|      ');
        var e1subs = '^                !                        ';
        var e2 = cold('--x-----------y--------z---|              ');
        var e2subs = '^                !                        ';
        var close = [cold('-------------s---|                     '),
            cold('-----(s|)                   '),
            cold('---------------(s|)')];
        var closeSubs = ['  ^            !                          ',
            '              ^  !                        '];
        var expected = '--x-----------y---                        ';
        var x = cold('--b---c---d--|                          ');
        var y = cold('--e-                        ');
        var unsub = '                 !                        ';
        var values = { x: x, y: y };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return close[i++]; });
        expectObservable(result, unsub).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
        expectSubscriptions(close[0].subscriptions).toBe(closeSubs[0]);
        expectSubscriptions(close[1].subscriptions).toBe(closeSubs[1]);
        expectSubscriptions(close[2].subscriptions).toBe([]);
    });
    it('should not break unsubscription chains when result is unsubscribed explicitly', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|      ');
        var e1subs = '^              !                          ';
        var e2 = cold('--x-----------y--------z---|              ');
        var e2subs = '^              !                          ';
        var close = [cold('---------------s--|                     '),
            cold('----(s|)                    '),
            cold('---------------(s|)')];
        var closeSubs = ['  ^            !                          ',
            '              ^!                          '];
        var expected = '--x-----------y-                          ';
        var x = cold('--b---c---d---                          ');
        var y = cold('--                          ');
        var unsub = '               !                          ';
        var values = { x: x, y: y };
        var i = 0;
        var result = e1
            .mergeMap(function (x) { return Observable.of(x); })
            .windowToggle(e2, function () { return close[i++]; })
            .mergeMap(function (x) { return Observable.of(x); });
        expectObservable(result, unsub).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
        expectSubscriptions(close[0].subscriptions).toBe(closeSubs[0]);
        expectSubscriptions(close[1].subscriptions).toBe(closeSubs[1]);
    });
    it('should dispose window Subjects if the outer is unsubscribed early', function () {
        var source = hot('--a--b--c--d--e--f--g--h--|');
        var open = cold('o-------------------------|');
        var sourceSubs = '^        !                 ';
        var expected = 'x---------                 ';
        var x = cold('--a--b--c-                 ');
        var unsub = '         !                 ';
        var late = time('---------------|           ');
        var values = { x: x };
        var window;
        var result = source
            .windowToggle(open, function () { return Observable.never(); })
            .do(function (w) { window = w; });
        expectObservable(result, unsub).toBe(expected, values);
        expectSubscriptions(source.subscriptions).toBe(sourceSubs);
        rxTestScheduler.schedule(function () {
            chai_1.expect(function () {
                window.subscribe();
            }).to.throw(Rx.ObjectUnsubscribedError);
        }, late);
    });
    it('should propagate error thrown from closingSelector', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|      ');
        var e1subs = '^             !                           ';
        var e2 = cold('--x-----------y--------z---|              ');
        var e2subs = '^             !                           ';
        var close = [cold('---------------s--|                     '),
            cold('----(s|)                    '),
            cold('---------------(s|)')];
        var expected = '--x-----------#----                       ';
        var x = cold('--b---c---d-#                           ');
        var values = { x: x };
        var i = 0;
        var result = e1.windowToggle(e2, function () {
            if (i === 1) {
                throw 'error';
            }
            return close[i++];
        });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should propagate error emitted from a closing', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|');
        var e1subs = '^             !                     ';
        var e2 = cold('--x-----------y--------z---|        ');
        var e2subs = '^             !                     ';
        var close = [cold('---------------s--|               '),
            cold('#                     ')];
        var expected = '--x-----------(y#)                  ';
        var x = cold('--b---c---d-#                     ');
        var y = cold('#                     ');
        var values = { x: x, y: y };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return close[i++]; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should propagate error emitted late from a closing', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|');
        var e1subs = '^                  !                ';
        var e2 = cold('--x-----------y--------z---|        ');
        var e2subs = '^                  !                ';
        var close = [cold('---------------s--|               '),
            cold('-----#                ')];
        var expected = '--x-----------y----#                ';
        var x = cold('--b---c---d---e|                  ');
        var y = cold('--e--#                ');
        var values = { x: x, y: y };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return close[i++]; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle errors', function () {
        var e1 = hot('--a--^---b---c---d---e--#                ');
        var e1subs = '^                  !                ';
        var e2 = cold('--x-----------y--------z---|        ');
        var e2subs = '^                  !                ';
        var close = [cold('---------------s--|               '),
            cold('-------s|             ')];
        var expected = '--x-----------y----#                ';
        var x = cold('--b---c---d---e|                  ');
        var y = cold('--e--#                ');
        var values = { x: x, y: y };
        var i = 0;
        var result = e1.windowToggle(e2, function () { return close[i++]; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle empty source', function () {
        var e1 = cold('|');
        var e1subs = '(^!)';
        var e2 = cold('--o-----|');
        var e2subs = '(^!)';
        var e3 = cold('-----c--|');
        var expected = '|';
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle throw', function () {
        var e1 = cold('#');
        var e1subs = '(^!)';
        var e2 = cold('--o-----|');
        var e2subs = '(^!)';
        var e3 = cold('-----c--|');
        var expected = '#';
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle never', function () {
        var e1 = hot('-');
        var e1subs = '^                                           !';
        var e2 = cold('--o-----o------o-----o---o-----|             ');
        var e2subs = '^                              !             ';
        var e3 = cold('--c-|                                      ');
        var expected = '--u-----v------x-----y---z-------------------';
        var u = cold('--|                                        ');
        var v = cold('--|                                  ');
        var x = cold('--|                           ');
        var y = cold('--|                     ');
        var z = cold('--|                 ');
        var unsub = '                                            !';
        var values = { u: u, v: v, x: x, y: y, z: z };
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result, unsub).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle a never opening Observable', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|');
        var e1subs = '^                                  !';
        var e2 = cold('-');
        var e2subs = '^                                  !';
        var e3 = cold('--c-|                               ');
        var expected = '-----------------------------------|';
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle a never closing Observable', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|');
        var e1subs = '^                                  !';
        var e2 = cold('---o---------------o-----------|    ');
        var e2subs = '^                              !    ';
        var e3 = cold('-');
        var expected = '---x---------------y---------------|';
        var x = cold('-b---c---d---e---f---g---h------|');
        var y = cold('-f---g---h------|');
        var values = { x: x, y: y };
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle opening Observable that just throws', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|');
        var e1subs = '(^!)';
        var e2 = cold('#');
        var e2subs = '(^!)';
        var e3 = cold('--c-|');
        var subs = '(^!)';
        var expected = '#';
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected);
        expectSubscriptions(e1.subscriptions).toBe(subs);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
    it('should handle empty closing observable', function () {
        var e1 = hot('--a--^---b---c---d---e---f---g---h------|');
        var e1subs = '^                                  !';
        var e2 = cold('---o---------------o-----------|    ');
        var e2subs = '^                              !    ';
        var e3 = Observable.empty();
        var expected = '---x---------------y---------------|';
        var x = cold('|');
        var y = cold('|');
        var values = { x: x, y: y };
        var result = e1.windowToggle(e2, function () { return e3; });
        expectObservable(result).toBe(expected, values);
        expectSubscriptions(e1.subscriptions).toBe(e1subs);
        expectSubscriptions(e2.subscriptions).toBe(e2subs);
    });
});
//# sourceMappingURL=windowToggle-spec.js.map