"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var cssAttributes_1 = __importDefault(require("./cssAttributes"));
var Counter = /** @class */ (function () {
    function Counter() {
        this.map = {};
    }
    Counter.prototype.increase = function (element) {
        if (element.tagName in this.map) {
            this.map[element.tagName] = this.map[element.tagName] + 1;
        }
        else {
            this.map[element.tagName] = 1;
        }
        return this.map[element.tagName];
    };
    return Counter;
}());
exports.Counter = Counter;
function getAllElementsByXPath(node) {
    var rootNode = document.getElementsByTagName('html')[0];
    var rootPath = '//html[1]';
    if (node) {
        rootNode = node;
        rootPath = getElementXPath(rootNode);
    }
    var root = transform(rootNode);
    var allElements = {};
    allElements[rootPath] = root;
    allElements = mapElement(rootNode, rootPath, allElements);
    return allElements;
}
exports.default = getAllElementsByXPath;
function getElementXPath(node) {
    var paths = [];
    for (; node && node.nodeType === Node.ELEMENT_NODE; node = node.parentNode) {
        var index = 0;
        for (var sibling = node.previousSibling; sibling; sibling = sibling.previousSibling) {
            if (sibling.nodeType === Node.DOCUMENT_TYPE_NODE) {
                continue;
            }
            if (sibling.nodeName === node.nodeName) {
                ++index;
            }
        }
        var tagName = node.nodeName.toLowerCase();
        var pathIndex = '[' + (index + 1) + ']';
        paths.unshift(tagName + pathIndex);
    }
    return paths.length ? '/' + paths.join('/') : '';
}
exports.getElementXPath = getElementXPath;
function mapElement(element, parentPath, allElements) {
    if (!element || !element.children) {
        return allElements;
    }
    var counter = new Counter();
    for (var i = 0; i < element.childNodes.length; i++) {
        var child = element.childNodes[i];
        if (child.nodeType === child.ELEMENT_NODE ||
            (isNonEmptyTextNode(child) && containsOtherElements(element))) {
            if (child.nodeType === child.TEXT_NODE) {
                child.tagName = 'textnode';
            }
            var cnt = counter.increase(child);
            var path = parentPath + '/' + child.tagName.toLowerCase() + '[' + cnt + ']';
            allElements[path] = transform(child);
            mapElement(child, path, allElements);
        }
    }
    return allElements;
}
exports.mapElement = mapElement;
function transform(node) {
    var extractedAttributes = {
        tagName: node.tagName.toLowerCase(),
        text: getText(node),
        value: node.value,
        'tab-index': node.tabIndex,
        shown: isShown(node),
    };
    if (node.nodeType === node.TEXT_NODE) {
        addCoordinates(extractedAttributes, node.parentNode);
        return extractedAttributes;
    }
    // extract *all* HTML element attributes
    var attrs = node.attributes;
    for (var i = 0; i < attrs.length; i++) {
        var attributeName = attrs[i].name;
        var attributeValue = attrs[i].value;
        extractedAttributes[attributeName] = attributeValue;
    }
    // overwrite empty attributes (e.g. 'disabled')
    extractedAttributes['checked'] = node.checked;
    extractedAttributes['disabled'] = isDisabled(node);
    extractedAttributes['read-only'] = node.readOnly;
    // extract *given* CSS style attributes
    var style = getComputedStyleSafely(node);
    var parentStyle = getComputedStyleSafely(node.parentNode);
    for (var i = 0; i < cssAttributes_1.default.length; i++) {
        var attrName = cssAttributes_1.default[i];
        if (!extractedAttributes[attrName]) {
            if (parentStyle[attrName] != style[attrName]) {
                extractedAttributes[attrName] = style[attrName];
            }
        }
    }
    addCoordinates(extractedAttributes, node);
    return extractedAttributes;
}
exports.transform = transform;
function getText(node) {
    var firstNode = node.childNodes[0];
    if (firstNode && firstNode.nodeType === node.TEXT_NODE) {
        return firstNode.nodeValue;
    }
    if (node.nodeType === node.TEXT_NODE) {
        return node.nodeValue;
    }
    return '';
}
exports.getText = getText;
function getX(node) {
    var rect = node.getBoundingClientRect();
    return rect.left + window.scrollX;
}
exports.getX = getX;
function getY(node) {
    var rect = node.getBoundingClientRect();
    return rect.top + window.scrollY;
}
exports.getY = getY;
function addCoordinates(attributes, node) {
    // these attributes need special treatment
    attributes['absolute-x'] = getX(node);
    attributes['absolute-y'] = getY(node);
    attributes['absolute-width'] = node.getBoundingClientRect().width;
    attributes['absolute-height'] = node.getBoundingClientRect().height;
    var parentNode = node.parentNode;
    if (typeof parentNode.getBoundingClientRect === 'function') {
        attributes['x'] = getX(node) - getX(parentNode);
        attributes['y'] = getY(node) - getY(parentNode);
        attributes['width'] = node.getBoundingClientRect().width - parentNode.getBoundingClientRect().width;
        attributes['height'] = node.getBoundingClientRect().height - parentNode.getBoundingClientRect().height;
    }
    else {
        attributes['x'] = getX(node);
        attributes['y'] = getY(node);
        attributes['width'] = node.getBoundingClientRect().width;
        attributes['height'] = node.getBoundingClientRect().height;
    }
}
exports.addCoordinates = addCoordinates;
function isDisabled(node) {
    if (!node.disabled) {
        return false;
    }
    if (node.disabled === '') {
        return false;
    }
    if (node.disabled === 'disabled') {
        return true;
    }
    return node.disabled ? true : false;
}
exports.isDisabled = isDisabled;
//extract *given* CSS style attributes
function getComputedStyleSafely(node) {
    if (node === null) {
        return {};
    }
    try {
        return window.getComputedStyle(node) || {};
    }
    catch (err) { }
    return {};
}
exports.getComputedStyleSafely = getComputedStyleSafely;
function isShown(e) {
    if (e.nodeType === e.TEXT_NODE) {
        return isShown(e.parentNode);
    }
    return !!(e.offsetWidth || e.offsetHeight || e.getClientRects().length);
}
exports.isShown = isShown;
function isNonEmptyTextNode(node) {
    var nodeValue = node.nodeValue === null ? '' : node.nodeValue;
    return node.nodeType === node.TEXT_NODE && nodeValue.trim().length > 0;
}
exports.isNonEmptyTextNode = isNonEmptyTextNode;
function containsOtherElements(element) {
    return element.children.length > 0;
}
exports.containsOtherElements = containsOtherElements;
