import { Action, AnyAction } from './actions';
/**
 * A *reducer* (also called a *reducing function*) is a function that accepts
 * an accumulation and a value and returns a new accumulation. They are used
 * to reduce a collection of values down to a single value
 *
 * Reducers are not unique to Redux—they are a fundamental concept in
 * functional programming.  Even most non-functional languages, like
 * JavaScript, have a built-in API for reducing. In JavaScript, it's
 * `Array.prototype.reduce()`.
 *
 * In Redux, the accumulated value is the state object, and the values being
 * accumulated are actions. Reducers calculate a new state given the previous
 * state and an action. They must be *pure functions*—functions that return
 * the exact same output for given inputs. They should also be free of
 * side-effects. This is what enables exciting features like hot reloading and
 * time travel.
 *
 * Reducers are the most important concept in Redux.
 *
 * *Do not put API calls into reducers.*
 *
 * @template S The type of state consumed and produced by this reducer.
 * @template A The type of actions the reducer can potentially respond to.
 */
export declare type Reducer<S = any, A extends Action = AnyAction> = (state: S | undefined, action: A) => S;
/**
 * Object whose values correspond to different reducer functions.
 *
 * @template A The type of actions the reducers can potentially respond to.
 */
export declare type ReducersMapObject<S = any, A extends Action = AnyAction> = {
    [K in keyof S]: Reducer<S[K], A>;
};
/**
 * Infer a combined state shape from a `ReducersMapObject`.
 *
 * @template M Object map of reducers as provided to `combineReducers(map: M)`.
 */
export declare type StateFromReducersMapObject<M> = M extends ReducersMapObject ? {
    [P in keyof M]: M[P] extends Reducer<infer S, any> ? S : never;
} : never;
/**
 * Infer reducer union type from a `ReducersMapObject`.
 *
 * @template M Object map of reducers as provided to `combineReducers(map: M)`.
 */
export declare type ReducerFromReducersMapObject<M> = M extends {
    [P in keyof M]: infer R;
} ? R extends Reducer<any, any> ? R : never : never;
/**
 * Infer action type from a reducer function.
 *
 * @template R Type of reducer.
 */
export declare type ActionFromReducer<R> = R extends Reducer<any, infer A> ? A : never;
/**
 * Infer action union type from a `ReducersMapObject`.
 *
 * @template M Object map of reducers as provided to `combineReducers(map: M)`.
 */
export declare type ActionFromReducersMapObject<M> = M extends ReducersMapObject ? ActionFromReducer<ReducerFromReducersMapObject<M>> : never;
