'use strict'

const rgba = require('color-normalize')
const getBounds = require('array-bounds')
const colorId = require('color-id')
const snapPoints = require('snap-points-2d')
const extend = require('object-assign')
const glslify = require('glslify')
const search = require('binary-search-bounds')
const pick = require('pick-by-alias')
const updateDiff = require('update-diff')
const flatten = require('flatten-vertex-data')
const ie = require('is-iexplorer')
const {float32, fract32} = require('to-float32')
const arrayRange = require('array-range')
const parseRect = require('parse-rect')

module.exports = Scatter


function Scatter (regl, options) {
	if (typeof regl === 'function') {
		if (!options) options = {}
		options.regl = regl
	}
	else {
		options = regl
		regl = null
	}

	if (options && options.length) options.positions = options
	regl = options.regl

	// persistent variables
	let gl = regl._gl,
		drawMarker, drawCircle,
		sizeBuffer, positionBuffer, positionFractBuffer, colorBuffer,
		paletteTexture, palette = [], paletteIds = {},
		defaults = {
			color: 'black',
			borderColor: 'transparent',
			borderSize: 1,
			size: 12,
			opacity: 1,
			marker: undefined,
			viewport: null,
			range: null,
			pixelSize: null,
			offset: 0,
			count: 0,
			bounds: null,
			positions: [],
			snap: 1e4
		},

		// state
		groups = [],

		// textures for marker keys
		markerTextures = [null],
		markerCache = [null]

	const maxColors = 4096, maxSize = 100

	// texture with color palette
	paletteTexture = regl.texture({
		data: new Uint8Array(maxColors * 4),
		width: maxColors,
		height: 1,
		type: 'uint8',
		format: 'rgba',
		wrapS: 'clamp',
		wrapT: 'clamp',
		mag: 'nearest',
		min: 'nearest'
	})

	// buffers to reuse
	sizeBuffer = regl.buffer({
		usage: 'dynamic',
		type: 'float',
		data: null
	})
	colorBuffer = regl.buffer({
		usage: 'dynamic',
		type: 'uint16',
		data: null
	})
	positionBuffer = regl.buffer({
		usage: 'dynamic',
		type: 'float',
		data: null
	})
	positionFractBuffer = regl.buffer({
		usage: 'dynamic',
		type: 'float',
		data: null
	})

	// fast-create from existing regl-scatter instance
	if (options.clone) {
		groups = options.clone.groups.map(group => {
			group = extend({}, group)
			if (group.markerIds) {
				group.markerIds = group.markerIds.map(ids => {
					// recreate regl elements here
					let newIds = ids.slice()
					newIds.lod = ids.lod
					newIds.snap = ids.snap
					newIds.data = ids.data
					newIds.id = ids.id
					newIds.x = ids.x
					newIds.w = ids.w
					newIds.elements = regl.elements({
						primitive: 'points',
						type: 'uint32',
						data: ids.data
					})
					return newIds
				})
			}
			return group
		})

		// create marker textures
		options.clone.markers.forEach(markers => {
			addMarker(markers)
		})
		// clone palette texture
		updatePalette(options.clone.palette)
		updateBuffers({point: true, color: true, size: true})
	}
	// full create from options
	else {
		update(options)
	}


	// common shader options
	let shaderOptions = {
		uniforms: {
			pixelRatio: regl.context('pixelRatio'),
			palette: paletteTexture,
			paletteSize: (ctx, prop) => [maxColors, paletteTexture.height],
			scale: regl.prop('scale'),
			scaleFract: regl.prop('scaleFract'),
			translate: regl.prop('translate'),
			translateFract: regl.prop('translateFract'),
			opacity: regl.prop('opacity'),
			marker: regl.prop('marker'),
		},

		attributes: {
			position: positionBuffer,
			positionFract: positionFractBuffer,
			size: (ctx, prop) => prop.size.length ? {
				buffer: sizeBuffer,
				stride: 2,
				offset: 0
			} : {constant: [Math.round(prop.size * 255 / maxSize)]},
			borderSize: (ctx, prop) => prop.borderSize.length ? {
				buffer: sizeBuffer,
				stride: 2,
				offset: 1
			} : {constant: [Math.round(prop.borderSize * 255 / maxSize)]},
			colorId: (ctx, prop) => prop.color.length ? {
				buffer: colorBuffer,
				stride: 8,
				offset: 0
			} : {constant: [prop.color]},
			borderColorId: (ctx, prop) => prop.borderColor.length ? {
				buffer: colorBuffer,
				stride: 8,
				offset: 4
			} : {constant: [prop.borderColor]}
		},


		blend: {
			enable: true,
			color: [0,0,0,1],
			func: {
				srcRGB: 'src alpha',
				dstRGB: 'one minus src alpha',
				srcAlpha: 'one minus dst alpha',
				dstAlpha: 'one'
			}
		},

		scissor: {
			enable: true,
			box: regl.prop('viewport')
		},
		viewport: regl.prop('viewport'),

		stencil: {enable: false},
		depth: {enable: false},


		elements: regl.prop('elements'),
		count: regl.prop('count'),
		offset: regl.prop('offset'),

		primitive: 'points'
	}

	// IE11 shader workaround
	if (ie) {
		drawCircle = regl(extend({}, shaderOptions, {
			frag: glslify('./ie-frag.glsl'),
			vert: glslify('./ie-vert.glsl'),
			uniforms: {
				// FIXME: generate attribute color data
				color: (ctx, p) => {
					let id = p.color.length ? p.color[0] : p.color;
					return palette.slice(id * 4, id * 4 + 4).map(v => v / 255)
				},
				borderColor: (ctx, p) => {
					let id = p.borderColor.length ? p.borderColor[0] : p.borderColor;
					return palette.slice(id * 4, id * 4 + 4).map(v => v / 255)
				},
				pixelRatio: regl.context('pixelRatio'),
				palette: paletteTexture,
				paletteSize: (ctx, prop) => [maxColors, paletteTexture.height],
				scale: regl.prop('scale'),
				scaleFract: regl.prop('scaleFract'),
				translate: regl.prop('translate'),
				translateFract: regl.prop('translateFract'),
				opacity: regl.prop('opacity'),
				marker: regl.prop('marker')
			},
			attributes: {
				position: positionBuffer,
				positionFract: positionFractBuffer,
				size: shaderOptions.attributes.size,
				borderSize: shaderOptions.attributes.borderSize
			}
		}))
	}
	else {
		// draw sdf-marker
		let markerOptions = extend({}, shaderOptions)
		markerOptions.frag = glslify('./marker-frag.glsl')
		markerOptions.vert = glslify('./marker-vert.glsl')

		drawMarker = regl(markerOptions)

		// draw circle
		let circleOptions = extend({}, shaderOptions)
		circleOptions.frag = glslify('./circle-frag.glsl')
		circleOptions.vert = glslify('./circle-vert.glsl')

		drawCircle = regl(circleOptions)
	}

	// expose API
	extend(scatter2d, {
		update: update,
		draw: draw,
		destroy: destroy,
		regl: regl,
		gl: gl,
		canvas: gl.canvas,
		groups: groups,
		markers: markerCache,
		palette: palette
	})


	// main update/draw function
	function scatter2d (opts) {
		// update
		if (opts) {
			update(opts)
		}

		// destroy
		else if (opts === null) {
			destroy()
		}

		draw()
	}

	// draw all groups or only indicated ones
	function draw (opts) {
		if (typeof opts === 'number') return drawGroup(opts)

		// highlight elements
		if (Array.isArray(opts)) {
			opts.forEach((els, i) => {
				if (els == null) return
				if (els.length) return drawGroup(els, i)
				return drawGroup(els)
			})
			return
		}

		// make options a batch
		groups.forEach((group, i) => {
			if (!group) return

			drawGroup(i)
		})
	}

	// draw specific scatter group
	function drawGroup (group, id) {
		if (typeof group === 'number') group = groups[group]

		let els

		if (Array.isArray(group)) {
			els = group
			group = groups[id]
		}

		if (!(group && group.count && group.opacity)) return

		// if subset of elements to redraw passed - form a whitelist
		let whitelist
		if (els) {
			whitelist = Array(group.count);

			for (let i = 0; i < els.length; i++) {
				whitelist[els[i]] = true
			}
		}

		// draw circles
		// FIXME remove regl._refresh hooks once regl issue #427 is fixed
		if (group.markerIds[0]) {
			regl._refresh()

			let opts = getMarkerDrawOptions(group.markerIds[0], group, whitelist)

			drawCircle(opts)
		}

		// draw all other available markers
		let batch = []
		for (let i = 1; i < group.markerIds.length; i++) {
			let ids = group.markerIds[i]

			if (!ids || !ids.length) continue

			[].push.apply(batch, getMarkerDrawOptions(ids, group, whitelist))
		}

		if (batch.length) {
			regl._refresh()
			drawMarker(batch)
		}
	}

	// get options for the marker ids
	function getMarkerDrawOptions(ids, group, whitelist) {
		let {range, offset} = group
		// unsnapped options
		if (!ids.snap) {
			let elements = whitelist ? filter(ids.data, whitelist) : ids.elements;

			return [extend({}, group, {
				elements: elements,
				offset: 0,
				count: whitelist ? elements.length : ids.length,
				marker: markerTextures[ids.id]
			})]
		}

		// scales batch
		let batch = []
		let {lod, x, id} = ids

		let pixelSize = Math.min((range[2] - range[0]) / group.viewport.width, (range[3] - range[1]) / group.viewport.height)

		for (let scaleNum = lod.length; scaleNum--;) {
			let level = lod[scaleNum]

			// FIXME: use minSize-adaptive coeff here, if makes sense, mb we need dist tho
			if (level.pixelSize && level.pixelSize < pixelSize && scaleNum > 1) {
				continue
			}

			let intervalStart = level.offset
			let intervalEnd = level.count + intervalStart

			let startOffset = search.ge(x, range[0], intervalStart, intervalEnd - 1)
			let endOffset = search.lt(x, range[2], startOffset, intervalEnd - 1) + 1

			if (endOffset <= startOffset) continue

			// whitelisted level requires subelements from the range
			if (whitelist) {
				let elements = filter(ids.data.subarray(startOffset, endOffset), whitelist)

				batch.push(extend({}, group, {
					elements: elements,
					marker: markerTextures[id],
					offset: 0,
					count: elements.length
				}))
			}
			else {
				batch.push(extend({}, group, {
					elements: ids.elements,
					marker: markerTextures[id],
					offset: startOffset,
					count: endOffset - startOffset
				}))
			}
		}

		function filter(offsets, whitelist) {
			let subEls = []
			for (let i = 0, l = offsets.length; i < l; i++) {
				let el = offsets[i]
				let id = el - offset
				if (whitelist[id]) {
					subEls.push(el)
				}
			}
			return subEls
		}

		return batch
	}

	// update groups options
	function update (options) {
		if (!options) return

		// direct points argument
		if (options.length != null) {
			if (typeof options[0] === 'number') options = [{positions: options}]
		}
		// make options a batch
		else if (!Array.isArray(options)) options = [options]

		// global count of points
		let pointCount = 0, sizeCount = 0, colorCount = 0

		scatter2d.groups = groups = options.map((options, i) => {
			let group = groups[i]

			if (options === undefined) return group

			if (options === null) options = { positions: null }
			else if (typeof options === 'function') options = {after: options}
			else if (typeof options[0] === 'number') options = {positions: options}

			// copy options to avoid mutation & handle aliases
			options = pick(options, {
				positions: 'positions data points',
				snap: 'snap cluster',
				size: 'sizes size radius',
				borderSize: 'borderSizes borderSize stroke-width strokeWidth outline',
				color: 'colors color fill fill-color fillColor',
				borderColor: 'borderColors borderColor stroke stroke-color strokeColor',
				palette: 'palette swatch',
				marker: 'markers marker shape',
				range: 'range dataBox',
				viewport: 'viewport viewBox',
				opacity: 'opacity alpha'
			})

			if (options.positions === null) options.positions = []

			if (!group) {
				groups[i] = group = {
					id: i,
					scale: null,
					translate: null,
					scaleFract: null,
					translateFract: null,

					// list of ids corresponding to markers, with inner props
					markerIds: []
				}
				options = extend({}, defaults, options)
			}

			// force update triggers
			if (options.positions && !('marker' in options)) {
				options.marker = group.marker
				delete group.marker
			}

			// updating markers cause recalculating snapping
			if (options.marker && !('positions' in options)) {
				options.positions = group.positions
				delete group.positions
			}

			updateDiff(group, options, [{
				snap: true,
				size: s => {
					sizeCount += s.length ? 1 : 0
					return s
				},
				borderSize: s => {
					sizeCount += s.length ? 1 : 0
					return s
				},
				opacity: parseFloat,

				// add colors to palette, save references
				color: c => {
					c = updateColor(c)
					colorCount++
					return c
				},
				borderColor: c => {
					c = updateColor(c)
					colorCount++
					return c
				},

				positions: (positions, group, options) => {
					positions = flatten(positions, 'float64')

					let count = group.count = Math.floor(positions.length / 2)
					let bounds = group.bounds = count ? getBounds(positions, 2) : null

					// if range is not provided updated - recalc it
					if (!options.range && !group.range) {
						delete group.range
						options.range = bounds
					}

					group.offset = pointCount
					pointCount += count

					// reset marker
					if (!options.marker && !group.marker) {
						delete group.marker;
						options.marker = null;
					}

					return positions
				}
			}, {
				// create marker ids corresponding to known marker textures
				marker: (markers, group, options) => {
					// reset marker elements
					group.markerIds.length = 0

					// single sdf marker
					if (!markers || typeof markers[0] === 'number') {
						let id = addMarker(markers)

						let elements = arrayRange(group.count)

						group.markerIds[id] = elements
					}
					// per-point markers
					else {
						for (let i = 0, l = markers.length; i < l; i++) {
							let id = addMarker(markers[i])

							if (!group.markerIds[id]) group.markerIds[id] = []
							group.markerIds[id].push(i)
						}
					}

					return markers
				}
			}, {
				// recalculate per-marker snapping
				// first, it is faster to snap 100 points 100 times than 10000 points once (practically, not theoretically)
				// second, it is easier to subset render per-marker than per-generic set
				positions: (positions, group) => {
					let {markerIds, snap, bounds, offset} = group

					for (let i = 0; i < markerIds.length; i++) {
						let ids = markerIds[i]
						if (!ids || !ids.length) continue

						let l = ids.length, els

						ids.id = i;

						if (snap && (snap === true || l > snap)) {
							ids.snap = true
							let x = ids.x = Array(l)
							let w = ids.w = Array(l)
							let markerPoints

							// multimarker snapping is computationally more intense
							if (markerIds.length > 1) {
								markerPoints = Array(l * 2)

								for (let i = 0; i < l; i++) {
									let id = ids[i]
									markerPoints[i * 2] = positions[id * 2]
									markerPoints[i * 2 + 1] = positions[id * 2 + 1]
								}
							}
							else {
								markerPoints = new Float64Array(positions.length)
								markerPoints.set(positions)
							}

							// shuffled_id: real_id
							let i2id = new Uint32Array(l)

							ids.lod = snapPoints(markerPoints, i2id, w, bounds)

							els = new Uint32Array(l)
							for (let i = 0; i < l; i++) {
								let id = i2id[i], iid = ids[id]
								els[i] = iid + offset
								x[i] = positions[iid * 2]
							}
						}
						else {
							els = new Uint32Array(l)
							for (let i = 0; i < l; i++) {
								els[i] = ids[i] + offset
							}
						}

						ids.data = els;
						ids.elements = regl.elements({
							primitive: 'points',
							type: 'uint32',
							data: els
						})
					}
				},

				range: (range, group, options) => {
					let bounds = group.bounds

					if (!bounds) return
					if (!range) range = bounds

					group.scale = [1 / (range[2] - range[0]), 1 / (range[3] - range[1])]
					group.translate = [-range[0], -range[1]]

					group.scaleFract = fract32(group.scale)
					group.translateFract = fract32(group.translate)

					return range
				},

				viewport: vp => {
					return parseRect(vp || [
						gl.drawingBufferWidth,
						gl.drawingBufferHeight
					])
				}
			}])

			return group
		})

		updateBuffers({
			point: pointCount,
			size: sizeCount,
			color: colorCount
		})
	}

	// update buffers data based on existing groups
	function updateBuffers({point, size, color}) {
		// put point/color data into buffers, if updated any of them
		let len = groups.reduce((acc, group, i) => {
			return acc + (group ? group.count : 0)
		}, 0)

		if (point) {
			let positionData = new Float32Array(len * 2)
			let positionFractData = new Float32Array(len * 2)

			groups.forEach((group, i) => {
				if (!group) return
				let {positions, count, offset} = group
				if (!count) return
				positionData.set(float32(positions), offset * 2)
				positionFractData.set(fract32(positions), offset * 2)
			})

			positionBuffer(positionData)
			positionFractBuffer(positionFractData)
		}

		if (size) {
			let sizeData = new Uint8Array(len * 2)

			groups.forEach((group, i) => {
				if (!group) return
				let {count, offset, size, borderSize} = group
				if (!count) return

				if (size.length || borderSize.length) {
					let sizes = new Uint8Array(count*2)
					for (let i = 0; i < count; i++) {
						// we downscale size to allow for fractions
						sizes[i*2] = Math.round((size[i] == null ? size : size[i]) * 255 / maxSize)
						sizes[i*2 + 1] = Math.round((borderSize[i] == null ? borderSize : borderSize[i]) * 255 / maxSize)
					}
					sizeData.set(sizes, offset * 2)
				}
			})
			sizeBuffer(sizeData)
		}

		if (color) {
			let colorData = new Uint16Array(len * 4)

			groups.forEach((group, i) => {
				if (!group) return
				let {count, offset, color, borderColor} = group
				if (!count) return

				if (color.length || borderColor.length) {
					let colorIds = new Uint16Array(count * 4)
					for (let i = 0; i < count; i++) {
						if (color[i] != null) {
							colorIds[i*4] = color[i] % maxColors
							colorIds[i*4 + 1] = Math.floor(color[i] / maxColors)
						}
						if (borderColor[i] != null) {
							colorIds[i*4 + 2] = borderColor[i] % maxColors
							colorIds[i*4 + 3] = Math.floor(borderColor[i] / maxColors)
						}
					}

					colorData.set(colorIds, offset * 4)
				}
			})


			colorBuffer(colorData)
		}
	}

	// get (and create) marker texture id
	function addMarker (sdf) {
		let pos = sdf == null ? 0 : markerCache.indexOf(sdf)

		if (pos >= 0) return pos

		// convert sdf to 0..255 range
		let distArr
		if (sdf instanceof Uint8Array || sdf instanceof Uint8ClampedArray) {
			distArr = sdf
		}
		else {
			distArr = new Uint8Array(sdf.length)
			for (let i = 0, l = sdf.length; i < l; i++) {
				distArr[i] = sdf[i] * 255
			}
		}

		let radius = Math.floor(Math.sqrt(distArr.length))

		pos = markerTextures.length

		markerCache.push(sdf)
		markerTextures.push(regl.texture({
			channels: 1,
			data: distArr,
			radius: radius,
			mag: 'linear',
			min: 'linear'
		}))

		return pos
	}

	// register color to palette, return it's index or list of indexes
	function updateColor (colors) {
		if (!Array.isArray(colors)) {
			colors = [colors]
		}

		let idx = []

		for (let i = 0; i < colors.length; i++) {
			let color = colors[i]

			// idx colors directly
			if (typeof color === 'number') {
				idx[i] = color
				continue
			}

			color = rgba(color, 'uint8')

			let id = colorId(color, false)

			// if new color - save it
			if (paletteIds[id] == null) {
				let pos = palette.length
				paletteIds[id] = Math.floor(pos / 4)
				palette[pos] = color[0]
				palette[pos+1] = color[1]
				palette[pos+2] = color[2]
				palette[pos+3] = color[3]
			}

			idx[i] = paletteIds[id]
		}

		// limit max color
		updatePalette(palette)

		// keep static index for single-color property
		return idx.length === 1 ? idx[0] : idx
	}

	function updatePalette(palette) {
		let requiredHeight = Math.ceil(palette.length * .25 / maxColors)

		// pad data
		if (requiredHeight > 1) {
			palette = palette.slice()
			for (let i = (palette.length * .25) % maxColors; i < requiredHeight * maxColors; i++) {
				palette.push(0, 0, 0, 0)
			}
		}

		// ensure height
		if (paletteTexture.height < requiredHeight) {
			paletteTexture.resize(maxColors, requiredHeight)
		}

		// update full data
		paletteTexture.subimage({
			width: Math.min(palette.length * .25, maxColors),
			height: requiredHeight,
			data: palette
		}, 0, 0)
	}

	// remove unused stuff
	function destroy () {
		groups.length = 0

		sizeBuffer.destroy()
		positionBuffer.destroy()
		positionFractBuffer.destroy()
		colorBuffer.destroy()
		paletteTexture.destroy()
	}

	return scatter2d
}
