import { propertiesToMdxJsxAttributes } from 'hast-util-properties-to-mdx-jsx-attributes';
import parseSrcset from 'parse-srcset';
import { visit } from 'unist-util-visit';
const urlPattern = /^(https?:)?\//;
const relativePathPattern = /\.\.?\//;
export const defaultAttributes = {
    audio: 'src',
    embed: 'src',
    img: ['src', 'srcset'],
    object: 'data',
    source: ['src', 'srcset'],
    track: 'src',
    video: ['poster', 'src']
};
/**
 * A rehype MDX plugin for converting media sources into imports.
 */
const rehypeMdxImportMedia = ({ attributes = defaultAttributes, elementAttributeNameCase, preserveHash = 'import', preserveQuery = 'import', resolve = true } = {}) => {
    const elementMap = new Map(Object.entries(attributes).map(([tagName, attributeNames]) => {
        const set = new Set();
        if (typeof attributeNames === 'string') {
            set.add(attributeNames.toLowerCase());
        }
        else {
            for (const name of attributeNames) {
                set.add(name.toLowerCase());
            }
        }
        return [tagName.toLowerCase(), set];
    }));
    return (ast) => {
        const imports = [];
        const imported = new Map();
        visit(ast, 'element', (node, index, parent) => {
            const attributeNames = elementMap.get(node.tagName);
            if (!attributeNames) {
                return;
            }
            let shouldReplace = false;
            // Don’t even bother continuing if there are no properties to replace.
            for (const name in node.properties) {
                if (attributeNames.has(name.toLowerCase())) {
                    shouldReplace = true;
                    break;
                }
            }
            if (!shouldReplace) {
                return;
            }
            shouldReplace = false;
            /**
             * Generate an identifier node for an import path.
             *
             * If the path should not be replaced, nothing is returned. If an identifier was already
             * calculated for this path, it is returned instead.
             *
             * @param importSource
             *   The path to get an identifier for.
             * @returns
             *   The matching identifier, or none.
             */
            function getIdentifier(importSource) {
                let value = importSource;
                if (urlPattern.test(value)) {
                    return [];
                }
                if (!relativePathPattern.test(value) && resolve) {
                    value = `./${value}`;
                }
                const hashIndex = value.indexOf('#');
                const hash = hashIndex === -1 ? '' : value.slice(hashIndex);
                const remainder = hashIndex === -1 ? value : value.slice(0, hashIndex);
                const queryIndex = remainder.indexOf('?');
                const query = queryIndex === -1 ? '' : remainder.slice(queryIndex);
                value = queryIndex === -1 ? remainder : remainder.slice(0, queryIndex);
                let propChunk = '';
                if (preserveQuery === 'import') {
                    value += query;
                }
                else if (preserveQuery === 'jsx') {
                    propChunk += query;
                }
                else if (preserveQuery === 'both') {
                    value += query;
                    propChunk += query;
                }
                if (preserveHash === 'import') {
                    value += hash;
                }
                else if (preserveHash === 'jsx') {
                    propChunk += hash;
                }
                else if (preserveHash === 'both') {
                    value += hash;
                    propChunk += hash;
                }
                let name = imported.get(value);
                if (!name) {
                    name = `_rehypeMdxImportMedia${imported.size}`;
                    imports.push({
                        type: 'ImportDeclaration',
                        source: { type: 'Literal', value },
                        specifiers: [{ type: 'ImportDefaultSpecifier', local: { type: 'Identifier', name } }]
                    });
                    imported.set(value, name);
                }
                shouldReplace = true;
                return [{ type: 'Identifier', name }, propChunk];
            }
            const replacements = propertiesToMdxJsxAttributes(node.properties, {
                elementAttributeNameCase,
                transform(name, value) {
                    if (!value) {
                        return value;
                    }
                    const lower = name.toLowerCase();
                    if (!attributeNames.has(lower)) {
                        return value;
                    }
                    if (lower !== 'srcset') {
                        const [identifier, extra] = getIdentifier(value);
                        if (!identifier) {
                            return value;
                        }
                        if (extra) {
                            return {
                                type: 'TemplateLiteral',
                                expressions: [identifier],
                                quasis: [
                                    { type: 'TemplateElement', tail: false, value: { raw: '' } },
                                    { type: 'TemplateElement', tail: true, value: { raw: extra } }
                                ]
                            };
                        }
                        return identifier;
                    }
                    const srcset = parseSrcset(value);
                    const expressions = [];
                    const quasis = [];
                    let raw = '';
                    for (const [srcIndex, src] of srcset.entries()) {
                        const [identifier, extra] = getIdentifier(src.url);
                        if (identifier) {
                            quasis.push({ type: 'TemplateElement', tail: false, value: { raw } });
                            expressions.push(identifier);
                            raw = extra;
                        }
                        else {
                            raw += src.url;
                        }
                        if (src.d) {
                            raw += ` ${src.d}x`;
                        }
                        if (src.w) {
                            raw += ` ${src.w}w`;
                        }
                        if (src.h) {
                            raw += ` ${src.h}h`;
                        }
                        if (srcIndex < srcset.length - 1) {
                            raw += ',';
                        }
                    }
                    if (!expressions.length) {
                        return value;
                    }
                    quasis.push({ type: 'TemplateElement', tail: true, value: { raw } });
                    return { type: 'TemplateLiteral', expressions, quasis };
                }
            });
            if (shouldReplace) {
                parent.children[index] = {
                    type: 'mdxJsxTextElement',
                    name: node.tagName,
                    attributes: replacements,
                    children: node.children,
                    data: node.data,
                    position: node.position
                };
            }
        });
        if (imports.length) {
            ast.children.unshift({
                type: 'mdxjsEsm',
                value: '',
                data: {
                    estree: {
                        type: 'Program',
                        sourceType: 'module',
                        body: imports
                    }
                }
            });
        }
    };
};
export default rehypeMdxImportMedia;
//# sourceMappingURL=rehype-mdx-import-media.js.map