/**
 * @public
 */
declare const defaultEqualityCheck: EqualityFn;
/**
 * @public
 */
interface DefaultMemoizeOptions {
    /**
     * Used to compare the individual arguments of the provided calculation function.
     *
     * @default defaultEqualityCheck
     */
    equalityCheck?: EqualityFn;
    /**
     * If provided, used to compare a newly generated output value against previous values in the cache.
     * If a match is found, the old value is returned. This addresses the common
     * ```ts
     * todos.map(todo => todo.id)
     * ```
     * use case, where an update to another field in the original data causes a recalculation
     * due to changed references, but the output is still effectively the same.
     */
    resultEqualityCheck?: EqualityFn;
    /**
     * The cache size for the selector. If greater than 1, the selector will use an LRU cache internally.
     *
     * @default 1
     */
    maxSize?: number;
}
/**
 * The standard memoize function used by `createSelector`.
 * @param func - The function to be memoized.
 * @param equalityCheckOrOptions - Either an equality check function or an options object.
 * @returns A memoized function with a `.clearCache()` method attached.
 *
 * @template Func - The type of the function that is memoized.
 *
 * @see {@link https://github.com/reduxjs/reselect#defaultmemoizefunc-equalitycheckoroptions--defaultequalitycheck defaultMemoize}
 *
 * @public
 */
declare function defaultMemoize<Func extends AnyFunction>(func: Func, equalityCheckOrOptions?: EqualityFn | DefaultMemoizeOptions): Func & {
    clearCache: () => void;
};

/**
 * @internal
 */
type LongestTuple<T extends readonly unknown[][]> = T extends [
    infer U extends unknown[]
] ? U : T extends [infer U, ...infer R extends unknown[][]] ? MostProperties<U, LongestTuple<R>> : never;
/**
 * @internal
 */
type MostProperties<T, U> = keyof U extends keyof T ? T : U;
/**
 * @internal
 */
type ElementAt<T extends unknown[], N extends keyof any> = N extends keyof T ? T[N] : unknown;
/**
 * @internal
 */
type ElementsAt<T extends readonly unknown[][], N extends keyof any> = {
    [K in keyof T]: ElementAt<T[K], N>;
};
/**
 * @internal
 */
type Intersect<T extends readonly unknown[]> = T extends [] ? unknown : T extends [infer H, ...infer T] ? H & Intersect<T> : T[number];
/**
 * @internal
 */
type MergeTuples<T extends readonly unknown[][], L extends unknown[] = LongestTuple<T>> = {
    [K in keyof L]: Intersect<ElementsAt<T, K>>;
};
/**
 * @internal
 */
type ExtractParameters<T extends readonly AnyFunction[]> = {
    [K in keyof T]: Parameters<T[K]>;
};
/**
 * @internal
 */
type MergeParameters<T extends readonly AnyFunction[]> = '0' extends keyof T ? MergeTuples<ExtractParameters<T>> : Parameters<T[number]>;

/**
 * A standard selector function.
 * @template State - The first value, often a Redux root state object.
 * @template Result - The final result returned by the selector.
 * @template Params - All additional arguments passed into the selector.
 *
 * @public
 */
type Selector<State = any, Result = unknown, Params extends readonly any[] = any[]> = Distribute<
/**
 * A function that takes a state and returns data that is based on that state.
 *
 * @param state - The first argument, often a Redux root state object.
 * @param params - All additional arguments passed into the selector.
 * @returns A derived value from the state.
 */
(state: State, ...params: FallbackIfNever<Params, []>) => Result>;
/**
 * An array of input selectors.
 *
 * @public
 */
type SelectorArray = readonly Selector[];
/**
 * Extracts an array of all return types from all input selectors.
 *
 * @public
 */
type SelectorResultArray<Selectors extends SelectorArray> = ExtractReturnType<Selectors>;
/**
 * The options object used inside `createSelector` and `createSelectorCreator`.
 *
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 * @template OverrideMemoizeFunction - The type of the optional `memoize` function that could be passed into the options object inside `createSelector` to override the original `memoize` function that was initially passed into `createSelectorCreator`.
 * @template OverrideArgsMemoizeFunction - The type of the optional `argsMemoize` function that could be passed into the options object inside `createSelector` to override the original `argsMemoize` function that was initially passed into `createSelectorCreator`. If none was initially provided, `defaultMemoize` will be used.
 *
 * @public
 */
interface CreateSelectorOptions<MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, OverrideMemoizeFunction extends UnknownMemoizer = never, OverrideArgsMemoizeFunction extends UnknownMemoizer = never> {
    /**
     * Overrides the global input stability check for the selector.
     * - `once` - Run only the first time the selector is called.
     * - `always` - Run every time the selector is called.
     * - `never` - Never run the input stability check.
     *
     * @default 'once'
     *
     * @see {@link https://github.com/reduxjs/reselect#development-only-checks development-only-checks}
     * @see {@link https://github.com/reduxjs/reselect#inputstabilitycheck inputStabilityCheck}
     * @see {@link https://github.com/reduxjs/reselect#per-selector-configuration per-selector-configuration}
     *
     * @since 5.0.0
     */
    inputStabilityCheck?: StabilityCheckFrequency;
    /**
     * The memoize function that is used to memoize the {@linkcode OutputSelectorFields.resultFunc resultFunc}
     * inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
     *
     * When passed directly into `createSelector`, it overrides the `memoize` function initially passed into `createSelectorCreator`.
     *
     * @example
     * ```ts
     * import { createSelector, weakMapMemoize } from 'reselect'
     *
     * const selectTodoById = createSelector(
     *   [
     *     (state: RootState) => state.todos,
     *     (state: RootState, id: number) => id
     *   ],
     *   (todos) => todos[id],
     *   { memoize: weakMapMemoize }
     * )
     * ```
     *
     * @since 5.0.0
     */
    memoize: FallbackIfNever<OverrideMemoizeFunction, MemoizeFunction>;
    /**
     * The optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
     *
     * When passed directly into `createSelector`, it overrides the `argsMemoize` function initially passed into `createSelectorCreator`. If none was initially provided, `defaultMemoize` will be used.
     *
     * @example
     * ```ts
     * import { createSelector, weakMapMemoize } from 'reselect'
     *
     * const selectTodoById = createSelector(
     *   [
     *     (state: RootState) => state.todos,
     *     (state: RootState, id: number) => id
     *   ],
     *   (todos) => todos[id],
     *   { argsMemoize: weakMapMemoize }
     * )
     * ```
     *
     * @default defaultMemoize
     *
     * @since 5.0.0
     */
    argsMemoize?: FallbackIfNever<OverrideArgsMemoizeFunction, ArgsMemoizeFunction>;
    /**
     * Optional configuration options for the {@linkcode CreateSelectorOptions.memoize memoize} function.
     * These options are passed to the {@linkcode CreateSelectorOptions.memoize memoize} function as the second argument.
     *
     * @since 5.0.0
     */
    memoizeOptions?: OverrideMemoizeOptions<MemoizeFunction, OverrideMemoizeFunction>;
    /**
     * Optional configuration options for the {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} function.
     * These options are passed to the {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} function as the second argument.
     *
     * @since 5.0.0
     */
    argsMemoizeOptions?: OverrideMemoizeOptions<ArgsMemoizeFunction, OverrideArgsMemoizeFunction>;
}
/**
 * The additional fields attached to the output selector generated by `createSelector`.
 *
 * **Note**: Although {@linkcode CreateSelectorOptions.memoize memoize}
 * and {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} are included in the attached fields,
 * the fields themselves are independent of the type of
 * {@linkcode CreateSelectorOptions.memoize memoize} and {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} functions.
 * Meaning this type is not going to generate additional fields based on what functions we use to memoize our selectors.
 *
 * _This type is not to be confused with {@linkcode ExtractMemoizerFields ExtractMemoizerFields}._
 *
 * @template InputSelectors - The type of the input selectors.
 * @template Result - The type of the result returned by the `resultFunc`.
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 *
 * @public
 */
type OutputSelectorFields<InputSelectors extends SelectorArray = SelectorArray, Result = unknown, MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize> = {
    /** The final function passed to `createSelector`. Otherwise known as the `combiner`.*/
    resultFunc: Combiner<InputSelectors, Result>;
    /** The memoized version of {@linkcode OutputSelectorFields.resultFunc resultFunc}. */
    memoizedResultFunc: Combiner<InputSelectors, Result> & ExtractMemoizerFields<MemoizeFunction>;
    /** Returns the last result calculated by the output selector. */
    lastResult: () => Result;
    /** An array of the input selectors. */
    dependencies: InputSelectors;
    /** Counts the number of times the output has been recalculated. */
    recomputations: () => number;
    /** Resets the count of `recomputations` count to 0. */
    resetRecomputations: () => 0;
} & Simplify<Required<Pick<CreateSelectorOptions<MemoizeFunction, ArgsMemoizeFunction>, 'argsMemoize' | 'memoize'>>>;
/**
 * Represents the actual selectors generated by `createSelector`.
 *
 * @template InputSelectors - The type of the input selectors.
 * @template Result - The type of the result returned by the `resultFunc`.
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 *
 * @public
 */
type OutputSelector<InputSelectors extends SelectorArray = SelectorArray, Result = unknown, MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize> = Selector<GetStateFromSelectors<InputSelectors>, Result, GetParamsFromSelectors<InputSelectors>> & ExtractMemoizerFields<ArgsMemoizeFunction> & OutputSelectorFields<InputSelectors, Result, MemoizeFunction, ArgsMemoizeFunction>;
/**
 * A function that takes input selectors' return values as arguments and returns a result. Otherwise known as `resultFunc`.
 *
 * @template InputSelectors - An array of input selectors.
 * @template Result - Result returned by `resultFunc`.
 *
 * @public
 */
type Combiner<InputSelectors extends SelectorArray, Result> = Distribute<
/**
 * A function that takes input selectors' return values as arguments and returns a result. Otherwise known as `resultFunc`.
 *
 * @param resultFuncArgs - Return values of input selectors.
 * @returns The return value of {@linkcode OutputSelectorFields.resultFunc resultFunc}.
 */
(...resultFuncArgs: SelectorResultArray<InputSelectors>) => Result>;
/**
 * A selector that is assumed to have one additional argument, such as
 * the props from a React component.
 *
 * @public
 */
type ParametricSelector<State, Props, Result> = Selector<State, Result, [
    Props,
    ...any
]>;
/**
 * A generated selector that is assumed to have one additional argument.
 *
 * @public
 */
type OutputParametricSelector<State, Props, Result> = ParametricSelector<State, Props, Result> & OutputSelectorFields<SelectorArray, Result>;
/**
 * A standard function returning true if two values are considered equal.
 *
 * @public
 */
type EqualityFn = (a: any, b: any) => boolean;
/**
 * The frequency of input stability checks.
 *
 * @since 5.0.0
 * @public
 */
type StabilityCheckFrequency = 'always' | 'once' | 'never';
/**
 * Determines the combined single "State" type (first arg) from all input selectors.
 *
 * @public
 */
type GetStateFromSelectors<Selectors extends SelectorArray> = MergeParameters<Selectors>[0];
/**
 * Determines the combined  "Params" type (all remaining args) from all input selectors.
 *
 * @public
 */
type GetParamsFromSelectors<Selectors extends SelectorArray> = ArrayTail<MergeParameters<Selectors>>;
/**
 * Any function with any arguments.
 *
 * @internal
 */
type AnyFunction = (...args: any[]) => any;
/**
 * Any function with unknown arguments.
 *
 * @internal
 */
type UnknownFunction = (...args: unknown[]) => unknown;
/**
 * Any Memoizer function. A memoizer is a function that accepts another function and returns it.
 *
 * @template FunctionType - The type of the function that is memoized.
 *
 * @internal
 */
type UnknownMemoizer<FunctionType extends UnknownFunction = UnknownFunction> = (func: FunctionType, ...options: any[]) => FunctionType;
/**
 * When a generic type parameter is using its default value of `never`, fallback to a different type.
 *
 * @template T - Type to be checked.
 * @template FallbackTo - Type to fallback to if `T` resolves to `never`.
 *
 * @internal
 */
type FallbackIfNever<T, FallbackTo> = IfNever<T, FallbackTo, T>;
/**
 * Derive the type of memoize options object based on whether the memoize function itself was overridden.
 *
 * _This type can be used for both `memoizeOptions` and `argsMemoizeOptions`._
 *
 * @template MemoizeFunction - The type of the `memoize` or `argsMemoize` function initially passed into `createSelectorCreator`.
 * @template OverrideMemoizeFunction - The type of the optional `memoize` or `argsMemoize` function passed directly into `createSelector` which then overrides the original `memoize` or `argsMemoize` function passed into `createSelectorCreator`.
 *
 * @internal
 */
type OverrideMemoizeOptions<MemoizeFunction extends UnknownMemoizer, OverrideMemoizeFunction extends UnknownMemoizer = never> = IfNever<OverrideMemoizeFunction, MemoizeOptionsFromParameters<MemoizeFunction>, MemoizeOptionsFromParameters<OverrideMemoizeFunction>>;
/**
 * Extracts the non-function part of a type.
 *
 * @template T - The input type to be refined by excluding function types and index signatures.
 *
 * @internal
 */
type NonFunctionType<T> = OmitIndexSignature<Exclude<T, AnyFunction>>;
/**
 * Extracts the function part of a type.
 *
 * @template T - The input type to be refined by extracting function types.
 *
 * @internal
 */
type FunctionType<T> = Extract<T, AnyFunction>;
/**
 * Extracts the options type for a memoization function based on its parameters.
 * The first parameter of the function is expected to be the function to be memoized,
 * followed by options for the memoization process.
 *
 * @template MemoizeFunction - The type of the memoize function to be checked.
 *
 * @internal
 */
type MemoizeOptionsFromParameters<MemoizeFunction extends UnknownMemoizer> = (Simplify<NonFunctionType<DropFirstParameter<MemoizeFunction>[0]>> | FunctionType<DropFirstParameter<MemoizeFunction>[0]>) | (Simplify<NonFunctionType<DropFirstParameter<MemoizeFunction>[number]>> | FunctionType<DropFirstParameter<MemoizeFunction>[number]>)[];
/**
 * Extracts the additional fields that a memoize function attaches to
 * the function it memoizes (e.g., `clearCache`).
 *
 * @template MemoizeFunction - The type of the memoize function to be checked.
 *
 * @internal
 */
type ExtractMemoizerFields<MemoizeFunction extends UnknownMemoizer> = Simplify<OmitIndexSignature<ReturnType<MemoizeFunction>>>;
/**
 * Extracts the return type from all functions as a tuple.
 *
 * @internal
 */
type ExtractReturnType<FunctionsArray extends readonly AnyFunction[]> = {
    [Index in keyof FunctionsArray]: FunctionsArray[Index] extends FunctionsArray[number] ? ReturnType<FunctionsArray[Index]> : never;
};
/**
 * Utility type to infer the type of "all params of a function except the first",
 * so we can determine what arguments a memoize function accepts.
 *
 * @internal
 */
type DropFirstParameter<Func extends AnyFunction> = Func extends (firstArg: any, ...restArgs: infer Rest) => any ? Rest : never;
/**
 * Distributes over a type. It is used mostly to expand a function type
 * in hover previews while preserving their original JSDoc information.
 *
 * If preserving JSDoc information is not a concern, you can use {@linkcode ExpandFunction ExpandFunction}.
 *
 * @template T The type to be distributed.
 *
 * @internal
 */
type Distribute<T> = T extends T ? T : never;
/**
 * Extracts the type of an array or tuple minus the first element.
 *
 * @internal
 */
type ArrayTail<TArray> = TArray extends readonly [
    unknown,
    ...infer TTail
] ? TTail : [];
/**
 * An alias for type `{}`. Represents any value that is not `null` or `undefined`.
 * It is mostly used for semantic purposes to help distinguish between an
 * empty object type and `{}` as they are not the same.
 *
 * @internal
 */
type AnyNonNullishValue = NonNullable<unknown>;
/**
 * Same as {@linkcode AnyNonNullishValue AnyNonNullishValue} but aliased
 * for semantic purposes. It is intended to be used in scenarios where
 * a recursive type definition needs to be interrupted to ensure type safety
 * and to avoid excessively deep recursion that could lead to performance issues.
 *
 * @internal
 */
type InterruptRecursion = AnyNonNullishValue;
/**
 * An if-else-like type that resolves depending on whether the given type is `never`.
 * This is mainly used to conditionally resolve the type of a `memoizeOptions` object based on whether `memoize` is provided or not.
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/if-never.d.ts Source}
 *
 * @internal
 */
type IfNever<T, TypeIfNever, TypeIfNotNever> = [T] extends [never] ? TypeIfNever : TypeIfNotNever;
/**
 * Omit any index signatures from the given object type, leaving only explicitly defined properties.
 * This is mainly used to remove explicit `any`s from the return type of some memoizers (e.g, `microMemoize`).
 *
 * __Disclaimer:__ When used on an intersection of a function and an object,
 * the function is erased.
 *
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/omit-index-signature.d.ts Source}
 *
 * @internal
 */
type OmitIndexSignature<ObjectType> = {
    [KeyType in keyof ObjectType as {} extends Record<KeyType, unknown> ? never : KeyType]: ObjectType[KeyType];
};
/**
 * The infamous "convert a union type to an intersection type" hack
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/union-to-intersection.d.ts Source}
 * @see {@link https://github.com/microsoft/TypeScript/issues/29594 Reference}
 *
 * @internal
 */
type UnionToIntersection<Union> = (Union extends unknown ? (distributedUnion: Union) => void : never) extends (mergedIntersection: infer Intersection) => void ? Intersection : never;
/**
 * Code to convert a union of values into a tuple.
 * @see {@link https://stackoverflow.com/a/55128956/62937 Source}
 *
 * @internal
 */
type Push<T extends any[], V> = [...T, V];
/**
 * @see {@link https://stackoverflow.com/a/55128956/62937 Source}
 *
 * @internal
 */
type LastOf<T> = UnionToIntersection<T extends any ? () => T : never> extends () => infer R ? R : never;
/**
 * TS4.1+
 * @see {@link https://stackoverflow.com/a/55128956/62937 Source}
 *
 * @internal
 */
type TuplifyUnion<T, L = LastOf<T>, N = [T] extends [never] ? true : false> = true extends N ? [] : Push<TuplifyUnion<Exclude<T, L>>, L>;
/**
 * Converts "the values of an object" into a tuple, like a type-level `Object.values()`
 * @see {@link https://stackoverflow.com/a/68695508/62937 Source}
 *
 * @internal
 */
type ObjectValuesToTuple<T, KS extends any[] = TuplifyUnion<keyof T>, R extends any[] = []> = KS extends [infer K, ...infer KT] ? ObjectValuesToTuple<T, KT, [...R, T[K & keyof T]]> : R;
/**
 * Useful to flatten the type output to improve type hints shown in editors.
 * And also to transform an interface into a type to aide with assignability.
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/simplify.d.ts Source}
 *
 * @internal
 */
type Simplify<T> = {
    [KeyType in keyof T]: T[KeyType];
} & AnyNonNullishValue;

/**
 * Uses an "auto-tracking" approach inspired by the work of the Ember Glimmer team.
 * It uses a Proxy to wrap arguments and track accesses to nested fields
 * in your selector on first read. Later, when the selector is called with
 * new arguments, it identifies which accessed fields have changed and
 * only recalculates the result if one or more of those accessed fields have changed.
 * This allows it to be more precise than the shallow equality checks in `defaultMemoize`.
 *
 * __Design Tradeoffs for `autotrackMemoize`:__
 * - Pros:
 *    - It is likely to avoid excess calculations and recalculate fewer times than `defaultMemoize` will,
 *    which may also result in fewer component re-renders.
 * - Cons:
 *    - It only has a cache size of 1.
 *    - It is slower than `defaultMemoize`, because it has to do more work. (How much slower is dependent on the number of accessed fields in a selector, number of calls, frequency of input changes, etc)
 *    - It can have some unexpected behavior. Because it tracks nested field accesses,
 *    cases where you don't access a field will not recalculate properly.
 *    For example, a badly-written selector like:
 *      ```ts
 *      createSelector([state => state.todos], todos => todos)
 *      ```
 *      that just immediately returns the extracted value will never update, because it doesn't see any field accesses to check.
 *
 * __Use Cases for `autotrackMemoize`:__
 * - It is likely best used for cases where you need to access specific nested fields
 * in data, and avoid recalculating if other fields in the same data objects are immutably updated.
 *
 * @param func - The function to be memoized.
 * @returns A memoized function with a `.clearCache()` method attached.
 *
 * @example
 * <caption>Using `createSelector`</caption>
 * ```ts
 * import { unstable_autotrackMemoize as autotrackMemoize, createSelector } from 'reselect'
 *
 * const selectTodoIds = createSelector(
 *   [(state: RootState) => state.todos],
 *   (todos) => todos.map(todo => todo.id),
 *   { memoize: autotrackMemoize }
 * )
 * ```
 *
 * @example
 * <caption>Using `createSelectorCreator`</caption>
 * ```ts
 * import { unstable_autotrackMemoize as autotrackMemoize, createSelectorCreator } from 'reselect'
 *
 * const createSelectorAutotrack = createSelectorCreator(autotrackMemoize)
 *
 * const selectTodoIds = createSelectorAutotrack(
 *   [(state: RootState) => state.todos],
 *   (todos) => todos.map(todo => todo.id)
 * )
 * ```
 *
 * @template Func - The type of the function that is memoized.
 *
 * @since 5.0.0
 * @public
 * @experimental
 */
declare function autotrackMemoize<Func extends AnyFunction>(func: Func): Func & {
    clearCache: () => void;
};

/**
 * An instance of `createSelector`, customized with a given memoize implementation.
 *
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 *
 * @public
 */
interface CreateSelectorFunction<MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize> {
    /**
     * Creates a memoized selector function.
     *
     * @param createSelectorArgs - An arbitrary number of input selectors as separate inline arguments and a `combiner` function.
     * @returns A memoized output selector.
     *
     * @template InputSelectors - The type of the input selectors as an array.
     * @template Result - The return type of the `combiner` as well as the output selector.
     * @template OverrideMemoizeFunction - The type of the optional `memoize` function that could be passed into the options object to override the original `memoize` function that was initially passed into `createSelectorCreator`.
     * @template OverrideArgsMemoizeFunction - The type of the optional `argsMemoize` function that could be passed into the options object to override the original `argsMemoize` function that was initially passed into `createSelectorCreator`.
     *
     * @see {@link https://github.com/reduxjs/reselect#createselectorinputselectors--inputselectors-resultfunc-selectoroptions createSelector}
     */
    <InputSelectors extends SelectorArray, Result>(...createSelectorArgs: [
        ...inputSelectors: InputSelectors,
        combiner: Combiner<InputSelectors, Result>
    ]): OutputSelector<InputSelectors, Result, MemoizeFunction, ArgsMemoizeFunction> & InterruptRecursion;
    /**
     * Creates a memoized selector function.
     *
     * @param createSelectorArgs - An arbitrary number of input selectors as separate inline arguments, a `combiner` function and an `options` object.
     * @returns A memoized output selector.
     *
     * @template InputSelectors - The type of the input selectors as an array.
     * @template Result - The return type of the `combiner` as well as the output selector.
     * @template OverrideMemoizeFunction - The type of the optional `memoize` function that could be passed into the options object to override the original `memoize` function that was initially passed into `createSelectorCreator`.
     * @template OverrideArgsMemoizeFunction - The type of the optional `argsMemoize` function that could be passed into the options object to override the original `argsMemoize` function that was initially passed into `createSelectorCreator`.
     *
     * @see {@link https://github.com/reduxjs/reselect#createselectorinputselectors--inputselectors-resultfunc-selectoroptions createSelector}
     */
    <InputSelectors extends SelectorArray, Result, OverrideMemoizeFunction extends UnknownMemoizer = MemoizeFunction, OverrideArgsMemoizeFunction extends UnknownMemoizer = ArgsMemoizeFunction>(...createSelectorArgs: [
        ...inputSelectors: InputSelectors,
        combiner: Combiner<InputSelectors, Result>,
        createSelectorOptions: Partial<CreateSelectorOptions<MemoizeFunction, ArgsMemoizeFunction, OverrideMemoizeFunction, OverrideArgsMemoizeFunction>>
    ]): OutputSelector<InputSelectors, Result, OverrideMemoizeFunction, OverrideArgsMemoizeFunction> & InterruptRecursion;
    /**
     * Creates a memoized selector function.
     *
     * @param inputSelectors - An array of input selectors.
     * @param combiner - A function that Combines the input selectors and returns an output selector. Otherwise known as the result function.
     * @param createSelectorOptions - An optional options object that allows for further customization per selector.
     * @returns A memoized output selector.
     *
     * @template InputSelectors - The type of the input selectors array.
     * @template Result - The return type of the `combiner` as well as the output selector.
     * @template OverrideMemoizeFunction - The type of the optional `memoize` function that could be passed into the options object to override the original `memoize` function that was initially passed into `createSelectorCreator`.
     * @template OverrideArgsMemoizeFunction - The type of the optional `argsMemoize` function that could be passed into the options object to override the original `argsMemoize` function that was initially passed into `createSelectorCreator`.
     *
     * @see {@link https://github.com/reduxjs/reselect#createselectorinputselectors--inputselectors-resultfunc-selectoroptions createSelector}
     */
    <InputSelectors extends SelectorArray, Result, OverrideMemoizeFunction extends UnknownMemoizer = MemoizeFunction, OverrideArgsMemoizeFunction extends UnknownMemoizer = ArgsMemoizeFunction>(inputSelectors: [...InputSelectors], combiner: Combiner<InputSelectors, Result>, createSelectorOptions?: Partial<CreateSelectorOptions<MemoizeFunction, ArgsMemoizeFunction, OverrideMemoizeFunction, OverrideArgsMemoizeFunction>>): OutputSelector<InputSelectors, Result, OverrideMemoizeFunction, OverrideArgsMemoizeFunction> & InterruptRecursion;
}
/**
 * In development mode, an extra check is conducted on your input selectors.
 * It runs your input selectors an extra time with the same arguments, and
 * warns in the console if they return a different result _(based on your `memoize` method)_.
 *
 * This function allows you to override this setting for all of your selectors.
 *
 * **Note**: This setting can still be overridden per selector inside `createSelector`'s `options` object.
 * See {@link https://github.com/reduxjs/reselect#per-selector-configuration per-selector-configuration}
 * and {@linkcode CreateSelectorOptions.inputStabilityCheck inputStabilityCheck} for more details.
 *
 * _The input stability check does not run in production builds._
 *
 * @param inputStabilityCheckFrequency - How often the `inputStabilityCheck` should run for all selectors.
 *
 * @example
 * ```ts
 * import { setInputStabilityCheckEnabled } from 'reselect'
import { assert } from './autotrackMemoize/utils';
import { OutputSelectorFields, Mapped } from './types';
 *
 * // Run only the first time the selector is called. (default)
 * setInputStabilityCheckEnabled('once')
 *
 * // Run every time the selector is called.
 * setInputStabilityCheckEnabled('always')
 *
 * // Never run the input stability check.
 * setInputStabilityCheckEnabled('never')
 * ```
 * @see {@link https://github.com/reduxjs/reselect#development-only-checks development-only-checks}
 * @see {@link https://github.com/reduxjs/reselect#global-configuration global-configuration}
 *
 * @since 5.0.0
 * @public
 */
declare function setInputStabilityCheckEnabled(inputStabilityCheckFrequency: StabilityCheckFrequency): void;
/**
 * Creates a selector creator function with the specified memoization function and options for customizing memoization behavior.
 *
 * @param options - An options object containing the `memoize` function responsible for memoizing the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). It also provides additional options for customizing memoization. While the `memoize` property is mandatory, the rest are optional.
 * @returns A customized `createSelector` function.
 *
 * @example
 * ```ts
 * const customCreateSelector = createSelectorCreator({
 *   memoize: customMemoize, // Function to be used to memoize `resultFunc`
 *   memoizeOptions: [memoizeOption1, memoizeOption2], // Options passed to `customMemoize` as the second argument onwards
 *   argsMemoize: customArgsMemoize, // Function to be used to memoize the selector's arguments
 *   argsMemoizeOptions: [argsMemoizeOption1, argsMemoizeOption2] // Options passed to `customArgsMemoize` as the second argument onwards
 * })
 *
 * const customSelector = customCreateSelector(
 *   [inputSelector1, inputSelector2],
 *   resultFunc // `resultFunc` will be passed as the first argument to `customMemoize`
 * )
 *
 * customSelector(
 *   ...selectorArgs // Will be memoized by `customArgsMemoize`
 * )
 * ```
 *
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 *
 * @see {@link https://github.com/reduxjs/reselect#createselectorcreatormemoize-memoizeoptions createSelectorCreator}
 *
 * @since 5.0.0
 * @public
 */
declare function createSelectorCreator<MemoizeFunction extends UnknownMemoizer, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize>(options: CreateSelectorOptions<typeof defaultMemoize, typeof defaultMemoize, MemoizeFunction, ArgsMemoizeFunction>): CreateSelectorFunction<MemoizeFunction, ArgsMemoizeFunction>;
/**
 * Creates a selector creator function with the specified memoization function and options for customizing memoization behavior.
 *
 * @param memoize - The `memoize` function responsible for memoizing the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @param memoizeOptionsFromArgs - Optional configuration options for the memoization function. These options are then passed to the memoize function as the second argument onwards.
 * @returns A customized `createSelector` function.
 *
 * @example
 * ```ts
 * const customCreateSelector = createSelectorCreator(customMemoize, // Function to be used to memoize `resultFunc`
 *   option1, // Will be passed as second argument to `customMemoize`
 *   option2, // Will be passed as third argument to `customMemoize`
 *   option3 // Will be passed as fourth argument to `customMemoize`
 * )
 *
 * const customSelector = customCreateSelector(
 *   [inputSelector1, inputSelector2],
 *   resultFunc // `resultFunc` will be passed as the first argument to `customMemoize`
 * )
 * ```
 *
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 *
 * @see {@link https://github.com/reduxjs/reselect#createselectorcreatormemoize-memoizeoptions createSelectorCreator}
 *
 * @public
 */
declare function createSelectorCreator<MemoizeFunction extends UnknownMemoizer>(memoize: MemoizeFunction, ...memoizeOptionsFromArgs: DropFirstParameter<MemoizeFunction>): CreateSelectorFunction<MemoizeFunction>;
/**
 * Accepts one or more "input selectors" (either as separate arguments or a single array),
 * a single "result function" / "combiner", and an optional options object, and
 * generates a memoized selector function.
 *
 * @see {@link https://github.com/reduxjs/reselect#createselectorinputselectors--inputselectors-resultfunc-selectoroptions createSelector}
 *
 * @public
 */
declare const createSelector: CreateSelectorFunction<typeof defaultMemoize, typeof defaultMemoize>;

/**
 *
 * @WIP
 */
type SelectorsMap<T extends SelectorsObject> = {
    [Key in keyof T]: ReturnType<T[Key]>;
};
/**
 * Allows you to create a pre-typed version of {@linkcode createStructuredSelector createStructuredSelector}
 * For your root state.
 *
 * @since 5.0.0
 * @public
 * @WIP
 */
interface TypedStructuredSelectorCreator<RootState = any> {
    <InputSelectorsObject extends {
        [Key in keyof RootState]: Selector<RootState, RootState[Key], []>;
    } = {
        [Key in keyof RootState]: Selector<RootState, RootState[Key], []>;
    }, MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize>(selectors: InputSelectorsObject, selectorCreator?: CreateSelectorFunction<MemoizeFunction, ArgsMemoizeFunction>): OutputSelector<ObjectValuesToTuple<InputSelectorsObject>, SelectorsMap<InputSelectorsObject>, MemoizeFunction, ArgsMemoizeFunction>;
}
interface SelectorsObject {
    [key: string]: Selector;
}
/**
 * It provides a way to create structured selectors.
 * The structured selector can take multiple input selectors
 * and map their output to an object with specific keys.
 *
 * @see {@link https://github.com/reduxjs/reselect#createstructuredselectorinputselectors-selectorcreator--createselector createStructuredSelector}
 *
 * @public
 */
interface StructuredSelectorCreator {
    /**
     * A convenience function for a common pattern that arises when using Reselect.
     * The selector passed to a `connect` decorator often just takes the
     * values of its input selectors and maps them to keys in an object.
     *
     * @param selectorMap - A key value pair consisting of input selectors.
     * @param selectorCreator - A custom selector creator function. It defaults to `createSelector`.
     * @returns A memoized structured selector.
     *
     * @example
     * <caption>Modern Use Case</caption>
     * ```ts
     * import { createSelector, createStructuredSelector } from 'reselect'
     *
     * interface State {
     *   todos: {
     *     id: number
     *     title: string
     *     description: string
     *     completed: boolean
     *   }[]
     *   alerts: {
     *     id: number
     *     message: string
     *     type: 'reminder' | 'notification'
     *     read: boolean
     *   }[]
     * }
     *
     * const state: State = {
     *   todos: [
     *     {
     *       id: 0,
     *       title: 'Buy groceries',
     *       description: 'Milk, bread, eggs, and fruits',
     *       completed: false
     *     },
     *     {
     *       id: 1,
     *       title: 'Schedule dentist appointment',
     *       description: 'Check available slots for next week',
     *       completed: true
     *     }
     *   ],
     *   alerts: [
     *     {
     *       id: 0,
     *       message: 'You have an upcoming meeting at 3 PM.',
     *       type: 'reminder',
     *       read: false
     *     },
     *     {
     *       id: 1,
     *       message: 'New software update available.',
     *       type: 'notification',
     *       read: true
     *     }
     *   ]
     * }
     *
     * // This:
     * const structuredSelector = createStructuredSelector(
     *   {
     *     allTodos: (state: State) => state.todos,
     *     allAlerts: (state: State) => state.alerts,
     *     selectedTodo: (state: State, id: number) => state.todos[id]
     *   },
     *   createSelector
     * )
     *
     * // Is essentially the same as this:
     * const selector = createSelector(
     *   [
     *     (state: State) => state.todos,
     *     (state: State) => state.alerts,
     *     (state: State, id: number) => state.todos[id]
     *   ],
     *   (allTodos, allAlerts, selectedTodo) => {
     *     return {
     *       allTodos,
     *       allAlerts,
     *       selectedTodo
     *     }
     *   }
     * )
     * ```
     *
     * @example
     * <caption>Simple Use Case</caption>
     * ```ts
     * const selectA = state => state.a
     * const selectB = state => state.b
     *
     * // The result function in the following selector
     * // is simply building an object from the input selectors
     * const structuredSelector = createSelector(selectA, selectB, (a, b) => ({
     *   a,
     *   b
     * }))
     *
     * const result = structuredSelector({ a: 1, b: 2 }) // will produce { x: 1, y: 2 }
     * ```
     *
     * @template InputSelectorsObject - The shape of the input selectors object.
     * @template MemoizeFunction - The type of the memoize function that is used to create the structured selector. It defaults to `defaultMemoize`.
     * @template ArgsMemoizeFunction - The type of the of the memoize function that is used to memoize the arguments passed into the generated structured selector. It defaults to `defaultMemoize`.
     *
     * @see {@link https://github.com/reduxjs/reselect#createstructuredselectorinputselectors-selectorcreator--createselector createStructuredSelector}
     */
    <InputSelectorsObject extends SelectorsObject, MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize, ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize>(selectorMap: InputSelectorsObject, selectorCreator?: CreateSelectorFunction<MemoizeFunction, ArgsMemoizeFunction>): OutputSelector<ObjectValuesToTuple<InputSelectorsObject>, SelectorsMap<InputSelectorsObject>, MemoizeFunction, ArgsMemoizeFunction>;
}
/**
 * A convenience function for a common pattern that arises when using Reselect.
 * The selector passed to a `connect` decorator often just takes the values of its input selectors
 * and maps them to keys in an object.
 *
 * @example
 * <caption>Simple Use Case</caption>
 * ```ts
 * const selectA = state => state.a
 * const selectB = state => state.b
 *
 * // The result function in the following selector
 * // is simply building an object from the input selectors
 * const structuredSelector = createSelector(selectA, selectB, (a, b) => ({
 *   a,
 *   b
 * }))
 * ```
 *
 * @see {@link https://github.com/reduxjs/reselect#createstructuredselectorinputselectors-selectorcreator--createselector createStructuredSelector}
 *
 * @public
 */
declare const createStructuredSelector: StructuredSelectorCreator;

/**
 * Creates a tree of `WeakMap`-based cache nodes based on the identity of the
 * arguments it's been called with (in this case, the extracted values from your input selectors).
 * This allows `weakmapMemoize` to have an effectively infinite cache size.
 * Cache results will be kept in memory as long as references to the arguments still exist,
 * and then cleared out as the arguments are garbage-collected.
 *
 * __Design Tradeoffs for `weakmapMemoize`:__
 * - Pros:
 *   - It has an effectively infinite cache size, but you have no control over
 *   how long values are kept in cache as it's based on garbage collection and `WeakMap`s.
 * - Cons:
 *   - There's currently no way to alter the argument comparisons.
 *   They're based on strict reference equality.
 *   - It's roughly the same speed as `defaultMemoize`, although likely a fraction slower.
 *
 * __Use Cases for `weakmapMemoize`:__
 * - This memoizer is likely best used for cases where you need to call the
 * same selector instance with many different arguments, such as a single
 * selector instance that is used in a list item component and called with
 * item IDs like:
 *   ```ts
 *   useSelector(state => selectSomeData(state, props.category))
 *   ```
 * @param func - The function to be memoized.
 * @returns A memoized function with a `.clearCache()` method attached.
 *
 * @example
 * <caption>Using `createSelector`</caption>
 * ```ts
 * import { createSelector, weakMapMemoize } from 'reselect'
 *
 * const selectTodoById = createSelector(
 *   [
 *     (state: RootState) => state.todos,
 *     (state: RootState, id: number) => id
 *   ],
 *   (todos) => todos[id],
 *   { memoize: weakMapMemoize }
 * )
 * ```
 *
 * @example
 * <caption>Using `createSelectorCreator`</caption>
 * ```ts
 * import { createSelectorCreator, weakMapMemoize } from 'reselect'
 *
 * const createSelectorWeakmap = createSelectorCreator(weakMapMemoize)
 *
 * const selectTodoById = createSelectorWeakmap(
 *   [
 *     (state: RootState) => state.todos,
 *     (state: RootState, id: number) => id
 *   ],
 *   (todos) => todos[id]
 * )
 * ```
 *
 * @template Func - The type of the function that is memoized.
 *
 * @since 5.0.0
 * @public
 * @experimental
 */
declare function weakMapMemoize<Func extends AnyFunction>(func: Func): Func & {
    clearCache: () => void;
};

export { Combiner, CreateSelectorFunction, CreateSelectorOptions, DefaultMemoizeOptions, EqualityFn, GetParamsFromSelectors, GetStateFromSelectors, OutputParametricSelector, OutputSelector, OutputSelectorFields, ParametricSelector, Selector, SelectorArray, SelectorResultArray, StabilityCheckFrequency, StructuredSelectorCreator, TypedStructuredSelectorCreator, createSelector, createSelectorCreator, createStructuredSelector, defaultEqualityCheck, defaultMemoize, setInputStabilityCheckEnabled, autotrackMemoize as unstable_autotrackMemoize, weakMapMemoize };
