import type { defaultMemoize } from './defaultMemoize'
import type { MergeParameters } from './versionedTypes'

export type { MergeParameters } from './versionedTypes'

/*
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 *
 * Reselect Data Types
 *
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 */

/**
 * A standard selector function.
 * @template State - The first value, often a Redux root state object.
 * @template Result - The final result returned by the selector.
 * @template Params - All additional arguments passed into the selector.
 *
 * @public
 */
export type Selector<
  State = any,
  Result = unknown,
  Params extends readonly any[] = any[]
> = Distribute<
  /**
   * A function that takes a state and returns data that is based on that state.
   *
   * @param state - The first argument, often a Redux root state object.
   * @param params - All additional arguments passed into the selector.
   * @returns A derived value from the state.
   */
  (state: State, ...params: FallbackIfNever<Params, []>) => Result
>

/**
 * An array of input selectors.
 *
 * @public
 */
export type SelectorArray = readonly Selector[]

/**
 * Extracts an array of all return types from all input selectors.
 *
 * @public
 */
export type SelectorResultArray<Selectors extends SelectorArray> =
  ExtractReturnType<Selectors>

/**
 * The options object used inside `createSelector` and `createSelectorCreator`.
 *
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 * @template OverrideMemoizeFunction - The type of the optional `memoize` function that could be passed into the options object inside `createSelector` to override the original `memoize` function that was initially passed into `createSelectorCreator`.
 * @template OverrideArgsMemoizeFunction - The type of the optional `argsMemoize` function that could be passed into the options object inside `createSelector` to override the original `argsMemoize` function that was initially passed into `createSelectorCreator`. If none was initially provided, `defaultMemoize` will be used.
 *
 * @public
 */
export interface CreateSelectorOptions<
  MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize,
  ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize,
  OverrideMemoizeFunction extends UnknownMemoizer = never,
  OverrideArgsMemoizeFunction extends UnknownMemoizer = never
> {
  /**
   * Overrides the global input stability check for the selector.
   * - `once` - Run only the first time the selector is called.
   * - `always` - Run every time the selector is called.
   * - `never` - Never run the input stability check.
   *
   * @default 'once'
   *
   * @see {@link https://github.com/reduxjs/reselect#development-only-checks development-only-checks}
   * @see {@link https://github.com/reduxjs/reselect#inputstabilitycheck inputStabilityCheck}
   * @see {@link https://github.com/reduxjs/reselect#per-selector-configuration per-selector-configuration}
   *
   * @since 5.0.0
   */
  inputStabilityCheck?: StabilityCheckFrequency

  /**
   * The memoize function that is used to memoize the {@linkcode OutputSelectorFields.resultFunc resultFunc}
   * inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
   *
   * When passed directly into `createSelector`, it overrides the `memoize` function initially passed into `createSelectorCreator`.
   *
   * @example
   * ```ts
   * import { createSelector, weakMapMemoize } from 'reselect'
   *
   * const selectTodoById = createSelector(
   *   [
   *     (state: RootState) => state.todos,
   *     (state: RootState, id: number) => id
   *   ],
   *   (todos) => todos[id],
   *   { memoize: weakMapMemoize }
   * )
   * ```
   *
   * @since 5.0.0
   */
  // If `memoize` is not provided inside the options object, fallback to `MemoizeFunction` which is the original memoize function passed into `createSelectorCreator`.
  memoize: FallbackIfNever<OverrideMemoizeFunction, MemoizeFunction>

  /**
   * The optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
   *
   * When passed directly into `createSelector`, it overrides the `argsMemoize` function initially passed into `createSelectorCreator`. If none was initially provided, `defaultMemoize` will be used.
   *
   * @example
   * ```ts
   * import { createSelector, weakMapMemoize } from 'reselect'
   *
   * const selectTodoById = createSelector(
   *   [
   *     (state: RootState) => state.todos,
   *     (state: RootState, id: number) => id
   *   ],
   *   (todos) => todos[id],
   *   { argsMemoize: weakMapMemoize }
   * )
   * ```
   *
   * @default defaultMemoize
   *
   * @since 5.0.0
   */
  // If `argsMemoize` is not provided inside the options object,
  // fallback to `ArgsMemoizeFunction` which is the original `argsMemoize` function passed into `createSelectorCreator`.
  // If none was passed originally to `createSelectorCreator`, it should fallback to `defaultMemoize`.
  argsMemoize?: FallbackIfNever<
    OverrideArgsMemoizeFunction,
    ArgsMemoizeFunction
  >

  /**
   * Optional configuration options for the {@linkcode CreateSelectorOptions.memoize memoize} function.
   * These options are passed to the {@linkcode CreateSelectorOptions.memoize memoize} function as the second argument.
   *
   * @since 5.0.0
   */
  // Should dynamically change to the options argument of `memoize`.
  memoizeOptions?: OverrideMemoizeOptions<
    MemoizeFunction,
    OverrideMemoizeFunction
  >

  /**
   * Optional configuration options for the {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} function.
   * These options are passed to the {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} function as the second argument.
   *
   * @since 5.0.0
   */
  argsMemoizeOptions?: OverrideMemoizeOptions<
    ArgsMemoizeFunction,
    OverrideArgsMemoizeFunction
  >
}

/**
 * The additional fields attached to the output selector generated by `createSelector`.
 *
 * **Note**: Although {@linkcode CreateSelectorOptions.memoize memoize}
 * and {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} are included in the attached fields,
 * the fields themselves are independent of the type of
 * {@linkcode CreateSelectorOptions.memoize memoize} and {@linkcode CreateSelectorOptions.argsMemoize argsMemoize} functions.
 * Meaning this type is not going to generate additional fields based on what functions we use to memoize our selectors.
 *
 * _This type is not to be confused with {@linkcode ExtractMemoizerFields ExtractMemoizerFields}._
 *
 * @template InputSelectors - The type of the input selectors.
 * @template Result - The type of the result returned by the `resultFunc`.
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 *
 * @public
 */
export type OutputSelectorFields<
  InputSelectors extends SelectorArray = SelectorArray,
  Result = unknown,
  MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize,
  ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize
> = {
  /** The final function passed to `createSelector`. Otherwise known as the `combiner`.*/
  resultFunc: Combiner<InputSelectors, Result>
  /** The memoized version of {@linkcode OutputSelectorFields.resultFunc resultFunc}. */
  memoizedResultFunc: Combiner<InputSelectors, Result> &
    ExtractMemoizerFields<MemoizeFunction>
  /** Returns the last result calculated by the output selector. */
  lastResult: () => Result
  /** An array of the input selectors. */
  dependencies: InputSelectors
  /** Counts the number of times the output has been recalculated. */
  recomputations: () => number
  /** Resets the count of `recomputations` count to 0. */
  resetRecomputations: () => 0
} & Simplify<
  Required<
    Pick<
      CreateSelectorOptions<MemoizeFunction, ArgsMemoizeFunction>,
      'argsMemoize' | 'memoize'
    >
  >
>

/**
 * Represents the actual selectors generated by `createSelector`.
 *
 * @template InputSelectors - The type of the input selectors.
 * @template Result - The type of the result returned by the `resultFunc`.
 * @template MemoizeFunction - The type of the memoize function that is used to memoize the `resultFunc` inside `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`).
 * @template ArgsMemoizeFunction - The type of the optional memoize function that is used to memoize the arguments passed into the output selector generated by `createSelector` (e.g., `defaultMemoize` or `weakMapMemoize`). If none is explicitly provided, `defaultMemoize` will be used.
 *
 * @public
 */
export type OutputSelector<
  InputSelectors extends SelectorArray = SelectorArray,
  Result = unknown,
  MemoizeFunction extends UnknownMemoizer = typeof defaultMemoize,
  ArgsMemoizeFunction extends UnknownMemoizer = typeof defaultMemoize
> = Selector<
  GetStateFromSelectors<InputSelectors>,
  Result,
  GetParamsFromSelectors<InputSelectors>
> &
  ExtractMemoizerFields<ArgsMemoizeFunction> &
  OutputSelectorFields<
    InputSelectors,
    Result,
    MemoizeFunction,
    ArgsMemoizeFunction
  >

/**
 * A function that takes input selectors' return values as arguments and returns a result. Otherwise known as `resultFunc`.
 *
 * @template InputSelectors - An array of input selectors.
 * @template Result - Result returned by `resultFunc`.
 *
 * @public
 */
export type Combiner<InputSelectors extends SelectorArray, Result> = Distribute<
  /**
   * A function that takes input selectors' return values as arguments and returns a result. Otherwise known as `resultFunc`.
   *
   * @param resultFuncArgs - Return values of input selectors.
   * @returns The return value of {@linkcode OutputSelectorFields.resultFunc resultFunc}.
   */
  (...resultFuncArgs: SelectorResultArray<InputSelectors>) => Result
>

/**
 * A selector that is assumed to have one additional argument, such as
 * the props from a React component.
 *
 * @public
 */
export type ParametricSelector<State, Props, Result> = Selector<
  State,
  Result,
  [Props, ...any]
>

/**
 * A generated selector that is assumed to have one additional argument.
 *
 * @public
 */
export type OutputParametricSelector<State, Props, Result> = ParametricSelector<
  State,
  Props,
  Result
> &
  OutputSelectorFields<SelectorArray, Result>

/**
 * A standard function returning true if two values are considered equal.
 *
 * @public
 */
export type EqualityFn = (a: any, b: any) => boolean

/**
 * The frequency of input stability checks.
 *
 * @since 5.0.0
 * @public
 */
export type StabilityCheckFrequency = 'always' | 'once' | 'never'

/**
 * Determines the combined single "State" type (first arg) from all input selectors.
 *
 * @public
 */
export type GetStateFromSelectors<Selectors extends SelectorArray> =
  MergeParameters<Selectors>[0]

/**
 * Determines the combined  "Params" type (all remaining args) from all input selectors.
 *
 * @public
 */
export type GetParamsFromSelectors<Selectors extends SelectorArray> = ArrayTail<
  MergeParameters<Selectors>
>

/*
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 *
 * Reselect Internal Utility Types
 *
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 */

/**
 * Any function with any arguments.
 *
 * @internal
 */
export type AnyFunction = (...args: any[]) => any

/**
 * Any function with unknown arguments.
 *
 * @internal
 */
export type UnknownFunction = (...args: unknown[]) => unknown

/**
 * Any Memoizer function. A memoizer is a function that accepts another function and returns it.
 *
 * @template FunctionType - The type of the function that is memoized.
 *
 * @internal
 */
export type UnknownMemoizer<
  FunctionType extends UnknownFunction = UnknownFunction
> = (func: FunctionType, ...options: any[]) => FunctionType

/**
 * When a generic type parameter is using its default value of `never`, fallback to a different type.
 *
 * @template T - Type to be checked.
 * @template FallbackTo - Type to fallback to if `T` resolves to `never`.
 *
 * @internal
 */
export type FallbackIfNever<T, FallbackTo> = IfNever<T, FallbackTo, T>

/**
 * Derive the type of memoize options object based on whether the memoize function itself was overridden.
 *
 * _This type can be used for both `memoizeOptions` and `argsMemoizeOptions`._
 *
 * @template MemoizeFunction - The type of the `memoize` or `argsMemoize` function initially passed into `createSelectorCreator`.
 * @template OverrideMemoizeFunction - The type of the optional `memoize` or `argsMemoize` function passed directly into `createSelector` which then overrides the original `memoize` or `argsMemoize` function passed into `createSelectorCreator`.
 *
 * @internal
 */
export type OverrideMemoizeOptions<
  MemoizeFunction extends UnknownMemoizer,
  OverrideMemoizeFunction extends UnknownMemoizer = never
> = IfNever<
  OverrideMemoizeFunction,
  MemoizeOptionsFromParameters<MemoizeFunction>,
  MemoizeOptionsFromParameters<OverrideMemoizeFunction>
>

/**
 * Extracts the non-function part of a type.
 *
 * @template T - The input type to be refined by excluding function types and index signatures.
 *
 * @internal
 */
export type NonFunctionType<T> = OmitIndexSignature<Exclude<T, AnyFunction>>

/**
 * Extracts the function part of a type.
 *
 * @template T - The input type to be refined by extracting function types.
 *
 * @internal
 */
export type FunctionType<T> = Extract<T, AnyFunction>

/**
 * Extracts the options type for a memoization function based on its parameters.
 * The first parameter of the function is expected to be the function to be memoized,
 * followed by options for the memoization process.
 *
 * @template MemoizeFunction - The type of the memoize function to be checked.
 *
 * @internal
 */
export type MemoizeOptionsFromParameters<
  MemoizeFunction extends UnknownMemoizer
> =
  | (
      | Simplify<NonFunctionType<DropFirstParameter<MemoizeFunction>[0]>>
      | FunctionType<DropFirstParameter<MemoizeFunction>[0]>
    )
  | (
      | Simplify<NonFunctionType<DropFirstParameter<MemoizeFunction>[number]>>
      | FunctionType<DropFirstParameter<MemoizeFunction>[number]>
    )[]

/**
 * Extracts the additional fields that a memoize function attaches to
 * the function it memoizes (e.g., `clearCache`).
 *
 * @template MemoizeFunction - The type of the memoize function to be checked.
 *
 * @internal
 */
export type ExtractMemoizerFields<MemoizeFunction extends UnknownMemoizer> =
  Simplify<OmitIndexSignature<ReturnType<MemoizeFunction>>>

/**
 * Extracts the return type from all functions as a tuple.
 *
 * @internal
 */
export type ExtractReturnType<FunctionsArray extends readonly AnyFunction[]> = {
  [Index in keyof FunctionsArray]: FunctionsArray[Index] extends FunctionsArray[number]
    ? ReturnType<FunctionsArray[Index]>
    : never
}

/**
 * Utility type to infer the type of "all params of a function except the first",
 * so we can determine what arguments a memoize function accepts.
 *
 * @internal
 */
export type DropFirstParameter<Func extends AnyFunction> = Func extends (
  firstArg: any,
  ...restArgs: infer Rest
) => any
  ? Rest
  : never

/**
 * Distributes over a type. It is used mostly to expand a function type
 * in hover previews while preserving their original JSDoc information.
 *
 * If preserving JSDoc information is not a concern, you can use {@linkcode ExpandFunction ExpandFunction}.
 *
 * @template T The type to be distributed.
 *
 * @internal
 */
export type Distribute<T> = T extends T ? T : never

/**
 * Extracts the type of the first element of an array or tuple.
 *
 * @internal
 */
export type FirstArrayElement<TArray> = TArray extends readonly [
  unknown,
  ...unknown[]
]
  ? TArray[0]
  : never

/**
 * Extracts the type of an array or tuple minus the first element.
 *
 * @internal
 */
export type ArrayTail<TArray> = TArray extends readonly [
  unknown,
  ...infer TTail
]
  ? TTail
  : []

/**
 * An alias for type `{}`. Represents any value that is not `null` or `undefined`.
 * It is mostly used for semantic purposes to help distinguish between an
 * empty object type and `{}` as they are not the same.
 *
 * @internal
 */
export type AnyNonNullishValue = NonNullable<unknown>

/**
 * Same as {@linkcode AnyNonNullishValue AnyNonNullishValue} but aliased
 * for semantic purposes. It is intended to be used in scenarios where
 * a recursive type definition needs to be interrupted to ensure type safety
 * and to avoid excessively deep recursion that could lead to performance issues.
 *
 * @internal
 */
export type InterruptRecursion = AnyNonNullishValue

/*
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 *
 * External/Copied Utility Types
 *
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 *
 */

/**
 * An if-else-like type that resolves depending on whether the given type is `never`.
 * This is mainly used to conditionally resolve the type of a `memoizeOptions` object based on whether `memoize` is provided or not.
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/if-never.d.ts Source}
 *
 * @internal
 */
export type IfNever<T, TypeIfNever, TypeIfNotNever> = [T] extends [never]
  ? TypeIfNever
  : TypeIfNotNever

/**
 * Omit any index signatures from the given object type, leaving only explicitly defined properties.
 * This is mainly used to remove explicit `any`s from the return type of some memoizers (e.g, `microMemoize`).
 *
 * __Disclaimer:__ When used on an intersection of a function and an object,
 * the function is erased.
 *
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/omit-index-signature.d.ts Source}
 *
 * @internal
 */
export type OmitIndexSignature<ObjectType> = {
  [KeyType in keyof ObjectType as {} extends Record<KeyType, unknown>
    ? never
    : KeyType]: ObjectType[KeyType]
}

/**
 * The infamous "convert a union type to an intersection type" hack
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/union-to-intersection.d.ts Source}
 * @see {@link https://github.com/microsoft/TypeScript/issues/29594 Reference}
 *
 * @internal
 */
export type UnionToIntersection<Union> =
  // `extends unknown` is always going to be the case and is used to convert the
  // `Union` into a [distributive conditional
  // type](https://www.typescriptlang.org/docs/handbook/release-notes/typescript-2-8.html#distributive-conditional-types).
  (
    Union extends unknown
      ? // The union type is used as the only argument to a function since the union
        // of function arguments is an intersection.
        (distributedUnion: Union) => void
      : // This won't happen.
        never
  ) extends // Infer the `Intersection` type since TypeScript represents the positional
  // arguments of unions of functions as an intersection of the union.
  (mergedIntersection: infer Intersection) => void
    ? Intersection
    : never

/**
 * Code to convert a union of values into a tuple.
 * @see {@link https://stackoverflow.com/a/55128956/62937 Source}
 *
 * @internal
 */
type Push<T extends any[], V> = [...T, V]

/**
 * @see {@link https://stackoverflow.com/a/55128956/62937 Source}
 *
 * @internal
 */
type LastOf<T> = UnionToIntersection<
  T extends any ? () => T : never
> extends () => infer R
  ? R
  : never

/**
 * TS4.1+
 * @see {@link https://stackoverflow.com/a/55128956/62937 Source}
 *
 * @internal
 */
export type TuplifyUnion<
  T,
  L = LastOf<T>,
  N = [T] extends [never] ? true : false
> = true extends N ? [] : Push<TuplifyUnion<Exclude<T, L>>, L>

/**
 * Converts "the values of an object" into a tuple, like a type-level `Object.values()`
 * @see {@link https://stackoverflow.com/a/68695508/62937 Source}
 *
 * @internal
 */
export type ObjectValuesToTuple<
  T,
  KS extends any[] = TuplifyUnion<keyof T>,
  R extends any[] = []
> = KS extends [infer K, ...infer KT]
  ? ObjectValuesToTuple<T, KT, [...R, T[K & keyof T]]>
  : R

/**
 *
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 *
 * Type Expansion Utilities
 *
 * -----------------------------------------------------------------------------
 * -----------------------------------------------------------------------------
 *
 */

/**
 * Check whether `U` contains `U1`.
 * @see {@link https://millsp.github.io/ts-toolbelt/modules/union_has.html Source}
 *
 * @internal
 */
export type Has<U, U1> = [U1] extends [U] ? 1 : 0

/**
 * @internal
 */
export type Boolean2 = 0 | 1

/**
 * @internal
 */
export type If2<B extends Boolean2, Then, Else = never> = B extends 1
  ? Then
  : Else

/**
 * @internal
 */
export type BuiltIn =
  | Function
  | Error
  | Date
  | { readonly [Symbol.toStringTag]: string }
  | RegExp
  | Generator

/**
 * Expand an item a single level.
 * @see {@link https://stackoverflow.com/a/69288824/62937 Source}
 *
 * @internal
 */
export type Expand<T> = T extends (...args: infer A) => infer R
  ? (...args: Expand<A>) => Expand<R>
  : T extends infer O
  ? { [K in keyof O]: O[K] }
  : never

/**
 * Expand an item recursively.
 * @see {@link https://stackoverflow.com/a/69288824/62937 Source}
 *
 * @internal
 */
export type ExpandRecursively<T> = T extends (...args: infer A) => infer R
  ? (...args: ExpandRecursively<A>) => ExpandRecursively<R>
  : T extends object
  ? T extends infer O
    ? { [K in keyof O]: ExpandRecursively<O[K]> }
    : never
  : T

/**
 * @internal
 */
export type Identity<T> = T

/**
 * Another form of type value expansion
 * @see {@link https://github.com/microsoft/TypeScript/issues/35247 Source}
 *
 * @internal
 */
export type Mapped<T> = Identity<{ [k in keyof T]: T[k] }>

/**
 * This utility type is primarily used to expand a function type in order to
 * improve its visual display in hover previews within IDEs.
 *
 * __Disclaimer:__ Functions expanded using this type will not display their
 * original JSDoc information in hover previews.
 *
 * @template FunctionType - The type of the function to be expanded.
 *
 * @internal
 */
export type ExpandFunction<FunctionType extends AnyFunction> =
  FunctionType extends FunctionType
    ? (...args: Parameters<FunctionType>) => ReturnType<FunctionType>
    : never

/**
 * Useful to flatten the type output to improve type hints shown in editors.
 * And also to transform an interface into a type to aide with assignability.
 * @see {@link https://github.com/sindresorhus/type-fest/blob/main/source/simplify.d.ts Source}
 *
 * @internal
 */
export type Simplify<T> = {
  [KeyType in keyof T]: T[KeyType]
} & AnyNonNullishValue

/**
 * Fully expand a type, deeply
 * @see {@link https://github.com/millsp/ts-toolbelt Any.Compute}
 *
 * @internal
 */
export type ComputeDeep<A, Seen = never> = A extends BuiltIn
  ? A
  : If2<
      Has<Seen, A>,
      A,
      A extends any[]
        ? A extends Record<PropertyKey, any>[]
          ? ({
              [K in keyof A[number]]: ComputeDeep<A[number][K], A | Seen>
            } & unknown)[]
          : A
        : A extends readonly any[]
        ? A extends readonly Record<PropertyKey, any>[]
          ? readonly ({
              [K in keyof A[number]]: ComputeDeep<A[number][K], A | Seen>
            } & unknown)[]
          : A
        : { [K in keyof A]: ComputeDeep<A[K], A | Seen> } & unknown
    >
