/**
 * Contains classes for classic scheme such as Node, Input, Output, Control, Socket, Connection
 * @module
 * @group Primary
 */
import { ConnectionBase, NodeBase } from '../types';
declare type PortId = string;
/**
 * The socket class
 * @priority 7
 */
export declare class Socket {
    name: string;
    /**
     * @constructor
     * @param name Name of the socket
     */
    constructor(name: string);
}
/**
 * General port class
 */
export declare class Port<S extends Socket> {
    socket: S;
    label?: string;
    multipleConnections?: boolean;
    /**
     * Port id, unique string generated by `getUID` function
     */
    id: PortId;
    /**
     * Port index, used for sorting ports. Default is `0`
     */
    index?: number;
    /**
     * @constructor
     * @param socket Socket instance
     * @param label Label of the port
     * @param multipleConnections Whether the output port can have multiple connections
     */
    constructor(socket: S, label?: string, multipleConnections?: boolean);
}
/**
 * The input port class
 * @priority 6
 */
export declare class Input<S extends Socket> extends Port<S> {
    socket: S;
    label?: string;
    multipleConnections?: boolean;
    /**
     * Control instance
     */
    control: Control | null;
    /**
     * Whether the control is visible. Can be managed dynamically by extensions. Default is `true`
     */
    showControl: boolean;
    /**
     * @constructor
     * @param socket Socket instance
     * @param label Label of the input port
     * @param multipleConnections Whether the output port can have multiple connections. Default is `false`
     */
    constructor(socket: S, label?: string, multipleConnections?: boolean);
    /**
     * Add control to the input port
     * @param control Control instance
     */
    addControl(control: Control): void;
    /**
     * Remove control from the input port
     */
    removeControl(): void;
}
/**
 * The output port class
 * @priority 5
 */
export declare class Output<S extends Socket> extends Port<S> {
    /**
     * @constructor
     * @param socket Socket instance
     * @param label Label of the output port
     * @param multipleConnections Whether the output port can have multiple connections. Default is `true`
     */
    constructor(socket: S, label?: string, multipleConnections?: boolean);
}
/**
 * General control class
 * @priority 5
 */
export declare class Control {
    /**
     * Control id, unique string generated by `getUID` function
     */
    id: string;
    /**
     * Control index, used for sorting controls. Default is `0`
     */
    index?: number;
    constructor();
}
/**
 * Input control options
 */
declare type InputControlOptions<N> = {
    /** Whether the control is readonly. Default is `false` */
    readonly?: boolean;
    /** Initial value of the control */
    initial?: N;
    /** Callback function that is called when the control value changes */
    change?: (value: N) => void;
};
/**
 * The input control class
 * @example new InputControl('text', { readonly: true, initial: 'hello' })
 */
export declare class InputControl<T extends 'text' | 'number', N = T extends 'text' ? string : number> extends Control {
    type: T;
    options?: InputControlOptions<N>;
    value?: N;
    readonly: boolean;
    /**
     * @constructor
     * @param type Type of the control: `text` or `number`
     * @param options Control options
     */
    constructor(type: T, options?: InputControlOptions<N>);
    /**
     * Set control value
     * @param value Value to set
     */
    setValue(value?: N): void;
}
/**
 * The node class
 * @priority 10
 * @example new Node('math')
 */
export declare class Node<Inputs extends {
    [key in string]?: Socket;
} = {
    [key in string]?: Socket;
}, Outputs extends {
    [key in string]?: Socket;
} = {
    [key in string]?: Socket;
}, Controls extends {
    [key in string]?: Control;
} = {
    [key in string]?: Control;
}> implements NodeBase {
    label: string;
    /**
     * Node id, unique string generated by `getUID` function
     */
    id: NodeBase['id'];
    /**
     * Node inputs
     */
    inputs: {
        [key in keyof Inputs]?: Input<Exclude<Inputs[key], undefined>>;
    };
    /**
     * Node outputs
     */
    outputs: {
        [key in keyof Outputs]?: Output<Exclude<Outputs[key], undefined>>;
    };
    /**
     * Node controls
     */
    controls: Controls;
    /**
     * Whether the node is selected. Default is `false`
     */
    selected?: boolean;
    constructor(label: string);
    hasInput<K extends keyof Inputs>(key: K): any;
    addInput<K extends keyof Inputs>(key: K, input: Input<Exclude<Inputs[K], undefined>>): void;
    removeInput(key: keyof Inputs): void;
    hasOutput<K extends keyof Outputs>(key: K): any;
    addOutput<K extends keyof Outputs>(key: K, output: Output<Exclude<Outputs[K], undefined>>): void;
    removeOutput(key: keyof Outputs): void;
    hasControl<K extends keyof Controls>(key: K): any;
    addControl<K extends keyof Controls>(key: K, control: Controls[K]): void;
    removeControl(key: keyof Controls): void;
}
/**
 * The connection class
 * @priority 9
 */
export declare class Connection<Source extends Node, Target extends Node> implements ConnectionBase {
    sourceOutput: keyof Source['outputs'];
    targetInput: keyof Target['inputs'];
    /**
     * Connection id, unique string generated by `getUID` function
     */
    id: ConnectionBase['id'];
    /**
     * Source node id
     */
    source: NodeBase['id'];
    /**
     * Target node id
     */
    target: NodeBase['id'];
    /**
     * @constructor
     * @param source Source node instance
     * @param sourceOutput Source node output key
     * @param target Target node instance
     * @param targetInput Target node input key
     */
    constructor(source: Source, sourceOutput: keyof Source['outputs'], target: Target, targetInput: keyof Target['inputs']);
}
export {};
//# sourceMappingURL=classic.d.ts.map