/*!
 * Built by Revolist
 */
import { proxyCustomElement, HTMLElement, createEvent, h, Host } from '@stencil/core/internal/client';
import { c as createStore, d as setStore, D as DataStore } from './data.store.js';
import { a as getItemByPosition } from './dimension.helpers.js';
import { U as UUID } from './consts.js';
import { d as defineCustomElement$3 } from './revogr-data2.js';
import { a as defineCustomElement$2 } from './revogr-header2.js';
import { d as defineCustomElement$1 } from './revogr-viewport-scroll2.js';

/**
 * Update items based on new scroll position
 * If viewport wasn't changed fully simple recombination of positions
 * Otherwise rebuild viewport items
 */
function getUpdatedItemsByPosition(pos, items, realCount, virtualSize, dimension) {
  const activeItem = getItemByPosition(dimension, pos);
  const firstItem = getFirstItem(items);
  let toUpdate;
  // do simple position replacement if items already present in viewport
  if (firstItem) {
    let changedOffsetStart = activeItem.itemIndex - (firstItem.itemIndex || 0);
    if (changedOffsetStart) {
      // simple recombination
      const newData = recombineByOffset(Math.abs(changedOffsetStart), Object.assign(Object.assign({ positiveDirection: changedOffsetStart > -1 }, dimension), items));
      if (newData) {
        toUpdate = newData;
      }
      // if partial replacement add items if revo-viewport has some space left
      if (toUpdate) {
        const extra = addMissingItems(activeItem, realCount, virtualSize, toUpdate, dimension);
        if (extra.length) {
          updateMissingAndRange(toUpdate.items, extra, toUpdate);
        }
      }
    }
  }
  // new collection if no items after replacement full replacement
  if (!toUpdate) {
    const items = getItems({
      start: activeItem.start,
      startIndex: activeItem.itemIndex,
      origSize: dimension.originItemSize,
      maxSize: virtualSize,
      maxCount: realCount,
      sizes: dimension.sizes,
    });
    // range now comes from 0 to length - 1
    toUpdate = {
      items,
      start: 0,
      end: items.length - 1,
    };
  }
  return toUpdate;
}
function updateMissingAndRange(items, missing, range) {
  items.splice(range.end + 1, 0, ...missing);
  // update range if start larger after recombination
  if (range.start >= range.end && !(range.start === range.end && range.start === 0)) {
    range.start += missing.length;
  }
  range.end += missing.length;
}
// if partial replacement add items if revo-viewport has some space left
function addMissingItems(firstItem, realCount, virtualSize, existingCollection, dimension) {
  const lastItem = getLastItem(existingCollection);
  const items = getItems({
    sizes: dimension.sizes,
    start: lastItem.end,
    startIndex: lastItem.itemIndex + 1,
    origSize: dimension.originItemSize,
    maxSize: virtualSize - (lastItem.end - firstItem.start),
    maxCount: realCount,
  });
  return items;
}
// get revo-viewport items parameters, caching position and calculating items count in revo-viewport
function getItems(opt, currentSize = 0) {
  const items = [];
  let index = opt.startIndex;
  let size = currentSize;
  while (size <= opt.maxSize && index < opt.maxCount) {
    const newSize = getItemSize(index, opt.sizes, opt.origSize);
    items.push({
      start: opt.start + size,
      end: opt.start + size + newSize,
      itemIndex: index,
      size: newSize,
    });
    size += newSize;
    index++;
  }
  return items;
}
/**
 * Do batch items recombination
 * If items not overlapped with existing viewport returns null
 */
function recombineByOffset(offset, data) {
  const newItems = [...data.items];
  const itemsCount = newItems.length;
  let newRange = {
    start: data.start,
    end: data.end,
  };
  // if offset out of revo-viewport, makes sense whole redraw
  if (offset > itemsCount) {
    return null;
  }
  // is direction of scroll positive
  if (data.positiveDirection) {
    // push item to the end
    let lastItem = getLastItem(data);
    let i = newRange.start;
    const length = i + offset;
    for (; i < length; i++) {
      const newIndex = lastItem.itemIndex + 1;
      const size = getItemSize(newIndex, data.sizes, data.originItemSize);
      // if item overlapped limit break a loop
      if (lastItem.end + size > data.realSize) {
        break;
      }
      // new item index to recombine
      let newEnd = i % itemsCount;
      // item should always present, we do not create new item, we recombine them
      if (!newItems[newEnd]) {
        throw new Error('incorrect index');
      }
      // do recombination
      newItems[newEnd] = lastItem = {
        start: lastItem.end,
        end: lastItem.end + size,
        itemIndex: newIndex,
        size: size,
      };
      // update range
      newRange.start++;
      newRange.end = newEnd;
    }
    // direction is negative
  }
  else {
    // push item to the start
    let firstItem = getFirstItem(data);
    const end = newRange.end;
    for (let i = 0; i < offset; i++) {
      const newIndex = firstItem.itemIndex - 1;
      const size = getItemSize(newIndex, data.sizes, data.originItemSize);
      // new item index to recombine
      let newStart = end - i;
      newStart = (newStart < 0 ? itemsCount + newStart : newStart) % itemsCount;
      // item should always present, we do not create new item, we recombine them
      if (!newItems[newStart]) {
        throw new Error('incorrect index');
      }
      // do recombination
      newItems[newStart] = firstItem = {
        start: firstItem.start - size,
        end: firstItem.start,
        itemIndex: newIndex,
        size: size,
      };
      // update range
      newRange.start = newStart;
      newRange.end--;
    }
  }
  const range = {
    start: (newRange.start < 0 ? itemsCount + newRange.start : newRange.start) % itemsCount,
    end: (newRange.end < 0 ? itemsCount + newRange.end : newRange.end) % itemsCount,
  };
  return Object.assign({ items: newItems }, range);
}
function getItemSize(index, sizes, origSize = 0) {
  if (sizes && sizes[index]) {
    return sizes[index];
  }
  return origSize;
}
function isActiveRange(pos, item) {
  return item && pos >= item.start && pos <= item.end;
}
function getFirstItem(s) {
  return s.items[s.start];
}
function getLastItem(s) {
  return s.items[s.end];
}
/**
 * Set items sizes from start index to end
 * @param vpItems
 * @param start
 * @param size
 * @param lastCoordinate
 * @returns
 */
function setItemSizes(vpItems, initialIndex, size, lastCoordinate) {
  const items = [...vpItems];
  const count = items.length;
  let pos = lastCoordinate;
  let i = 0;
  let start = initialIndex;
  // viewport not inited
  if (!count) {
    return [];
  }
  // loop through array from initial item after recombination
  while (i < count) {
    const item = items[start];
    item.start = pos;
    item.size = size;
    item.end = item.start + size;
    pos = item.end;
    // loop by start index
    start++;
    i++;
    // if start index out of array, reset it
    if (start === count) {
      start = 0;
    }
  }
  return items;
}

/**
 * Store is responsible for visible
 * Viewport information for each dimension
 * Redraw items during scrolling
 */
function initialState() {
  return {
    // virtual item information per rendered item
    items: [],
    // virtual dom item order to render
    start: 0,
    end: 0,
    // size of viewport in px
    virtualSize: 0,
    // total number of items
    realCount: 0,
    // last coordinate for store position restore
    lastCoordinate: 0,
  };
}
class ViewportStore {
  constructor() {
    this.store = createStore(initialState());
    this.store.onChange('realCount', () => this.clear());
  }
  /** Render viewport based on coordinate, this is main method for draw */
  setViewPortCoordinate(position, dimension) {
    let virtualSize = this.store.get('virtualSize');
    // no visible data to calculate
    if (!virtualSize) {
      return;
    }
    const frameOffset = dimension.frameOffset;
    const outsize = frameOffset * 2 * dimension.originItemSize;
    virtualSize += outsize;
    let maxCoordinate = virtualSize;
    if (dimension.realSize > virtualSize) {
      maxCoordinate = dimension.realSize - virtualSize;
    }
    let toUpdate = {
      lastCoordinate: position,
    };
    let pos = position;
    pos -= frameOffset * dimension.originItemSize;
    pos = pos < 0 ? 0 : pos < maxCoordinate ? pos : maxCoordinate;
    const firstItem = getFirstItem(this.getItems());
    const lastItem = getLastItem(this.getItems());
    // left position changed
    if (!isActiveRange(pos, firstItem)) {
      toUpdate = Object.assign(Object.assign({}, toUpdate), getUpdatedItemsByPosition(pos, this.getItems(), this.store.get('realCount'), virtualSize, dimension));
      setStore(this.store, Object.assign({}, toUpdate));
      // right position changed
    }
    else if (firstItem && this.store.get('virtualSize') + pos > (lastItem === null || lastItem === void 0 ? void 0 : lastItem.end)) {
      // check is any item missing for full fill content
      const missing = addMissingItems(firstItem, this.store.get('realCount'), virtualSize + pos - firstItem.start, this.getItems(), dimension);
      if (missing.length) {
        const items = [...this.store.get('items')];
        const range = {
          start: this.store.get('start'),
          end: this.store.get('end'),
        };
        updateMissingAndRange(items, missing, range);
        toUpdate = Object.assign(Object.assign(Object.assign({}, toUpdate), { items: [...items] }), range);
        setStore(this.store, Object.assign({}, toUpdate));
      }
    }
  }
  /** Update viewport sizes */
  setViewPortDimension(sizes, dropToOriginalSize) {
    let items = this.store.get('items');
    const count = items.length;
    // viewport not inited
    if (!count) {
      return;
    }
    let changedCoordinate = 0;
    let i = 0;
    let start = this.store.get('start');
    // drop to original size if requested
    if (dropToOriginalSize) {
      items = setItemSizes(items, start, dropToOriginalSize, this.store.get('lastCoordinate'));
    }
    // loop through array from initial item after recombination
    while (i < count) {
      const item = items[start];
      // change pos if size change present before
      if (changedCoordinate) {
        item.start += changedCoordinate;
        item.end += changedCoordinate;
      }
      // change size
      const size = sizes[item.itemIndex];
      if (size) {
        const changedSize = size - item.size;
        changedCoordinate += changedSize;
        item.size = size;
        item.end = item.start + size;
      }
      // loop by start index
      start++;
      i++;
      if (start === count) {
        start = 0;
      }
    }
    setStore(this.store, { items: [...items] });
  }
  getItems() {
    return {
      items: this.store.get('items'),
      start: this.store.get('start'),
      end: this.store.get('end'),
    };
  }
  setViewport(data) {
    setStore(this.store, data);
  }
  clear() {
    this.store.set('items', []);
  }
}

const RowHeaderRender = s => (__, { rowIndex: i }) => s + i;

const LETTER_BLOCK_SIZE = 10;
const RevogrRowHeaders = /*@__PURE__*/ proxyCustomElement(class extends HTMLElement {
  constructor() {
    super();
    this.__registerHost();
    this.scrollViewport = createEvent(this, "scrollViewport", 3);
    this.elementToScroll = createEvent(this, "elementToScroll", 3);
  }
  render() {
    var _a;
    const dataViews = [];
    const viewport = new ViewportStore();
    /** render viewports rows */
    let totalLength = 1;
    for (let data of this.dataPorts) {
      const itemCount = data.dataStore.get('items').length;
      // initiate row data
      const dataStore = new DataStore(data.type);
      dataStore.updateData(data.dataStore.get('source'));
      // initiate column data
      const colData = new DataStore('colPinStart');
      const column = Object.assign({ cellTemplate: RowHeaderRender(totalLength) }, this.rowHeaderColumn);
      colData.updateData([column]);
      const viewData = Object.assign(Object.assign({}, data), { dataStore: dataStore.store, colData: colData.store, viewportCol: viewport.store, readonly: true, range: false });
      dataViews.push(h("revogr-data", Object.assign({}, viewData)));
      totalLength += itemCount;
    }
    const colSize = ((_a = this.rowHeaderColumn) === null || _a === void 0 ? void 0 : _a.size) || (totalLength.toString().length + 1) * LETTER_BLOCK_SIZE;
    viewport.setViewport({
      realCount: 1,
      virtualSize: 0,
      items: [
        {
          size: colSize,
          start: 0,
          end: colSize,
          itemIndex: 0,
        },
      ],
    });
    const parent = `${this.uiid}-rowHeaders`;
    const viewportScroll = {
      [UUID]: parent,
      contentHeight: this.height,
      contentWidth: 0,
      style: { minWidth: `${colSize}px` },
      ref: (el) => this.elementToScroll.emit(el),
      onScrollViewport: (e) => this.scrollViewport.emit(e.detail),
    };
    const viewportHeader = Object.assign(Object.assign({}, this.headerProp), { colData: typeof this.rowHeaderColumn === 'object' ? [this.rowHeaderColumn] : [], viewportCol: viewport.store, canResize: false, parent, slot: 'header' });
    return h(Host, { class: "rowHeaders", key: "rowHeaders" }, h("revogr-viewport-scroll", Object.assign({}, viewportScroll), h("revogr-header", Object.assign({}, viewportHeader)), dataViews));
  }
}, [0, "revogr-row-headers", {
    "height": [2],
    "dataPorts": [16],
    "headerProp": [16],
    "uiid": [1],
    "resize": [4],
    "rowHeaderColumn": [16]
  }]);
function defineCustomElement() {
  if (typeof customElements === "undefined") {
    return;
  }
  const components = ["revogr-row-headers", "revogr-data", "revogr-header", "revogr-viewport-scroll"];
  components.forEach(tagName => { switch (tagName) {
    case "revogr-row-headers":
      if (!customElements.get(tagName)) {
        customElements.define(tagName, RevogrRowHeaders);
      }
      break;
    case "revogr-data":
      if (!customElements.get(tagName)) {
        defineCustomElement$3();
      }
      break;
    case "revogr-header":
      if (!customElements.get(tagName)) {
        defineCustomElement$2();
      }
      break;
    case "revogr-viewport-scroll":
      if (!customElements.get(tagName)) {
        defineCustomElement$1();
      }
      break;
  } });
}
defineCustomElement();

export { RevogrRowHeaders as R, ViewportStore as V, defineCustomElement as d };
