/* Riot Compiler WIP, @license MIT */
'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var recastUtil = _interopDefault(require('recast/lib/util'));
var sourceMap = require('source-map');
var recast = _interopDefault(require('recast'));
var globalScope = _interopDefault(require('globals'));
var curry = _interopDefault(require('curri'));
var riotParser = require('@riotjs/parser');
var riotParser__default = _interopDefault(riotParser);
var ruit = _interopDefault(require('ruit'));

const TAG_LOGIC_PROPERTY = 'tag';
const TAG_CSS_PROPERTY = 'css';
const TAG_TEMPLATE_PROPERTY = 'template';

/**
 * Detect node js environements
 * @returns { boolean } true if the runtime is node
 */
function isNode() {
  return typeof process !== 'undefined'
}

/**
 * Return a source map as JSON, it it has not the toJSON method it means it can
 * be used right the way
 * @param   { SourceMapGenerator|Object } map - a sourcemap generator or simply an json object
 * @returns { Object } the source map as JSON
 */
function asJSON(map) {
  if (map.toJSON) return map.toJSON()
  return map
}
/**
 * Compose two sourcemaps
 * @param   { SourceMapGenerator } formerMap - original sourcemap
 * @param   { SourceMapGenerator } latterMap - target sourcemap
 * @returns { Object } sourcemap json
 */
function composeSourcemaps(formerMap, latterMap) {
  if (
    isNode() &&
    formerMap && latterMap && latterMap.mappings
  ) {
    return recastUtil.composeSourceMaps(asJSON(formerMap), asJSON(latterMap))
  } else if (isNode() && formerMap) {
    return asJSON(formerMap)
  }

  return {}
}

/**
 * Create a new sourcemap generator
 * @param   { Object } options - sourcemap options
 * @returns { SourceMapGenerator } SourceMapGenerator instance
 */
function createSourcemap(options) {
  return new sourceMap.SourceMapGenerator(options)
}

const Output = Object.freeze({
  code: '',
  ast: [],
  map: null
});

/**
 * Create the right output data result of a parsing
 * @param   { Object } data - output data
 * @param   { string } data.code - code generated
 * @param   { AST } data.ast - ast representing the code
 * @param   { SourceMapGenerator } data.map - source map generated along with the code
 * @param   { Object } options - user options, probably containing the path to the source file
 * @returns { Output } output container object
 */
function createOutput(data, options) {
  const output = Object.seal({
    ...Output,
    ...data
  });

  if (!output.map && options && options.file) Object.assign(output, {
    map: createSourcemap({ file: options.file })
  });

  return output
}

/**
 * Transform the source code received via a compiler function
 * @param   { Function } compiler - function needed to generate the output code
 * @param   { Object } options - options to pass to the compilert
 * @param   { string } source - source code
 * @returns { Promise<Output> } output - the result of the compiler
 */
async function transform(compiler, options, source) {
  const result = await (compiler ? compiler(source, options) : { code: source });
  return createOutput(result, options)
}

/**
 * Throw an error with a descriptive message
 * @param   { string } message - error message
 * @returns { undefined } hoppla.. at this point the program should stop working
 */
function panic(message) {
  throw new Error(message)
}

const postprocessors = new Set();

/**
 * Register a postprocessor that will be used after the parsing and compilation of the riot tags
 * @param { Function } postprocessor - transformer that will receive the output code ans sourcemap
 * @returns { Set } the postprocessors collection
 */
function register(postprocessor) {
  if (postprocessors.has(postprocessor)) {
    panic(`This postprocessor "${postprocessor.name || postprocessor.toString()}" was already registered`);
  }

  postprocessors.add(postprocessor);

  return postprocessors
}

/**
 * Exec all the postprocessors in sequence combining the sourcemaps generated
 * @param   { Output } compilerOutput - output generated by the compiler
 * @param   { Object } options - user options received by the compiler
 * @returns { Promise<Output> } object containing output code and source map
 */
async function execute(compilerOutput, options) {
  return Array.from(postprocessors).reduce(async function(acc, postprocessor) {
    const { code, map } = await acc;
    const output = await postprocessor(code, options);

    return {
      code: output.code,
      map: composeSourcemaps(output.map, map)
    }
  }, Promise.resolve(createOutput(compilerOutput, options)))
}

/**
 * Parsers that can be registered by users to preparse components fragments
 * @type { Object }
 */
const preprocessors = Object.freeze({
  javascript: new Map(),
  css: new Map(),
  template: new Map()
    .set('default', function(code) { return { code } })
});

// throw a processor type error
function preprocessorTypeError(type) {
  panic(`No preprocessor of type "${type}" was found, please make sure to use one of these: 'javascript', 'css' or 'template'`);
}

// throw an error if the preprocessor was not registered
function preprocessorNameNotFoundError(name) {
  panic(`No preprocessor named "${name}" was found, are you sure you have registered it?'`);
}

/**
 * Register a custom preprocessor
 * @param   { string } type - preprocessor type either 'js', 'css' or 'template'
 * @param   { string } name - unique preprocessor id
 * @param   { Function } preprocessor - preprocessor function
 * @returns { Map } - the preprocessors map
 */
function register$1(type, name, preprocessor) {
  if (!type) panic('Please define the type of preprocessor you want to register \'javascript\', \'css\' or \'template\'');
  if (!name) panic('Please define a name for your preprocessor');
  if (!preprocessor) panic('Please provide a preprocessor function');
  if (!preprocessors[type]) preprocessorTypeError(type);
  if (preprocessors[type].has(name)) panic(`The preprocessor ${name} was already registered before`);

  preprocessors[type].set(name, preprocessor);

  return preprocessors
}

/**
 * Exec the compilation of a preprocessor
 * @param   { string } type - preprocessor type either 'js', 'css' or 'template'
 * @param   { string } name - unique preprocessor id
 * @param   { Object } options - preprocessor options
 * @param   { string } source - source code
 * @returns { Promise<Output> } object containing a sourcemap and a code string
 */
async function execute$1(type, name, options, source) {
  if (!preprocessors[type]) preprocessorTypeError(type);
  if (!preprocessors[type].has(name)) preprocessorNameNotFoundError(name);

  return await transform(preprocessors[type].get(name), options, source)
}

const ATTRIBUTE_TYPE_NAME = 'type';

/**
 * Get the type attribute from a node generated by the riot parser
 * @param   { Object} sourceNode - riot parser node
 * @returns { string|null } a valid type to identify the preprocessor to use or nothing
 */
function getPreprocessorTypeByAttribute(sourceNode) {
  const typeAttribute = sourceNode.attributes ?
    sourceNode.attributes.find(attribute => attribute.name === ATTRIBUTE_TYPE_NAME) :
    null;

  return typeAttribute ? normalize(typeAttribute.value) : null
}


/**
 * Remove the noise in case a user has defined the preprocessor type='text/scss'
 * @param   { string } value - input string
 * @returns { string } normalized string
 */
function normalize(value) {
  return value.replace('text/', '')
}

/**
 * Count spaces before first character of a string
 * @param   { string } string - target string
 * @returns { number } amount of spaces before the first char
 */
function getColum(string) {
  const spacesAmount = string.search(/\S/);
  return spacesAmount > 0 ? spacesAmount : 0
}

const LINES_RE = /\r\n?|\n/g;

/**
 * Split a string into a rows array generated from its EOL matches
 * @param   { string } string [description]
 * @returns { Array } array containing all the string rows
 */
function splitStringByEOL(string) {
  return string.split(LINES_RE)
}

/**
 * Get the line and the column of a source text based on its position in the string
 * @param   { string } string - target string
 * @param   { number } position - target position
 * @returns { Object } object containing the source text line and column
 */
function getLineAndColumnByPosition(string, position) {
  const lines = splitStringByEOL(string.slice(0, position));

  return {
    line: lines.length,
    column: getColum(lines[lines.length - 1])
  }
}

/**
 * Preprocess a riot parser node
 * @param   { string } preprocessorType - either css, js
 * @param   { string } preprocessorName - preprocessor id
 * @param   { Object } options - options that will be passed to the compiler
 * @param   { string } source - tag source code
 * @param   { RiotParser.nodeTypes } node - css node detected by the parser
 * @returns { Output } code and sourcemap generated by the preprocessor
 */
async function preprocess(preprocessorType, preprocessorName, options, source, node) {
  const { column } = getLineAndColumnByPosition(source, node.start);
  const offsetTop = '\n'.repeat(column);
  const code = `${offsetTop}\n${node.text}`;

  return await (preprocessorName ?
    execute$1(preprocessorType, preprocessorName, options, code) :
    { code }
  )
}

const types = recast.types;
const builders = types.builders;
const namedTypes = types.namedTypes;

/**
 * Generate the component css
 * @param   { Object } sourceNode - node generated by the riot compiler
 * @param   { string } source - original component source code
 * @param   { Object } options - user options
 * @param   { Object } output - current compiler output
 * @returns { Promise<Output> } - the current ast program and the original sourcemap
 */
async function css(sourceNode, source, options, { ast, map }) {
  const preprocessorName = getPreprocessorTypeByAttribute(sourceNode);
  const cssNode = sourceNode.text;
  const preprocessorOutput = await preprocess('css', preprocessorName, options, source, cssNode);
  const generatedCss = recast.parse(`\`${preprocessorOutput.code.trim()}\``, {
    sourceFileName: options.file,
    inputSourceMap: composeSourcemaps(map, preprocessorOutput.map)
  });

  types.visit(ast, {
    visitProperty(path) {
      if (path.value.key.name === TAG_CSS_PROPERTY) {
        path.value.value = generatedCss.program.body[0].expression;
        return false
      }

      this.traverse(path);
    }
  });

  return { ast, map, code: recast.print(ast).code }
}

const isExportDefaultStatement = namedTypes.ExportDefaultDeclaration.check;

/**
 * Find the export default statement
 * @param   { Array } body - tree structure containing the program code
 * @returns { Object } node containing only the code of the export default statement
 */
function findExportDefaultStatement(body) {
  return body.find(isExportDefaultStatement)
}

/**
 * Find all the code in an ast program except for the export default statements
 * @param   { Array } body - tree structure containing the program code
 * @returns { Array } array containing all the program code except the export default expressions
 */
function filterNonExportDefaultStatements(body) {
  return body.filter(node => !isExportDefaultStatement(node))
}

/**
 * Get the body of the AST structure
 * @param   { Object } ast - ast object generated by recast
 * @returns { Array } array containing the program code
 */
function getProgramBody(ast) {
  return ast.program.body
}

/**
 * Extend the AST adding the new tag method containing our tag sourcecode
 * @param   { Object } ast - current output ast
 * @param   { Object } exportDefaultNode - tag export default node
 * @returns { Object } the output ast having the "tag" key extended with the content of the export default
 */
function extendTagProperty(ast, exportDefaultNode) {
  types.visit(ast, {
    visitProperty(path) {
      if (path.value.key.name === TAG_LOGIC_PROPERTY) {
        path.value.value = exportDefaultNode.declaration;
        return false
      }

      this.traverse(path);
    }
  });

  return ast
}

/**
 * Generate the component javascript logic
 * @param   { Object } sourceNode - node generated by the riot compiler
 * @param   { string } source - original component source code
 * @param   { Object } options - user options
 * @param   { Output } output - current compiler output
 * @returns { Promise<Output> } - enhanced output with the result of the current generator
 */
async function javascript(sourceNode, source, options, { ast, map }) {
  const preprocessorName = getPreprocessorTypeByAttribute(sourceNode);
  const javascriptNode = sourceNode.text;
  const preprocessorOutput = await preprocess('js', preprocessorName, options, source, javascriptNode);
  const generatedAst = recast.parse(preprocessorOutput.code, {
    sourceFileName: options.file,
    inputSourceMap: composeSourcemaps(map, preprocessorOutput.map)
  });
  const generatedAstBody = getProgramBody(generatedAst);
  const bodyWithoutExportDefault = filterNonExportDefaultStatements(generatedAstBody);
  const exportDefaultNode = findExportDefaultStatement(generatedAstBody);
  const outputBody = getProgramBody(ast);

  // add to the ast the "private" javascript content of our tag script node
  outputBody.unshift(...bodyWithoutExportDefault);

  // convert the export default adding its content to the "tag" property exported
  if (exportDefaultNode) extendTagProperty(ast, exportDefaultNode);

  return {
    ast,
    map,
    code: recast.print(ast).code
  }
}

const BINDING_TYPES = 'bindingTypes';
const EACH_BINDING_TYPE = 'EACH';
const IF_BINDING_TYPE = 'IF';
const TAG_BINDING_TYPE = 'TAG';


const EXPRESSION_TYPES = 'expressionTypes';
const ATTRIBUTE_EXPRESSION_TYPE = 'ATTRIBUTE';
const VALUE_EXPRESSION_TYPE = 'VALUE';
const TEXT_EXPRESSION_TYPE = 'TEXT';
const EVENT_EXPRESSION_TYPE = 'EVENT';

const TEMPLATE_FN = 'template';
const SCOPE = 'scope';
const COMPONENTS_REGISTRY = 'components';

// keys needed to create the DOM bindings
const BINDING_SELECTOR_KEY = 'selector';
const BINDING_COMPONENTS_KEY = 'components';
const BINDING_TEMPLATE_KEY = 'template';
const BINDING_TYPE_KEY = 'type';
const BINDING_REDUNDANT_ATTRIBUTE_KEY = 'redundantAttribute';
const BINDING_CONDITION_KEY = 'condition';
const BINDING_ITEM_NAME_KEY = 'itemName';
const BINDING_GET_KEY_KEY = 'getKey';
const BINDING_INDEX_NAME_KEY = 'indexName';
const BINDING_EVALUATE_KEY = 'evaluate';
const BINDING_NAME_KEY = 'name';
const BINDING_SLOTS_KEY = 'slots';
const BINDING_EXPRESSIONS_KEY = 'expressions';
const BINDING_CHILD_NODE_INDEX_KEY = 'childNodeIndex';
// slots keys
const BINDING_BINDINGS_KEY = 'bindings';
const BINDING_ID_KEY = 'id';
const BINDING_HTML_KEY = 'html';
const BINDING_ATTRIBUTES_KEY = 'attributes';

// DOM directives
const IF_DIRECTIVE = 'if';
const EACH_DIRECTIVE = 'each';
const KEY_ATTRIBUTE = 'key';
const SLOT_ATTRIBUTE = 'slot';

const TEXT_NODE_EXPRESSION_PLACEHOLDER = '<!---->';
const BINDING_SELECTOR_PREFIX = 'expr';
const IS_VOID_NODE = 'isVoid';
const IS_CUSTOM_NODE = 'isCustom';
const IS_BOOLEAN_ATTRIBUTE = 'isBoolean';

const browserAPIs = Object.keys(globalScope.browser);
const builtinAPIs = Object.keys(globalScope.builtin);

const isIdentifier = namedTypes.Identifier.check;
const isLiteral = namedTypes.Literal.check;
const isExpressionStatement = namedTypes.ExpressionStatement.check;
const isObjectExpression = namedTypes.ObjectExpression.check;
const isThisExpression = namedTypes.ThisExpression.check;
const isSequenceExpression = namedTypes.SequenceExpression.check;
const isBinaryExpression = namedTypes.BinaryExpression.check;

const isBrowserAPI = ({name}) => browserAPIs.includes(name);
const isBuiltinAPI = ({name}) => builtinAPIs.includes(name);

function nullNode() {
  return builders.literal(null)
}

function simplePropertyNode(key, value) {
  return builders.property('init', builders.literal(key), value, false)
}

/* eslint-disable */
// source: https://30secondsofcode.org/function#compose
var compose = (...fns) => fns.reduce((f, g) => (...args) => f(g(...args)));

const scope = builders.identifier(SCOPE);
const getName = node => node && node.name ? node.name : node;

/**
 * Find the attribute node
 * @param   { string } name -  name of the attribute we want to find
 * @param   { riotParser.nodeTypes.TAG } node - a tag node
 * @returns { riotParser.nodeTypes.ATTR } attribute node
 */
function findAttribute(name, node) {
  return node.attributes && node.attributes.find(attr => getName(attr) === name)
}

const findIfAttribute = curry(findAttribute)(IF_DIRECTIVE);
const findEachAttribute = curry(findAttribute)(EACH_DIRECTIVE);
const findKeyAttribute = curry(findAttribute)(KEY_ATTRIBUTE);
const hasIfAttribute = compose(Boolean, findIfAttribute);
const hasEachAttribute = compose(Boolean, findEachAttribute);
const hasKeyAttribute = compose(Boolean, findKeyAttribute);

function createExpressionSourcemap(expression, sourceFile, sourceCode) {
  const sourcemap = createSourcemap({ file: sourceFile })

  ;[expression.start, expression.end].forEach(position => {
    const location = getLineAndColumnByPosition(sourceCode, position);

    sourcemap.addMapping({
      source: sourceFile,
      generated: location,
      original: location
    });
  });
}

/**
 * Check if a node name is part of the browser or builtin javascript api or it belongs to the current scope
 * @param   { types.NodePath } path - containing the current node visited
 * @returns {boolean} true if it's a global api variable
 */
function isGlobal({ scope, node }) {
  return isBuiltinAPI(node) || isBrowserAPI(node) || scope.lookup(getName(node))
}

/**
 * Replace the path scope with a member Expression
 * @param   { types.NodePath } path - containing the current node visited
 * @param   { types.Node } property - node we want to prefix with the scope identifier
 * @returns {undefined} this is a void function
 */
function replacePathScope(path, property) {
  path.replace(builders.memberExpression(
    scope,
    property,
    false
  ));
}

/**
 * Change the nodes scope adding the `scope` prefix
 * @param   { types.NodePath } path - containing the current node visited
 * @returns { boolean } return false if we want to stop the tree traversal
 * @context { types.visit }
 */
function updateNodeScope(path) {
  if (!isGlobal(path)) {
    replacePathScope(path, isThisExpression(path.node.object) ? path.node.property : path.node);

    return false
  }

  this.traverse(path);
}

/**
 * Objects properties should be handled a bit differently from the Identifier
 * @param   { types.NodePath } path - containing the current node visited
 * @returns { boolean } return false if we want to stop the tree traversal
 */
function visitProperty(path) {
  const value = path.node.value;

  if (isIdentifier(value)) {
    updateNodeScope(path.get('value'));
  } else if (isObjectExpression(value)) {
    this.traverse(path.get('value'));
  }

  return false
}

/**
 * The this expressions should be replaced with the scope
 * @param   { types.NodePath } path - containing the current node visited
 * @returns { boolean } return false if we want to stop the tree traversal
 */
function visitThisExpression(path) {
  path.replace(scope);
  this.traverse(path);
}

/**
 * Update the scope of the global nodes
 * @param   { Object } ast - ast program
 * @returns { Object } the ast program with all the global nodes updated
 */
function updateNodesScope(ast) {
  const ignorePath = () => false;

  types.visit(ast, {
    visitIdentifier: updateNodeScope,
    visitMemberExpression: updateNodeScope,
    visitProperty,
    visitThisExpression,
    visitClassExpression: ignorePath
  });

  return ast
}

/**
 * Convert any expression to an AST tree
 * @param   { Object } expression - expression parsed by the riot parser
 * @param   { string } sourceFile - original tag file
 * @param   { string } sourceCode - original tag source code
 * @returns { Object } the ast generated
 */
function createASTFromExpression(expression, sourceFile, sourceCode) {
  return recast.parse(`(${expression.text})`, {
    sourceFileName: sourceFile,
    inputSourceMap: sourceFile && createExpressionSourcemap(expression, sourceFile, sourceCode)
  })
}

const getEachItemName = expression => isSequenceExpression(expression.left) ? expression.left.expressions[0] : expression.left;
const getEachIndexName = expression => isSequenceExpression(expression.left) ? expression.left.expressions[1] : null;
const getEachValue = expression => expression.right;
const nameToliteral = compose(builders.literal, getName);

/**
 * Get the each expression properties to create properly the template binding
 * @param   { DomBinding.Expression } eachExpression - original each expression data
 * @param   { string } sourceFile - original tag file
 * @param   { string } sourceCode - original tag source code
 * @returns { Array } AST nodes that are needed to build an each binding
 */
function getEachExpressionProperties(eachExpression, sourceFile, sourceCode) {
  const ast = createASTFromExpression(eachExpression, sourceFile, sourceCode);
  const body = ast.program.body;
  const firstNode = body[0];


  if (!isExpressionStatement(firstNode)) {
    panic(`The each directives supported should be of type "ExpressionStatement",you have provided a "${firstNode.type}"`);
  }

  const { expression } = firstNode;

  return [
    simplePropertyNode(
      BINDING_ITEM_NAME_KEY,
      compose(nameToliteral, getEachItemName)(expression)
    ),
    simplePropertyNode(
      BINDING_INDEX_NAME_KEY,
      compose(nameToliteral, getEachIndexName)(expression)
    ),
    simplePropertyNode(
      BINDING_EVALUATE_KEY,
      compose(
        e => toScopedFunction(e, sourceFile, sourceCode),
        e => ({
          ...eachExpression,
          text: recast.print(e).code
        }),
        getEachValue
      )(expression)
    )
  ]
}

/**
 * Create the bindings template property
 * @param   {Array} args - arguments to pass to the template function
 * @returns {ASTNode} a binding template key
 */
function createTemplateProperty(args) {
  return simplePropertyNode(
    BINDING_TEMPLATE_KEY,
    args ? callTemplateFunction(...args) : nullNode()
  )
}

/**
 * Try to get the expression of an attribute node
 * @param   { RiotParser.Node.Attribute } attribute - riot parser attribute node
 * @returns { RiotParser.Node.Expression } attribute expression value
 */
function getAttributeExpression(attribute) {
  return attribute.expressions ? attribute.expressions[0] : {
    // if no expression was found try to typecast the attribute value
    ...attribute,
    text: attribute.value
  }
}

/**
 * Convert any parser option to a valid template one
 * @param   { RiotParser.Node.Expression } expression - expression parsed by the riot parser
 * @param   { string } sourceFile - original tag file
 * @param   { string } sourceCode - original tag source code
 * @returns { Object } a FunctionExpression object
 *
 * @example
 *  toScopedFunction('foo + bar') // scope.foo + scope.bar
 *
 * @example
 *  toScopedFunction('foo.baz + bar') // scope.foo.baz + scope.bar
 */
function toScopedFunction(expression, sourceFile, sourceCode) {
  const ast = createASTFromExpression(expression, sourceFile, sourceCode);
  const generatedAST = updateNodesScope(ast);
  const astBody = generatedAST.program.body;
  const expressionAST = astBody[0] ? astBody[0].expression : astBody;

  return builders.functionExpression(
    null,
    [builders.identifier(SCOPE)],
    builders.blockStatement([builders.returnStatement(
      expressionAST
    )])
  )
}

/**
 * Wrap a string in a template literal expression
 * @param   {string} string - target string
 * @returns {string} a template literal
 */
function wrapInBacktick(string) {
  return `\`${string}\``
}

/**
 * Simple bindings might contain multiple expressions like for example: "{foo} and {bar}"
 * This helper aims to merge them in a template literal if it's necessary
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {string} either a string representing a template literal or simply the first expression matched
 */
function mergeNodeExpressions(node) {
  if (node.expressions.length === 1) {
    return node.expressions[0].text
  }

  const charAddedProIteration = 3; // ${}

  // a tricky way to merge nodes containing multiple expressions in the same string
  const values = node.expressions.reduce((string, expression, index) => {
    const bracketsLength = expression.end - expression.start - expression.text.length;
    const offset = index * (charAddedProIteration - bracketsLength);
    const expressionStart = expression.start - node.start + offset;
    const expressionEnd = expression.end - node.start + offset;

    return `${string.substring(0, expressionStart)}$\{${expression.text}}${string.substring(expressionEnd)}`
  }, node.text);

  return wrapInBacktick(values)
}

/**
 * Create the template call function
 * @param   {Array|string|Node.Literal} template - template string
 * @param   {Array<AST.Nodes>} bindings - template bindings provided as AST nodes
 * @returns {Node.CallExpression} template call expression
 */
function callTemplateFunction(template, bindings) {
  return builders.callExpression(builders.identifier(TEMPLATE_FN), [
    template ? builders.literal(template) : nullNode(),
    bindings ? builders.arrayExpression(bindings) : nullNode()
  ])
}

/**
 * Convert any DOM attribute into a valid DOM selector useful for the querySelector API
 * @param   { string } attributeName - name of the attribute to query
 * @returns { string } the attribute transformed to a query selector
 */
const attributeNameToDOMQuerySelector = attributeName => `[${attributeName}]`;

/**
 * Create the properties to query a DOM node
 * @param   { string } attributeName - attribute name needed to identify a DOM node
 * @returns { Array<AST.Node> } array containing the selector properties needed for the binding
 */
function createSelectorProperties(attributeName) {
  return attributeName ? [
    simplePropertyNode(BINDING_REDUNDANT_ATTRIBUTE_KEY, builders.literal(attributeName)),
    simplePropertyNode(BINDING_SELECTOR_KEY,
      compose(builders.literal, attributeNameToDOMQuerySelector)(attributeName)
    )
  ] : []
}

/**
 * Clean binding or custom attributes
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {Array<RiotParser.Node.Attr>} only the attributes that are not bindings or directives
 */
function cleanAttributes(node) {
  return getNodeAttributes(node).filter(attribute => ![IF_DIRECTIVE, EACH_DIRECTIVE, KEY_ATTRIBUTE, SLOT_ATTRIBUTE].includes(attribute.name))
}

/**
 * Create a root node proxing only its nodes and attributes
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {RiotParser.Node} root node
 */
function createRootNode(node) {
  return {
    nodes: getChildrenNodes(node),
    isRoot: true,
    // root nodes shuold't have directives
    attributes: cleanAttributes(node)
  }
}

/**
 * Get all the child nodes of a RiotParser.Node
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {Array<RiotParser.Node>} all the child nodes found
 */
function getChildrenNodes(node) {
  return node && node.nodes ? node.nodes : []
}

/**
 * Get all the attributes of a riot parser node
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {Array<RiotParser.Node.Attribute>} all the attributes find
 */
function getNodeAttributes(node) {
  return node.attributes ? node.attributes : []
}

/**
 * Find all the node attributes that are not expressions
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {Array} list of all the static attributes
 */
function findStaticAttributes(node) {
  return getNodeAttributes(node).filter(attribute => !hasExpressions(attribute))
}

/**
 * Find all the node attributes that have expressions
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {Array} list of all the dynamic attributes
 */
function findDynamicAttributes(node) {
  return getNodeAttributes(node).filter(hasExpressions)
}

/**
 * True if the node has the isCustom attribute set
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true if either it's a riot component or a custom element
 */
function isCustomNode(node) {
  return !!node[IS_CUSTOM_NODE]
}

/**
 * True if the node has the isVoid attribute set
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true if the node is self closing
 */
function isVoidNode(node) {
  return !!node[IS_VOID_NODE]
}

/**
 * True if the riot parser did find a tag node
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only for the tag nodes
 */
function isTagNode(node) {
  return node.type === riotParser.nodeTypes.TAG
}

/**
 * True if the riot parser did find a text node
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only for the text nodes
 */
function isTextNode(node) {
  return node.type === riotParser.nodeTypes.TEXT
}

/**
 * True if the node parsed is the root one
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only for the root nodes
 */
function isRootNode(node) {
  return node.isRoot
}

/**
 * True if the node is an attribute and its name is "value"
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only for value attribute nodes
 */
function isValueAttribute(node) {
  return node.name === 'value'
}

/**
 * True if the node is an attribute and a DOM handler
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only for dom listener attribute nodes
 */
const isEventAttribute = (() => {
  const EVENT_ATTR_RE = /^on/;
  return node => EVENT_ATTR_RE.test(node.name)
})();

/**
 * True if the node has expressions or expression attributes
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} ditto
 */
function hasExpressions(node) {
  return !!(
    node.expressions ||
    // has expression attributes
    (getNodeAttributes(node).some(attribute => hasExpressions(attribute))) ||
    // has child text nodes with expressions
    (node.nodes && node.nodes.some(node => isTextNode(node) && hasExpressions(node)))
  )
}

/**
 * Convert all the node static attributes to strings
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {string} all the node static concatenated as string
 */
function staticAttributesToString(node) {
  return findStaticAttributes(node)
    .map(attribute => attribute[IS_BOOLEAN_ATTRIBUTE] || !attribute.value ?
      attribute.name :
      `${attribute.name}="${attribute.value}"`
    ).join(' ')
}


/**
 * Convert a riot parser opening node into a string
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {string} the node as string
 */
function nodeToString(node) {
  const attributes = staticAttributesToString(node);

  switch(true) {
  case isTagNode(node):
    return `<${node.name}${attributes ? ` ${attributes}` : ''}${isVoidNode(node) ? '/' : ''}>`
  case isTextNode(node):
    return hasExpressions(node) ? TEXT_NODE_EXPRESSION_PLACEHOLDER : node.text
  default:
    return ''
  }
}

/**
 * Close an html node
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {string} the closing tag of the html tag node passed to this function
 */
function closeTag(node) {
  return node.name ? `</${node.name}>` : ''
}

/**
 * True if the node has not expression set nor bindings directives
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only if it's a static node that doesn't need bindings or expressions
 */
function isStaticNode(node) {
  return [
    hasExpressions,
    findEachAttribute,
    findIfAttribute,
    isCustomNode
  ].every(test => !test(node))
}

/**
 * True if the node is a directive having its own template
 * @param   {RiotParser.Node} node - riot parser node
 * @returns {boolean} true only for the IF EACH and TAG bindings
 */
function hasItsOwnTemplate(node) {
  return [
    findEachAttribute,
    findIfAttribute,
    isCustomNode
  ].some(test => test(node))
}


/**
 * Create a selector that will be used to find the node via dom-bindings
 * @param   {number} id - temporary variable that will be increased anytime this function will be called
 * @returns {string} selector attribute needed to bind a riot expression
 */
const createBindingSelector = (function createSelector(id = 0) {
  return () => `${BINDING_SELECTOR_PREFIX}${id++}`
}());

/**
 * Simple clone deep function, do not use it for classes or recursive objects!
 * @param   {*} source - possibily an object to clone
 * @returns {*} the object we wanted to clone
 */
function cloneDeep(source) {
  return JSON.parse(JSON.stringify(source))
}

/**
 * Create a simple attribute expression
 * @param   {RiotParser.Node.Attr} sourceNode - the custom tag
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @returns {AST.Node} object containing the expression binding keys
 */
function createAttributeExpression(sourceNode, sourceFile, sourceCode) {
  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(EXPRESSION_TYPES),
        builders.identifier(ATTRIBUTE_EXPRESSION_TYPE),
        false
      ),
    ),
    simplePropertyNode(BINDING_NAME_KEY, builders.literal(sourceNode.name)),
    simplePropertyNode(
      BINDING_EVALUATE_KEY,
      hasExpressions(sourceNode) ?
        // dynamic attribute
        toScopedFunction(sourceNode.expressions[0], sourceFile, sourceCode) :
        // static attribute
        builders.functionExpression(
          null,
          [],
          builders.blockStatement([
            builders.returnStatement(builders.literal(sourceNode.value || true))
          ])
        )
    )
  ])
}

/**
 * Find the slots in the current component and group them under the same id
 * @param   {RiotParser.Node.Tag} sourceNode - the custom tag
 * @returns {Object} object containing all the slots grouped by name
 */
function groupSlots(sourceNode) {
  return sourceNode.nodes.reduce((acc, node) => {
    const slotAttribute = findSlotAttribute(node);

    if (slotAttribute) {
      acc[slotAttribute.value] = node;
    } else {
      acc.default = {
        nodes: [...getChildrenNodes(acc.default), node]
      };
    }

    return acc
  }, {
    default: null
  })
}

/**
 * Create the slot entity to pass to the riot-dom bindings
 * @param   {string} id - slot id
 * @param   {RiotParser.Node.Tag} sourceNode - slot root node
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @returns {AST.Node} ast node containing the slot object properties
 */
function buildSlot(id, sourceNode, sourceFile, sourceCode) {
  const cloneNode = {
    ...sourceNode,
    // avoid to render the slot attribute
    attributes: getNodeAttributes(sourceNode).filter(attribute => attribute.name !== SLOT_ATTRIBUTE)
  };
  const [html, bindings] = build(cloneNode, sourceFile, sourceCode);

  return builders.objectExpression([
    simplePropertyNode(BINDING_ID_KEY, builders.literal(id)),
    simplePropertyNode(BINDING_HTML_KEY, builders.literal(html)),
    simplePropertyNode(BINDING_BINDINGS_KEY, builders.arrayExpression(bindings))
  ])
}

/**
 * Find the slot attribute if it exists
 * @param   {RiotParser.Node.Tag} sourceNode - the custom tag
 * @returns {RiotParser.Node.Attr|undefined} the slot attribute found
 */
function findSlotAttribute(sourceNode) {
  return getNodeAttributes(sourceNode).find(attribute => attribute.name === SLOT_ATTRIBUTE)
}

/**
 * Transform a RiotParser.Node.Tag into a tag binding
 * @param   { RiotParser.Node.Tag } sourceNode - the custom tag
 * @param   { string } selectorAttribute - attribute needed to select the target node
 * @param   { stiring } sourceFile - source file path
 * @param   { string } sourceCode - original source
 * @returns { AST.Node } an each binding node
 */
function createTagBinding(sourceNode, selectorAttribute, sourceFile, sourceCode) {
  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(BINDING_TYPES),
        builders.identifier(TAG_BINDING_TYPE),
        false
      ),
    ),
    simplePropertyNode(BINDING_COMPONENTS_KEY, builders.memberExpression(
      builders.identifier(COMPONENTS_REGISTRY),
      builders.literal(sourceNode.name),
      true
    )),
    simplePropertyNode(BINDING_SLOTS_KEY, builders.arrayExpression([
      ...Object.entries(groupSlots(sourceNode)).map(([key, value]) => buildSlot(key, value, sourceFile, sourceCode))
    ])),
    simplePropertyNode(BINDING_ATTRIBUTES_KEY, builders.arrayExpression([
      ...cleanAttributes(sourceNode)
        .filter(attribute => attribute.name !== selectorAttribute)
        .map(attribute => createAttributeExpression(attribute, sourceFile, sourceCode))
    ])),
    ...createSelectorProperties(selectorAttribute)
  ])
}

/**
 * Transform a RiotParser.Node.Tag into an each binding
 * @param   { RiotParser.Node.Tag } sourceNode - tag containing the each attribute
 * @param   { string } selectorAttribute - attribute needed to select the target node
 * @param   { stiring } sourceFile - source file path
 * @param   { string } sourceCode - original source
 * @returns { AST.Node } an each binding node
 */
function createEachBinding(sourceNode, selectorAttribute, sourceFile, sourceCode) {
  const [ifAttribute, eachAttribute, keyAttribute] = [
    findIfAttribute,
    findEachAttribute,
    findKeyAttribute
  ].map(f => f(sourceNode));
  const mightBeARiotComponent = isCustomNode(sourceNode);
  const attributeOrNull = attribute => attribute ? toScopedFunction(getAttributeExpression(attribute), sourceFile, sourceCode) : nullNode();

  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(BINDING_TYPES),
        builders.identifier(EACH_BINDING_TYPE),
        false
      ),
    ),
    simplePropertyNode(BINDING_GET_KEY_KEY, attributeOrNull(keyAttribute)),
    simplePropertyNode(BINDING_CONDITION_KEY, attributeOrNull(ifAttribute)),
    createTemplateProperty(mightBeARiotComponent ?
      [null, [createTagBinding(sourceNode, null, sourceCode, sourceCode)]] :
      build(createRootNode(sourceNode), sourceCode, sourceCode)
    ),
    ...createSelectorProperties(selectorAttribute),
    ...compose(getEachExpressionProperties, getAttributeExpression)(eachAttribute)
  ])
}

/**
 * Transform a RiotParser.Node.Tag into an if binding
 * @param   { RiotParser.Node.Tag } sourceNode - tag containing the if attribute
 * @param   { string } selectorAttribute - attribute needed to select the target node
 * @param   { stiring } sourceFile - source file path
 * @param   { string } sourceCode - original source
 * @returns { AST.Node } an each binding node
 */
function createIfBinding(sourceNode, selectorAttribute, sourceFile, sourceCode) {
  const ifAttribute = findIfAttribute(sourceNode);
  const mightBeARiotComponent = isCustomNode(sourceNode);

  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(BINDING_TYPES),
        builders.identifier(IF_BINDING_TYPE),
        false
      ),
    ),
    simplePropertyNode(
      BINDING_EVALUATE_KEY,
      toScopedFunction(ifAttribute.expressions[0], sourceFile, sourceCode)
    ),
    ...createSelectorProperties(selectorAttribute),
    createTemplateProperty(mightBeARiotComponent ?
      [null, [createTagBinding(sourceNode, null, sourceCode, sourceCode)]] :
      build(createRootNode(sourceNode), sourceCode, sourceCode)
    )
  ])
}

/**
 * Create a simple event expression
 * @param   {RiotParser.Node.Attr} sourceNode - attribute containing the event handlers
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @returns {AST.Node} object containing the expression binding keys
 */
function createEventExpression(sourceNode, sourceFile, sourceCode) {
  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(EXPRESSION_TYPES),
        builders.identifier(EVENT_EXPRESSION_TYPE),
        false
      ),
    ),
    simplePropertyNode(BINDING_NAME_KEY, builders.literal(sourceNode.name)),
    simplePropertyNode(
      BINDING_EVALUATE_KEY,
      toScopedFunction(sourceNode.expressions[0], sourceFile, sourceCode)
    )
  ])
}

/**
 * Create a text expression
 * @param   {RiotParser.Node.Text} sourceNode - text node to parse
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @param   {number} childNodeIndex - position of the child text node in its parent children nodes
 * @returns {AST.Node} object containing the expression binding keys
 */
function createTextExpression(sourceNode, sourceFile, sourceCode, childNodeIndex) {
  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(EXPRESSION_TYPES),
        builders.identifier(TEXT_EXPRESSION_TYPE),
        false
      ),
    ),
    simplePropertyNode(
      BINDING_CHILD_NODE_INDEX_KEY,
      builders.literal(childNodeIndex)
    ),
    simplePropertyNode(
      BINDING_EVALUATE_KEY,
      toScopedFunction({
        start: sourceNode.start,
        end: sourceNode.end,
        text: mergeNodeExpressions(sourceNode)
      }, sourceFile, sourceCode)
    )
  ])
}

function createValueExpression(sourceNode, sourceFile, sourceCode) {
  return builders.objectExpression([
    simplePropertyNode(BINDING_TYPE_KEY,
      builders.memberExpression(
        builders.identifier(EXPRESSION_TYPES),
        builders.identifier(VALUE_EXPRESSION_TYPE),
        false
      ),
    ),
    simplePropertyNode(
      BINDING_EVALUATE_KEY,
      toScopedFunction(sourceNode.expressions[0], sourceFile, sourceCode)
    )
  ])
}

function createExpression(sourceNode, sourceFile, sourceCode, childNodeIndex) {
  switch (true) {
  case isTextNode(sourceNode):
    return createTextExpression(sourceNode, sourceFile, sourceCode, childNodeIndex)
  case isValueAttribute(sourceNode):
    return createValueExpression(sourceNode, sourceFile, sourceCode)
  case isEventAttribute(sourceNode):
    return createEventExpression(sourceNode, sourceFile, sourceCode)
  default:
    return createAttributeExpression(sourceNode, sourceFile, sourceCode)
  }
}

/**
 * Create the attribute expressions
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @returns {Array} array containing all the attribute expressions
 */
function createAttributeExpressions(sourceNode, sourceFile, sourceCode) {
  return findDynamicAttributes(sourceNode)
    .map(attribute => createExpression(attribute, sourceFile, sourceCode))
}

/**
 * Create the text node expressions
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @returns {Array} array containing all the text node expressions
 */
function createTextNodeExpressions(sourceNode, sourceFile, sourceCode) {
  const childrenNodes = getChildrenNodes(sourceNode);

  return childrenNodes
    .filter(isTextNode)
    .filter(hasExpressions)
    .map(node => createExpression(
      node,
      sourceFile,
      sourceCode,
      childrenNodes.indexOf(node)
    ))
}

/**
 * Add a simple binding to a riot parser node
 * @param   { RiotParser.Node.Tag } sourceNode - tag containing the if attribute
 * @param   { string } selectorAttribute - attribute needed to select the target node
 * @param   { stiring } sourceFile - source file path
 * @param   { string } sourceCode - original source
 * @returns { AST.Node } an each binding node
 */
function createSimpleBinding(sourceNode, selectorAttribute, sourceFile, sourceCode) {
  return builders.objectExpression([
    ...createSelectorProperties(selectorAttribute),
    simplePropertyNode(
      BINDING_EXPRESSIONS_KEY,
      builders.arrayExpression([
        ...createTextNodeExpressions(sourceNode, sourceFile, sourceCode),
        ...createAttributeExpressions(sourceNode, sourceFile, sourceCode)
      ])
    )
  ])
}

const BuildingState = Object.freeze({
  html: [],
  bindings: [],
  parent: null
});

/**
 * Nodes having bindings should be cloned and new selector properties should be added to them
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {string} bindingsSelector - temporary string to identify the current node
 * @returns {RiotParser.Node} the original node parsed having the new binding selector attribute
 */
function createBindingsTag(sourceNode, bindingsSelector) {
  if (!bindingsSelector) return sourceNode

  return {
    ...sourceNode,
    // inject the selector bindings into the node attributes
    attributes: [{
      name: bindingsSelector
    }, ...getNodeAttributes(sourceNode)]
  }
}

/**
 * Create a generic dynamic node (text or tag) and generate its bindings
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @param   {BuildingState} state - state representing the current building tree state during the recursion
 * @returns {Array} array containing the html output and bindings for the current node
 */
function createDynamicNode(sourceNode, sourceFile, sourceCode, state) {
  switch (true) {
  case isTextNode(sourceNode):
    // text nodes will not have any bindings
    return [nodeToString(sourceNode), []]
  default:
    return createTagWithBindings(sourceNode, sourceFile, sourceCode, state)
  }
}

/**
 * Create only a dynamic tag node with generating a custom selector and its bindings
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @param   {BuildingState} state - state representing the current building tree state during the recursion
 * @returns {Array} array containing the html output and bindings for the current node
 */
function createTagWithBindings(sourceNode, sourceFile, sourceCode) {
  const bindingsSelector = isRootNode(sourceNode) ? null : createBindingSelector();
  const cloneNode = createBindingsTag(sourceNode, bindingsSelector);
  const tagOpeningHTML = nodeToString(cloneNode);

  switch(true) {
  // EACH bindings have prio 1
  case hasEachAttribute(cloneNode):
    return [tagOpeningHTML, [createEachBinding(cloneNode, bindingsSelector, sourceFile, sourceCode)]]
  // IF bindings have prio 2
  case hasIfAttribute(cloneNode):
    return [tagOpeningHTML, [createIfBinding(cloneNode, bindingsSelector, sourceFile, sourceCode)]]
  // TAG bindings have prio 3
  case isCustomNode(cloneNode):
    return [tagOpeningHTML, [createTagBinding(cloneNode, bindingsSelector, sourceFile, sourceCode)]]
  // this node has expressions bound to it
  default:
    return [tagOpeningHTML, [createSimpleBinding(cloneNode, bindingsSelector, sourceFile, sourceCode)]]
  }
}

/**
 * Parse a node trying to extract its template and bindings
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @param   {BuildingState} state - state representing the current building tree state during the recursion
 * @returns {Array} array containing the html output and bindings for the current node
 */
function parseNode(sourceNode, sourceFile, sourceCode, state) {
  // static nodes have no bindings
  if (isStaticNode(sourceNode)) return [nodeToString(sourceNode), []]
  return createDynamicNode(sourceNode, sourceFile, sourceCode, state)
}

/**
 * Build the template and the bindings
 * @param   {RiotParser.Node} sourceNode - any kind of node parsed via riot parser
 * @param   {stiring} sourceFile - source file path
 * @param   {string} sourceCode - original source
 * @param   {BuildingState} state - state representing the current building tree state during the recursion
 * @returns {Array} array containing the html output and the dom bindings
 */
function build(
  sourceNode,
  sourceFile,
  sourceCode,
  state
) {
  if (!sourceNode) panic('Something went wrong with your tag DOM parsing, your tag template can\'t be created');

  const [nodeHTML, nodeBindings] = parseNode(sourceNode, sourceFile, sourceCode, state);
  const childrenNodes = getChildrenNodes(sourceNode);
  const currentState = { ...cloneDeep(BuildingState), ...state };

  // mutate the original arrays
  currentState.html.push(...nodeHTML);
  currentState.bindings.push(...nodeBindings);

  // do recursion if
  // this tag has children and it has no special directives bound to it
  if (childrenNodes.length && !hasItsOwnTemplate(sourceNode)) {
    childrenNodes.forEach(node => build(node, sourceFile, sourceCode, { parent: sourceNode, ...currentState }));
  }

  // close the tag if it's not a void one
  if (isTagNode(sourceNode) && !isVoidNode(sourceNode)) {
    currentState.html.push(closeTag(sourceNode));
  }

  return [
    currentState.html.join(''),
    currentState.bindings
  ]
}

/**
 * Extend the AST adding the new template property containing our template call to render the component
 * @param   { Object } ast - current output ast
* @param   { stiring } sourceFile - source file path
 * @param   { string } sourceCode - original source
 * @param   { Object } sourceNode - node generated by the riot compiler
 * @returns { Object } the output ast having the "template" key
 */
function extendTemplateProperty(ast, sourceFile, sourceCode, sourceNode) {
  types.visit(ast, {
    visitProperty(path) {
      if (path.value.key.name === TAG_TEMPLATE_PROPERTY) {
        path.value.value = builders.functionExpression(
          null,
          [
            TEMPLATE_FN,
            EXPRESSION_TYPES,
            BINDING_TYPES,
            COMPONENTS_REGISTRY
          ].map(builders.identifier),
          builders.blockStatement([
            builders.returnStatement(
              callTemplateFunction(
                ...build(
                  createRootNode(sourceNode),
                  sourceFile,
                  sourceCode
                )
              )
            )
          ])
        );

        return false
      }

      this.traverse(path);
    }
  });

  return ast
}

/**
 * Generate the component template logic
 * @param   { Object } sourceNode - node generated by the riot compiler
 * @param   { string } source - original component source code
 * @param   { Object } options - user options
 * @param   { Output } output - current compiler output
 * @returns { Promise<Output> } - enhanced output with the result of the current generator
 */
async function template(sourceNode, source, options, { ast, map }) {
  const output = extendTemplateProperty(ast, options.file, source, sourceNode);

  return { ast: output, map, code: recast.print(output).code }
}

/**
 * Create the initial output
 * @param { Sourcemap } map - initial sourcemap
 * @returns { Output } containing the initial code and  AST
 *
 * @example
 * // the output represents the following string in AST
 */
function createInitialInput(map) {
  const code = `export default { ${TAG_CSS_PROPERTY}: null, ${TAG_LOGIC_PROPERTY}: null, ${TAG_TEMPLATE_PROPERTY}: null }`;
  return {
    ast: recast.parse(code),
    code,
    map
  }
}

/**
 * Generate the output code source together with the sourcemap
 * @param   { string } source - source code of the tag we will need to compile
 * @param { string } options - compiling options
 * @returns { Promise<Output> } object containing output code and source map
 */
async function compile(source, options = {
  template: 'default',
  file: '[unknown-source-file]'
}) {
  const { code, map } = await execute$1('template', 'default', options, source);
  const { template: template$$1, css: css$$1, javascript: javascript$$1 } = riotParser__default(options).parse(code).output;

  return ruit(createInitialInput(map),
    hookGenerator(css, css$$1, code, options),
    hookGenerator(javascript, javascript$$1, code, options),
    hookGenerator(template, template$$1, code, options),
    ({ ast }) => recast.prettyPrint(ast),
    execute,
  )
}

/**
 * Prepare the riot parser node transformers
 * @param   { Function } transformer - transformer function
 * @param   { Object } sourceNode - riot parser node
 * @param   { string } source - component source code
 * @param   { Object } options - compiling options
 * @returns { Promise<Output> } object containing output code and source map
 */
function hookGenerator(transformer, sourceNode, source, options) {
  if (!sourceNode) {
    return result => result
  }

  return curry(transformer)(sourceNode, source, options)
}

// This function can be used to register new preprocessors
// a preprocessor can target either only the css or javascript nodes
// or the complete tag source file ('template')
const registerPreprocessor = register$1;

// This function can allow you to register postprocessors that will parse the output code
// here we can run prettifiers, eslint fixes...
const registerPostprocessor = register;

exports.createInitialInput = createInitialInput;
exports.compile = compile;
exports.registerPreprocessor = registerPreprocessor;
exports.registerPostprocessor = registerPostprocessor;
