import composeSourcemaps from './utils/compose-sourcemaps'
import { createOutput } from './transformer'
import panic from './utils/panic'

export const postprocessors = new Set()

/**
 * Register a postprocessor that will be used after the parsing and compilation of the riot tags
 * @param { Function } postprocessor - transformer that will receive the output code ans sourcemap
 * @returns { Set } the postprocessors collection
 */
export function register(postprocessor) {
  if (postprocessors.has(postprocessor)) {
    panic(`This postprocessor "${postprocessor.name || postprocessor.toString()}" was already registered`)
  }

  postprocessors.add(postprocessor)

  return postprocessors
}

/**
 * Unregister a postprocessor
 * @param { Function } postprocessor - possibly a postprocessor previously registered
 * @returns { Set } the postprocessors collection
 */
export function unregister(postprocessor) {
  if (!postprocessors.has(postprocessor)) {
    panic(`This postprocessor "${postprocessor.name || postprocessor.toString()}" was never registered`)
  }

  postprocessors.delete(postprocessor)

  return postprocessors
}

/**
 * Exec all the postprocessors in sequence combining the sourcemaps generated
 * @param   { Output } compilerOutput - output generated by the compiler
 * @param   { Object } options - user options received by the compiler
 * @returns { Promise<Output> } object containing output code and source map
 */
export async function execute(compilerOutput, options) {
  return Array.from(postprocessors).reduce(async function(acc, postprocessor) {
    const { code, map } = await acc
    const output = await postprocessor(code, options)

    return {
      code: output.code,
      map: composeSourcemaps(output.map, map)
    }
  }, Promise.resolve(createOutput(compilerOutput, options)))
}