"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppApi = void 0;
const metadata_1 = require("../../definition/metadata");
const api_1 = require("../../definition/api");
const methods = [
    'get',
    'post',
    'put',
    'delete',
    'head',
    'options',
    'patch',
];
class AppApi {
    constructor(app, api, endpoint) {
        this.app = app;
        this.api = api;
        this.endpoint = endpoint;
        this.appId = app.getID();
        switch (this.api.visibility) {
            case api_1.ApiVisibility.PUBLIC:
                this.basePath = `/api/apps/public/${app.getID()}`;
                break;
            case api_1.ApiVisibility.PRIVATE:
                this.basePath = `/api/apps/private/${app.getID()}/${app.getStorageItem()._id}`;
                this.hash = app.getStorageItem()._id;
                break;
        }
        this.computedPath = `${this.basePath}/${endpoint.path}`;
        this.implementedMethods = methods.filter((m) => typeof endpoint[m] === 'function');
    }
    runExecutor(request, logStorage, accessors) {
        return __awaiter(this, void 0, void 0, function* () {
            const { path } = this.endpoint;
            const method = request.method;
            // Ensure the api has the property before going on
            if (typeof this.endpoint[method] !== 'function') {
                return;
            }
            if (!this.validateVisibility(request)) {
                return {
                    status: 404,
                };
            }
            if (!this.validateSecurity(request)) {
                return {
                    status: 401,
                };
            }
            const endpoint = {
                basePath: this.basePath,
                fullPath: this.computedPath,
                appId: this.appId,
                hash: this.hash,
            };
            const logger = this.app.setupLogger(metadata_1.AppMethod._API_EXECUTOR);
            logger.debug(`${path}'s ${method} is being executed...`, request);
            const runCode = `module.exports = endpoint.${method}.apply(endpoint, args)`;
            try {
                const result = yield this.app.getRuntime().runInSandbox(runCode, {
                    endpoint: this.endpoint,
                    args: [
                        request,
                        endpoint,
                        accessors.getReader(this.app.getID()),
                        accessors.getModifier(this.app.getID()),
                        accessors.getHttp(this.app.getID()),
                        accessors.getPersistence(this.app.getID()),
                    ],
                });
                logger.debug(`${path}'s ${method} was successfully executed.`);
                logStorage.storeEntries(this.app.getID(), logger);
                return result;
            }
            catch (e) {
                logger.error(e);
                logger.debug(`${path}'s ${method} was unsuccessful.`);
                logStorage.storeEntries(this.app.getID(), logger);
                throw e;
            }
        });
    }
    validateVisibility(request) {
        if (this.api.visibility === api_1.ApiVisibility.PUBLIC) {
            return true;
        }
        if (this.api.visibility === api_1.ApiVisibility.PRIVATE) {
            return this.app.getStorageItem()._id === request.privateHash;
        }
        return false;
    }
    validateSecurity(request) {
        if (this.api.security === api_1.ApiSecurity.UNSECURE) {
            return true;
        }
        return false;
    }
}
exports.AppApi = AppApi;

//# sourceMappingURL=AppApi.js.map
