'use strict';

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var fs = _interopDefault(require('fs'));
var path = _interopDefault(require('path'));
var colors = require('colors');
var replace = _interopDefault(require('replace'));
var MagicString = _interopDefault(require('magic-string'));
var rollupPluginutils = require('rollup-pluginutils');

var moduleIdRegex = /moduleId\s*:(.*)/g;
var componentRegex = /@Component\(\s?{([\s\S]*)}\s?\)$|type:\s?Component,\s?args:\s?\[\s?{([\s\S]*)},\s?\]/gm;
var commentRegex = /\/\*[\s\S]*?\*\/|([^\\:]|^)\/\/.*$/gm; // http://www.regextester.com/?fam=96247
var templateUrlRegex = /templateUrl\s*:(.*)/g;
var styleUrlsRegex = /styleUrls\s*:(\s*\[[\s\S]*?\])/g; // http://www.regextester.com/?fam=98594
var stringRegex = /(['"`])((?:[^\\]\\\1|.)*?)\1/g;

function insertText(str, dir, preprocessor, processFilename, sourceType) {
  if ( preprocessor === void 0 ) preprocessor = function (res) { return res; };
  if ( processFilename === void 0 ) processFilename = false;
  if ( sourceType === void 0 ) sourceType = 'ts';

  var quoteChar = sourceType === 'ts' ? '`' : '"';
  return str.replace(stringRegex, function (match, quote, url) {
    var includePath = path.join(dir, url);
    if (processFilename) {
      return quoteChar + preprocessor(includePath) + quoteChar;
    }
    var text = fs.readFileSync(includePath).toString();
    return quoteChar + preprocessor(text, includePath) + quoteChar;
  });
}

function angular(options) {
  if ( options === void 0 ) options = {};

  options.preprocessors = options.preprocessors || {}; // set default preprocessors to `{}`
  options.replace = (typeof options.replace === 'boolean') ? options.replace  : true; // set default replace to `true`

  // ignore @angular/** modules
  options.exclude = options.exclude || [];
  if (typeof options.exclude === 'string' || options.exclude instanceof String) {
    options.exclude = [options.exclude];
  }
  if (options.exclude.indexOf('node_modules/@angular/**') === -1) {
    options.exclude.push('node_modules/@angular/**');
  }

  var filter = rollupPluginutils.createFilter(options.include, options.exclude);

  return {
    name: 'angular',
    transform: function transform(source, map) {
      if (!filter(map)) { return; }
      // replace comments in source
      source = source.replace(commentRegex, '');
      // use MagicString
      var magicString = new MagicString(source);
      // get dir from `map`
      var dir = path.parse(map).dir;
      // get file extension from `map`
      var fileExt = map.split('.').pop();

      var hasReplacements = false;
      var match;
      var start, end, replacement;

      while ((match = componentRegex.exec(source)) !== null) {
        start = match.index;
        end = start + match[0].length;

        replacement = match[0]
          .replace(templateUrlRegex, function (match, url) {
            hasReplacements = true;
            var toReplace = 'template:' + insertText(url, dir, options.preprocessors.template, options.processFilename, options.sourcetype);
            if (fileExt === 'js' && options.replace === true) {
              /* replace templateUrl in files generated by ngc */
              replace({
                regex: match,
                replacement: toReplace,
                paths: [map],
                recursive: true,
                silent: true,
              });
              console.info(("templateUrl in file " + map + " has been changed from " + (colors.green(match)) + "  to " + (colors.green(toReplace))));
            }
            return toReplace;
          })
          .replace(styleUrlsRegex, function (match, urls) {
            hasReplacements = true;
            var toReplace = 'styles:' + insertText(urls, dir, options.preprocessors.style, options.processFilename, options.sourcetype);
            /* replace styles in files generated by ngc */
            if (fileExt === 'js' && options.replace === true) {
              replace({
                regex: styleUrlsRegex,
                replacement: toReplace,
                paths: [map],
                recursive: true,
                silent: true,
              });
              console.info(("styleUrls in file " + map + " has been changed from " + (colors.green(match)) + " to " + (colors.green(toReplace))));
            }
            return toReplace;
          })
          .replace(moduleIdRegex, function (match, moduleId) {
            hasReplacements = true;
            return '';
          });
        if (hasReplacements) {
          magicString.overwrite(start, end, replacement);
        }
      }

      if (!hasReplacements) {
        return null;
      }

      var result = { code: magicString.toString() };

      if (options.sourceMap !== false) {
        result.map = magicString.generateMap({ hires: true });
      }
      return result;
    }
  };
}

module.exports = angular;
