'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var path = require('path');
var estreeWalker = require('estree-walker');
var mm = _interopDefault(require('micromatch'));

function addExtension ( filename, ext ) {
	if ( ext === void 0 ) ext = '.js';

	if ( !path.extname( filename ) ) { filename += ext; }
	return filename;
}

var blockDeclarations = {
	'const': true,
	'let': true
};

var extractors = {
	Literal: function Literal ( names, param ) {
		names.push( param.value );
	},

	Identifier: function Identifier ( names, param ) {
		names.push( param.name );
	},

	ObjectPattern: function ObjectPattern ( names, param ) {
		param.properties.forEach( function (prop) {
			if ( prop.type === 'RestElement' ) {
				extractors.RestElement( names, prop );
			} else {
				extractors[ (prop.value || prop.key).type ]( names, prop.value || prop.key );
			}
		});
	},

	ArrayPattern: function ArrayPattern ( names, param ) {
		param.elements.forEach( function (element) {
			if ( element ) { extractors[ element.type ]( names, element ); }
		});
	},

	RestElement: function RestElement ( names, param ) {
		extractors[ param.argument.type ]( names, param.argument );
	},

	AssignmentPattern: function AssignmentPattern ( names, param ) {
		return extractors[ param.left.type ]( names, param.left );
	}
};

function extractNames ( param ) {
	var names = [];

	extractors[ param.type ]( names, param );
	return names;
}

var Scope = function Scope ( options ) {
	var this$1 = this;

	options = options || {};

	this.parent = options.parent;
	this.isBlockScope = !!options.block;

	this.declarations = Object.create( null );

	if ( options.params ) {
		options.params.forEach( function (param) {
			extractNames( param ).forEach( function (name) {
				this$1.declarations[ name ] = true;
			});
		});
	}
};

Scope.prototype.addDeclaration = function addDeclaration ( node, isBlockDeclaration, isVar ) {
		var this$1 = this;

	if ( !isBlockDeclaration && this.isBlockScope ) {
		// it's a `var` or function node, and this
		// is a block scope, so we need to go up
		this.parent.addDeclaration( node, isBlockDeclaration, isVar );
	} else if ( node.id ) {
		extractNames( node.id ).forEach( function (name) {
			this$1.declarations[ name ] = true;
		});
	}
};

Scope.prototype.contains = function contains ( name ) {
	return this.declarations[ name ] ||
		       ( this.parent ? this.parent.contains( name ) : false );
};


function attachScopes ( ast, propertyName ) {
	if ( propertyName === void 0 ) propertyName = 'scope';

	var scope = new Scope();

	estreeWalker.walk( ast, {
		enter: function enter ( node, parent ) {
			// function foo () {...}
			// class Foo {...}
			if ( /(Function|Class)Declaration/.test( node.type ) ) {
				scope.addDeclaration( node, false, false );
			}

			// var foo = 1
			if ( node.type === 'VariableDeclaration' ) {
				var isBlockDeclaration = blockDeclarations[ node.kind ];

				node.declarations.forEach( function (declaration) {
					scope.addDeclaration( declaration, isBlockDeclaration, true );
				});
			}

			var newScope;

			// create new function scope
			if ( /Function/.test( node.type ) ) {
				newScope = new Scope({
					parent: scope,
					block: false,
					params: node.params
				});

				// named function expressions - the name is considered
				// part of the function's scope
				if ( node.type === 'FunctionExpression' && node.id ) {
					newScope.addDeclaration( node, false, false );
				}
			}

			// create new block scope
			if ( node.type === 'BlockStatement' && !/Function/.test( parent.type ) ) {
				newScope = new Scope({
					parent: scope,
					block: true
				});
			}

			// catch clause has its own block scope
			if ( node.type === 'CatchClause' ) {
				newScope = new Scope({
					parent: scope,
					params: [ node.param ],
					block: true
				});
			}

			if ( newScope ) {
				Object.defineProperty( node, propertyName, {
					value: newScope,
					configurable: true
				});

				scope = newScope;
			}
		},
		leave: function leave ( node ) {
			if ( node[ propertyName ] ) { scope = scope.parent; }
		}
	});

	return scope;
}

function ensureArray ( thing ) {
	if ( Array.isArray( thing ) ) { return thing; }
	if ( thing == undefined ) { return []; }
	return [ thing ];
}

function createFilter ( include, exclude ) {
	var getMatcher = function (id) { return ( isRegexp( id ) ? id : { test: mm.matcher( path.resolve( id ).split ( path.sep ).join( '/' ) ) } ); };
	include = ensureArray( include ).map( getMatcher );
	exclude = ensureArray( exclude ).map( getMatcher );

	return function ( id ) {

		if ( typeof id !== 'string' ) { return false; }
		if ( /\0/.test( id ) ) { return false; }

		id = id.split( path.sep ).join( '/' );

		for ( var i = 0; i < exclude.length; ++i ) {
			var matcher = exclude[i];
			if ( matcher.test( id ) ) { return false; }
		}

		for ( var i$1 = 0; i$1 < include.length; ++i$1 ) {
			var matcher$1 = include[i$1];
			if ( matcher$1.test( id ) ) { return true; }
		}

		return !include.length;
	};
}

function isRegexp ( val ) {
	return val instanceof RegExp;
}

var reservedWords = 'break case class catch const continue debugger default delete do else export extends finally for function if import in instanceof let new return super switch this throw try typeof var void while with yield enum await implements package protected static interface private public'.split( ' ' );
var builtins = 'arguments Infinity NaN undefined null true false eval uneval isFinite isNaN parseFloat parseInt decodeURI decodeURIComponent encodeURI encodeURIComponent escape unescape Object Function Boolean Symbol Error EvalError InternalError RangeError ReferenceError SyntaxError TypeError URIError Number Math Date String RegExp Array Int8Array Uint8Array Uint8ClampedArray Int16Array Uint16Array Int32Array Uint32Array Float32Array Float64Array Map Set WeakMap WeakSet SIMD ArrayBuffer DataView JSON Promise Generator GeneratorFunction Reflect Proxy Intl'.split( ' ' );

var blacklisted = Object.create( null );
reservedWords.concat( builtins ).forEach( function (word) { return blacklisted[ word ] = true; } );

function makeLegalIdentifier ( str ) {
	str = str
		.replace( /-(\w)/g, function ( _, letter ) { return letter.toUpperCase(); } )
		.replace( /[^$_a-zA-Z0-9]/g, '_' );

	if ( /\d/.test( str[0] ) || blacklisted[ str ] ) { str = "_" + str; }

	return str;
}

function serializeArray (arr, indent, baseIndent) {
	var output = '[';
	var separator = indent ? '\n' + baseIndent + indent : '';
	for (var i = 0; i < arr.length; i++) {
		var key = arr[i];
		output += "" + (i > 0 ? ',' : '') + separator + (serialize(key, indent, baseIndent + indent));
	}
	return output + (indent ? '\n' + baseIndent : '') + "]";
}

function serializeObject (obj, indent, baseIndent) {
	var output = '{';
	var separator = indent ? '\n' + baseIndent + indent : '';
	var keys = Object.keys(obj);
	for (var i = 0; i < keys.length; i++) {
		var key = keys[i];
		var stringKey = makeLegalIdentifier(key) === key ? key : JSON.stringify(key);
		output += "" + (i > 0 ? ',' : '') + separator + stringKey + ":" + (indent ? ' ' : '') + (serialize(obj[key], indent, baseIndent + indent));
	}
	return output + (indent ? '\n' + baseIndent : '') + "}";
}

function serialize (obj, indent, baseIndent) {
	if (obj === Infinity)
		{ return 'Infinity'; }
	if (obj instanceof Date)
		{ return 'new Date(' + obj.getTime() + ')'; }
	if (obj instanceof RegExp)
		{ return obj.toString(); }
	if (typeof obj === 'number' && isNaN(obj))
		{ return 'NaN'; }
	if (Array.isArray(obj))
		{ return serializeArray(obj, indent, baseIndent); }
	if (obj === null)
		{ return 'null'; }
	if (typeof obj === 'object')
		{ return serializeObject(obj, indent, baseIndent); }
	return JSON.stringify(obj);
}

// convert data object into separate named exports (and default)
function dataToNamedExports (data, options) {
	if ( options === void 0 ) options = {};

	var t = options.compact ? '' : 'indent' in options ? options.indent : '\t';
	var _ = options.compact ? '' : ' ';
	var n = options.compact ? '' : '\n';
	var declarationType = options.preferConst ? 'const' : 'var';
	
	if (options.namedExports === false || typeof data !== 'object' || Array.isArray(data) || data === null)
		{ return ("export default" + _ + (serialize( data, options.compact ? null : t, '' )) + ";"); }

	var namedExportCode = '';
	var defaultExportRows = [];
	var dataKeys = Object.keys(data);
	for (var i = 0; i < dataKeys.length; i++) {
		var key = dataKeys[i];
		if (key === makeLegalIdentifier(key)) {
			if (options.objectShorthand)
				{ defaultExportRows.push(key); }
			else
				{ defaultExportRows.push((key + ":" + _ + key)); }
			namedExportCode += "export " + declarationType + " " + key + _ + "=" + _ + (serialize(data[key], options.compact ? null : t, '')) + ";" + n;
		} else {
			defaultExportRows.push(((JSON.stringify(key)) + ": " + (serialize(data[key], options.compact ? null : t, ''))));
		}
	}
	return namedExportCode + "export default" + _ + "{" + n + t + (defaultExportRows.join(("," + n + t))) + n + "};" + n;
}

exports.addExtension = addExtension;
exports.attachScopes = attachScopes;
exports.createFilter = createFilter;
exports.makeLegalIdentifier = makeLegalIdentifier;
exports.dataToEsm = dataToNamedExports;
