
// We are modularizing this manually because the current modularize setting in Emscripten has some issues:
// https://github.com/kripken/emscripten/issues/5820
// In addition, When you use emcc's modularization, it still expects to export a global object called `Module`,
// which is able to be used/called before the WASM is loaded.
// The modularization below exports a promise that loads and resolves to the actual sql.js module.
// That way, this module can't be used before the WASM is finished loading.

// We are going to define a function that a user will call to start loading initializing our Sql.js library
// However, that function might be called multiple times, and on subsequent calls, we don't actually want it to instantiate a new instance of the Module
// Instead, we want to return the previously loaded module

// TODO: Make this not declare a global if used in the browser
var initSqlJsPromise = undefined;

var initSqlJs = function (moduleConfig) {

    if (initSqlJsPromise){
      return initSqlJsPromise;
    }
    // If we're here, we've never called this function before
    initSqlJsPromise = new Promise((resolveModule, reject) => {

        // We are modularizing this manually because the current modularize setting in Emscripten has some issues:
        // https://github.com/kripken/emscripten/issues/5820

        // The way to affect the loading of emcc compiled modules is to create a variable called `Module` and add
        // properties to it, like `preRun`, `postRun`, etc
        // We are using that to get notified when the WASM has finished loading.
        // Only then will we return our promise

        // If they passed in a moduleConfig object, use that
        // Otherwise, initialize Module to the empty object
        var Module = typeof moduleConfig !== 'undefined' ? moduleConfig : {};

        // EMCC only allows for a single onAbort function (not an array of functions)
        // So if the user defined their own onAbort function, we remember it and call it
        var originalOnAbortFunction = Module['onAbort'];
        Module['onAbort'] = function (errorThatCausedAbort) {
            reject(new Error(errorThatCausedAbort));
            if (originalOnAbortFunction){
              originalOnAbortFunction(errorThatCausedAbort);
            }
        };

        Module['postRun'] = Module['postRun'] || [];
        Module['postRun'].push(function () {
            // When Emscripted calls postRun, this promise resolves with the built Module
            resolveModule(Module);
        });

        // There is a section of code in the emcc-generated code below that looks like this:
        // (Note that this is lowercase `module`)
        // if (typeof module !== 'undefined') {
        //     module['exports'] = Module;
        // }
        // When that runs, it's going to overwrite our own modularization export efforts in shell-post.js!
        // The only way to tell emcc not to emit it is to pass the MODULARIZE=1 or MODULARIZE_INSTANCE=1 flags,
        // but that carries with it additional unnecessary baggage/bugs we don't want either.
        // So, we have three options:
        // 1) We undefine `module`
        // 2) We remember what `module['exports']` was at the beginning of this function and we restore it later
        // 3) We write a script to remove those lines of code as part of the Make process.
        //
        // Since those are the only lines of code that care about module, we will undefine it. It's the most straightforward
        // of the options, and has the side effect of reducing emcc's efforts to modify the module if its output were to change in the future.
        // That's a nice side effect since we're handling the modularization efforts ourselves
        module = undefined;
        
        // The emcc-generated code and shell-post.js code goes below,
        // meaning that all of it runs inside of this promise. If anything throws an exception, our promise will abort
// Copyright 2010 The Emscripten Authors.  All rights reserved.
// Emscripten is available under two separate licenses, the MIT license and the
// University of Illinois/NCSA Open Source License.  Both these licenses can be
// found in the LICENSE file.

// The Module object: Our interface to the outside world. We import
// and export values on it. There are various ways Module can be used:
// 1. Not defined. We create it here
// 2. A function parameter, function(Module) { ..generated code.. }
// 3. pre-run appended it, var Module = {}; ..generated code..
// 4. External script tag defines var Module.
// We need to check if Module already exists (e.g. case 3 above).
// Substitution will be replaced with actual code on later stage of the build,
// this way Closure Compiler will not mangle it (e.g. case 4. above).
// Note that if you want to run closure, and also to use Module
// after the generated code, you will need to define   var Module = {};
// before the code. Then that object will be used in the code, and you
// can continue to use Module afterwards as well.
var Module = typeof Module !== 'undefined' ? Module : {};

// --pre-jses are emitted after the Module integration code, so that they can
// refer to Module (if they choose; they can also define Module)


// Wait for preRun to run, and then finish our initialization
var runCompiledCode = (function() {// Generated by CoffeeScript 1.12.7
var Database, NULL, RegisterExtensionFunctions, SQLite, Statement, apiTemp, i, sqlite3_bind_blob, sqlite3_bind_double, sqlite3_bind_int, sqlite3_bind_parameter_index, sqlite3_bind_text, sqlite3_changes, sqlite3_clear_bindings, sqlite3_close_v2, sqlite3_column_blob, sqlite3_column_bytes, sqlite3_column_double, sqlite3_column_name, sqlite3_column_text, sqlite3_column_type, sqlite3_create_function_v2, sqlite3_data_count, sqlite3_errmsg, sqlite3_exec, sqlite3_finalize, sqlite3_free, sqlite3_open, sqlite3_prepare_v2, sqlite3_prepare_v2_sqlptr, sqlite3_reset, sqlite3_result_blob, sqlite3_result_double, sqlite3_result_error, sqlite3_result_int, sqlite3_result_int64, sqlite3_result_null, sqlite3_result_text, sqlite3_step, sqlite3_value_blob, sqlite3_value_bytes, sqlite3_value_double, sqlite3_value_int, sqlite3_value_text, sqlite3_value_type;

apiTemp = stackAlloc(4);

SQLite = {};


/* Represents a prepared statement.

Prepared statements allow you to have a template sql string,
that you can execute multiple times with different parameters.

You can't instantiate this class directly, you have to use a [Database](Database.html)
object in order to create a statement.

**Warning**: When you close a database (using db.close()), all its statements are
closed too and become unusable.

@see Database.html#prepare-dynamic
@see https://en.wikipedia.org/wiki/Prepared_statement
 */

Statement = (function() {
  function Statement(stmt1, db) {
    this.stmt = stmt1;
    this.db = db;
    this.pos = 1;
    this.allocatedmem = [];
  }


  /* Bind values to the parameters, after having reseted the statement
  
  SQL statements can have parameters, named *'?', '?NNN', ':VVV', '@VVV', '$VVV'*,
  where NNN is a number and VVV a string.
  This function binds these parameters to the given values.
  
  *Warning*: ':', '@', and '$' are included in the parameters names
  
  ## Binding values to named parameters
  @example Bind values to named parameters
      var stmt = db.prepare("UPDATE test SET a=@newval WHERE id BETWEEN $mini AND $maxi");
      stmt.bind({$mini:10, $maxi:20, '@newval':5});
  - Create a statement that contains parameters like '$VVV', ':VVV', '@VVV'
  - Call Statement.bind with an object as parameter
  
  ## Binding values to parameters
  @example Bind values to anonymous parameters
      var stmt = db.prepare("UPDATE test SET a=? WHERE id BETWEEN ? AND ?");
      stmt.bind([5, 10, 20]);
   - Create a statement that contains parameters like '?', '?NNN'
   - Call Statement.bind with an array as parameter
  
  ## Value types
  Javascript type | SQLite type
  --- | ---
  number | REAL, INTEGER
  boolean | INTEGER
  string | TEXT
  Array, Uint8Array | BLOB
  null | NULL
  @see http://www.sqlite.org/datatype3.html
  
  @see http://www.sqlite.org/lang_expr.html#varparam
  @param values [Array,Object] The values to bind
  @return [Boolean] true if it worked
  @throw [String] SQLite Error
   */

  Statement.prototype['bind'] = function(values) {
    if (!this.stmt) {
      throw "Statement closed";
    }
    this['reset']();
    if (Array.isArray(values)) {
      return this.bindFromArray(values);
    } else {
      return this.bindFromObject(values);
    }
  };


  /* Execute the statement, fetching the the next line of result,
  that can be retrieved with [Statement.get()](#get-dynamic) .
  
  @return [Boolean] true if a row of result available
  @throw [String] SQLite Error
   */

  Statement.prototype['step'] = function() {
    var ret;
    if (!this.stmt) {
      throw "Statement closed";
    }
    this.pos = 1;
    switch (ret = sqlite3_step(this.stmt)) {
      case SQLite.ROW:
        return true;
      case SQLite.DONE:
        return false;
      default:
        return this.db.handleError(ret);
    }
  };

  Statement.prototype.getNumber = function(pos) {
    if (pos == null) {
      pos = this.pos++;
    }
    return sqlite3_column_double(this.stmt, pos);
  };

  Statement.prototype.getString = function(pos) {
    if (pos == null) {
      pos = this.pos++;
    }
    return sqlite3_column_text(this.stmt, pos);
  };

  Statement.prototype.getBlob = function(pos) {
    var i, k, ptr, ref, result, size;
    if (pos == null) {
      pos = this.pos++;
    }
    size = sqlite3_column_bytes(this.stmt, pos);
    ptr = sqlite3_column_blob(this.stmt, pos);
    result = new Uint8Array(size);
    for (i = k = 0, ref = size; 0 <= ref ? k < ref : k > ref; i = 0 <= ref ? ++k : --k) {
      result[i] = HEAP8[ptr + i];
    }
    return result;
  };


  /* Get one row of results of a statement.
  If the first parameter is not provided, step must have been called before get.
  @param [Array,Object] Optional: If set, the values will be bound to the statement, and it will be executed
  @return [Array<String,Number,Uint8Array,null>] One row of result
  
  @example Print all the rows of the table test to the console
  
      var stmt = db.prepare("SELECT * FROM test");
      while (stmt.step()) console.log(stmt.get());
   */

  Statement.prototype['get'] = function(params) {
    var field, k, ref, results1;
    if (params != null) {
      this['bind'](params) && this['step']();
    }
    results1 = [];
    for (field = k = 0, ref = sqlite3_data_count(this.stmt); 0 <= ref ? k < ref : k > ref; field = 0 <= ref ? ++k : --k) {
      switch (sqlite3_column_type(this.stmt, field)) {
        case SQLite.INTEGER:
        case SQLite.FLOAT:
          results1.push(this.getNumber(field));
          break;
        case SQLite.TEXT:
          results1.push(this.getString(field));
          break;
        case SQLite.BLOB:
          results1.push(this.getBlob(field));
          break;
        default:
          results1.push(null);
      }
    }
    return results1;
  };


  /* Get the list of column names of a row of result of a statement.
  @return [Array<String>] The names of the columns
  @example
  
      var stmt = db.prepare("SELECT 5 AS nbr, x'616200' AS data, NULL AS null_value;");
      stmt.step(); // Execute the statement
      console.log(stmt.getColumnNames()); // Will print ['nbr','data','null_value']
   */

  Statement.prototype['getColumnNames'] = function() {
    var i, k, ref, results1;
    results1 = [];
    for (i = k = 0, ref = sqlite3_data_count(this.stmt); 0 <= ref ? k < ref : k > ref; i = 0 <= ref ? ++k : --k) {
      results1.push(sqlite3_column_name(this.stmt, i));
    }
    return results1;
  };


  /* Get one row of result as a javascript object, associating column names with
  their value in the current row.
  @param [Array,Object] Optional: If set, the values will be bound to the statement, and it will be executed
  @return [Object] The row of result
  @see [Statement.get](#get-dynamic)
  
  @example
  
      var stmt = db.prepare("SELECT 5 AS nbr, x'616200' AS data, NULL AS null_value;");
      stmt.step(); // Execute the statement
      console.log(stmt.getAsObject()); // Will print {nbr:5, data: Uint8Array([1,2,3]), null_value:null}
   */

  Statement.prototype['getAsObject'] = function(params) {
    var i, k, len, name, names, rowObject, values;
    values = this['get'](params);
    names = this['getColumnNames']();
    rowObject = {};
    for (i = k = 0, len = names.length; k < len; i = ++k) {
      name = names[i];
      rowObject[name] = values[i];
    }
    return rowObject;
  };


  /* Shorthand for bind + step + reset
  Bind the values, execute the statement, ignoring the rows it returns, and resets it
  @param [Array,Object] Value to bind to the statement
   */

  Statement.prototype['run'] = function(values) {
    if (values != null) {
      this['bind'](values);
    }
    this['step']();
    return this['reset']();
  };

  Statement.prototype.bindString = function(string, pos) {
    var bytes, strptr;
    if (pos == null) {
      pos = this.pos++;
    }
    bytes = intArrayFromString(string);
    this.allocatedmem.push(strptr = allocate(bytes, 'i8', ALLOC_NORMAL));
    this.db.handleError(sqlite3_bind_text(this.stmt, pos, strptr, bytes.length - 1, 0));
    return true;
  };

  Statement.prototype.bindBlob = function(array, pos) {
    var blobptr;
    if (pos == null) {
      pos = this.pos++;
    }
    this.allocatedmem.push(blobptr = allocate(array, 'i8', ALLOC_NORMAL));
    this.db.handleError(sqlite3_bind_blob(this.stmt, pos, blobptr, array.length, 0));
    return true;
  };

  Statement.prototype.bindNumber = function(num, pos) {
    var bindfunc;
    if (pos == null) {
      pos = this.pos++;
    }
    bindfunc = num === (num | 0) ? sqlite3_bind_int : sqlite3_bind_double;
    this.db.handleError(bindfunc(this.stmt, pos, num));
    return true;
  };

  Statement.prototype.bindNull = function(pos) {
    if (pos == null) {
      pos = this.pos++;
    }
    return sqlite3_bind_blob(this.stmt, pos, 0, 0, 0) === SQLite.OK;
  };

  Statement.prototype.bindValue = function(val, pos) {
    if (pos == null) {
      pos = this.pos++;
    }
    switch (typeof val) {
      case "string":
        return this.bindString(val, pos);
      case "number":
      case "boolean":
        return this.bindNumber(val + 0, pos);
      case "object":
        if (val === null) {
          return this.bindNull(pos);
        } else if (val.length != null) {
          return this.bindBlob(val, pos);
        } else {
          throw "Wrong API use : tried to bind a value of an unknown type (" + val + ").";
        }
    }
  };


  /* Bind names and values of an object to the named parameters of the statement
  @param [Object]
  @private
  @nodoc
   */

  Statement.prototype.bindFromObject = function(valuesObj) {
    var name, num, value;
    for (name in valuesObj) {
      value = valuesObj[name];
      num = sqlite3_bind_parameter_index(this.stmt, name);
      if (num !== 0) {
        this.bindValue(value, num);
      }
    }
    return true;
  };


  /* Bind values to numbered parameters
  @param [Array]
  @private
  @nodoc
   */

  Statement.prototype.bindFromArray = function(values) {
    var k, len, num, value;
    for (num = k = 0, len = values.length; k < len; num = ++k) {
      value = values[num];
      this.bindValue(value, num + 1);
    }
    return true;
  };


  /* Reset a statement, so that it's parameters can be bound to new values
  It also clears all previous bindings, freeing the memory used by bound parameters.
   */

  Statement.prototype['reset'] = function() {
    this.freemem();
    return sqlite3_clear_bindings(this.stmt) === SQLite.OK && sqlite3_reset(this.stmt) === SQLite.OK;
  };


  /* Free the memory allocated during parameter binding
   */

  Statement.prototype.freemem = function() {
    var mem;
    while (mem = this.allocatedmem.pop()) {
      _free(mem);
    }
    return null;
  };


  /* Free the memory used by the statement
  @return [Boolean] true in case of success
   */

  Statement.prototype['free'] = function() {
    var res;
    this.freemem();
    res = sqlite3_finalize(this.stmt) === SQLite.OK;
    delete this.db.statements[this.stmt];
    this.stmt = NULL;
    return res;
  };

  return Statement;

})();

Database = (function() {
  function Database(data) {
    this.filename = 'dbfile_' + (0xffffffff * Math.random() >>> 0);
    if (data != null) {
      FS.createDataFile('/', this.filename, data, true, true);
    }
    this.handleError(sqlite3_open(this.filename, apiTemp));
    this.db = getValue(apiTemp, 'i32');
    RegisterExtensionFunctions(this.db);
    this.statements = {};
    this.functions = {};
  }


  /* Execute an SQL query, ignoring the rows it returns.
  
  @param sql [String] a string containing some SQL text to execute
  @param params [Array] (*optional*) When the SQL statement contains placeholders, you can pass them in here. They will be bound to the statement before it is executed.
  
  If you use the params argument, you **cannot** provide an sql string that contains several
  queries (separated by ';')
  
  @example Insert values in a table
      db.run("INSERT INTO test VALUES (:age, :name)", {':age':18, ':name':'John'});
  
  @return [Database] The database object (useful for method chaining)
   */

  Database.prototype['run'] = function(sql, params) {
    var stmt;
    if (!this.db) {
      throw "Database closed";
    }
    if (params) {
      stmt = this['prepare'](sql, params);
      stmt['step']();
      stmt['free']();
    } else {
      this.handleError(sqlite3_exec(this.db, sql, 0, 0, apiTemp));
    }
    return this;
  };


  /* Execute an SQL query, and returns the result.
  
  This is a wrapper against Database.prepare, Statement.step, Statement.get,
  and Statement.free.
  
  The result is an array of result elements. There are as many result elements
  as the number of statements in your sql string (statements are separated by a semicolon)
  
  Each result element is an object with two properties:
      'columns' : the name of the columns of the result (as returned by Statement.getColumnNames())
      'values' : an array of rows. Each row is itself an array of values
  
  ## Example use
  We have the following table, named *test* :
  
  | id | age |  name  |
  |:--:|:---:|:------:|
  | 1  |  1  | Ling   |
  | 2  |  18 | Paul   |
  | 3  |  3  | Markus |
  
  
  We query it like that:
  ```javascript
  var db = new SQL.Database();
  var res = db.exec("SELECT id FROM test; SELECT age,name FROM test;");
  ```
  
  `res` is now :
  ```javascript
      [
          {columns: ['id'], values:[[1],[2],[3]]},
          {columns: ['age','name'], values:[[1,'Ling'],[18,'Paul'],[3,'Markus']]}
      ]
  ```
  
  @param sql [String] a string containing some SQL text to execute
  @return [Array<QueryResults>] An array of results.
   */

  Database.prototype['exec'] = function(sql) {
    var curresult, nextSqlPtr, pStmt, pzTail, results, stack, stmt;
    if (!this.db) {
      throw "Database closed";
    }
    stack = stackSave();
    nextSqlPtr = allocateUTF8OnStack(sql);
    pzTail = stackAlloc(4);
    results = [];
    while (getValue(nextSqlPtr, 'i8') !== NULL) {
      setValue(apiTemp, 0, 'i32');
      setValue(pzTail, 0, 'i32');
      this.handleError(sqlite3_prepare_v2_sqlptr(this.db, nextSqlPtr, -1, apiTemp, pzTail));
      pStmt = getValue(apiTemp, 'i32');
      nextSqlPtr = getValue(pzTail, 'i32');
      if (pStmt === NULL) {
        continue;
      }
      stmt = new Statement(pStmt, this);
      curresult = null;
      while (stmt['step']()) {
        if (curresult === null) {
          curresult = {
            'columns': stmt['getColumnNames'](),
            'values': []
          };
          results.push(curresult);
        }
        curresult['values'].push(stmt['get']());
      }
      stmt['free']();
    }
    stackRestore(stack);
    return results;
  };


  /* Execute an sql statement, and call a callback for each row of result.
  
  **Currently** this method is synchronous, it will not return until the callback has
  been called on every row of the result. But this might change.
  
  @param sql [String] A string of SQL text. Can contain placeholders that will be
  bound to the parameters given as the second argument
  @param params [Array<String,Number,null,Uint8Array>] (*optional*) Parameters to bind
  to the query
  @param callback [Function(Object)] A function that will be called on each row of result
  @param done [Function] A function that will be called when all rows have been retrieved
  
  @return [Database] The database object. Useful for method chaining
  
  @example Read values from a table
      db.each("SELECT name,age FROM users WHERE age >= $majority",
                      {$majority:18},
                      function(row){console.log(row.name + " is a grown-up.")}
                  );
   */

  Database.prototype['each'] = function(sql, params, callback, done) {
    var stmt;
    if (typeof params === 'function') {
      done = callback;
      callback = params;
      params = void 0;
    }
    stmt = this['prepare'](sql, params);
    while (stmt['step']()) {
      callback(stmt['getAsObject']());
    }
    stmt['free']();
    if (typeof done === 'function') {
      return done();
    }
  };


  /* Prepare an SQL statement
  @param sql [String] a string of SQL, that can contain placeholders ('?', ':VVV', ':AAA', '@AAA')
  @param params [Array] (*optional*) values to bind to placeholders
  @return [Statement] the resulting statement
  @throw [String] SQLite error
   */

  Database.prototype['prepare'] = function(sql, params) {
    var pStmt, stmt;
    setValue(apiTemp, 0, 'i32');
    this.handleError(sqlite3_prepare_v2(this.db, sql, -1, apiTemp, NULL));
    pStmt = getValue(apiTemp, 'i32');
    if (pStmt === NULL) {
      throw 'Nothing to prepare';
    }
    stmt = new Statement(pStmt, this);
    if (params != null) {
      stmt.bind(params);
    }
    this.statements[pStmt] = stmt;
    return stmt;
  };


  /* Exports the contents of the database to a binary array
  @return [Uint8Array] An array of bytes of the SQLite3 database file
   */

  Database.prototype['export'] = function() {
    var _, binaryDb, func, ref, ref1, stmt;
    ref = this.statements;
    for (_ in ref) {
      stmt = ref[_];
      stmt['free']();
    }
    ref1 = this.functions;
    for (_ in ref1) {
      func = ref1[_];
      removeFunction(func);
    }
    this.functions = {};
    this.handleError(sqlite3_close_v2(this.db));
    binaryDb = FS.readFile(this.filename, {
      encoding: 'binary'
    });
    this.handleError(sqlite3_open(this.filename, apiTemp));
    this.db = getValue(apiTemp, 'i32');
    return binaryDb;
  };


  /* Close the database, and all associated prepared statements.
  
  The memory associated to the database and all associated statements
  will be freed.
  
  **Warning**: A statement belonging to a database that has been closed cannot
  be used anymore.
  
  Databases **must** be closed, when you're finished with them, or the
  memory consumption will grow forever
   */

  Database.prototype['close'] = function() {
    var _, func, ref, ref1, stmt;
    ref = this.statements;
    for (_ in ref) {
      stmt = ref[_];
      stmt['free']();
    }
    ref1 = this.functions;
    for (_ in ref1) {
      func = ref1[_];
      removeFunction(func);
    }
    this.functions = {};
    this.handleError(sqlite3_close_v2(this.db));
    FS.unlink('/' + this.filename);
    return this.db = null;
  };


  /* Analyze a result code, return null if no error occured, and throw
  an error with a descriptive message otherwise
  @nodoc
   */

  Database.prototype.handleError = function(returnCode) {
    var errmsg;
    if (returnCode === SQLite.OK) {
      return null;
    } else {
      errmsg = sqlite3_errmsg(this.db);
      throw new Error(errmsg);
    }
  };


  /* Returns the number of rows modified, inserted or deleted by the
  most recently completed INSERT, UPDATE or DELETE statement on the
  database Executing any other type of SQL statement does not modify
  the value returned by this function.
  
  @return [Number] the number of rows modified
   */

  Database.prototype['getRowsModified'] = function() {
    return sqlite3_changes(this.db);
  };


  /* Register a custom function with SQLite
  @example Register a simple function
      db.create_function("addOne", function(x) {return x+1;})
      db.exec("SELECT addOne(1)") // = 2
  
  @param name [String] the name of the function as referenced in SQL statements.
  @param func [Function] the actual function to be executed.
   */

  Database.prototype['create_function'] = function(name, func) {
    var func_ptr, wrapped_func;
    wrapped_func = function(cx, argc, argv) {
      var arg, args, blobptr, data_func, error, i, k, ref, result, value_ptr, value_type;
      args = [];
      for (i = k = 0, ref = argc; 0 <= ref ? k < ref : k > ref; i = 0 <= ref ? ++k : --k) {
        value_ptr = getValue(argv + (4 * i), 'i32');
        value_type = sqlite3_value_type(value_ptr);
        data_func = (function() {
          switch (false) {
            case value_type !== 1:
              return sqlite3_value_double;
            case value_type !== 2:
              return sqlite3_value_double;
            case value_type !== 3:
              return sqlite3_value_text;
            case value_type !== 4:
              return function(ptr) {
                var blob_arg, blob_ptr, j, l, ref1, size;
                size = sqlite3_value_bytes(ptr);
                blob_ptr = sqlite3_value_blob(ptr);
                blob_arg = new Uint8Array(size);
                for (j = l = 0, ref1 = size; 0 <= ref1 ? l < ref1 : l > ref1; j = 0 <= ref1 ? ++l : --l) {
                  blob_arg[j] = HEAP8[blob_ptr + j];
                }
                return blob_arg;
              };
            default:
              return function(ptr) {
                return null;
              };
          }
        })();
        arg = data_func(value_ptr);
        args.push(arg);
      }
      try {
        result = func.apply(null, args);
      } catch (error1) {
        error = error1;
        sqlite3_result_error(cx, error, -1);
        return;
      }
      switch (typeof result) {
        case 'boolean':
          sqlite3_result_int(cx, result ? 1 : 0);
          break;
        case 'number':
          sqlite3_result_double(cx, result);
          break;
        case 'string':
          sqlite3_result_text(cx, result, -1, -1);
          break;
        case 'object':
          if (result === null) {
            sqlite3_result_null(cx);
          } else if (result.length != null) {
            blobptr = allocate(result, 'i8', ALLOC_NORMAL);
            sqlite3_result_blob(cx, blobptr, result.length, -1);
            _free(blobptr);
          } else {
            sqlite3_result_error(cx, "Wrong API use : tried to return a value of an unknown type (" + result + ").", -1);
          }
          break;
        default:
          sqlite3_result_null(cx);
      }
    };
    if (name in this.functions) {
      removeFunction(this.functions[name]);
      delete this.functions[name];
    }
    func_ptr = addFunction(wrapped_func);
    this.functions[name] = func_ptr;
    this.handleError(sqlite3_create_function_v2(this.db, name, func.length, SQLite.UTF8, 0, func_ptr, 0, 0, 0));
    return this;
  };

  return Database;

})();

sqlite3_open = Module['cwrap']('sqlite3_open', 'number', ['string', 'number']);

sqlite3_close_v2 = Module['cwrap']('sqlite3_close_v2', 'number', ['number']);

sqlite3_exec = Module['cwrap']('sqlite3_exec', 'number', ['number', 'string', 'number', 'number', 'number']);

sqlite3_free = Module['cwrap']('sqlite3_free', '', ['number']);

sqlite3_changes = Module['cwrap']('sqlite3_changes', 'number', ['number']);

sqlite3_prepare_v2 = Module['cwrap']('sqlite3_prepare_v2', 'number', ['number', 'string', 'number', 'number', 'number']);

sqlite3_prepare_v2_sqlptr = Module['cwrap']('sqlite3_prepare_v2', 'number', ['number', 'number', 'number', 'number', 'number']);

sqlite3_bind_text = Module['cwrap']('sqlite3_bind_text', 'number', ['number', 'number', 'number', 'number', 'number']);

sqlite3_bind_blob = Module['cwrap']('sqlite3_bind_blob', 'number', ['number', 'number', 'number', 'number', 'number']);

sqlite3_bind_double = Module['cwrap']('sqlite3_bind_double', 'number', ['number', 'number', 'number']);

sqlite3_bind_int = Module['cwrap']('sqlite3_bind_int', 'number', ['number', 'number', 'number']);

sqlite3_bind_parameter_index = Module['cwrap']('sqlite3_bind_parameter_index', 'number', ['number', 'string']);

sqlite3_step = Module['cwrap']('sqlite3_step', 'number', ['number']);

sqlite3_errmsg = Module['cwrap']('sqlite3_errmsg', 'string', ['number']);

sqlite3_data_count = Module['cwrap']('sqlite3_data_count', 'number', ['number']);

sqlite3_column_double = Module['cwrap']('sqlite3_column_double', 'number', ['number', 'number']);

sqlite3_column_text = Module['cwrap']('sqlite3_column_text', 'string', ['number', 'number']);

sqlite3_column_blob = Module['cwrap']('sqlite3_column_blob', 'number', ['number', 'number']);

sqlite3_column_bytes = Module['cwrap']('sqlite3_column_bytes', 'number', ['number', 'number']);

sqlite3_column_type = Module['cwrap']('sqlite3_column_type', 'number', ['number', 'number']);

sqlite3_column_name = Module['cwrap']('sqlite3_column_name', 'string', ['number', 'number']);

sqlite3_reset = Module['cwrap']('sqlite3_reset', 'number', ['number']);

sqlite3_clear_bindings = Module['cwrap']('sqlite3_clear_bindings', 'number', ['number']);

sqlite3_finalize = Module['cwrap']('sqlite3_finalize', 'number', ['number']);

sqlite3_create_function_v2 = Module['cwrap']('sqlite3_create_function_v2', 'number', ['number', 'string', 'number', 'number', 'number', 'number', 'number', 'number', 'number']);

sqlite3_value_type = Module['cwrap']('sqlite3_value_type', 'number', ['number']);

sqlite3_value_bytes = Module['cwrap']('sqlite3_value_bytes', 'number', ['number']);

sqlite3_value_text = Module['cwrap']('sqlite3_value_text', 'string', ['number']);

sqlite3_value_int = Module['cwrap']('sqlite3_value_int', 'number', ['number']);

sqlite3_value_blob = Module['cwrap']('sqlite3_value_blob', 'number', ['number']);

sqlite3_value_double = Module['cwrap']('sqlite3_value_double', 'number', ['number']);

sqlite3_result_double = Module['cwrap']('sqlite3_result_double', '', ['number', 'number']);

sqlite3_result_null = Module['cwrap']('sqlite3_result_null', '', ['number']);

sqlite3_result_text = Module['cwrap']('sqlite3_result_text', '', ['number', 'string', 'number', 'number']);

sqlite3_result_blob = Module['cwrap']('sqlite3_result_blob', '', ['number', 'number', 'number', 'number']);

sqlite3_result_int = Module['cwrap']('sqlite3_result_int', '', ['number', 'number']);

sqlite3_result_int64 = Module['cwrap']('sqlite3_result_int64', '', ['number', 'number']);

sqlite3_result_error = Module['cwrap']('sqlite3_result_error', '', ['number', 'string', 'number']);

RegisterExtensionFunctions = Module['cwrap']('RegisterExtensionFunctions', 'number', ['number']);

this['SQL'] = {
  'Database': Database
};

for (i in this['SQL']) {
  Module[i] = this['SQL'][i];
}

NULL = 0;

SQLite.OK = 0;

SQLite.ERROR = 1;

SQLite.INTERNAL = 2;

SQLite.PERM = 3;

SQLite.ABORT = 4;

SQLite.BUSY = 5;

SQLite.LOCKED = 6;

SQLite.NOMEM = 7;

SQLite.READONLY = 8;

SQLite.INTERRUPT = 9;

SQLite.IOERR = 10;

SQLite.CORRUPT = 11;

SQLite.NOTFOUND = 12;

SQLite.FULL = 13;

SQLite.CANTOPEN = 14;

SQLite.PROTOCOL = 15;

SQLite.EMPTY = 16;

SQLite.SCHEMA = 17;

SQLite.TOOBIG = 18;

SQLite.CONSTRAINT = 19;

SQLite.MISMATCH = 20;

SQLite.MISUSE = 21;

SQLite.NOLFS = 22;

SQLite.AUTH = 23;

SQLite.FORMAT = 24;

SQLite.RANGE = 25;

SQLite.NOTADB = 26;

SQLite.NOTICE = 27;

SQLite.WARNING = 28;

SQLite.ROW = 100;

SQLite.DONE = 101;

SQLite.INTEGER = 1;

SQLite.FLOAT = 2;

SQLite.TEXT = 3;

SQLite.BLOB = 4;

SQLite.NULL = 5;

SQLite.UTF8 = 1;

}).bind(this);

Module['preRun'] = Module['preRun'] || [];
Module['preRun'].push(runCompiledCode);


// Sometimes an existing Module object exists with properties
// meant to overwrite the default module functionality. Here
// we collect those properties and reapply _after_ we configure
// the current environment's defaults to avoid having to be so
// defensive during initialization.
var moduleOverrides = {};
var key;
for (key in Module) {
  if (Module.hasOwnProperty(key)) {
    moduleOverrides[key] = Module[key];
  }
}

Module['arguments'] = [];
Module['thisProgram'] = './this.program';
Module['quit'] = function(status, toThrow) {
  throw toThrow;
};
Module['preRun'] = [];
Module['postRun'] = [];

// Determine the runtime environment we are in. You can customize this by
// setting the ENVIRONMENT setting at compile time (see settings.js).

var ENVIRONMENT_IS_WEB = false;
var ENVIRONMENT_IS_WORKER = false;
var ENVIRONMENT_IS_NODE = false;
var ENVIRONMENT_IS_SHELL = false;
ENVIRONMENT_IS_WEB = typeof window === 'object';
ENVIRONMENT_IS_WORKER = typeof importScripts === 'function';
ENVIRONMENT_IS_NODE = typeof process === 'object' && typeof require === 'function' && !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_WORKER;
ENVIRONMENT_IS_SHELL = !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_NODE && !ENVIRONMENT_IS_WORKER;



// Three configurations we can be running in:
// 1) We could be the application main() thread running in the main JS UI thread. (ENVIRONMENT_IS_WORKER == false and ENVIRONMENT_IS_PTHREAD == false)
// 2) We could be the application main() thread proxied to worker. (with Emscripten -s PROXY_TO_WORKER=1) (ENVIRONMENT_IS_WORKER == true, ENVIRONMENT_IS_PTHREAD == false)
// 3) We could be an application pthread running in a worker. (ENVIRONMENT_IS_WORKER == true and ENVIRONMENT_IS_PTHREAD == true)




// `/` should be present at the end if `scriptDirectory` is not empty
var scriptDirectory = '';
function locateFile(path) {
  if (Module['locateFile']) {
    return Module['locateFile'](path, scriptDirectory);
  } else {
    return scriptDirectory + path;
  }
}

if (ENVIRONMENT_IS_NODE) {
  scriptDirectory = __dirname + '/';

  // Expose functionality in the same simple way that the shells work
  // Note that we pollute the global namespace here, otherwise we break in node
  var nodeFS;
  var nodePath;

  Module['read'] = function shell_read(filename, binary) {
    var ret;
    ret = tryParseAsDataURI(filename);
    if (!ret) {
      if (!nodeFS) nodeFS = require('fs');
      if (!nodePath) nodePath = require('path');
      filename = nodePath['normalize'](filename);
      ret = nodeFS['readFileSync'](filename);
    }
    return binary ? ret : ret.toString();
  };

  Module['readBinary'] = function readBinary(filename) {
    var ret = Module['read'](filename, true);
    if (!ret.buffer) {
      ret = new Uint8Array(ret);
    }
    assert(ret.buffer);
    return ret;
  };

  if (process['argv'].length > 1) {
    Module['thisProgram'] = process['argv'][1].replace(/\\/g, '/');
  }

  Module['arguments'] = process['argv'].slice(2);

  if (typeof module !== 'undefined') {
    module['exports'] = Module;
  }

  // Currently node will swallow unhandled rejections, but this behavior is
  // deprecated, and in the future it will exit with error status.
  process['on']('unhandledRejection', abort);

  Module['quit'] = function(status) {
    process['exit'](status);
  };

  Module['inspect'] = function () { return '[Emscripten Module object]'; };
} else
if (ENVIRONMENT_IS_SHELL) {


  if (typeof read != 'undefined') {
    Module['read'] = function shell_read(f) {
      var data = tryParseAsDataURI(f);
      if (data) {
        return intArrayToString(data);
      }
      return read(f);
    };
  }

  Module['readBinary'] = function readBinary(f) {
    var data;
    data = tryParseAsDataURI(f);
    if (data) {
      return data;
    }
    if (typeof readbuffer === 'function') {
      return new Uint8Array(readbuffer(f));
    }
    data = read(f, 'binary');
    assert(typeof data === 'object');
    return data;
  };

  if (typeof scriptArgs != 'undefined') {
    Module['arguments'] = scriptArgs;
  } else if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }

  if (typeof quit === 'function') {
    Module['quit'] = function(status) {
      quit(status);
    }
  }
} else
if (ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) {
  if (ENVIRONMENT_IS_WORKER) { // Check worker, not web, since window could be polyfilled
    scriptDirectory = self.location.href;
  } else if (document.currentScript) { // web
    scriptDirectory = document.currentScript.src;
  }
  // blob urls look like blob:http://site.com/etc/etc and we cannot infer anything from them.
  // otherwise, slice off the final part of the url to find the script directory.
  // if scriptDirectory does not contain a slash, lastIndexOf will return -1,
  // and scriptDirectory will correctly be replaced with an empty string.
  if (scriptDirectory.indexOf('blob:') !== 0) {
    scriptDirectory = scriptDirectory.substr(0, scriptDirectory.lastIndexOf('/')+1);
  } else {
    scriptDirectory = '';
  }


  Module['read'] = function shell_read(url) {
    try {
      var xhr = new XMLHttpRequest();
      xhr.open('GET', url, false);
      xhr.send(null);
      return xhr.responseText;
    } catch (err) {
      var data = tryParseAsDataURI(url);
      if (data) {
        return intArrayToString(data);
      }
      throw err;
    }
  };

  if (ENVIRONMENT_IS_WORKER) {
    Module['readBinary'] = function readBinary(url) {
      try {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', url, false);
        xhr.responseType = 'arraybuffer';
        xhr.send(null);
        return new Uint8Array(xhr.response);
      } catch (err) {
        var data = tryParseAsDataURI(url);
        if (data) {
          return data;
        }
        throw err;
      }
    };
  }

  Module['readAsync'] = function readAsync(url, onload, onerror) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, true);
    xhr.responseType = 'arraybuffer';
    xhr.onload = function xhr_onload() {
      if (xhr.status == 200 || (xhr.status == 0 && xhr.response)) { // file URLs can return 0
        onload(xhr.response);
        return;
      }
      var data = tryParseAsDataURI(url);
      if (data) {
        onload(data.buffer);
        return;
      }
      onerror();
    };
    xhr.onerror = onerror;
    xhr.send(null);
  };

  Module['setWindowTitle'] = function(title) { document.title = title };
} else
{
}

// Set up the out() and err() hooks, which are how we can print to stdout or
// stderr, respectively.
// If the user provided Module.print or printErr, use that. Otherwise,
// console.log is checked first, as 'print' on the web will open a print dialogue
// printErr is preferable to console.warn (works better in shells)
// bind(console) is necessary to fix IE/Edge closed dev tools panel behavior.
var out = Module['print'] || (typeof console !== 'undefined' ? console.log.bind(console) : (typeof print !== 'undefined' ? print : null));
var err = Module['printErr'] || (typeof printErr !== 'undefined' ? printErr : ((typeof console !== 'undefined' && console.warn.bind(console)) || out));

// Merge back in the overrides
for (key in moduleOverrides) {
  if (moduleOverrides.hasOwnProperty(key)) {
    Module[key] = moduleOverrides[key];
  }
}
// Free the object hierarchy contained in the overrides, this lets the GC
// reclaim data used e.g. in memoryInitializerRequest, which is a large typed array.
moduleOverrides = undefined;

// perform assertions in shell.js after we set up out() and err(), as otherwise if an assertion fails it cannot print the message



// Copyright 2017 The Emscripten Authors.  All rights reserved.
// Emscripten is available under two separate licenses, the MIT license and the
// University of Illinois/NCSA Open Source License.  Both these licenses can be
// found in the LICENSE file.

// {{PREAMBLE_ADDITIONS}}

var STACK_ALIGN = 16;


function dynamicAlloc(size) {
  var ret = HEAP32[DYNAMICTOP_PTR>>2];
  var end = (ret + size + 15) & -16;
  if (end <= _emscripten_get_heap_size()) {
    HEAP32[DYNAMICTOP_PTR>>2] = end;
  } else {
    return 0;
  }
  return ret;
}

function alignMemory(size, factor) {
  if (!factor) factor = STACK_ALIGN; // stack alignment (16-byte) by default
  return Math.ceil(size / factor) * factor;
}

function getNativeTypeSize(type) {
  switch (type) {
    case 'i1': case 'i8': return 1;
    case 'i16': return 2;
    case 'i32': return 4;
    case 'i64': return 8;
    case 'float': return 4;
    case 'double': return 8;
    default: {
      if (type[type.length-1] === '*') {
        return 4; // A pointer
      } else if (type[0] === 'i') {
        var bits = parseInt(type.substr(1));
        assert(bits % 8 === 0, 'getNativeTypeSize invalid bits ' + bits + ', type ' + type);
        return bits / 8;
      } else {
        return 0;
      }
    }
  }
}

function warnOnce(text) {
  if (!warnOnce.shown) warnOnce.shown = {};
  if (!warnOnce.shown[text]) {
    warnOnce.shown[text] = 1;
    err(text);
  }
}

var asm2wasmImports = { // special asm2wasm imports
    "f64-rem": function(x, y) {
        return x % y;
    },
    "debugger": function() {
        debugger;
    }
};



var jsCallStartIndex = 1;
var functionPointers = new Array(64);


// 'sig' parameter is required for the llvm backend but only when func is not
// already a WebAssembly function.
function addFunction(func, sig) {


  var base = 0;
  for (var i = base; i < base + 64; i++) {
    if (!functionPointers[i]) {
      functionPointers[i] = func;
      return jsCallStartIndex + i;
    }
  }
  throw 'Finished up all reserved function pointers. Use a higher value for RESERVED_FUNCTION_POINTERS.';

}

function removeFunction(index) {

  functionPointers[index-jsCallStartIndex] = null;
}

var funcWrappers = {};

function getFuncWrapper(func, sig) {
  if (!func) return; // on null pointer, return undefined
  assert(sig);
  if (!funcWrappers[sig]) {
    funcWrappers[sig] = {};
  }
  var sigCache = funcWrappers[sig];
  if (!sigCache[func]) {
    // optimize away arguments usage in common cases
    if (sig.length === 1) {
      sigCache[func] = function dynCall_wrapper() {
        return dynCall(sig, func);
      };
    } else if (sig.length === 2) {
      sigCache[func] = function dynCall_wrapper(arg) {
        return dynCall(sig, func, [arg]);
      };
    } else {
      // general case
      sigCache[func] = function dynCall_wrapper() {
        return dynCall(sig, func, Array.prototype.slice.call(arguments));
      };
    }
  }
  return sigCache[func];
}


function makeBigInt(low, high, unsigned) {
  return unsigned ? ((+((low>>>0)))+((+((high>>>0)))*4294967296.0)) : ((+((low>>>0)))+((+((high|0)))*4294967296.0));
}

function dynCall(sig, ptr, args) {
  if (args && args.length) {
    return Module['dynCall_' + sig].apply(null, [ptr].concat(args));
  } else {
    return Module['dynCall_' + sig].call(null, ptr);
  }
}

var tempRet0 = 0;

var setTempRet0 = function(value) {
  tempRet0 = value;
}

var getTempRet0 = function() {
  return tempRet0;
}


var Runtime = {
};

// The address globals begin at. Very low in memory, for code size and optimization opportunities.
// Above 0 is static memory, starting with globals.
// Then the stack.
// Then 'dynamic' memory for sbrk.
var GLOBAL_BASE = 8;




// === Preamble library stuff ===

// Documentation for the public APIs defined in this file must be updated in:
//    site/source/docs/api_reference/preamble.js.rst
// A prebuilt local version of the documentation is available at:
//    site/build/text/docs/api_reference/preamble.js.txt
// You can also build docs locally as HTML or other formats in site/
// An online HTML version (which may be of a different version of Emscripten)
//    is up at http://kripken.github.io/emscripten-site/docs/api_reference/preamble.js.html




/** @type {function(number, string, boolean=)} */
function getValue(ptr, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': return HEAP8[((ptr)>>0)];
      case 'i8': return HEAP8[((ptr)>>0)];
      case 'i16': return HEAP16[((ptr)>>1)];
      case 'i32': return HEAP32[((ptr)>>2)];
      case 'i64': return HEAP32[((ptr)>>2)];
      case 'float': return HEAPF32[((ptr)>>2)];
      case 'double': return HEAPF64[((ptr)>>3)];
      default: abort('invalid type for getValue: ' + type);
    }
  return null;
}




// Wasm globals

var wasmMemory;

// Potentially used for direct table calls.
var wasmTable;


//========================================
// Runtime essentials
//========================================

// whether we are quitting the application. no code should run after this.
// set in exit() and abort()
var ABORT = false;

// set by exit() and abort().  Passed to 'onExit' handler.
// NOTE: This is also used as the process return code code in shell environments
// but only when noExitRuntime is false.
var EXITSTATUS = 0;

/** @type {function(*, string=)} */
function assert(condition, text) {
  if (!condition) {
    abort('Assertion failed: ' + text);
  }
}

// Returns the C function with a specified identifier (for C++, you need to do manual name mangling)
function getCFunc(ident) {
  var func = Module['_' + ident]; // closure exported function
  assert(func, 'Cannot call unknown function ' + ident + ', make sure it is exported');
  return func;
}

// C calling interface.
function ccall(ident, returnType, argTypes, args, opts) {
  // For fast lookup of conversion functions
  var toC = {
    'string': function(str) {
      var ret = 0;
      if (str !== null && str !== undefined && str !== 0) { // null string
        // at most 4 bytes per UTF-8 code point, +1 for the trailing '\0'
        var len = (str.length << 2) + 1;
        ret = stackAlloc(len);
        stringToUTF8(str, ret, len);
      }
      return ret;
    },
    'array': function(arr) {
      var ret = stackAlloc(arr.length);
      writeArrayToMemory(arr, ret);
      return ret;
    }
  };

  function convertReturnValue(ret) {
    if (returnType === 'string') return UTF8ToString(ret);
    if (returnType === 'boolean') return Boolean(ret);
    return ret;
  }

  var func = getCFunc(ident);
  var cArgs = [];
  var stack = 0;
  if (args) {
    for (var i = 0; i < args.length; i++) {
      var converter = toC[argTypes[i]];
      if (converter) {
        if (stack === 0) stack = stackSave();
        cArgs[i] = converter(args[i]);
      } else {
        cArgs[i] = args[i];
      }
    }
  }
  var ret = func.apply(null, cArgs);
  ret = convertReturnValue(ret);
  if (stack !== 0) stackRestore(stack);
  return ret;
}

function cwrap(ident, returnType, argTypes, opts) {
  argTypes = argTypes || [];
  // When the function takes numbers and returns a number, we can just return
  // the original function
  var numericArgs = argTypes.every(function(type){ return type === 'number'});
  var numericRet = returnType !== 'string';
  if (numericRet && numericArgs && !opts) {
    return getCFunc(ident);
  }
  return function() {
    return ccall(ident, returnType, argTypes, arguments, opts);
  }
}

/** @type {function(number, number, string, boolean=)} */
function setValue(ptr, value, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': HEAP8[((ptr)>>0)]=value; break;
      case 'i8': HEAP8[((ptr)>>0)]=value; break;
      case 'i16': HEAP16[((ptr)>>1)]=value; break;
      case 'i32': HEAP32[((ptr)>>2)]=value; break;
      case 'i64': (tempI64 = [value>>>0,(tempDouble=value,(+(Math_abs(tempDouble))) >= 1.0 ? (tempDouble > 0.0 ? ((Math_min((+(Math_floor((tempDouble)/4294967296.0))), 4294967295.0))|0)>>>0 : (~~((+(Math_ceil((tempDouble - +(((~~(tempDouble)))>>>0))/4294967296.0)))))>>>0) : 0)],HEAP32[((ptr)>>2)]=tempI64[0],HEAP32[(((ptr)+(4))>>2)]=tempI64[1]); break;
      case 'float': HEAPF32[((ptr)>>2)]=value; break;
      case 'double': HEAPF64[((ptr)>>3)]=value; break;
      default: abort('invalid type for setValue: ' + type);
    }
}

var ALLOC_NORMAL = 0; // Tries to use _malloc()
var ALLOC_STACK = 1; // Lives for the duration of the current function call
var ALLOC_DYNAMIC = 2; // Cannot be freed except through sbrk
var ALLOC_NONE = 3; // Do not allocate

// allocate(): This is for internal use. You can use it yourself as well, but the interface
//             is a little tricky (see docs right below). The reason is that it is optimized
//             for multiple syntaxes to save space in generated code. So you should
//             normally not use allocate(), and instead allocate memory using _malloc(),
//             initialize it with setValue(), and so forth.
// @slab: An array of data, or a number. If a number, then the size of the block to allocate,
//        in *bytes* (note that this is sometimes confusing: the next parameter does not
//        affect this!)
// @types: Either an array of types, one for each byte (or 0 if no type at that position),
//         or a single type which is used for the entire block. This only matters if there
//         is initial data - if @slab is a number, then this does not matter at all and is
//         ignored.
// @allocator: How to allocate memory, see ALLOC_*
/** @type {function((TypedArray|Array<number>|number), string, number, number=)} */
function allocate(slab, types, allocator, ptr) {
  var zeroinit, size;
  if (typeof slab === 'number') {
    zeroinit = true;
    size = slab;
  } else {
    zeroinit = false;
    size = slab.length;
  }

  var singleType = typeof types === 'string' ? types : null;

  var ret;
  if (allocator == ALLOC_NONE) {
    ret = ptr;
  } else {
    ret = [_malloc,
    stackAlloc,
    dynamicAlloc][allocator](Math.max(size, singleType ? 1 : types.length));
  }

  if (zeroinit) {
    var stop;
    ptr = ret;
    assert((ret & 3) == 0);
    stop = ret + (size & ~3);
    for (; ptr < stop; ptr += 4) {
      HEAP32[((ptr)>>2)]=0;
    }
    stop = ret + size;
    while (ptr < stop) {
      HEAP8[((ptr++)>>0)]=0;
    }
    return ret;
  }

  if (singleType === 'i8') {
    if (slab.subarray || slab.slice) {
      HEAPU8.set(/** @type {!Uint8Array} */ (slab), ret);
    } else {
      HEAPU8.set(new Uint8Array(slab), ret);
    }
    return ret;
  }

  var i = 0, type, typeSize, previousType;
  while (i < size) {
    var curr = slab[i];

    type = singleType || types[i];
    if (type === 0) {
      i++;
      continue;
    }

    if (type == 'i64') type = 'i32'; // special case: we have one i32 here, and one i32 later

    setValue(ret+i, curr, type);

    // no need to look up size unless type changes, so cache it
    if (previousType !== type) {
      typeSize = getNativeTypeSize(type);
      previousType = type;
    }
    i += typeSize;
  }

  return ret;
}

// Allocate memory during any stage of startup - static memory early on, dynamic memory later, malloc when ready
function getMemory(size) {
  if (!runtimeInitialized) return dynamicAlloc(size);
  return _malloc(size);
}




/** @type {function(number, number=)} */
function Pointer_stringify(ptr, length) {
  abort("this function has been removed - you should use UTF8ToString(ptr, maxBytesToRead) instead!");
}

// Given a pointer 'ptr' to a null-terminated ASCII-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function AsciiToString(ptr) {
  var str = '';
  while (1) {
    var ch = HEAPU8[((ptr++)>>0)];
    if (!ch) return str;
    str += String.fromCharCode(ch);
  }
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in ASCII form. The copy will require at most str.length+1 bytes of space in the HEAP.

function stringToAscii(str, outPtr) {
  return writeAsciiToMemory(str, outPtr, false);
}


// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the given array that contains uint8 values, returns
// a copy of that string as a Javascript String object.

var UTF8Decoder = typeof TextDecoder !== 'undefined' ? new TextDecoder('utf8') : undefined;

/**
 * @param {number} idx
 * @param {number=} maxBytesToRead
 * @return {string}
 */
function UTF8ArrayToString(u8Array, idx, maxBytesToRead) {
  var endIdx = idx + maxBytesToRead;
  var endPtr = idx;
  // TextDecoder needs to know the byte length in advance, it doesn't stop on null terminator by itself.
  // Also, use the length info to avoid running tiny strings through TextDecoder, since .subarray() allocates garbage.
  // (As a tiny code save trick, compare endPtr against endIdx using a negation, so that undefined means Infinity)
  while (u8Array[endPtr] && !(endPtr >= endIdx)) ++endPtr;

  if (endPtr - idx > 16 && u8Array.subarray && UTF8Decoder) {
    return UTF8Decoder.decode(u8Array.subarray(idx, endPtr));
  } else {
    var str = '';
    // If building with TextDecoder, we have already computed the string length above, so test loop end condition against that
    while (idx < endPtr) {
      // For UTF8 byte structure, see:
      // http://en.wikipedia.org/wiki/UTF-8#Description
      // https://www.ietf.org/rfc/rfc2279.txt
      // https://tools.ietf.org/html/rfc3629
      var u0 = u8Array[idx++];
      if (!(u0 & 0x80)) { str += String.fromCharCode(u0); continue; }
      var u1 = u8Array[idx++] & 63;
      if ((u0 & 0xE0) == 0xC0) { str += String.fromCharCode(((u0 & 31) << 6) | u1); continue; }
      var u2 = u8Array[idx++] & 63;
      if ((u0 & 0xF0) == 0xE0) {
        u0 = ((u0 & 15) << 12) | (u1 << 6) | u2;
      } else {
        u0 = ((u0 & 7) << 18) | (u1 << 12) | (u2 << 6) | (u8Array[idx++] & 63);
      }

      if (u0 < 0x10000) {
        str += String.fromCharCode(u0);
      } else {
        var ch = u0 - 0x10000;
        str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
      }
    }
  }
  return str;
}

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the emscripten HEAP, returns a
// copy of that string as a Javascript String object.
// maxBytesToRead: an optional length that specifies the maximum number of bytes to read. You can omit
//                 this parameter to scan the string until the first \0 byte. If maxBytesToRead is
//                 passed, and the string at [ptr, ptr+maxBytesToReadr[ contains a null byte in the
//                 middle, then the string will cut short at that byte index (i.e. maxBytesToRead will
//                 not produce a string of exact length [ptr, ptr+maxBytesToRead[)
//                 N.B. mixing frequent uses of UTF8ToString() with and without maxBytesToRead may
//                 throw JS JIT optimizations off, so it is worth to consider consistently using one
//                 style or the other.
/**
 * @param {number} ptr
 * @param {number=} maxBytesToRead
 * @return {string}
 */
function UTF8ToString(ptr, maxBytesToRead) {
  return ptr ? UTF8ArrayToString(HEAPU8, ptr, maxBytesToRead) : '';
}

// Copies the given Javascript String object 'str' to the given byte array at address 'outIdx',
// encoded in UTF8 form and null-terminated. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8 to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outU8Array: the array to copy to. Each index in this array is assumed to be one 8-byte element.
//   outIdx: The starting offset in the array to begin the copying.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array.
//                    This count should include the null terminator,
//                    i.e. if maxBytesToWrite=1, only the null terminator will be written and nothing else.
//                    maxBytesToWrite=0 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8Array(str, outU8Array, outIdx, maxBytesToWrite) {
  if (!(maxBytesToWrite > 0)) // Parameter maxBytesToWrite is not optional. Negative values, 0, null, undefined and false each don't write out any bytes.
    return 0;

  var startIdx = outIdx;
  var endIdx = outIdx + maxBytesToWrite - 1; // -1 for string null terminator.
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    // For UTF8 byte structure, see http://en.wikipedia.org/wiki/UTF-8#Description and https://www.ietf.org/rfc/rfc2279.txt and https://tools.ietf.org/html/rfc3629
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) {
      var u1 = str.charCodeAt(++i);
      u = 0x10000 + ((u & 0x3FF) << 10) | (u1 & 0x3FF);
    }
    if (u <= 0x7F) {
      if (outIdx >= endIdx) break;
      outU8Array[outIdx++] = u;
    } else if (u <= 0x7FF) {
      if (outIdx + 1 >= endIdx) break;
      outU8Array[outIdx++] = 0xC0 | (u >> 6);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0xFFFF) {
      if (outIdx + 2 >= endIdx) break;
      outU8Array[outIdx++] = 0xE0 | (u >> 12);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else {
      if (outIdx + 3 >= endIdx) break;
      outU8Array[outIdx++] = 0xF0 | (u >> 18);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    }
  }
  // Null-terminate the pointer to the buffer.
  outU8Array[outIdx] = 0;
  return outIdx - startIdx;
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF8 form. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8 to compute the exact number of bytes (excluding null terminator) that this function will write.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8(str, outPtr, maxBytesToWrite) {
  return stringToUTF8Array(str, HEAPU8,outPtr, maxBytesToWrite);
}

// Returns the number of bytes the given Javascript string takes if encoded as a UTF8 byte array, EXCLUDING the null terminator byte.
function lengthBytesUTF8(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) u = 0x10000 + ((u & 0x3FF) << 10) | (str.charCodeAt(++i) & 0x3FF);
    if (u <= 0x7F) ++len;
    else if (u <= 0x7FF) len += 2;
    else if (u <= 0xFFFF) len += 3;
    else len += 4;
  }
  return len;
}


// Given a pointer 'ptr' to a null-terminated UTF16LE-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

var UTF16Decoder = typeof TextDecoder !== 'undefined' ? new TextDecoder('utf-16le') : undefined;
function UTF16ToString(ptr) {
  var endPtr = ptr;
  // TextDecoder needs to know the byte length in advance, it doesn't stop on null terminator by itself.
  // Also, use the length info to avoid running tiny strings through TextDecoder, since .subarray() allocates garbage.
  var idx = endPtr >> 1;
  while (HEAP16[idx]) ++idx;
  endPtr = idx << 1;

  if (endPtr - ptr > 32 && UTF16Decoder) {
    return UTF16Decoder.decode(HEAPU8.subarray(ptr, endPtr));
  } else {
    var i = 0;

    var str = '';
    while (1) {
      var codeUnit = HEAP16[(((ptr)+(i*2))>>1)];
      if (codeUnit == 0) return str;
      ++i;
      // fromCharCode constructs a character from a UTF-16 code unit, so we can pass the UTF16 string right through.
      str += String.fromCharCode(codeUnit);
    }
  }
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF16 form. The copy will require at most str.length*4+2 bytes of space in the HEAP.
// Use the function lengthBytesUTF16() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=2, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<2 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF16(str, outPtr, maxBytesToWrite) {
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 2) return 0;
  maxBytesToWrite -= 2; // Null terminator.
  var startPtr = outPtr;
  var numCharsToWrite = (maxBytesToWrite < str.length*2) ? (maxBytesToWrite / 2) : str.length;
  for (var i = 0; i < numCharsToWrite; ++i) {
    // charCodeAt returns a UTF-16 encoded code unit, so it can be directly written to the HEAP.
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    HEAP16[((outPtr)>>1)]=codeUnit;
    outPtr += 2;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP16[((outPtr)>>1)]=0;
  return outPtr - startPtr;
}

// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF16(str) {
  return str.length*2;
}

function UTF32ToString(ptr) {
  var i = 0;

  var str = '';
  while (1) {
    var utf32 = HEAP32[(((ptr)+(i*4))>>2)];
    if (utf32 == 0)
      return str;
    ++i;
    // Gotcha: fromCharCode constructs a character from a UTF-16 encoded code (pair), not from a Unicode code point! So encode the code point to UTF-16 for constructing.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    if (utf32 >= 0x10000) {
      var ch = utf32 - 0x10000;
      str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
    } else {
      str += String.fromCharCode(utf32);
    }
  }
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF32 form. The copy will require at most str.length*4+4 bytes of space in the HEAP.
// Use the function lengthBytesUTF32() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=4, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<4 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF32(str, outPtr, maxBytesToWrite) {
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 4) return 0;
  var startPtr = outPtr;
  var endPtr = startPtr + maxBytesToWrite - 4;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) {
      var trailSurrogate = str.charCodeAt(++i);
      codeUnit = 0x10000 + ((codeUnit & 0x3FF) << 10) | (trailSurrogate & 0x3FF);
    }
    HEAP32[((outPtr)>>2)]=codeUnit;
    outPtr += 4;
    if (outPtr + 4 > endPtr) break;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP32[((outPtr)>>2)]=0;
  return outPtr - startPtr;
}

// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF32(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i);
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) ++i; // possibly a lead surrogate, so skip over the tail surrogate.
    len += 4;
  }

  return len;
}

// Allocate heap space for a JS string, and write it there.
// It is the responsibility of the caller to free() that memory.
function allocateUTF8(str) {
  var size = lengthBytesUTF8(str) + 1;
  var ret = _malloc(size);
  if (ret) stringToUTF8Array(str, HEAP8, ret, size);
  return ret;
}

// Allocate stack space for a JS string, and write it there.
function allocateUTF8OnStack(str) {
  var size = lengthBytesUTF8(str) + 1;
  var ret = stackAlloc(size);
  stringToUTF8Array(str, HEAP8, ret, size);
  return ret;
}

// Deprecated: This function should not be called because it is unsafe and does not provide
// a maximum length limit of how many bytes it is allowed to write. Prefer calling the
// function stringToUTF8Array() instead, which takes in a maximum length that can be used
// to be secure from out of bounds writes.
/** @deprecated */
function writeStringToMemory(string, buffer, dontAddNull) {
  warnOnce('writeStringToMemory is deprecated and should not be called! Use stringToUTF8() instead!');

  var /** @type {number} */ lastChar, /** @type {number} */ end;
  if (dontAddNull) {
    // stringToUTF8Array always appends null. If we don't want to do that, remember the
    // character that existed at the location where the null will be placed, and restore
    // that after the write (below).
    end = buffer + lengthBytesUTF8(string);
    lastChar = HEAP8[end];
  }
  stringToUTF8(string, buffer, Infinity);
  if (dontAddNull) HEAP8[end] = lastChar; // Restore the value under the null character.
}

function writeArrayToMemory(array, buffer) {
  HEAP8.set(array, buffer);
}

function writeAsciiToMemory(str, buffer, dontAddNull) {
  for (var i = 0; i < str.length; ++i) {
    HEAP8[((buffer++)>>0)]=str.charCodeAt(i);
  }
  // Null-terminate the pointer to the HEAP.
  if (!dontAddNull) HEAP8[((buffer)>>0)]=0;
}





function demangle(func) {
  return func;
}

function demangleAll(text) {
  var regex =
    /__Z[\w\d_]+/g;
  return text.replace(regex,
    function(x) {
      var y = demangle(x);
      return x === y ? x : (y + ' [' + x + ']');
    });
}

function jsStackTrace() {
  var err = new Error();
  if (!err.stack) {
    // IE10+ special cases: It does have callstack info, but it is only populated if an Error object is thrown,
    // so try that as a special-case.
    try {
      throw new Error(0);
    } catch(e) {
      err = e;
    }
    if (!err.stack) {
      return '(no stack trace available)';
    }
  }
  return err.stack.toString();
}

function stackTrace() {
  var js = jsStackTrace();
  if (Module['extraStackTrace']) js += '\n' + Module['extraStackTrace']();
  return demangleAll(js);
}



// Memory management

var PAGE_SIZE = 16384;
var WASM_PAGE_SIZE = 65536;
var ASMJS_PAGE_SIZE = 16777216;

function alignUp(x, multiple) {
  if (x % multiple > 0) {
    x += multiple - (x % multiple);
  }
  return x;
}

var HEAP,
/** @type {ArrayBuffer} */
  buffer,
/** @type {Int8Array} */
  HEAP8,
/** @type {Uint8Array} */
  HEAPU8,
/** @type {Int16Array} */
  HEAP16,
/** @type {Uint16Array} */
  HEAPU16,
/** @type {Int32Array} */
  HEAP32,
/** @type {Uint32Array} */
  HEAPU32,
/** @type {Float32Array} */
  HEAPF32,
/** @type {Float64Array} */
  HEAPF64;

function updateGlobalBufferViews() {
  Module['HEAP8'] = HEAP8 = new Int8Array(buffer);
  Module['HEAP16'] = HEAP16 = new Int16Array(buffer);
  Module['HEAP32'] = HEAP32 = new Int32Array(buffer);
  Module['HEAPU8'] = HEAPU8 = new Uint8Array(buffer);
  Module['HEAPU16'] = HEAPU16 = new Uint16Array(buffer);
  Module['HEAPU32'] = HEAPU32 = new Uint32Array(buffer);
  Module['HEAPF32'] = HEAPF32 = new Float32Array(buffer);
  Module['HEAPF64'] = HEAPF64 = new Float64Array(buffer);
}


var STATIC_BASE = 8,
    STACK_BASE = 60496,
    STACKTOP = STACK_BASE,
    STACK_MAX = 5303376,
    DYNAMIC_BASE = 5303376,
    DYNAMICTOP_PTR = 60240;




var TOTAL_STACK = 5242880;

var INITIAL_TOTAL_MEMORY = Module['TOTAL_MEMORY'] || 16777216;
if (INITIAL_TOTAL_MEMORY < TOTAL_STACK) err('TOTAL_MEMORY should be larger than TOTAL_STACK, was ' + INITIAL_TOTAL_MEMORY + '! (TOTAL_STACK=' + TOTAL_STACK + ')');

// Initialize the runtime's memory







// Use a provided buffer, if there is one, or else allocate a new one
if (Module['buffer']) {
  buffer = Module['buffer'];
} else {
  // Use a WebAssembly memory where available
  {
    buffer = new ArrayBuffer(INITIAL_TOTAL_MEMORY);
  }
}
updateGlobalBufferViews();


HEAP32[DYNAMICTOP_PTR>>2] = DYNAMIC_BASE;






// Endianness check (note: assumes compiler arch was little-endian)

function callRuntimeCallbacks(callbacks) {
  while(callbacks.length > 0) {
    var callback = callbacks.shift();
    if (typeof callback == 'function') {
      callback();
      continue;
    }
    var func = callback.func;
    if (typeof func === 'number') {
      if (callback.arg === undefined) {
        Module['dynCall_v'](func);
      } else {
        Module['dynCall_vi'](func, callback.arg);
      }
    } else {
      func(callback.arg === undefined ? null : callback.arg);
    }
  }
}

var __ATPRERUN__  = []; // functions called before the runtime is initialized
var __ATINIT__    = []; // functions called during startup
var __ATMAIN__    = []; // functions called when main() is to be run
var __ATEXIT__    = []; // functions called during shutdown
var __ATPOSTRUN__ = []; // functions called after the main() is called

var runtimeInitialized = false;
var runtimeExited = false;


function preRun() {
  // compatibility - merge in anything from Module['preRun'] at this time
  if (Module['preRun']) {
    if (typeof Module['preRun'] == 'function') Module['preRun'] = [Module['preRun']];
    while (Module['preRun'].length) {
      addOnPreRun(Module['preRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPRERUN__);
}

function ensureInitRuntime() {
  if (runtimeInitialized) return;
  runtimeInitialized = true;
  if (!Module["noFSInit"] && !FS.init.initialized) FS.init();
TTY.init();
  callRuntimeCallbacks(__ATINIT__);
}

function preMain() {
  FS.ignorePermissions = false;
  callRuntimeCallbacks(__ATMAIN__);
}

function exitRuntime() {
  runtimeExited = true;
}

function postRun() {
  // compatibility - merge in anything from Module['postRun'] at this time
  if (Module['postRun']) {
    if (typeof Module['postRun'] == 'function') Module['postRun'] = [Module['postRun']];
    while (Module['postRun'].length) {
      addOnPostRun(Module['postRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPOSTRUN__);
}

function addOnPreRun(cb) {
  __ATPRERUN__.unshift(cb);
}

function addOnInit(cb) {
  __ATINIT__.unshift(cb);
}

function addOnPreMain(cb) {
  __ATMAIN__.unshift(cb);
}

function addOnExit(cb) {
}

function addOnPostRun(cb) {
  __ATPOSTRUN__.unshift(cb);
}

function unSign(value, bits, ignore) {
  if (value >= 0) {
    return value;
  }
  return bits <= 32 ? 2*Math.abs(1 << (bits-1)) + value // Need some trickery, since if bits == 32, we are right at the limit of the bits JS uses in bitshifts
                    : Math.pow(2, bits)         + value;
}
function reSign(value, bits, ignore) {
  if (value <= 0) {
    return value;
  }
  var half = bits <= 32 ? Math.abs(1 << (bits-1)) // abs is needed if bits == 32
                        : Math.pow(2, bits-1);
  if (value >= half && (bits <= 32 || value > half)) { // for huge values, we can hit the precision limit and always get true here. so don't do that
                                                       // but, in general there is no perfect solution here. With 64-bit ints, we get rounding and errors
                                                       // TODO: In i64 mode 1, resign the two parts separately and safely
    value = -2*half + value; // Cannot bitshift half, as it may be at the limit of the bits JS uses in bitshifts
  }
  return value;
}



var Math_abs = Math.abs;
var Math_cos = Math.cos;
var Math_sin = Math.sin;
var Math_tan = Math.tan;
var Math_acos = Math.acos;
var Math_asin = Math.asin;
var Math_atan = Math.atan;
var Math_atan2 = Math.atan2;
var Math_exp = Math.exp;
var Math_log = Math.log;
var Math_sqrt = Math.sqrt;
var Math_ceil = Math.ceil;
var Math_floor = Math.floor;
var Math_pow = Math.pow;
var Math_imul = Math.imul;
var Math_fround = Math.fround;
var Math_round = Math.round;
var Math_min = Math.min;
var Math_max = Math.max;
var Math_clz32 = Math.clz32;
var Math_trunc = Math.trunc;



// A counter of dependencies for calling run(). If we need to
// do asynchronous work before running, increment this and
// decrement it. Incrementing must happen in a place like
// Module.preRun (used by emcc to add file preloading).
// Note that you can add dependencies in preRun, even though
// it happens right before run - run will be postponed until
// the dependencies are met.
var runDependencies = 0;
var runDependencyWatcher = null;
var dependenciesFulfilled = null; // overridden to take different actions when all run dependencies are fulfilled

function getUniqueRunDependency(id) {
  return id;
}

function addRunDependency(id) {
  runDependencies++;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
}

function removeRunDependency(id) {
  runDependencies--;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (runDependencies == 0) {
    if (runDependencyWatcher !== null) {
      clearInterval(runDependencyWatcher);
      runDependencyWatcher = null;
    }
    if (dependenciesFulfilled) {
      var callback = dependenciesFulfilled;
      dependenciesFulfilled = null;
      callback(); // can add another dependenciesFulfilled
    }
  }
}

Module["preloadedImages"] = {}; // maps url to image data
Module["preloadedAudios"] = {}; // maps url to audio data


var memoryInitializer = null;






// Copyright 2017 The Emscripten Authors.  All rights reserved.
// Emscripten is available under two separate licenses, the MIT license and the
// University of Illinois/NCSA Open Source License.  Both these licenses can be
// found in the LICENSE file.

// Prefix of data URIs emitted by SINGLE_FILE and related options.
var dataURIPrefix = 'data:application/octet-stream;base64,';

// Indicates whether filename is a base64 data URI.
function isDataURI(filename) {
  return String.prototype.startsWith ?
      filename.startsWith(dataURIPrefix) :
      filename.indexOf(dataURIPrefix) === 0;
}





// === Body ===

var ASM_CONSTS = [];





// STATICTOP = STATIC_BASE + 60488;
/* global initializers */  __ATINIT__.push({ func: function() { ___emscripten_environ_constructor() } });


memoryInitializer = "data:application/octet-stream;base64,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";





/* no memory initializer */
var tempDoublePtr = 60480

function copyTempFloat(ptr) { // functions, because inlining this code increases code size too much
  HEAP8[tempDoublePtr] = HEAP8[ptr];
  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];
  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];
  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];
}

function copyTempDouble(ptr) {
  HEAP8[tempDoublePtr] = HEAP8[ptr];
  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];
  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];
  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];
  HEAP8[tempDoublePtr+4] = HEAP8[ptr+4];
  HEAP8[tempDoublePtr+5] = HEAP8[ptr+5];
  HEAP8[tempDoublePtr+6] = HEAP8[ptr+6];
  HEAP8[tempDoublePtr+7] = HEAP8[ptr+7];
}

// {{PRE_LIBRARY}}


  function ___assert_fail(condition, filename, line, func) {
      abort('Assertion failed: ' + UTF8ToString(condition) + ', at: ' + [filename ? UTF8ToString(filename) : 'unknown filename', line, func ? UTF8ToString(func) : 'unknown function']);
    }

  
  var ENV={};function ___buildEnvironment(environ) {
      // WARNING: Arbitrary limit!
      var MAX_ENV_VALUES = 64;
      var TOTAL_ENV_SIZE = 1024;
  
      // Statically allocate memory for the environment.
      var poolPtr;
      var envPtr;
      if (!___buildEnvironment.called) {
        ___buildEnvironment.called = true;
        // Set default values. Use string keys for Closure Compiler compatibility.
        ENV['USER'] = ENV['LOGNAME'] = 'web_user';
        ENV['PATH'] = '/';
        ENV['PWD'] = '/';
        ENV['HOME'] = '/home/web_user';
        ENV['LANG'] = 'C.UTF-8';
        ENV['_'] = Module['thisProgram'];
        // Allocate memory.
        poolPtr = getMemory(TOTAL_ENV_SIZE);
        envPtr = getMemory(MAX_ENV_VALUES * 4);
        HEAP32[((envPtr)>>2)]=poolPtr;
        HEAP32[((environ)>>2)]=envPtr;
      } else {
        envPtr = HEAP32[((environ)>>2)];
        poolPtr = HEAP32[((envPtr)>>2)];
      }
  
      // Collect key=value lines.
      var strings = [];
      var totalSize = 0;
      for (var key in ENV) {
        if (typeof ENV[key] === 'string') {
          var line = key + '=' + ENV[key];
          strings.push(line);
          totalSize += line.length;
        }
      }
      if (totalSize > TOTAL_ENV_SIZE) {
        throw new Error('Environment size exceeded TOTAL_ENV_SIZE!');
      }
  
      // Make new.
      var ptrSize = 4;
      for (var i = 0; i < strings.length; i++) {
        var line = strings[i];
        writeAsciiToMemory(line, poolPtr);
        HEAP32[(((envPtr)+(i * ptrSize))>>2)]=poolPtr;
        poolPtr += line.length + 1;
      }
      HEAP32[(((envPtr)+(strings.length * ptrSize))>>2)]=0;
    }

  
  
   
  
   
  
     

  
    

   

  
  
  
  function ___setErrNo(value) {
      if (Module['___errno_location']) HEAP32[((Module['___errno_location']())>>2)]=value;
      return value;
    }
  
  var PATH={splitPath:function (filename) {
        var splitPathRe = /^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/;
        return splitPathRe.exec(filename).slice(1);
      },normalizeArray:function (parts, allowAboveRoot) {
        // if the path tries to go above the root, `up` ends up > 0
        var up = 0;
        for (var i = parts.length - 1; i >= 0; i--) {
          var last = parts[i];
          if (last === '.') {
            parts.splice(i, 1);
          } else if (last === '..') {
            parts.splice(i, 1);
            up++;
          } else if (up) {
            parts.splice(i, 1);
            up--;
          }
        }
        // if the path is allowed to go above the root, restore leading ..s
        if (allowAboveRoot) {
          for (; up; up--) {
            parts.unshift('..');
          }
        }
        return parts;
      },normalize:function (path) {
        var isAbsolute = path.charAt(0) === '/',
            trailingSlash = path.substr(-1) === '/';
        // Normalize the path
        path = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), !isAbsolute).join('/');
        if (!path && !isAbsolute) {
          path = '.';
        }
        if (path && trailingSlash) {
          path += '/';
        }
        return (isAbsolute ? '/' : '') + path;
      },dirname:function (path) {
        var result = PATH.splitPath(path),
            root = result[0],
            dir = result[1];
        if (!root && !dir) {
          // No dirname whatsoever
          return '.';
        }
        if (dir) {
          // It has a dirname, strip trailing slash
          dir = dir.substr(0, dir.length - 1);
        }
        return root + dir;
      },basename:function (path) {
        // EMSCRIPTEN return '/'' for '/', not an empty string
        if (path === '/') return '/';
        var lastSlash = path.lastIndexOf('/');
        if (lastSlash === -1) return path;
        return path.substr(lastSlash+1);
      },extname:function (path) {
        return PATH.splitPath(path)[3];
      },join:function () {
        var paths = Array.prototype.slice.call(arguments, 0);
        return PATH.normalize(paths.join('/'));
      },join2:function (l, r) {
        return PATH.normalize(l + '/' + r);
      },resolve:function () {
        var resolvedPath = '',
          resolvedAbsolute = false;
        for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
          var path = (i >= 0) ? arguments[i] : FS.cwd();
          // Skip empty and invalid entries
          if (typeof path !== 'string') {
            throw new TypeError('Arguments to path.resolve must be strings');
          } else if (!path) {
            return ''; // an invalid portion invalidates the whole thing
          }
          resolvedPath = path + '/' + resolvedPath;
          resolvedAbsolute = path.charAt(0) === '/';
        }
        // At this point the path should be resolved to a full absolute path, but
        // handle relative paths to be safe (might happen when process.cwd() fails)
        resolvedPath = PATH.normalizeArray(resolvedPath.split('/').filter(function(p) {
          return !!p;
        }), !resolvedAbsolute).join('/');
        return ((resolvedAbsolute ? '/' : '') + resolvedPath) || '.';
      },relative:function (from, to) {
        from = PATH.resolve(from).substr(1);
        to = PATH.resolve(to).substr(1);
        function trim(arr) {
          var start = 0;
          for (; start < arr.length; start++) {
            if (arr[start] !== '') break;
          }
          var end = arr.length - 1;
          for (; end >= 0; end--) {
            if (arr[end] !== '') break;
          }
          if (start > end) return [];
          return arr.slice(start, end - start + 1);
        }
        var fromParts = trim(from.split('/'));
        var toParts = trim(to.split('/'));
        var length = Math.min(fromParts.length, toParts.length);
        var samePartsLength = length;
        for (var i = 0; i < length; i++) {
          if (fromParts[i] !== toParts[i]) {
            samePartsLength = i;
            break;
          }
        }
        var outputParts = [];
        for (var i = samePartsLength; i < fromParts.length; i++) {
          outputParts.push('..');
        }
        outputParts = outputParts.concat(toParts.slice(samePartsLength));
        return outputParts.join('/');
      }};
  
  var TTY={ttys:[],init:function () {
        // https://github.com/emscripten-core/emscripten/pull/1555
        // if (ENVIRONMENT_IS_NODE) {
        //   // currently, FS.init does not distinguish if process.stdin is a file or TTY
        //   // device, it always assumes it's a TTY device. because of this, we're forcing
        //   // process.stdin to UTF8 encoding to at least make stdin reading compatible
        //   // with text files until FS.init can be refactored.
        //   process['stdin']['setEncoding']('utf8');
        // }
      },shutdown:function () {
        // https://github.com/emscripten-core/emscripten/pull/1555
        // if (ENVIRONMENT_IS_NODE) {
        //   // inolen: any idea as to why node -e 'process.stdin.read()' wouldn't exit immediately (with process.stdin being a tty)?
        //   // isaacs: because now it's reading from the stream, you've expressed interest in it, so that read() kicks off a _read() which creates a ReadReq operation
        //   // inolen: I thought read() in that case was a synchronous operation that just grabbed some amount of buffered data if it exists?
        //   // isaacs: it is. but it also triggers a _read() call, which calls readStart() on the handle
        //   // isaacs: do process.stdin.pause() and i'd think it'd probably close the pending call
        //   process['stdin']['pause']();
        // }
      },register:function (dev, ops) {
        TTY.ttys[dev] = { input: [], output: [], ops: ops };
        FS.registerDevice(dev, TTY.stream_ops);
      },stream_ops:{open:function (stream) {
          var tty = TTY.ttys[stream.node.rdev];
          if (!tty) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          stream.tty = tty;
          stream.seekable = false;
        },close:function (stream) {
          // flush any pending line data
          stream.tty.ops.flush(stream.tty);
        },flush:function (stream) {
          stream.tty.ops.flush(stream.tty);
        },read:function (stream, buffer, offset, length, pos /* ignored */) {
          if (!stream.tty || !stream.tty.ops.get_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          var bytesRead = 0;
          for (var i = 0; i < length; i++) {
            var result;
            try {
              result = stream.tty.ops.get_char(stream.tty);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            if (result === undefined && bytesRead === 0) {
              throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
            }
            if (result === null || result === undefined) break;
            bytesRead++;
            buffer[offset+i] = result;
          }
          if (bytesRead) {
            stream.node.timestamp = Date.now();
          }
          return bytesRead;
        },write:function (stream, buffer, offset, length, pos) {
          if (!stream.tty || !stream.tty.ops.put_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          try {
            for (var i = 0; i < length; i++) {
              stream.tty.ops.put_char(stream.tty, buffer[offset+i]);
            }
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES.EIO);
          }
          if (length) {
            stream.node.timestamp = Date.now();
          }
          return i;
        }},default_tty_ops:{get_char:function (tty) {
          if (!tty.input.length) {
            var result = null;
            if (ENVIRONMENT_IS_NODE) {
              // we will read data by chunks of BUFSIZE
              var BUFSIZE = 256;
              var buf = new Buffer(BUFSIZE);
              var bytesRead = 0;
  
              var isPosixPlatform = (process.platform != 'win32'); // Node doesn't offer a direct check, so test by exclusion
  
              var fd = process.stdin.fd;
              if (isPosixPlatform) {
                // Linux and Mac cannot use process.stdin.fd (which isn't set up as sync)
                var usingDevice = false;
                try {
                  fd = fs.openSync('/dev/stdin', 'r');
                  usingDevice = true;
                } catch (e) {}
              }
  
              try {
                bytesRead = fs.readSync(fd, buf, 0, BUFSIZE, null);
              } catch(e) {
                // Cross-platform differences: on Windows, reading EOF throws an exception, but on other OSes,
                // reading EOF returns 0. Uniformize behavior by treating the EOF exception to return 0.
                if (e.toString().indexOf('EOF') != -1) bytesRead = 0;
                else throw e;
              }
  
              if (usingDevice) { fs.closeSync(fd); }
              if (bytesRead > 0) {
                result = buf.slice(0, bytesRead).toString('utf-8');
              } else {
                result = null;
              }
            } else
            if (typeof window != 'undefined' &&
              typeof window.prompt == 'function') {
              // Browser.
              result = window.prompt('Input: ');  // returns null on cancel
              if (result !== null) {
                result += '\n';
              }
            } else if (typeof readline == 'function') {
              // Command line.
              result = readline();
              if (result !== null) {
                result += '\n';
              }
            }
            if (!result) {
              return null;
            }
            tty.input = intArrayFromString(result, true);
          }
          return tty.input.shift();
        },put_char:function (tty, val) {
          if (val === null || val === 10) {
            out(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          } else {
            if (val != 0) tty.output.push(val); // val == 0 would cut text output off in the middle.
          }
        },flush:function (tty) {
          if (tty.output && tty.output.length > 0) {
            out(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          }
        }},default_tty1_ops:{put_char:function (tty, val) {
          if (val === null || val === 10) {
            err(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          } else {
            if (val != 0) tty.output.push(val);
          }
        },flush:function (tty) {
          if (tty.output && tty.output.length > 0) {
            err(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          }
        }}};
  
  var MEMFS={ops_table:null,mount:function (mount) {
        return MEMFS.createNode(null, '/', 16384 | 511 /* 0777 */, 0);
      },createNode:function (parent, name, mode, dev) {
        if (FS.isBlkdev(mode) || FS.isFIFO(mode)) {
          // no supported
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (!MEMFS.ops_table) {
          MEMFS.ops_table = {
            dir: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr,
                lookup: MEMFS.node_ops.lookup,
                mknod: MEMFS.node_ops.mknod,
                rename: MEMFS.node_ops.rename,
                unlink: MEMFS.node_ops.unlink,
                rmdir: MEMFS.node_ops.rmdir,
                readdir: MEMFS.node_ops.readdir,
                symlink: MEMFS.node_ops.symlink
              },
              stream: {
                llseek: MEMFS.stream_ops.llseek
              }
            },
            file: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr
              },
              stream: {
                llseek: MEMFS.stream_ops.llseek,
                read: MEMFS.stream_ops.read,
                write: MEMFS.stream_ops.write,
                allocate: MEMFS.stream_ops.allocate,
                mmap: MEMFS.stream_ops.mmap,
                msync: MEMFS.stream_ops.msync
              }
            },
            link: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr,
                readlink: MEMFS.node_ops.readlink
              },
              stream: {}
            },
            chrdev: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr
              },
              stream: FS.chrdev_stream_ops
            }
          };
        }
        var node = FS.createNode(parent, name, mode, dev);
        if (FS.isDir(node.mode)) {
          node.node_ops = MEMFS.ops_table.dir.node;
          node.stream_ops = MEMFS.ops_table.dir.stream;
          node.contents = {};
        } else if (FS.isFile(node.mode)) {
          node.node_ops = MEMFS.ops_table.file.node;
          node.stream_ops = MEMFS.ops_table.file.stream;
          node.usedBytes = 0; // The actual number of bytes used in the typed array, as opposed to contents.length which gives the whole capacity.
          // When the byte data of the file is populated, this will point to either a typed array, or a normal JS array. Typed arrays are preferred
          // for performance, and used by default. However, typed arrays are not resizable like normal JS arrays are, so there is a small disk size
          // penalty involved for appending file writes that continuously grow a file similar to std::vector capacity vs used -scheme.
          node.contents = null; 
        } else if (FS.isLink(node.mode)) {
          node.node_ops = MEMFS.ops_table.link.node;
          node.stream_ops = MEMFS.ops_table.link.stream;
        } else if (FS.isChrdev(node.mode)) {
          node.node_ops = MEMFS.ops_table.chrdev.node;
          node.stream_ops = MEMFS.ops_table.chrdev.stream;
        }
        node.timestamp = Date.now();
        // add the new node to the parent
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },getFileDataAsRegularArray:function (node) {
        if (node.contents && node.contents.subarray) {
          var arr = [];
          for (var i = 0; i < node.usedBytes; ++i) arr.push(node.contents[i]);
          return arr; // Returns a copy of the original data.
        }
        return node.contents; // No-op, the file contents are already in a JS array. Return as-is.
      },getFileDataAsTypedArray:function (node) {
        if (!node.contents) return new Uint8Array;
        if (node.contents.subarray) return node.contents.subarray(0, node.usedBytes); // Make sure to not return excess unused bytes.
        return new Uint8Array(node.contents);
      },expandFileStorage:function (node, newCapacity) {
        var prevCapacity = node.contents ? node.contents.length : 0;
        if (prevCapacity >= newCapacity) return; // No need to expand, the storage was already large enough.
        // Don't expand strictly to the given requested limit if it's only a very small increase, but instead geometrically grow capacity.
        // For small filesizes (<1MB), perform size*2 geometric increase, but for large sizes, do a much more conservative size*1.125 increase to
        // avoid overshooting the allocation cap by a very large margin.
        var CAPACITY_DOUBLING_MAX = 1024 * 1024;
        newCapacity = Math.max(newCapacity, (prevCapacity * (prevCapacity < CAPACITY_DOUBLING_MAX ? 2.0 : 1.125)) | 0);
        if (prevCapacity != 0) newCapacity = Math.max(newCapacity, 256); // At minimum allocate 256b for each file when expanding.
        var oldContents = node.contents;
        node.contents = new Uint8Array(newCapacity); // Allocate new storage.
        if (node.usedBytes > 0) node.contents.set(oldContents.subarray(0, node.usedBytes), 0); // Copy old data over to the new storage.
        return;
      },resizeFileStorage:function (node, newSize) {
        if (node.usedBytes == newSize) return;
        if (newSize == 0) {
          node.contents = null; // Fully decommit when requesting a resize to zero.
          node.usedBytes = 0;
          return;
        }
        if (!node.contents || node.contents.subarray) { // Resize a typed array if that is being used as the backing store.
          var oldContents = node.contents;
          node.contents = new Uint8Array(new ArrayBuffer(newSize)); // Allocate new storage.
          if (oldContents) {
            node.contents.set(oldContents.subarray(0, Math.min(newSize, node.usedBytes))); // Copy old data over to the new storage.
          }
          node.usedBytes = newSize;
          return;
        }
        // Backing with a JS array.
        if (!node.contents) node.contents = [];
        if (node.contents.length > newSize) node.contents.length = newSize;
        else while (node.contents.length < newSize) node.contents.push(0);
        node.usedBytes = newSize;
      },node_ops:{getattr:function (node) {
          var attr = {};
          // device numbers reuse inode numbers.
          attr.dev = FS.isChrdev(node.mode) ? node.id : 1;
          attr.ino = node.id;
          attr.mode = node.mode;
          attr.nlink = 1;
          attr.uid = 0;
          attr.gid = 0;
          attr.rdev = node.rdev;
          if (FS.isDir(node.mode)) {
            attr.size = 4096;
          } else if (FS.isFile(node.mode)) {
            attr.size = node.usedBytes;
          } else if (FS.isLink(node.mode)) {
            attr.size = node.link.length;
          } else {
            attr.size = 0;
          }
          attr.atime = new Date(node.timestamp);
          attr.mtime = new Date(node.timestamp);
          attr.ctime = new Date(node.timestamp);
          // NOTE: In our implementation, st_blocks = Math.ceil(st_size/st_blksize),
          //       but this is not required by the standard.
          attr.blksize = 4096;
          attr.blocks = Math.ceil(attr.size / attr.blksize);
          return attr;
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
          if (attr.size !== undefined) {
            MEMFS.resizeFileStorage(node, attr.size);
          }
        },lookup:function (parent, name) {
          throw FS.genericErrors[ERRNO_CODES.ENOENT];
        },mknod:function (parent, name, mode, dev) {
          return MEMFS.createNode(parent, name, mode, dev);
        },rename:function (old_node, new_dir, new_name) {
          // if we're overwriting a directory at new_name, make sure it's empty.
          if (FS.isDir(old_node.mode)) {
            var new_node;
            try {
              new_node = FS.lookupNode(new_dir, new_name);
            } catch (e) {
            }
            if (new_node) {
              for (var i in new_node.contents) {
                throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
              }
            }
          }
          // do the internal rewiring
          delete old_node.parent.contents[old_node.name];
          old_node.name = new_name;
          new_dir.contents[new_name] = old_node;
          old_node.parent = new_dir;
        },unlink:function (parent, name) {
          delete parent.contents[name];
        },rmdir:function (parent, name) {
          var node = FS.lookupNode(parent, name);
          for (var i in node.contents) {
            throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
          }
          delete parent.contents[name];
        },readdir:function (node) {
          var entries = ['.', '..']
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newname, oldpath) {
          var node = MEMFS.createNode(parent, newname, 511 /* 0777 */ | 40960, 0);
          node.link = oldpath;
          return node;
        },readlink:function (node) {
          if (!FS.isLink(node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return node.link;
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          var contents = stream.node.contents;
          if (position >= stream.node.usedBytes) return 0;
          var size = Math.min(stream.node.usedBytes - position, length);
          if (size > 8 && contents.subarray) { // non-trivial, and typed array
            buffer.set(contents.subarray(position, position + size), offset);
          } else {
            for (var i = 0; i < size; i++) buffer[offset + i] = contents[position + i];
          }
          return size;
        },write:function (stream, buffer, offset, length, position, canOwn) {
  
          if (!length) return 0;
          var node = stream.node;
          node.timestamp = Date.now();
  
          if (buffer.subarray && (!node.contents || node.contents.subarray)) { // This write is from a typed array to a typed array?
            if (canOwn) {
              node.contents = buffer.subarray(offset, offset + length);
              node.usedBytes = length;
              return length;
            } else if (node.usedBytes === 0 && position === 0) { // If this is a simple first write to an empty file, do a fast set since we don't need to care about old data.
              node.contents = new Uint8Array(buffer.subarray(offset, offset + length));
              node.usedBytes = length;
              return length;
            } else if (position + length <= node.usedBytes) { // Writing to an already allocated and used subrange of the file?
              node.contents.set(buffer.subarray(offset, offset + length), position);
              return length;
            }
          }
  
          // Appending to an existing file and we need to reallocate, or source data did not come as a typed array.
          MEMFS.expandFileStorage(node, position+length);
          if (node.contents.subarray && buffer.subarray) node.contents.set(buffer.subarray(offset, offset + length), position); // Use typed array write if available.
          else {
            for (var i = 0; i < length; i++) {
             node.contents[position + i] = buffer[offset + i]; // Or fall back to manual write if not.
            }
          }
          node.usedBytes = Math.max(node.usedBytes, position+length);
          return length;
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.usedBytes;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return position;
        },allocate:function (stream, offset, length) {
          MEMFS.expandFileStorage(stream.node, offset + length);
          stream.node.usedBytes = Math.max(stream.node.usedBytes, offset + length);
        },mmap:function (stream, buffer, offset, length, position, prot, flags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          var ptr;
          var allocated;
          var contents = stream.node.contents;
          // Only make a new copy when MAP_PRIVATE is specified.
          if ( !(flags & 2) &&
                (contents.buffer === buffer || contents.buffer === buffer.buffer) ) {
            // We can't emulate MAP_SHARED when the file is not backed by the buffer
            // we're mapping to (e.g. the HEAP buffer).
            allocated = false;
            ptr = contents.byteOffset;
          } else {
            // Try to avoid unnecessary slices.
            if (position > 0 || position + length < stream.node.usedBytes) {
              if (contents.subarray) {
                contents = contents.subarray(position, position + length);
              } else {
                contents = Array.prototype.slice.call(contents, position, position + length);
              }
            }
            allocated = true;
            ptr = _malloc(length);
            if (!ptr) {
              throw new FS.ErrnoError(ERRNO_CODES.ENOMEM);
            }
            buffer.set(contents, ptr);
          }
          return { ptr: ptr, allocated: allocated };
        },msync:function (stream, buffer, offset, length, mmapFlags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          if (mmapFlags & 2) {
            // MAP_PRIVATE calls need not to be synced back to underlying fs
            return 0;
          }
  
          var bytesWritten = MEMFS.stream_ops.write(stream, buffer, 0, length, offset, false);
          // should we check if bytesWritten and length are the same?
          return 0;
        }}};
  
  var IDBFS={dbs:{},indexedDB:function () {
        if (typeof indexedDB !== 'undefined') return indexedDB;
        var ret = null;
        if (typeof window === 'object') ret = window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;
        assert(ret, 'IDBFS used, but indexedDB not supported');
        return ret;
      },DB_VERSION:21,DB_STORE_NAME:"FILE_DATA",mount:function (mount) {
        // reuse all of the core MEMFS functionality
        return MEMFS.mount.apply(null, arguments);
      },syncfs:function (mount, populate, callback) {
        IDBFS.getLocalSet(mount, function(err, local) {
          if (err) return callback(err);
  
          IDBFS.getRemoteSet(mount, function(err, remote) {
            if (err) return callback(err);
  
            var src = populate ? remote : local;
            var dst = populate ? local : remote;
  
            IDBFS.reconcile(src, dst, callback);
          });
        });
      },getDB:function (name, callback) {
        // check the cache first
        var db = IDBFS.dbs[name];
        if (db) {
          return callback(null, db);
        }
  
        var req;
        try {
          req = IDBFS.indexedDB().open(name, IDBFS.DB_VERSION);
        } catch (e) {
          return callback(e);
        }
        if (!req) {
          return callback("Unable to connect to IndexedDB");
        }
        req.onupgradeneeded = function(e) {
          var db = e.target.result;
          var transaction = e.target.transaction;
  
          var fileStore;
  
          if (db.objectStoreNames.contains(IDBFS.DB_STORE_NAME)) {
            fileStore = transaction.objectStore(IDBFS.DB_STORE_NAME);
          } else {
            fileStore = db.createObjectStore(IDBFS.DB_STORE_NAME);
          }
  
          if (!fileStore.indexNames.contains('timestamp')) {
            fileStore.createIndex('timestamp', 'timestamp', { unique: false });
          }
        };
        req.onsuccess = function() {
          db = req.result;
  
          // add to the cache
          IDBFS.dbs[name] = db;
          callback(null, db);
        };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },getLocalSet:function (mount, callback) {
        var entries = {};
  
        function isRealDir(p) {
          return p !== '.' && p !== '..';
        };
        function toAbsolute(root) {
          return function(p) {
            return PATH.join2(root, p);
          }
        };
  
        var check = FS.readdir(mount.mountpoint).filter(isRealDir).map(toAbsolute(mount.mountpoint));
  
        while (check.length) {
          var path = check.pop();
          var stat;
  
          try {
            stat = FS.stat(path);
          } catch (e) {
            return callback(e);
          }
  
          if (FS.isDir(stat.mode)) {
            check.push.apply(check, FS.readdir(path).filter(isRealDir).map(toAbsolute(path)));
          }
  
          entries[path] = { timestamp: stat.mtime };
        }
  
        return callback(null, { type: 'local', entries: entries });
      },getRemoteSet:function (mount, callback) {
        var entries = {};
  
        IDBFS.getDB(mount.mountpoint, function(err, db) {
          if (err) return callback(err);
  
          try {
            var transaction = db.transaction([IDBFS.DB_STORE_NAME], 'readonly');
            transaction.onerror = function(e) {
              callback(this.error);
              e.preventDefault();
            };
  
            var store = transaction.objectStore(IDBFS.DB_STORE_NAME);
            var index = store.index('timestamp');
  
            index.openKeyCursor().onsuccess = function(event) {
              var cursor = event.target.result;
  
              if (!cursor) {
                return callback(null, { type: 'remote', db: db, entries: entries });
              }
  
              entries[cursor.primaryKey] = { timestamp: cursor.key };
  
              cursor.continue();
            };
          } catch (e) {
            return callback(e);
          }
        });
      },loadLocalEntry:function (path, callback) {
        var stat, node;
  
        try {
          var lookup = FS.lookupPath(path);
          node = lookup.node;
          stat = FS.stat(path);
        } catch (e) {
          return callback(e);
        }
  
        if (FS.isDir(stat.mode)) {
          return callback(null, { timestamp: stat.mtime, mode: stat.mode });
        } else if (FS.isFile(stat.mode)) {
          // Performance consideration: storing a normal JavaScript array to a IndexedDB is much slower than storing a typed array.
          // Therefore always convert the file contents to a typed array first before writing the data to IndexedDB.
          node.contents = MEMFS.getFileDataAsTypedArray(node);
          return callback(null, { timestamp: stat.mtime, mode: stat.mode, contents: node.contents });
        } else {
          return callback(new Error('node type not supported'));
        }
      },storeLocalEntry:function (path, entry, callback) {
        try {
          if (FS.isDir(entry.mode)) {
            FS.mkdir(path, entry.mode);
          } else if (FS.isFile(entry.mode)) {
            FS.writeFile(path, entry.contents, { canOwn: true });
          } else {
            return callback(new Error('node type not supported'));
          }
  
          FS.chmod(path, entry.mode);
          FS.utime(path, entry.timestamp, entry.timestamp);
        } catch (e) {
          return callback(e);
        }
  
        callback(null);
      },removeLocalEntry:function (path, callback) {
        try {
          var lookup = FS.lookupPath(path);
          var stat = FS.stat(path);
  
          if (FS.isDir(stat.mode)) {
            FS.rmdir(path);
          } else if (FS.isFile(stat.mode)) {
            FS.unlink(path);
          }
        } catch (e) {
          return callback(e);
        }
  
        callback(null);
      },loadRemoteEntry:function (store, path, callback) {
        var req = store.get(path);
        req.onsuccess = function(event) { callback(null, event.target.result); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },storeRemoteEntry:function (store, path, entry, callback) {
        var req = store.put(entry, path);
        req.onsuccess = function() { callback(null); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },removeRemoteEntry:function (store, path, callback) {
        var req = store.delete(path);
        req.onsuccess = function() { callback(null); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },reconcile:function (src, dst, callback) {
        var total = 0;
  
        var create = [];
        Object.keys(src.entries).forEach(function (key) {
          var e = src.entries[key];
          var e2 = dst.entries[key];
          if (!e2 || e.timestamp > e2.timestamp) {
            create.push(key);
            total++;
          }
        });
  
        var remove = [];
        Object.keys(dst.entries).forEach(function (key) {
          var e = dst.entries[key];
          var e2 = src.entries[key];
          if (!e2) {
            remove.push(key);
            total++;
          }
        });
  
        if (!total) {
          return callback(null);
        }
  
        var errored = false;
        var completed = 0;
        var db = src.type === 'remote' ? src.db : dst.db;
        var transaction = db.transaction([IDBFS.DB_STORE_NAME], 'readwrite');
        var store = transaction.objectStore(IDBFS.DB_STORE_NAME);
  
        function done(err) {
          if (err) {
            if (!done.errored) {
              done.errored = true;
              return callback(err);
            }
            return;
          }
          if (++completed >= total) {
            return callback(null);
          }
        };
  
        transaction.onerror = function(e) {
          done(this.error);
          e.preventDefault();
        };
  
        // sort paths in ascending order so directory entries are created
        // before the files inside them
        create.sort().forEach(function (path) {
          if (dst.type === 'local') {
            IDBFS.loadRemoteEntry(store, path, function (err, entry) {
              if (err) return done(err);
              IDBFS.storeLocalEntry(path, entry, done);
            });
          } else {
            IDBFS.loadLocalEntry(path, function (err, entry) {
              if (err) return done(err);
              IDBFS.storeRemoteEntry(store, path, entry, done);
            });
          }
        });
  
        // sort paths in descending order so files are deleted before their
        // parent directories
        remove.sort().reverse().forEach(function(path) {
          if (dst.type === 'local') {
            IDBFS.removeLocalEntry(path, done);
          } else {
            IDBFS.removeRemoteEntry(store, path, done);
          }
        });
      }};
  
  var NODEFS={isWindows:false,staticInit:function () {
        NODEFS.isWindows = !!process.platform.match(/^win/);
        var flags = process["binding"]("constants");
        // Node.js 4 compatibility: it has no namespaces for constants
        if (flags["fs"]) {
          flags = flags["fs"];
        }
        NODEFS.flagsForNodeMap = {
          "1024": flags["O_APPEND"],
          "64": flags["O_CREAT"],
          "128": flags["O_EXCL"],
          "0": flags["O_RDONLY"],
          "2": flags["O_RDWR"],
          "4096": flags["O_SYNC"],
          "512": flags["O_TRUNC"],
          "1": flags["O_WRONLY"]
        };
      },bufferFrom:function (arrayBuffer) {
        // Node.js < 4.5 compatibility: Buffer.from does not support ArrayBuffer
        // Buffer.from before 4.5 was just a method inherited from Uint8Array
        // Buffer.alloc has been added with Buffer.from together, so check it instead
        return Buffer.alloc ? Buffer.from(arrayBuffer) : new Buffer(arrayBuffer);
      },mount:function (mount) {
        assert(ENVIRONMENT_IS_NODE);
        return NODEFS.createNode(null, '/', NODEFS.getMode(mount.opts.root), 0);
      },createNode:function (parent, name, mode, dev) {
        if (!FS.isDir(mode) && !FS.isFile(mode) && !FS.isLink(mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var node = FS.createNode(parent, name, mode);
        node.node_ops = NODEFS.node_ops;
        node.stream_ops = NODEFS.stream_ops;
        return node;
      },getMode:function (path) {
        var stat;
        try {
          stat = fs.lstatSync(path);
          if (NODEFS.isWindows) {
            // Node.js on Windows never represents permission bit 'x', so
            // propagate read bits to execute bits
            stat.mode = stat.mode | ((stat.mode & 292) >> 2);
          }
        } catch (e) {
          if (!e.code) throw e;
          throw new FS.ErrnoError(ERRNO_CODES[e.code]);
        }
        return stat.mode;
      },realPath:function (node) {
        var parts = [];
        while (node.parent !== node) {
          parts.push(node.name);
          node = node.parent;
        }
        parts.push(node.mount.opts.root);
        parts.reverse();
        return PATH.join.apply(null, parts);
      },flagsForNode:function (flags) {
        flags &= ~0x200000 /*O_PATH*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x800 /*O_NONBLOCK*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x8000 /*O_LARGEFILE*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x80000 /*O_CLOEXEC*/; // Some applications may pass it; it makes no sense for a single process.
        var newFlags = 0;
        for (var k in NODEFS.flagsForNodeMap) {
          if (flags & k) {
            newFlags |= NODEFS.flagsForNodeMap[k];
            flags ^= k;
          }
        }
  
        if (!flags) {
          return newFlags;
        } else {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
      },node_ops:{getattr:function (node) {
          var path = NODEFS.realPath(node);
          var stat;
          try {
            stat = fs.lstatSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          // node.js v0.10.20 doesn't report blksize and blocks on Windows. Fake them with default blksize of 4096.
          // See http://support.microsoft.com/kb/140365
          if (NODEFS.isWindows && !stat.blksize) {
            stat.blksize = 4096;
          }
          if (NODEFS.isWindows && !stat.blocks) {
            stat.blocks = (stat.size+stat.blksize-1)/stat.blksize|0;
          }
          return {
            dev: stat.dev,
            ino: stat.ino,
            mode: stat.mode,
            nlink: stat.nlink,
            uid: stat.uid,
            gid: stat.gid,
            rdev: stat.rdev,
            size: stat.size,
            atime: stat.atime,
            mtime: stat.mtime,
            ctime: stat.ctime,
            blksize: stat.blksize,
            blocks: stat.blocks
          };
        },setattr:function (node, attr) {
          var path = NODEFS.realPath(node);
          try {
            if (attr.mode !== undefined) {
              fs.chmodSync(path, attr.mode);
              // update the common node structure mode as well
              node.mode = attr.mode;
            }
            if (attr.timestamp !== undefined) {
              var date = new Date(attr.timestamp);
              fs.utimesSync(path, date, date);
            }
            if (attr.size !== undefined) {
              fs.truncateSync(path, attr.size);
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },lookup:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          var mode = NODEFS.getMode(path);
          return NODEFS.createNode(parent, name, mode);
        },mknod:function (parent, name, mode, dev) {
          var node = NODEFS.createNode(parent, name, mode, dev);
          // create the backing node for this in the fs root as well
          var path = NODEFS.realPath(node);
          try {
            if (FS.isDir(node.mode)) {
              fs.mkdirSync(path, node.mode);
            } else {
              fs.writeFileSync(path, '', { mode: node.mode });
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          return node;
        },rename:function (oldNode, newDir, newName) {
          var oldPath = NODEFS.realPath(oldNode);
          var newPath = PATH.join2(NODEFS.realPath(newDir), newName);
          try {
            fs.renameSync(oldPath, newPath);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },unlink:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          try {
            fs.unlinkSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },rmdir:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          try {
            fs.rmdirSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },readdir:function (node) {
          var path = NODEFS.realPath(node);
          try {
            return fs.readdirSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },symlink:function (parent, newName, oldPath) {
          var newPath = PATH.join2(NODEFS.realPath(parent), newName);
          try {
            fs.symlinkSync(oldPath, newPath);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },readlink:function (node) {
          var path = NODEFS.realPath(node);
          try {
            path = fs.readlinkSync(path);
            path = NODEJS_PATH.relative(NODEJS_PATH.resolve(node.mount.opts.root), path);
            return path;
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        }},stream_ops:{open:function (stream) {
          var path = NODEFS.realPath(stream.node);
          try {
            if (FS.isFile(stream.node.mode)) {
              stream.nfd = fs.openSync(path, NODEFS.flagsForNode(stream.flags));
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },close:function (stream) {
          try {
            if (FS.isFile(stream.node.mode) && stream.nfd) {
              fs.closeSync(stream.nfd);
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },read:function (stream, buffer, offset, length, position) {
          // Node.js < 6 compatibility: node errors on 0 length reads
          if (length === 0) return 0;
          try {
            return fs.readSync(stream.nfd, NODEFS.bufferFrom(buffer.buffer), offset, length, position);
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },write:function (stream, buffer, offset, length, position) {
          try {
            return fs.writeSync(stream.nfd, NODEFS.bufferFrom(buffer.buffer), offset, length, position);
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              try {
                var stat = fs.fstatSync(stream.nfd);
                position += stat.size;
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES[e.code]);
              }
            }
          }
  
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
  
          return position;
        }}};
  
  var WORKERFS={DIR_MODE:16895,FILE_MODE:33279,reader:null,mount:function (mount) {
        assert(ENVIRONMENT_IS_WORKER);
        if (!WORKERFS.reader) WORKERFS.reader = new FileReaderSync();
        var root = WORKERFS.createNode(null, '/', WORKERFS.DIR_MODE, 0);
        var createdParents = {};
        function ensureParent(path) {
          // return the parent node, creating subdirs as necessary
          var parts = path.split('/');
          var parent = root;
          for (var i = 0; i < parts.length-1; i++) {
            var curr = parts.slice(0, i+1).join('/');
            // Issue 4254: Using curr as a node name will prevent the node
            // from being found in FS.nameTable when FS.open is called on
            // a path which holds a child of this node,
            // given that all FS functions assume node names
            // are just their corresponding parts within their given path,
            // rather than incremental aggregates which include their parent's
            // directories.
            if (!createdParents[curr]) {
              createdParents[curr] = WORKERFS.createNode(parent, parts[i], WORKERFS.DIR_MODE, 0);
            }
            parent = createdParents[curr];
          }
          return parent;
        }
        function base(path) {
          var parts = path.split('/');
          return parts[parts.length-1];
        }
        // We also accept FileList here, by using Array.prototype
        Array.prototype.forEach.call(mount.opts["files"] || [], function(file) {
          WORKERFS.createNode(ensureParent(file.name), base(file.name), WORKERFS.FILE_MODE, 0, file, file.lastModifiedDate);
        });
        (mount.opts["blobs"] || []).forEach(function(obj) {
          WORKERFS.createNode(ensureParent(obj["name"]), base(obj["name"]), WORKERFS.FILE_MODE, 0, obj["data"]);
        });
        (mount.opts["packages"] || []).forEach(function(pack) {
          pack['metadata'].files.forEach(function(file) {
            var name = file.filename.substr(1); // remove initial slash
            WORKERFS.createNode(ensureParent(name), base(name), WORKERFS.FILE_MODE, 0, pack['blob'].slice(file.start, file.end));
          });
        });
        return root;
      },createNode:function (parent, name, mode, dev, contents, mtime) {
        var node = FS.createNode(parent, name, mode);
        node.mode = mode;
        node.node_ops = WORKERFS.node_ops;
        node.stream_ops = WORKERFS.stream_ops;
        node.timestamp = (mtime || new Date).getTime();
        assert(WORKERFS.FILE_MODE !== WORKERFS.DIR_MODE);
        if (mode === WORKERFS.FILE_MODE) {
          node.size = contents.size;
          node.contents = contents;
        } else {
          node.size = 4096;
          node.contents = {};
        }
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },node_ops:{getattr:function (node) {
          return {
            dev: 1,
            ino: undefined,
            mode: node.mode,
            nlink: 1,
            uid: 0,
            gid: 0,
            rdev: undefined,
            size: node.size,
            atime: new Date(node.timestamp),
            mtime: new Date(node.timestamp),
            ctime: new Date(node.timestamp),
            blksize: 4096,
            blocks: Math.ceil(node.size / 4096),
          };
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
        },lookup:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        },mknod:function (parent, name, mode, dev) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },rename:function (oldNode, newDir, newName) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },unlink:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },rmdir:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },readdir:function (node) {
          var entries = ['.', '..'];
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newName, oldPath) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },readlink:function (node) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          if (position >= stream.node.size) return 0;
          var chunk = stream.node.contents.slice(position, position + length);
          var ab = WORKERFS.reader.readAsArrayBuffer(chunk);
          buffer.set(new Uint8Array(ab), offset);
          return chunk.size;
        },write:function (stream, buffer, offset, length, position) {
          throw new FS.ErrnoError(ERRNO_CODES.EIO);
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.size;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return position;
        }}};
  
  var _stdin=60256;
  
  var _stdout=60272;
  
  var _stderr=60288;var FS={root:null,mounts:[],devices:{},streams:[],nextInode:1,nameTable:null,currentPath:"/",initialized:false,ignorePermissions:true,trackingDelegate:{},tracking:{openFlags:{READ:1,WRITE:2}},ErrnoError:null,genericErrors:{},filesystems:null,syncFSRequests:0,handleFSError:function (e) {
        if (!(e instanceof FS.ErrnoError)) throw e + ' : ' + stackTrace();
        return ___setErrNo(e.errno);
      },lookupPath:function (path, opts) {
        path = PATH.resolve(FS.cwd(), path);
        opts = opts || {};
  
        if (!path) return { path: '', node: null };
  
        var defaults = {
          follow_mount: true,
          recurse_count: 0
        };
        for (var key in defaults) {
          if (opts[key] === undefined) {
            opts[key] = defaults[key];
          }
        }
  
        if (opts.recurse_count > 8) {  // max recursive lookup of 8
          throw new FS.ErrnoError(40);
        }
  
        // split the path
        var parts = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), false);
  
        // start at the root
        var current = FS.root;
        var current_path = '/';
  
        for (var i = 0; i < parts.length; i++) {
          var islast = (i === parts.length-1);
          if (islast && opts.parent) {
            // stop resolving
            break;
          }
  
          current = FS.lookupNode(current, parts[i]);
          current_path = PATH.join2(current_path, parts[i]);
  
          // jump to the mount's root node if this is a mountpoint
          if (FS.isMountpoint(current)) {
            if (!islast || (islast && opts.follow_mount)) {
              current = current.mounted.root;
            }
          }
  
          // by default, lookupPath will not follow a symlink if it is the final path component.
          // setting opts.follow = true will override this behavior.
          if (!islast || opts.follow) {
            var count = 0;
            while (FS.isLink(current.mode)) {
              var link = FS.readlink(current_path);
              current_path = PATH.resolve(PATH.dirname(current_path), link);
  
              var lookup = FS.lookupPath(current_path, { recurse_count: opts.recurse_count });
              current = lookup.node;
  
              if (count++ > 40) {  // limit max consecutive symlinks to 40 (SYMLOOP_MAX).
                throw new FS.ErrnoError(40);
              }
            }
          }
        }
  
        return { path: current_path, node: current };
      },getPath:function (node) {
        var path;
        while (true) {
          if (FS.isRoot(node)) {
            var mount = node.mount.mountpoint;
            if (!path) return mount;
            return mount[mount.length-1] !== '/' ? mount + '/' + path : mount + path;
          }
          path = path ? node.name + '/' + path : node.name;
          node = node.parent;
        }
      },hashName:function (parentid, name) {
        var hash = 0;
  
  
        for (var i = 0; i < name.length; i++) {
          hash = ((hash << 5) - hash + name.charCodeAt(i)) | 0;
        }
        return ((parentid + hash) >>> 0) % FS.nameTable.length;
      },hashAddNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        node.name_next = FS.nameTable[hash];
        FS.nameTable[hash] = node;
      },hashRemoveNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        if (FS.nameTable[hash] === node) {
          FS.nameTable[hash] = node.name_next;
        } else {
          var current = FS.nameTable[hash];
          while (current) {
            if (current.name_next === node) {
              current.name_next = node.name_next;
              break;
            }
            current = current.name_next;
          }
        }
      },lookupNode:function (parent, name) {
        var err = FS.mayLookup(parent);
        if (err) {
          throw new FS.ErrnoError(err, parent);
        }
        var hash = FS.hashName(parent.id, name);
        for (var node = FS.nameTable[hash]; node; node = node.name_next) {
          var nodeName = node.name;
          if (node.parent.id === parent.id && nodeName === name) {
            return node;
          }
        }
        // if we failed to find it in the cache, call into the VFS
        return FS.lookup(parent, name);
      },createNode:function (parent, name, mode, rdev) {
        if (!FS.FSNode) {
          FS.FSNode = function(parent, name, mode, rdev) {
            if (!parent) {
              parent = this;  // root node sets parent to itself
            }
            this.parent = parent;
            this.mount = parent.mount;
            this.mounted = null;
            this.id = FS.nextInode++;
            this.name = name;
            this.mode = mode;
            this.node_ops = {};
            this.stream_ops = {};
            this.rdev = rdev;
          };
  
          FS.FSNode.prototype = {};
  
          // compatibility
          var readMode = 292 | 73;
          var writeMode = 146;
  
          // NOTE we must use Object.defineProperties instead of individual calls to
          // Object.defineProperty in order to make closure compiler happy
          Object.defineProperties(FS.FSNode.prototype, {
            read: {
              get: function() { return (this.mode & readMode) === readMode; },
              set: function(val) { val ? this.mode |= readMode : this.mode &= ~readMode; }
            },
            write: {
              get: function() { return (this.mode & writeMode) === writeMode; },
              set: function(val) { val ? this.mode |= writeMode : this.mode &= ~writeMode; }
            },
            isFolder: {
              get: function() { return FS.isDir(this.mode); }
            },
            isDevice: {
              get: function() { return FS.isChrdev(this.mode); }
            }
          });
        }
  
        var node = new FS.FSNode(parent, name, mode, rdev);
  
        FS.hashAddNode(node);
  
        return node;
      },destroyNode:function (node) {
        FS.hashRemoveNode(node);
      },isRoot:function (node) {
        return node === node.parent;
      },isMountpoint:function (node) {
        return !!node.mounted;
      },isFile:function (mode) {
        return (mode & 61440) === 32768;
      },isDir:function (mode) {
        return (mode & 61440) === 16384;
      },isLink:function (mode) {
        return (mode & 61440) === 40960;
      },isChrdev:function (mode) {
        return (mode & 61440) === 8192;
      },isBlkdev:function (mode) {
        return (mode & 61440) === 24576;
      },isFIFO:function (mode) {
        return (mode & 61440) === 4096;
      },isSocket:function (mode) {
        return (mode & 49152) === 49152;
      },flagModes:{"r":0,"rs":1052672,"r+":2,"w":577,"wx":705,"xw":705,"w+":578,"wx+":706,"xw+":706,"a":1089,"ax":1217,"xa":1217,"a+":1090,"ax+":1218,"xa+":1218},modeStringToFlags:function (str) {
        var flags = FS.flagModes[str];
        if (typeof flags === 'undefined') {
          throw new Error('Unknown file open mode: ' + str);
        }
        return flags;
      },flagsToPermissionString:function (flag) {
        var perms = ['r', 'w', 'rw'][flag & 3];
        if ((flag & 512)) {
          perms += 'w';
        }
        return perms;
      },nodePermissions:function (node, perms) {
        if (FS.ignorePermissions) {
          return 0;
        }
        // return 0 if any user, group or owner bits are set.
        if (perms.indexOf('r') !== -1 && !(node.mode & 292)) {
          return 13;
        } else if (perms.indexOf('w') !== -1 && !(node.mode & 146)) {
          return 13;
        } else if (perms.indexOf('x') !== -1 && !(node.mode & 73)) {
          return 13;
        }
        return 0;
      },mayLookup:function (dir) {
        var err = FS.nodePermissions(dir, 'x');
        if (err) return err;
        if (!dir.node_ops.lookup) return 13;
        return 0;
      },mayCreate:function (dir, name) {
        try {
          var node = FS.lookupNode(dir, name);
          return 17;
        } catch (e) {
        }
        return FS.nodePermissions(dir, 'wx');
      },mayDelete:function (dir, name, isdir) {
        var node;
        try {
          node = FS.lookupNode(dir, name);
        } catch (e) {
          return e.errno;
        }
        var err = FS.nodePermissions(dir, 'wx');
        if (err) {
          return err;
        }
        if (isdir) {
          if (!FS.isDir(node.mode)) {
            return 20;
          }
          if (FS.isRoot(node) || FS.getPath(node) === FS.cwd()) {
            return 16;
          }
        } else {
          if (FS.isDir(node.mode)) {
            return 21;
          }
        }
        return 0;
      },mayOpen:function (node, flags) {
        if (!node) {
          return 2;
        }
        if (FS.isLink(node.mode)) {
          return 40;
        } else if (FS.isDir(node.mode)) {
          if (FS.flagsToPermissionString(flags) !== 'r' || // opening for write
              (flags & 512)) { // TODO: check for O_SEARCH? (== search for dir only)
            return 21;
          }
        }
        return FS.nodePermissions(node, FS.flagsToPermissionString(flags));
      },MAX_OPEN_FDS:4096,nextfd:function (fd_start, fd_end) {
        fd_start = fd_start || 0;
        fd_end = fd_end || FS.MAX_OPEN_FDS;
        for (var fd = fd_start; fd <= fd_end; fd++) {
          if (!FS.streams[fd]) {
            return fd;
          }
        }
        throw new FS.ErrnoError(24);
      },getStream:function (fd) {
        return FS.streams[fd];
      },createStream:function (stream, fd_start, fd_end) {
        if (!FS.FSStream) {
          FS.FSStream = function(){};
          FS.FSStream.prototype = {};
          // compatibility
          Object.defineProperties(FS.FSStream.prototype, {
            object: {
              get: function() { return this.node; },
              set: function(val) { this.node = val; }
            },
            isRead: {
              get: function() { return (this.flags & 2097155) !== 1; }
            },
            isWrite: {
              get: function() { return (this.flags & 2097155) !== 0; }
            },
            isAppend: {
              get: function() { return (this.flags & 1024); }
            }
          });
        }
        // clone it, so we can return an instance of FSStream
        var newStream = new FS.FSStream();
        for (var p in stream) {
          newStream[p] = stream[p];
        }
        stream = newStream;
        var fd = FS.nextfd(fd_start, fd_end);
        stream.fd = fd;
        FS.streams[fd] = stream;
        return stream;
      },closeStream:function (fd) {
        FS.streams[fd] = null;
      },chrdev_stream_ops:{open:function (stream) {
          var device = FS.getDevice(stream.node.rdev);
          // override node's stream ops with the device's
          stream.stream_ops = device.stream_ops;
          // forward the open call
          if (stream.stream_ops.open) {
            stream.stream_ops.open(stream);
          }
        },llseek:function () {
          throw new FS.ErrnoError(29);
        }},major:function (dev) {
        return ((dev) >> 8);
      },minor:function (dev) {
        return ((dev) & 0xff);
      },makedev:function (ma, mi) {
        return ((ma) << 8 | (mi));
      },registerDevice:function (dev, ops) {
        FS.devices[dev] = { stream_ops: ops };
      },getDevice:function (dev) {
        return FS.devices[dev];
      },getMounts:function (mount) {
        var mounts = [];
        var check = [mount];
  
        while (check.length) {
          var m = check.pop();
  
          mounts.push(m);
  
          check.push.apply(check, m.mounts);
        }
  
        return mounts;
      },syncfs:function (populate, callback) {
        if (typeof(populate) === 'function') {
          callback = populate;
          populate = false;
        }
  
        FS.syncFSRequests++;
  
        if (FS.syncFSRequests > 1) {
          console.log('warning: ' + FS.syncFSRequests + ' FS.syncfs operations in flight at once, probably just doing extra work');
        }
  
        var mounts = FS.getMounts(FS.root.mount);
        var completed = 0;
  
        function doCallback(err) {
          FS.syncFSRequests--;
          return callback(err);
        }
  
        function done(err) {
          if (err) {
            if (!done.errored) {
              done.errored = true;
              return doCallback(err);
            }
            return;
          }
          if (++completed >= mounts.length) {
            doCallback(null);
          }
        };
  
        // sync all mounts
        mounts.forEach(function (mount) {
          if (!mount.type.syncfs) {
            return done(null);
          }
          mount.type.syncfs(mount, populate, done);
        });
      },mount:function (type, opts, mountpoint) {
        var root = mountpoint === '/';
        var pseudo = !mountpoint;
        var node;
  
        if (root && FS.root) {
          throw new FS.ErrnoError(16);
        } else if (!root && !pseudo) {
          var lookup = FS.lookupPath(mountpoint, { follow_mount: false });
  
          mountpoint = lookup.path;  // use the absolute path
          node = lookup.node;
  
          if (FS.isMountpoint(node)) {
            throw new FS.ErrnoError(16);
          }
  
          if (!FS.isDir(node.mode)) {
            throw new FS.ErrnoError(20);
          }
        }
  
        var mount = {
          type: type,
          opts: opts,
          mountpoint: mountpoint,
          mounts: []
        };
  
        // create a root node for the fs
        var mountRoot = type.mount(mount);
        mountRoot.mount = mount;
        mount.root = mountRoot;
  
        if (root) {
          FS.root = mountRoot;
        } else if (node) {
          // set as a mountpoint
          node.mounted = mount;
  
          // add the new mount to the current mount's children
          if (node.mount) {
            node.mount.mounts.push(mount);
          }
        }
  
        return mountRoot;
      },unmount:function (mountpoint) {
        var lookup = FS.lookupPath(mountpoint, { follow_mount: false });
  
        if (!FS.isMountpoint(lookup.node)) {
          throw new FS.ErrnoError(22);
        }
  
        // destroy the nodes for this mount, and all its child mounts
        var node = lookup.node;
        var mount = node.mounted;
        var mounts = FS.getMounts(mount);
  
        Object.keys(FS.nameTable).forEach(function (hash) {
          var current = FS.nameTable[hash];
  
          while (current) {
            var next = current.name_next;
  
            if (mounts.indexOf(current.mount) !== -1) {
              FS.destroyNode(current);
            }
  
            current = next;
          }
        });
  
        // no longer a mountpoint
        node.mounted = null;
  
        // remove this mount from the child mounts
        var idx = node.mount.mounts.indexOf(mount);
        node.mount.mounts.splice(idx, 1);
      },lookup:function (parent, name) {
        return parent.node_ops.lookup(parent, name);
      },mknod:function (path, mode, dev) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        if (!name || name === '.' || name === '..') {
          throw new FS.ErrnoError(22);
        }
        var err = FS.mayCreate(parent, name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.mknod) {
          throw new FS.ErrnoError(1);
        }
        return parent.node_ops.mknod(parent, name, mode, dev);
      },create:function (path, mode) {
        mode = mode !== undefined ? mode : 438 /* 0666 */;
        mode &= 4095;
        mode |= 32768;
        return FS.mknod(path, mode, 0);
      },mkdir:function (path, mode) {
        mode = mode !== undefined ? mode : 511 /* 0777 */;
        mode &= 511 | 512;
        mode |= 16384;
        return FS.mknod(path, mode, 0);
      },mkdirTree:function (path, mode) {
        var dirs = path.split('/');
        var d = '';
        for (var i = 0; i < dirs.length; ++i) {
          if (!dirs[i]) continue;
          d += '/' + dirs[i];
          try {
            FS.mkdir(d, mode);
          } catch(e) {
            if (e.errno != 17) throw e;
          }
        }
      },mkdev:function (path, mode, dev) {
        if (typeof(dev) === 'undefined') {
          dev = mode;
          mode = 438 /* 0666 */;
        }
        mode |= 8192;
        return FS.mknod(path, mode, dev);
      },symlink:function (oldpath, newpath) {
        if (!PATH.resolve(oldpath)) {
          throw new FS.ErrnoError(2);
        }
        var lookup = FS.lookupPath(newpath, { parent: true });
        var parent = lookup.node;
        if (!parent) {
          throw new FS.ErrnoError(2);
        }
        var newname = PATH.basename(newpath);
        var err = FS.mayCreate(parent, newname);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.symlink) {
          throw new FS.ErrnoError(1);
        }
        return parent.node_ops.symlink(parent, newname, oldpath);
      },rename:function (old_path, new_path) {
        var old_dirname = PATH.dirname(old_path);
        var new_dirname = PATH.dirname(new_path);
        var old_name = PATH.basename(old_path);
        var new_name = PATH.basename(new_path);
        // parents must exist
        var lookup, old_dir, new_dir;
        try {
          lookup = FS.lookupPath(old_path, { parent: true });
          old_dir = lookup.node;
          lookup = FS.lookupPath(new_path, { parent: true });
          new_dir = lookup.node;
        } catch (e) {
          throw new FS.ErrnoError(16);
        }
        if (!old_dir || !new_dir) throw new FS.ErrnoError(2);
        // need to be part of the same mount
        if (old_dir.mount !== new_dir.mount) {
          throw new FS.ErrnoError(18);
        }
        // source must exist
        var old_node = FS.lookupNode(old_dir, old_name);
        // old path should not be an ancestor of the new path
        var relative = PATH.relative(old_path, new_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(22);
        }
        // new path should not be an ancestor of the old path
        relative = PATH.relative(new_path, old_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(39);
        }
        // see if the new path already exists
        var new_node;
        try {
          new_node = FS.lookupNode(new_dir, new_name);
        } catch (e) {
          // not fatal
        }
        // early out if nothing needs to change
        if (old_node === new_node) {
          return;
        }
        // we'll need to delete the old entry
        var isdir = FS.isDir(old_node.mode);
        var err = FS.mayDelete(old_dir, old_name, isdir);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        // need delete permissions if we'll be overwriting.
        // need create permissions if new doesn't already exist.
        err = new_node ?
          FS.mayDelete(new_dir, new_name, isdir) :
          FS.mayCreate(new_dir, new_name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!old_dir.node_ops.rename) {
          throw new FS.ErrnoError(1);
        }
        if (FS.isMountpoint(old_node) || (new_node && FS.isMountpoint(new_node))) {
          throw new FS.ErrnoError(16);
        }
        // if we are going to change the parent, check write permissions
        if (new_dir !== old_dir) {
          err = FS.nodePermissions(old_dir, 'w');
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        try {
          if (FS.trackingDelegate['willMovePath']) {
            FS.trackingDelegate['willMovePath'](old_path, new_path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willMovePath']('"+old_path+"', '"+new_path+"') threw an exception: " + e.message);
        }
        // remove the node from the lookup hash
        FS.hashRemoveNode(old_node);
        // do the underlying fs rename
        try {
          old_dir.node_ops.rename(old_node, new_dir, new_name);
        } catch (e) {
          throw e;
        } finally {
          // add the node back to the hash (in case node_ops.rename
          // changed its name)
          FS.hashAddNode(old_node);
        }
        try {
          if (FS.trackingDelegate['onMovePath']) FS.trackingDelegate['onMovePath'](old_path, new_path);
        } catch(e) {
          console.log("FS.trackingDelegate['onMovePath']('"+old_path+"', '"+new_path+"') threw an exception: " + e.message);
        }
      },rmdir:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, true);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.rmdir) {
          throw new FS.ErrnoError(1);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(16);
        }
        try {
          if (FS.trackingDelegate['willDeletePath']) {
            FS.trackingDelegate['willDeletePath'](path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: " + e.message);
        }
        parent.node_ops.rmdir(parent, name);
        FS.destroyNode(node);
        try {
          if (FS.trackingDelegate['onDeletePath']) FS.trackingDelegate['onDeletePath'](path);
        } catch(e) {
          console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: " + e.message);
        }
      },readdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        if (!node.node_ops.readdir) {
          throw new FS.ErrnoError(20);
        }
        return node.node_ops.readdir(node);
      },unlink:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, false);
        if (err) {
          // According to POSIX, we should map EISDIR to EPERM, but
          // we instead do what Linux does (and we must, as we use
          // the musl linux libc).
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.unlink) {
          throw new FS.ErrnoError(1);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(16);
        }
        try {
          if (FS.trackingDelegate['willDeletePath']) {
            FS.trackingDelegate['willDeletePath'](path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: " + e.message);
        }
        parent.node_ops.unlink(parent, name);
        FS.destroyNode(node);
        try {
          if (FS.trackingDelegate['onDeletePath']) FS.trackingDelegate['onDeletePath'](path);
        } catch(e) {
          console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: " + e.message);
        }
      },readlink:function (path) {
        var lookup = FS.lookupPath(path);
        var link = lookup.node;
        if (!link) {
          throw new FS.ErrnoError(2);
        }
        if (!link.node_ops.readlink) {
          throw new FS.ErrnoError(22);
        }
        return PATH.resolve(FS.getPath(link.parent), link.node_ops.readlink(link));
      },stat:function (path, dontFollow) {
        var lookup = FS.lookupPath(path, { follow: !dontFollow });
        var node = lookup.node;
        if (!node) {
          throw new FS.ErrnoError(2);
        }
        if (!node.node_ops.getattr) {
          throw new FS.ErrnoError(1);
        }
        return node.node_ops.getattr(node);
      },lstat:function (path) {
        return FS.stat(path, true);
      },chmod:function (path, mode, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(1);
        }
        node.node_ops.setattr(node, {
          mode: (mode & 4095) | (node.mode & ~4095),
          timestamp: Date.now()
        });
      },lchmod:function (path, mode) {
        FS.chmod(path, mode, true);
      },fchmod:function (fd, mode) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(9);
        }
        FS.chmod(stream.node, mode);
      },chown:function (path, uid, gid, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(1);
        }
        node.node_ops.setattr(node, {
          timestamp: Date.now()
          // we ignore the uid / gid for now
        });
      },lchown:function (path, uid, gid) {
        FS.chown(path, uid, gid, true);
      },fchown:function (fd, uid, gid) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(9);
        }
        FS.chown(stream.node, uid, gid);
      },truncate:function (path, len) {
        if (len < 0) {
          throw new FS.ErrnoError(22);
        }
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: true });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(1);
        }
        if (FS.isDir(node.mode)) {
          throw new FS.ErrnoError(21);
        }
        if (!FS.isFile(node.mode)) {
          throw new FS.ErrnoError(22);
        }
        var err = FS.nodePermissions(node, 'w');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        node.node_ops.setattr(node, {
          size: len,
          timestamp: Date.now()
        });
      },ftruncate:function (fd, len) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(9);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(22);
        }
        FS.truncate(stream.node, len);
      },utime:function (path, atime, mtime) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        node.node_ops.setattr(node, {
          timestamp: Math.max(atime, mtime)
        });
      },open:function (path, flags, mode, fd_start, fd_end) {
        if (path === "") {
          throw new FS.ErrnoError(2);
        }
        flags = typeof flags === 'string' ? FS.modeStringToFlags(flags) : flags;
        mode = typeof mode === 'undefined' ? 438 /* 0666 */ : mode;
        if ((flags & 64)) {
          mode = (mode & 4095) | 32768;
        } else {
          mode = 0;
        }
        var node;
        if (typeof path === 'object') {
          node = path;
        } else {
          path = PATH.normalize(path);
          try {
            var lookup = FS.lookupPath(path, {
              follow: !(flags & 131072)
            });
            node = lookup.node;
          } catch (e) {
            // ignore
          }
        }
        // perhaps we need to create the node
        var created = false;
        if ((flags & 64)) {
          if (node) {
            // if O_CREAT and O_EXCL are set, error out if the node already exists
            if ((flags & 128)) {
              throw new FS.ErrnoError(17);
            }
          } else {
            // node doesn't exist, try to create it
            node = FS.mknod(path, mode, 0);
            created = true;
          }
        }
        if (!node) {
          throw new FS.ErrnoError(2);
        }
        // can't truncate a device
        if (FS.isChrdev(node.mode)) {
          flags &= ~512;
        }
        // if asked only for a directory, then this must be one
        if ((flags & 65536) && !FS.isDir(node.mode)) {
          throw new FS.ErrnoError(20);
        }
        // check permissions, if this is not a file we just created now (it is ok to
        // create and write to a file with read-only permissions; it is read-only
        // for later use)
        if (!created) {
          var err = FS.mayOpen(node, flags);
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        // do truncation if necessary
        if ((flags & 512)) {
          FS.truncate(node, 0);
        }
        // we've already handled these, don't pass down to the underlying vfs
        flags &= ~(128 | 512);
  
        // register the stream with the filesystem
        var stream = FS.createStream({
          node: node,
          path: FS.getPath(node),  // we want the absolute path to the node
          flags: flags,
          seekable: true,
          position: 0,
          stream_ops: node.stream_ops,
          // used by the file family libc calls (fopen, fwrite, ferror, etc.)
          ungotten: [],
          error: false
        }, fd_start, fd_end);
        // call the new stream's open function
        if (stream.stream_ops.open) {
          stream.stream_ops.open(stream);
        }
        if (Module['logReadFiles'] && !(flags & 1)) {
          if (!FS.readFiles) FS.readFiles = {};
          if (!(path in FS.readFiles)) {
            FS.readFiles[path] = 1;
            console.log("FS.trackingDelegate error on read file: " + path);
          }
        }
        try {
          if (FS.trackingDelegate['onOpenFile']) {
            var trackingFlags = 0;
            if ((flags & 2097155) !== 1) {
              trackingFlags |= FS.tracking.openFlags.READ;
            }
            if ((flags & 2097155) !== 0) {
              trackingFlags |= FS.tracking.openFlags.WRITE;
            }
            FS.trackingDelegate['onOpenFile'](path, trackingFlags);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['onOpenFile']('"+path+"', flags) threw an exception: " + e.message);
        }
        return stream;
      },close:function (stream) {
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(9);
        }
        if (stream.getdents) stream.getdents = null; // free readdir state
        try {
          if (stream.stream_ops.close) {
            stream.stream_ops.close(stream);
          }
        } catch (e) {
          throw e;
        } finally {
          FS.closeStream(stream.fd);
        }
        stream.fd = null;
      },isClosed:function (stream) {
        return stream.fd === null;
      },llseek:function (stream, offset, whence) {
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(9);
        }
        if (!stream.seekable || !stream.stream_ops.llseek) {
          throw new FS.ErrnoError(29);
        }
        if (whence != 0 /* SEEK_SET */ && whence != 1 /* SEEK_CUR */ && whence != 2 /* SEEK_END */) {
          throw new FS.ErrnoError(22);
        }
        stream.position = stream.stream_ops.llseek(stream, offset, whence);
        stream.ungotten = [];
        return stream.position;
      },read:function (stream, buffer, offset, length, position) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(22);
        }
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(9);
        }
        if ((stream.flags & 2097155) === 1) {
          throw new FS.ErrnoError(9);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(21);
        }
        if (!stream.stream_ops.read) {
          throw new FS.ErrnoError(22);
        }
        var seeking = typeof position !== 'undefined';
        if (!seeking) {
          position = stream.position;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(29);
        }
        var bytesRead = stream.stream_ops.read(stream, buffer, offset, length, position);
        if (!seeking) stream.position += bytesRead;
        return bytesRead;
      },write:function (stream, buffer, offset, length, position, canOwn) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(22);
        }
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(9);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(9);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(21);
        }
        if (!stream.stream_ops.write) {
          throw new FS.ErrnoError(22);
        }
        if (stream.flags & 1024) {
          // seek to the end before writing in append mode
          FS.llseek(stream, 0, 2);
        }
        var seeking = typeof position !== 'undefined';
        if (!seeking) {
          position = stream.position;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(29);
        }
        var bytesWritten = stream.stream_ops.write(stream, buffer, offset, length, position, canOwn);
        if (!seeking) stream.position += bytesWritten;
        try {
          if (stream.path && FS.trackingDelegate['onWriteToFile']) FS.trackingDelegate['onWriteToFile'](stream.path);
        } catch(e) {
          console.log("FS.trackingDelegate['onWriteToFile']('"+stream.path+"') threw an exception: " + e.message);
        }
        return bytesWritten;
      },allocate:function (stream, offset, length) {
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(9);
        }
        if (offset < 0 || length <= 0) {
          throw new FS.ErrnoError(22);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(9);
        }
        if (!FS.isFile(stream.node.mode) && !FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(19);
        }
        if (!stream.stream_ops.allocate) {
          throw new FS.ErrnoError(95);
        }
        stream.stream_ops.allocate(stream, offset, length);
      },mmap:function (stream, buffer, offset, length, position, prot, flags) {
        // TODO if PROT is PROT_WRITE, make sure we have write access
        if ((stream.flags & 2097155) === 1) {
          throw new FS.ErrnoError(13);
        }
        if (!stream.stream_ops.mmap) {
          throw new FS.ErrnoError(19);
        }
        return stream.stream_ops.mmap(stream, buffer, offset, length, position, prot, flags);
      },msync:function (stream, buffer, offset, length, mmapFlags) {
        if (!stream || !stream.stream_ops.msync) {
          return 0;
        }
        return stream.stream_ops.msync(stream, buffer, offset, length, mmapFlags);
      },munmap:function (stream) {
        return 0;
      },ioctl:function (stream, cmd, arg) {
        if (!stream.stream_ops.ioctl) {
          throw new FS.ErrnoError(25);
        }
        return stream.stream_ops.ioctl(stream, cmd, arg);
      },readFile:function (path, opts) {
        opts = opts || {};
        opts.flags = opts.flags || 'r';
        opts.encoding = opts.encoding || 'binary';
        if (opts.encoding !== 'utf8' && opts.encoding !== 'binary') {
          throw new Error('Invalid encoding type "' + opts.encoding + '"');
        }
        var ret;
        var stream = FS.open(path, opts.flags);
        var stat = FS.stat(path);
        var length = stat.size;
        var buf = new Uint8Array(length);
        FS.read(stream, buf, 0, length, 0);
        if (opts.encoding === 'utf8') {
          ret = UTF8ArrayToString(buf, 0);
        } else if (opts.encoding === 'binary') {
          ret = buf;
        }
        FS.close(stream);
        return ret;
      },writeFile:function (path, data, opts) {
        opts = opts || {};
        opts.flags = opts.flags || 'w';
        var stream = FS.open(path, opts.flags, opts.mode);
        if (typeof data === 'string') {
          var buf = new Uint8Array(lengthBytesUTF8(data)+1);
          var actualNumBytes = stringToUTF8Array(data, buf, 0, buf.length);
          FS.write(stream, buf, 0, actualNumBytes, undefined, opts.canOwn);
        } else if (ArrayBuffer.isView(data)) {
          FS.write(stream, data, 0, data.byteLength, undefined, opts.canOwn);
        } else {
          throw new Error('Unsupported data type');
        }
        FS.close(stream);
      },cwd:function () {
        return FS.currentPath;
      },chdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        if (lookup.node === null) {
          throw new FS.ErrnoError(2);
        }
        if (!FS.isDir(lookup.node.mode)) {
          throw new FS.ErrnoError(20);
        }
        var err = FS.nodePermissions(lookup.node, 'x');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        FS.currentPath = lookup.path;
      },createDefaultDirectories:function () {
        FS.mkdir('/tmp');
        FS.mkdir('/home');
        FS.mkdir('/home/web_user');
      },createDefaultDevices:function () {
        // create /dev
        FS.mkdir('/dev');
        // setup /dev/null
        FS.registerDevice(FS.makedev(1, 3), {
          read: function() { return 0; },
          write: function(stream, buffer, offset, length, pos) { return length; }
        });
        FS.mkdev('/dev/null', FS.makedev(1, 3));
        // setup /dev/tty and /dev/tty1
        // stderr needs to print output using Module['printErr']
        // so we register a second tty just for it.
        TTY.register(FS.makedev(5, 0), TTY.default_tty_ops);
        TTY.register(FS.makedev(6, 0), TTY.default_tty1_ops);
        FS.mkdev('/dev/tty', FS.makedev(5, 0));
        FS.mkdev('/dev/tty1', FS.makedev(6, 0));
        // setup /dev/[u]random
        var random_device;
        if (typeof crypto === 'object' && typeof crypto['getRandomValues'] === 'function') {
          // for modern web browsers
          var randomBuffer = new Uint8Array(1);
          random_device = function() { crypto.getRandomValues(randomBuffer); return randomBuffer[0]; };
        } else
        if (ENVIRONMENT_IS_NODE) {
          // for nodejs with or without crypto support included
          try {
            var crypto_module = require('crypto');
            // nodejs has crypto support
            random_device = function() { return crypto_module['randomBytes'](1)[0]; };
          } catch (e) {
            // nodejs doesn't have crypto support
          }
        } else
        {}
        if (!random_device) {
          // we couldn't find a proper implementation, as Math.random() is not suitable for /dev/random, see emscripten-core/emscripten/pull/7096
          random_device = function() { abort("random_device"); };
        }
        FS.createDevice('/dev', 'random', random_device);
        FS.createDevice('/dev', 'urandom', random_device);
        // we're not going to emulate the actual shm device,
        // just create the tmp dirs that reside in it commonly
        FS.mkdir('/dev/shm');
        FS.mkdir('/dev/shm/tmp');
      },createSpecialDirectories:function () {
        // create /proc/self/fd which allows /proc/self/fd/6 => readlink gives the name of the stream for fd 6 (see test_unistd_ttyname)
        FS.mkdir('/proc');
        FS.mkdir('/proc/self');
        FS.mkdir('/proc/self/fd');
        FS.mount({
          mount: function() {
            var node = FS.createNode('/proc/self', 'fd', 16384 | 511 /* 0777 */, 73);
            node.node_ops = {
              lookup: function(parent, name) {
                var fd = +name;
                var stream = FS.getStream(fd);
                if (!stream) throw new FS.ErrnoError(9);
                var ret = {
                  parent: null,
                  mount: { mountpoint: 'fake' },
                  node_ops: { readlink: function() { return stream.path } }
                };
                ret.parent = ret; // make it look like a simple root node
                return ret;
              }
            };
            return node;
          }
        }, {}, '/proc/self/fd');
      },createStandardStreams:function () {
        // TODO deprecate the old functionality of a single
        // input / output callback and that utilizes FS.createDevice
        // and instead require a unique set of stream ops
  
        // by default, we symlink the standard streams to the
        // default tty devices. however, if the standard streams
        // have been overwritten we create a unique device for
        // them instead.
        if (Module['stdin']) {
          FS.createDevice('/dev', 'stdin', Module['stdin']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdin');
        }
        if (Module['stdout']) {
          FS.createDevice('/dev', 'stdout', null, Module['stdout']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdout');
        }
        if (Module['stderr']) {
          FS.createDevice('/dev', 'stderr', null, Module['stderr']);
        } else {
          FS.symlink('/dev/tty1', '/dev/stderr');
        }
  
        // open default streams for the stdin, stdout and stderr devices
        var stdin = FS.open('/dev/stdin', 'r');
        var stdout = FS.open('/dev/stdout', 'w');
        var stderr = FS.open('/dev/stderr', 'w');
      },ensureErrnoError:function () {
        if (FS.ErrnoError) return;
        FS.ErrnoError = function ErrnoError(errno, node) {
          this.node = node;
          this.setErrno = function(errno) {
            this.errno = errno;
          };
          this.setErrno(errno);
          this.message = 'FS error';
          // Node.js compatibility: assigning on this.stack fails on Node 4 (but fixed on Node 8)
          if (this.stack) Object.defineProperty(this, "stack", { value: (new Error).stack, writable: true });
        };
        FS.ErrnoError.prototype = new Error();
        FS.ErrnoError.prototype.constructor = FS.ErrnoError;
        // Some errors may happen quite a bit, to avoid overhead we reuse them (and suffer a lack of stack info)
        [2].forEach(function(code) {
          FS.genericErrors[code] = new FS.ErrnoError(code);
          FS.genericErrors[code].stack = '<generic error, no stack>';
        });
      },staticInit:function () {
        FS.ensureErrnoError();
  
        FS.nameTable = new Array(4096);
  
        FS.mount(MEMFS, {}, '/');
  
        FS.createDefaultDirectories();
        FS.createDefaultDevices();
        FS.createSpecialDirectories();
  
        FS.filesystems = {
          'MEMFS': MEMFS,
          'IDBFS': IDBFS,
          'NODEFS': NODEFS,
          'WORKERFS': WORKERFS,
        };
      },init:function (input, output, error) {
        FS.init.initialized = true;
  
        FS.ensureErrnoError();
  
        // Allow Module.stdin etc. to provide defaults, if none explicitly passed to us here
        Module['stdin'] = input || Module['stdin'];
        Module['stdout'] = output || Module['stdout'];
        Module['stderr'] = error || Module['stderr'];
  
        FS.createStandardStreams();
      },quit:function () {
        FS.init.initialized = false;
        // force-flush all streams, so we get musl std streams printed out
        var fflush = Module['_fflush'];
        if (fflush) fflush(0);
        // close all of our streams
        for (var i = 0; i < FS.streams.length; i++) {
          var stream = FS.streams[i];
          if (!stream) {
            continue;
          }
          FS.close(stream);
        }
      },getMode:function (canRead, canWrite) {
        var mode = 0;
        if (canRead) mode |= 292 | 73;
        if (canWrite) mode |= 146;
        return mode;
      },joinPath:function (parts, forceRelative) {
        var path = PATH.join.apply(null, parts);
        if (forceRelative && path[0] == '/') path = path.substr(1);
        return path;
      },absolutePath:function (relative, base) {
        return PATH.resolve(base, relative);
      },standardizePath:function (path) {
        return PATH.normalize(path);
      },findObject:function (path, dontResolveLastLink) {
        var ret = FS.analyzePath(path, dontResolveLastLink);
        if (ret.exists) {
          return ret.object;
        } else {
          ___setErrNo(ret.error);
          return null;
        }
      },analyzePath:function (path, dontResolveLastLink) {
        // operate from within the context of the symlink's target
        try {
          var lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          path = lookup.path;
        } catch (e) {
        }
        var ret = {
          isRoot: false, exists: false, error: 0, name: null, path: null, object: null,
          parentExists: false, parentPath: null, parentObject: null
        };
        try {
          var lookup = FS.lookupPath(path, { parent: true });
          ret.parentExists = true;
          ret.parentPath = lookup.path;
          ret.parentObject = lookup.node;
          ret.name = PATH.basename(path);
          lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          ret.exists = true;
          ret.path = lookup.path;
          ret.object = lookup.node;
          ret.name = lookup.node.name;
          ret.isRoot = lookup.path === '/';
        } catch (e) {
          ret.error = e.errno;
        };
        return ret;
      },createFolder:function (parent, name, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.mkdir(path, mode);
      },createPath:function (parent, path, canRead, canWrite) {
        parent = typeof parent === 'string' ? parent : FS.getPath(parent);
        var parts = path.split('/').reverse();
        while (parts.length) {
          var part = parts.pop();
          if (!part) continue;
          var current = PATH.join2(parent, part);
          try {
            FS.mkdir(current);
          } catch (e) {
            // ignore EEXIST
          }
          parent = current;
        }
        return current;
      },createFile:function (parent, name, properties, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.create(path, mode);
      },createDataFile:function (parent, name, data, canRead, canWrite, canOwn) {
        var path = name ? PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name) : parent;
        var mode = FS.getMode(canRead, canWrite);
        var node = FS.create(path, mode);
        if (data) {
          if (typeof data === 'string') {
            var arr = new Array(data.length);
            for (var i = 0, len = data.length; i < len; ++i) arr[i] = data.charCodeAt(i);
            data = arr;
          }
          // make sure we can write to the file
          FS.chmod(node, mode | 146);
          var stream = FS.open(node, 'w');
          FS.write(stream, data, 0, data.length, 0, canOwn);
          FS.close(stream);
          FS.chmod(node, mode);
        }
        return node;
      },createDevice:function (parent, name, input, output) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(!!input, !!output);
        if (!FS.createDevice.major) FS.createDevice.major = 64;
        var dev = FS.makedev(FS.createDevice.major++, 0);
        // Create a fake device that a set of stream ops to emulate
        // the old behavior.
        FS.registerDevice(dev, {
          open: function(stream) {
            stream.seekable = false;
          },
          close: function(stream) {
            // flush any pending line data
            if (output && output.buffer && output.buffer.length) {
              output(10);
            }
          },
          read: function(stream, buffer, offset, length, pos /* ignored */) {
            var bytesRead = 0;
            for (var i = 0; i < length; i++) {
              var result;
              try {
                result = input();
              } catch (e) {
                throw new FS.ErrnoError(5);
              }
              if (result === undefined && bytesRead === 0) {
                throw new FS.ErrnoError(11);
              }
              if (result === null || result === undefined) break;
              bytesRead++;
              buffer[offset+i] = result;
            }
            if (bytesRead) {
              stream.node.timestamp = Date.now();
            }
            return bytesRead;
          },
          write: function(stream, buffer, offset, length, pos) {
            for (var i = 0; i < length; i++) {
              try {
                output(buffer[offset+i]);
              } catch (e) {
                throw new FS.ErrnoError(5);
              }
            }
            if (length) {
              stream.node.timestamp = Date.now();
            }
            return i;
          }
        });
        return FS.mkdev(path, mode, dev);
      },createLink:function (parent, name, target, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        return FS.symlink(target, path);
      },forceLoadFile:function (obj) {
        if (obj.isDevice || obj.isFolder || obj.link || obj.contents) return true;
        var success = true;
        if (typeof XMLHttpRequest !== 'undefined') {
          throw new Error("Lazy loading should have been performed (contents set) in createLazyFile, but it was not. Lazy loading only works in web workers. Use --embed-file or --preload-file in emcc on the main thread.");
        } else if (Module['read']) {
          // Command-line.
          try {
            // WARNING: Can't read binary files in V8's d8 or tracemonkey's js, as
            //          read() will try to parse UTF8.
            obj.contents = intArrayFromString(Module['read'](obj.url), true);
            obj.usedBytes = obj.contents.length;
          } catch (e) {
            success = false;
          }
        } else {
          throw new Error('Cannot load without read() or XMLHttpRequest.');
        }
        if (!success) ___setErrNo(5);
        return success;
      },createLazyFile:function (parent, name, url, canRead, canWrite) {
        // Lazy chunked Uint8Array (implements get and length from Uint8Array). Actual getting is abstracted away for eventual reuse.
        function LazyUint8Array() {
          this.lengthKnown = false;
          this.chunks = []; // Loaded chunks. Index is the chunk number
        }
        LazyUint8Array.prototype.get = function LazyUint8Array_get(idx) {
          if (idx > this.length-1 || idx < 0) {
            return undefined;
          }
          var chunkOffset = idx % this.chunkSize;
          var chunkNum = (idx / this.chunkSize)|0;
          return this.getter(chunkNum)[chunkOffset];
        }
        LazyUint8Array.prototype.setDataGetter = function LazyUint8Array_setDataGetter(getter) {
          this.getter = getter;
        }
        LazyUint8Array.prototype.cacheLength = function LazyUint8Array_cacheLength() {
          // Find length
          var xhr = new XMLHttpRequest();
          xhr.open('HEAD', url, false);
          xhr.send(null);
          if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
          var datalength = Number(xhr.getResponseHeader("Content-length"));
          var header;
          var hasByteServing = (header = xhr.getResponseHeader("Accept-Ranges")) && header === "bytes";
          var usesGzip = (header = xhr.getResponseHeader("Content-Encoding")) && header === "gzip";
  
          var chunkSize = 1024*1024; // Chunk size in bytes
  
          if (!hasByteServing) chunkSize = datalength;
  
          // Function to get a range from the remote URL.
          var doXHR = (function(from, to) {
            if (from > to) throw new Error("invalid range (" + from + ", " + to + ") or no bytes requested!");
            if (to > datalength-1) throw new Error("only " + datalength + " bytes available! programmer error!");
  
            // TODO: Use mozResponseArrayBuffer, responseStream, etc. if available.
            var xhr = new XMLHttpRequest();
            xhr.open('GET', url, false);
            if (datalength !== chunkSize) xhr.setRequestHeader("Range", "bytes=" + from + "-" + to);
  
            // Some hints to the browser that we want binary data.
            if (typeof Uint8Array != 'undefined') xhr.responseType = 'arraybuffer';
            if (xhr.overrideMimeType) {
              xhr.overrideMimeType('text/plain; charset=x-user-defined');
            }
  
            xhr.send(null);
            if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
            if (xhr.response !== undefined) {
              return new Uint8Array(xhr.response || []);
            } else {
              return intArrayFromString(xhr.responseText || '', true);
            }
          });
          var lazyArray = this;
          lazyArray.setDataGetter(function(chunkNum) {
            var start = chunkNum * chunkSize;
            var end = (chunkNum+1) * chunkSize - 1; // including this byte
            end = Math.min(end, datalength-1); // if datalength-1 is selected, this is the last block
            if (typeof(lazyArray.chunks[chunkNum]) === "undefined") {
              lazyArray.chunks[chunkNum] = doXHR(start, end);
            }
            if (typeof(lazyArray.chunks[chunkNum]) === "undefined") throw new Error("doXHR failed!");
            return lazyArray.chunks[chunkNum];
          });
  
          if (usesGzip || !datalength) {
            // if the server uses gzip or doesn't supply the length, we have to download the whole file to get the (uncompressed) length
            chunkSize = datalength = 1; // this will force getter(0)/doXHR do download the whole file
            datalength = this.getter(0).length;
            chunkSize = datalength;
            console.log("LazyFiles on gzip forces download of the whole file when length is accessed");
          }
  
          this._length = datalength;
          this._chunkSize = chunkSize;
          this.lengthKnown = true;
        }
        if (typeof XMLHttpRequest !== 'undefined') {
          if (!ENVIRONMENT_IS_WORKER) throw 'Cannot do synchronous binary XHRs outside webworkers in modern browsers. Use --embed-file or --preload-file in emcc';
          var lazyArray = new LazyUint8Array();
          Object.defineProperties(lazyArray, {
            length: {
              get: function() {
                if(!this.lengthKnown) {
                  this.cacheLength();
                }
                return this._length;
              }
            },
            chunkSize: {
              get: function() {
                if(!this.lengthKnown) {
                  this.cacheLength();
                }
                return this._chunkSize;
              }
            }
          });
  
          var properties = { isDevice: false, contents: lazyArray };
        } else {
          var properties = { isDevice: false, url: url };
        }
  
        var node = FS.createFile(parent, name, properties, canRead, canWrite);
        // This is a total hack, but I want to get this lazy file code out of the
        // core of MEMFS. If we want to keep this lazy file concept I feel it should
        // be its own thin LAZYFS proxying calls to MEMFS.
        if (properties.contents) {
          node.contents = properties.contents;
        } else if (properties.url) {
          node.contents = null;
          node.url = properties.url;
        }
        // Add a function that defers querying the file size until it is asked the first time.
        Object.defineProperties(node, {
          usedBytes: {
            get: function() { return this.contents.length; }
          }
        });
        // override each stream op with one that tries to force load the lazy file first
        var stream_ops = {};
        var keys = Object.keys(node.stream_ops);
        keys.forEach(function(key) {
          var fn = node.stream_ops[key];
          stream_ops[key] = function forceLoadLazyFile() {
            if (!FS.forceLoadFile(node)) {
              throw new FS.ErrnoError(5);
            }
            return fn.apply(null, arguments);
          };
        });
        // use a custom read function
        stream_ops.read = function stream_ops_read(stream, buffer, offset, length, position) {
          if (!FS.forceLoadFile(node)) {
            throw new FS.ErrnoError(5);
          }
          var contents = stream.node.contents;
          if (position >= contents.length)
            return 0;
          var size = Math.min(contents.length - position, length);
          if (contents.slice) { // normal array
            for (var i = 0; i < size; i++) {
              buffer[offset + i] = contents[position + i];
            }
          } else {
            for (var i = 0; i < size; i++) { // LazyUint8Array from sync binary XHR
              buffer[offset + i] = contents.get(position + i);
            }
          }
          return size;
        };
        node.stream_ops = stream_ops;
        return node;
      },createPreloadedFile:function (parent, name, url, canRead, canWrite, onload, onerror, dontCreateFile, canOwn, preFinish) {
        Browser.init(); // XXX perhaps this method should move onto Browser?
        // TODO we should allow people to just pass in a complete filename instead
        // of parent and name being that we just join them anyways
        var fullname = name ? PATH.resolve(PATH.join2(parent, name)) : parent;
        var dep = getUniqueRunDependency('cp ' + fullname); // might have several active requests for the same fullname
        function processData(byteArray) {
          function finish(byteArray) {
            if (preFinish) preFinish();
            if (!dontCreateFile) {
              FS.createDataFile(parent, name, byteArray, canRead, canWrite, canOwn);
            }
            if (onload) onload();
            removeRunDependency(dep);
          }
          var handled = false;
          Module['preloadPlugins'].forEach(function(plugin) {
            if (handled) return;
            if (plugin['canHandle'](fullname)) {
              plugin['handle'](byteArray, fullname, finish, function() {
                if (onerror) onerror();
                removeRunDependency(dep);
              });
              handled = true;
            }
          });
          if (!handled) finish(byteArray);
        }
        addRunDependency(dep);
        if (typeof url == 'string') {
          Browser.asyncLoad(url, function(byteArray) {
            processData(byteArray);
          }, onerror);
        } else {
          processData(url);
        }
      },indexedDB:function () {
        return window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;
      },DB_NAME:function () {
        return 'EM_FS_' + window.location.pathname;
      },DB_VERSION:20,DB_STORE_NAME:"FILE_DATA",saveFilesToDB:function (paths, onload, onerror) {
        onload = onload || function(){};
        onerror = onerror || function(){};
        var indexedDB = FS.indexedDB();
        try {
          var openRequest = indexedDB.open(FS.DB_NAME(), FS.DB_VERSION);
        } catch (e) {
          return onerror(e);
        }
        openRequest.onupgradeneeded = function openRequest_onupgradeneeded() {
          console.log('creating db');
          var db = openRequest.result;
          db.createObjectStore(FS.DB_STORE_NAME);
        };
        openRequest.onsuccess = function openRequest_onsuccess() {
          var db = openRequest.result;
          var transaction = db.transaction([FS.DB_STORE_NAME], 'readwrite');
          var files = transaction.objectStore(FS.DB_STORE_NAME);
          var ok = 0, fail = 0, total = paths.length;
          function finish() {
            if (fail == 0) onload(); else onerror();
          }
          paths.forEach(function(path) {
            var putRequest = files.put(FS.analyzePath(path).object.contents, path);
            putRequest.onsuccess = function putRequest_onsuccess() { ok++; if (ok + fail == total) finish() };
            putRequest.onerror = function putRequest_onerror() { fail++; if (ok + fail == total) finish() };
          });
          transaction.onerror = onerror;
        };
        openRequest.onerror = onerror;
      },loadFilesFromDB:function (paths, onload, onerror) {
        onload = onload || function(){};
        onerror = onerror || function(){};
        var indexedDB = FS.indexedDB();
        try {
          var openRequest = indexedDB.open(FS.DB_NAME(), FS.DB_VERSION);
        } catch (e) {
          return onerror(e);
        }
        openRequest.onupgradeneeded = onerror; // no database to load from
        openRequest.onsuccess = function openRequest_onsuccess() {
          var db = openRequest.result;
          try {
            var transaction = db.transaction([FS.DB_STORE_NAME], 'readonly');
          } catch(e) {
            onerror(e);
            return;
          }
          var files = transaction.objectStore(FS.DB_STORE_NAME);
          var ok = 0, fail = 0, total = paths.length;
          function finish() {
            if (fail == 0) onload(); else onerror();
          }
          paths.forEach(function(path) {
            var getRequest = files.get(path);
            getRequest.onsuccess = function getRequest_onsuccess() {
              if (FS.analyzePath(path).exists) {
                FS.unlink(path);
              }
              FS.createDataFile(PATH.dirname(path), PATH.basename(path), getRequest.result, true, true, true);
              ok++;
              if (ok + fail == total) finish();
            };
            getRequest.onerror = function getRequest_onerror() { fail++; if (ok + fail == total) finish() };
          });
          transaction.onerror = onerror;
        };
        openRequest.onerror = onerror;
      }};
  
  var ERRNO_CODES={EPERM:1,ENOENT:2,ESRCH:3,EINTR:4,EIO:5,ENXIO:6,E2BIG:7,ENOEXEC:8,EBADF:9,ECHILD:10,EAGAIN:11,EWOULDBLOCK:11,ENOMEM:12,EACCES:13,EFAULT:14,ENOTBLK:15,EBUSY:16,EEXIST:17,EXDEV:18,ENODEV:19,ENOTDIR:20,EISDIR:21,EINVAL:22,ENFILE:23,EMFILE:24,ENOTTY:25,ETXTBSY:26,EFBIG:27,ENOSPC:28,ESPIPE:29,EROFS:30,EMLINK:31,EPIPE:32,EDOM:33,ERANGE:34,ENOMSG:42,EIDRM:43,ECHRNG:44,EL2NSYNC:45,EL3HLT:46,EL3RST:47,ELNRNG:48,EUNATCH:49,ENOCSI:50,EL2HLT:51,EDEADLK:35,ENOLCK:37,EBADE:52,EBADR:53,EXFULL:54,ENOANO:55,EBADRQC:56,EBADSLT:57,EDEADLOCK:35,EBFONT:59,ENOSTR:60,ENODATA:61,ETIME:62,ENOSR:63,ENONET:64,ENOPKG:65,EREMOTE:66,ENOLINK:67,EADV:68,ESRMNT:69,ECOMM:70,EPROTO:71,EMULTIHOP:72,EDOTDOT:73,EBADMSG:74,ENOTUNIQ:76,EBADFD:77,EREMCHG:78,ELIBACC:79,ELIBBAD:80,ELIBSCN:81,ELIBMAX:82,ELIBEXEC:83,ENOSYS:38,ENOTEMPTY:39,ENAMETOOLONG:36,ELOOP:40,EOPNOTSUPP:95,EPFNOSUPPORT:96,ECONNRESET:104,ENOBUFS:105,EAFNOSUPPORT:97,EPROTOTYPE:91,ENOTSOCK:88,ENOPROTOOPT:92,ESHUTDOWN:108,ECONNREFUSED:111,EADDRINUSE:98,ECONNABORTED:103,ENETUNREACH:101,ENETDOWN:100,ETIMEDOUT:110,EHOSTDOWN:112,EHOSTUNREACH:113,EINPROGRESS:115,EALREADY:114,EDESTADDRREQ:89,EMSGSIZE:90,EPROTONOSUPPORT:93,ESOCKTNOSUPPORT:94,EADDRNOTAVAIL:99,ENETRESET:102,EISCONN:106,ENOTCONN:107,ETOOMANYREFS:109,EUSERS:87,EDQUOT:122,ESTALE:116,ENOTSUP:95,ENOMEDIUM:123,EILSEQ:84,EOVERFLOW:75,ECANCELED:125,ENOTRECOVERABLE:131,EOWNERDEAD:130,ESTRPIPE:86};var SYSCALLS={DEFAULT_POLLMASK:5,mappings:{},umask:511,calculateAt:function (dirfd, path) {
        if (path[0] !== '/') {
          // relative path
          var dir;
          if (dirfd === -100) {
            dir = FS.cwd();
          } else {
            var dirstream = FS.getStream(dirfd);
            if (!dirstream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
            dir = dirstream.path;
          }
          path = PATH.join2(dir, path);
        }
        return path;
      },doStat:function (func, path, buf) {
        try {
          var stat = func(path);
        } catch (e) {
          if (e && e.node && PATH.normalize(path) !== PATH.normalize(FS.getPath(e.node))) {
            // an error occurred while trying to look up the path; we should just report ENOTDIR
            return -ERRNO_CODES.ENOTDIR;
          }
          throw e;
        }
        HEAP32[((buf)>>2)]=stat.dev;
        HEAP32[(((buf)+(4))>>2)]=0;
        HEAP32[(((buf)+(8))>>2)]=stat.ino;
        HEAP32[(((buf)+(12))>>2)]=stat.mode;
        HEAP32[(((buf)+(16))>>2)]=stat.nlink;
        HEAP32[(((buf)+(20))>>2)]=stat.uid;
        HEAP32[(((buf)+(24))>>2)]=stat.gid;
        HEAP32[(((buf)+(28))>>2)]=stat.rdev;
        HEAP32[(((buf)+(32))>>2)]=0;
        HEAP32[(((buf)+(36))>>2)]=stat.size;
        HEAP32[(((buf)+(40))>>2)]=4096;
        HEAP32[(((buf)+(44))>>2)]=stat.blocks;
        HEAP32[(((buf)+(48))>>2)]=(stat.atime.getTime() / 1000)|0;
        HEAP32[(((buf)+(52))>>2)]=0;
        HEAP32[(((buf)+(56))>>2)]=(stat.mtime.getTime() / 1000)|0;
        HEAP32[(((buf)+(60))>>2)]=0;
        HEAP32[(((buf)+(64))>>2)]=(stat.ctime.getTime() / 1000)|0;
        HEAP32[(((buf)+(68))>>2)]=0;
        HEAP32[(((buf)+(72))>>2)]=stat.ino;
        return 0;
      },doMsync:function (addr, stream, len, flags) {
        var buffer = new Uint8Array(HEAPU8.subarray(addr, addr + len));
        FS.msync(stream, buffer, 0, len, flags);
      },doMkdir:function (path, mode) {
        // remove a trailing slash, if one - /a/b/ has basename of '', but
        // we want to create b in the context of this function
        path = PATH.normalize(path);
        if (path[path.length-1] === '/') path = path.substr(0, path.length-1);
        FS.mkdir(path, mode, 0);
        return 0;
      },doMknod:function (path, mode, dev) {
        // we don't want this in the JS API as it uses mknod to create all nodes.
        switch (mode & 61440) {
          case 32768:
          case 8192:
          case 24576:
          case 4096:
          case 49152:
            break;
          default: return -ERRNO_CODES.EINVAL;
        }
        FS.mknod(path, mode, dev);
        return 0;
      },doReadlink:function (path, buf, bufsize) {
        if (bufsize <= 0) return -ERRNO_CODES.EINVAL;
        var ret = FS.readlink(path);
  
        var len = Math.min(bufsize, lengthBytesUTF8(ret));
        var endChar = HEAP8[buf+len];
        stringToUTF8(ret, buf, bufsize+1);
        // readlink is one of the rare functions that write out a C string, but does never append a null to the output buffer(!)
        // stringToUTF8() always appends a null byte, so restore the character under the null byte after the write.
        HEAP8[buf+len] = endChar;
  
        return len;
      },doAccess:function (path, amode) {
        if (amode & ~7) {
          // need a valid mode
          return -ERRNO_CODES.EINVAL;
        }
        var node;
        var lookup = FS.lookupPath(path, { follow: true });
        node = lookup.node;
        var perms = '';
        if (amode & 4) perms += 'r';
        if (amode & 2) perms += 'w';
        if (amode & 1) perms += 'x';
        if (perms /* otherwise, they've just passed F_OK */ && FS.nodePermissions(node, perms)) {
          return -ERRNO_CODES.EACCES;
        }
        return 0;
      },doDup:function (path, flags, suggestFD) {
        var suggest = FS.getStream(suggestFD);
        if (suggest) FS.close(suggest);
        return FS.open(path, flags, 0, suggestFD, suggestFD).fd;
      },doReadv:function (stream, iov, iovcnt, offset) {
        var ret = 0;
        for (var i = 0; i < iovcnt; i++) {
          var ptr = HEAP32[(((iov)+(i*8))>>2)];
          var len = HEAP32[(((iov)+(i*8 + 4))>>2)];
          var curr = FS.read(stream, HEAP8,ptr, len, offset);
          if (curr < 0) return -1;
          ret += curr;
          if (curr < len) break; // nothing more to read
        }
        return ret;
      },doWritev:function (stream, iov, iovcnt, offset) {
        var ret = 0;
        for (var i = 0; i < iovcnt; i++) {
          var ptr = HEAP32[(((iov)+(i*8))>>2)];
          var len = HEAP32[(((iov)+(i*8 + 4))>>2)];
          var curr = FS.write(stream, HEAP8,ptr, len, offset);
          if (curr < 0) return -1;
          ret += curr;
        }
        return ret;
      },varargs:0,get:function (varargs) {
        SYSCALLS.varargs += 4;
        var ret = HEAP32[(((SYSCALLS.varargs)-(4))>>2)];
        return ret;
      },getStr:function () {
        var ret = UTF8ToString(SYSCALLS.get());
        return ret;
      },getStreamFromFD:function () {
        var stream = FS.getStream(SYSCALLS.get());
        if (!stream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        return stream;
      },getSocketFromFD:function () {
        var socket = SOCKFS.getSocket(SYSCALLS.get());
        if (!socket) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        return socket;
      },getSocketAddress:function (allowNull) {
        var addrp = SYSCALLS.get(), addrlen = SYSCALLS.get();
        if (allowNull && addrp === 0) return null;
        var info = __read_sockaddr(addrp, addrlen);
        if (info.errno) throw new FS.ErrnoError(info.errno);
        info.addr = DNS.lookup_addr(info.addr) || info.addr;
        return info;
      },get64:function () {
        var low = SYSCALLS.get(), high = SYSCALLS.get();
        return low;
      },getZero:function () {
        SYSCALLS.get();
      }};function ___syscall10(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // unlink
      var path = SYSCALLS.getStr();
      FS.unlink(path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall118(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fsync
      var stream = SYSCALLS.getStreamFromFD();
      return 0; // we can't do anything synchronously; the in-memory FS is already synced to
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall140(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // llseek
      var stream = SYSCALLS.getStreamFromFD(), offset_high = SYSCALLS.get(), offset_low = SYSCALLS.get(), result = SYSCALLS.get(), whence = SYSCALLS.get();
      // NOTE: offset_high is unused - Emscripten's off_t is 32-bit
      var offset = offset_low;
      FS.llseek(stream, offset, whence);
      HEAP32[((result)>>2)]=stream.position;
      if (stream.getdents && offset === 0 && whence === 0) stream.getdents = null; // reset readdir state
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall15(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // chmod
      var path = SYSCALLS.getStr(), mode = SYSCALLS.get();
      FS.chmod(path, mode);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall183(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // getcwd
      var buf = SYSCALLS.get(), size = SYSCALLS.get();
      if (size === 0) return -ERRNO_CODES.EINVAL;
      var cwd = FS.cwd();
      var cwdLengthInBytes = lengthBytesUTF8(cwd);
      if (size < cwdLengthInBytes + 1) return -ERRNO_CODES.ERANGE;
      stringToUTF8(cwd, buf, size);
      return buf;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall192(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // mmap2
      var addr = SYSCALLS.get(), len = SYSCALLS.get(), prot = SYSCALLS.get(), flags = SYSCALLS.get(), fd = SYSCALLS.get(), off = SYSCALLS.get()
      off <<= 12; // undo pgoffset
      var ptr;
      var allocated = false;
      if (fd === -1) {
        ptr = _memalign(PAGE_SIZE, len);
        if (!ptr) return -ERRNO_CODES.ENOMEM;
        _memset(ptr, 0, len);
        allocated = true;
      } else {
        var info = FS.getStream(fd);
        if (!info) return -ERRNO_CODES.EBADF;
        var res = FS.mmap(info, HEAPU8, addr, len, off, prot, flags);
        ptr = res.ptr;
        allocated = res.allocated;
      }
      SYSCALLS.mappings[ptr] = { malloc: ptr, len: len, allocated: allocated, fd: fd, flags: flags };
      return ptr;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall194(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // ftruncate64
      var fd = SYSCALLS.get(), zero = SYSCALLS.getZero(), length = SYSCALLS.get64();
      FS.ftruncate(fd, length);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall195(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_stat64
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.stat, path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall196(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_lstat64
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.lstat, path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall197(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_fstat64
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.stat, stream.path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  
  var PROCINFO={ppid:1,pid:42,sid:42,pgid:42};function ___syscall20(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // getpid
      return PROCINFO.pid;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  
  function ___syscall202(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // getgid32
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }function ___syscall201(a0,a1
  /*``*/) {
  return ___syscall202(a0,a1);
  }

  function ___syscall207(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fchown32
      var fd = SYSCALLS.get(), owner = SYSCALLS.get(), group = SYSCALLS.get();
      FS.fchown(fd, owner, group);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall212(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // chown32
      var path = SYSCALLS.getStr(), owner = SYSCALLS.get(), group = SYSCALLS.get();
      FS.chown(path, owner, group);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall221(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fcntl64
      var stream = SYSCALLS.getStreamFromFD(), cmd = SYSCALLS.get();
      switch (cmd) {
        case 0: {
          var arg = SYSCALLS.get();
          if (arg < 0) {
            return -ERRNO_CODES.EINVAL;
          }
          var newStream;
          newStream = FS.open(stream.path, stream.flags, 0, arg);
          return newStream.fd;
        }
        case 1:
        case 2:
          return 0;  // FD_CLOEXEC makes no sense for a single process.
        case 3:
          return stream.flags;
        case 4: {
          var arg = SYSCALLS.get();
          stream.flags |= arg;
          return 0;
        }
        case 12:
        /* case 12: Currently in musl F_GETLK64 has same value as F_GETLK, so omitted to avoid duplicate case blocks. If that changes, uncomment this */ {
          
          var arg = SYSCALLS.get();
          var offset = 0;
          // We're always unlocked.
          HEAP16[(((arg)+(offset))>>1)]=2;
          return 0;
        }
        case 13:
        case 14:
        /* case 13: Currently in musl F_SETLK64 has same value as F_SETLK, so omitted to avoid duplicate case blocks. If that changes, uncomment this */
        /* case 14: Currently in musl F_SETLKW64 has same value as F_SETLKW, so omitted to avoid duplicate case blocks. If that changes, uncomment this */
          
          
          return 0; // Pretend that the locking is successful.
        case 16:
        case 8:
          return -ERRNO_CODES.EINVAL; // These are for sockets. We don't have them fully implemented yet.
        case 9:
          // musl trusts getown return values, due to a bug where they must be, as they overlap with errors. just return -1 here, so fnctl() returns that, and we set errno ourselves.
          ___setErrNo(ERRNO_CODES.EINVAL);
          return -1;
        default: {
          return -ERRNO_CODES.EINVAL;
        }
      }
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall3(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // read
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get(), count = SYSCALLS.get();
      return FS.read(stream, HEAP8,buf, count);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall33(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // access
      var path = SYSCALLS.getStr(), amode = SYSCALLS.get();
      return SYSCALLS.doAccess(path, amode);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall39(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // mkdir
      var path = SYSCALLS.getStr(), mode = SYSCALLS.get();
      return SYSCALLS.doMkdir(path, mode);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall4(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // write
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get(), count = SYSCALLS.get();
      return FS.write(stream, HEAP8,buf, count);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall40(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // rmdir
      var path = SYSCALLS.getStr();
      FS.rmdir(path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall5(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // open
      var pathname = SYSCALLS.getStr(), flags = SYSCALLS.get(), mode = SYSCALLS.get() // optional TODO
      var stream = FS.open(pathname, flags, mode);
      return stream.fd;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall6(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // close
      var stream = SYSCALLS.getStreamFromFD();
      FS.close(stream);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall85(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // readlink
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get(), bufsize = SYSCALLS.get();
      return SYSCALLS.doReadlink(path, buf, bufsize);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall91(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // munmap
      var addr = SYSCALLS.get(), len = SYSCALLS.get();
      // TODO: support unmmap'ing parts of allocations
      var info = SYSCALLS.mappings[addr];
      if (!info) return 0;
      if (len === info.len) {
        var stream = FS.getStream(info.fd);
        SYSCALLS.doMsync(addr, stream, len, info.flags)
        FS.munmap(stream);
        SYSCALLS.mappings[addr] = null;
        if (info.allocated) {
          _free(info.malloc);
        }
      }
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall94(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fchmod
      var fd = SYSCALLS.get(), mode = SYSCALLS.get();
      FS.fchmod(fd, mode);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

   

   

   

   

   

  function _emscripten_get_heap_size() {
      return HEAP8.length;
    }

  
  function abortOnCannotGrowMemory(requestedSize) {
      abort('OOM');
    }function _emscripten_resize_heap(requestedSize) {
      abortOnCannotGrowMemory(requestedSize);
    }

  function _getenv(name) {
      // char *getenv(const char *name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/getenv.html
      if (name === 0) return 0;
      name = UTF8ToString(name);
      if (!ENV.hasOwnProperty(name)) return 0;
  
      if (_getenv.ret) _free(_getenv.ret);
      _getenv.ret = allocateUTF8(ENV[name]);
      return _getenv.ret;
    }

  function _gettimeofday(ptr) {
      var now = Date.now();
      HEAP32[((ptr)>>2)]=(now/1000)|0; // seconds
      HEAP32[(((ptr)+(4))>>2)]=((now % 1000)*1000)|0; // microseconds
      return 0;
    }



   

  
  function _llvm_log10_f32(x) {
      return Math.log(x) / Math.LN10; // TODO: Math.log10, when browser support is there
    }function _llvm_log10_f64(a0
  /*``*/) {
  return _llvm_log10_f32(a0);
  }

  function _llvm_trap() {
      abort('trap!');
    }

  
  var ___tm_current=60336;
  
  
  var ___tm_timezone=(stringToUTF8("GMT", 60384, 4), 60384);
  
  function _tzset() {
      // TODO: Use (malleable) environment variables instead of system settings.
      if (_tzset.called) return;
      _tzset.called = true;
  
      // timezone is specified as seconds west of UTC ("The external variable
      // `timezone` shall be set to the difference, in seconds, between
      // Coordinated Universal Time (UTC) and local standard time."), the same
      // as returned by getTimezoneOffset().
      // See http://pubs.opengroup.org/onlinepubs/009695399/functions/tzset.html
      HEAP32[((__get_timezone())>>2)]=(new Date()).getTimezoneOffset() * 60;
  
      var winter = new Date(2000, 0, 1);
      var summer = new Date(2000, 6, 1);
      HEAP32[((__get_daylight())>>2)]=Number(winter.getTimezoneOffset() != summer.getTimezoneOffset());
  
      function extractZone(date) {
        var match = date.toTimeString().match(/\(([A-Za-z ]+)\)$/);
        return match ? match[1] : "GMT";
      };
      var winterName = extractZone(winter);
      var summerName = extractZone(summer);
      var winterNamePtr = allocate(intArrayFromString(winterName), 'i8', ALLOC_NORMAL);
      var summerNamePtr = allocate(intArrayFromString(summerName), 'i8', ALLOC_NORMAL);
      if (summer.getTimezoneOffset() < winter.getTimezoneOffset()) {
        // Northern hemisphere
        HEAP32[((__get_tzname())>>2)]=winterNamePtr;
        HEAP32[(((__get_tzname())+(4))>>2)]=summerNamePtr;
      } else {
        HEAP32[((__get_tzname())>>2)]=summerNamePtr;
        HEAP32[(((__get_tzname())+(4))>>2)]=winterNamePtr;
      }
    }function _localtime_r(time, tmPtr) {
      _tzset();
      var date = new Date(HEAP32[((time)>>2)]*1000);
      HEAP32[((tmPtr)>>2)]=date.getSeconds();
      HEAP32[(((tmPtr)+(4))>>2)]=date.getMinutes();
      HEAP32[(((tmPtr)+(8))>>2)]=date.getHours();
      HEAP32[(((tmPtr)+(12))>>2)]=date.getDate();
      HEAP32[(((tmPtr)+(16))>>2)]=date.getMonth();
      HEAP32[(((tmPtr)+(20))>>2)]=date.getFullYear()-1900;
      HEAP32[(((tmPtr)+(24))>>2)]=date.getDay();
  
      var start = new Date(date.getFullYear(), 0, 1);
      var yday = ((date.getTime() - start.getTime()) / (1000 * 60 * 60 * 24))|0;
      HEAP32[(((tmPtr)+(28))>>2)]=yday;
      HEAP32[(((tmPtr)+(36))>>2)]=-(date.getTimezoneOffset() * 60);
  
      // Attention: DST is in December in South, and some regions don't have DST at all.
      var summerOffset = new Date(2000, 6, 1).getTimezoneOffset();
      var winterOffset = start.getTimezoneOffset();
      var dst = (summerOffset != winterOffset && date.getTimezoneOffset() == Math.min(winterOffset, summerOffset))|0;
      HEAP32[(((tmPtr)+(32))>>2)]=dst;
  
      var zonePtr = HEAP32[(((__get_tzname())+(dst ? 4 : 0))>>2)];
      HEAP32[(((tmPtr)+(40))>>2)]=zonePtr;
  
      return tmPtr;
    }function _localtime(time) {
      return _localtime_r(time, ___tm_current);
    }

  
  function _emscripten_memcpy_big(dest, src, num) {
      HEAPU8.set(HEAPU8.subarray(src, src+num), dest);
    }
  
   

   

   

  
  function _usleep(useconds) {
      // int usleep(useconds_t useconds);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/usleep.html
      // We're single-threaded, so use a busy loop. Super-ugly.
      var msec = useconds / 1000;
      if ((ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) && self['performance'] && self['performance']['now']) {
        var start = self['performance']['now']();
        while (self['performance']['now']() - start < msec) {
          // Do nothing.
        }
      } else {
        var start = Date.now();
        while (Date.now() - start < msec) {
          // Do nothing.
        }
      }
      return 0;
    }
  Module["_usleep"] = _usleep;function _nanosleep(rqtp, rmtp) {
      // int nanosleep(const struct timespec  *rqtp, struct timespec *rmtp);
      var seconds = HEAP32[((rqtp)>>2)];
      var nanoseconds = HEAP32[(((rqtp)+(4))>>2)];
      if (rmtp !== 0) {
        HEAP32[((rmtp)>>2)]=0;
        HEAP32[(((rmtp)+(4))>>2)]=0;
      }
      return _usleep((seconds * 1e6) + (nanoseconds / 1000));
    }

   

  function _sysconf(name) {
      // long sysconf(int name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/sysconf.html
      switch(name) {
        case 30: return PAGE_SIZE;
        case 85:
          var maxHeapSize = 2*1024*1024*1024 - 16777216;
          maxHeapSize = HEAPU8.length;
          return maxHeapSize / PAGE_SIZE;
        case 132:
        case 133:
        case 12:
        case 137:
        case 138:
        case 15:
        case 235:
        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 149:
        case 13:
        case 10:
        case 236:
        case 153:
        case 9:
        case 21:
        case 22:
        case 159:
        case 154:
        case 14:
        case 77:
        case 78:
        case 139:
        case 80:
        case 81:
        case 82:
        case 68:
        case 67:
        case 164:
        case 11:
        case 29:
        case 47:
        case 48:
        case 95:
        case 52:
        case 51:
        case 46:
          return 200809;
        case 79:
          return 0;
        case 27:
        case 246:
        case 127:
        case 128:
        case 23:
        case 24:
        case 160:
        case 161:
        case 181:
        case 182:
        case 242:
        case 183:
        case 184:
        case 243:
        case 244:
        case 245:
        case 165:
        case 178:
        case 179:
        case 49:
        case 50:
        case 168:
        case 169:
        case 175:
        case 170:
        case 171:
        case 172:
        case 97:
        case 76:
        case 32:
        case 173:
        case 35:
          return -1;
        case 176:
        case 177:
        case 7:
        case 155:
        case 8:
        case 157:
        case 125:
        case 126:
        case 92:
        case 93:
        case 129:
        case 130:
        case 131:
        case 94:
        case 91:
          return 1;
        case 74:
        case 60:
        case 69:
        case 70:
        case 4:
          return 1024;
        case 31:
        case 42:
        case 72:
          return 32;
        case 87:
        case 26:
        case 33:
          return 2147483647;
        case 34:
        case 1:
          return 47839;
        case 38:
        case 36:
          return 99;
        case 43:
        case 37:
          return 2048;
        case 0: return 2097152;
        case 3: return 65536;
        case 28: return 32768;
        case 44: return 32767;
        case 75: return 16384;
        case 39: return 1000;
        case 89: return 700;
        case 71: return 256;
        case 40: return 255;
        case 2: return 100;
        case 180: return 64;
        case 25: return 20;
        case 5: return 16;
        case 6: return 6;
        case 73: return 4;
        case 84: {
          if (typeof navigator === 'object') return navigator['hardwareConcurrency'] || 1;
          return 1;
        }
      }
      ___setErrNo(22);
      return -1;
    }

  function _time(ptr) {
      var ret = (Date.now()/1000)|0;
      if (ptr) {
        HEAP32[((ptr)>>2)]=ret;
      }
      return ret;
    }

  function _utimes(path, times) {
      var time;
      if (times) {
        var offset = 8 + 0;
        time = HEAP32[(((times)+(offset))>>2)] * 1000;
        offset = 8 + 4;
        time += HEAP32[(((times)+(offset))>>2)] / 1000;
      } else {
        time = Date.now();
      }
      path = UTF8ToString(path);
      try {
        FS.utime(path, time, time);
        return 0;
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }
FS.staticInit();;
if (ENVIRONMENT_IS_NODE) { var fs = require("fs"); var NODEJS_PATH = require("path"); NODEFS.staticInit(); };
var ASSERTIONS = false;

// Copyright 2017 The Emscripten Authors.  All rights reserved.
// Emscripten is available under two separate licenses, the MIT license and the
// University of Illinois/NCSA Open Source License.  Both these licenses can be
// found in the LICENSE file.

/** @type {function(string, boolean=, number=)} */
function intArrayFromString(stringy, dontAddNull, length) {
  var len = length > 0 ? length : lengthBytesUTF8(stringy)+1;
  var u8array = new Array(len);
  var numBytesWritten = stringToUTF8Array(stringy, u8array, 0, u8array.length);
  if (dontAddNull) u8array.length = numBytesWritten;
  return u8array;
}

function intArrayToString(array) {
  var ret = [];
  for (var i = 0; i < array.length; i++) {
    var chr = array[i];
    if (chr > 0xFF) {
      if (ASSERTIONS) {
        assert(false, 'Character code ' + chr + ' (' + String.fromCharCode(chr) + ')  at offset ' + i + ' not in 0x00-0xFF.');
      }
      chr &= 0xFF;
    }
    ret.push(String.fromCharCode(chr));
  }
  return ret.join('');
}


// Copied from https://github.com/strophe/strophejs/blob/e06d027/src/polyfills.js#L149

// This code was written by Tyler Akins and has been placed in the
// public domain.  It would be nice if you left this header intact.
// Base64 code from Tyler Akins -- http://rumkin.com

/**
 * Decodes a base64 string.
 * @param {String} input The string to decode.
 */
var decodeBase64 = typeof atob === 'function' ? atob : function (input) {
  var keyStr = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';

  var output = '';
  var chr1, chr2, chr3;
  var enc1, enc2, enc3, enc4;
  var i = 0;
  // remove all characters that are not A-Z, a-z, 0-9, +, /, or =
  input = input.replace(/[^A-Za-z0-9\+\/\=]/g, '');
  do {
    enc1 = keyStr.indexOf(input.charAt(i++));
    enc2 = keyStr.indexOf(input.charAt(i++));
    enc3 = keyStr.indexOf(input.charAt(i++));
    enc4 = keyStr.indexOf(input.charAt(i++));

    chr1 = (enc1 << 2) | (enc2 >> 4);
    chr2 = ((enc2 & 15) << 4) | (enc3 >> 2);
    chr3 = ((enc3 & 3) << 6) | enc4;

    output = output + String.fromCharCode(chr1);

    if (enc3 !== 64) {
      output = output + String.fromCharCode(chr2);
    }
    if (enc4 !== 64) {
      output = output + String.fromCharCode(chr3);
    }
  } while (i < input.length);
  return output;
};

// Converts a string of base64 into a byte array.
// Throws error on invalid input.
function intArrayFromBase64(s) {
  if (typeof ENVIRONMENT_IS_NODE === 'boolean' && ENVIRONMENT_IS_NODE) {
    var buf;
    try {
      buf = Buffer.from(s, 'base64');
    } catch (_) {
      buf = new Buffer(s, 'base64');
    }
    return new Uint8Array(buf.buffer, buf.byteOffset, buf.byteLength);
  }

  try {
    var decoded = decodeBase64(s);
    var bytes = new Uint8Array(decoded.length);
    for (var i = 0 ; i < decoded.length ; ++i) {
      bytes[i] = decoded.charCodeAt(i);
    }
    return bytes;
  } catch (_) {
    throw new Error('Converting base64 string to bytes failed.');
  }
}

// If filename is a base64 data URI, parses and returns data (Buffer on node,
// Uint8Array otherwise). If filename is not a base64 data URI, returns undefined.
function tryParseAsDataURI(filename) {
  if (!isDataURI(filename)) {
    return;
  }

  return intArrayFromBase64(filename.slice(dataURIPrefix.length));
}


// ASM_LIBRARY EXTERN PRIMITIVES: Math_clz32,Math_imul,Int8Array,Int32Array


function jsCall_i(index) {
    return functionPointers[index]();
}

function jsCall_ii(index,a1) {
    return functionPointers[index](a1);
}

function jsCall_iii(index,a1,a2) {
    return functionPointers[index](a1,a2);
}

function jsCall_iiii(index,a1,a2,a3) {
    return functionPointers[index](a1,a2,a3);
}

function jsCall_iiiii(index,a1,a2,a3,a4) {
    return functionPointers[index](a1,a2,a3,a4);
}

function jsCall_iiiiii(index,a1,a2,a3,a4,a5) {
    return functionPointers[index](a1,a2,a3,a4,a5);
}

function jsCall_iiiiiii(index,a1,a2,a3,a4,a5,a6) {
    return functionPointers[index](a1,a2,a3,a4,a5,a6);
}

function jsCall_vi(index,a1) {
    functionPointers[index](a1);
}

function jsCall_vii(index,a1,a2) {
    functionPointers[index](a1,a2);
}

function jsCall_viii(index,a1,a2,a3) {
    functionPointers[index](a1,a2,a3);
}

function jsCall_viiii(index,a1,a2,a3,a4) {
    functionPointers[index](a1,a2,a3,a4);
}

function jsCall_viiiiii(index,a1,a2,a3,a4,a5,a6) {
    functionPointers[index](a1,a2,a3,a4,a5,a6);
}

var asmGlobalArg = { "Math": Math, "Int8Array": Int8Array, "Int16Array": Int16Array, "Int32Array": Int32Array, "Uint8Array": Uint8Array, "Uint16Array": Uint16Array, "Float32Array": Float32Array, "Float64Array": Float64Array, "Infinity": Infinity }

var asmLibraryArg = {
  "a": abort,
  "b": setTempRet0,
  "c": getTempRet0,
  "d": jsCall_i,
  "e": jsCall_ii,
  "f": jsCall_iii,
  "g": jsCall_iiii,
  "h": jsCall_iiiii,
  "i": jsCall_iiiiii,
  "j": jsCall_iiiiiii,
  "k": jsCall_vi,
  "l": jsCall_vii,
  "m": jsCall_viii,
  "n": jsCall_viiii,
  "o": jsCall_viiiiii,
  "p": ___assert_fail,
  "q": ___buildEnvironment,
  "r": ___setErrNo,
  "s": ___syscall10,
  "t": ___syscall118,
  "u": ___syscall140,
  "v": ___syscall15,
  "w": ___syscall183,
  "x": ___syscall192,
  "y": ___syscall194,
  "z": ___syscall195,
  "A": ___syscall196,
  "B": ___syscall197,
  "C": ___syscall20,
  "D": ___syscall201,
  "E": ___syscall202,
  "F": ___syscall207,
  "G": ___syscall212,
  "H": ___syscall221,
  "I": ___syscall3,
  "J": ___syscall33,
  "K": ___syscall39,
  "L": ___syscall4,
  "M": ___syscall40,
  "N": ___syscall5,
  "O": ___syscall6,
  "P": ___syscall85,
  "Q": ___syscall91,
  "R": ___syscall94,
  "S": _emscripten_get_heap_size,
  "T": _emscripten_memcpy_big,
  "U": _emscripten_resize_heap,
  "V": _getenv,
  "W": _gettimeofday,
  "X": _llvm_log10_f32,
  "Y": _llvm_log10_f64,
  "Z": _llvm_trap,
  "_": _localtime,
  "$": _localtime_r,
  "aa": _nanosleep,
  "ab": _sysconf,
  "ac": _time,
  "ad": _tzset,
  "ae": _usleep,
  "af": _utimes,
  "ag": abortOnCannotGrowMemory,
  "ah": tempDoublePtr,
  "ai": DYNAMICTOP_PTR
}
// EMSCRIPTEN_START_ASM
var asm = (/** @suppress {uselessCode} */ function(global, env, buffer) {
'use asm';

  var HEAP8 = new global.Int8Array(buffer),
  HEAP16 = new global.Int16Array(buffer),
  HEAP32 = new global.Int32Array(buffer),
  HEAPU8 = new global.Uint8Array(buffer),
  HEAPU16 = new global.Uint16Array(buffer),
  HEAPF32 = new global.Float32Array(buffer),
  HEAPF64 = new global.Float64Array(buffer),
  tempDoublePtr=env.ah|0,
  DYNAMICTOP_PTR=env.ai|0,
  __THREW__ = 0,
  threwValue = 0,
  setjmpId = 0,
  tempInt = 0,
  tempBigInt = 0,
  tempBigIntS = 0,
  tempValue = 0,
  tempDouble = 0.0,
  inf = global.Infinity,
  Math_floor=global.Math.floor,
  Math_abs=global.Math.abs,
  Math_sqrt=global.Math.sqrt,
  Math_pow=global.Math.pow,
  Math_cos=global.Math.cos,
  Math_sin=global.Math.sin,
  Math_tan=global.Math.tan,
  Math_acos=global.Math.acos,
  Math_asin=global.Math.asin,
  Math_atan=global.Math.atan,
  Math_atan2=global.Math.atan2,
  Math_exp=global.Math.exp,
  Math_log=global.Math.log,
  Math_ceil=global.Math.ceil,
  Math_imul=global.Math.imul,
  Math_min=global.Math.min,
  Math_clz32=global.Math.clz32,
  abort=env.a,
  setTempRet0=env.b,
  getTempRet0=env.c,
  jsCall_i=env.d,
  jsCall_ii=env.e,
  jsCall_iii=env.f,
  jsCall_iiii=env.g,
  jsCall_iiiii=env.h,
  jsCall_iiiiii=env.i,
  jsCall_iiiiiii=env.j,
  jsCall_vi=env.k,
  jsCall_vii=env.l,
  jsCall_viii=env.m,
  jsCall_viiii=env.n,
  jsCall_viiiiii=env.o,
  ___assert_fail=env.p,
  ___buildEnvironment=env.q,
  ___setErrNo=env.r,
  ___syscall10=env.s,
  ___syscall118=env.t,
  ___syscall140=env.u,
  ___syscall15=env.v,
  ___syscall183=env.w,
  ___syscall192=env.x,
  ___syscall194=env.y,
  ___syscall195=env.z,
  ___syscall196=env.A,
  ___syscall197=env.B,
  ___syscall20=env.C,
  ___syscall201=env.D,
  ___syscall202=env.E,
  ___syscall207=env.F,
  ___syscall212=env.G,
  ___syscall221=env.H,
  ___syscall3=env.I,
  ___syscall33=env.J,
  ___syscall39=env.K,
  ___syscall4=env.L,
  ___syscall40=env.M,
  ___syscall5=env.N,
  ___syscall6=env.O,
  ___syscall85=env.P,
  ___syscall91=env.Q,
  ___syscall94=env.R,
  _emscripten_get_heap_size=env.S,
  _emscripten_memcpy_big=env.T,
  _emscripten_resize_heap=env.U,
  _getenv=env.V,
  _gettimeofday=env.W,
  _llvm_log10_f32=env.X,
  _llvm_log10_f64=env.Y,
  _llvm_trap=env.Z,
  _localtime=env._,
  _localtime_r=env.$,
  _nanosleep=env.aa,
  _sysconf=env.ab,
  _time=env.ac,
  _tzset=env.ad,
  _usleep=env.ae,
  _utimes=env.af,
  abortOnCannotGrowMemory=env.ag,
  STACKTOP = 60496,
  STACK_MAX = 5303376,
  tempFloat = 0.0;

// EMSCRIPTEN_START_FUNCS

function stackAlloc(size) {
  size = size|0;
  var ret = 0;
  ret = STACKTOP;
  STACKTOP = (STACKTOP + size)|0;
  STACKTOP = (STACKTOP + 15)&-16;
  
  return ret|0;
}
function stackSave() {
  return STACKTOP|0;
}
function stackRestore(top) {
  top = top|0;
  STACKTOP = top;
}
function establishStackSpace(stackBase, stackMax) {
  stackBase = stackBase|0;
  stackMax = stackMax|0;
  STACKTOP = stackBase;
  STACK_MAX = stackMax;
}

function _sqlite3BtreeEnterAll($0) {
 $0 = $0|0;
 var $$010 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 20|0);
 $2 = HEAP32[$1>>2]|0;
 $3 = ($2|0)>(0);
 if (!($3)) {
  return;
 }
 $4 = ((($0)) + 16|0);
 $5 = HEAP32[$4>>2]|0;
 $$010 = 0;
 while(1) {
  $6 = (((($5) + ($$010<<4)|0)) + 4|0);
  $7 = HEAP32[$6>>2]|0;
  $8 = ($7|0)==(0|0);
  if (!($8)) {
   $9 = HEAP32[$7>>2]|0;
   $10 = ((($7)) + 4|0);
   $11 = HEAP32[$10>>2]|0;
   $12 = ((($11)) + 4|0);
   HEAP32[$12>>2] = $9;
  }
  $13 = (($$010) + 1)|0;
  $14 = ($13|0)<($2|0);
  if ($14) {
   $$010 = $13;
  } else {
   break;
  }
 }
 return;
}
function _sqlite3VdbeClearObject($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i = 0, $$0$i$us = 0, $$0$i32 = 0, $$0$i32$us = 0, $$012$i = 0, $$012$i46 = 0, $$1$i = 0, $$1$i$us = 0, $$1$i33 = 0, $$1$i33$us = 0, $$in = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0;
 var $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0;
 var $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0;
 var $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0;
 var $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0;
 var $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0;
 var $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0;
 var $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0;
 var $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0;
 var $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0;
 var $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0;
 var $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0;
 var $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0;
 var $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0;
 var $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0;
 var $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0;
 var $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0;
 var $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0;
 var $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0;
 var $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0;
 var $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0;
 var $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0;
 var $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $or$cond$i = 0, $or$cond$i31 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($1)) + 116|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ((($1)) + 144|0);
 $5 = HEAP16[$4>>1]|0;
 $6 = ($3|0)!=(0|0);
 $7 = ($5<<16>>16)!=(0);
 $or$cond$i = $6 & $7;
 L1: do {
  if ($or$cond$i) {
   $8 = $5&65535;
   $9 = $8 << 1;
   $10 = (($3) + (($9*40)|0)|0);
   $11 = ((($3)) + 32|0);
   $12 = HEAP32[$11>>2]|0;
   $13 = ((($12)) + 480|0);
   $14 = HEAP32[$13>>2]|0;
   $15 = ($14|0)==(0|0);
   $16 = ($12|0)==(0|0);
   $17 = ((($12)) + 304|0);
   $18 = ((($12)) + 308|0);
   $19 = ((($12)) + 300|0);
   if (!($15)) {
    if ($16) {
     $$0$i$us = $3;
     while(1) {
      $20 = ((($$0$i$us)) + 24|0);
      $21 = HEAP32[$20>>2]|0;
      $22 = ($21|0)==(0);
      do {
       if (!($22)) {
        $23 = ((($$0$i$us)) + 20|0);
        $24 = HEAP32[$23>>2]|0;
        $25 = ($24|0)==(0|0);
        if (!($25)) {
         $26 = HEAP32[7324]|0;
         $27 = ($26|0)==(0);
         if ($27) {
          $35 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$35 & 127]($24);
          break;
         } else {
          $28 = HEAP32[(29352)>>2]|0;
          $29 = (FUNCTION_TABLE_ii[$28 & 127]($24)|0);
          $30 = HEAP32[14978]|0;
          $31 = (($30) - ($29))|0;
          HEAP32[14978] = $31;
          $32 = HEAP32[14981]|0;
          $33 = (($32) + -1)|0;
          HEAP32[14981] = $33;
          $34 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$34 & 127]($24);
          break;
         }
        }
       }
      } while(0);
      $36 = ((($$0$i$us)) + 40|0);
      $37 = ($36>>>0)<($10>>>0);
      if ($37) {
       $$0$i$us = $36;
      } else {
       break L1;
      }
     }
    }
    $$0$i = $3;
    while(1) {
     $38 = ((($$0$i)) + 24|0);
     $39 = HEAP32[$38>>2]|0;
     $40 = ($39|0)==(0);
     do {
      if (!($40)) {
       $41 = ((($$0$i)) + 20|0);
       $42 = HEAP32[$41>>2]|0;
       $43 = ($42|0)==(0|0);
       if (!($43)) {
        $44 = HEAP32[$13>>2]|0;
        $45 = ($44|0)==(0|0);
        if (!($45)) {
         _measureAllocationSize($12,$42);
         break;
        }
        $46 = $42;
        $47 = HEAP32[$17>>2]|0;
        $48 = ($47>>>0)>($46>>>0);
        if (!($48)) {
         $49 = HEAP32[$18>>2]|0;
         $50 = ($49>>>0)>($46>>>0);
         if ($50) {
          $51 = HEAP32[$19>>2]|0;
          HEAP32[$42>>2] = $51;
          HEAP32[$19>>2] = $42;
          break;
         }
        }
        $52 = HEAP32[7324]|0;
        $53 = ($52|0)==(0);
        if ($53) {
         $61 = HEAP32[(29344)>>2]|0;
         FUNCTION_TABLE_vi[$61 & 127]($42);
         break;
        } else {
         $54 = HEAP32[(29352)>>2]|0;
         $55 = (FUNCTION_TABLE_ii[$54 & 127]($42)|0);
         $56 = HEAP32[14978]|0;
         $57 = (($56) - ($55))|0;
         HEAP32[14978] = $57;
         $58 = HEAP32[14981]|0;
         $59 = (($58) + -1)|0;
         HEAP32[14981] = $59;
         $60 = HEAP32[(29344)>>2]|0;
         FUNCTION_TABLE_vi[$60 & 127]($42);
         break;
        }
       }
      }
     } while(0);
     $62 = ((($$0$i)) + 40|0);
     $63 = ($62>>>0)<($10>>>0);
     if ($63) {
      $$0$i = $62;
     } else {
      break L1;
     }
    }
   }
   if ($16) {
    $$1$i$us = $3;
    while(1) {
     $64 = ((($$1$i$us)) + 8|0);
     $65 = HEAP16[$64>>1]|0;
     $66 = $65 & 9216;
     $67 = ($66<<16>>16)==(0);
     if ($67) {
      $68 = ((($$1$i$us)) + 24|0);
      $69 = HEAP32[$68>>2]|0;
      $70 = ($69|0)==(0);
      if (!($70)) {
       $71 = ((($$1$i$us)) + 20|0);
       $72 = HEAP32[$71>>2]|0;
       $73 = ($72|0)==(0|0);
       do {
        if (!($73)) {
         $74 = HEAP32[7324]|0;
         $75 = ($74|0)==(0);
         if ($75) {
          $83 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$83 & 127]($72);
          break;
         } else {
          $76 = HEAP32[(29352)>>2]|0;
          $77 = (FUNCTION_TABLE_ii[$76 & 127]($72)|0);
          $78 = HEAP32[14978]|0;
          $79 = (($78) - ($77))|0;
          HEAP32[14978] = $79;
          $80 = HEAP32[14981]|0;
          $81 = (($80) + -1)|0;
          HEAP32[14981] = $81;
          $82 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$82 & 127]($72);
          break;
         }
        }
       } while(0);
       HEAP32[$68>>2] = 0;
      }
     } else {
      _vdbeMemClear($$1$i$us);
     }
     HEAP16[$64>>1] = 128;
     $84 = ((($$1$i$us)) + 40|0);
     $85 = ($84>>>0)<($10>>>0);
     if ($85) {
      $$1$i$us = $84;
     } else {
      break L1;
     }
    }
   }
   $$1$i = $3;
   while(1) {
    $86 = ((($$1$i)) + 8|0);
    $87 = HEAP16[$86>>1]|0;
    $88 = $87 & 9216;
    $89 = ($88<<16>>16)==(0);
    if ($89) {
     $90 = ((($$1$i)) + 24|0);
     $91 = HEAP32[$90>>2]|0;
     $92 = ($91|0)==(0);
     if (!($92)) {
      $93 = ((($$1$i)) + 20|0);
      $94 = HEAP32[$93>>2]|0;
      $95 = HEAP32[$13>>2]|0;
      $96 = ($95|0)==(0|0);
      do {
       if ($96) {
        $97 = $94;
        $98 = HEAP32[$17>>2]|0;
        $99 = ($98>>>0)>($97>>>0);
        if (!($99)) {
         $100 = HEAP32[$18>>2]|0;
         $101 = ($100>>>0)>($97>>>0);
         if ($101) {
          $102 = HEAP32[$19>>2]|0;
          HEAP32[$94>>2] = $102;
          HEAP32[$19>>2] = $94;
          break;
         }
        }
        $103 = ($94|0)==(0|0);
        if (!($103)) {
         $104 = HEAP32[7324]|0;
         $105 = ($104|0)==(0);
         if ($105) {
          $113 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$113 & 127]($94);
          break;
         } else {
          $106 = HEAP32[(29352)>>2]|0;
          $107 = (FUNCTION_TABLE_ii[$106 & 127]($94)|0);
          $108 = HEAP32[14978]|0;
          $109 = (($108) - ($107))|0;
          HEAP32[14978] = $109;
          $110 = HEAP32[14981]|0;
          $111 = (($110) + -1)|0;
          HEAP32[14981] = $111;
          $112 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$112 & 127]($94);
          break;
         }
        }
       } else {
        _measureAllocationSize($12,$94);
       }
      } while(0);
      HEAP32[$90>>2] = 0;
     }
    } else {
     _vdbeMemClear($$1$i);
    }
    HEAP16[$86>>1] = 128;
    $114 = ((($$1$i)) + 40|0);
    $115 = ($114>>>0)<($10>>>0);
    if ($115) {
     $$1$i = $114;
    } else {
     break;
    }
   }
  }
 } while(0);
 $116 = ((($1)) + 212|0);
 $117 = HEAP32[$116>>2]|0;
 $118 = ($117|0)==(0|0);
 if (!($118)) {
  $119 = ($0|0)==(0|0);
  $120 = ((($0)) + 480|0);
  $121 = ((($0)) + 304|0);
  $122 = ((($0)) + 308|0);
  $123 = ((($0)) + 300|0);
  $$in = $117;
  while(1) {
   $124 = ((($$in)) + 24|0);
   $125 = HEAP32[$124>>2]|0;
   $126 = HEAP32[$$in>>2]|0;
   $127 = ((($$in)) + 4|0);
   $128 = HEAP32[$127>>2]|0;
   $129 = ($126|0)==(0|0);
   if ($129) {
    label = 67;
   } else {
    $130 = ($128|0)<(1);
    if (!($130)) {
     $131 = (($128) + -1)|0;
     $132 = (($126) + (($131*20)|0)|0);
     $$012$i = $132;
     while(1) {
      $133 = ((($$012$i)) + 1|0);
      $134 = HEAP8[$133>>0]|0;
      $135 = ($134<<24>>24)<(-6);
      if ($135) {
       $136 = $134 << 24 >> 24;
       $137 = ((($$012$i)) + 16|0);
       $138 = HEAP32[$137>>2]|0;
       _freeP4($0,$136,$138);
      }
      $139 = ((($$012$i)) + -20|0);
      $140 = ($139>>>0)<($126>>>0);
      if ($140) {
       break;
      } else {
       $$012$i = $139;
      }
     }
    }
    do {
     if ($119) {
      label = 63;
     } else {
      $141 = HEAP32[$120>>2]|0;
      $142 = ($141|0)==(0|0);
      if (!($142)) {
       _measureAllocationSize($0,$126);
       break;
      }
      $143 = $126;
      $144 = HEAP32[$121>>2]|0;
      $145 = ($144>>>0)>($143>>>0);
      if ($145) {
       label = 63;
      } else {
       $146 = HEAP32[$122>>2]|0;
       $147 = ($146>>>0)>($143>>>0);
       if ($147) {
        $148 = HEAP32[$123>>2]|0;
        HEAP32[$126>>2] = $148;
        HEAP32[$123>>2] = $126;
       } else {
        label = 63;
       }
      }
     }
    } while(0);
    do {
     if ((label|0) == 63) {
      label = 0;
      $149 = HEAP32[7324]|0;
      $150 = ($149|0)==(0);
      if ($150) {
       $158 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$158 & 127]($126);
       break;
      } else {
       $151 = HEAP32[(29352)>>2]|0;
       $152 = (FUNCTION_TABLE_ii[$151 & 127]($126)|0);
       $153 = HEAP32[14978]|0;
       $154 = (($153) - ($152))|0;
       HEAP32[14978] = $154;
       $155 = HEAP32[14981]|0;
       $156 = (($155) + -1)|0;
       HEAP32[14981] = $156;
       $157 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$157 & 127]($126);
       break;
      }
     }
    } while(0);
    $159 = ($$in|0)==(0|0);
    if (!($159)) {
     label = 67;
    }
   }
   do {
    if ((label|0) == 67) {
     label = 0;
     if (!($119)) {
      $160 = HEAP32[$120>>2]|0;
      $161 = ($160|0)==(0|0);
      if (!($161)) {
       _measureAllocationSize($0,$$in);
       break;
      }
      $162 = $$in;
      $163 = HEAP32[$121>>2]|0;
      $164 = ($163>>>0)>($162>>>0);
      if (!($164)) {
       $165 = HEAP32[$122>>2]|0;
       $166 = ($165>>>0)>($162>>>0);
       if ($166) {
        $167 = HEAP32[$123>>2]|0;
        HEAP32[$$in>>2] = $167;
        HEAP32[$123>>2] = $$in;
        break;
       }
      }
     }
     $168 = HEAP32[7324]|0;
     $169 = ($168|0)==(0);
     if ($169) {
      $177 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$177 & 127]($$in);
      break;
     } else {
      $170 = HEAP32[(29352)>>2]|0;
      $171 = (FUNCTION_TABLE_ii[$170 & 127]($$in)|0);
      $172 = HEAP32[14978]|0;
      $173 = (($172) - ($171))|0;
      HEAP32[14978] = $173;
      $174 = HEAP32[14981]|0;
      $175 = (($174) + -1)|0;
      HEAP32[14981] = $175;
      $176 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$176 & 127]($$in);
      break;
     }
    }
   } while(0);
   $178 = ($125|0)==(0|0);
   if ($178) {
    break;
   } else {
    $$in = $125;
   }
  }
 }
 $179 = ((($1)) + 20|0);
 $180 = HEAP32[$179>>2]|0;
 $181 = ($180|0)==(381479589);
 do {
  if (!($181)) {
   $182 = ((($1)) + 100|0);
   $183 = HEAP32[$182>>2]|0;
   $184 = ((($1)) + 16|0);
   $185 = HEAP16[$184>>1]|0;
   $186 = ($183|0)!=(0|0);
   $187 = ($185<<16>>16)!=(0);
   $or$cond$i31 = $186 & $187;
   L114: do {
    if ($or$cond$i31) {
     $188 = $185 << 16 >> 16;
     $189 = (($183) + (($188*40)|0)|0);
     $190 = ((($183)) + 32|0);
     $191 = HEAP32[$190>>2]|0;
     $192 = ((($191)) + 480|0);
     $193 = HEAP32[$192>>2]|0;
     $194 = ($193|0)==(0|0);
     $195 = ($191|0)==(0|0);
     $196 = ((($191)) + 304|0);
     $197 = ((($191)) + 308|0);
     $198 = ((($191)) + 300|0);
     if (!($194)) {
      if ($195) {
       $$0$i32$us = $183;
       while(1) {
        $199 = ((($$0$i32$us)) + 24|0);
        $200 = HEAP32[$199>>2]|0;
        $201 = ($200|0)==(0);
        do {
         if (!($201)) {
          $202 = ((($$0$i32$us)) + 20|0);
          $203 = HEAP32[$202>>2]|0;
          $204 = ($203|0)==(0|0);
          if (!($204)) {
           $205 = HEAP32[7324]|0;
           $206 = ($205|0)==(0);
           if ($206) {
            $214 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$214 & 127]($203);
            break;
           } else {
            $207 = HEAP32[(29352)>>2]|0;
            $208 = (FUNCTION_TABLE_ii[$207 & 127]($203)|0);
            $209 = HEAP32[14978]|0;
            $210 = (($209) - ($208))|0;
            HEAP32[14978] = $210;
            $211 = HEAP32[14981]|0;
            $212 = (($211) + -1)|0;
            HEAP32[14981] = $212;
            $213 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$213 & 127]($203);
            break;
           }
          }
         }
        } while(0);
        $215 = ((($$0$i32$us)) + 40|0);
        $216 = ($215>>>0)<($189>>>0);
        if ($216) {
         $$0$i32$us = $215;
        } else {
         break L114;
        }
       }
      }
      $$0$i32 = $183;
      while(1) {
       $217 = ((($$0$i32)) + 24|0);
       $218 = HEAP32[$217>>2]|0;
       $219 = ($218|0)==(0);
       do {
        if (!($219)) {
         $220 = ((($$0$i32)) + 20|0);
         $221 = HEAP32[$220>>2]|0;
         $222 = ($221|0)==(0|0);
         if (!($222)) {
          $223 = HEAP32[$192>>2]|0;
          $224 = ($223|0)==(0|0);
          if (!($224)) {
           _measureAllocationSize($191,$221);
           break;
          }
          $225 = $221;
          $226 = HEAP32[$196>>2]|0;
          $227 = ($226>>>0)>($225>>>0);
          if (!($227)) {
           $228 = HEAP32[$197>>2]|0;
           $229 = ($228>>>0)>($225>>>0);
           if ($229) {
            $230 = HEAP32[$198>>2]|0;
            HEAP32[$221>>2] = $230;
            HEAP32[$198>>2] = $221;
            break;
           }
          }
          $231 = HEAP32[7324]|0;
          $232 = ($231|0)==(0);
          if ($232) {
           $240 = HEAP32[(29344)>>2]|0;
           FUNCTION_TABLE_vi[$240 & 127]($221);
           break;
          } else {
           $233 = HEAP32[(29352)>>2]|0;
           $234 = (FUNCTION_TABLE_ii[$233 & 127]($221)|0);
           $235 = HEAP32[14978]|0;
           $236 = (($235) - ($234))|0;
           HEAP32[14978] = $236;
           $237 = HEAP32[14981]|0;
           $238 = (($237) + -1)|0;
           HEAP32[14981] = $238;
           $239 = HEAP32[(29344)>>2]|0;
           FUNCTION_TABLE_vi[$239 & 127]($221);
           break;
          }
         }
        }
       } while(0);
       $241 = ((($$0$i32)) + 40|0);
       $242 = ($241>>>0)<($189>>>0);
       if ($242) {
        $$0$i32 = $241;
       } else {
        break L114;
       }
      }
     }
     if ($195) {
      $$1$i33$us = $183;
      while(1) {
       $243 = ((($$1$i33$us)) + 8|0);
       $244 = HEAP16[$243>>1]|0;
       $245 = $244 & 9216;
       $246 = ($245<<16>>16)==(0);
       if ($246) {
        $247 = ((($$1$i33$us)) + 24|0);
        $248 = HEAP32[$247>>2]|0;
        $249 = ($248|0)==(0);
        if (!($249)) {
         $250 = ((($$1$i33$us)) + 20|0);
         $251 = HEAP32[$250>>2]|0;
         $252 = ($251|0)==(0|0);
         do {
          if (!($252)) {
           $253 = HEAP32[7324]|0;
           $254 = ($253|0)==(0);
           if ($254) {
            $262 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$262 & 127]($251);
            break;
           } else {
            $255 = HEAP32[(29352)>>2]|0;
            $256 = (FUNCTION_TABLE_ii[$255 & 127]($251)|0);
            $257 = HEAP32[14978]|0;
            $258 = (($257) - ($256))|0;
            HEAP32[14978] = $258;
            $259 = HEAP32[14981]|0;
            $260 = (($259) + -1)|0;
            HEAP32[14981] = $260;
            $261 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$261 & 127]($251);
            break;
           }
          }
         } while(0);
         HEAP32[$247>>2] = 0;
        }
       } else {
        _vdbeMemClear($$1$i33$us);
       }
       HEAP16[$243>>1] = 128;
       $263 = ((($$1$i33$us)) + 40|0);
       $264 = ($263>>>0)<($189>>>0);
       if ($264) {
        $$1$i33$us = $263;
       } else {
        break L114;
       }
      }
     }
     $$1$i33 = $183;
     while(1) {
      $265 = ((($$1$i33)) + 8|0);
      $266 = HEAP16[$265>>1]|0;
      $267 = $266 & 9216;
      $268 = ($267<<16>>16)==(0);
      if ($268) {
       $269 = ((($$1$i33)) + 24|0);
       $270 = HEAP32[$269>>2]|0;
       $271 = ($270|0)==(0);
       if (!($271)) {
        $272 = ((($$1$i33)) + 20|0);
        $273 = HEAP32[$272>>2]|0;
        $274 = HEAP32[$192>>2]|0;
        $275 = ($274|0)==(0|0);
        do {
         if ($275) {
          $276 = $273;
          $277 = HEAP32[$196>>2]|0;
          $278 = ($277>>>0)>($276>>>0);
          if (!($278)) {
           $279 = HEAP32[$197>>2]|0;
           $280 = ($279>>>0)>($276>>>0);
           if ($280) {
            $281 = HEAP32[$198>>2]|0;
            HEAP32[$273>>2] = $281;
            HEAP32[$198>>2] = $273;
            break;
           }
          }
          $282 = ($273|0)==(0|0);
          if (!($282)) {
           $283 = HEAP32[7324]|0;
           $284 = ($283|0)==(0);
           if ($284) {
            $292 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$292 & 127]($273);
            break;
           } else {
            $285 = HEAP32[(29352)>>2]|0;
            $286 = (FUNCTION_TABLE_ii[$285 & 127]($273)|0);
            $287 = HEAP32[14978]|0;
            $288 = (($287) - ($286))|0;
            HEAP32[14978] = $288;
            $289 = HEAP32[14981]|0;
            $290 = (($289) + -1)|0;
            HEAP32[14981] = $290;
            $291 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$291 & 127]($273);
            break;
           }
          }
         } else {
          _measureAllocationSize($191,$273);
         }
        } while(0);
        HEAP32[$269>>2] = 0;
       }
      } else {
       _vdbeMemClear($$1$i33);
      }
      HEAP16[$265>>1] = 128;
      $293 = ((($$1$i33)) + 40|0);
      $294 = ($293>>>0)<($189>>>0);
      if ($294) {
       $$1$i33 = $293;
      } else {
       break;
      }
     }
    }
   } while(0);
   $295 = ((($1)) + 128|0);
   $296 = HEAP32[$295>>2]|0;
   $297 = ($296|0)==(0|0);
   do {
    if (!($297)) {
     $298 = ($0|0)==(0|0);
     if (!($298)) {
      $299 = ((($0)) + 480|0);
      $300 = HEAP32[$299>>2]|0;
      $301 = ($300|0)==(0|0);
      if (!($301)) {
       _measureAllocationSize($0,$296);
       break;
      }
      $302 = $296;
      $303 = ((($0)) + 304|0);
      $304 = HEAP32[$303>>2]|0;
      $305 = ($304>>>0)>($302>>>0);
      if (!($305)) {
       $306 = ((($0)) + 308|0);
       $307 = HEAP32[$306>>2]|0;
       $308 = ($307>>>0)>($302>>>0);
       if ($308) {
        $309 = ((($0)) + 300|0);
        $310 = HEAP32[$309>>2]|0;
        HEAP32[$296>>2] = $310;
        HEAP32[$309>>2] = $296;
        break;
       }
      }
     }
     $311 = HEAP32[7324]|0;
     $312 = ($311|0)==(0);
     if ($312) {
      $320 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$320 & 127]($296);
      break;
     } else {
      $313 = HEAP32[(29352)>>2]|0;
      $314 = (FUNCTION_TABLE_ii[$313 & 127]($296)|0);
      $315 = HEAP32[14978]|0;
      $316 = (($315) - ($314))|0;
      HEAP32[14978] = $316;
      $317 = HEAP32[14981]|0;
      $318 = (($317) + -1)|0;
      HEAP32[14981] = $318;
      $319 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$319 & 127]($296);
      break;
     }
    }
   } while(0);
   $321 = ((($1)) + 192|0);
   $322 = HEAP32[$321>>2]|0;
   $323 = ($322|0)==(0|0);
   if (!($323)) {
    $324 = ($0|0)==(0|0);
    if (!($324)) {
     $325 = ((($0)) + 480|0);
     $326 = HEAP32[$325>>2]|0;
     $327 = ($326|0)==(0|0);
     if (!($327)) {
      _measureAllocationSize($0,$322);
      break;
     }
     $328 = $322;
     $329 = ((($0)) + 304|0);
     $330 = HEAP32[$329>>2]|0;
     $331 = ($330>>>0)>($328>>>0);
     if (!($331)) {
      $332 = ((($0)) + 308|0);
      $333 = HEAP32[$332>>2]|0;
      $334 = ($333>>>0)>($328>>>0);
      if ($334) {
       $335 = ((($0)) + 300|0);
       $336 = HEAP32[$335>>2]|0;
       HEAP32[$322>>2] = $336;
       HEAP32[$335>>2] = $322;
       break;
      }
     }
    }
    $337 = HEAP32[7324]|0;
    $338 = ($337|0)==(0);
    if ($338) {
     $346 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$346 & 127]($322);
     break;
    } else {
     $339 = HEAP32[(29352)>>2]|0;
     $340 = (FUNCTION_TABLE_ii[$339 & 127]($322)|0);
     $341 = HEAP32[14978]|0;
     $342 = (($341) - ($340))|0;
     HEAP32[14978] = $342;
     $343 = HEAP32[14981]|0;
     $344 = (($343) + -1)|0;
     HEAP32[14981] = $344;
     $345 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$345 & 127]($322);
     break;
    }
   }
  }
 } while(0);
 $347 = ((($1)) + 104|0);
 $348 = HEAP32[$347>>2]|0;
 $349 = ((($1)) + 108|0);
 $350 = HEAP32[$349>>2]|0;
 $351 = ($348|0)==(0|0);
 do {
  if (!($351)) {
   $352 = ($350|0)<(1);
   if (!($352)) {
    $353 = (($350) + -1)|0;
    $354 = (($348) + (($353*20)|0)|0);
    $$012$i46 = $354;
    while(1) {
     $355 = ((($$012$i46)) + 1|0);
     $356 = HEAP8[$355>>0]|0;
     $357 = ($356<<24>>24)<(-6);
     if ($357) {
      $358 = $356 << 24 >> 24;
      $359 = ((($$012$i46)) + 16|0);
      $360 = HEAP32[$359>>2]|0;
      _freeP4($0,$358,$360);
     }
     $361 = ((($$012$i46)) + -20|0);
     $362 = ($361>>>0)<($348>>>0);
     if ($362) {
      break;
     } else {
      $$012$i46 = $361;
     }
    }
   }
   $363 = ($0|0)==(0|0);
   if (!($363)) {
    $364 = ((($0)) + 480|0);
    $365 = HEAP32[$364>>2]|0;
    $366 = ($365|0)==(0|0);
    if (!($366)) {
     _measureAllocationSize($0,$348);
     break;
    }
    $367 = $348;
    $368 = ((($0)) + 304|0);
    $369 = HEAP32[$368>>2]|0;
    $370 = ($369>>>0)>($367>>>0);
    if (!($370)) {
     $371 = ((($0)) + 308|0);
     $372 = HEAP32[$371>>2]|0;
     $373 = ($372>>>0)>($367>>>0);
     if ($373) {
      $374 = ((($0)) + 300|0);
      $375 = HEAP32[$374>>2]|0;
      HEAP32[$348>>2] = $375;
      HEAP32[$374>>2] = $348;
      break;
     }
    }
   }
   $376 = HEAP32[7324]|0;
   $377 = ($376|0)==(0);
   if ($377) {
    $385 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$385 & 127]($348);
    break;
   } else {
    $378 = HEAP32[(29352)>>2]|0;
    $379 = (FUNCTION_TABLE_ii[$378 & 127]($348)|0);
    $380 = HEAP32[14978]|0;
    $381 = (($380) - ($379))|0;
    HEAP32[14978] = $381;
    $382 = HEAP32[14981]|0;
    $383 = (($382) + -1)|0;
    HEAP32[14981] = $383;
    $384 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$384 & 127]($348);
    break;
   }
  }
 } while(0);
 $386 = HEAP32[$2>>2]|0;
 $387 = ($386|0)==(0|0);
 do {
  if (!($387)) {
   $388 = ($0|0)==(0|0);
   if (!($388)) {
    $389 = ((($0)) + 480|0);
    $390 = HEAP32[$389>>2]|0;
    $391 = ($390|0)==(0|0);
    if (!($391)) {
     _measureAllocationSize($0,$386);
     break;
    }
    $392 = $386;
    $393 = ((($0)) + 304|0);
    $394 = HEAP32[$393>>2]|0;
    $395 = ($394>>>0)>($392>>>0);
    if (!($395)) {
     $396 = ((($0)) + 308|0);
     $397 = HEAP32[$396>>2]|0;
     $398 = ($397>>>0)>($392>>>0);
     if ($398) {
      $399 = ((($0)) + 300|0);
      $400 = HEAP32[$399>>2]|0;
      HEAP32[$386>>2] = $400;
      HEAP32[$399>>2] = $386;
      break;
     }
    }
   }
   $401 = HEAP32[7324]|0;
   $402 = ($401|0)==(0);
   if ($402) {
    $410 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$410 & 127]($386);
    break;
   } else {
    $403 = HEAP32[(29352)>>2]|0;
    $404 = (FUNCTION_TABLE_ii[$403 & 127]($386)|0);
    $405 = HEAP32[14978]|0;
    $406 = (($405) - ($404))|0;
    HEAP32[14978] = $406;
    $407 = HEAP32[14981]|0;
    $408 = (($407) + -1)|0;
    HEAP32[14981] = $408;
    $409 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$409 & 127]($386);
    break;
   }
  }
 } while(0);
 $411 = ((($1)) + 188|0);
 $412 = HEAP32[$411>>2]|0;
 $413 = ($412|0)==(0|0);
 if ($413) {
  return;
 }
 $414 = ($0|0)==(0|0);
 if (!($414)) {
  $415 = ((($0)) + 480|0);
  $416 = HEAP32[$415>>2]|0;
  $417 = ($416|0)==(0|0);
  if (!($417)) {
   _measureAllocationSize($0,$412);
   return;
  }
  $418 = $412;
  $419 = ((($0)) + 304|0);
  $420 = HEAP32[$419>>2]|0;
  $421 = ($420>>>0)>($418>>>0);
  if (!($421)) {
   $422 = ((($0)) + 308|0);
   $423 = HEAP32[$422>>2]|0;
   $424 = ($423>>>0)>($418>>>0);
   if ($424) {
    $425 = ((($0)) + 300|0);
    $426 = HEAP32[$425>>2]|0;
    HEAP32[$412>>2] = $426;
    HEAP32[$425>>2] = $412;
    return;
   }
  }
 }
 $427 = HEAP32[7324]|0;
 $428 = ($427|0)==(0);
 if ($428) {
  $436 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$436 & 127]($412);
  return;
 } else {
  $429 = HEAP32[(29352)>>2]|0;
  $430 = (FUNCTION_TABLE_ii[$429 & 127]($412)|0);
  $431 = HEAP32[14978]|0;
  $432 = (($431) - ($430))|0;
  HEAP32[14978] = $432;
  $433 = HEAP32[14981]|0;
  $434 = (($433) + -1)|0;
  HEAP32[14981] = $434;
  $435 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$435 & 127]($412);
  return;
 }
}
function _sqlite3DbFree($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $3 = 0, $4 = 0, $5 = 0;
 var $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($1|0)==(0|0);
 if ($2) {
  return;
 }
 $3 = ($0|0)==(0|0);
 if (!($3)) {
  $4 = ((($0)) + 480|0);
  $5 = HEAP32[$4>>2]|0;
  $6 = ($5|0)==(0|0);
  if (!($6)) {
   _measureAllocationSize($0,$1);
   return;
  }
  $7 = $1;
  $8 = ((($0)) + 304|0);
  $9 = HEAP32[$8>>2]|0;
  $10 = ($9>>>0)>($7>>>0);
  if (!($10)) {
   $11 = ((($0)) + 308|0);
   $12 = HEAP32[$11>>2]|0;
   $13 = ($12>>>0)>($7>>>0);
   if ($13) {
    $14 = ((($0)) + 300|0);
    $15 = HEAP32[$14>>2]|0;
    HEAP32[$1>>2] = $15;
    HEAP32[$14>>2] = $1;
    return;
   }
  }
 }
 $16 = HEAP32[7324]|0;
 $17 = ($16|0)==(0);
 if ($17) {
  $25 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$25 & 127]($1);
  return;
 } else {
  $18 = HEAP32[(29352)>>2]|0;
  $19 = (FUNCTION_TABLE_ii[$18 & 127]($1)|0);
  $20 = HEAP32[14978]|0;
  $21 = (($20) - ($19))|0;
  HEAP32[14978] = $21;
  $22 = HEAP32[14981]|0;
  $23 = (($22) + -1)|0;
  HEAP32[14981] = $23;
  $24 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$24 & 127]($1);
  return;
 }
}
function _sqlite3_initialize() {
 var $$ = 0, $$0$i$1$i$i = 0, $$0$i$2$i$i = 0, $$0$i$i = 0, $$0$i$i$1$i$i = 0, $$0$i$i$2$i$i = 0, $$0$i$i$i = 0, $$0$i$i$i$i = 0, $$0$i$i$i1$i = 0, $$0$i$i$i17$i = 0, $$0$i$i$i19 = 0, $$0$i$i$i19$1 = 0, $$0$i$i$i19$2 = 0, $$0$i$i$i19$3 = 0, $$0$i$i13$i = 0, $$0$i$i29$i = 0, $$0$i$i33$i = 0, $$0$i1725 = 0, $$0$lcssa$i = 0, $$01315$i$i$1$i$i = 0;
 var $$01315$i$i$2$i$i = 0, $$01315$i$i$i = 0, $$01315$i$i$i$i = 0, $$01315$i$i$i25$i = 0, $$01315$i$i$i9$i = 0, $$014 = 0, $$016$i$i$1$i$i = 0, $$016$i$i$2$i$i = 0, $$016$i$i$i = 0, $$016$i$i$i$i = 0, $$016$i$i$i24$i = 0, $$016$i$i$i8$i = 0, $$019$i = 0, $$033$i$i = 0, $$033$i$i$i = 0, $$033$i$i16$i = 0, $$07$i$1$i$i = 0, $$07$i$2$i$i = 0, $$07$i$i = 0, $$07$i$i$i = 0;
 var $$07$i$i18$i = 0, $$07$i$i2$i = 0, $$08$i$1$i$i = 0, $$08$i$2$i$i = 0, $$08$i$i = 0, $$08$i$i$i = 0, $$08$i$i21$i = 0, $$08$i$i5$i = 0, $$118$i = 0, $$326 = 0, $$4 = 0, $$lcssa$i$i$1$i$i = 0, $$lcssa$i$i$2$i$i = 0, $$lcssa$i$i$i = 0, $$lcssa$i$i$i$i = 0, $$lcssa$i$i$i11$i = 0, $$lcssa$i$i$i27$i = 0, $$op$i = 0, $$pr = 0, $$pre = 0;
 var $$sink = 0, $$sink89 = 0, $$sink90 = 0, $$sink91 = 0, $$sink92 = 0, $$sink93 = 0, $0 = 0, $1 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0;
 var $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0;
 var $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0;
 var $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0;
 var $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0;
 var $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $2 = 0, $20 = 0;
 var $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0;
 var $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0;
 var $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0;
 var $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0;
 var $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0;
 var $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0;
 var $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0;
 var $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0;
 var $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0;
 var $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0;
 var $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0;
 var $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0;
 var $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0;
 var $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0;
 var $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0;
 var $471 = 0, $472 = 0, $473 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0;
 var $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0;
 var $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0;
 var $exitcond$i$i = 0, $exitcond$i$i$i = 0, $exitcond$i$i32$i = 0, $or$cond$i = 0, $or$cond$i$i$1$i$i = 0, $or$cond$i$i$2$i$i = 0, $or$cond$i$i$i = 0, $or$cond$i$i$i$i = 0, $or$cond$i$i$i10$i = 0, $or$cond$i$i$i26$i = 0, $or$cond14$i$i$1$i$i = 0, $or$cond14$i$i$2$i$i = 0, $or$cond14$i$i$i = 0, $or$cond14$i$i$i$i = 0, $or$cond14$i$i$i22$i = 0, $or$cond14$i$i$i6$i = 0, $or$cond3$i = 0, $scevgep = 0, $scevgep$i = 0, $spec$select$i$i$i = 0;
 var $spec$select$i$i$i$1 = 0, $spec$select$i$i$i$2 = 0, $spec$select$i$i$i$3 = 0, $spec$select17$i = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vfsList$sink = 0, $vfsList$sink95 = 0, $vfsList$sink96 = 0, dest = 0, label = 0, sp = 0, stop = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $vararg_buffer1 = sp + 8|0;
 $vararg_buffer = sp;
 $0 = HEAP32[(29520)>>2]|0;
 $1 = ($0|0)==(0);
 if (!($1)) {
  $$014 = 0;
  STACKTOP = sp;return ($$014|0);
 }
 HEAP32[(29528)>>2] = 1;
 $2 = HEAP32[(29532)>>2]|0;
 $3 = ($2|0)==(0);
 if ($3) {
  $4 = HEAP32[(29340)>>2]|0;
  $5 = ($4|0)==(0|0);
  if ($5) {
   HEAP32[$vararg_buffer>>2] = 30308;
   (_sqlite3_config(4,$vararg_buffer)|0);
  }
  ;HEAP32[59056>>2]=0|0;HEAP32[59056+4>>2]=0|0;HEAP32[59056+8>>2]=0|0;HEAP32[59056+12>>2]=0|0;HEAP32[59056+16>>2]=0|0;HEAP32[59056+20>>2]=0|0;
  HEAP32[14764] = (8);
  $6 = HEAP32[(29496)>>2]|0;
  $7 = ($6|0)==(0|0);
  $8 = HEAP32[(29500)>>2]|0;
  $9 = ($8|0)<(512);
  $or$cond$i = $7 | $9;
  $10 = HEAP32[(29504)>>2]|0;
  $11 = ($10|0)<(1);
  $or$cond3$i = $or$cond$i | $11;
  if ($or$cond3$i) {
   HEAP32[(29496)>>2] = 0;
   HEAP32[(29500)>>2] = 0;
  }
  $12 = HEAP32[(29360)>>2]|0;
  $13 = HEAP32[(29368)>>2]|0;
  $14 = (FUNCTION_TABLE_ii[$12 & 127]($13)|0);
  $15 = ($14|0)==(0);
  if (!($15)) {
   ;HEAP32[59056>>2]=0|0;HEAP32[59056+4>>2]=0|0;HEAP32[59056+8>>2]=0|0;HEAP32[59056+12>>2]=0|0;HEAP32[59056+16>>2]=0|0;HEAP32[59056+20>>2]=0|0;
   $$014 = $14;
   STACKTOP = sp;return ($$014|0);
  }
 }
 HEAP32[(29532)>>2] = 1;
 $16 = HEAP32[(29544)>>2]|0;
 $17 = ($16|0)==(0|0);
 if ($17) {
  HEAP32[(29544)>>2] = (8);
 }
 $18 = HEAP32[(29540)>>2]|0;
 $19 = (($18) + 1)|0;
 HEAP32[(29540)>>2] = $19;
 $20 = HEAP32[(29520)>>2]|0;
 $21 = HEAP32[(29524)>>2]|0;
 $22 = $21 | $20;
 $23 = ($22|0)==(0);
 if ($23) {
  HEAP32[(29524)>>2] = 1;
  dest=59088; stop=dest+92|0; do { HEAP32[dest>>2]=0|0; dest=dest+4|0; } while ((dest|0) < (stop|0));
  $24 = HEAP32[(21792)>>2]|0;
  $25 = ($24|0)==(0|0);
  if ($25) {
   $$0$i$i$i$i = 0;
  } else {
   $26 = (_strlen($24)|0);
   $27 = $26 & 1073741823;
   $$0$i$i$i$i = $27;
  }
  $28 = HEAP8[$24>>0]|0;
  $29 = $28 << 24 >> 24;
  $30 = (($$0$i$i$i$i) + ($29))|0;
  $31 = (($30|0) % 23)&-1;
  $32 = (59088 + ($31<<2)|0);
  $$07$i$i$i = HEAP32[$32>>2]|0;
  $33 = ($$07$i$i$i|0)==(0|0);
  $34 = $$07$i$i$i;
  L23: do {
   if ($33) {
    label = 23;
   } else {
    $35 = $28&255;
    $36 = (208 + ($35)|0);
    $37 = HEAP8[$36>>0]|0;
    $38 = $37&255;
    $$08$i$i$i = $$07$i$i$i;
    while(1) {
     $39 = ((($$08$i$i$i)) + 32|0);
     $40 = HEAP32[$39>>2]|0;
     $41 = HEAP8[$40>>0]|0;
     $42 = $41&255;
     $43 = (208 + ($42)|0);
     $44 = HEAP8[$43>>0]|0;
     $45 = $44&255;
     $46 = (($45) - ($38))|0;
     $47 = ($46|0)!=(0);
     $48 = ($41<<24>>24)==(0);
     $or$cond14$i$i$i$i = $48 | $47;
     if ($or$cond14$i$i$i$i) {
      $$lcssa$i$i$i$i = $46;
     } else {
      $$01315$i$i$i$i = $24;$$016$i$i$i$i = $40;
      while(1) {
       $49 = ((($$016$i$i$i$i)) + 1|0);
       $50 = ((($$01315$i$i$i$i)) + 1|0);
       $51 = HEAP8[$49>>0]|0;
       $52 = $51&255;
       $53 = (208 + ($52)|0);
       $54 = HEAP8[$53>>0]|0;
       $55 = $54&255;
       $56 = HEAP8[$50>>0]|0;
       $57 = $56&255;
       $58 = (208 + ($57)|0);
       $59 = HEAP8[$58>>0]|0;
       $60 = $59&255;
       $61 = (($55) - ($60))|0;
       $62 = ($61|0)!=(0);
       $63 = ($51<<24>>24)==(0);
       $or$cond$i$i$i$i = $63 | $62;
       if ($or$cond$i$i$i$i) {
        $$lcssa$i$i$i$i = $61;
        break;
       } else {
        $$01315$i$i$i$i = $50;$$016$i$i$i$i = $49;
       }
      }
     }
     $64 = ($$lcssa$i$i$i$i|0)==(0);
     if ($64) {
      break;
     }
     $65 = ((($$08$i$i$i)) + 36|0);
     $$0$i$i$i = HEAP32[$65>>2]|0;
     $66 = ($$0$i$i$i|0)==(0|0);
     if ($66) {
      label = 23;
      break L23;
     } else {
      $$08$i$i$i = $$0$i$i$i;
     }
    }
    $67 = ($$08$i$i$i|0)==(0|0);
    if ($67) {
     label = 23;
    } else {
     $68 = ((($$08$i$i$i)) + 12|0);
     $69 = HEAP32[$68>>2]|0;
     HEAP32[(21772)>>2] = $69;
     $$sink = $68;
    }
   }
  } while(0);
  if ((label|0) == 23) {
   HEAP32[(21772)>>2] = 0;
   HEAP32[(21796)>>2] = $34;
   $$sink = $32;
  }
  HEAP32[$$sink>>2] = 21760;
  $70 = HEAP32[(21832)>>2]|0;
  $71 = ($70|0)==(0|0);
  if ($71) {
   $$0$i$i$1$i$i = 0;
  } else {
   $72 = (_strlen($70)|0);
   $73 = $72 & 1073741823;
   $$0$i$i$1$i$i = $73;
  }
  $74 = HEAP8[$70>>0]|0;
  $75 = $74 << 24 >> 24;
  $76 = (($$0$i$i$1$i$i) + ($75))|0;
  $77 = (($76|0) % 23)&-1;
  $78 = (59088 + ($77<<2)|0);
  $$07$i$1$i$i = HEAP32[$78>>2]|0;
  $79 = ($$07$i$1$i$i|0)==(0|0);
  $80 = $$07$i$1$i$i;
  L41: do {
   if ($79) {
    label = 35;
   } else {
    $81 = $74&255;
    $82 = (208 + ($81)|0);
    $83 = HEAP8[$82>>0]|0;
    $84 = $83&255;
    $$08$i$1$i$i = $$07$i$1$i$i;
    while(1) {
     $85 = ((($$08$i$1$i$i)) + 32|0);
     $86 = HEAP32[$85>>2]|0;
     $87 = HEAP8[$86>>0]|0;
     $88 = $87&255;
     $89 = (208 + ($88)|0);
     $90 = HEAP8[$89>>0]|0;
     $91 = $90&255;
     $92 = (($91) - ($84))|0;
     $93 = ($92|0)!=(0);
     $94 = ($87<<24>>24)==(0);
     $or$cond14$i$i$1$i$i = $94 | $93;
     if ($or$cond14$i$i$1$i$i) {
      $$lcssa$i$i$1$i$i = $92;
     } else {
      $$01315$i$i$1$i$i = $70;$$016$i$i$1$i$i = $86;
      while(1) {
       $95 = ((($$016$i$i$1$i$i)) + 1|0);
       $96 = ((($$01315$i$i$1$i$i)) + 1|0);
       $97 = HEAP8[$95>>0]|0;
       $98 = $97&255;
       $99 = (208 + ($98)|0);
       $100 = HEAP8[$99>>0]|0;
       $101 = $100&255;
       $102 = HEAP8[$96>>0]|0;
       $103 = $102&255;
       $104 = (208 + ($103)|0);
       $105 = HEAP8[$104>>0]|0;
       $106 = $105&255;
       $107 = (($101) - ($106))|0;
       $108 = ($107|0)!=(0);
       $109 = ($97<<24>>24)==(0);
       $or$cond$i$i$1$i$i = $109 | $108;
       if ($or$cond$i$i$1$i$i) {
        $$lcssa$i$i$1$i$i = $107;
        break;
       } else {
        $$01315$i$i$1$i$i = $96;$$016$i$i$1$i$i = $95;
       }
      }
     }
     $110 = ($$lcssa$i$i$1$i$i|0)==(0);
     if ($110) {
      break;
     }
     $111 = ((($$08$i$1$i$i)) + 36|0);
     $$0$i$1$i$i = HEAP32[$111>>2]|0;
     $112 = ($$0$i$1$i$i|0)==(0|0);
     if ($112) {
      label = 35;
      break L41;
     } else {
      $$08$i$1$i$i = $$0$i$1$i$i;
     }
    }
    $113 = ($$08$i$1$i$i|0)==(0|0);
    if ($113) {
     label = 35;
    } else {
     $114 = ((($$08$i$1$i$i)) + 12|0);
     $115 = HEAP32[$114>>2]|0;
     HEAP32[(21812)>>2] = $115;
     $$sink89 = $114;
    }
   }
  } while(0);
  if ((label|0) == 35) {
   HEAP32[(21812)>>2] = 0;
   HEAP32[(21836)>>2] = $80;
   $$sink89 = $78;
  }
  HEAP32[$$sink89>>2] = (21800);
  $116 = HEAP32[(21872)>>2]|0;
  $117 = ($116|0)==(0|0);
  if ($117) {
   $$0$i$i$2$i$i = 0;
  } else {
   $118 = (_strlen($116)|0);
   $119 = $118 & 1073741823;
   $$0$i$i$2$i$i = $119;
  }
  $120 = HEAP8[$116>>0]|0;
  $121 = $120 << 24 >> 24;
  $122 = (($$0$i$i$2$i$i) + ($121))|0;
  $123 = (($122|0) % 23)&-1;
  $124 = (59088 + ($123<<2)|0);
  $$07$i$2$i$i = HEAP32[$124>>2]|0;
  $125 = ($$07$i$2$i$i|0)==(0|0);
  $126 = $$07$i$2$i$i;
  L59: do {
   if ($125) {
    label = 47;
   } else {
    $127 = $120&255;
    $128 = (208 + ($127)|0);
    $129 = HEAP8[$128>>0]|0;
    $130 = $129&255;
    $$08$i$2$i$i = $$07$i$2$i$i;
    while(1) {
     $131 = ((($$08$i$2$i$i)) + 32|0);
     $132 = HEAP32[$131>>2]|0;
     $133 = HEAP8[$132>>0]|0;
     $134 = $133&255;
     $135 = (208 + ($134)|0);
     $136 = HEAP8[$135>>0]|0;
     $137 = $136&255;
     $138 = (($137) - ($130))|0;
     $139 = ($138|0)!=(0);
     $140 = ($133<<24>>24)==(0);
     $or$cond14$i$i$2$i$i = $140 | $139;
     if ($or$cond14$i$i$2$i$i) {
      $$lcssa$i$i$2$i$i = $138;
     } else {
      $$01315$i$i$2$i$i = $116;$$016$i$i$2$i$i = $132;
      while(1) {
       $141 = ((($$016$i$i$2$i$i)) + 1|0);
       $142 = ((($$01315$i$i$2$i$i)) + 1|0);
       $143 = HEAP8[$141>>0]|0;
       $144 = $143&255;
       $145 = (208 + ($144)|0);
       $146 = HEAP8[$145>>0]|0;
       $147 = $146&255;
       $148 = HEAP8[$142>>0]|0;
       $149 = $148&255;
       $150 = (208 + ($149)|0);
       $151 = HEAP8[$150>>0]|0;
       $152 = $151&255;
       $153 = (($147) - ($152))|0;
       $154 = ($153|0)!=(0);
       $155 = ($143<<24>>24)==(0);
       $or$cond$i$i$2$i$i = $155 | $154;
       if ($or$cond$i$i$2$i$i) {
        $$lcssa$i$i$2$i$i = $153;
        break;
       } else {
        $$01315$i$i$2$i$i = $142;$$016$i$i$2$i$i = $141;
       }
      }
     }
     $156 = ($$lcssa$i$i$2$i$i|0)==(0);
     if ($156) {
      break;
     }
     $157 = ((($$08$i$2$i$i)) + 36|0);
     $$0$i$2$i$i = HEAP32[$157>>2]|0;
     $158 = ($$0$i$2$i$i|0)==(0|0);
     if ($158) {
      label = 47;
      break L59;
     } else {
      $$08$i$2$i$i = $$0$i$2$i$i;
     }
    }
    $159 = ($$08$i$2$i$i|0)==(0|0);
    if ($159) {
     label = 47;
    } else {
     $160 = ((($$08$i$2$i$i)) + 12|0);
     $161 = HEAP32[$160>>2]|0;
     HEAP32[(21852)>>2] = $161;
     $$sink90 = $160;
    }
   }
  } while(0);
  if ((label|0) == 47) {
   HEAP32[(21852)>>2] = 0;
   HEAP32[(21876)>>2] = $126;
   $$sink90 = $124;
  }
  HEAP32[$$sink90>>2] = (21840);
  $$033$i$i$i = 0;
  while(1) {
   $162 = (21888 + (($$033$i$i$i*40)|0)|0);
   $163 = (((21888 + (($$033$i$i$i*40)|0)|0)) + 32|0);
   $164 = HEAP32[$163>>2]|0;
   $165 = ($164|0)==(0|0);
   if ($165) {
    $$0$i$i$i1$i = 0;
   } else {
    $166 = (_strlen($164)|0);
    $167 = $166 & 1073741823;
    $$0$i$i$i1$i = $167;
   }
   $168 = HEAP8[$164>>0]|0;
   $169 = $168 << 24 >> 24;
   $170 = (($$0$i$i$i1$i) + ($169))|0;
   $171 = (($170|0) % 23)&-1;
   $172 = (59088 + ($171<<2)|0);
   $$07$i$i2$i = HEAP32[$172>>2]|0;
   $173 = ($$07$i$i2$i|0)==(0|0);
   $174 = $$07$i$i2$i;
   L79: do {
    if ($173) {
     label = 60;
    } else {
     $175 = $168&255;
     $176 = (208 + ($175)|0);
     $177 = HEAP8[$176>>0]|0;
     $178 = $177&255;
     $$08$i$i5$i = $$07$i$i2$i;
     while(1) {
      $179 = ((($$08$i$i5$i)) + 32|0);
      $180 = HEAP32[$179>>2]|0;
      $181 = HEAP8[$180>>0]|0;
      $182 = $181&255;
      $183 = (208 + ($182)|0);
      $184 = HEAP8[$183>>0]|0;
      $185 = $184&255;
      $186 = (($185) - ($178))|0;
      $187 = ($186|0)!=(0);
      $188 = ($181<<24>>24)==(0);
      $or$cond14$i$i$i6$i = $188 | $187;
      if ($or$cond14$i$i$i6$i) {
       $$lcssa$i$i$i11$i = $186;
      } else {
       $$01315$i$i$i9$i = $164;$$016$i$i$i8$i = $180;
       while(1) {
        $189 = ((($$016$i$i$i8$i)) + 1|0);
        $190 = ((($$01315$i$i$i9$i)) + 1|0);
        $191 = HEAP8[$189>>0]|0;
        $192 = $191&255;
        $193 = (208 + ($192)|0);
        $194 = HEAP8[$193>>0]|0;
        $195 = $194&255;
        $196 = HEAP8[$190>>0]|0;
        $197 = $196&255;
        $198 = (208 + ($197)|0);
        $199 = HEAP8[$198>>0]|0;
        $200 = $199&255;
        $201 = (($195) - ($200))|0;
        $202 = ($201|0)!=(0);
        $203 = ($191<<24>>24)==(0);
        $or$cond$i$i$i10$i = $203 | $202;
        if ($or$cond$i$i$i10$i) {
         $$lcssa$i$i$i11$i = $201;
         break;
        } else {
         $$01315$i$i$i9$i = $190;$$016$i$i$i8$i = $189;
        }
       }
      }
      $204 = ($$lcssa$i$i$i11$i|0)==(0);
      if ($204) {
       break;
      }
      $205 = ((($$08$i$i5$i)) + 36|0);
      $$0$i$i13$i = HEAP32[$205>>2]|0;
      $206 = ($$0$i$i13$i|0)==(0|0);
      if ($206) {
       label = 60;
       break L79;
      } else {
       $$08$i$i5$i = $$0$i$i13$i;
      }
     }
     $207 = ($$08$i$i5$i|0)==(0|0);
     if ($207) {
      label = 60;
     } else {
      $208 = ((($$08$i$i5$i)) + 12|0);
      $209 = HEAP32[$208>>2]|0;
      $210 = (((21888 + (($$033$i$i$i*40)|0)|0)) + 12|0);
      HEAP32[$210>>2] = $209;
      $$sink91 = $208;
     }
    }
   } while(0);
   if ((label|0) == 60) {
    label = 0;
    $211 = (((21888 + (($$033$i$i$i*40)|0)|0)) + 12|0);
    HEAP32[$211>>2] = 0;
    $212 = (((21888 + (($$033$i$i$i*40)|0)|0)) + 36|0);
    HEAP32[$212>>2] = $174;
    $$sink91 = $172;
   }
   HEAP32[$$sink91>>2] = $162;
   $213 = (($$033$i$i$i) + 1)|0;
   $exitcond$i$i$i = ($213|0)==(15);
   if ($exitcond$i$i$i) {
    break;
   } else {
    $$033$i$i$i = $213;
   }
  }
  $$033$i$i16$i = 0;
  while(1) {
   $214 = (22496 + (($$033$i$i16$i*40)|0)|0);
   $215 = (((22496 + (($$033$i$i16$i*40)|0)|0)) + 32|0);
   $216 = HEAP32[$215>>2]|0;
   $217 = ($216|0)==(0|0);
   if ($217) {
    $$0$i$i$i17$i = 0;
   } else {
    $218 = (_strlen($216)|0);
    $219 = $218 & 1073741823;
    $$0$i$i$i17$i = $219;
   }
   $220 = HEAP8[$216>>0]|0;
   $221 = $220 << 24 >> 24;
   $222 = (($$0$i$i$i17$i) + ($221))|0;
   $223 = (($222|0) % 23)&-1;
   $224 = (59088 + ($223<<2)|0);
   $$07$i$i18$i = HEAP32[$224>>2]|0;
   $225 = ($$07$i$i18$i|0)==(0|0);
   $226 = $$07$i$i18$i;
   L100: do {
    if ($225) {
     label = 74;
    } else {
     $227 = $220&255;
     $228 = (208 + ($227)|0);
     $229 = HEAP8[$228>>0]|0;
     $230 = $229&255;
     $$08$i$i21$i = $$07$i$i18$i;
     while(1) {
      $231 = ((($$08$i$i21$i)) + 32|0);
      $232 = HEAP32[$231>>2]|0;
      $233 = HEAP8[$232>>0]|0;
      $234 = $233&255;
      $235 = (208 + ($234)|0);
      $236 = HEAP8[$235>>0]|0;
      $237 = $236&255;
      $238 = (($237) - ($230))|0;
      $239 = ($238|0)!=(0);
      $240 = ($233<<24>>24)==(0);
      $or$cond14$i$i$i22$i = $240 | $239;
      if ($or$cond14$i$i$i22$i) {
       $$lcssa$i$i$i27$i = $238;
      } else {
       $$01315$i$i$i25$i = $216;$$016$i$i$i24$i = $232;
       while(1) {
        $241 = ((($$016$i$i$i24$i)) + 1|0);
        $242 = ((($$01315$i$i$i25$i)) + 1|0);
        $243 = HEAP8[$241>>0]|0;
        $244 = $243&255;
        $245 = (208 + ($244)|0);
        $246 = HEAP8[$245>>0]|0;
        $247 = $246&255;
        $248 = HEAP8[$242>>0]|0;
        $249 = $248&255;
        $250 = (208 + ($249)|0);
        $251 = HEAP8[$250>>0]|0;
        $252 = $251&255;
        $253 = (($247) - ($252))|0;
        $254 = ($253|0)!=(0);
        $255 = ($243<<24>>24)==(0);
        $or$cond$i$i$i26$i = $255 | $254;
        if ($or$cond$i$i$i26$i) {
         $$lcssa$i$i$i27$i = $253;
         break;
        } else {
         $$01315$i$i$i25$i = $242;$$016$i$i$i24$i = $241;
        }
       }
      }
      $256 = ($$lcssa$i$i$i27$i|0)==(0);
      if ($256) {
       break;
      }
      $257 = ((($$08$i$i21$i)) + 36|0);
      $$0$i$i29$i = HEAP32[$257>>2]|0;
      $258 = ($$0$i$i29$i|0)==(0|0);
      if ($258) {
       label = 74;
       break L100;
      } else {
       $$08$i$i21$i = $$0$i$i29$i;
      }
     }
     $259 = ($$08$i$i21$i|0)==(0|0);
     if ($259) {
      label = 74;
     } else {
      $260 = ((($$08$i$i21$i)) + 12|0);
      $261 = HEAP32[$260>>2]|0;
      $262 = (((22496 + (($$033$i$i16$i*40)|0)|0)) + 12|0);
      HEAP32[$262>>2] = $261;
      $$sink92 = $260;
     }
    }
   } while(0);
   if ((label|0) == 74) {
    label = 0;
    $263 = (((22496 + (($$033$i$i16$i*40)|0)|0)) + 12|0);
    HEAP32[$263>>2] = 0;
    $264 = (((22496 + (($$033$i$i16$i*40)|0)|0)) + 36|0);
    HEAP32[$264>>2] = $226;
    $$sink92 = $224;
   }
   HEAP32[$$sink92>>2] = $214;
   $265 = (($$033$i$i16$i) + 1)|0;
   $exitcond$i$i32$i = ($265|0)==(8);
   if ($exitcond$i$i32$i) {
    break;
   } else {
    $$033$i$i16$i = $265;
   }
  }
  $$033$i$i = 0;
  while(1) {
   $266 = (19424 + (($$033$i$i*40)|0)|0);
   $267 = (((19424 + (($$033$i$i*40)|0)|0)) + 32|0);
   $268 = HEAP32[$267>>2]|0;
   $269 = ($268|0)==(0|0);
   if ($269) {
    $$0$i$i33$i = 0;
   } else {
    $270 = (_strlen($268)|0);
    $271 = $270 & 1073741823;
    $$0$i$i33$i = $271;
   }
   $272 = HEAP8[$268>>0]|0;
   $273 = $272 << 24 >> 24;
   $274 = (($$0$i$i33$i) + ($273))|0;
   $275 = (($274|0) % 23)&-1;
   $276 = (59088 + ($275<<2)|0);
   $$07$i$i = HEAP32[$276>>2]|0;
   $277 = ($$07$i$i|0)==(0|0);
   $278 = $$07$i$i;
   L121: do {
    if ($277) {
     label = 88;
    } else {
     $279 = $272&255;
     $280 = (208 + ($279)|0);
     $281 = HEAP8[$280>>0]|0;
     $282 = $281&255;
     $$08$i$i = $$07$i$i;
     while(1) {
      $283 = ((($$08$i$i)) + 32|0);
      $284 = HEAP32[$283>>2]|0;
      $285 = HEAP8[$284>>0]|0;
      $286 = $285&255;
      $287 = (208 + ($286)|0);
      $288 = HEAP8[$287>>0]|0;
      $289 = $288&255;
      $290 = (($289) - ($282))|0;
      $291 = ($290|0)!=(0);
      $292 = ($285<<24>>24)==(0);
      $or$cond14$i$i$i = $292 | $291;
      if ($or$cond14$i$i$i) {
       $$lcssa$i$i$i = $290;
      } else {
       $$01315$i$i$i = $268;$$016$i$i$i = $284;
       while(1) {
        $293 = ((($$016$i$i$i)) + 1|0);
        $294 = ((($$01315$i$i$i)) + 1|0);
        $295 = HEAP8[$293>>0]|0;
        $296 = $295&255;
        $297 = (208 + ($296)|0);
        $298 = HEAP8[$297>>0]|0;
        $299 = $298&255;
        $300 = HEAP8[$294>>0]|0;
        $301 = $300&255;
        $302 = (208 + ($301)|0);
        $303 = HEAP8[$302>>0]|0;
        $304 = $303&255;
        $305 = (($299) - ($304))|0;
        $306 = ($305|0)!=(0);
        $307 = ($295<<24>>24)==(0);
        $or$cond$i$i$i = $307 | $306;
        if ($or$cond$i$i$i) {
         $$lcssa$i$i$i = $305;
         break;
        } else {
         $$01315$i$i$i = $294;$$016$i$i$i = $293;
        }
       }
      }
      $308 = ($$lcssa$i$i$i|0)==(0);
      if ($308) {
       break;
      }
      $309 = ((($$08$i$i)) + 36|0);
      $$0$i$i = HEAP32[$309>>2]|0;
      $310 = ($$0$i$i|0)==(0|0);
      if ($310) {
       label = 88;
       break L121;
      } else {
       $$08$i$i = $$0$i$i;
      }
     }
     $311 = ($$08$i$i|0)==(0|0);
     if ($311) {
      label = 88;
     } else {
      $312 = ((($$08$i$i)) + 12|0);
      $313 = HEAP32[$312>>2]|0;
      $314 = (((19424 + (($$033$i$i*40)|0)|0)) + 12|0);
      HEAP32[$314>>2] = $313;
      $$sink93 = $312;
     }
    }
   } while(0);
   if ((label|0) == 88) {
    label = 0;
    $315 = (((19424 + (($$033$i$i*40)|0)|0)) + 12|0);
    HEAP32[$315>>2] = 0;
    $316 = (((19424 + (($$033$i$i*40)|0)|0)) + 36|0);
    HEAP32[$316>>2] = $278;
    $$sink93 = $276;
   }
   HEAP32[$$sink93>>2] = $266;
   $317 = (($$033$i$i) + 1)|0;
   $exitcond$i$i = ($317|0)==(57);
   if ($exitcond$i$i) {
    break;
   } else {
    $$033$i$i = $317;
   }
  }
  $318 = HEAP32[(29536)>>2]|0;
  $319 = ($318|0)==(0);
  if ($319) {
   $320 = HEAP32[(29416)>>2]|0;
   $321 = ($320|0)==(0|0);
   if ($321) {
    HEAP32[$vararg_buffer1>>2] = 30340;
    (_sqlite3_config(18,$vararg_buffer1)|0);
    $$pre = HEAP32[(29416)>>2]|0;
    $323 = $$pre;
   } else {
    $323 = $320;
   }
   $322 = HEAP32[(29412)>>2]|0;
   $324 = (FUNCTION_TABLE_ii[$323 & 127]($322)|0);
   $325 = ($324|0)==(0);
   if ($325) {
    label = 94;
   } else {
    $$326 = $324;
   }
  } else {
   label = 94;
  }
  L142: do {
   if ((label|0) == 94) {
    HEAP32[(29536)>>2] = 1;
    $326 = (_sqlite3_initialize()|0);
    $327 = ($326|0)==(0);
    if ($327) {
     $328 = HEAP32[7324]|0;
     $329 = ($328|0)==(0);
     do {
      if ($329) {
       $368 = HEAP32[(29340)>>2]|0;
       $369 = (FUNCTION_TABLE_ii[$368 & 127](10)|0);
       $370 = ($369|0)==(0|0);
       if ($370) {
        $$326 = 7;
        break L142;
       } else {
        $$0$i1725 = $369;
       }
      } else {
       $330 = HEAP32[(29356)>>2]|0;
       $331 = (FUNCTION_TABLE_ii[$330 & 127](10)|0);
       $332 = HEAP32[14985]|0;
       $333 = ($332>>>0)<(10);
       if ($333) {
        HEAP32[14985] = 10;
       }
       $334 = (59064);
       $335 = $334;
       $336 = HEAP32[$335>>2]|0;
       $337 = (($334) + 4)|0;
       $338 = $337;
       $339 = HEAP32[$338>>2]|0;
       $340 = ($339|0)>(0);
       $341 = ($336>>>0)>(0);
       $342 = ($339|0)==(0);
       $343 = $342 & $341;
       $344 = $340 | $343;
       if ($344) {
        $345 = HEAP32[14978]|0;
        $346 = ($331|0)<(0);
        $347 = $346 << 31 >> 31;
        $348 = (_i64Subtract(($336|0),($339|0),($331|0),($347|0))|0);
        $349 = (getTempRet0() | 0);
        $350 = ($349|0)<(0);
        $351 = ($348>>>0)<=($345>>>0);
        $352 = ($349|0)==(0);
        $353 = $352 & $351;
        $354 = $350 | $353;
        $$ = $354&1;
        HEAP32[(59072)>>2] = $$;
       }
       $355 = HEAP32[(29340)>>2]|0;
       $356 = (FUNCTION_TABLE_ii[$355 & 127]($331)|0);
       $357 = ($356|0)==(0|0);
       if ($357) {
        $$326 = 7;
        break L142;
       }
       $358 = HEAP32[(29352)>>2]|0;
       $359 = (FUNCTION_TABLE_ii[$358 & 127]($356)|0);
       $360 = HEAP32[14978]|0;
       $361 = (($360) + ($359))|0;
       HEAP32[14978] = $361;
       $362 = HEAP32[14982]|0;
       $363 = ($361>>>0)>($362>>>0);
       if ($363) {
        HEAP32[14982] = $361;
       }
       $364 = HEAP32[14981]|0;
       $365 = (($364) + 1)|0;
       HEAP32[14981] = $365;
       $366 = HEAP32[14987]|0;
       $367 = ($365>>>0)>($366>>>0);
       if (!($367)) {
        $$0$i1725 = $356;
        break;
       }
       HEAP32[14987] = $365;
       $$0$i1725 = $356;
      }
     } while(0);
     $371 = HEAP32[7324]|0;
     $372 = ($371|0)==(0);
     if ($372) {
      $380 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$380 & 127]($$0$i1725);
     } else {
      $373 = HEAP32[(29352)>>2]|0;
      $374 = (FUNCTION_TABLE_ii[$373 & 127]($$0$i1725)|0);
      $375 = HEAP32[14978]|0;
      $376 = (($375) - ($374))|0;
      HEAP32[14978] = $376;
      $377 = HEAP32[14981]|0;
      $378 = (($377) + -1)|0;
      HEAP32[14981] = $378;
      $379 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$379 & 127]($$0$i1725);
     }
     $381 = (_sqlite3_initialize()|0);
     $382 = ($381|0)==(0);
     if ($382) {
      $383 = HEAP32[14770]|0;
      $384 = ($383|0)==(464|0);
      do {
       if ($384) {
        $385 = HEAP32[(476)>>2]|0;
        HEAP32[14770] = $385;
        $386 = $385;
        $395 = $386;
       } else {
        $387 = ($383|0)==(0|0);
        if ($387) {
         $395 = 0;
         break;
        }
        $$0$i$i$i19 = $383;
        while(1) {
         $388 = ((($$0$i$i$i19)) + 12|0);
         $389 = HEAP32[$388>>2]|0;
         $390 = ($389|0)!=(0|0);
         $391 = ($389|0)!=(464|0);
         $spec$select$i$i$i = $391 & $390;
         if ($spec$select$i$i$i) {
          $$0$i$i$i19 = $389;
         } else {
          break;
         }
        }
        $392 = ($389|0)==(464|0);
        if (!($392)) {
         $395 = $383;
         break;
        }
        $393 = ((($$0$i$i$i19)) + 12|0);
        $394 = HEAP32[(476)>>2]|0;
        HEAP32[$393>>2] = $394;
        $395 = $383;
       }
      } while(0);
      HEAP32[(476)>>2] = $395;
      HEAP32[14770] = 464;
     }
     $396 = (_sqlite3_initialize()|0);
     $397 = ($396|0)==(0);
     if ($397) {
      $420 = HEAP32[14770]|0;
      $421 = ($420|0)==((552)|0);
      do {
       if ($421) {
        $430 = HEAP32[(564)>>2]|0;
        HEAP32[14770] = $430;
        $431 = $430;
        $433 = $431;
        label = 138;
       } else {
        $422 = ($420|0)==(0|0);
        if ($422) {
         label = 140;
         break;
        }
        $$0$i$i$i19$1 = $420;
        while(1) {
         $423 = ((($$0$i$i$i19$1)) + 12|0);
         $424 = HEAP32[$423>>2]|0;
         $425 = ($424|0)!=(0|0);
         $426 = ($424|0)!=((552)|0);
         $spec$select$i$i$i$1 = $426 & $425;
         if ($spec$select$i$i$i$1) {
          $$0$i$i$i19$1 = $424;
         } else {
          break;
         }
        }
        $427 = ($424|0)==((552)|0);
        if (!($427)) {
         $433 = $420;
         label = 138;
         break;
        }
        $428 = ((($$0$i$i$i19$1)) + 12|0);
        $429 = HEAP32[(564)>>2]|0;
        HEAP32[$428>>2] = $429;
        $433 = $420;
        label = 138;
       }
      } while(0);
      do {
       if ((label|0) == 138) {
        $432 = ($433|0)==(0|0);
        if ($432) {
         label = 140;
         break;
        }
        $434 = ((($433)) + 12|0);
        $435 = HEAP32[$434>>2]|0;
        HEAP32[(564)>>2] = $435;
        $vfsList$sink = $434;
       }
      } while(0);
      if ((label|0) == 140) {
       HEAP32[(564)>>2] = 0;
       $vfsList$sink = 59080;
      }
      HEAP32[$vfsList$sink>>2] = (552);
     }
     $436 = (_sqlite3_initialize()|0);
     $437 = ($436|0)==(0);
     if ($437) {
      $438 = HEAP32[14770]|0;
      $439 = ($438|0)==((640)|0);
      do {
       if ($439) {
        $448 = HEAP32[(652)>>2]|0;
        HEAP32[14770] = $448;
        $449 = $448;
        $451 = $449;
        label = 150;
       } else {
        $440 = ($438|0)==(0|0);
        if ($440) {
         label = 152;
         break;
        }
        $$0$i$i$i19$2 = $438;
        while(1) {
         $441 = ((($$0$i$i$i19$2)) + 12|0);
         $442 = HEAP32[$441>>2]|0;
         $443 = ($442|0)!=(0|0);
         $444 = ($442|0)!=((640)|0);
         $spec$select$i$i$i$2 = $444 & $443;
         if ($spec$select$i$i$i$2) {
          $$0$i$i$i19$2 = $442;
         } else {
          break;
         }
        }
        $445 = ($442|0)==((640)|0);
        if (!($445)) {
         $451 = $438;
         label = 150;
         break;
        }
        $446 = ((($$0$i$i$i19$2)) + 12|0);
        $447 = HEAP32[(652)>>2]|0;
        HEAP32[$446>>2] = $447;
        $451 = $438;
        label = 150;
       }
      } while(0);
      do {
       if ((label|0) == 150) {
        $450 = ($451|0)==(0|0);
        if ($450) {
         label = 152;
         break;
        }
        $452 = ((($451)) + 12|0);
        $453 = HEAP32[$452>>2]|0;
        HEAP32[(652)>>2] = $453;
        $vfsList$sink95 = $452;
       }
      } while(0);
      if ((label|0) == 152) {
       HEAP32[(652)>>2] = 0;
       $vfsList$sink95 = 59080;
      }
      HEAP32[$vfsList$sink95>>2] = (640);
     }
     $454 = (_sqlite3_initialize()|0);
     $455 = ($454|0)==(0);
     if ($455) {
      $456 = HEAP32[14770]|0;
      $457 = ($456|0)==((728)|0);
      do {
       if ($457) {
        $466 = HEAP32[(740)>>2]|0;
        HEAP32[14770] = $466;
        $467 = $466;
        $469 = $467;
        label = 162;
       } else {
        $458 = ($456|0)==(0|0);
        if ($458) {
         label = 164;
         break;
        }
        $$0$i$i$i19$3 = $456;
        while(1) {
         $459 = ((($$0$i$i$i19$3)) + 12|0);
         $460 = HEAP32[$459>>2]|0;
         $461 = ($460|0)!=(0|0);
         $462 = ($460|0)!=((728)|0);
         $spec$select$i$i$i$3 = $462 & $461;
         if ($spec$select$i$i$i$3) {
          $$0$i$i$i19$3 = $460;
         } else {
          break;
         }
        }
        $463 = ($460|0)==((728)|0);
        if (!($463)) {
         $469 = $456;
         label = 162;
         break;
        }
        $464 = ((($$0$i$i$i19$3)) + 12|0);
        $465 = HEAP32[(740)>>2]|0;
        HEAP32[$464>>2] = $465;
        $469 = $456;
        label = 162;
       }
      } while(0);
      do {
       if ((label|0) == 162) {
        $468 = ($469|0)==(0|0);
        if ($468) {
         label = 164;
         break;
        }
        $470 = ((($469)) + 12|0);
        $471 = HEAP32[$470>>2]|0;
        HEAP32[(740)>>2] = $471;
        $vfsList$sink96 = $470;
       }
      } while(0);
      if ((label|0) == 164) {
       HEAP32[(740)>>2] = 0;
       $vfsList$sink96 = 59080;
      }
      HEAP32[$vfsList$sink96>>2] = (728);
     }
     $401 = HEAP32[(29496)>>2]|0;
     $472 = HEAP32[(59240)>>2]|0;
     $473 = ($472|0)==(0);
     if (!($473)) {
      $398 = HEAP32[(29504)>>2]|0;
      $399 = HEAP32[(29500)>>2]|0;
      $400 = ($401|0)==(0|0);
      $spec$select17$i = $400 ? 0 : $398;
      $402 = ($spec$select17$i|0)==(0);
      $403 = $400 | $402;
      $$op$i = $399 & -8;
      $404 = $403 ? 0 : $$op$i;
      HEAP32[(59252)>>2] = $404;
      HEAP32[(59280)>>2] = $spec$select17$i;
      HEAP32[(59256)>>2] = $spec$select17$i;
      $405 = ($spec$select17$i|0)>(90);
      if ($405) {
       $408 = 10;
      } else {
       $406 = (($spec$select17$i|0) / 10)&-1;
       $407 = (($406) + 1)|0;
       $408 = $407;
      }
      HEAP32[(59260)>>2] = $408;
      HEAP32[(59264)>>2] = $401;
      HEAP32[(59276)>>2] = 0;
      HEAP32[(59284)>>2] = 0;
      if ($402) {
       $$0$lcssa$i = $401;
      } else {
       $409 = (($spec$select17$i) + -1)|0;
       $410 = Math_imul($404, $409)|0;
       $$019$i = $401;$$118$i = $spec$select17$i;$412 = 0;
       while(1) {
        $411 = (($$118$i) + -1)|0;
        HEAP32[$$019$i>>2] = $412;
        $413 = (($$019$i) + ($404)|0);
        $414 = ($411|0)==(0);
        $415 = $$019$i;
        if ($414) {
         break;
        } else {
         $$019$i = $413;$$118$i = $411;$412 = $415;
        }
       }
       $416 = Math_imul($404, $spec$select17$i)|0;
       $scevgep = (($401) + ($410)|0);
       HEAP32[(59276)>>2] = $scevgep;
       $scevgep$i = (($401) + ($416)|0);
       $$0$lcssa$i = $scevgep$i;
      }
      HEAP32[(59268)>>2] = $$0$lcssa$i;
     }
     HEAP32[(29520)>>2] = 1;
     $$326 = 0;
    } else {
     $$326 = 7;
    }
   }
  } while(0);
  HEAP32[(29524)>>2] = 0;
  $$pr = HEAP32[(29540)>>2]|0;
  $$4 = $$326;$418 = $$pr;
 } else {
  $$4 = 0;$418 = $19;
 }
 $417 = (($418) + -1)|0;
 HEAP32[(29540)>>2] = $417;
 $419 = ($418|0)<(2);
 if (!($419)) {
  $$014 = $$4;
  STACKTOP = sp;return ($$014|0);
 }
 HEAP32[(29544)>>2] = 0;
 $$014 = $$4;
 STACKTOP = sp;return ($$014|0);
}
function _sqlite3_malloc($0) {
 $0 = $0|0;
 var $$ = 0, $$0 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0;
 var $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = (_sqlite3_initialize()|0);
 $2 = ($1|0)!=(0);
 $3 = ($0|0)<(1);
 $or$cond = $3 | $2;
 if ($or$cond) {
  $$0 = 0;
  return ($$0|0);
 }
 $4 = ($0|0)<(0);
 $5 = $4 << 31 >> 31;
 $6 = (_i64Add(($0|0),($5|0),-1,-1)|0);
 $7 = (getTempRet0() | 0);
 $8 = ($7>>>0)>(0);
 $9 = ($6>>>0)>(2147483390);
 $10 = ($7|0)==(0);
 $11 = $10 & $9;
 $12 = $8 | $11;
 if ($12) {
  $$0 = 0;
  return ($$0|0);
 }
 $13 = HEAP32[7324]|0;
 $14 = ($13|0)==(0);
 if ($14) {
  $53 = HEAP32[(29340)>>2]|0;
  $54 = (FUNCTION_TABLE_ii[$53 & 127]($0)|0);
  $$0 = $54;
  return ($$0|0);
 }
 $15 = HEAP32[(29356)>>2]|0;
 $16 = (FUNCTION_TABLE_ii[$15 & 127]($0)|0);
 $17 = HEAP32[14985]|0;
 $18 = ($17>>>0)<($0>>>0);
 if ($18) {
  HEAP32[14985] = $0;
 }
 $19 = (59064);
 $20 = $19;
 $21 = HEAP32[$20>>2]|0;
 $22 = (($19) + 4)|0;
 $23 = $22;
 $24 = HEAP32[$23>>2]|0;
 $25 = ($24|0)>(0);
 $26 = ($21>>>0)>(0);
 $27 = ($24|0)==(0);
 $28 = $27 & $26;
 $29 = $25 | $28;
 if ($29) {
  $30 = HEAP32[14978]|0;
  $31 = ($16|0)<(0);
  $32 = $31 << 31 >> 31;
  $33 = (_i64Subtract(($21|0),($24|0),($16|0),($32|0))|0);
  $34 = (getTempRet0() | 0);
  $35 = ($34|0)<(0);
  $36 = ($33>>>0)<=($30>>>0);
  $37 = ($34|0)==(0);
  $38 = $37 & $36;
  $39 = $35 | $38;
  $$ = $39&1;
  HEAP32[(59072)>>2] = $$;
 }
 $40 = HEAP32[(29340)>>2]|0;
 $41 = (FUNCTION_TABLE_ii[$40 & 127]($16)|0);
 $42 = ($41|0)==(0|0);
 if ($42) {
  $$0 = 0;
  return ($$0|0);
 }
 $43 = HEAP32[(29352)>>2]|0;
 $44 = (FUNCTION_TABLE_ii[$43 & 127]($41)|0);
 $45 = HEAP32[14978]|0;
 $46 = (($45) + ($44))|0;
 HEAP32[14978] = $46;
 $47 = HEAP32[14982]|0;
 $48 = ($46>>>0)>($47>>>0);
 if ($48) {
  HEAP32[14982] = $46;
 }
 $49 = HEAP32[14981]|0;
 $50 = (($49) + 1)|0;
 HEAP32[14981] = $50;
 $51 = HEAP32[14987]|0;
 $52 = ($50>>>0)>($51>>>0);
 if (!($52)) {
  $$0 = $41;
  return ($$0|0);
 }
 HEAP32[14987] = $50;
 $$0 = $41;
 return ($$0|0);
}
function _sqlite3Malloc($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$ = 0, $$0 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0;
 var $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = (_i64Add(($0|0),($1|0),-1,-1)|0);
 $3 = (getTempRet0() | 0);
 $4 = ($3>>>0)>(0);
 $5 = ($2>>>0)>(2147483390);
 $6 = ($3|0)==(0);
 $7 = $6 & $5;
 $8 = $4 | $7;
 if ($8) {
  $$0 = 0;
  return ($$0|0);
 }
 $9 = HEAP32[7324]|0;
 $10 = ($9|0)==(0);
 if ($10) {
  $49 = HEAP32[(29340)>>2]|0;
  $50 = (FUNCTION_TABLE_ii[$49 & 127]($0)|0);
  $$0 = $50;
  return ($$0|0);
 }
 $11 = HEAP32[(29356)>>2]|0;
 $12 = (FUNCTION_TABLE_ii[$11 & 127]($0)|0);
 $13 = HEAP32[14985]|0;
 $14 = ($13>>>0)<($0>>>0);
 if ($14) {
  HEAP32[14985] = $0;
 }
 $15 = (59064);
 $16 = $15;
 $17 = HEAP32[$16>>2]|0;
 $18 = (($15) + 4)|0;
 $19 = $18;
 $20 = HEAP32[$19>>2]|0;
 $21 = ($20|0)>(0);
 $22 = ($17>>>0)>(0);
 $23 = ($20|0)==(0);
 $24 = $23 & $22;
 $25 = $21 | $24;
 if ($25) {
  $26 = HEAP32[14978]|0;
  $27 = ($12|0)<(0);
  $28 = $27 << 31 >> 31;
  $29 = (_i64Subtract(($17|0),($20|0),($12|0),($28|0))|0);
  $30 = (getTempRet0() | 0);
  $31 = ($30|0)<(0);
  $32 = ($29>>>0)<=($26>>>0);
  $33 = ($30|0)==(0);
  $34 = $33 & $32;
  $35 = $31 | $34;
  $$ = $35&1;
  HEAP32[(59072)>>2] = $$;
 }
 $36 = HEAP32[(29340)>>2]|0;
 $37 = (FUNCTION_TABLE_ii[$36 & 127]($12)|0);
 $38 = ($37|0)==(0|0);
 if ($38) {
  $$0 = 0;
  return ($$0|0);
 }
 $39 = HEAP32[(29352)>>2]|0;
 $40 = (FUNCTION_TABLE_ii[$39 & 127]($37)|0);
 $41 = HEAP32[14978]|0;
 $42 = (($41) + ($40))|0;
 HEAP32[14978] = $42;
 $43 = HEAP32[14982]|0;
 $44 = ($42>>>0)>($43>>>0);
 if ($44) {
  HEAP32[14982] = $42;
 }
 $45 = HEAP32[14981]|0;
 $46 = (($45) + 1)|0;
 HEAP32[14981] = $46;
 $47 = HEAP32[14987]|0;
 $48 = ($46>>>0)>($47>>>0);
 if (!($48)) {
  $$0 = $37;
  return ($$0|0);
 }
 HEAP32[14987] = $46;
 $$0 = $37;
 return ($$0|0);
}
function _sqlite3_malloc64($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$ = 0, $$0 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0;
 var $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = (_sqlite3_initialize()|0);
 $3 = ($2|0)!=(0);
 $4 = (_i64Add(($0|0),($1|0),-1,-1)|0);
 $5 = (getTempRet0() | 0);
 $6 = ($5>>>0)>(0);
 $7 = ($4>>>0)>(2147483390);
 $8 = ($5|0)==(0);
 $9 = $8 & $7;
 $10 = $6 | $9;
 $or$cond = $10 | $3;
 if ($or$cond) {
  $$0 = 0;
  return ($$0|0);
 }
 $11 = HEAP32[7324]|0;
 $12 = ($11|0)==(0);
 if ($12) {
  $51 = HEAP32[(29340)>>2]|0;
  $52 = (FUNCTION_TABLE_ii[$51 & 127]($0)|0);
  $$0 = $52;
  return ($$0|0);
 }
 $13 = HEAP32[(29356)>>2]|0;
 $14 = (FUNCTION_TABLE_ii[$13 & 127]($0)|0);
 $15 = HEAP32[14985]|0;
 $16 = ($15>>>0)<($0>>>0);
 if ($16) {
  HEAP32[14985] = $0;
 }
 $17 = (59064);
 $18 = $17;
 $19 = HEAP32[$18>>2]|0;
 $20 = (($17) + 4)|0;
 $21 = $20;
 $22 = HEAP32[$21>>2]|0;
 $23 = ($22|0)>(0);
 $24 = ($19>>>0)>(0);
 $25 = ($22|0)==(0);
 $26 = $25 & $24;
 $27 = $23 | $26;
 if ($27) {
  $28 = HEAP32[14978]|0;
  $29 = ($14|0)<(0);
  $30 = $29 << 31 >> 31;
  $31 = (_i64Subtract(($19|0),($22|0),($14|0),($30|0))|0);
  $32 = (getTempRet0() | 0);
  $33 = ($32|0)<(0);
  $34 = ($31>>>0)<=($28>>>0);
  $35 = ($32|0)==(0);
  $36 = $35 & $34;
  $37 = $33 | $36;
  $$ = $37&1;
  HEAP32[(59072)>>2] = $$;
 }
 $38 = HEAP32[(29340)>>2]|0;
 $39 = (FUNCTION_TABLE_ii[$38 & 127]($14)|0);
 $40 = ($39|0)==(0|0);
 if ($40) {
  $$0 = 0;
  return ($$0|0);
 }
 $41 = HEAP32[(29352)>>2]|0;
 $42 = (FUNCTION_TABLE_ii[$41 & 127]($39)|0);
 $43 = HEAP32[14978]|0;
 $44 = (($43) + ($42))|0;
 HEAP32[14978] = $44;
 $45 = HEAP32[14982]|0;
 $46 = ($44>>>0)>($45>>>0);
 if ($46) {
  HEAP32[14982] = $44;
 }
 $47 = HEAP32[14981]|0;
 $48 = (($47) + 1)|0;
 HEAP32[14981] = $48;
 $49 = HEAP32[14987]|0;
 $50 = ($48>>>0)>($49>>>0);
 if (!($50)) {
  $$0 = $39;
  return ($$0|0);
 }
 HEAP32[14987] = $48;
 $$0 = $39;
 return ($$0|0);
}
function _sqlite3_free($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0, $11 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  return;
 }
 $2 = HEAP32[7324]|0;
 $3 = ($2|0)==(0);
 if ($3) {
  $11 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$11 & 127]($0);
  return;
 } else {
  $4 = HEAP32[(29352)>>2]|0;
  $5 = (FUNCTION_TABLE_ii[$4 & 127]($0)|0);
  $6 = HEAP32[14978]|0;
  $7 = (($6) - ($5))|0;
  HEAP32[14978] = $7;
  $8 = HEAP32[14981]|0;
  $9 = (($8) + -1)|0;
  HEAP32[14981] = $9;
  $10 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$10 & 127]($0);
  return;
 }
}
function _sqlite3MallocSize($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = HEAP32[(29352)>>2]|0;
 $2 = (FUNCTION_TABLE_ii[$1 & 127]($0)|0);
 return ($2|0);
}
function _sqlite3Realloc($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$ = 0, $$0 = 0, $$036 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0;
 var $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0;
 var $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0;
 var $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0;
 var $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0;
 var $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $or$cond = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = ($0|0)==(0|0);
 if ($3) {
  $4 = (_i64Add(($1|0),($2|0),-1,-1)|0);
  $5 = (getTempRet0() | 0);
  $6 = ($5>>>0)>(0);
  $7 = ($4>>>0)>(2147483390);
  $8 = ($5|0)==(0);
  $9 = $8 & $7;
  $10 = $6 | $9;
  if ($10) {
   $$036 = 0;
   return ($$036|0);
  }
  $11 = HEAP32[7324]|0;
  $12 = ($11|0)==(0);
  if ($12) {
   $51 = HEAP32[(29340)>>2]|0;
   $52 = (FUNCTION_TABLE_ii[$51 & 127]($1)|0);
   $$036 = $52;
   return ($$036|0);
  }
  $13 = HEAP32[(29356)>>2]|0;
  $14 = (FUNCTION_TABLE_ii[$13 & 127]($1)|0);
  $15 = HEAP32[14985]|0;
  $16 = ($15>>>0)<($1>>>0);
  if ($16) {
   HEAP32[14985] = $1;
  }
  $17 = (59064);
  $18 = $17;
  $19 = HEAP32[$18>>2]|0;
  $20 = (($17) + 4)|0;
  $21 = $20;
  $22 = HEAP32[$21>>2]|0;
  $23 = ($22|0)>(0);
  $24 = ($19>>>0)>(0);
  $25 = ($22|0)==(0);
  $26 = $25 & $24;
  $27 = $23 | $26;
  if ($27) {
   $28 = HEAP32[14978]|0;
   $29 = ($14|0)<(0);
   $30 = $29 << 31 >> 31;
   $31 = (_i64Subtract(($19|0),($22|0),($14|0),($30|0))|0);
   $32 = (getTempRet0() | 0);
   $33 = ($32|0)<(0);
   $34 = ($31>>>0)<=($28>>>0);
   $35 = ($32|0)==(0);
   $36 = $35 & $34;
   $37 = $33 | $36;
   $$ = $37&1;
   HEAP32[(59072)>>2] = $$;
  }
  $38 = HEAP32[(29340)>>2]|0;
  $39 = (FUNCTION_TABLE_ii[$38 & 127]($14)|0);
  $40 = ($39|0)==(0|0);
  if ($40) {
   $$036 = 0;
   return ($$036|0);
  }
  $41 = HEAP32[(29352)>>2]|0;
  $42 = (FUNCTION_TABLE_ii[$41 & 127]($39)|0);
  $43 = HEAP32[14978]|0;
  $44 = (($43) + ($42))|0;
  HEAP32[14978] = $44;
  $45 = HEAP32[14982]|0;
  $46 = ($44>>>0)>($45>>>0);
  if ($46) {
   HEAP32[14982] = $44;
  }
  $47 = HEAP32[14981]|0;
  $48 = (($47) + 1)|0;
  HEAP32[14981] = $48;
  $49 = HEAP32[14987]|0;
  $50 = ($48>>>0)>($49>>>0);
  if (!($50)) {
   $$036 = $39;
   return ($$036|0);
  }
  HEAP32[14987] = $48;
  $$036 = $39;
  return ($$036|0);
 }
 $53 = ($1|0)==(0);
 $54 = ($2|0)==(0);
 $55 = $53 & $54;
 if ($55) {
  $56 = HEAP32[7324]|0;
  $57 = ($56|0)==(0);
  if ($57) {
   $65 = HEAP32[(29344)>>2]|0;
   FUNCTION_TABLE_vi[$65 & 127]($0);
   $$036 = 0;
   return ($$036|0);
  } else {
   $58 = HEAP32[(29352)>>2]|0;
   $59 = (FUNCTION_TABLE_ii[$58 & 127]($0)|0);
   $60 = HEAP32[14978]|0;
   $61 = (($60) - ($59))|0;
   HEAP32[14978] = $61;
   $62 = HEAP32[14981]|0;
   $63 = (($62) + -1)|0;
   HEAP32[14981] = $63;
   $64 = HEAP32[(29344)>>2]|0;
   FUNCTION_TABLE_vi[$64 & 127]($0);
   $$036 = 0;
   return ($$036|0);
  }
 }
 $66 = ($2>>>0)>(0);
 $67 = ($1>>>0)>(2147483391);
 $68 = ($2|0)==(0);
 $69 = $68 & $67;
 $70 = $66 | $69;
 if ($70) {
  $$036 = 0;
  return ($$036|0);
 }
 $71 = HEAP32[(29352)>>2]|0;
 $72 = (FUNCTION_TABLE_ii[$71 & 127]($0)|0);
 $73 = HEAP32[(29356)>>2]|0;
 $74 = (FUNCTION_TABLE_ii[$73 & 127]($1)|0);
 $75 = ($74|0)==($72|0);
 if ($75) {
  $$036 = $0;
  return ($$036|0);
 }
 $76 = HEAP32[7324]|0;
 $77 = ($76|0)==(0);
 if ($77) {
  $104 = HEAP32[(29348)>>2]|0;
  $105 = (FUNCTION_TABLE_iii[$104 & 255]($0,$74)|0);
  $$036 = $105;
  return ($$036|0);
 }
 $78 = HEAP32[14985]|0;
 $79 = ($78>>>0)<($1>>>0);
 if ($79) {
  HEAP32[14985] = $1;
 }
 $80 = HEAP32[(29348)>>2]|0;
 $81 = (FUNCTION_TABLE_iii[$80 & 255]($0,$74)|0);
 $82 = ($81|0)==(0|0);
 $83 = (59064);
 $84 = $83;
 $85 = HEAP32[$84>>2]|0;
 $86 = (($83) + 4)|0;
 $87 = $86;
 $88 = HEAP32[$87>>2]|0;
 $89 = ($88|0)>(0);
 $90 = ($85>>>0)>(0);
 $91 = ($88|0)==(0);
 $92 = $91 & $90;
 $93 = $89 | $92;
 $or$cond = $82 & $93;
 if ($or$cond) {
  $94 = HEAP32[(29348)>>2]|0;
  $95 = (FUNCTION_TABLE_iii[$94 & 255]($0,$74)|0);
  $$0 = $95;
 } else {
  $$0 = $81;
 }
 $96 = ($$0|0)==(0|0);
 if ($96) {
  $$036 = 0;
  return ($$036|0);
 }
 $97 = HEAP32[(29352)>>2]|0;
 $98 = (FUNCTION_TABLE_ii[$97 & 127]($$0)|0);
 $99 = (($98) - ($72))|0;
 $100 = HEAP32[14978]|0;
 $101 = (($100) + ($99))|0;
 HEAP32[14978] = $101;
 $102 = HEAP32[14982]|0;
 $103 = ($101>>>0)>($102>>>0);
 if (!($103)) {
  $$036 = $$0;
  return ($$036|0);
 }
 HEAP32[14982] = $101;
 $$036 = $$0;
 return ($$036|0);
}
function _sqlite3_str_vappendf($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$ = 0, $$$6713 = 0, $$0 = 0, $$0$$0$$0$$i = 0.0, $$0$$0$$0$2$i = 0.0, $$0$$0$$0$3$i = 0.0, $$0$i$i787 = 0, $$0$i741 = 0, $$0$i742 = 0, $$0$i743 = 0, $$0$i748 = 0, $$0$i778 = 0, $$0$i798 = 0, $$0$i813 = 0, $$0$i829 = 0, $$0561 = 0, $$0567 = 0, $$0578 = 0, $$06031059$lcssa = 0, $$0607 = 0;
 var $$0619 = 0, $$0622 = 0, $$0624 = 0, $$0626 = 0, $$0629 = 0, $$0632$lcssa = 0, $$06321062 = 0, $$06321062$us = 0, $$06341061 = 0, $$06341061$us = 0, $$0635 = 0, $$0639$lcssa = 0, $$06391060 = 0, $$06391060$us = 0, $$06431097 = 0, $$06431097$in = 0, $$0644$lcssa = 0, $$06441075 = 0, $$0647 = 0, $$0657 = 0;
 var $$06611142 = 0, $$0666 = 0.0, $$0669 = 0, $$06731162 = 0, $$06741158 = 0, $$06761157 = 0, $$0684 = 0, $$0692 = 0, $$0693 = 0, $$0707 = 0, $$0720$lcssa = 0.0, $$07201101 = 0.0, $$0722$lcssa = 0, $$0722997 = 0, $$0724 = 0, $$0725 = 0, $$0725$lcssa = 0, $$07251003 = 0, $$07251005 = 0, $$08731122 = 0;
 var $$0877 = 0.0, $$1 = 0, $$1$i1090 = 0, $$1$i1090$ph = 0, $$1$i7901170 = 0, $$1$i7901170$ph = 0, $$1$i7951167 = 0, $$1$i7951167$ph = 0, $$10 = 0, $$105881138 = 0, $$10886 = 0.0, $$11$ph = 0, $$115891093 = 0, $$11602 = 0, $$11618 = 0, $$12 = 0, $$131124 = 0, $$14 = 0, $$15 = 0, $$1568 = 0;
 var $$1579 = 0, $$1592$lcssa = 0, $$16$lcssa = 0, $$16041159 = 0, $$1608 = 0, $$160812401242$ph = 0, $$161130 = 0, $$1623 = 0, $$1625898 = 0, $$1627897 = 0, $$1630895 = 0, $$1636$lcssa = 0, $$16361070 = 0, $$1640$us = 0, $$1645 = 0, $$1648892 = 0, $$1658888 = 0, $$1667 = 0.0, $$1671 = 0, $$1671916 = 0;
 var $$1671917 = 0, $$1675 = 0, $$1677 = 0, $$1694 = 0, $$1698 = 0, $$17$lcssa = 0, $$1702 = 0, $$1703 = 0, $$1708 = 0, $$171139 = 0, $$17171123 = 0, $$18 = 0, $$1874 = 0, $$19 = 0, $$2 = 0, $$20 = 0, $$21 = 0, $$221089 = 0, $$221089$1 = 0, $$221089$2 = 0;
 var $$23 = 0, $$24 = 0, $$24934 = 0, $$25 = 0, $$2563$lcssa = 0, $$2563$off = 0, $$2563$off996 = 0, $$2563998 = 0, $$2569 = 0, $$26 = 0, $$2621900 = 0, $$2621901 = 0, $$2631 = 0, $$2637 = 0, $$2646 = 0, $$2651890 = 0, $$2659 = 0, $$2665 = 0, $$2668 = 0.0, $$2672 = 0;
 var $$2695 = 0, $$2709 = 0, $$2718 = 0, $$28 = 0, $$2875 = 0, $$2879$lcssa = 0.0, $$28791107 = 0.0, $$3 = 0, $$3564 = 0, $$3570 = 0, $$3581 = 0, $$36061102 = 0, $$3610902 = 0, $$36109021253 = 0, $$3610903 = 0, $$3638 = 0, $$36421069 = 0, $$3652 = 0, $$3696 = 0, $$3700$ph = 0;
 var $$3710$lcssa = 0, $$37101108 = 0, $$3719 = 0, $$3719$in1128 = 0, $$38761137 = 0, $$3880$lcssa = 0.0, $$38801114 = 0.0, $$4 = 0, $$4571$lcssa = 0, $$4582904 = 0, $$4582905 = 0, $$4611 = 0, $$4653 = 0, $$4701 = 0, $$4701933 = 0, $$4711$lcssa = 0, $$47111115 = 0, $$4881 = 0.0, $$5$lcssa = 0, $$5566 = 0;
 var $$5572 = 0, $$5596 = 0, $$5702 = 0, $$5712 = 0, $$5882 = 0.0, $$61121 = 0.0, $$6573 = 0, $$65971081 = 0, $$6613 = 0, $$6703 = 0, $$6713 = 0, $$7$lcssa = 0, $$75741163 = 0, $$7598 = 0, $$76141096 = 0, $$7714 = 0, $$7883 = 0.0, $$7999 = 0, $$8 = 0, $$8575 = 0;
 var $$8705 = 0, $$8715 = 0, $$8884 = 0.0, $$9587$lcssa = 0, $$95871129 = 0, $$98851136 = 0.0, $$9906 = 0, $$idx$i$i$i = 0, $$idx$i$i$i750 = 0, $$idx$i$i$i753 = 0, $$idx$i$i$i761 = 0, $$idx$i$i$i769 = 0, $$idx$val$i$i$i = 0, $$idx$val$i$i$i751 = 0, $$idx$val$i$i$i754 = 0, $$idx$val$i$i$i762 = 0, $$idx$val$i$i$i770 = 0, $$idx5$i$i$i = 0, $$idx5$val$i$i$i = 0, $$idx6$i$i$i = 0;
 var $$idx6$val$i$i$i = 0, $$idx8$i$i$i = 0, $$idx8$i$i$i755 = 0, $$idx8$i$i$i763 = 0, $$idx8$i$i$i771 = 0, $$idx8$val$i$i$i = 0, $$idx8$val$i$i$i756 = 0, $$idx8$val$i$i$i764 = 0, $$idx8$val$i$i$i772 = 0, $$idx9$i$i$i = 0, $$idx9$i$i$i757 = 0, $$idx9$i$i$i765 = 0, $$idx9$i$i$i773 = 0, $$idx9$val$i$i$i = 0, $$idx9$val$i$i$i758 = 0, $$idx9$val$i$i$i766 = 0, $$idx9$val$i$i$i774 = 0, $$in = 0, $$lcssa961 = 0, $$lcssa962 = 0;
 var $$lcssa963 = 0, $$mux1594 = 0, $$mux1596 = 0, $$neg = 0, $$off = 0, $$off1004 = 0, $$pre = 0, $$pre$phi1237Z2D = 0, $$pre$phi1239Z2D = 0, $$pre$phiZ2D = 0, $$pre1238 = 0, $$sink = 0, $$sink1692 = 0, $$sink1699 = 0, $10 = 0, $100 = 0, $1000 = 0, $1001 = 0, $1002 = 0, $1003 = 0;
 var $1004 = 0, $1005 = 0, $1006 = 0, $1007 = 0, $1008 = 0, $1009 = 0, $101 = 0, $1010 = 0, $1011 = 0, $1012 = 0, $1013 = 0, $1014 = 0, $1015 = 0, $1016 = 0, $1017 = 0, $1018 = 0, $1019 = 0, $102 = 0, $1020 = 0, $1021 = 0;
 var $1022 = 0, $1023 = 0, $1024 = 0, $1025 = 0, $1026 = 0, $1027 = 0, $1028 = 0, $1029 = 0, $103 = 0, $1030 = 0, $1031 = 0, $1032 = 0, $1033 = 0, $1034 = 0, $1035 = 0, $1036 = 0, $1037 = 0, $1038 = 0, $1039 = 0, $104 = 0;
 var $1040 = 0, $1041 = 0, $1042 = 0, $1043 = 0, $1044 = 0, $1045 = 0, $1046 = 0, $1047 = 0, $1048 = 0, $1049 = 0, $105 = 0, $1050 = 0, $1051 = 0, $1052 = 0, $1053 = 0, $1054 = 0, $1055 = 0, $1056 = 0, $1057 = 0, $1058 = 0;
 var $1059 = 0, $106 = 0, $1060 = 0, $1061 = 0, $1062 = 0, $1063 = 0, $1064 = 0, $1065 = 0, $1066 = 0, $1067 = 0, $1068 = 0, $1069 = 0, $107 = 0, $1070 = 0, $1071 = 0, $1072 = 0, $1073 = 0, $1074 = 0, $1075 = 0, $1076 = 0;
 var $1077 = 0, $1078 = 0, $1079 = 0, $108 = 0, $1080 = 0, $1081 = 0, $1082 = 0, $1083 = 0, $1084 = 0, $1085 = 0, $1086 = 0, $1087 = 0, $1088 = 0, $1089 = 0, $109 = 0, $1090 = 0, $1091 = 0, $1092 = 0, $1093 = 0, $1094 = 0;
 var $1095 = 0, $1096 = 0, $1097 = 0, $1098 = 0, $1099 = 0, $11 = 0, $110 = 0, $1100 = 0, $1101 = 0, $1102 = 0, $1103 = 0, $1104 = 0, $1105 = 0, $1106 = 0, $1107 = 0, $1108 = 0, $1109 = 0, $111 = 0, $1110 = 0, $1111 = 0;
 var $1112 = 0, $1113 = 0, $1114 = 0, $1115 = 0, $1116 = 0, $1117 = 0, $1118 = 0, $1119 = 0, $112 = 0, $1120 = 0, $1121 = 0, $1122 = 0, $1123 = 0, $1124 = 0, $1125 = 0, $1126 = 0, $1127 = 0, $1128 = 0, $1129 = 0, $113 = 0;
 var $1130 = 0, $1131 = 0, $1132 = 0, $1133 = 0, $1134 = 0, $1135 = 0, $1136 = 0, $1137 = 0, $1138 = 0, $1139 = 0, $114 = 0, $1140 = 0, $1141 = 0, $1142 = 0, $1143 = 0, $1144 = 0, $1145 = 0, $1146 = 0, $1147 = 0, $1148 = 0;
 var $1149 = 0, $115 = 0, $1150 = 0, $1151 = 0, $1152 = 0, $1153 = 0, $1154 = 0, $1155 = 0, $1156 = 0, $1157 = 0, $1158 = 0, $1159 = 0, $116 = 0, $1160 = 0, $1161 = 0, $1162 = 0, $1163 = 0, $1164 = 0, $1165 = 0, $1166 = 0;
 var $1167 = 0, $1168 = 0, $1169 = 0, $117 = 0, $1170 = 0, $1171 = 0, $1172 = 0, $1173 = 0, $1174 = 0, $1175 = 0, $1176 = 0, $1177 = 0, $1178 = 0, $1179 = 0, $118 = 0, $1180 = 0, $1181 = 0, $1182 = 0, $1183 = 0, $1184 = 0;
 var $1185 = 0, $1186 = 0, $1187 = 0, $1188 = 0, $1189 = 0, $119 = 0, $1190 = 0, $1191 = 0, $1192 = 0, $1193 = 0, $1194 = 0, $1195 = 0, $1196 = 0, $1197 = 0, $1198 = 0, $1199 = 0, $12 = 0, $120 = 0, $1200 = 0, $1201 = 0;
 var $1202 = 0, $1203 = 0, $1204 = 0, $1205 = 0, $1206 = 0, $1207 = 0, $1208 = 0, $1209 = 0, $121 = 0, $1210 = 0, $1211 = 0, $1212 = 0, $1213 = 0, $1214 = 0, $1215 = 0, $1216 = 0, $1217 = 0, $1218 = 0, $1219 = 0, $122 = 0;
 var $1220 = 0, $1221 = 0, $1222 = 0, $1223 = 0, $1224 = 0, $1225 = 0, $1226 = 0, $1227 = 0, $1228 = 0, $1229 = 0, $123 = 0, $1230 = 0, $1231 = 0, $1232 = 0, $1233 = 0, $1234 = 0, $1235 = 0, $1236 = 0, $1237 = 0, $1238 = 0;
 var $1239 = 0, $124 = 0, $1240 = 0, $1241 = 0, $1242 = 0, $1243 = 0, $1244 = 0, $1245 = 0, $1246 = 0, $1247 = 0, $1248 = 0, $1249 = 0, $125 = 0, $1250 = 0, $1251 = 0, $1252 = 0, $1253 = 0, $1254 = 0, $1255 = 0, $1256 = 0;
 var $1257 = 0, $1258 = 0, $1259 = 0, $126 = 0, $1260 = 0, $1261 = 0, $1262 = 0, $1263 = 0, $1264 = 0, $1265 = 0, $1266 = 0, $1267 = 0, $1268 = 0, $1269 = 0, $127 = 0, $1270 = 0, $1271 = 0, $1272 = 0, $1273 = 0, $1274 = 0;
 var $1275 = 0, $1276 = 0, $1277 = 0, $1278 = 0, $1279 = 0, $128 = 0, $1280 = 0, $1281 = 0, $1282 = 0, $1283 = 0, $1284 = 0, $1285 = 0, $1286 = 0, $1287 = 0, $1288 = 0, $1289 = 0, $129 = 0, $1290 = 0, $1291 = 0, $1292 = 0;
 var $1293 = 0, $1294 = 0, $1295 = 0, $1296 = 0, $1297 = 0, $1298 = 0, $1299 = 0, $13 = 0, $130 = 0, $1300 = 0, $1301 = 0, $1302 = 0, $1303 = 0, $1304 = 0, $1305 = 0, $1306 = 0, $1307 = 0, $1308 = 0, $1309 = 0, $131 = 0;
 var $1310 = 0, $1311 = 0, $1312 = 0, $1313 = 0, $1314 = 0, $1315 = 0, $1316 = 0, $1317 = 0, $1318 = 0, $1319 = 0, $132 = 0, $1320 = 0, $1321 = 0, $1322 = 0, $1323 = 0, $1324 = 0, $1325 = 0, $1326 = 0, $1327 = 0, $1328 = 0;
 var $1329 = 0, $133 = 0, $1330 = 0, $1331 = 0, $1332 = 0, $1333 = 0, $1334 = 0, $1335 = 0, $1336 = 0, $1337 = 0, $1338 = 0, $1339 = 0, $134 = 0.0, $1340 = 0, $1341 = 0, $1342 = 0, $1343 = 0, $1344 = 0, $1345 = 0, $1346 = 0;
 var $1347 = 0, $1348 = 0, $1349 = 0, $135 = 0, $1350 = 0, $1351 = 0, $1352 = 0, $1353 = 0, $1354 = 0, $1355 = 0, $1356 = 0, $1357 = 0, $1358 = 0, $1359 = 0, $136 = 0, $1360 = 0, $1361 = 0, $1362 = 0, $1363 = 0, $1364 = 0;
 var $1365 = 0, $1366 = 0, $1367 = 0, $1368 = 0, $1369 = 0, $137 = 0, $1370 = 0, $1371 = 0, $1372 = 0, $1373 = 0, $1374 = 0, $1375 = 0, $1376 = 0, $1377 = 0, $1378 = 0, $1379 = 0, $138 = 0, $1380 = 0, $1381 = 0, $1382 = 0;
 var $1383 = 0, $1384 = 0, $1385 = 0, $1386 = 0, $1387 = 0, $1388 = 0, $1389 = 0, $139 = 0, $1390 = 0, $1391 = 0, $1392 = 0, $1393 = 0, $1394 = 0, $1395 = 0, $1396 = 0, $1397 = 0, $1398 = 0, $1399 = 0, $14 = 0, $140 = 0;
 var $1400 = 0, $1401 = 0, $1402 = 0, $1403 = 0, $1404 = 0, $1405 = 0, $1406 = 0, $1407 = 0, $1408 = 0, $1409 = 0, $141 = 0, $1410 = 0, $1411 = 0, $1412 = 0, $1413 = 0, $1414 = 0, $1415 = 0, $1416 = 0, $1417 = 0, $1418 = 0;
 var $1419 = 0, $142 = 0, $1420 = 0, $1421 = 0, $1422 = 0, $1423 = 0, $1424 = 0, $1425 = 0, $1426 = 0, $1427 = 0, $1428 = 0, $1429 = 0, $143 = 0, $1430 = 0, $1431 = 0, $1432 = 0, $1433 = 0, $1434 = 0, $1435 = 0, $1436 = 0;
 var $1437 = 0, $1438 = 0, $1439 = 0, $144 = 0, $1440 = 0, $1441 = 0, $1442 = 0, $1443 = 0, $1444 = 0, $1445 = 0, $1446 = 0, $1447 = 0, $1448 = 0, $1449 = 0, $145 = 0, $1450 = 0, $1451 = 0, $1452 = 0, $1453 = 0, $1454 = 0;
 var $1455 = 0, $1456 = 0, $1457 = 0, $1458 = 0, $1459 = 0, $146 = 0, $1460 = 0, $1461 = 0, $1462 = 0, $1463 = 0, $1464 = 0, $1465 = 0, $1466 = 0, $1467 = 0, $1468 = 0, $1469 = 0, $147 = 0, $1470 = 0, $1471 = 0, $1472 = 0;
 var $1473 = 0, $1474 = 0, $1475 = 0, $1476 = 0, $1477 = 0, $1478 = 0, $1479 = 0, $148 = 0, $1480 = 0, $1481 = 0, $1482 = 0, $1483 = 0, $1484 = 0, $1485 = 0, $1486 = 0, $1487 = 0, $1488 = 0, $1489 = 0, $149 = 0, $1490 = 0;
 var $1491 = 0, $1492 = 0, $1493 = 0, $1494 = 0, $1495 = 0, $1496 = 0, $1497 = 0, $1498 = 0, $1499 = 0, $15 = 0, $150 = 0, $1500 = 0, $1501 = 0, $1502 = 0, $1503 = 0, $1504 = 0, $1505 = 0, $1506 = 0, $1507 = 0, $1508 = 0;
 var $1509 = 0, $151 = 0, $1510 = 0, $1511 = 0, $1512 = 0, $1513 = 0, $1514 = 0, $1515 = 0, $1516 = 0, $1517 = 0, $1518 = 0, $1519 = 0, $152 = 0, $1520 = 0, $1521 = 0, $1522 = 0, $1523 = 0, $1524 = 0, $1525 = 0, $1526 = 0;
 var $1527 = 0, $1528 = 0, $1529 = 0, $153 = 0, $1530 = 0, $1531 = 0, $1532 = 0, $1533 = 0, $1534 = 0, $1535 = 0, $1536 = 0, $1537 = 0, $1538 = 0, $1539 = 0, $154 = 0, $1540 = 0, $1541 = 0, $1542 = 0, $1543 = 0, $1544 = 0;
 var $1545 = 0, $1546 = 0, $1547 = 0, $1548 = 0, $1549 = 0, $155 = 0, $1550 = 0, $1551 = 0, $1552 = 0, $1553 = 0, $1554 = 0, $1555 = 0, $1556 = 0, $1557 = 0, $1558 = 0, $1559 = 0, $156 = 0, $1560 = 0, $1561 = 0, $1562 = 0;
 var $1563 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0;
 var $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0;
 var $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0.0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0;
 var $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0;
 var $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0;
 var $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0.0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0;
 var $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0;
 var $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0;
 var $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0;
 var $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0;
 var $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0;
 var $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0;
 var $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0;
 var $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0;
 var $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0;
 var $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0;
 var $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0;
 var $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0;
 var $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0;
 var $5 = 0, $50 = 0, $500 = 0, $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $509 = 0, $51 = 0, $510 = 0, $511 = 0, $512 = 0, $513 = 0, $514 = 0, $515 = 0, $516 = 0;
 var $517 = 0, $518 = 0, $519 = 0, $52 = 0, $520 = 0, $521 = 0, $522 = 0, $523 = 0, $524 = 0, $525 = 0, $526 = 0, $527 = 0, $528 = 0, $529 = 0, $53 = 0, $530 = 0, $531 = 0, $532 = 0, $533 = 0, $534 = 0;
 var $535 = 0, $536 = 0, $537 = 0, $538 = 0, $539 = 0, $54 = 0, $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $545 = 0, $546 = 0, $547 = 0, $548 = 0, $549 = 0, $55 = 0, $550 = 0, $551 = 0, $552 = 0;
 var $553 = 0, $554 = 0, $555 = 0, $556 = 0, $557 = 0, $558 = 0, $559 = 0, $56 = 0, $560 = 0, $561 = 0, $562 = 0, $563 = 0, $564 = 0, $565 = 0, $566 = 0, $567 = 0, $568 = 0, $569 = 0, $57 = 0, $570 = 0;
 var $571 = 0, $572 = 0, $573 = 0, $574 = 0, $575 = 0, $576 = 0, $577 = 0, $578 = 0, $579 = 0, $58 = 0, $580 = 0, $581 = 0, $582 = 0, $583 = 0, $584 = 0, $585 = 0, $586 = 0, $587 = 0, $588 = 0, $589 = 0;
 var $59 = 0, $590 = 0, $591 = 0, $592 = 0, $593 = 0.0, $594 = 0, $595 = 0, $596 = 0, $597 = 0, $598 = 0, $599 = 0, $6 = 0, $60 = 0, $600 = 0, $601 = 0, $602 = 0.0, $603 = 0, $604 = 0, $605 = 0.0, $606 = 0;
 var $607 = 0, $608 = 0, $609 = 0, $61 = 0, $610 = 0, $611 = 0, $612 = 0.0, $613 = 0, $614 = 0, $615 = 0, $616 = 0.0, $617 = 0, $618 = 0, $619 = 0, $62 = 0, $620 = 0, $621 = 0, $622 = 0, $623 = 0, $624 = 0;
 var $625 = 0.0, $626 = 0, $627 = 0, $628 = 0.0, $629 = 0, $63 = 0, $630 = 0, $631 = 0.0, $632 = 0, $633 = 0, $634 = 0, $635 = 0, $636 = 0.0, $637 = 0, $638 = 0, $639 = 0, $64 = 0, $640 = 0, $641 = 0.0, $642 = 0;
 var $643 = 0, $644 = 0, $645 = 0, $646 = 0.0, $647 = 0, $648 = 0.0, $649 = 0, $65 = 0, $650 = 0, $651 = 0, $652 = 0.0, $653 = 0, $654 = 0, $655 = 0, $656 = 0, $657 = 0, $658 = 0, $659 = 0, $66 = 0, $660 = 0.0;
 var $661 = 0, $662 = 0, $663 = 0.0, $664 = 0, $665 = 0, $666 = 0, $667 = 0, $668 = 0, $669 = 0, $67 = 0, $670 = 0, $671 = 0, $672 = 0, $673 = 0, $674 = 0, $675 = 0, $676 = 0, $677 = 0, $678 = 0, $679 = 0;
 var $68 = 0, $680 = 0, $681 = 0, $682 = 0, $683 = 0, $684 = 0, $685 = 0, $686 = 0, $687 = 0, $688 = 0, $689 = 0, $69 = 0, $690 = 0, $691 = 0, $692 = 0, $693 = 0, $694 = 0, $695 = 0, $696 = 0, $697 = 0;
 var $698 = 0, $699 = 0, $7 = 0, $70 = 0, $700 = 0, $701 = 0, $702 = 0, $703 = 0, $704 = 0, $705 = 0, $706 = 0, $707 = 0, $708 = 0, $709 = 0, $71 = 0, $710 = 0, $711 = 0, $712 = 0, $713 = 0, $714 = 0;
 var $715 = 0, $716 = 0, $717 = 0, $718 = 0, $719 = 0, $72 = 0, $720 = 0, $721 = 0, $722 = 0, $723 = 0, $724 = 0, $725 = 0, $726 = 0, $727 = 0, $728 = 0, $729 = 0, $73 = 0, $730 = 0, $731 = 0, $732 = 0;
 var $733 = 0, $734 = 0, $735 = 0, $736 = 0, $737 = 0, $738 = 0, $739 = 0, $74 = 0, $740 = 0, $741 = 0, $742 = 0, $743 = 0, $744 = 0, $745 = 0, $746 = 0, $747 = 0, $748 = 0, $749 = 0, $75 = 0, $750 = 0;
 var $751 = 0, $752 = 0, $753 = 0, $754 = 0, $755 = 0, $756 = 0, $757 = 0, $758 = 0, $759 = 0, $76 = 0, $760 = 0, $761 = 0, $762 = 0, $763 = 0, $764 = 0, $765 = 0, $766 = 0, $767 = 0, $768 = 0, $769 = 0;
 var $77 = 0, $770 = 0, $771 = 0, $772 = 0, $773 = 0, $774 = 0, $775 = 0, $776 = 0, $777 = 0, $778 = 0, $779 = 0, $78 = 0, $780 = 0, $781 = 0, $782 = 0, $783 = 0, $784 = 0, $785 = 0, $786 = 0, $787 = 0;
 var $788 = 0, $789 = 0, $79 = 0, $790 = 0, $791 = 0, $792 = 0, $793 = 0, $794 = 0, $795 = 0, $796 = 0, $797 = 0, $798 = 0, $799 = 0, $8 = 0, $80 = 0, $800 = 0, $801 = 0, $802 = 0, $803 = 0, $804 = 0;
 var $805 = 0, $806 = 0, $807 = 0, $808 = 0, $809 = 0, $81 = 0, $810 = 0, $811 = 0, $812 = 0, $813 = 0, $814 = 0, $815 = 0, $816 = 0, $817 = 0, $818 = 0, $819 = 0, $82 = 0, $820 = 0, $821 = 0, $822 = 0;
 var $823 = 0, $824 = 0, $825 = 0, $826 = 0, $827 = 0, $828 = 0, $829 = 0, $83 = 0, $830 = 0, $831 = 0, $832 = 0, $833 = 0, $834 = 0, $835 = 0, $836 = 0, $837 = 0, $838 = 0, $839 = 0, $84 = 0, $840 = 0;
 var $841 = 0, $842 = 0, $843 = 0, $844 = 0, $845 = 0, $846 = 0, $847 = 0, $848 = 0, $849 = 0, $85 = 0, $850 = 0, $851 = 0, $852 = 0, $853 = 0, $854 = 0, $855 = 0, $856 = 0, $857 = 0, $858 = 0, $859 = 0;
 var $86 = 0, $860 = 0, $861 = 0, $862 = 0, $863 = 0, $864 = 0, $865 = 0, $866 = 0, $867 = 0, $868 = 0, $869 = 0, $87 = 0, $870 = 0, $871 = 0, $872 = 0, $873 = 0, $874 = 0.0, $875 = 0, $876 = 0.0, $877 = 0.0;
 var $878 = 0, $879 = 0, $88 = 0, $880 = 0, $881 = 0, $882 = 0, $883 = 0, $884 = 0, $885 = 0, $886 = 0, $887 = 0, $888 = 0, $889 = 0, $89 = 0, $890 = 0, $891 = 0, $892 = 0, $893 = 0, $894 = 0.0, $895 = 0;
 var $896 = 0.0, $897 = 0.0, $898 = 0, $899 = 0, $9 = 0, $90 = 0, $900 = 0, $901 = 0, $902 = 0, $903 = 0, $904 = 0, $905 = 0, $906 = 0, $907 = 0, $908 = 0, $909 = 0, $91 = 0, $910 = 0, $911 = 0, $912 = 0;
 var $913 = 0, $914 = 0, $915 = 0, $916 = 0, $917 = 0, $918 = 0, $919 = 0, $92 = 0.0, $920 = 0, $921 = 0, $922 = 0, $923 = 0, $924 = 0, $925 = 0, $926 = 0, $927 = 0, $928 = 0, $929 = 0, $93 = 0, $930 = 0;
 var $931 = 0, $932 = 0, $933 = 0, $934 = 0, $935 = 0, $936 = 0, $937 = 0, $938 = 0, $939 = 0, $94 = 0, $940 = 0, $941 = 0, $942 = 0, $943 = 0, $944 = 0, $945 = 0, $946 = 0, $947 = 0, $948 = 0, $949 = 0;
 var $95 = 0, $950 = 0, $951 = 0, $952 = 0, $953 = 0, $954 = 0, $955 = 0, $956 = 0, $957 = 0, $958 = 0, $959 = 0, $96 = 0, $960 = 0, $961 = 0, $962 = 0, $963 = 0, $964 = 0, $965 = 0, $966 = 0, $967 = 0;
 var $968 = 0, $969 = 0, $97 = 0, $970 = 0, $971 = 0, $972 = 0, $973 = 0, $974 = 0, $975 = 0, $976 = 0, $977 = 0, $978 = 0, $979 = 0, $98 = 0, $980 = 0, $981 = 0, $982 = 0, $983 = 0, $984 = 0, $985 = 0;
 var $986 = 0, $987 = 0, $988 = 0, $989 = 0, $99 = 0, $990 = 0, $991 = 0, $992 = 0, $993 = 0, $994 = 0, $995 = 0, $996 = 0, $997 = 0, $998 = 0, $999 = 0, $arglist_current = 0, $arglist_current11 = 0, $arglist_current14 = 0, $arglist_current17 = 0, $arglist_current2 = 0;
 var $arglist_current20 = 0, $arglist_current23 = 0, $arglist_current26 = 0, $arglist_current29 = 0, $arglist_current32 = 0, $arglist_current35 = 0, $arglist_current38 = 0, $arglist_current41 = 0, $arglist_current44 = 0, $arglist_current5 = 0, $arglist_current8 = 0, $arglist_next = 0, $arglist_next12 = 0, $arglist_next15 = 0, $arglist_next18 = 0, $arglist_next21 = 0, $arglist_next24 = 0, $arglist_next27 = 0, $arglist_next3 = 0, $arglist_next30 = 0;
 var $arglist_next33 = 0, $arglist_next36 = 0, $arglist_next39 = 0, $arglist_next42 = 0, $arglist_next45 = 0, $arglist_next6 = 0, $arglist_next9 = 0, $cond = 0, $exitcond = 0, $expanded = 0, $expanded101 = 0, $expanded102 = 0, $expanded104 = 0, $expanded105 = 0, $expanded106 = 0, $expanded108 = 0, $expanded109 = 0, $expanded111 = 0, $expanded112 = 0, $expanded113 = 0;
 var $expanded115 = 0, $expanded116 = 0, $expanded118 = 0, $expanded119 = 0, $expanded120 = 0, $expanded122 = 0, $expanded123 = 0, $expanded125 = 0, $expanded126 = 0, $expanded127 = 0, $expanded129 = 0, $expanded130 = 0, $expanded132 = 0, $expanded133 = 0, $expanded134 = 0, $expanded136 = 0, $expanded137 = 0, $expanded139 = 0, $expanded140 = 0, $expanded141 = 0;
 var $expanded143 = 0, $expanded144 = 0, $expanded146 = 0, $expanded147 = 0, $expanded148 = 0, $expanded150 = 0, $expanded151 = 0, $expanded153 = 0, $expanded154 = 0, $expanded155 = 0, $expanded46 = 0, $expanded48 = 0, $expanded49 = 0, $expanded50 = 0, $expanded52 = 0, $expanded53 = 0, $expanded55 = 0, $expanded56 = 0, $expanded57 = 0, $expanded59 = 0;
 var $expanded60 = 0, $expanded62 = 0, $expanded63 = 0, $expanded64 = 0, $expanded66 = 0, $expanded67 = 0, $expanded69 = 0, $expanded70 = 0, $expanded71 = 0, $expanded73 = 0, $expanded74 = 0, $expanded76 = 0, $expanded77 = 0, $expanded78 = 0, $expanded80 = 0, $expanded81 = 0, $expanded83 = 0, $expanded84 = 0, $expanded85 = 0, $expanded87 = 0;
 var $expanded88 = 0, $expanded90 = 0, $expanded91 = 0, $expanded92 = 0, $expanded94 = 0, $expanded95 = 0, $expanded97 = 0, $expanded98 = 0, $expanded99 = 0, $not$ = 0, $not$946 = 0, $not$947 = 0, $or$cond = 0, $or$cond10 = 0, $or$cond12$not = 0, $or$cond15 = 0, $or$cond1597 = 0, $or$cond17 = 0, $or$cond1841 = 0, $or$cond19 = 0;
 var $or$cond22 = 0, $or$cond729 = 0, $or$cond730 = 0, $scevgep = 0, $scevgep1235 = 0, $spec$select = 0, $spec$select726 = 0, $spec$select727 = 0, $spec$select727923 = 0, $spec$select727924 = 0, $spec$select728 = 0, $spec$select732 = 0, $spec$select733 = 0, $spec$select733$us = 0, $spec$select739 = 0, $spec$select740 = 0, $spec$select941 = 0, $spec$select942 = 0.0, $spec$select943 = 0, $spec$select944 = 0;
 var $spec$select945 = 0, $storemerge = 0.0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 96|0;
 $3 = sp + 80|0;
 $4 = sp + 72|0;
 $5 = sp;
 $6 = ((($0)) + 21|0);
 $7 = HEAP8[$6>>0]|0;
 $8 = $7 & 2;
 $9 = ($8<<24>>24)==(0);
 if ($9) {
  $$0669 = 1;$$0724 = 0;
 } else {
  $arglist_current = HEAP32[$2>>2]|0;
  $10 = $arglist_current;
  $11 = ((0) + 4|0);
  $expanded46 = $11;
  $expanded = (($expanded46) - 1)|0;
  $12 = (($10) + ($expanded))|0;
  $13 = ((0) + 4|0);
  $expanded50 = $13;
  $expanded49 = (($expanded50) - 1)|0;
  $expanded48 = $expanded49 ^ -1;
  $14 = $12 & $expanded48;
  $15 = $14;
  $16 = HEAP32[$15>>2]|0;
  $arglist_next = ((($15)) + 4|0);
  HEAP32[$2>>2] = $arglist_next;
  $$0669 = 0;$$0724 = $16;
 }
 $17 = ((($$0724)) + 4|0);
 $18 = ((($$0724)) + 8|0);
 $19 = ((($0)) + 16|0);
 $20 = ((($0)) + 8|0);
 $21 = ((($0)) + 4|0);
 $22 = ((($0)) + 20|0);
 $23 = ((($5)) + 1|0);
 $24 = ((($5)) + 2|0);
 $25 = ((($5)) + 3|0);
 $26 = ((($0)) + 12|0);
 $$0 = $1;$$0567 = 0;$$0693 = 0;
 L4: while(1) {
  $27 = HEAP8[$$0>>0]|0;
  switch ($27<<24>>24) {
  case 0:  {
   label = 543;
   break L4;
   break;
  }
  case 37:  {
   $$1568 = $$0567;$$2 = $$0;
   break;
  }
  default: {
   $$1 = $$0;
   L8: while(1) {
    $28 = ((($$1)) + 1|0);
    $29 = HEAP8[$28>>0]|0;
    switch ($29<<24>>24) {
    case 37: case 0:  {
     break L8;
     break;
    }
    default: {
     $$1 = $28;
    }
    }
   }
   $30 = $28;
   $31 = $$0;
   $32 = (($30) - ($31))|0;
   $33 = HEAP32[$19>>2]|0;
   $34 = (($33) + ($32))|0;
   $35 = HEAP32[$20>>2]|0;
   $36 = ($34>>>0)<($35>>>0);
   if ($36) {
    $37 = ($32|0)==(0);
    if (!($37)) {
     HEAP32[$19>>2] = $34;
     $38 = HEAP32[$21>>2]|0;
     $39 = (($38) + ($33)|0);
     _memcpy(($39|0),($$0|0),($32|0))|0;
    }
   } else {
    _enlargeAndAppend($0,$$0,$32);
   }
   $40 = HEAP8[$28>>0]|0;
   $41 = ($40<<24>>24)==(0);
   if ($41) {
    label = 543;
    break L4;
   } else {
    $$1568 = $$0;$$2 = $28;
   }
  }
  }
  $42 = ((($$2)) + 1|0);
  $43 = HEAP8[$42>>0]|0;
  $44 = ($43<<24>>24)==(0);
  if ($44) {
   label = 13;
   break;
  }
  $51 = $43 << 24 >> 24;
  $$0561 = $51;$$0578 = -1;$$0607 = 0;$$0619 = 0;$$0624 = 0;$$0626 = 0;$$0629 = 0;$$0647 = 0;$$0657 = 0;$$3 = $42;
  L18: while(1) {
   switch ($$0561|0) {
   case 108:  {
    label = 24;
    break L18;
    break;
   }
   case 45:  {
    $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = 1;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 43:  {
    $$1625898 = 43;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 32:  {
    $$1625898 = 32;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 35:  {
    $$1625898 = $$0624;$$1627897 = 1;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 33:  {
    $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = 1;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 48:  {
    $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = 1;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 44:  {
    $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = 44;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$0578;$$9906 = $$3;
    break;
   }
   case 57: case 56: case 55: case 54: case 53: case 52: case 51: case 50: case 49:  {
    $$07251003 = (($$0561) + -48)|0;
    $59 = ((($$3)) + 1|0);
    $60 = HEAP8[$59>>0]|0;
    $61 = $60 << 24 >> 24;
    $$off1004 = (($60) + -48)<<24>>24;
    $62 = ($$off1004&255)<(10);
    if ($62) {
     $$07251005 = $$07251003;$65 = $61;$67 = $59;
     while(1) {
      $63 = ($$07251005*10)|0;
      $64 = (($65) + ($63))|0;
      $$0725 = (($64) + -48)|0;
      $66 = ((($67)) + 1|0);
      $68 = HEAP8[$66>>0]|0;
      $69 = $68 << 24 >> 24;
      $$off = (($68) + -48)<<24>>24;
      $70 = ($$off&255)<(10);
      if ($70) {
       $$07251005 = $$0725;$65 = $69;$67 = $66;
      } else {
       $$0725$lcssa = $$0725;$$5$lcssa = $67;$$lcssa961 = $69;$$lcssa962 = $68;$$lcssa963 = $66;
       break;
      }
     }
    } else {
     $$0725$lcssa = $$07251003;$$5$lcssa = $$3;$$lcssa961 = $61;$$lcssa962 = $60;$$lcssa963 = $59;
    }
    $71 = $$0725$lcssa & 2147483647;
    switch ($$lcssa962<<24>>24) {
    case 46: case 108:  {
     $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $71;$$4582905 = $$0578;$$9906 = $$5$lcssa;
     break;
    }
    default: {
     $$10 = $$lcssa963;$$2621900 = $$0619;$$2651890 = 0;$$3610902 = $71;$$4582904 = $$0578;$$5566 = $$lcssa961;
     break L18;
    }
    }
    break;
   }
   case 42:  {
    do {
     if ($$0669) {
      $arglist_current2 = HEAP32[$2>>2]|0;
      $99 = $arglist_current2;
      $100 = ((0) + 4|0);
      $expanded53 = $100;
      $expanded52 = (($expanded53) - 1)|0;
      $101 = (($99) + ($expanded52))|0;
      $102 = ((0) + 4|0);
      $expanded57 = $102;
      $expanded56 = (($expanded57) - 1)|0;
      $expanded55 = $expanded56 ^ -1;
      $103 = $101 & $expanded55;
      $104 = $103;
      $105 = HEAP32[$104>>2]|0;
      $arglist_next3 = ((($104)) + 4|0);
      HEAP32[$2>>2] = $arglist_next3;
      $$1608 = $105;
      label = 39;
     } else {
      $72 = HEAP32[$$0724>>2]|0;
      $73 = HEAP32[$17>>2]|0;
      $74 = ($72|0)>($73|0);
      if ($74) {
       $75 = HEAP32[$18>>2]|0;
       $76 = (($73) + 1)|0;
       HEAP32[$17>>2] = $76;
       $77 = (($75) + ($73<<2)|0);
       $78 = HEAP32[$77>>2]|0;
       $79 = ((($78)) + 8|0);
       $80 = HEAP16[$79>>1]|0;
       $81 = $80&65535;
       $82 = $81 & 4;
       $83 = ($82|0)==(0);
       if (!($83)) {
        $84 = $78;
        $85 = $84;
        $86 = HEAP32[$85>>2]|0;
        $87 = (($84) + 4)|0;
        $88 = $87;
        $89 = HEAP32[$88>>2]|0;
        $$1608 = $86;
        label = 39;
        break;
       }
       $90 = $81 & 8;
       $91 = ($90|0)==(0);
       if (!($91)) {
        $92 = +HEAPF64[$78>>3];
        $93 = (_doubleToInt64($92)|0);
        $94 = (getTempRet0() | 0);
        $$1608 = $93;
        label = 39;
        break;
       }
       $95 = $81 & 18;
       $96 = ($95|0)==(0);
       if ($96) {
        $$160812401242$ph = 0;
        label = 40;
       } else {
        $$idx$i$i$i769 = ((($78)) + 10|0);
        $$idx$val$i$i$i770 = HEAP8[$$idx$i$i$i769>>0]|0;
        $$idx8$i$i$i771 = ((($78)) + 12|0);
        $$idx8$val$i$i$i772 = HEAP32[$$idx8$i$i$i771>>2]|0;
        $$idx9$i$i$i773 = ((($78)) + 16|0);
        $$idx9$val$i$i$i774 = HEAP32[$$idx9$i$i$i773>>2]|0;
        $97 = (_memIntValue($$idx$val$i$i$i770,$$idx8$val$i$i$i772,$$idx9$val$i$i$i774)|0);
        $98 = (getTempRet0() | 0);
        $$1608 = $97;
        label = 39;
       }
      } else {
       $$160812401242$ph = 0;
       label = 40;
      }
     }
    } while(0);
    if ((label|0) == 39) {
     label = 0;
     $106 = ($$1608|0)<(0);
     $107 = ($$1608|0)==(-2147483648);
     $108 = (0 - ($$1608))|0;
     $$mux1594 = $107 ? 0 : $108;
     if ($106) {
      $1559 = 1;$1560 = $$mux1594;
     } else {
      $$160812401242$ph = $$1608;
      label = 40;
     }
    }
    if ((label|0) == 40) {
     label = 0;
     $1559 = $$0619;$1560 = $$160812401242$ph;
    }
    $109 = ((($$3)) + 1|0);
    $110 = HEAP8[$109>>0]|0;
    switch ($110<<24>>24) {
    case 46: case 108:  {
     $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $1559;$$3610903 = $1560;$$4582905 = $$0578;$$9906 = $$3;
     break;
    }
    default: {
     label = 61;
     break L18;
    }
    }
    break;
   }
   case 46:  {
    $111 = ((($$3)) + 1|0);
    $112 = HEAP8[$111>>0]|0;
    $113 = ($112<<24>>24)==(42);
    if ($113) {
     do {
      if ($$0669) {
       $arglist_current5 = HEAP32[$2>>2]|0;
       $141 = $arglist_current5;
       $142 = ((0) + 4|0);
       $expanded60 = $142;
       $expanded59 = (($expanded60) - 1)|0;
       $143 = (($141) + ($expanded59))|0;
       $144 = ((0) + 4|0);
       $expanded64 = $144;
       $expanded63 = (($expanded64) - 1)|0;
       $expanded62 = $expanded63 ^ -1;
       $145 = $143 & $expanded62;
       $146 = $145;
       $147 = HEAP32[$146>>2]|0;
       $arglist_next6 = ((($146)) + 4|0);
       HEAP32[$2>>2] = $arglist_next6;
       $$1579 = $147;
       label = 52;
      } else {
       $114 = HEAP32[$$0724>>2]|0;
       $115 = HEAP32[$17>>2]|0;
       $116 = ($114|0)>($115|0);
       if ($116) {
        $117 = HEAP32[$18>>2]|0;
        $118 = (($115) + 1)|0;
        HEAP32[$17>>2] = $118;
        $119 = (($117) + ($115<<2)|0);
        $120 = HEAP32[$119>>2]|0;
        $121 = ((($120)) + 8|0);
        $122 = HEAP16[$121>>1]|0;
        $123 = $122&65535;
        $124 = $123 & 4;
        $125 = ($124|0)==(0);
        if (!($125)) {
         $126 = $120;
         $127 = $126;
         $128 = HEAP32[$127>>2]|0;
         $129 = (($126) + 4)|0;
         $130 = $129;
         $131 = HEAP32[$130>>2]|0;
         $$1579 = $128;
         label = 52;
         break;
        }
        $132 = $123 & 8;
        $133 = ($132|0)==(0);
        if (!($133)) {
         $134 = +HEAPF64[$120>>3];
         $135 = (_doubleToInt64($134)|0);
         $136 = (getTempRet0() | 0);
         $$1579 = $135;
         label = 52;
         break;
        }
        $137 = $123 & 18;
        $138 = ($137|0)==(0);
        if ($138) {
         $1561 = 0;
        } else {
         $$idx$i$i$i761 = ((($120)) + 10|0);
         $$idx$val$i$i$i762 = HEAP8[$$idx$i$i$i761>>0]|0;
         $$idx8$i$i$i763 = ((($120)) + 12|0);
         $$idx8$val$i$i$i764 = HEAP32[$$idx8$i$i$i763>>2]|0;
         $$idx9$i$i$i765 = ((($120)) + 16|0);
         $$idx9$val$i$i$i766 = HEAP32[$$idx9$i$i$i765>>2]|0;
         $139 = (_memIntValue($$idx$val$i$i$i762,$$idx8$val$i$i$i764,$$idx9$val$i$i$i766)|0);
         $140 = (getTempRet0() | 0);
         $$1579 = $139;
         label = 52;
        }
       } else {
        $1561 = 0;
       }
      }
     } while(0);
     if ((label|0) == 52) {
      label = 0;
      $148 = ($$1579|0)<(0);
      $149 = ($$1579|0)==(-2147483648);
      $150 = (0 - ($$1579))|0;
      $$mux1596 = $149 ? -1 : $150;
      $spec$select = $148 ? $$mux1596 : $$1579;
      $1561 = $spec$select;
     }
     $151 = ((($$3)) + 2|0);
     $152 = HEAP8[$151>>0]|0;
     $153 = $152 << 24 >> 24;
     $$3564 = $153;$$3581 = $1561;$$8 = $151;
    } else {
     $154 = $112 << 24 >> 24;
     $$2563$off996 = (($154) + -48)|0;
     $155 = ($$2563$off996>>>0)<(10);
     if ($155) {
      $$0722997 = 0;$$2563998 = $154;$$7999 = $111;
      while(1) {
       $156 = ($$0722997*10)|0;
       $157 = (($$2563998) + -48)|0;
       $158 = (($157) + ($156))|0;
       $159 = ((($$7999)) + 1|0);
       $160 = HEAP8[$159>>0]|0;
       $161 = $160 << 24 >> 24;
       $$2563$off = (($161) + -48)|0;
       $162 = ($$2563$off>>>0)<(10);
       if ($162) {
        $$0722997 = $158;$$2563998 = $161;$$7999 = $159;
       } else {
        break;
       }
      }
      $163 = $160 << 24 >> 24;
      $$0722$lcssa = $158;$$2563$lcssa = $163;$$7$lcssa = $159;
     } else {
      $$0722$lcssa = 0;$$2563$lcssa = $154;$$7$lcssa = $111;
     }
     $164 = $$0722$lcssa & 2147483647;
     $$3564 = $$2563$lcssa;$$3581 = $164;$$8 = $$7$lcssa;
    }
    $165 = ($$3564|0)==(108);
    $166 = ((($$8)) + -1|0);
    if ($165) {
     $$1625898 = $$0624;$$1627897 = $$0626;$$1630895 = $$0629;$$1648892 = $$0647;$$1658888 = $$0657;$$2621901 = $$0619;$$3610903 = $$0607;$$4582905 = $$3581;$$9906 = $166;
    } else {
     $$10 = $$8;$$2621900 = $$0619;$$2651890 = 0;$$3610902 = $$0607;$$4582904 = $$3581;$$5566 = $$3564;
     break L18;
    }
    break;
   }
   default: {
    $$10 = $$3;$$2621900 = $$0619;$$2651890 = 0;$$3610902 = $$0607;$$4582904 = $$0578;$$5566 = $$0561;
    break L18;
   }
   }
   $167 = ((($$9906)) + 1|0);
   $168 = HEAP8[$167>>0]|0;
   $169 = $168 << 24 >> 24;
   $170 = ($168<<24>>24)==(0);
   if ($170) {
    label = 543;
    break L4;
   } else {
    $$0561 = $169;$$0578 = $$4582905;$$0607 = $$3610903;$$0619 = $$2621901;$$0624 = $$1625898;$$0626 = $$1627897;$$0629 = $$1630895;$$0647 = $$1648892;$$0657 = $$1658888;$$3 = $167;
   }
  }
  if ((label|0) == 24) {
   label = 0;
   $52 = ((($$3)) + 1|0);
   $53 = HEAP8[$52>>0]|0;
   $54 = $53 << 24 >> 24;
   $55 = ($53<<24>>24)==(108);
   if ($55) {
    $56 = ((($$3)) + 2|0);
    $57 = HEAP8[$56>>0]|0;
    $58 = $57 << 24 >> 24;
    $$10 = $56;$$2621900 = $$0619;$$2651890 = 2;$$3610902 = $$0607;$$4582904 = $$0578;$$5566 = $58;
   } else {
    $$10 = $52;$$2621900 = $$0619;$$2651890 = 1;$$3610902 = $$0607;$$4582904 = $$0578;$$5566 = $54;
   }
  }
  else if ((label|0) == 61) {
   label = 0;
   $171 = ((($$3)) + 1|0);
   $172 = $110 << 24 >> 24;
   $$10 = $171;$$2621900 = $1559;$$2651890 = 0;$$3610902 = $1560;$$4582904 = $$0578;$$5566 = $172;
  }
  L81: do {
   switch ($$5566|0) {
   case 100:  {
    $$06031059$lcssa = 0;
    break;
   }
   case 115:  {
    $$06031059$lcssa = 1;
    break;
   }
   case 103:  {
    $$06031059$lcssa = 2;
    break;
   }
   case 122:  {
    $$06031059$lcssa = 3;
    break;
   }
   case 113:  {
    $$06031059$lcssa = 4;
    break;
   }
   case 81:  {
    $$06031059$lcssa = 5;
    break;
   }
   case 119:  {
    $$06031059$lcssa = 6;
    break;
   }
   case 99:  {
    $$06031059$lcssa = 7;
    break;
   }
   case 111:  {
    $$06031059$lcssa = 8;
    break;
   }
   case 117:  {
    $$06031059$lcssa = 9;
    break;
   }
   case 120:  {
    $$06031059$lcssa = 10;
    break;
   }
   case 88:  {
    $$06031059$lcssa = 11;
    break;
   }
   case 102:  {
    $$06031059$lcssa = 12;
    break;
   }
   case 101:  {
    $$06031059$lcssa = 13;
    break;
   }
   case 69:  {
    $$06031059$lcssa = 14;
    break;
   }
   case 71:  {
    $$06031059$lcssa = 15;
    break;
   }
   case 105:  {
    $$06031059$lcssa = 16;
    break;
   }
   case 110:  {
    $$06031059$lcssa = 17;
    break;
   }
   case 37:  {
    $$06031059$lcssa = 18;
    break;
   }
   case 112:  {
    $$06031059$lcssa = 19;
    break;
   }
   case 84:  {
    $$06031059$lcssa = 20;
    break;
   }
   default: {
    switch ($$5566|0) {
    case 83:  {
     $$06031059$lcssa = 21;
     break L81;
     break;
    }
    case 114:  {
     break;
    }
    default: {
     label = 543;
     break L4;
    }
    }
    $$06031059$lcssa = 22;
   }
   }
  } while(0);
  $173 = (((16 + (($$06031059$lcssa*6)|0)|0)) + 3|0);
  $174 = HEAP8[$173>>0]|0;
  L105: do {
   switch ($174<<24>>24) {
   case 13:  {
    $$3652 = 1;
    label = 86;
    break;
   }
   case 0: case 15:  {
    $$3652 = $$2651890;
    label = 86;
    break;
   }
   case 16:  {
    $$2659 = $$0657;$$4653 = $$2651890;
    label = 87;
    break;
   }
   case 3: case 2: case 1:  {
    do {
     if ($$0669) {
      $arglist_current23 = HEAP32[$2>>2]|0;
      $606 = $arglist_current23;
      $607 = ((0) + 8|0);
      $expanded102 = $607;
      $expanded101 = (($expanded102) - 1)|0;
      $608 = (($606) + ($expanded101))|0;
      $609 = ((0) + 8|0);
      $expanded106 = $609;
      $expanded105 = (($expanded106) - 1)|0;
      $expanded104 = $expanded105 ^ -1;
      $610 = $608 & $expanded104;
      $611 = $610;
      $612 = +HEAPF64[$611>>3];
      $arglist_next24 = ((($611)) + 8|0);
      HEAP32[$2>>2] = $arglist_next24;
      $storemerge = $612;
      label = 208;
     } else {
      $581 = HEAP32[$$0724>>2]|0;
      $582 = HEAP32[$17>>2]|0;
      $583 = ($581|0)>($582|0);
      if ($583) {
       $584 = HEAP32[$18>>2]|0;
       $585 = (($582) + 1)|0;
       HEAP32[$17>>2] = $585;
       $586 = (($584) + ($582<<2)|0);
       $587 = HEAP32[$586>>2]|0;
       $588 = ((($587)) + 8|0);
       $589 = HEAP16[$588>>1]|0;
       $590 = $589&65535;
       $591 = $590 & 8;
       $592 = ($591|0)==(0);
       if (!($592)) {
        $593 = +HEAPF64[$587>>3];
        $storemerge = $593;
        label = 208;
        break;
       }
       $594 = $590 & 4;
       $595 = ($594|0)==(0);
       if (!($595)) {
        $596 = $587;
        $597 = $596;
        $598 = HEAP32[$597>>2]|0;
        $599 = (($596) + 4)|0;
        $600 = $599;
        $601 = HEAP32[$600>>2]|0;
        $602 = (+($598>>>0)) + (4294967296.0*(+($601|0)));
        $storemerge = $602;
        label = 208;
        break;
       }
       $603 = $590 & 18;
       $604 = ($603|0)==(0);
       if (!($604)) {
        $$idx$i$i$i = ((($587)) + 10|0);
        $$idx$val$i$i$i = HEAP8[$$idx$i$i$i>>0]|0;
        $$idx5$i$i$i = ((($587)) + 12|0);
        $$idx5$val$i$i$i = HEAP32[$$idx5$i$i$i>>2]|0;
        $$idx6$i$i$i = ((($587)) + 16|0);
        $$idx6$val$i$i$i = HEAP32[$$idx6$i$i$i>>2]|0;
        $605 = (+_memRealValue($$idx$val$i$i$i,$$idx5$val$i$i$i,$$idx6$val$i$i$i));
        $storemerge = $605;
        label = 208;
        break;
       }
      }
      $613 = ($$4582904|0)<(0);
      $spec$select727923 = $613 ? 6 : $$4582904;
      $$0877 = 0.0;$$2672 = $$0624;$spec$select727924 = $spec$select727923;
     }
    } while(0);
    if ((label|0) == 208) {
     label = 0;
     $614 = ($$4582904|0)<(0);
     $spec$select727 = $614 ? 6 : $$4582904;
     $615 = $storemerge < 0.0;
     if ($615) {
      $616 = - $storemerge;
      $$0877 = $616;$$2672 = 45;$spec$select727924 = $spec$select727;
     } else {
      $$0877 = $storemerge;$$2672 = $$0624;$spec$select727924 = $spec$select727;
     }
    }
    $617 = ($$06031059$lcssa|0)==(2);
    $618 = ($$06031059$lcssa|0)==(15);
    $619 = $617 | $618;
    $620 = ($spec$select727924|0)>(0);
    $or$cond10 = $619 & $620;
    $621 = $or$cond10 << 31 >> 31;
    $spec$select728 = (($spec$select727924) + ($621))|0;
    $622 = $spec$select728 & 4095;
    $623 = ($622|0)==(0);
    if ($623) {
     $$0720$lcssa = 0.5;
    } else {
     $$07201101 = 0.5;$$36061102 = $622;
     while(1) {
      $624 = (($$36061102) + -1)|0;
      $625 = $$07201101 * 0.10000000000000001;
      $626 = ($$36061102|0)>(1);
      if ($626) {
       $$07201101 = $625;$$36061102 = $624;
      } else {
       $$0720$lcssa = $625;
       break;
      }
     }
    }
    $627 = ($$06031059$lcssa|0)==(12);
    $628 = $$0877 + $$0720$lcssa;
    $spec$select942 = $627 ? $628 : $$0877;
    HEAPF64[$3>>3] = $spec$select942;
    $$0$$0$$0$2$i = +HEAPF64[$3>>3];
    HEAPF64[$4>>3] = $$0$$0$$0$2$i;
    $$0$$0$$0$3$i = +HEAPF64[$3>>3];
    $$0$$0$$0$$i = +HEAPF64[$4>>3];
    $629 = $$0$$0$$0$3$i == $$0$$0$$0$$i;
    if ($629) {
     $630 = $spec$select942 > 0.0;
     if ($630) {
      $$0666 = 1.0;$$0707 = 0;
      while(1) {
       $631 = $$0666 * 1.0E+100;
       $632 = $spec$select942 >= $631;
       $633 = ($$0707>>>0)<(351);
       $634 = $633 & $632;
       $635 = (($$0707) + 100)|0;
       if ($634) {
        $$0666 = $631;$$0707 = $635;
       } else {
        break;
       }
      }
      $$1667 = $$0666;$$1708 = $$0707;
      while(1) {
       $636 = $$1667 * 1.0E+10;
       $637 = $spec$select942 >= $636;
       $638 = ($$1708>>>0)<(351);
       $639 = $638 & $637;
       $640 = (($$1708) + 10)|0;
       if ($639) {
        $$1667 = $636;$$1708 = $640;
       } else {
        break;
       }
      }
      $$2668 = $$1667;$$2709 = $$1708;
      while(1) {
       $641 = $$2668 * 10.0;
       $642 = $spec$select942 >= $641;
       $643 = ($$2709>>>0)<(351);
       $644 = $643 & $642;
       $645 = (($$2709) + 1)|0;
       if ($644) {
        $$2668 = $641;$$2709 = $645;
       } else {
        break;
       }
      }
      $646 = $spec$select942 / $$2668;
      $647 = $646 < 1.0E-8;
      if ($647) {
       $$28791107 = $646;$$37101108 = $$2709;
       while(1) {
        $648 = $$28791107 * 1.0E+8;
        $649 = (($$37101108) + -8)|0;
        $650 = $648 < 1.0E-8;
        if ($650) {
         $$28791107 = $648;$$37101108 = $649;
        } else {
         $$2879$lcssa = $648;$$3710$lcssa = $649;
         break;
        }
       }
      } else {
       $$2879$lcssa = $646;$$3710$lcssa = $$2709;
      }
      $651 = $$2879$lcssa < 1.0;
      if ($651) {
       $$38801114 = $$2879$lcssa;$$47111115 = $$3710$lcssa;
       while(1) {
        $652 = $$38801114 * 10.0;
        $653 = (($$47111115) + -1)|0;
        $654 = $652 < 1.0;
        if ($654) {
         $$38801114 = $652;$$47111115 = $653;
        } else {
         $$3880$lcssa = $652;$$4711$lcssa = $653;
         break;
        }
       }
      } else {
       $$3880$lcssa = $$2879$lcssa;$$4711$lcssa = $$3710$lcssa;
      }
      $655 = ($$4711$lcssa|0)>(350);
      $656 = ($$2672<<24>>24)!=(0);
      if ($655) {
       $657 = $656 ? 4 : 3;
       $658 = $656&1;
       $659 = (($5) + ($658)|0);
       HEAP8[$5>>0] = $$2672;
       HEAP8[$659>>0]=6712905&255;HEAP8[$659+1>>0]=(6712905>>8)&255;HEAP8[$659+2>>0]=(6712905>>16)&255;HEAP8[$659+3>>0]=6712905>>24;
       $$11602 = $657;$$11618 = $$3610902;$$28 = $5;$$3696 = $$0693;$$8705 = 0;
       break L105;
      } else {
       $$4881 = $$3880$lcssa;$$5712 = $$4711$lcssa;
      }
     } else {
      $$4881 = $spec$select942;$$5712 = 0;
     }
     if ($627) {
      $$5882 = $$4881;$$6713 = $$5712;
     } else {
      $660 = $$0720$lcssa + $$4881;
      $661 = !($660 >= 10.0);
      if ($661) {
       $$5882 = $660;$$6713 = $$5712;
      } else {
       $662 = (($$5712) + 1)|0;
       $663 = $660 * 0.10000000000000001;
       $$5882 = $663;$$6713 = $662;
      }
     }
     $664 = ($$0626<<24>>24)==(0);
     $665 = $664&1;
     $666 = ($$6713|0)<(-4);
     $667 = ($spec$select728|0)<($$6713|0);
     $or$cond = $666 | $667;
     $spec$select739 = $or$cond ? 2 : 1;
     $668 = $or$cond ? 0 : $$6713;
     $spec$select740 = (($spec$select728) - ($668))|0;
     $spec$select943 = $619 ? $665 : $$0629;
     $spec$select944 = $619 ? $spec$select739 : $174;
     $spec$select945 = $619 ? $spec$select740 : $spec$select727924;
     $669 = ($spec$select944<<24>>24)==(2);
     $$$6713 = $669 ? 0 : $$6713;
     $670 = ($$$6713|0)>(0);
     $671 = $670 ? $$$6713 : 0;
     $672 = ($spec$select945|0)<(0);
     $673 = $672 << 31 >> 31;
     $674 = ($$3610902|0)<(0);
     $675 = $674 << 31 >> 31;
     $676 = (_i64Add(($spec$select945|0),($673|0),($$3610902|0),($675|0))|0);
     $677 = (getTempRet0() | 0);
     $678 = (_i64Add(($676|0),($677|0),($671|0),0)|0);
     $679 = (getTempRet0() | 0);
     $680 = ($679|0)>(0);
     $681 = ($678>>>0)>(55);
     $682 = ($679|0)==(0);
     $683 = $682 & $681;
     $684 = $680 | $683;
     L158: do {
      if ($684) {
       $685 = (_i64Add(($678|0),($679|0),15,0)|0);
       $686 = (getTempRet0() | 0);
       $687 = HEAP8[$22>>0]|0;
       $688 = ($687<<24>>24)==(0);
       if (!($688)) {
        label = 543;
        break L4;
       }
       $689 = HEAP32[$20>>2]|0;
       $690 = ($686|0)>(0);
       $691 = ($685>>>0)>($689>>>0);
       $692 = ($686|0)==(0);
       $693 = $692 & $691;
       $694 = $690 | $693;
       if ($694) {
        $695 = HEAP32[$26>>2]|0;
        $696 = ($686|0)>(0);
        $697 = ($685>>>0)>($695>>>0);
        $698 = ($686|0)==(0);
        $699 = $698 & $697;
        $700 = $696 | $699;
        if ($700) {
         label = 236;
         break L4;
        }
       }
       $742 = HEAP32[$0>>2]|0;
       $743 = ($742|0)==(0|0);
       L164: do {
        if ($743) {
         $777 = (_i64Add(($678|0),($679|0),14,0)|0);
         $778 = (getTempRet0() | 0);
         $779 = ($778>>>0)>(0);
         $780 = ($777>>>0)>(2147483390);
         $781 = ($778|0)==(0);
         $782 = $781 & $780;
         $783 = $779 | $782;
         if ($783) {
          label = 278;
          break L4;
         }
         $784 = HEAP32[7324]|0;
         $785 = ($784|0)==(0);
         if ($785) {
          $824 = HEAP32[(29340)>>2]|0;
          $825 = (FUNCTION_TABLE_ii[$824 & 127]($685)|0);
          $$0$i813 = $825;
          break;
         }
         $786 = HEAP32[(29356)>>2]|0;
         $787 = (FUNCTION_TABLE_ii[$786 & 127]($685)|0);
         $788 = HEAP32[14985]|0;
         $789 = ($788>>>0)<($685>>>0);
         if ($789) {
          HEAP32[14985] = $685;
         }
         $790 = (59064);
         $791 = $790;
         $792 = HEAP32[$791>>2]|0;
         $793 = (($790) + 4)|0;
         $794 = $793;
         $795 = HEAP32[$794>>2]|0;
         $796 = ($795|0)>(0);
         $797 = ($792>>>0)>(0);
         $798 = ($795|0)==(0);
         $799 = $798 & $797;
         $800 = $796 | $799;
         if ($800) {
          $801 = HEAP32[14978]|0;
          $802 = ($787|0)<(0);
          $803 = $802 << 31 >> 31;
          $804 = (_i64Subtract(($792|0),($795|0),($787|0),($803|0))|0);
          $805 = (getTempRet0() | 0);
          $806 = ($805|0)<(0);
          $807 = ($804>>>0)<=($801>>>0);
          $808 = ($805|0)==(0);
          $809 = $808 & $807;
          $810 = $806 | $809;
          $$1702 = $810&1;
          HEAP32[(59072)>>2] = $$1702;
         }
         $811 = HEAP32[(29340)>>2]|0;
         $812 = (FUNCTION_TABLE_ii[$811 & 127]($787)|0);
         $813 = ($812|0)==(0|0);
         if ($813) {
          label = 278;
          break L4;
         }
         $814 = HEAP32[(29352)>>2]|0;
         $815 = (FUNCTION_TABLE_ii[$814 & 127]($812)|0);
         $816 = HEAP32[14978]|0;
         $817 = (($816) + ($815))|0;
         HEAP32[14978] = $817;
         $818 = HEAP32[14982]|0;
         $819 = ($817>>>0)>($818>>>0);
         if ($819) {
          HEAP32[14982] = $817;
         }
         $820 = HEAP32[14981]|0;
         $821 = (($820) + 1)|0;
         HEAP32[14981] = $821;
         $822 = HEAP32[14987]|0;
         $823 = ($821>>>0)>($822>>>0);
         if (!($823)) {
          $$11$ph = $812;$$3700$ph = $812;
          break L158;
         }
         HEAP32[14987] = $821;
         $$11$ph = $812;$$3700$ph = $812;
         break L158;
        } else {
         $744 = ((($742)) + 272|0);
         $745 = HEAP32[$744>>2]|0;
         $746 = ($745|0)==(0);
         if ($746) {
          $747 = ((($742)) + 276|0);
          $748 = HEAP16[$747>>1]|0;
          $749 = $748&65535;
          $750 = ($686>>>0)>(0);
          $751 = ($685>>>0)>($749>>>0);
          $752 = ($686|0)==(0);
          $753 = $752 & $751;
          $754 = $750 | $753;
          do {
           if ($754) {
            $755 = ((($742)) + 288|0);
            $$sink1692 = $755;
           } else {
            $756 = ((($742)) + 300|0);
            $757 = HEAP32[$756>>2]|0;
            $758 = ($757|0)==(0|0);
            if (!($758)) {
             $759 = HEAP32[$757>>2]|0;
             HEAP32[$756>>2] = $759;
             $760 = ((($742)) + 284|0);
             $761 = HEAP32[$760>>2]|0;
             $762 = (($761) + 1)|0;
             HEAP32[$760>>2] = $762;
             $$0$i813 = $757;
             break L164;
            }
            $763 = ((($742)) + 296|0);
            $764 = HEAP32[$763>>2]|0;
            $765 = ($764|0)==(0|0);
            if ($765) {
             $770 = ((($742)) + 292|0);
             $$sink1692 = $770;
             break;
            } else {
             $766 = HEAP32[$764>>2]|0;
             HEAP32[$763>>2] = $766;
             $767 = ((($742)) + 284|0);
             $768 = HEAP32[$767>>2]|0;
             $769 = (($768) + 1)|0;
             HEAP32[$767>>2] = $769;
             $$0$i813 = $764;
             break L164;
            }
           }
          } while(0);
          $774 = HEAP32[$$sink1692>>2]|0;
          $775 = (($774) + 1)|0;
          HEAP32[$$sink1692>>2] = $775;
         } else {
          $771 = ((($742)) + 81|0);
          $772 = HEAP8[$771>>0]|0;
          $773 = ($772<<24>>24)==(0);
          if (!($773)) {
           label = 278;
           break L4;
          }
         }
         $776 = (_dbMallocRawFinish($742,$685,$686)|0);
         $$0$i813 = $776;
        }
       } while(0);
       $826 = ($$0$i813|0)==(0|0);
       if ($826) {
        label = 278;
        break L4;
       } else {
        $$11$ph = $$0$i813;$$3700$ph = $$0$i813;
       }
      } else {
       $$11$ph = $5;$$3700$ph = 0;
      }
     } while(0);
     $860 = $$0629&255;
     $861 = ($860*10)|0;
     $862 = (($861) + 16)|0;
     $863 = ($spec$select945|0)>(0);
     $864 = $863&1;
     $865 = $$0629 | $$0626;
     $866 = $865&255;
     $867 = $866 | $864;
     $868 = ($$2672<<24>>24)==(0);
     if ($868) {
      $$12 = $$11$ph;
     } else {
      $869 = ((($$11$ph)) + 1|0);
      HEAP8[$$11$ph>>0] = $$2672;
      $$12 = $869;
     }
     $870 = ($$$6713|0)<(0);
     if ($870) {
      $871 = ((($$12)) + 1|0);
      HEAP8[$$12>>0] = 48;
      $$14 = $871;$$2718 = $$$6713;$$2875 = $862;$$8884 = $$5882;
     } else {
      $$08731122 = $862;$$131124 = $$12;$$17171123 = $$$6713;$$61121 = $$5882;
      while(1) {
       $872 = ($$08731122|0)<(1);
       $873 = (~~(($$61121)));
       $874 = (+($873|0));
       $875 = (($873) + 48)|0;
       $876 = $$61121 - $874;
       $877 = $876 * 10.0;
       $878 = $875&255;
       $$7883 = $872 ? $$61121 : $877;
       $not$947 = $872 ^ 1;
       $879 = $not$947 << 31 >> 31;
       $$1874 = (($$08731122) + ($879))|0;
       $$0$i743 = $872 ? 48 : $878;
       $880 = ((($$131124)) + 1|0);
       HEAP8[$$131124>>0] = $$0$i743;
       $881 = (($$17171123) + -1)|0;
       $882 = ($$17171123|0)>(0);
       if ($882) {
        $$08731122 = $$1874;$$131124 = $880;$$17171123 = $881;$$61121 = $$7883;
       } else {
        $$14 = $880;$$2718 = -1;$$2875 = $$1874;$$8884 = $$7883;
        break;
       }
      }
     }
     $883 = ($867|0)==(0);
     if ($883) {
      $$15 = $$14;
     } else {
      $884 = ((($$14)) + 1|0);
      HEAP8[$$14>>0] = 46;
      $$15 = $884;
     }
     $885 = ($$2718|0)<(-1);
     if ($885) {
      $886 = $$2718 ^ -1;
      _memset(($$15|0),48,($886|0))|0;
      $$161130 = $$15;$$3719$in1128 = $$2718;$$95871129 = $spec$select945;
      while(1) {
       $$3719 = (($$3719$in1128) + 1)|0;
       $887 = ((($$161130)) + 1|0);
       $888 = (($$95871129) + -1)|0;
       $889 = ($$3719$in1128|0)<(-2);
       if ($889) {
        $$161130 = $887;$$3719$in1128 = $$3719;$$95871129 = $888;
       } else {
        $$16$lcssa = $887;$$9587$lcssa = $888;
        break;
       }
      }
     } else {
      $$16$lcssa = $$15;$$9587$lcssa = $spec$select945;
     }
     $890 = ($$9587$lcssa|0)>(0);
     if ($890) {
      $$105881138 = $$9587$lcssa;$$171139 = $$16$lcssa;$$38761137 = $$2875;$$98851136 = $$8884;
      while(1) {
       $891 = (($$105881138) + -1)|0;
       $892 = ($$38761137|0)<(1);
       $893 = (~~(($$98851136)));
       $894 = (+($893|0));
       $895 = (($893) + 48)|0;
       $896 = $$98851136 - $894;
       $897 = $896 * 10.0;
       $898 = $895&255;
       $$10886 = $892 ? $$98851136 : $897;
       $not$946 = $892 ^ 1;
       $899 = $not$946 << 31 >> 31;
       $$4 = (($$38761137) + ($899))|0;
       $$0$i742 = $892 ? 48 : $898;
       $900 = ((($$171139)) + 1|0);
       HEAP8[$$171139>>0] = $$0$i742;
       $901 = ($$105881138|0)>(1);
       if ($901) {
        $$105881138 = $891;$$171139 = $900;$$38761137 = $$4;$$98851136 = $$10886;
       } else {
        $$17$lcssa = $900;
        break;
       }
      }
     } else {
      $$17$lcssa = $$16$lcssa;
     }
     $902 = ($spec$select943<<24>>24)==(0);
     $or$cond729 = $902 | $883;
     L220: do {
      if ($or$cond729) {
       $$19 = $$17$lcssa;
      } else {
       $$18 = $$17$lcssa;
       L222: while(1) {
        $903 = ((($$18)) + -1|0);
        $904 = HEAP8[$903>>0]|0;
        switch ($904<<24>>24) {
        case 46:  {
         break L222;
         break;
        }
        case 48:  {
         break;
        }
        default: {
         $$19 = $$18;
         break L220;
        }
        }
        HEAP8[$903>>0] = 0;
        $$18 = $903;
       }
       $905 = ($$0629<<24>>24)==(0);
       if ($905) {
        HEAP8[$903>>0] = 0;
        $$19 = $903;
        break;
       } else {
        $906 = ((($$18)) + 1|0);
        HEAP8[$$18>>0] = 48;
        $$19 = $906;
        break;
       }
      }
     } while(0);
     if ($669) {
      $907 = (((16 + (($$06031059$lcssa*6)|0)|0)) + 4|0);
      $908 = HEAP8[$907>>0]|0;
      $909 = $908&255;
      $910 = (160 + ($909)|0);
      $911 = HEAP8[$910>>0]|0;
      $912 = ((($$19)) + 1|0);
      HEAP8[$$19>>0] = $911;
      $913 = ($$6713|0)<(0);
      $914 = ((($$19)) + 2|0);
      if ($913) {
       HEAP8[$912>>0] = 45;
       $915 = (0 - ($$6713))|0;
       $$7714 = $915;
      } else {
       HEAP8[$912>>0] = 43;
       $$7714 = $$6713;
      }
      $916 = ($$7714|0)>(99);
      if ($916) {
       $917 = (($$7714>>>0) / 100)&-1;
       $918 = (($917) + 48)|0;
       $919 = $918&255;
       $920 = ((($$19)) + 3|0);
       HEAP8[$914>>0] = $919;
       $921 = ($917*100)|0;
       $922 = (($$7714) - ($921))|0;
       $$20 = $920;$$8715 = $922;
      } else {
       $$20 = $914;$$8715 = $$7714;
      }
      $923 = (($$8715|0) / 10)&-1;
      $924 = (($923) + 48)|0;
      $925 = $924&255;
      $926 = ((($$20)) + 1|0);
      HEAP8[$$20>>0] = $925;
      $927 = ($923*10)|0;
      $928 = (($$8715) - ($927))|0;
      $929 = (($928) + 48)|0;
      $930 = $929&255;
      $931 = ((($$20)) + 2|0);
      HEAP8[$926>>0] = $930;
      $$21 = $931;
     } else {
      $$21 = $$19;
     }
     HEAP8[$$21>>0] = 0;
     $932 = $$21;
     $933 = $$11$ph;
     $934 = (($932) - ($933))|0;
     $935 = ($$0647<<24>>24)!=(0);
     $936 = ($$2621900<<24>>24)==(0);
     $or$cond12$not = $936 & $935;
     $937 = ($$3610902|0)>($934|0);
     $or$cond730 = $or$cond12$not & $937;
     if ($or$cond730) {
      $938 = (($$3610902) - ($934))|0;
      $939 = ($934|0)<(0);
      if (!($939)) {
       $$06611142 = $$3610902;
       while(1) {
        $940 = (($$06611142) - ($938))|0;
        $941 = (($$11$ph) + ($940)|0);
        $942 = HEAP8[$941>>0]|0;
        $943 = (($$11$ph) + ($$06611142)|0);
        HEAP8[$943>>0] = $942;
        $944 = (($$06611142) + -1)|0;
        $945 = ($$06611142|0)>($938|0);
        if ($945) {
         $$06611142 = $944;
        } else {
         break;
        }
       }
      }
      $946 = ($938|0)==(0);
      if ($946) {
       $$11602 = $$3610902;$$11618 = $$3610902;$$28 = $$11$ph;$$3696 = $$0693;$$8705 = $$3700$ph;
      } else {
       $947 = ($$2672<<24>>24)!=(0);
       $948 = $947&1;
       $scevgep = (($$11$ph) + ($948)|0);
       _memset(($scevgep|0),48,($938|0))|0;
       $$11602 = $$3610902;$$11618 = $$3610902;$$28 = $$11$ph;$$3696 = $$0693;$$8705 = $$3700$ph;
      }
     } else {
      $$11602 = $934;$$11618 = $$3610902;$$28 = $$11$ph;$$3696 = $$0693;$$8705 = $$3700$ph;
     }
    } else {
     $$11602 = 3;$$11618 = $$3610902;$$28 = 31167;$$3696 = $$0693;$$8705 = 0;
    }
    break;
   }
   case 4:  {
    if ($$0669) {
     $949 = HEAP32[$19>>2]|0;
     $arglist_current26 = HEAP32[$2>>2]|0;
     $950 = $arglist_current26;
     $951 = ((0) + 4|0);
     $expanded109 = $951;
     $expanded108 = (($expanded109) - 1)|0;
     $952 = (($950) + ($expanded108))|0;
     $953 = ((0) + 4|0);
     $expanded113 = $953;
     $expanded112 = (($expanded113) - 1)|0;
     $expanded111 = $expanded112 ^ -1;
     $954 = $952 & $expanded111;
     $955 = $954;
     $956 = HEAP32[$955>>2]|0;
     $arglist_next27 = ((($955)) + 4|0);
     HEAP32[$2>>2] = $arglist_next27;
     HEAP32[$956>>2] = $949;
     $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
    } else {
     $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
    }
    break;
   }
   case 7:  {
    HEAP8[$5>>0] = 37;
    $$11602 = 1;$$11618 = $$3610902;$$28 = $5;$$3696 = $$0693;$$8705 = 0;
    break;
   }
   case 8:  {
    L251: do {
     if ($$0669) {
      $arglist_current29 = HEAP32[$2>>2]|0;
      $986 = $arglist_current29;
      $987 = ((0) + 4|0);
      $expanded116 = $987;
      $expanded115 = (($expanded116) - 1)|0;
      $988 = (($986) + ($expanded115))|0;
      $989 = ((0) + 4|0);
      $expanded120 = $989;
      $expanded119 = (($expanded120) - 1)|0;
      $expanded118 = $expanded119 ^ -1;
      $990 = $988 & $expanded118;
      $991 = $990;
      $992 = HEAP32[$991>>2]|0;
      $arglist_next30 = ((($991)) + 4|0);
      HEAP32[$2>>2] = $arglist_next30;
      $993 = ($992>>>0)<(128);
      if ($993) {
       $994 = $992&255;
       HEAP8[$5>>0] = $994;
       $$5596 = 1;
       break;
      }
      $995 = ($992>>>0)<(2048);
      if ($995) {
       $996 = $992 >>> 6;
       $997 = $996 & 31;
       $998 = $997 | 192;
       $999 = $998&255;
       HEAP8[$5>>0] = $999;
       $1000 = $992 & 63;
       $1001 = $1000 | 128;
       $1002 = $1001&255;
       HEAP8[$23>>0] = $1002;
       $$5596 = 2;
       break;
      }
      $1003 = ($992>>>0)<(65536);
      if ($1003) {
       $1004 = $992 >>> 12;
       $1005 = $1004 & 15;
       $1006 = $1005 | 224;
       $1007 = $1006&255;
       HEAP8[$5>>0] = $1007;
       $1008 = $992 >>> 6;
       $1009 = $1008 & 63;
       $1010 = $1009 | 128;
       $1011 = $1010&255;
       HEAP8[$23>>0] = $1011;
       $1012 = $992 & 63;
       $1013 = $1012 | 128;
       $1014 = $1013&255;
       HEAP8[$24>>0] = $1014;
       $$5596 = 3;
       break;
      } else {
       $1015 = $992 >>> 18;
       $1016 = $1015 & 7;
       $1017 = $1016 | 240;
       $1018 = $1017&255;
       HEAP8[$5>>0] = $1018;
       $1019 = $992 >>> 12;
       $1020 = $1019 & 63;
       $1021 = $1020 | 128;
       $1022 = $1021&255;
       HEAP8[$23>>0] = $1022;
       $1023 = $992 >>> 6;
       $1024 = $1023 & 63;
       $1025 = $1024 | 128;
       $1026 = $1025&255;
       HEAP8[$24>>0] = $1026;
       $1027 = $992 & 63;
       $1028 = $1027 | 128;
       $1029 = $1028&255;
       HEAP8[$25>>0] = $1029;
       $$5596 = 4;
       break;
      }
     } else {
      $957 = HEAP32[$$0724>>2]|0;
      $958 = HEAP32[$17>>2]|0;
      $959 = ($957|0)>($958|0);
      do {
       if ($959) {
        $960 = HEAP32[$18>>2]|0;
        $961 = (($958) + 1)|0;
        HEAP32[$17>>2] = $961;
        $962 = (($960) + ($958<<2)|0);
        $963 = HEAP32[$962>>2]|0;
        $964 = ($963|0)==(0|0);
        if (!($964)) {
         $965 = ((($963)) + 8|0);
         $966 = HEAP16[$965>>1]|0;
         $967 = $966 & 514;
         $968 = ($967<<16>>16)==(514);
         if ($968) {
          $969 = ((($963)) + 10|0);
          $970 = HEAP8[$969>>0]|0;
          $971 = ($970<<24>>24)==(1);
          if ($971) {
           $972 = ((($963)) + 16|0);
           $973 = HEAP32[$972>>2]|0;
           $$0$i778 = $973;
          } else {
           label = 335;
          }
         } else {
          label = 335;
         }
         if ((label|0) == 335) {
          label = 0;
          $974 = $966 & 1;
          $975 = ($974<<16>>16)==(0);
          if (!($975)) {
           break;
          }
          $976 = (_valueToText($963,1)|0);
          $$0$i778 = $976;
         }
         $977 = ($$0$i778|0)==(0|0);
         if (!($977)) {
          $978 = HEAP8[$$0$i778>>0]|0;
          HEAP8[$5>>0] = $978;
          $979 = ($978&255)>(191);
          if (!($979)) {
           $$5596 = 1;
           break L251;
          }
          $$221089 = ((($$0$i778)) + 1|0);
          $980 = HEAP8[$$221089>>0]|0;
          $981 = $980 & -64;
          $982 = ($981<<24>>24)==(-128);
          if (!($982)) {
           $$5596 = 1;
           break L251;
          }
          HEAP8[$23>>0] = $980;
          $$221089$1 = ((($$0$i778)) + 2|0);
          $983 = HEAP8[$$221089$1>>0]|0;
          $984 = $983 & -64;
          $985 = ($984<<24>>24)==(-128);
          if (!($985)) {
           $$5596 = 2;
           break L251;
          }
          HEAP8[$24>>0] = $983;
          $$221089$2 = ((($$0$i778)) + 3|0);
          $1556 = HEAP8[$$221089$2>>0]|0;
          $1557 = $1556 & -64;
          $1558 = ($1557<<24>>24)==(-128);
          if (!($1558)) {
           $$5596 = 3;
           break L251;
          }
          HEAP8[$25>>0] = $1556;
          $$5596 = 4;
          break L251;
         }
        }
       }
      } while(0);
      HEAP8[$5>>0] = 0;
      $$5596 = 1;
     }
    } while(0);
    $1030 = ($$4582904|0)>(1);
    if ($1030) {
     $$neg = (1 - ($$4582904))|0;
     $1031 = (($$neg) + ($$3610902))|0;
     $1032 = ($1031|0)<(2);
     $1033 = ($$2621900<<24>>24)!=(0);
     $or$cond15 = $1032 | $1033;
     do {
      if ($or$cond15) {
       $$4611 = $1031;
      } else {
       $1034 = (($1031) + -1)|0;
       $1035 = HEAP32[$19>>2]|0;
       $1036 = ($1034|0)<(0);
       $1037 = $1036 << 31 >> 31;
       $1038 = (_i64Add(($1035|0),0,($1034|0),($1037|0))|0);
       $1039 = (getTempRet0() | 0);
       $1040 = HEAP32[$20>>2]|0;
       $1041 = ($1039|0)<(0);
       $1042 = ($1038>>>0)<($1040>>>0);
       $1043 = ($1039|0)==(0);
       $1044 = $1043 & $1042;
       $1045 = $1041 | $1044;
       if ($1045) {
        $$1$i1090$ph = $1034;
       } else {
        $1046 = (_sqlite3StrAccumEnlarge($0,$1034)|0);
        $1047 = ($1046|0)<(1);
        if ($1047) {
         $$4611 = 0;
         break;
        } else {
         $$1$i1090$ph = $1046;
        }
       }
       $$1$i1090 = $$1$i1090$ph;
       while(1) {
        $1048 = (($$1$i1090) + -1)|0;
        $1049 = HEAP32[$21>>2]|0;
        $1050 = HEAP32[$19>>2]|0;
        $1051 = (($1050) + 1)|0;
        HEAP32[$19>>2] = $1051;
        $1052 = (($1049) + ($1050)|0);
        HEAP8[$1052>>0] = 32;
        $1053 = ($$1$i1090|0)>(1);
        if ($1053) {
         $$1$i1090 = $1048;
        } else {
         $$4611 = 0;
         break;
        }
       }
      }
     } while(0);
     $$115891093 = $$4582904;
     while(1) {
      $1054 = (($$115891093) + -1)|0;
      $1055 = HEAP32[$19>>2]|0;
      $1056 = (($1055) + ($$5596))|0;
      $1057 = HEAP32[$20>>2]|0;
      $1058 = ($1056>>>0)<($1057>>>0);
      if ($1058) {
       HEAP32[$19>>2] = $1056;
       $1059 = HEAP32[$21>>2]|0;
       $1060 = (($1059) + ($1055)|0);
       _memcpy(($1060|0),($5|0),($$5596|0))|0;
      } else {
       _enlargeAndAppend($0,$5,$$5596);
      }
      $1061 = ($$115891093|0)>(2);
      if ($1061) {
       $$115891093 = $1054;
      } else {
       $$25 = $5;$$2631 = 1;$$5702 = 0;$$6613 = $$4611;$$7598 = $$5596;
       label = 395;
       break;
      }
     }
    } else {
     $$25 = $5;$$2631 = 1;$$5702 = 0;$$6613 = $$3610902;$$7598 = $$5596;
     label = 395;
    }
    break;
   }
   case 6: case 5:  {
    do {
     if ($$0669) {
      $arglist_current32 = HEAP32[$2>>2]|0;
      $1082 = $arglist_current32;
      $1083 = ((0) + 4|0);
      $expanded123 = $1083;
      $expanded122 = (($expanded123) - 1)|0;
      $1084 = (($1082) + ($expanded122))|0;
      $1085 = ((0) + 4|0);
      $expanded127 = $1085;
      $expanded126 = (($expanded127) - 1)|0;
      $expanded125 = $expanded126 ^ -1;
      $1086 = $1084 & $expanded125;
      $1087 = $1086;
      $1088 = HEAP32[$1087>>2]|0;
      $arglist_next33 = ((($1087)) + 4|0);
      HEAP32[$2>>2] = $arglist_next33;
      $$23 = $1088;$$2665 = $174;
      label = 369;
     } else {
      $1062 = HEAP32[$$0724>>2]|0;
      $1063 = HEAP32[$17>>2]|0;
      $1064 = ($1062|0)>($1063|0);
      if ($1064) {
       $1065 = HEAP32[$18>>2]|0;
       $1066 = (($1063) + 1)|0;
       HEAP32[$17>>2] = $1066;
       $1067 = (($1065) + ($1063<<2)|0);
       $1068 = HEAP32[$1067>>2]|0;
       $1069 = ($1068|0)==(0|0);
       if ($1069) {
        $$24 = 59952;$$4701 = 0;
        label = 380;
       } else {
        $1070 = ((($1068)) + 8|0);
        $1071 = HEAP16[$1070>>1]|0;
        $1072 = $1071 & 514;
        $1073 = ($1072<<16>>16)==(514);
        if ($1073) {
         $1074 = ((($1068)) + 10|0);
         $1075 = HEAP8[$1074>>0]|0;
         $1076 = ($1075<<24>>24)==(1);
         if ($1076) {
          $1077 = ((($1068)) + 16|0);
          $1078 = HEAP32[$1077>>2]|0;
          $$23 = $1078;$$2665 = 5;
          label = 369;
          break;
         }
        }
        $1079 = $1071 & 1;
        $1080 = ($1079<<16>>16)==(0);
        if ($1080) {
         $1081 = (_valueToText($1068,1)|0);
         $$23 = $1081;$$2665 = 5;
         label = 369;
        } else {
         $$24 = 59952;$$4701 = 0;
         label = 380;
        }
       }
      } else {
       $$24 = 59952;$$4701 = 0;
       label = 380;
      }
     }
    } while(0);
    if ((label|0) == 369) {
     label = 0;
     $1089 = ($$23|0)==(0|0);
     if ($1089) {
      $$24 = 59952;$$4701 = 0;
      label = 380;
     } else {
      $1090 = ($$2665<<24>>24)==(6);
      if ($1090) {
       $1091 = HEAP32[$19>>2]|0;
       $1092 = ($1091|0)==(0);
       if ($1092) {
        $1093 = HEAP32[$26>>2]|0;
        $1094 = ($1093|0)!=(0);
        $1095 = ($$3610902|0)==(0);
        $or$cond17 = $1095 & $1094;
        $1096 = ($$4582904|0)<(0);
        $or$cond19 = $1096 & $or$cond17;
        if ($or$cond19) {
         $1097 = HEAP8[$22>>0]|0;
         $1098 = ($1097<<24>>24)==(0);
         if ($1098) {
          HEAP32[$21>>2] = $$23;
          $1099 = HEAP32[$0>>2]|0;
          $1100 = ($1099|0)==(0|0);
          if ($1100) {
           label = 377;
          } else {
           $1101 = $$23;
           $1102 = ((($1099)) + 304|0);
           $1103 = HEAP32[$1102>>2]|0;
           $1104 = ($1103>>>0)>($1101>>>0);
           if ($1104) {
            label = 377;
           } else {
            $1105 = ((($1099)) + 308|0);
            $1106 = HEAP32[$1105>>2]|0;
            $1107 = ($1106>>>0)>($1101>>>0);
            if ($1107) {
             $1110 = ((($1099)) + 276|0);
             $1111 = HEAP16[$1110>>1]|0;
             $1112 = $1111&65535;
             $$0$i741 = $1112;
            } else {
             label = 377;
            }
           }
          }
          if ((label|0) == 377) {
           label = 0;
           $1108 = HEAP32[(29352)>>2]|0;
           $1109 = (FUNCTION_TABLE_ii[$1108 & 127]($$23)|0);
           $$0$i741 = $1109;
          }
          HEAP32[$20>>2] = $$0$i741;
          $1113 = (_strlen($$23)|0);
          $1114 = $1113 & 2147483647;
          HEAP32[$19>>2] = $1114;
          $1115 = HEAP8[$6>>0]|0;
          $1116 = $1115 | 4;
          HEAP8[$6>>0] = $1116;
          $$11602 = 0;$$11618 = 0;$$28 = $$23;$$3696 = $$0693;$$8705 = 0;
          break L105;
         } else {
          $$24934 = $$23;$$36109021253 = 0;$$4701933 = $$23;
         }
        } else {
         $$24 = $$23;$$4701 = $$23;
         label = 380;
        }
       } else {
        $$24 = $$23;$$4701 = $$23;
        label = 380;
       }
      } else {
       $$24 = $$23;$$4701 = 0;
       label = 380;
      }
     }
    }
    if ((label|0) == 380) {
     label = 0;
     $1117 = ($$4582904|0)>(-1);
     if ($1117) {
      $1118 = ($$0629<<24>>24)==(0);
      $1119 = ($$4582904|0)!=(0);
      if ($1118) {
       if (!($1119)) {
        $$11602 = 0;$$11618 = $$3610902;$$28 = $$24;$$3696 = $$0693;$$8705 = $$4701;
        break L105;
       }
       $$65971081 = 0;
       while(1) {
        $1133 = (($$24) + ($$65971081)|0);
        $1134 = HEAP8[$1133>>0]|0;
        $1135 = ($1134<<24>>24)==(0);
        if ($1135) {
         $$11602 = $$65971081;$$11618 = $$3610902;$$28 = $$24;$$3696 = $$0693;$$8705 = $$4701;
         break L105;
        }
        $1136 = (($$65971081) + 1)|0;
        $1137 = ($1136|0)<($$4582904|0);
        if ($1137) {
         $$65971081 = $1136;
        } else {
         $$11602 = $1136;$$11618 = $$3610902;$$28 = $$24;$$3696 = $$0693;$$8705 = $$4701;
         break L105;
        }
       }
      }
      L330: do {
       if ($1119) {
        $$06441075 = $$24;$$in = $$4582904;
        while(1) {
         $1120 = (($$in) + -1)|0;
         $1121 = HEAP8[$$06441075>>0]|0;
         $1122 = ($1121<<24>>24)==(0);
         if ($1122) {
          $$0644$lcssa = $$06441075;
          break L330;
         }
         $1123 = ((($$06441075)) + 1|0);
         $1124 = ($1121&255)>(191);
         if ($1124) {
          $$1645 = $1123;
          while(1) {
           $1125 = HEAP8[$$1645>>0]|0;
           $1126 = $1125 & -64;
           $1127 = ($1126<<24>>24)==(-128);
           $1128 = ((($$1645)) + 1|0);
           if ($1127) {
            $$1645 = $1128;
           } else {
            $$2646 = $$1645;
            break;
           }
          }
         } else {
          $$2646 = $1123;
         }
         $1129 = ($$in|0)>(1);
         if ($1129) {
          $$06441075 = $$2646;$$in = $1120;
         } else {
          $$0644$lcssa = $$2646;
          break;
         }
        }
       } else {
        $$0644$lcssa = $$24;
       }
      } while(0);
      $1130 = $$0644$lcssa;
      $1131 = $$24;
      $1132 = (($1130) - ($1131))|0;
      $$25 = $$24;$$2631 = $$0629;$$5702 = $$4701;$$6613 = $$3610902;$$7598 = $1132;
      label = 395;
      break L105;
     } else {
      $$24934 = $$24;$$36109021253 = $$3610902;$$4701933 = $$4701;
     }
    }
    $1138 = (_strlen($$24934)|0);
    $1139 = $1138 & 2147483647;
    $$25 = $$24934;$$2631 = $$0629;$$5702 = $$4701933;$$6613 = $$36109021253;$$7598 = $1139;
    label = 395;
    break;
   }
   case 14: case 10: case 9:  {
    $1149 = ($$06031059$lcssa|0)==(6);
    $1150 = $1149 ? 34 : 39;
    do {
     if ($$0669) {
      $arglist_current35 = HEAP32[$2>>2]|0;
      $1171 = $arglist_current35;
      $1172 = ((0) + 4|0);
      $expanded130 = $1172;
      $expanded129 = (($expanded130) - 1)|0;
      $1173 = (($1171) + ($expanded129))|0;
      $1174 = ((0) + 4|0);
      $expanded134 = $1174;
      $expanded133 = (($expanded134) - 1)|0;
      $expanded132 = $expanded133 ^ -1;
      $1175 = $1173 & $expanded132;
      $1176 = $1175;
      $1177 = HEAP32[$1176>>2]|0;
      $arglist_next36 = ((($1176)) + 4|0);
      HEAP32[$2>>2] = $arglist_next36;
      $$0622 = $1177;
     } else {
      $1151 = HEAP32[$$0724>>2]|0;
      $1152 = HEAP32[$17>>2]|0;
      $1153 = ($1151|0)>($1152|0);
      if ($1153) {
       $1154 = HEAP32[$18>>2]|0;
       $1155 = (($1152) + 1)|0;
       HEAP32[$17>>2] = $1155;
       $1156 = (($1154) + ($1152<<2)|0);
       $1157 = HEAP32[$1156>>2]|0;
       $1158 = ($1157|0)==(0|0);
       if ($1158) {
        $$0622 = 0;
       } else {
        $1159 = ((($1157)) + 8|0);
        $1160 = HEAP16[$1159>>1]|0;
        $1161 = $1160 & 514;
        $1162 = ($1161<<16>>16)==(514);
        if ($1162) {
         $1163 = ((($1157)) + 10|0);
         $1164 = HEAP8[$1163>>0]|0;
         $1165 = ($1164<<24>>24)==(1);
         if ($1165) {
          $1166 = ((($1157)) + 16|0);
          $1167 = HEAP32[$1166>>2]|0;
          $$0622 = $1167;
          break;
         }
        }
        $1168 = $1160 & 1;
        $1169 = ($1168<<16>>16)==(0);
        if ($1169) {
         $1170 = (_valueToText($1157,1)|0);
         $$0622 = $1170;
        } else {
         $$0622 = 0;
        }
       }
      } else {
       $$0622 = 0;
      }
     }
    } while(0);
    $1178 = ($$0622|0)==(0|0);
    $1179 = ($$06031059$lcssa|0)==(5);
    $1180 = $1179 ? 31171 : 31176;
    $$1623 = $1178 ? $1180 : $$0622;
    $1181 = ($$4582904|0)==(0);
    L354: do {
     if ($1181) {
      $$0632$lcssa = 0;$$0639$lcssa = 0;
     } else {
      $1182 = $1150&255;
      $1183 = ($$0629<<24>>24)==(0);
      if ($1183) {
       $$06321062 = 0;$$06341061 = $$4582904;$$06391060 = 0;
       while(1) {
        $1199 = (($$1623) + ($$06391060)|0);
        $1200 = HEAP8[$1199>>0]|0;
        $1201 = ($1200<<24>>24)==(0);
        if ($1201) {
         $$0632$lcssa = $$06321062;$$0639$lcssa = $$06391060;
         break L354;
        }
        $1202 = $1200 << 24 >> 24;
        $1203 = ($1202|0)==($1182|0);
        $1204 = $1203&1;
        $spec$select733 = (($$06321062) + ($1204))|0;
        $1205 = (($$06391060) + 1)|0;
        $1206 = (($$06341061) + -1)|0;
        $1207 = ($1206|0)==(0);
        if ($1207) {
         $$0632$lcssa = $spec$select733;$$0639$lcssa = $1205;
         break L354;
        } else {
         $$06321062 = $spec$select733;$$06341061 = $1206;$$06391060 = $1205;
        }
       }
      }
      $$06321062$us = 0;$$06341061$us = $$4582904;$$06391060$us = 0;
      while(1) {
       $1184 = (($$1623) + ($$06391060$us)|0);
       $1185 = HEAP8[$1184>>0]|0;
       $1186 = $1185 << 24 >> 24;
       $1187 = ($1185<<24>>24)==(0);
       if ($1187) {
        $$0632$lcssa = $$06321062$us;$$0639$lcssa = $$06391060$us;
        break L354;
       }
       $1188 = ($1186|0)==($1182|0);
       $1189 = $1188&1;
       $spec$select733$us = (($$06321062$us) + ($1189))|0;
       $1190 = $1186 & 192;
       $1191 = ($1190|0)==(192);
       if ($1191) {
        $$1640$us = $$06391060$us;
        while(1) {
         $1192 = (($$1640$us) + 1)|0;
         $1193 = (($$1623) + ($1192)|0);
         $1194 = HEAP8[$1193>>0]|0;
         $1195 = $1194 & -64;
         $1196 = ($1195<<24>>24)==(-128);
         if ($1196) {
          $$1640$us = $1192;
         } else {
          $$pre$phi1239Z2D = $1192;
          break;
         }
        }
       } else {
        $$pre1238 = (($$06391060$us) + 1)|0;
        $$pre$phi1239Z2D = $$pre1238;
       }
       $1197 = (($$06341061$us) + -1)|0;
       $1198 = ($1197|0)==(0);
       if ($1198) {
        $$0632$lcssa = $spec$select733$us;$$0639$lcssa = $$pre$phi1239Z2D;
        break;
       } else {
        $$06321062$us = $spec$select733$us;$$06341061$us = $1197;$$06391060$us = $$pre$phi1239Z2D;
       }
      }
     }
    } while(0);
    $not$ = $1178 ^ 1;
    $1208 = $1179 & $not$;
    $1209 = (($$0639$lcssa) + 3)|0;
    $1210 = (($1209) + ($$0632$lcssa))|0;
    $1211 = ($1210|0)>(70);
    L372: do {
     if ($1211) {
      $1212 = ($1210|0)<(0);
      $1213 = $1212 << 31 >> 31;
      $1214 = HEAP8[$22>>0]|0;
      $1215 = ($1214<<24>>24)==(0);
      if (!($1215)) {
       label = 543;
       break L4;
      }
      $1216 = HEAP32[$20>>2]|0;
      $1217 = (0)<($1213|0);
      $1218 = ($1216>>>0)<($1210>>>0);
      $1219 = (0)==($1213|0);
      $1220 = $1219 & $1218;
      $1221 = $1217 | $1220;
      if ($1221) {
       $1222 = HEAP32[$26>>2]|0;
       $1223 = (0)<($1213|0);
       $1224 = ($1222>>>0)<($1210>>>0);
       $1225 = (0)==($1213|0);
       $1226 = $1225 & $1224;
       $1227 = $1223 | $1226;
       if ($1227) {
        label = 423;
        break L4;
       }
      }
      $1269 = HEAP32[$0>>2]|0;
      $1270 = ($1269|0)==(0|0);
      L378: do {
       if ($1270) {
        $1304 = (_i64Add(($1210|0),($1213|0),-1,-1)|0);
        $1305 = (getTempRet0() | 0);
        $1306 = ($1305>>>0)>(0);
        $1307 = ($1304>>>0)>(2147483390);
        $1308 = ($1305|0)==(0);
        $1309 = $1308 & $1307;
        $1310 = $1306 | $1309;
        if ($1310) {
         label = 465;
         break L4;
        }
        $1311 = HEAP32[7324]|0;
        $1312 = ($1311|0)==(0);
        if ($1312) {
         $1351 = HEAP32[(29340)>>2]|0;
         $1352 = (FUNCTION_TABLE_ii[$1351 & 127]($1210)|0);
         $$0$i798 = $1352;
         break;
        }
        $1313 = HEAP32[(29356)>>2]|0;
        $1314 = (FUNCTION_TABLE_ii[$1313 & 127]($1210)|0);
        $1315 = HEAP32[14985]|0;
        $1316 = ($1315>>>0)<($1210>>>0);
        if ($1316) {
         HEAP32[14985] = $1210;
        }
        $1317 = (59064);
        $1318 = $1317;
        $1319 = HEAP32[$1318>>2]|0;
        $1320 = (($1317) + 4)|0;
        $1321 = $1320;
        $1322 = HEAP32[$1321>>2]|0;
        $1323 = ($1322|0)>(0);
        $1324 = ($1319>>>0)>(0);
        $1325 = ($1322|0)==(0);
        $1326 = $1325 & $1324;
        $1327 = $1323 | $1326;
        if ($1327) {
         $1328 = HEAP32[14978]|0;
         $1329 = ($1314|0)<(0);
         $1330 = $1329 << 31 >> 31;
         $1331 = (_i64Subtract(($1319|0),($1322|0),($1314|0),($1330|0))|0);
         $1332 = (getTempRet0() | 0);
         $1333 = ($1332|0)<(0);
         $1334 = ($1331>>>0)<=($1328>>>0);
         $1335 = ($1332|0)==(0);
         $1336 = $1335 & $1334;
         $1337 = $1333 | $1336;
         $$1703 = $1337&1;
         HEAP32[(59072)>>2] = $$1703;
        }
        $1338 = HEAP32[(29340)>>2]|0;
        $1339 = (FUNCTION_TABLE_ii[$1338 & 127]($1314)|0);
        $1340 = ($1339|0)==(0|0);
        if ($1340) {
         label = 465;
         break L4;
        }
        $1341 = HEAP32[(29352)>>2]|0;
        $1342 = (FUNCTION_TABLE_ii[$1341 & 127]($1339)|0);
        $1343 = HEAP32[14978]|0;
        $1344 = (($1343) + ($1342))|0;
        HEAP32[14978] = $1344;
        $1345 = HEAP32[14982]|0;
        $1346 = ($1344>>>0)>($1345>>>0);
        if ($1346) {
         HEAP32[14982] = $1344;
        }
        $1347 = HEAP32[14981]|0;
        $1348 = (($1347) + 1)|0;
        HEAP32[14981] = $1348;
        $1349 = HEAP32[14987]|0;
        $1350 = ($1348>>>0)>($1349>>>0);
        if (!($1350)) {
         $$26 = $1339;$$6703 = $1339;
         break L372;
        }
        HEAP32[14987] = $1348;
        $$26 = $1339;$$6703 = $1339;
        break L372;
       } else {
        $1271 = ((($1269)) + 272|0);
        $1272 = HEAP32[$1271>>2]|0;
        $1273 = ($1272|0)==(0);
        if ($1273) {
         $1274 = ((($1269)) + 276|0);
         $1275 = HEAP16[$1274>>1]|0;
         $1276 = $1275&65535;
         $1277 = (0)<($1213>>>0);
         $1278 = ($1276>>>0)<($1210>>>0);
         $1279 = (0)==($1213|0);
         $1280 = $1279 & $1278;
         $1281 = $1277 | $1280;
         do {
          if ($1281) {
           $1282 = ((($1269)) + 288|0);
           $$sink1699 = $1282;
          } else {
           $1283 = ((($1269)) + 300|0);
           $1284 = HEAP32[$1283>>2]|0;
           $1285 = ($1284|0)==(0|0);
           if (!($1285)) {
            $1286 = HEAP32[$1284>>2]|0;
            HEAP32[$1283>>2] = $1286;
            $1287 = ((($1269)) + 284|0);
            $1288 = HEAP32[$1287>>2]|0;
            $1289 = (($1288) + 1)|0;
            HEAP32[$1287>>2] = $1289;
            $$0$i798 = $1284;
            break L378;
           }
           $1290 = ((($1269)) + 296|0);
           $1291 = HEAP32[$1290>>2]|0;
           $1292 = ($1291|0)==(0|0);
           if ($1292) {
            $1297 = ((($1269)) + 292|0);
            $$sink1699 = $1297;
            break;
           } else {
            $1293 = HEAP32[$1291>>2]|0;
            HEAP32[$1290>>2] = $1293;
            $1294 = ((($1269)) + 284|0);
            $1295 = HEAP32[$1294>>2]|0;
            $1296 = (($1295) + 1)|0;
            HEAP32[$1294>>2] = $1296;
            $$0$i798 = $1291;
            break L378;
           }
          }
         } while(0);
         $1301 = HEAP32[$$sink1699>>2]|0;
         $1302 = (($1301) + 1)|0;
         HEAP32[$$sink1699>>2] = $1302;
        } else {
         $1298 = ((($1269)) + 81|0);
         $1299 = HEAP8[$1298>>0]|0;
         $1300 = ($1299<<24>>24)==(0);
         if (!($1300)) {
          label = 465;
          break L4;
         }
        }
        $1303 = (_dbMallocRawFinish($1269,$1210,$1213)|0);
        $$0$i798 = $1303;
       }
      } while(0);
      $1353 = ($$0$i798|0)==(0|0);
      if ($1353) {
       label = 465;
       break L4;
      } else {
       $$26 = $$0$i798;$$6703 = $$0$i798;
      }
     } else {
      $$26 = $5;$$6703 = 0;
     }
    } while(0);
    if ($1208) {
     HEAP8[$$26>>0] = $1150;
     $$0635 = 1;
    } else {
     $$0635 = 0;
    }
    $1387 = ($$0639$lcssa|0)>(0);
    if ($1387) {
     $1388 = $1150&255;
     $$16361070 = $$0635;$$36421069 = 0;
     while(1) {
      $1389 = (($$1623) + ($$36421069)|0);
      $1390 = HEAP8[$1389>>0]|0;
      $1391 = (($$16361070) + 1)|0;
      $1392 = (($$26) + ($$16361070)|0);
      HEAP8[$1392>>0] = $1390;
      $1393 = $1390 << 24 >> 24;
      $1394 = ($1393|0)==($1388|0);
      if ($1394) {
       $1395 = (($$26) + ($1391)|0);
       $1396 = (($$16361070) + 2)|0;
       HEAP8[$1395>>0] = $1390;
       $$2637 = $1396;
      } else {
       $$2637 = $1391;
      }
      $1397 = (($$36421069) + 1)|0;
      $exitcond = ($1397|0)==($$0639$lcssa|0);
      if ($exitcond) {
       $$1636$lcssa = $$2637;
       break;
      } else {
       $$16361070 = $$2637;$$36421069 = $1397;
      }
     }
    } else {
     $$1636$lcssa = $$0635;
    }
    if ($1208) {
     $1398 = (($$26) + ($$1636$lcssa)|0);
     $1399 = (($$1636$lcssa) + 1)|0;
     HEAP8[$1398>>0] = $1150;
     $$3638 = $1399;
    } else {
     $$3638 = $$1636$lcssa;
    }
    $1400 = (($$26) + ($$3638)|0);
    HEAP8[$1400>>0] = 0;
    $$25 = $$26;$$2631 = $$0629;$$5702 = $$6703;$$6613 = $$3610902;$$7598 = $$3638;
    label = 395;
    break;
   }
   case 11:  {
    $1401 = HEAP8[$6>>0]|0;
    $1402 = $1401 & 1;
    $1403 = ($1402<<24>>24)==(0);
    if ($1403) {
     label = 543;
     break L4;
    }
    $arglist_current38 = HEAP32[$2>>2]|0;
    $1404 = $arglist_current38;
    $1405 = ((0) + 4|0);
    $expanded137 = $1405;
    $expanded136 = (($expanded137) - 1)|0;
    $1406 = (($1404) + ($expanded136))|0;
    $1407 = ((0) + 4|0);
    $expanded141 = $1407;
    $expanded140 = (($expanded141) - 1)|0;
    $expanded139 = $expanded140 ^ -1;
    $1408 = $1406 & $expanded139;
    $1409 = $1408;
    $1410 = HEAP32[$1409>>2]|0;
    $arglist_next39 = ((($1409)) + 4|0);
    HEAP32[$2>>2] = $arglist_next39;
    $1411 = ($1410|0)==(0|0);
    if ($1411) {
     $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
    } else {
     $1412 = ((($1410)) + 4|0);
     $1413 = HEAP32[$1412>>2]|0;
     $1414 = ($1413|0)==(0);
     if ($1414) {
      $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
     } else {
      $1415 = HEAP32[$1410>>2]|0;
      $1416 = HEAP32[$19>>2]|0;
      $1417 = (($1416) + ($1413))|0;
      $1418 = HEAP32[$20>>2]|0;
      $1419 = ($1417>>>0)<($1418>>>0);
      if ($1419) {
       HEAP32[$19>>2] = $1417;
       $1420 = HEAP32[$21>>2]|0;
       $1421 = (($1420) + ($1416)|0);
       _memcpy(($1421|0),($1415|0),($1413|0))|0;
       $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
       break L105;
      } else {
       _enlargeAndAppend($0,$1415,$1413);
       $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
       break L105;
      }
     }
    }
    break;
   }
   case 12:  {
    $1422 = HEAP8[$6>>0]|0;
    $1423 = $1422 & 1;
    $1424 = ($1423<<24>>24)==(0);
    if ($1424) {
     label = 543;
     break L4;
    }
    $arglist_current41 = HEAP32[$2>>2]|0;
    $1425 = $arglist_current41;
    $1426 = ((0) + 4|0);
    $expanded144 = $1426;
    $expanded143 = (($expanded144) - 1)|0;
    $1427 = (($1425) + ($expanded143))|0;
    $1428 = ((0) + 4|0);
    $expanded148 = $1428;
    $expanded147 = (($expanded148) - 1)|0;
    $expanded146 = $expanded147 ^ -1;
    $1429 = $1427 & $expanded146;
    $1430 = $1429;
    $1431 = HEAP32[$1430>>2]|0;
    $arglist_next42 = ((($1430)) + 4|0);
    HEAP32[$2>>2] = $arglist_next42;
    $arglist_current44 = HEAP32[$2>>2]|0;
    $1432 = $arglist_current44;
    $1433 = ((0) + 4|0);
    $expanded151 = $1433;
    $expanded150 = (($expanded151) - 1)|0;
    $1434 = (($1432) + ($expanded150))|0;
    $1435 = ((0) + 4|0);
    $expanded155 = $1435;
    $expanded154 = (($expanded155) - 1)|0;
    $expanded153 = $expanded154 ^ -1;
    $1436 = $1434 & $expanded153;
    $1437 = $1436;
    $1438 = HEAP32[$1437>>2]|0;
    $arglist_next45 = ((($1437)) + 4|0);
    HEAP32[$2>>2] = $arglist_next45;
    $1439 = (((((($1431)) + 8|0) + (($1438*72)|0)|0)) + 4|0);
    $1440 = HEAP32[$1439>>2]|0;
    $1441 = ($1440|0)==(0|0);
    do {
     if (!($1441)) {
      $1442 = (_strlen($1440)|0);
      $1443 = $1442 & 1073741823;
      $1444 = HEAP32[$19>>2]|0;
      $1445 = (($1444) + ($1443))|0;
      $1446 = HEAP32[$20>>2]|0;
      $1447 = ($1445>>>0)<($1446>>>0);
      if ($1447) {
       $1448 = ($1443|0)==(0);
       if (!($1448)) {
        HEAP32[$19>>2] = $1445;
        $1449 = HEAP32[$21>>2]|0;
        $1450 = (($1449) + ($1444)|0);
        _memcpy(($1450|0),($1440|0),($1443|0))|0;
       }
      } else {
       _enlargeAndAppend($0,$1440,$1443);
      }
      $1451 = HEAP32[$19>>2]|0;
      $1452 = (($1451) + 1)|0;
      $1453 = HEAP32[$20>>2]|0;
      $1454 = ($1452>>>0)<($1453>>>0);
      if ($1454) {
       HEAP32[$19>>2] = $1452;
       $1455 = HEAP32[$21>>2]|0;
       $1456 = (($1455) + ($1451)|0);
       HEAP8[$1456>>0] = 46;
       break;
      } else {
       _enlargeAndAppend($0,31183,1);
       break;
      }
     }
    } while(0);
    $1457 = (((((($1431)) + 8|0) + (($1438*72)|0)|0)) + 8|0);
    $1458 = HEAP32[$1457>>2]|0;
    $1459 = ($1458|0)==(0|0);
    if ($1459) {
     $$0$i$i787 = 0;
    } else {
     $1460 = (_strlen($1458)|0);
     $1461 = $1460 & 1073741823;
     $$0$i$i787 = $1461;
    }
    $1462 = HEAP32[$19>>2]|0;
    $1463 = (($1462) + ($$0$i$i787))|0;
    $1464 = HEAP32[$20>>2]|0;
    $1465 = ($1463>>>0)<($1464>>>0);
    if (!($1465)) {
     _enlargeAndAppend($0,$1458,$$0$i$i787);
     $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
     break L105;
    }
    $1466 = ($$0$i$i787|0)==(0);
    if ($1466) {
     $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
    } else {
     HEAP32[$19>>2] = $1463;
     $1467 = HEAP32[$21>>2]|0;
     $1468 = (($1467) + ($1462)|0);
     _memcpy(($1468|0),($1458|0),($$0$i$i787|0))|0;
     $$11602 = 0;$$11618 = 0;$$28 = $$1568;$$3696 = $$0693;$$8705 = 0;
    }
    break;
   }
   default: {
    label = 543;
    break L4;
   }
   }
  } while(0);
  if ((label|0) == 86) {
   label = 0;
   $$2659 = 0;$$4653 = $$3652;
   label = 87;
  }
  else if ((label|0) == 395) {
   label = 0;
   $1140 = ($$2631<<24>>24)!=(0);
   $1141 = ($$6613|0)>(0);
   $or$cond22 = $1140 & $1141;
   $1142 = ($$7598|0)>(0);
   $or$cond1597 = $or$cond22 & $1142;
   if ($or$cond1597) {
    $$06431097$in = $$7598;$$76141096 = $$6613;
    while(1) {
     $$06431097 = (($$06431097$in) + -1)|0;
     $1143 = (($$25) + ($$06431097)|0);
     $1144 = HEAP8[$1143>>0]|0;
     $1145 = $1144 & -64;
     $1146 = ($1145<<24>>24)==(-128);
     $1147 = $1146&1;
     $spec$select732 = (($$76141096) + ($1147))|0;
     $1148 = ($$06431097$in|0)>(1);
     if ($1148) {
      $$06431097$in = $$06431097;$$76141096 = $spec$select732;
     } else {
      $$11602 = $$7598;$$11618 = $spec$select732;$$28 = $$25;$$3696 = $$0693;$$8705 = $$5702;
      break;
     }
    }
   } else {
    $$11602 = $$7598;$$11618 = $$6613;$$28 = $$25;$$3696 = $$0693;$$8705 = $$5702;
   }
  }
  if ((label|0) == 87) {
   label = 0;
   $175 = 4067322 >>> $$06031059$lcssa;
   $176 = $175 & 1;
   $177 = ($176|0)==(0);
   L461: do {
    if ($177) {
     do {
      if ($$0669) {
       $cond = ($$4653<<24>>24)==(2);
       if ($cond) {
        $arglist_current8 = HEAP32[$2>>2]|0;
        $205 = $arglist_current8;
        $206 = ((0) + 8|0);
        $expanded67 = $206;
        $expanded66 = (($expanded67) - 1)|0;
        $207 = (($205) + ($expanded66))|0;
        $208 = ((0) + 8|0);
        $expanded71 = $208;
        $expanded70 = (($expanded71) - 1)|0;
        $expanded69 = $expanded70 ^ -1;
        $209 = $207 & $expanded69;
        $210 = $209;
        $211 = $210;
        $212 = $211;
        $213 = HEAP32[$212>>2]|0;
        $214 = (($211) + 4)|0;
        $215 = $214;
        $216 = HEAP32[$215>>2]|0;
        $arglist_next9 = ((($210)) + 8|0);
        HEAP32[$2>>2] = $arglist_next9;
        $230 = $216;$232 = $213;
        label = 99;
        break;
       } else {
        $arglist_current11 = HEAP32[$2>>2]|0;
        $217 = $arglist_current11;
        $218 = ((0) + 4|0);
        $expanded74 = $218;
        $expanded73 = (($expanded74) - 1)|0;
        $219 = (($217) + ($expanded73))|0;
        $220 = ((0) + 4|0);
        $expanded78 = $220;
        $expanded77 = (($expanded78) - 1)|0;
        $expanded76 = $expanded77 ^ -1;
        $221 = $219 & $expanded76;
        $222 = $221;
        $223 = HEAP32[$222>>2]|0;
        $arglist_next12 = ((($222)) + 4|0);
        HEAP32[$2>>2] = $arglist_next12;
        $224 = ($223|0)<(0);
        $225 = $224 << 31 >> 31;
        $226 = ($223|0)<(0);
        $227 = (_i64Subtract(0,0,($223|0),($225|0))|0);
        $228 = (getTempRet0() | 0);
        if ($226) {
         $1562 = $227;$1563 = $228;
         break;
        } else {
         $$1671 = $$0624;$293 = $223;$295 = $225;
         label = 113;
         break L461;
        }
       }
      } else {
       $178 = HEAP32[$$0724>>2]|0;
       $179 = HEAP32[$17>>2]|0;
       $180 = ($178|0)>($179|0);
       if (!($180)) {
        $$1671917 = $$0624;
        label = 114;
        break L461;
       }
       $181 = HEAP32[$18>>2]|0;
       $182 = (($179) + 1)|0;
       HEAP32[$17>>2] = $182;
       $183 = (($181) + ($179<<2)|0);
       $184 = HEAP32[$183>>2]|0;
       $185 = ((($184)) + 8|0);
       $186 = HEAP16[$185>>1]|0;
       $187 = $186&65535;
       $188 = $187 & 4;
       $189 = ($188|0)==(0);
       if (!($189)) {
        $190 = $184;
        $191 = $190;
        $192 = HEAP32[$191>>2]|0;
        $193 = (($190) + 4)|0;
        $194 = $193;
        $195 = HEAP32[$194>>2]|0;
        $230 = $195;$232 = $192;
        label = 99;
        break;
       }
       $196 = $187 & 8;
       $197 = ($196|0)==(0);
       if (!($197)) {
        $198 = +HEAPF64[$184>>3];
        $199 = (_doubleToInt64($198)|0);
        $200 = (getTempRet0() | 0);
        $230 = $200;$232 = $199;
        label = 99;
        break;
       }
       $201 = $187 & 18;
       $202 = ($201|0)==(0);
       if ($202) {
        $$1671917 = $$0624;
        label = 114;
        break L461;
       }
       $$idx$i$i$i753 = ((($184)) + 10|0);
       $$idx$val$i$i$i754 = HEAP8[$$idx$i$i$i753>>0]|0;
       $$idx8$i$i$i755 = ((($184)) + 12|0);
       $$idx8$val$i$i$i756 = HEAP32[$$idx8$i$i$i755>>2]|0;
       $$idx9$i$i$i757 = ((($184)) + 16|0);
       $$idx9$val$i$i$i758 = HEAP32[$$idx9$i$i$i757>>2]|0;
       $203 = (_memIntValue($$idx$val$i$i$i754,$$idx8$val$i$i$i756,$$idx9$val$i$i$i758)|0);
       $204 = (getTempRet0() | 0);
       $230 = $204;$232 = $203;
       label = 99;
      }
     } while(0);
     if ((label|0) == 99) {
      label = 0;
      $229 = ($230|0)<(0);
      $231 = ($232|0)==(0);
      $233 = ($230|0)==(-2147483648);
      $234 = $231 & $233;
      $235 = (_i64Subtract(0,0,($232|0),($230|0))|0);
      $236 = (getTempRet0() | 0);
      $237 = $234 ? 0 : $235;
      $238 = $234 ? -2147483648 : $236;
      if ($229) {
       $1562 = $237;$1563 = $238;
      } else {
       $$1671 = $$0624;$293 = $232;$295 = $230;
       label = 113;
       break;
      }
     }
     $$1671 = 45;$293 = $1562;$295 = $1563;
     label = 113;
    } else {
     if ($$0669) {
      switch ($$4653<<24>>24) {
      case 0:  {
       $arglist_current20 = HEAP32[$2>>2]|0;
       $285 = $arglist_current20;
       $286 = ((0) + 4|0);
       $expanded95 = $286;
       $expanded94 = (($expanded95) - 1)|0;
       $287 = (($285) + ($expanded94))|0;
       $288 = ((0) + 4|0);
       $expanded99 = $288;
       $expanded98 = (($expanded99) - 1)|0;
       $expanded97 = $expanded98 ^ -1;
       $289 = $287 & $expanded97;
       $290 = $289;
       $291 = HEAP32[$290>>2]|0;
       $arglist_next21 = ((($290)) + 4|0);
       HEAP32[$2>>2] = $arglist_next21;
       $$1671 = 0;$293 = $291;$295 = 0;
       label = 113;
       break L461;
       break;
      }
      case 2:  {
       $arglist_current14 = HEAP32[$2>>2]|0;
       $266 = $arglist_current14;
       $267 = ((0) + 8|0);
       $expanded81 = $267;
       $expanded80 = (($expanded81) - 1)|0;
       $268 = (($266) + ($expanded80))|0;
       $269 = ((0) + 8|0);
       $expanded85 = $269;
       $expanded84 = (($expanded85) - 1)|0;
       $expanded83 = $expanded84 ^ -1;
       $270 = $268 & $expanded83;
       $271 = $270;
       $272 = $271;
       $273 = $272;
       $274 = HEAP32[$273>>2]|0;
       $275 = (($272) + 4)|0;
       $276 = $275;
       $277 = HEAP32[$276>>2]|0;
       $arglist_next15 = ((($271)) + 8|0);
       HEAP32[$2>>2] = $arglist_next15;
       $$1671 = 0;$293 = $274;$295 = $277;
       label = 113;
       break L461;
       break;
      }
      default: {
       $arglist_current17 = HEAP32[$2>>2]|0;
       $278 = $arglist_current17;
       $279 = ((0) + 4|0);
       $expanded88 = $279;
       $expanded87 = (($expanded88) - 1)|0;
       $280 = (($278) + ($expanded87))|0;
       $281 = ((0) + 4|0);
       $expanded92 = $281;
       $expanded91 = (($expanded92) - 1)|0;
       $expanded90 = $expanded91 ^ -1;
       $282 = $280 & $expanded90;
       $283 = $282;
       $284 = HEAP32[$283>>2]|0;
       $arglist_next18 = ((($283)) + 4|0);
       HEAP32[$2>>2] = $arglist_next18;
       $$1671 = 0;$293 = $284;$295 = 0;
       label = 113;
       break L461;
      }
      }
     }
     $239 = HEAP32[$$0724>>2]|0;
     $240 = HEAP32[$17>>2]|0;
     $241 = ($239|0)>($240|0);
     if ($241) {
      $242 = HEAP32[$18>>2]|0;
      $243 = (($240) + 1)|0;
      HEAP32[$17>>2] = $243;
      $244 = (($242) + ($240<<2)|0);
      $245 = HEAP32[$244>>2]|0;
      $246 = ((($245)) + 8|0);
      $247 = HEAP16[$246>>1]|0;
      $248 = $247&65535;
      $249 = $248 & 4;
      $250 = ($249|0)==(0);
      if (!($250)) {
       $251 = $245;
       $252 = $251;
       $253 = HEAP32[$252>>2]|0;
       $254 = (($251) + 4)|0;
       $255 = $254;
       $256 = HEAP32[$255>>2]|0;
       $$1671 = 0;$293 = $253;$295 = $256;
       label = 113;
       break;
      }
      $257 = $248 & 8;
      $258 = ($257|0)==(0);
      if (!($258)) {
       $259 = +HEAPF64[$245>>3];
       $260 = (_doubleToInt64($259)|0);
       $261 = (getTempRet0() | 0);
       $$1671 = 0;$293 = $260;$295 = $261;
       label = 113;
       break;
      }
      $262 = $248 & 18;
      $263 = ($262|0)==(0);
      if ($263) {
       $$1671917 = 0;
       label = 114;
      } else {
       $$idx$i$i$i750 = ((($245)) + 10|0);
       $$idx$val$i$i$i751 = HEAP8[$$idx$i$i$i750>>0]|0;
       $$idx8$i$i$i = ((($245)) + 12|0);
       $$idx8$val$i$i$i = HEAP32[$$idx8$i$i$i>>2]|0;
       $$idx9$i$i$i = ((($245)) + 16|0);
       $$idx9$val$i$i$i = HEAP32[$$idx9$i$i$i>>2]|0;
       $264 = (_memIntValue($$idx$val$i$i$i751,$$idx8$val$i$i$i,$$idx9$val$i$i$i)|0);
       $265 = (getTempRet0() | 0);
       $$1671 = 0;$293 = $264;$295 = $265;
       label = 113;
      }
     } else {
      $$1671917 = 0;
      label = 114;
     }
    }
   } while(0);
   if ((label|0) == 113) {
    label = 0;
    $292 = ($293|0)==(0);
    $294 = ($295|0)==(0);
    $296 = $292 & $294;
    if ($296) {
     $$1671917 = $$1671;
     label = 114;
    } else {
     $$1671916 = $$1671;$490 = $293;$491 = $295;$565 = $$0626;
    }
   }
   if ((label|0) == 114) {
    label = 0;
    $$1671916 = $$1671917;$490 = 0;$491 = 0;$565 = 0;
   }
   $297 = ($$0647<<24>>24)==(0);
   $298 = ($$1671916<<24>>24)!=(0);
   $299 = $298&1;
   $300 = (($$3610902) - ($299))|0;
   $301 = ($$4582904|0)<($300|0);
   $spec$select726 = $301 ? $300 : $$4582904;
   $spec$select941 = $297 ? $$4582904 : $spec$select726;
   $302 = ($spec$select941|0)<(37);
   if ($302) {
    $$0692 = $5;$$1698 = 0;$$2695 = 70;
   } else {
    $303 = ($spec$select941|0)<(0);
    $304 = $303 << 31 >> 31;
    $305 = (_i64Add(($spec$select941|0),($304|0),10,0)|0);
    $306 = (getTempRet0() | 0);
    $307 = ($$2659<<24>>24)==(0);
    if ($307) {
     $315 = $306;$317 = $305;
    } else {
     $308 = (($spec$select941>>>0) / 3)&-1;
     $309 = (_i64Add(($305|0),($306|0),($308|0),0)|0);
     $310 = (getTempRet0() | 0);
     $315 = $310;$317 = $309;
    }
    $311 = HEAP8[$22>>0]|0;
    $312 = ($311<<24>>24)==(0);
    if (!($312)) {
     label = 543;
     break;
    }
    $313 = HEAP32[$20>>2]|0;
    $314 = ($315|0)>(0);
    $316 = ($317>>>0)>($313>>>0);
    $318 = ($315|0)==(0);
    $319 = $318 & $316;
    $320 = $314 | $319;
    if ($320) {
     $321 = HEAP32[$26>>2]|0;
     $322 = ($315|0)>(0);
     $323 = ($317>>>0)>($321>>>0);
     $324 = ($315|0)==(0);
     $325 = $324 & $323;
     $326 = $322 | $325;
     if ($326) {
      label = 121;
      break;
     }
    }
    $368 = HEAP32[$0>>2]|0;
    $369 = ($368|0)==(0|0);
    L510: do {
     if ($369) {
      $403 = (_i64Add(($317|0),($315|0),-1,-1)|0);
      $404 = (getTempRet0() | 0);
      $405 = ($404>>>0)>(0);
      $406 = ($403>>>0)>(2147483390);
      $407 = ($404|0)==(0);
      $408 = $407 & $406;
      $409 = $405 | $408;
      if ($409) {
       label = 164;
       break L4;
      }
      $410 = HEAP32[7324]|0;
      $411 = ($410|0)==(0);
      if ($411) {
       $450 = HEAP32[(29340)>>2]|0;
       $451 = (FUNCTION_TABLE_ii[$450 & 127]($317)|0);
       $$0$i829 = $451;
       label = 162;
       break;
      }
      $412 = HEAP32[(29356)>>2]|0;
      $413 = (FUNCTION_TABLE_ii[$412 & 127]($317)|0);
      $414 = HEAP32[14985]|0;
      $415 = ($414>>>0)<($317>>>0);
      if ($415) {
       HEAP32[14985] = $317;
      }
      $416 = (59064);
      $417 = $416;
      $418 = HEAP32[$417>>2]|0;
      $419 = (($416) + 4)|0;
      $420 = $419;
      $421 = HEAP32[$420>>2]|0;
      $422 = ($421|0)>(0);
      $423 = ($418>>>0)>(0);
      $424 = ($421|0)==(0);
      $425 = $424 & $423;
      $426 = $422 | $425;
      if ($426) {
       $427 = HEAP32[14978]|0;
       $428 = ($413|0)<(0);
       $429 = $428 << 31 >> 31;
       $430 = (_i64Subtract(($418|0),($421|0),($413|0),($429|0))|0);
       $431 = (getTempRet0() | 0);
       $432 = ($431|0)<(0);
       $433 = ($430>>>0)<=($427>>>0);
       $434 = ($431|0)==(0);
       $435 = $434 & $433;
       $436 = $432 | $435;
       $$ = $436&1;
       HEAP32[(59072)>>2] = $$;
      }
      $437 = HEAP32[(29340)>>2]|0;
      $438 = (FUNCTION_TABLE_ii[$437 & 127]($413)|0);
      $439 = ($438|0)==(0|0);
      if ($439) {
       label = 164;
       break L4;
      }
      $440 = HEAP32[(29352)>>2]|0;
      $441 = (FUNCTION_TABLE_ii[$440 & 127]($438)|0);
      $442 = HEAP32[14978]|0;
      $443 = (($442) + ($441))|0;
      HEAP32[14978] = $443;
      $444 = HEAP32[14982]|0;
      $445 = ($443>>>0)>($444>>>0);
      if ($445) {
       HEAP32[14982] = $443;
      }
      $446 = HEAP32[14981]|0;
      $447 = (($446) + 1)|0;
      HEAP32[14981] = $447;
      $448 = HEAP32[14987]|0;
      $449 = ($447>>>0)>($448>>>0);
      if ($449) {
       HEAP32[14987] = $447;
       $$0$i748 = $438;$$pre$phi1237Z2D = $317;
      } else {
       $$0$i748 = $438;$$pre$phi1237Z2D = $317;
      }
     } else {
      $370 = ((($368)) + 272|0);
      $371 = HEAP32[$370>>2]|0;
      $372 = ($371|0)==(0);
      if ($372) {
       $373 = ((($368)) + 276|0);
       $374 = HEAP16[$373>>1]|0;
       $375 = $374&65535;
       $376 = ($315>>>0)>(0);
       $377 = ($317>>>0)>($375>>>0);
       $378 = ($315|0)==(0);
       $379 = $378 & $377;
       $380 = $376 | $379;
       do {
        if ($380) {
         $381 = ((($368)) + 288|0);
         $$sink = $381;
        } else {
         $382 = ((($368)) + 300|0);
         $383 = HEAP32[$382>>2]|0;
         $384 = ($383|0)==(0|0);
         if (!($384)) {
          $385 = HEAP32[$383>>2]|0;
          HEAP32[$382>>2] = $385;
          $386 = ((($368)) + 284|0);
          $387 = HEAP32[$386>>2]|0;
          $388 = (($387) + 1)|0;
          HEAP32[$386>>2] = $388;
          $$0$i829 = $383;
          label = 162;
          break L510;
         }
         $389 = ((($368)) + 296|0);
         $390 = HEAP32[$389>>2]|0;
         $391 = ($390|0)==(0|0);
         if ($391) {
          $396 = ((($368)) + 292|0);
          $$sink = $396;
          break;
         } else {
          $392 = HEAP32[$390>>2]|0;
          HEAP32[$389>>2] = $392;
          $393 = ((($368)) + 284|0);
          $394 = HEAP32[$393>>2]|0;
          $395 = (($394) + 1)|0;
          HEAP32[$393>>2] = $395;
          $$0$i829 = $390;
          label = 162;
          break L510;
         }
        }
       } while(0);
       $400 = HEAP32[$$sink>>2]|0;
       $401 = (($400) + 1)|0;
       HEAP32[$$sink>>2] = $401;
      } else {
       $397 = ((($368)) + 81|0);
       $398 = HEAP8[$397>>0]|0;
       $399 = ($398<<24>>24)==(0);
       if (!($399)) {
        label = 164;
        break L4;
       }
      }
      $402 = (_dbMallocRawFinish($368,$317,$315)|0);
      $$0$i829 = $402;
      label = 162;
     }
    } while(0);
    if ((label|0) == 162) {
     label = 0;
     $452 = ($$0$i829|0)==(0|0);
     if ($452) {
      label = 164;
      break;
     }
     $$0$i748 = $$0$i829;$$pre$phi1237Z2D = $317;
    }
    $486 = ($$0$i748|0)==(0|0);
    $$1694 = $486 ? $$0693 : $$pre$phi1237Z2D;
    $$0692 = $$0$i748;$$1698 = $$0$i748;$$2695 = $$1694;
   }
   $487 = (($$2695) + -1)|0;
   $488 = (($$0692) + ($487)|0);
   $489 = ($$06031059$lcssa|0)==(22);
   if ($489) {
    $492 = (___udivdi3(($490|0),($491|0),10,0)|0);
    $493 = (getTempRet0() | 0);
    $494 = (___muldi3(($492|0),($493|0),10,0)|0);
    $495 = (getTempRet0() | 0);
    $496 = (_i64Subtract(($490|0),($491|0),($494|0),($495|0))|0);
    $497 = (getTempRet0() | 0);
    $498 = ($496|0)>(3);
    $499 = (___uremdi3(($492|0),($493|0),10,0)|0);
    $500 = (getTempRet0() | 0);
    $501 = ($499|0)==(1);
    $502 = ($500|0)==(0);
    $503 = $501 & $502;
    $or$cond1841 = $498 | $503;
    $$0684 = $or$cond1841 ? 0 : $496;
    $504 = $$0684 << 1;
    $505 = $504 | 1;
    $506 = (31151 + ($505)|0);
    $507 = HEAP8[$506>>0]|0;
    $508 = ((($488)) + -1|0);
    HEAP8[$508>>0] = $507;
    $509 = (31151 + ($504)|0);
    $510 = HEAP8[$509>>0]|0;
    $511 = ((($508)) + -1|0);
    HEAP8[$511>>0] = $510;
    $$2569 = $511;
   } else {
    $$2569 = $488;
   }
   $512 = (((16 + (($$06031059$lcssa*6)|0)|0)) + 4|0);
   $513 = HEAP8[$512>>0]|0;
   $514 = $513&255;
   $515 = (160 + ($514)|0);
   $516 = (((16 + (($$06031059$lcssa*6)|0)|0)) + 1|0);
   $517 = HEAP8[$516>>0]|0;
   $518 = $517&255;
   $$3570 = $$2569;$519 = $490;$520 = $491;
   while(1) {
    $521 = (___udivdi3(($519|0),($520|0),($518|0),0)|0);
    $522 = (getTempRet0() | 0);
    $523 = (___muldi3(($521|0),($522|0),($518|0),0)|0);
    $524 = (getTempRet0() | 0);
    $525 = (_i64Subtract(($519|0),($520|0),($523|0),($524|0))|0);
    $526 = (getTempRet0() | 0);
    $527 = (($515) + ($525)|0);
    $528 = HEAP8[$527>>0]|0;
    $529 = ((($$3570)) + -1|0);
    HEAP8[$529>>0] = $528;
    $530 = ($520>>>0)<(0);
    $531 = ($519>>>0)<($518>>>0);
    $532 = ($520|0)==(0);
    $533 = $532 & $531;
    $534 = $530 | $533;
    if ($534) {
     break;
    } else {
     $$3570 = $529;$519 = $521;$520 = $522;
    }
   }
   $535 = $488;
   $536 = $529;
   $537 = (($535) - ($536))|0;
   $538 = ($spec$select941|0)>($537|0);
   if ($538) {
    $539 = (($535) - ($spec$select941))|0;
    $540 = (($539) - ($536))|0;
    $scevgep1235 = (($529) + ($540)|0);
    $541 = (($spec$select941) + ($536))|0;
    $542 = (($541) - ($535))|0;
    _memset(($scevgep1235|0),48,($542|0))|0;
    $$1592$lcssa = $spec$select941;$$4571$lcssa = $scevgep1235;
   } else {
    $$1592$lcssa = $537;$$4571$lcssa = $529;
   }
   $543 = ($$2659<<24>>24)==(0);
   if ($543) {
    $$5572 = $$4571$lcssa;
   } else {
    $544 = (($$1592$lcssa) + -1)|0;
    $545 = (($544|0) / 3)&-1;
    $546 = (0 - ($545))|0;
    $547 = (($$4571$lcssa) + ($546)|0);
    $548 = ($$1592$lcssa|0)>(3);
    if ($548) {
     $549 = ($545*3)|0;
     $550 = (($544) - ($549))|0;
     $551 = (($550) + 1)|0;
     $$06741158 = $551;$$06761157 = $545;$$16041159 = 0;
     while(1) {
      $552 = (($$16041159) + ($$06761157))|0;
      $553 = (($547) + ($552)|0);
      $554 = HEAP8[$553>>0]|0;
      $555 = (($547) + ($$16041159)|0);
      HEAP8[$555>>0] = $554;
      $556 = (($$06741158) + -1)|0;
      $557 = ($556|0)==(0);
      $558 = (($$16041159) + 1)|0;
      if ($557) {
       $559 = (($$06761157) + -1)|0;
       $560 = (($547) + ($558)|0);
       HEAP8[$560>>0] = $$2659;
       $$pre = (($$16041159) + 2)|0;
       $$1675 = 3;$$1677 = $559;$$pre$phiZ2D = $$pre;
      } else {
       $$1675 = $556;$$1677 = $$06761157;$$pre$phiZ2D = $558;
      }
      $561 = ($$1677|0)>(0);
      if ($561) {
       $$06741158 = $$1675;$$06761157 = $$1677;$$16041159 = $$pre$phiZ2D;
      } else {
       $$5572 = $547;
       break;
      }
     }
    } else {
     $$5572 = $547;
    }
   }
   $562 = ($$1671916<<24>>24)==(0);
   if ($562) {
    $$6573 = $$5572;
   } else {
    $563 = ((($$5572)) + -1|0);
    HEAP8[$563>>0] = $$1671916;
    $$6573 = $563;
   }
   $564 = ($565<<24>>24)==(0);
   if ($564) {
    $$8575 = $$6573;
   } else {
    $566 = 7860991 >>> $$06031059$lcssa;
    $567 = $566 & 1;
    $568 = ($567|0)==(0);
    if ($568) {
     $569 = (((16 + (($$06031059$lcssa*6)|0)|0)) + 5|0);
     $570 = HEAP8[$569>>0]|0;
     $571 = $570&255;
     $572 = (31160 + ($571)|0);
     $573 = HEAP8[$572>>0]|0;
     $$06731162 = $572;$$75741163 = $$6573;$575 = $573;
     while(1) {
      $574 = ((($$75741163)) + -1|0);
      HEAP8[$574>>0] = $575;
      $576 = ((($$06731162)) + 1|0);
      $577 = HEAP8[$576>>0]|0;
      $578 = ($577<<24>>24)==(0);
      if ($578) {
       $$8575 = $574;
       break;
      } else {
       $$06731162 = $576;$$75741163 = $574;$575 = $577;
      }
     }
    } else {
     $$8575 = $$6573;
    }
   }
   $579 = $$8575;
   $580 = (($535) - ($579))|0;
   $$11602 = $580;$$11618 = $$3610902;$$28 = $$8575;$$3696 = $$2695;$$8705 = $$1698;
  }
  $1469 = (($$11618) - ($$11602))|0;
  $1470 = ($1469|0)>(0);
  do {
   if ($1470) {
    $1471 = ($$2621900<<24>>24)==(0);
    $1472 = HEAP32[$19>>2]|0;
    if ($1471) {
     $1473 = ($1469|0)<(0);
     $1474 = $1473 << 31 >> 31;
     $1475 = (_i64Add(($1472|0),0,($1469|0),($1474|0))|0);
     $1476 = (getTempRet0() | 0);
     $1477 = HEAP32[$20>>2]|0;
     $1478 = ($1476|0)<(0);
     $1479 = ($1475>>>0)<($1477>>>0);
     $1480 = ($1476|0)==(0);
     $1481 = $1480 & $1479;
     $1482 = $1478 | $1481;
     if ($1482) {
      $$1$i7901170$ph = $1469;
      label = 514;
     } else {
      $1483 = (_sqlite3StrAccumEnlarge($0,$1469)|0);
      $1484 = ($1483|0)<(1);
      if (!($1484)) {
       $$1$i7901170$ph = $1483;
       label = 514;
      }
     }
     if ((label|0) == 514) {
      label = 0;
      $$1$i7901170 = $$1$i7901170$ph;
      while(1) {
       $1485 = (($$1$i7901170) + -1)|0;
       $1486 = HEAP32[$21>>2]|0;
       $1487 = HEAP32[$19>>2]|0;
       $1488 = (($1487) + 1)|0;
       HEAP32[$19>>2] = $1488;
       $1489 = (($1486) + ($1487)|0);
       HEAP8[$1489>>0] = 32;
       $1490 = ($$1$i7901170|0)>(1);
       if ($1490) {
        $$1$i7901170 = $1485;
       } else {
        break;
       }
      }
     }
     $1491 = HEAP32[$19>>2]|0;
     $1492 = (($1491) + ($$11602))|0;
     $1493 = HEAP32[$20>>2]|0;
     $1494 = ($1492>>>0)<($1493>>>0);
     if (!($1494)) {
      _enlargeAndAppend($0,$$28,$$11602);
      break;
     }
     $1495 = ($$11602|0)==(0);
     if ($1495) {
      break;
     }
     HEAP32[$19>>2] = $1492;
     $1496 = HEAP32[$21>>2]|0;
     $1497 = (($1496) + ($1491)|0);
     _memcpy(($1497|0),($$28|0),($$11602|0))|0;
     break;
    } else {
     $1498 = (($1472) + ($$11602))|0;
     $1499 = HEAP32[$20>>2]|0;
     $1500 = ($1498>>>0)<($1499>>>0);
     if ($1500) {
      $1501 = ($$11602|0)==(0);
      if (!($1501)) {
       HEAP32[$19>>2] = $1498;
       $1502 = HEAP32[$21>>2]|0;
       $1503 = (($1502) + ($1472)|0);
       _memcpy(($1503|0),($$28|0),($$11602|0))|0;
      }
     } else {
      _enlargeAndAppend($0,$$28,$$11602);
     }
     $1504 = HEAP32[$19>>2]|0;
     $1505 = ($1469|0)<(0);
     $1506 = $1505 << 31 >> 31;
     $1507 = (_i64Add(($1504|0),0,($1469|0),($1506|0))|0);
     $1508 = (getTempRet0() | 0);
     $1509 = HEAP32[$20>>2]|0;
     $1510 = ($1508|0)<(0);
     $1511 = ($1507>>>0)<($1509>>>0);
     $1512 = ($1508|0)==(0);
     $1513 = $1512 & $1511;
     $1514 = $1510 | $1513;
     if ($1514) {
      $$1$i7951167$ph = $1469;
     } else {
      $1515 = (_sqlite3StrAccumEnlarge($0,$1469)|0);
      $1516 = ($1515|0)<(1);
      if ($1516) {
       break;
      } else {
       $$1$i7951167$ph = $1515;
      }
     }
     $$1$i7951167 = $$1$i7951167$ph;
     while(1) {
      $1517 = (($$1$i7951167) + -1)|0;
      $1518 = HEAP32[$21>>2]|0;
      $1519 = HEAP32[$19>>2]|0;
      $1520 = (($1519) + 1)|0;
      HEAP32[$19>>2] = $1520;
      $1521 = (($1518) + ($1519)|0);
      HEAP8[$1521>>0] = 32;
      $1522 = ($$1$i7951167|0)>(1);
      if ($1522) {
       $$1$i7951167 = $1517;
      } else {
       break;
      }
     }
    }
   } else {
    $1523 = HEAP32[$19>>2]|0;
    $1524 = (($1523) + ($$11602))|0;
    $1525 = HEAP32[$20>>2]|0;
    $1526 = ($1524>>>0)<($1525>>>0);
    if (!($1526)) {
     _enlargeAndAppend($0,$$28,$$11602);
     break;
    }
    $1527 = ($$11602|0)==(0);
    if (!($1527)) {
     HEAP32[$19>>2] = $1524;
     $1528 = HEAP32[$21>>2]|0;
     $1529 = (($1528) + ($1523)|0);
     _memcpy(($1529|0),($$28|0),($$11602|0))|0;
    }
   }
  } while(0);
  $1530 = ($$8705|0)==(0|0);
  do {
   if (!($1530)) {
    $1531 = HEAP32[$0>>2]|0;
    $1532 = ($1531|0)==(0|0);
    if (!($1532)) {
     $1533 = ((($1531)) + 480|0);
     $1534 = HEAP32[$1533>>2]|0;
     $1535 = ($1534|0)==(0|0);
     if (!($1535)) {
      _measureAllocationSize($1531,$$8705);
      break;
     }
     $1536 = $$8705;
     $1537 = ((($1531)) + 304|0);
     $1538 = HEAP32[$1537>>2]|0;
     $1539 = ($1538>>>0)>($1536>>>0);
     if (!($1539)) {
      $1540 = ((($1531)) + 308|0);
      $1541 = HEAP32[$1540>>2]|0;
      $1542 = ($1541>>>0)>($1536>>>0);
      if ($1542) {
       $1543 = ((($1531)) + 300|0);
       $1544 = HEAP32[$1543>>2]|0;
       HEAP32[$$8705>>2] = $1544;
       HEAP32[$1543>>2] = $$8705;
       break;
      }
     }
    }
    $1545 = HEAP32[7324]|0;
    $1546 = ($1545|0)==(0);
    if ($1546) {
     $1554 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$1554 & 127]($$8705);
     break;
    } else {
     $1547 = HEAP32[(29352)>>2]|0;
     $1548 = (FUNCTION_TABLE_ii[$1547 & 127]($$8705)|0);
     $1549 = HEAP32[14978]|0;
     $1550 = (($1549) - ($1548))|0;
     HEAP32[14978] = $1550;
     $1551 = HEAP32[14981]|0;
     $1552 = (($1551) + -1)|0;
     HEAP32[14981] = $1552;
     $1553 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$1553 & 127]($$8705);
     break;
    }
   }
  } while(0);
  $1555 = ((($$10)) + 1|0);
  $$0 = $1555;$$0567 = $$28;$$0693 = $$3696;
 }
 if ((label|0) == 13) {
  $45 = HEAP32[$19>>2]|0;
  $46 = (($45) + 1)|0;
  $47 = HEAP32[$20>>2]|0;
  $48 = ($46>>>0)<($47>>>0);
  if ($48) {
   HEAP32[$19>>2] = $46;
   $49 = HEAP32[$21>>2]|0;
   $50 = (($49) + ($45)|0);
   HEAP8[$50>>0] = 37;
   STACKTOP = sp;return;
  } else {
   _enlargeAndAppend($0,31149,1);
   STACKTOP = sp;return;
  }
 }
 else if ((label|0) == 121) {
  HEAP8[$22>>0] = 18;
  $327 = ($321|0)==(0);
  if (!($327)) {
   $328 = HEAP8[$6>>0]|0;
   $329 = $328 & 4;
   $330 = ($329<<24>>24)==(0);
   if (!($330)) {
    $331 = HEAP32[$0>>2]|0;
    $332 = HEAP32[$21>>2]|0;
    $333 = ($332|0)==(0|0);
    do {
     if (!($333)) {
      $334 = ($331|0)==(0|0);
      if (!($334)) {
       $335 = ((($331)) + 480|0);
       $336 = HEAP32[$335>>2]|0;
       $337 = ($336|0)==(0|0);
       if (!($337)) {
        _measureAllocationSize($331,$332);
        break;
       }
       $338 = $332;
       $339 = ((($331)) + 304|0);
       $340 = HEAP32[$339>>2]|0;
       $341 = ($340>>>0)>($338>>>0);
       if (!($341)) {
        $342 = ((($331)) + 308|0);
        $343 = HEAP32[$342>>2]|0;
        $344 = ($343>>>0)>($338>>>0);
        if ($344) {
         $345 = ((($331)) + 300|0);
         $346 = HEAP32[$345>>2]|0;
         HEAP32[$332>>2] = $346;
         HEAP32[$345>>2] = $332;
         break;
        }
       }
      }
      $347 = HEAP32[7324]|0;
      $348 = ($347|0)==(0);
      if ($348) {
       $356 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$356 & 127]($332);
       break;
      } else {
       $349 = HEAP32[(29352)>>2]|0;
       $350 = (FUNCTION_TABLE_ii[$349 & 127]($332)|0);
       $351 = HEAP32[14978]|0;
       $352 = (($351) - ($350))|0;
       HEAP32[14978] = $352;
       $353 = HEAP32[14981]|0;
       $354 = (($353) + -1)|0;
       HEAP32[14981] = $354;
       $355 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$355 & 127]($332);
       break;
      }
     }
    } while(0);
    $357 = HEAP8[$6>>0]|0;
    $358 = $357 & -5;
    HEAP8[$6>>0] = $358;
   }
   HEAP32[$20>>2] = 0;
   HEAP32[$19>>2] = 0;
   HEAP32[$21>>2] = 0;
  }
  $359 = HEAP32[$0>>2]|0;
  $360 = ($359|0)==(0|0);
  if ($360) {
   STACKTOP = sp;return;
  }
  $361 = ((($359)) + 236|0);
  $362 = HEAP32[$361>>2]|0;
  $363 = ($362|0)==(0|0);
  if ($363) {
   STACKTOP = sp;return;
  }
  $364 = ((($362)) + 12|0);
  HEAP32[$364>>2] = 18;
  $365 = ((($362)) + 36|0);
  $366 = HEAP32[$365>>2]|0;
  $367 = (($366) + 1)|0;
  HEAP32[$365>>2] = $367;
  STACKTOP = sp;return;
 }
 else if ((label|0) == 164) {
  HEAP8[$22>>0] = 7;
  $453 = HEAP32[$26>>2]|0;
  $454 = ($453|0)==(0);
  if ($454) {
   STACKTOP = sp;return;
  }
  $455 = HEAP8[$6>>0]|0;
  $456 = $455 & 4;
  $457 = ($456<<24>>24)==(0);
  if (!($457)) {
   $458 = HEAP32[$0>>2]|0;
   $459 = HEAP32[$21>>2]|0;
   $460 = ($459|0)==(0|0);
   do {
    if (!($460)) {
     $461 = ($458|0)==(0|0);
     if (!($461)) {
      $462 = ((($458)) + 480|0);
      $463 = HEAP32[$462>>2]|0;
      $464 = ($463|0)==(0|0);
      if (!($464)) {
       _measureAllocationSize($458,$459);
       break;
      }
      $465 = $459;
      $466 = ((($458)) + 304|0);
      $467 = HEAP32[$466>>2]|0;
      $468 = ($467>>>0)>($465>>>0);
      if (!($468)) {
       $469 = ((($458)) + 308|0);
       $470 = HEAP32[$469>>2]|0;
       $471 = ($470>>>0)>($465>>>0);
       if ($471) {
        $472 = ((($458)) + 300|0);
        $473 = HEAP32[$472>>2]|0;
        HEAP32[$459>>2] = $473;
        HEAP32[$472>>2] = $459;
        break;
       }
      }
     }
     $474 = HEAP32[7324]|0;
     $475 = ($474|0)==(0);
     if ($475) {
      $483 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$483 & 127]($459);
      break;
     } else {
      $476 = HEAP32[(29352)>>2]|0;
      $477 = (FUNCTION_TABLE_ii[$476 & 127]($459)|0);
      $478 = HEAP32[14978]|0;
      $479 = (($478) - ($477))|0;
      HEAP32[14978] = $479;
      $480 = HEAP32[14981]|0;
      $481 = (($480) + -1)|0;
      HEAP32[14981] = $481;
      $482 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$482 & 127]($459);
      break;
     }
    }
   } while(0);
   $484 = HEAP8[$6>>0]|0;
   $485 = $484 & -5;
   HEAP8[$6>>0] = $485;
  }
  HEAP32[$20>>2] = 0;
  HEAP32[$19>>2] = 0;
  HEAP32[$21>>2] = 0;
  STACKTOP = sp;return;
 }
 else if ((label|0) == 236) {
  HEAP8[$22>>0] = 18;
  $701 = ($695|0)==(0);
  if (!($701)) {
   $702 = HEAP8[$6>>0]|0;
   $703 = $702 & 4;
   $704 = ($703<<24>>24)==(0);
   if (!($704)) {
    $705 = HEAP32[$0>>2]|0;
    $706 = HEAP32[$21>>2]|0;
    $707 = ($706|0)==(0|0);
    do {
     if (!($707)) {
      $708 = ($705|0)==(0|0);
      if (!($708)) {
       $709 = ((($705)) + 480|0);
       $710 = HEAP32[$709>>2]|0;
       $711 = ($710|0)==(0|0);
       if (!($711)) {
        _measureAllocationSize($705,$706);
        break;
       }
       $712 = $706;
       $713 = ((($705)) + 304|0);
       $714 = HEAP32[$713>>2]|0;
       $715 = ($714>>>0)>($712>>>0);
       if (!($715)) {
        $716 = ((($705)) + 308|0);
        $717 = HEAP32[$716>>2]|0;
        $718 = ($717>>>0)>($712>>>0);
        if ($718) {
         $719 = ((($705)) + 300|0);
         $720 = HEAP32[$719>>2]|0;
         HEAP32[$706>>2] = $720;
         HEAP32[$719>>2] = $706;
         break;
        }
       }
      }
      $721 = HEAP32[7324]|0;
      $722 = ($721|0)==(0);
      if ($722) {
       $730 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$730 & 127]($706);
       break;
      } else {
       $723 = HEAP32[(29352)>>2]|0;
       $724 = (FUNCTION_TABLE_ii[$723 & 127]($706)|0);
       $725 = HEAP32[14978]|0;
       $726 = (($725) - ($724))|0;
       HEAP32[14978] = $726;
       $727 = HEAP32[14981]|0;
       $728 = (($727) + -1)|0;
       HEAP32[14981] = $728;
       $729 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$729 & 127]($706);
       break;
      }
     }
    } while(0);
    $731 = HEAP8[$6>>0]|0;
    $732 = $731 & -5;
    HEAP8[$6>>0] = $732;
   }
   HEAP32[$20>>2] = 0;
   HEAP32[$19>>2] = 0;
   HEAP32[$21>>2] = 0;
  }
  $733 = HEAP32[$0>>2]|0;
  $734 = ($733|0)==(0|0);
  if ($734) {
   STACKTOP = sp;return;
  }
  $735 = ((($733)) + 236|0);
  $736 = HEAP32[$735>>2]|0;
  $737 = ($736|0)==(0|0);
  if ($737) {
   STACKTOP = sp;return;
  }
  $738 = ((($736)) + 12|0);
  HEAP32[$738>>2] = 18;
  $739 = ((($736)) + 36|0);
  $740 = HEAP32[$739>>2]|0;
  $741 = (($740) + 1)|0;
  HEAP32[$739>>2] = $741;
  STACKTOP = sp;return;
 }
 else if ((label|0) == 278) {
  HEAP8[$22>>0] = 7;
  $827 = HEAP32[$26>>2]|0;
  $828 = ($827|0)==(0);
  if ($828) {
   STACKTOP = sp;return;
  }
  $829 = HEAP8[$6>>0]|0;
  $830 = $829 & 4;
  $831 = ($830<<24>>24)==(0);
  if (!($831)) {
   $832 = HEAP32[$0>>2]|0;
   $833 = HEAP32[$21>>2]|0;
   $834 = ($833|0)==(0|0);
   do {
    if (!($834)) {
     $835 = ($832|0)==(0|0);
     if (!($835)) {
      $836 = ((($832)) + 480|0);
      $837 = HEAP32[$836>>2]|0;
      $838 = ($837|0)==(0|0);
      if (!($838)) {
       _measureAllocationSize($832,$833);
       break;
      }
      $839 = $833;
      $840 = ((($832)) + 304|0);
      $841 = HEAP32[$840>>2]|0;
      $842 = ($841>>>0)>($839>>>0);
      if (!($842)) {
       $843 = ((($832)) + 308|0);
       $844 = HEAP32[$843>>2]|0;
       $845 = ($844>>>0)>($839>>>0);
       if ($845) {
        $846 = ((($832)) + 300|0);
        $847 = HEAP32[$846>>2]|0;
        HEAP32[$833>>2] = $847;
        HEAP32[$846>>2] = $833;
        break;
       }
      }
     }
     $848 = HEAP32[7324]|0;
     $849 = ($848|0)==(0);
     if ($849) {
      $857 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$857 & 127]($833);
      break;
     } else {
      $850 = HEAP32[(29352)>>2]|0;
      $851 = (FUNCTION_TABLE_ii[$850 & 127]($833)|0);
      $852 = HEAP32[14978]|0;
      $853 = (($852) - ($851))|0;
      HEAP32[14978] = $853;
      $854 = HEAP32[14981]|0;
      $855 = (($854) + -1)|0;
      HEAP32[14981] = $855;
      $856 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$856 & 127]($833);
      break;
     }
    }
   } while(0);
   $858 = HEAP8[$6>>0]|0;
   $859 = $858 & -5;
   HEAP8[$6>>0] = $859;
  }
  HEAP32[$20>>2] = 0;
  HEAP32[$19>>2] = 0;
  HEAP32[$21>>2] = 0;
  STACKTOP = sp;return;
 }
 else if ((label|0) == 423) {
  HEAP8[$22>>0] = 18;
  $1228 = ($1222|0)==(0);
  if (!($1228)) {
   $1229 = HEAP8[$6>>0]|0;
   $1230 = $1229 & 4;
   $1231 = ($1230<<24>>24)==(0);
   if (!($1231)) {
    $1232 = HEAP32[$0>>2]|0;
    $1233 = HEAP32[$21>>2]|0;
    $1234 = ($1233|0)==(0|0);
    do {
     if (!($1234)) {
      $1235 = ($1232|0)==(0|0);
      if (!($1235)) {
       $1236 = ((($1232)) + 480|0);
       $1237 = HEAP32[$1236>>2]|0;
       $1238 = ($1237|0)==(0|0);
       if (!($1238)) {
        _measureAllocationSize($1232,$1233);
        break;
       }
       $1239 = $1233;
       $1240 = ((($1232)) + 304|0);
       $1241 = HEAP32[$1240>>2]|0;
       $1242 = ($1241>>>0)>($1239>>>0);
       if (!($1242)) {
        $1243 = ((($1232)) + 308|0);
        $1244 = HEAP32[$1243>>2]|0;
        $1245 = ($1244>>>0)>($1239>>>0);
        if ($1245) {
         $1246 = ((($1232)) + 300|0);
         $1247 = HEAP32[$1246>>2]|0;
         HEAP32[$1233>>2] = $1247;
         HEAP32[$1246>>2] = $1233;
         break;
        }
       }
      }
      $1248 = HEAP32[7324]|0;
      $1249 = ($1248|0)==(0);
      if ($1249) {
       $1257 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$1257 & 127]($1233);
       break;
      } else {
       $1250 = HEAP32[(29352)>>2]|0;
       $1251 = (FUNCTION_TABLE_ii[$1250 & 127]($1233)|0);
       $1252 = HEAP32[14978]|0;
       $1253 = (($1252) - ($1251))|0;
       HEAP32[14978] = $1253;
       $1254 = HEAP32[14981]|0;
       $1255 = (($1254) + -1)|0;
       HEAP32[14981] = $1255;
       $1256 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$1256 & 127]($1233);
       break;
      }
     }
    } while(0);
    $1258 = HEAP8[$6>>0]|0;
    $1259 = $1258 & -5;
    HEAP8[$6>>0] = $1259;
   }
   HEAP32[$20>>2] = 0;
   HEAP32[$19>>2] = 0;
   HEAP32[$21>>2] = 0;
  }
  $1260 = HEAP32[$0>>2]|0;
  $1261 = ($1260|0)==(0|0);
  if ($1261) {
   STACKTOP = sp;return;
  }
  $1262 = ((($1260)) + 236|0);
  $1263 = HEAP32[$1262>>2]|0;
  $1264 = ($1263|0)==(0|0);
  if ($1264) {
   STACKTOP = sp;return;
  }
  $1265 = ((($1263)) + 12|0);
  HEAP32[$1265>>2] = 18;
  $1266 = ((($1263)) + 36|0);
  $1267 = HEAP32[$1266>>2]|0;
  $1268 = (($1267) + 1)|0;
  HEAP32[$1266>>2] = $1268;
  STACKTOP = sp;return;
 }
 else if ((label|0) == 465) {
  HEAP8[$22>>0] = 7;
  $1354 = HEAP32[$26>>2]|0;
  $1355 = ($1354|0)==(0);
  if ($1355) {
   STACKTOP = sp;return;
  }
  $1356 = HEAP8[$6>>0]|0;
  $1357 = $1356 & 4;
  $1358 = ($1357<<24>>24)==(0);
  if (!($1358)) {
   $1359 = HEAP32[$0>>2]|0;
   $1360 = HEAP32[$21>>2]|0;
   $1361 = ($1360|0)==(0|0);
   do {
    if (!($1361)) {
     $1362 = ($1359|0)==(0|0);
     if (!($1362)) {
      $1363 = ((($1359)) + 480|0);
      $1364 = HEAP32[$1363>>2]|0;
      $1365 = ($1364|0)==(0|0);
      if (!($1365)) {
       _measureAllocationSize($1359,$1360);
       break;
      }
      $1366 = $1360;
      $1367 = ((($1359)) + 304|0);
      $1368 = HEAP32[$1367>>2]|0;
      $1369 = ($1368>>>0)>($1366>>>0);
      if (!($1369)) {
       $1370 = ((($1359)) + 308|0);
       $1371 = HEAP32[$1370>>2]|0;
       $1372 = ($1371>>>0)>($1366>>>0);
       if ($1372) {
        $1373 = ((($1359)) + 300|0);
        $1374 = HEAP32[$1373>>2]|0;
        HEAP32[$1360>>2] = $1374;
        HEAP32[$1373>>2] = $1360;
        break;
       }
      }
     }
     $1375 = HEAP32[7324]|0;
     $1376 = ($1375|0)==(0);
     if ($1376) {
      $1384 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$1384 & 127]($1360);
      break;
     } else {
      $1377 = HEAP32[(29352)>>2]|0;
      $1378 = (FUNCTION_TABLE_ii[$1377 & 127]($1360)|0);
      $1379 = HEAP32[14978]|0;
      $1380 = (($1379) - ($1378))|0;
      HEAP32[14978] = $1380;
      $1381 = HEAP32[14981]|0;
      $1382 = (($1381) + -1)|0;
      HEAP32[14981] = $1382;
      $1383 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$1383 & 127]($1360);
      break;
     }
    }
   } while(0);
   $1385 = HEAP8[$6>>0]|0;
   $1386 = $1385 & -5;
   HEAP8[$6>>0] = $1386;
  }
  HEAP32[$20>>2] = 0;
  HEAP32[$19>>2] = 0;
  HEAP32[$21>>2] = 0;
  STACKTOP = sp;return;
 }
 else if ((label|0) == 543) {
  STACKTOP = sp;return;
 }
}
function _sqlite3_str_append($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $12 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = ((($0)) + 16|0);
 $4 = HEAP32[$3>>2]|0;
 $5 = (($4) + ($2))|0;
 $6 = ((($0)) + 8|0);
 $7 = HEAP32[$6>>2]|0;
 $8 = ($5>>>0)<($7>>>0);
 if (!($8)) {
  _enlargeAndAppend($0,$1,$2);
  return;
 }
 $9 = ($2|0)==(0);
 if ($9) {
  return;
 }
 HEAP32[$3>>2] = $5;
 $10 = ((($0)) + 4|0);
 $11 = HEAP32[$10>>2]|0;
 $12 = (($11) + ($4)|0);
 _memcpy(($12|0),($1|0),($2|0))|0;
 return;
}
function _sqlite3StrAccumEnlarge($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i = 0, $$049 = 0, $$04986 = 0, $$pre = 0, $$pre$i = 0, $$pre$i52 = 0, $$pre$phi$i68Z2D = 0, $$pre$phi$i75Z2D = 0, $$pre$phi$iZ2D = 0, $$pre87 = 0, $$sink = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0;
 var $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0;
 var $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0;
 var $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0;
 var $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0;
 var $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0;
 var $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0;
 var $216 = 0, $217 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0;
 var $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0;
 var $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0;
 var $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0;
 var $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($0)) + 20|0);
 $3 = HEAP8[$2>>0]|0;
 $4 = ($3<<24>>24)==(0);
 if (!($4)) {
  $217 = 0;
  return ($217|0);
 }
 $5 = ((($0)) + 12|0);
 $6 = HEAP32[$5>>2]|0;
 $7 = ($6|0)==(0);
 if ($7) {
  HEAP8[$2>>0] = 18;
  $8 = HEAP32[$0>>2]|0;
  $9 = ($8|0)==(0|0);
  if (!($9)) {
   $10 = ((($8)) + 236|0);
   $11 = HEAP32[$10>>2]|0;
   $12 = ($11|0)==(0|0);
   if (!($12)) {
    $13 = ((($11)) + 12|0);
    HEAP32[$13>>2] = 18;
    $14 = ((($11)) + 36|0);
    $15 = HEAP32[$14>>2]|0;
    $16 = (($15) + 1)|0;
    HEAP32[$14>>2] = $16;
   }
  }
  $17 = ((($0)) + 8|0);
  $18 = HEAP32[$17>>2]|0;
  $19 = ((($0)) + 16|0);
  $20 = HEAP32[$19>>2]|0;
  $21 = (($18) + -1)|0;
  $22 = (($21) - ($20))|0;
  $217 = $22;
  return ($217|0);
 }
 $23 = ((($0)) + 21|0);
 $24 = HEAP8[$23>>0]|0;
 $25 = $24 & 4;
 $26 = ($25<<24>>24)==(0);
 if ($26) {
  $100 = 0;
 } else {
  $27 = ((($0)) + 4|0);
  $28 = HEAP32[$27>>2]|0;
  $100 = $28;
 }
 $29 = ((($0)) + 16|0);
 $30 = HEAP32[$29>>2]|0;
 $31 = (($1) + 1)|0;
 $32 = ($31|0)<(0);
 $33 = $32 << 31 >> 31;
 $34 = (_i64Add(($30|0),0,($31|0),($33|0))|0);
 $35 = (getTempRet0() | 0);
 $36 = (_i64Add(($34|0),($35|0),($30|0),0)|0);
 $37 = (getTempRet0() | 0);
 $38 = ($37|0)>(0);
 $39 = ($36>>>0)>($6>>>0);
 $40 = ($37|0)==(0);
 $41 = $40 & $39;
 $42 = $38 | $41;
 $43 = $42 ? $34 : $36;
 $44 = $42 ? $35 : $37;
 $45 = ($44|0)>(0);
 $46 = ($43>>>0)>($6>>>0);
 $47 = ($44|0)==(0);
 $48 = $47 & $46;
 $49 = $45 | $48;
 if ($49) {
  if ($26) {
   $$pre$i52 = ((($0)) + 4|0);
   $50 = ((($0)) + 8|0);
   HEAP32[$50>>2] = 0;
   HEAP32[$29>>2] = 0;
   HEAP32[$$pre$i52>>2] = 0;
   HEAP8[$2>>0] = 18;
   $83 = $24;$86 = $50;
   label = 23;
  } else {
   $51 = HEAP32[$0>>2]|0;
   $52 = ((($0)) + 4|0);
   $53 = HEAP32[$52>>2]|0;
   $54 = ($53|0)==(0|0);
   do {
    if (!($54)) {
     $55 = ($51|0)==(0|0);
     if (!($55)) {
      $56 = ((($51)) + 480|0);
      $57 = HEAP32[$56>>2]|0;
      $58 = ($57|0)==(0|0);
      if (!($58)) {
       _measureAllocationSize($51,$53);
       break;
      }
      $59 = $53;
      $60 = ((($51)) + 304|0);
      $61 = HEAP32[$60>>2]|0;
      $62 = ($61>>>0)>($59>>>0);
      if (!($62)) {
       $63 = ((($51)) + 308|0);
       $64 = HEAP32[$63>>2]|0;
       $65 = ($64>>>0)>($59>>>0);
       if ($65) {
        $66 = ((($51)) + 300|0);
        $67 = HEAP32[$66>>2]|0;
        HEAP32[$53>>2] = $67;
        HEAP32[$66>>2] = $53;
        break;
       }
      }
     }
     $68 = HEAP32[7324]|0;
     $69 = ($68|0)==(0);
     if ($69) {
      $77 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$77 & 127]($53);
      break;
     } else {
      $70 = HEAP32[(29352)>>2]|0;
      $71 = (FUNCTION_TABLE_ii[$70 & 127]($53)|0);
      $72 = HEAP32[14978]|0;
      $73 = (($72) - ($71))|0;
      HEAP32[14978] = $73;
      $74 = HEAP32[14981]|0;
      $75 = (($74) + -1)|0;
      HEAP32[14981] = $75;
      $76 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$76 & 127]($53);
      break;
     }
    }
   } while(0);
   $78 = HEAP8[$23>>0]|0;
   $79 = $78 & -5;
   HEAP8[$23>>0] = $79;
   $$pre87 = HEAP32[$5>>2]|0;
   $80 = ((($0)) + 8|0);
   HEAP32[$80>>2] = 0;
   HEAP32[$29>>2] = 0;
   HEAP32[$52>>2] = 0;
   HEAP8[$2>>0] = 18;
   $81 = ($$pre87|0)==(0);
   if (!($81)) {
    $83 = $79;$86 = $80;
    label = 23;
   }
  }
  if ((label|0) == 23) {
   $82 = $83 & 4;
   $84 = ($82<<24>>24)==(0);
   if (!($84)) {
    $85 = $83 & -5;
    HEAP8[$23>>0] = $85;
   }
   $$pre$phi$i75Z2D = ((($0)) + 4|0);
   HEAP32[$86>>2] = 0;
   HEAP32[$29>>2] = 0;
   HEAP32[$$pre$phi$i75Z2D>>2] = 0;
  }
  $87 = HEAP32[$0>>2]|0;
  $88 = ($87|0)==(0|0);
  if ($88) {
   $217 = 0;
   return ($217|0);
  }
  $89 = ((($87)) + 236|0);
  $90 = HEAP32[$89>>2]|0;
  $91 = ($90|0)==(0|0);
  if ($91) {
   $217 = 0;
   return ($217|0);
  }
  $92 = ((($90)) + 12|0);
  HEAP32[$92>>2] = 18;
  $93 = ((($90)) + 36|0);
  $94 = HEAP32[$93>>2]|0;
  $95 = (($94) + 1)|0;
  HEAP32[$93>>2] = $95;
  $217 = 0;
  return ($217|0);
 }
 $96 = ((($0)) + 8|0);
 HEAP32[$96>>2] = $43;
 $97 = HEAP32[$0>>2]|0;
 $98 = ($97|0)==(0|0);
 L48: do {
  if ($98) {
   $151 = (_sqlite3_initialize()|0);
   $152 = ($151|0)==(0);
   if ($152) {
    $153 = (_sqlite3Realloc($100,$43,0)|0);
    $$049 = $153;
    label = 49;
   }
  } else {
   $99 = ($100|0)==(0|0);
   if (!($99)) {
    $134 = $100;
    $135 = ((($97)) + 304|0);
    $136 = HEAP32[$135>>2]|0;
    $137 = ($136>>>0)>($134>>>0);
    if (!($137)) {
     $138 = ((($97)) + 308|0);
     $139 = HEAP32[$138>>2]|0;
     $140 = ($139>>>0)>($134>>>0);
     if ($140) {
      $141 = ((($97)) + 276|0);
      $142 = HEAP16[$141>>1]|0;
      $143 = $142&65535;
      $144 = (0)>(0);
      $145 = ($43>>>0)>($143>>>0);
      $146 = (0)==(0);
      $147 = $146 & $145;
      $148 = $144 | $147;
      if (!($148)) {
       $$04986 = $100;$156 = $24;
       label = 51;
       break;
      }
      $150 = (_dbReallocFinish($97,$100,$43,0)|0);
      $$049 = $150;
      label = 49;
      break;
     }
    }
    $149 = (_dbReallocFinish($97,$100,$43,0)|0);
    $$049 = $149;
    label = 49;
    break;
   }
   $101 = ((($97)) + 272|0);
   $102 = HEAP32[$101>>2]|0;
   $103 = ($102|0)==(0);
   if ($103) {
    $104 = ((($97)) + 276|0);
    $105 = HEAP16[$104>>1]|0;
    $106 = $105&65535;
    $107 = (0)>(0);
    $108 = ($43>>>0)>($106>>>0);
    $109 = (0)==(0);
    $110 = $109 & $108;
    $111 = $107 | $110;
    do {
     if ($111) {
      $112 = ((($97)) + 288|0);
      $$sink = $112;
     } else {
      $113 = ((($97)) + 300|0);
      $114 = HEAP32[$113>>2]|0;
      $115 = ($114|0)==(0|0);
      if (!($115)) {
       $116 = HEAP32[$114>>2]|0;
       HEAP32[$113>>2] = $116;
       $117 = ((($97)) + 284|0);
       $118 = HEAP32[$117>>2]|0;
       $119 = (($118) + 1)|0;
       HEAP32[$117>>2] = $119;
       $$049 = $114;
       label = 49;
       break L48;
      }
      $120 = ((($97)) + 296|0);
      $121 = HEAP32[$120>>2]|0;
      $122 = ($121|0)==(0|0);
      if ($122) {
       $127 = ((($97)) + 292|0);
       $$sink = $127;
       break;
      } else {
       $123 = HEAP32[$121>>2]|0;
       HEAP32[$120>>2] = $123;
       $124 = ((($97)) + 284|0);
       $125 = HEAP32[$124>>2]|0;
       $126 = (($125) + 1)|0;
       HEAP32[$124>>2] = $126;
       $$049 = $121;
       label = 49;
       break L48;
      }
     }
    } while(0);
    $131 = HEAP32[$$sink>>2]|0;
    $132 = (($131) + 1)|0;
    HEAP32[$$sink>>2] = $132;
   } else {
    $128 = ((($97)) + 81|0);
    $129 = HEAP8[$128>>0]|0;
    $130 = ($129<<24>>24)==(0);
    if (!($130)) {
     break;
    }
   }
   $133 = (_dbMallocRawFinish($97,$43,0)|0);
   $$049 = $133;
   label = 49;
  }
 } while(0);
 if ((label|0) == 49) {
  $154 = ($$049|0)==(0|0);
  if (!($154)) {
   $$pre = HEAP8[$23>>0]|0;
   $$04986 = $$049;$156 = $$pre;
   label = 51;
  }
 }
 if ((label|0) == 51) {
  $155 = $156 & 4;
  $157 = ($155<<24>>24)==(0);
  if ($157) {
   $158 = HEAP32[$29>>2]|0;
   $159 = ($158|0)==(0);
   if (!($159)) {
    $160 = ((($0)) + 4|0);
    $161 = HEAP32[$160>>2]|0;
    _memcpy(($$04986|0),($161|0),($158|0))|0;
   }
  }
  $162 = ((($0)) + 4|0);
  HEAP32[$162>>2] = $$04986;
  $163 = HEAP32[$0>>2]|0;
  $164 = ($163|0)==(0|0);
  if ($164) {
   label = 57;
  } else {
   $165 = $$04986;
   $166 = ((($163)) + 304|0);
   $167 = HEAP32[$166>>2]|0;
   $168 = ($167>>>0)>($165>>>0);
   if ($168) {
    label = 57;
   } else {
    $169 = ((($163)) + 308|0);
    $170 = HEAP32[$169>>2]|0;
    $171 = ($170>>>0)>($165>>>0);
    if ($171) {
     $174 = ((($163)) + 276|0);
     $175 = HEAP16[$174>>1]|0;
     $176 = $175&65535;
     $$0$i = $176;
    } else {
     label = 57;
    }
   }
  }
  if ((label|0) == 57) {
   $172 = HEAP32[(29352)>>2]|0;
   $173 = (FUNCTION_TABLE_ii[$172 & 127]($$04986)|0);
   $$0$i = $173;
  }
  HEAP32[$96>>2] = $$0$i;
  $215 = HEAP8[$23>>0]|0;
  $216 = $215 | 4;
  HEAP8[$23>>0] = $216;
  $217 = $1;
  return ($217|0);
 }
 $177 = HEAP8[$23>>0]|0;
 $178 = $177 & 4;
 $179 = ($178<<24>>24)==(0);
 if ($179) {
  $$pre$i = ((($0)) + 4|0);
  $$pre$phi$iZ2D = $$pre$i;$212 = $177;
 } else {
  $180 = HEAP32[$0>>2]|0;
  $181 = ((($0)) + 4|0);
  $182 = HEAP32[$181>>2]|0;
  $183 = ($182|0)==(0|0);
  do {
   if (!($183)) {
    $184 = ($180|0)==(0|0);
    if (!($184)) {
     $185 = ((($180)) + 480|0);
     $186 = HEAP32[$185>>2]|0;
     $187 = ($186|0)==(0|0);
     if (!($187)) {
      _measureAllocationSize($180,$182);
      break;
     }
     $188 = $182;
     $189 = ((($180)) + 304|0);
     $190 = HEAP32[$189>>2]|0;
     $191 = ($190>>>0)>($188>>>0);
     if (!($191)) {
      $192 = ((($180)) + 308|0);
      $193 = HEAP32[$192>>2]|0;
      $194 = ($193>>>0)>($188>>>0);
      if ($194) {
       $195 = ((($180)) + 300|0);
       $196 = HEAP32[$195>>2]|0;
       HEAP32[$182>>2] = $196;
       HEAP32[$195>>2] = $182;
       break;
      }
     }
    }
    $197 = HEAP32[7324]|0;
    $198 = ($197|0)==(0);
    if ($198) {
     $206 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$206 & 127]($182);
     break;
    } else {
     $199 = HEAP32[(29352)>>2]|0;
     $200 = (FUNCTION_TABLE_ii[$199 & 127]($182)|0);
     $201 = HEAP32[14978]|0;
     $202 = (($201) - ($200))|0;
     HEAP32[14978] = $202;
     $203 = HEAP32[14981]|0;
     $204 = (($203) + -1)|0;
     HEAP32[14981] = $204;
     $205 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$205 & 127]($182);
     break;
    }
   }
  } while(0);
  $207 = HEAP8[$23>>0]|0;
  $208 = $207 & -5;
  HEAP8[$23>>0] = $208;
  $$pre$phi$iZ2D = $181;$212 = $208;
 }
 HEAP32[$96>>2] = 0;
 HEAP32[$29>>2] = 0;
 HEAP32[$$pre$phi$iZ2D>>2] = 0;
 HEAP8[$2>>0] = 7;
 $209 = HEAP32[$5>>2]|0;
 $210 = ($209|0)==(0);
 if ($210) {
  $217 = 0;
  return ($217|0);
 }
 $211 = $212 & 4;
 $213 = ($211<<24>>24)==(0);
 if (!($213)) {
  $214 = $212 & -5;
  HEAP8[$23>>0] = $214;
 }
 $$pre$phi$i68Z2D = ((($0)) + 4|0);
 HEAP32[$96>>2] = 0;
 HEAP32[$29>>2] = 0;
 HEAP32[$$pre$phi$i68Z2D>>2] = 0;
 $217 = 0;
 return ($217|0);
}
function _enlargeAndAppend($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = (_sqlite3StrAccumEnlarge($0,$2)|0);
 $4 = ($3|0)>(0);
 if (!($4)) {
  return;
 }
 $5 = ((($0)) + 4|0);
 $6 = HEAP32[$5>>2]|0;
 $7 = ((($0)) + 16|0);
 $8 = HEAP32[$7>>2]|0;
 $9 = (($6) + ($8)|0);
 _memcpy(($9|0),($1|0),($3|0))|0;
 $10 = HEAP32[$7>>2]|0;
 $11 = (($10) + ($3))|0;
 HEAP32[$7>>2] = $11;
 return;
}
function _sqlite3Strlen30($0) {
 $0 = $0|0;
 var $$0 = 0, $1 = 0, $2 = 0, $3 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$0 = 0;
  return ($$0|0);
 }
 $2 = (_strlen($0)|0);
 $3 = $2 & 1073741823;
 $$0 = $3;
 return ($$0|0);
}
function _sqlite3StrAccumFinish($0) {
 $0 = $0|0;
 var $$0 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 4|0);
 $2 = HEAP32[$1>>2]|0;
 $3 = ($2|0)==(0|0);
 if (!($3)) {
  $4 = ((($0)) + 16|0);
  $5 = HEAP32[$4>>2]|0;
  $6 = (($2) + ($5)|0);
  HEAP8[$6>>0] = 0;
  $7 = ((($0)) + 12|0);
  $8 = HEAP32[$7>>2]|0;
  $9 = ($8|0)==(0);
  if (!($9)) {
   $10 = ((($0)) + 21|0);
   $11 = HEAP8[$10>>0]|0;
   $12 = $11 & 4;
   $13 = ($12<<24>>24)==(0);
   if ($13) {
    $14 = (_strAccumFinishRealloc($0)|0);
    $$0 = $14;
    return ($$0|0);
   }
  }
 }
 $15 = HEAP32[$1>>2]|0;
 $$0 = $15;
 return ($$0|0);
}
function _sqlite3_str_new($0) {
 $0 = $0|0;
 var $$ = 0, $$0 = 0, $$in = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0;
 var $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0;
 var $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0;
 var $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = (_sqlite3_initialize()|0);
 $2 = ($1|0)==(0);
 if (!($2)) {
  $$0 = 29632;
  return ($$0|0);
 }
 $3 = HEAP32[7324]|0;
 $4 = ($3|0)==(0);
 if ($4) {
  $43 = HEAP32[(29340)>>2]|0;
  $44 = (FUNCTION_TABLE_ii[$43 & 127](24)|0);
  $45 = ($44|0)==(0|0);
  if ($45) {
   $$0 = 29632;
   return ($$0|0);
  } else {
   $$in = $44;
  }
 } else {
  $5 = HEAP32[(29356)>>2]|0;
  $6 = (FUNCTION_TABLE_ii[$5 & 127](24)|0);
  $7 = HEAP32[14985]|0;
  $8 = ($7>>>0)<(24);
  if ($8) {
   HEAP32[14985] = 24;
  }
  $9 = (59064);
  $10 = $9;
  $11 = HEAP32[$10>>2]|0;
  $12 = (($9) + 4)|0;
  $13 = $12;
  $14 = HEAP32[$13>>2]|0;
  $15 = ($14|0)>(0);
  $16 = ($11>>>0)>(0);
  $17 = ($14|0)==(0);
  $18 = $17 & $16;
  $19 = $15 | $18;
  if ($19) {
   $20 = HEAP32[14978]|0;
   $21 = ($6|0)<(0);
   $22 = $21 << 31 >> 31;
   $23 = (_i64Subtract(($11|0),($14|0),($6|0),($22|0))|0);
   $24 = (getTempRet0() | 0);
   $25 = ($24|0)<(0);
   $26 = ($23>>>0)<=($20>>>0);
   $27 = ($24|0)==(0);
   $28 = $27 & $26;
   $29 = $25 | $28;
   $$ = $29&1;
   HEAP32[(59072)>>2] = $$;
  }
  $30 = HEAP32[(29340)>>2]|0;
  $31 = (FUNCTION_TABLE_ii[$30 & 127]($6)|0);
  $32 = ($31|0)==(0|0);
  if ($32) {
   $$0 = 29632;
   return ($$0|0);
  }
  $33 = HEAP32[(29352)>>2]|0;
  $34 = (FUNCTION_TABLE_ii[$33 & 127]($31)|0);
  $35 = HEAP32[14978]|0;
  $36 = (($35) + ($34))|0;
  HEAP32[14978] = $36;
  $37 = HEAP32[14982]|0;
  $38 = ($36>>>0)>($37>>>0);
  if ($38) {
   HEAP32[14982] = $36;
  }
  $39 = HEAP32[14981]|0;
  $40 = (($39) + 1)|0;
  HEAP32[14981] = $40;
  $41 = HEAP32[14987]|0;
  $42 = ($40>>>0)>($41>>>0);
  if ($42) {
   HEAP32[14987] = $40;
   $$in = $31;
  } else {
   $$in = $31;
  }
 }
 $46 = ($0|0)==(0|0);
 if ($46) {
  $55 = ((($$in)) + 4|0);
  HEAP32[$55>>2] = 0;
  HEAP32[$$in>>2] = 0;
  $56 = ((($$in)) + 8|0);
  HEAP32[$56>>2] = 0;
  $57 = ((($$in)) + 12|0);
  HEAP32[$57>>2] = 1000000000;
  $58 = ((($$in)) + 16|0);
  HEAP32[$58>>2] = 0;
  $59 = ((($$in)) + 20|0);
  HEAP8[$59>>0] = 0;
  $60 = ((($$in)) + 21|0);
  HEAP8[$60>>0] = 0;
  $$0 = $$in;
  return ($$0|0);
 } else {
  $47 = ((($0)) + 108|0);
  $48 = HEAP32[$47>>2]|0;
  $49 = ((($$in)) + 4|0);
  HEAP32[$49>>2] = 0;
  HEAP32[$$in>>2] = 0;
  $50 = ((($$in)) + 8|0);
  HEAP32[$50>>2] = 0;
  $51 = ((($$in)) + 12|0);
  HEAP32[$51>>2] = $48;
  $52 = ((($$in)) + 16|0);
  HEAP32[$52>>2] = 0;
  $53 = ((($$in)) + 20|0);
  HEAP8[$53>>0] = 0;
  $54 = ((($$in)) + 21|0);
  HEAP8[$54>>0] = 0;
  $$0 = $$in;
  return ($$0|0);
 }
 return (0)|0;
}
function _sqlite3_vmprintf($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0;
 var $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 96|0;
 $2 = sp;
 $3 = sp + 72|0;
 $4 = (_sqlite3_initialize()|0);
 $5 = ($4|0)==(0);
 if (!($5)) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 $6 = ((($3)) + 4|0);
 HEAP32[$6>>2] = $2;
 HEAP32[$3>>2] = 0;
 $7 = ((($3)) + 8|0);
 HEAP32[$7>>2] = 70;
 $8 = ((($3)) + 12|0);
 HEAP32[$8>>2] = 1000000000;
 $9 = ((($3)) + 16|0);
 HEAP32[$9>>2] = 0;
 $10 = ((($3)) + 20|0);
 HEAP8[$10>>0] = 0;
 $11 = ((($3)) + 21|0);
 HEAP8[$11>>0] = 0;
 _sqlite3_str_vappendf($3,$0,$1);
 $12 = HEAP32[$6>>2]|0;
 $13 = ($12|0)==(0|0);
 if (!($13)) {
  $14 = HEAP32[$9>>2]|0;
  $15 = (($12) + ($14)|0);
  HEAP8[$15>>0] = 0;
  $16 = HEAP32[$8>>2]|0;
  $17 = ($16|0)==(0);
  if (!($17)) {
   $18 = HEAP8[$11>>0]|0;
   $19 = $18 & 4;
   $20 = ($19<<24>>24)==(0);
   if ($20) {
    $21 = (_strAccumFinishRealloc($3)|0);
    $$0 = $21;
    STACKTOP = sp;return ($$0|0);
   }
  }
 }
 $22 = HEAP32[$6>>2]|0;
 $$0 = $22;
 STACKTOP = sp;return ($$0|0);
}
function _sqlite3_mprintf($0,$varargs) {
 $0 = $0|0;
 $varargs = $varargs|0;
 var $$0 = 0, $$0$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $3 = 0;
 var $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 112|0;
 $1 = sp + 16|0;
 $2 = sp + 88|0;
 $3 = sp;
 $4 = (_sqlite3_initialize()|0);
 $5 = ($4|0)==(0);
 if (!($5)) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 HEAP32[$3>>2] = $varargs;
 $6 = (_sqlite3_initialize()|0);
 $7 = ($6|0)==(0);
 do {
  if ($7) {
   $8 = ((($2)) + 4|0);
   HEAP32[$8>>2] = $1;
   HEAP32[$2>>2] = 0;
   $9 = ((($2)) + 8|0);
   HEAP32[$9>>2] = 70;
   $10 = ((($2)) + 12|0);
   HEAP32[$10>>2] = 1000000000;
   $11 = ((($2)) + 16|0);
   HEAP32[$11>>2] = 0;
   $12 = ((($2)) + 20|0);
   HEAP8[$12>>0] = 0;
   $13 = ((($2)) + 21|0);
   HEAP8[$13>>0] = 0;
   _sqlite3_str_vappendf($2,$0,$3);
   $14 = HEAP32[$8>>2]|0;
   $15 = ($14|0)==(0|0);
   if (!($15)) {
    $16 = HEAP32[$11>>2]|0;
    $17 = (($14) + ($16)|0);
    HEAP8[$17>>0] = 0;
    $18 = HEAP32[$10>>2]|0;
    $19 = ($18|0)==(0);
    if (!($19)) {
     $20 = HEAP8[$13>>0]|0;
     $21 = $20 & 4;
     $22 = ($21<<24>>24)==(0);
     if ($22) {
      $23 = (_strAccumFinishRealloc($2)|0);
      $$0$i = $23;
      break;
     }
    }
   }
   $24 = HEAP32[$8>>2]|0;
   $$0$i = $24;
  } else {
   $$0$i = 0;
  }
 } while(0);
 $$0 = $$0$i;
 STACKTOP = sp;return ($$0|0);
}
function _sqlite3_snprintf($0,$1,$2,$varargs) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $varargs = $varargs|0;
 var $10 = 0, $11 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 48|0;
 $3 = sp + 16|0;
 $4 = sp;
 HEAP32[$4>>2] = $varargs;
 $5 = ($0|0)<(1);
 if ($5) {
  STACKTOP = sp;return ($1|0);
 }
 $6 = ((($3)) + 4|0);
 HEAP32[$6>>2] = $1;
 HEAP32[$3>>2] = 0;
 $7 = ((($3)) + 8|0);
 HEAP32[$7>>2] = $0;
 $8 = ((($3)) + 12|0);
 $9 = ((($3)) + 16|0);
 ;HEAP32[$8>>2]=0|0;HEAP32[$8+4>>2]=0|0;HEAP16[$8+8>>1]=0|0;
 _sqlite3_str_vappendf($3,$2,$4);
 $10 = HEAP32[$9>>2]|0;
 $11 = (($1) + ($10)|0);
 HEAP8[$11>>0] = 0;
 STACKTOP = sp;return ($1|0);
}
function _sqlite3_log($0,$1,$varargs) {
 $0 = $0|0;
 $1 = $1|0;
 $varargs = $varargs|0;
 var $$0$i$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $3 = 0, $4 = 0, $5 = 0;
 var $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 256|0;
 $2 = sp + 232|0;
 $3 = sp + 16|0;
 $4 = sp;
 $5 = HEAP32[(29548)>>2]|0;
 $6 = ($5|0)==(0|0);
 if ($6) {
  STACKTOP = sp;return;
 }
 HEAP32[$4>>2] = $varargs;
 $7 = ((($2)) + 4|0);
 HEAP32[$7>>2] = $3;
 HEAP32[$2>>2] = 0;
 $8 = ((($2)) + 8|0);
 HEAP32[$8>>2] = 210;
 $9 = ((($2)) + 12|0);
 $10 = ((($2)) + 21|0);
 ;HEAP32[$9>>2]=0|0;HEAP32[$9+4>>2]=0|0;HEAP16[$9+8>>1]=0|0;
 _sqlite3_str_vappendf($2,$1,$4);
 $11 = HEAP32[(29548)>>2]|0;
 $12 = HEAP32[(29552)>>2]|0;
 $13 = HEAP32[$7>>2]|0;
 $14 = ($13|0)==(0|0);
 if ($14) {
  label = 6;
 } else {
  $15 = ((($2)) + 16|0);
  $16 = HEAP32[$15>>2]|0;
  $17 = (($13) + ($16)|0);
  HEAP8[$17>>0] = 0;
  $18 = HEAP32[$9>>2]|0;
  $19 = ($18|0)==(0);
  if ($19) {
   label = 6;
  } else {
   $20 = HEAP8[$10>>0]|0;
   $21 = $20 & 4;
   $22 = ($21<<24>>24)==(0);
   if ($22) {
    $23 = (_strAccumFinishRealloc($2)|0);
    $$0$i$i = $23;
   } else {
    label = 6;
   }
  }
 }
 if ((label|0) == 6) {
  $24 = HEAP32[$7>>2]|0;
  $$0$i$i = $24;
 }
 FUNCTION_TABLE_viii[$11 & 255]($12,$0,$$0$i$i);
 STACKTOP = sp;return;
}
function _sqlite3_str_appendf($0,$1,$varargs) {
 $0 = $0|0;
 $1 = $1|0;
 $varargs = $varargs|0;
 var $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $2 = sp;
 HEAP32[$2>>2] = $varargs;
 _sqlite3_str_vappendf($0,$1,$2);
 STACKTOP = sp;return;
}
function _sqlite3_randomness($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$01112$i = 0, $$023 = 0, $$024 = 0, $$026 = 0, $$125 = 0, $$3 = 0, $$31 = 0, $$32 = 0, $$33 = 0, $$34 = 0, $$35 = 0, $$promoted = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0;
 var $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0;
 var $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $exitcond = 0, $exitcond27 = 0, $or$cond = 0;
 var $spec$select = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 256|0;
 $2 = sp;
 $3 = (_sqlite3_initialize()|0);
 $4 = ($3|0)==(0);
 if (!($4)) {
  STACKTOP = sp;return;
 }
 $5 = ($0|0)<(1);
 $6 = ($1|0)==(0|0);
 $or$cond = $5 | $6;
 if ($or$cond) {
  HEAP8[59955] = 0;
  STACKTOP = sp;return;
 }
 $7 = HEAP8[59955]|0;
 $8 = ($7<<24>>24)==(0);
 if ($8) {
  HEAP8[59957] = 0;
  HEAP8[59956] = 0;
  $9 = (_sqlite3_initialize()|0);
  $10 = ($9|0)==(0);
  $$01112$i = HEAP32[14770]|0;
  $spec$select = $10 ? $$01112$i : 0;
  $11 = ((($spec$select)) + 56|0);
  $12 = HEAP32[$11>>2]|0;
  (FUNCTION_TABLE_iiii[$12 & 127]($spec$select,256,$2)|0);
  $$026 = 0;
  while(1) {
   $13 = $$026&255;
   $$3 = (59958 + ($$026)|0);
   HEAP8[$$3>>0] = $13;
   $14 = (($$026) + 1)|0;
   $exitcond27 = ($14|0)==(256);
   if ($exitcond27) {
    break;
   } else {
    $$026 = $14;
   }
  }
  $$promoted = HEAP8[59957]|0;
  $$125 = 0;$22 = $$promoted;
  while(1) {
   $$32 = (59958 + ($$125)|0);
   $15 = HEAP8[$$32>>0]|0;
   $16 = $15&255;
   $17 = (($2) + ($$125)|0);
   $18 = HEAP8[$17>>0]|0;
   $19 = $18&255;
   $20 = (($19) + ($16))|0;
   $21 = $22&255;
   $23 = (($20) + ($21))|0;
   $24 = $23&255;
   $25 = $23 & 255;
   $$31 = (59958 + ($25)|0);
   $26 = HEAP8[$$31>>0]|0;
   HEAP8[$$31>>0] = $15;
   HEAP8[$$32>>0] = $26;
   $27 = (($$125) + 1)|0;
   $exitcond = ($27|0)==(256);
   if ($exitcond) {
    break;
   } else {
    $$125 = $27;$22 = $24;
   }
  }
  HEAP8[59957] = $24;
  HEAP8[59955] = 1;
 }
 $$023 = $1;$$024 = $0;
 while(1) {
  $28 = HEAP8[59956]|0;
  $29 = (($28) + 1)<<24>>24;
  HEAP8[59956] = $29;
  $30 = $29&255;
  $$35 = (59958 + ($30)|0);
  $31 = HEAP8[$$35>>0]|0;
  $32 = $31&255;
  $33 = HEAP8[59957]|0;
  $34 = $33&255;
  $35 = (($34) + ($32))|0;
  $36 = $35&255;
  HEAP8[59957] = $36;
  $37 = $35 & 255;
  $$34 = (59958 + ($37)|0);
  $38 = HEAP8[$$34>>0]|0;
  HEAP8[$$35>>0] = $38;
  HEAP8[$$34>>0] = $31;
  $39 = HEAP8[$$35>>0]|0;
  $40 = $39&255;
  $41 = (($40) + ($32))|0;
  $42 = $41 & 255;
  $$33 = (59958 + ($42)|0);
  $43 = HEAP8[$$33>>0]|0;
  $44 = ((($$023)) + 1|0);
  HEAP8[$$023>>0] = $43;
  $45 = (($$024) + -1)|0;
  $46 = ($45|0)==(0);
  if ($46) {
   break;
  } else {
   $$023 = $44;$$024 = $45;
  }
 }
 STACKTOP = sp;return;
}
function _sqlite3_stricmp($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0 = 0, $$01315$i = 0, $$016$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i = 0, $or$cond14$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 if ($2) {
  $3 = ($1|0)!=(0|0);
  $4 = $3 << 31 >> 31;
  $$0 = $4;
  return ($$0|0);
 }
 $5 = ($1|0)==(0|0);
 if ($5) {
  $$0 = 1;
  return ($$0|0);
 }
 $6 = HEAP8[$0>>0]|0;
 $7 = $6&255;
 $8 = (208 + ($7)|0);
 $9 = HEAP8[$8>>0]|0;
 $10 = $9&255;
 $11 = HEAP8[$1>>0]|0;
 $12 = $11&255;
 $13 = (208 + ($12)|0);
 $14 = HEAP8[$13>>0]|0;
 $15 = $14&255;
 $16 = (($10) - ($15))|0;
 $17 = ($16|0)!=(0);
 $18 = ($6<<24>>24)==(0);
 $or$cond14$i = $18 | $17;
 if ($or$cond14$i) {
  $$0 = $16;
  return ($$0|0);
 }
 $$01315$i = $1;$$016$i = $0;
 while(1) {
  $19 = ((($$016$i)) + 1|0);
  $20 = ((($$01315$i)) + 1|0);
  $21 = HEAP8[$19>>0]|0;
  $22 = $21&255;
  $23 = (208 + ($22)|0);
  $24 = HEAP8[$23>>0]|0;
  $25 = $24&255;
  $26 = HEAP8[$20>>0]|0;
  $27 = $26&255;
  $28 = (208 + ($27)|0);
  $29 = HEAP8[$28>>0]|0;
  $30 = $29&255;
  $31 = (($25) - ($30))|0;
  $32 = ($31|0)!=(0);
  $33 = ($21<<24>>24)==(0);
  $or$cond$i = $33 | $32;
  if ($or$cond$i) {
   $$0 = $31;
   break;
  } else {
   $$01315$i = $20;$$016$i = $19;
  }
 }
 return ($$0|0);
}
function _sqlite3StrICmp($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$01315 = 0, $$016 = 0, $$lcssa = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond = 0, $or$cond14 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = HEAP8[$0>>0]|0;
 $3 = $2&255;
 $4 = (208 + ($3)|0);
 $5 = HEAP8[$4>>0]|0;
 $6 = $5&255;
 $7 = HEAP8[$1>>0]|0;
 $8 = $7&255;
 $9 = (208 + ($8)|0);
 $10 = HEAP8[$9>>0]|0;
 $11 = $10&255;
 $12 = (($6) - ($11))|0;
 $13 = ($12|0)!=(0);
 $14 = ($2<<24>>24)==(0);
 $or$cond14 = $14 | $13;
 if ($or$cond14) {
  $$lcssa = $12;
  return ($$lcssa|0);
 }
 $$01315 = $1;$$016 = $0;
 while(1) {
  $15 = ((($$016)) + 1|0);
  $16 = ((($$01315)) + 1|0);
  $17 = HEAP8[$15>>0]|0;
  $18 = $17&255;
  $19 = (208 + ($18)|0);
  $20 = HEAP8[$19>>0]|0;
  $21 = $20&255;
  $22 = HEAP8[$16>>0]|0;
  $23 = $22&255;
  $24 = (208 + ($23)|0);
  $25 = HEAP8[$24>>0]|0;
  $26 = $25&255;
  $27 = (($21) - ($26))|0;
  $28 = ($27|0)!=(0);
  $29 = ($17<<24>>24)==(0);
  $or$cond = $29 | $28;
  if ($or$cond) {
   $$lcssa = $27;
   break;
  } else {
   $$01315 = $16;$$016 = $15;
  }
 }
 return ($$lcssa|0);
}
function _sqlite3_strnicmp($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$01722 = 0, $$019 = 0, $$023 = 0, $$in = 0, $$phi$trans$insert28 = 0, $$phi$trans$insert29 = 0, $$pre$phi37Z2D = 0, $$pre27 = 0, $$pre30 = 0, $$pre36 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0;
 var $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = ($0|0)==(0|0);
 if ($3) {
  $4 = ($1|0)!=(0|0);
  $5 = $4 << 31 >> 31;
  $$019 = $5;
  return ($$019|0);
 }
 $6 = ($1|0)==(0|0);
 if ($6) {
  $$019 = 1;
  return ($$019|0);
 }
 $7 = ($2|0)>(0);
 if (!($7)) {
  $$019 = 0;
  return ($$019|0);
 }
 $$01722 = $0;$$023 = $1;$$in = $2;
 while(1) {
  $8 = (($$in) + -1)|0;
  $9 = HEAP8[$$01722>>0]|0;
  $10 = ($9<<24>>24)==(0);
  if ($10) {
   label = 7;
   break;
  }
  $11 = $9&255;
  $12 = (208 + ($11)|0);
  $13 = HEAP8[$12>>0]|0;
  $14 = HEAP8[$$023>>0]|0;
  $15 = $14&255;
  $16 = (208 + ($15)|0);
  $17 = HEAP8[$16>>0]|0;
  $18 = ($13<<24>>24)==($17<<24>>24);
  if (!($18)) {
   label = 9;
   break;
  }
  $19 = ((($$01722)) + 1|0);
  $20 = ((($$023)) + 1|0);
  $21 = ($$in|0)>(1);
  if ($21) {
   $$01722 = $19;$$023 = $20;$$in = $8;
  } else {
   $$019 = 0;
   label = 12;
   break;
  }
 }
 if ((label|0) == 7) {
  $$pre27 = HEAP8[$$023>>0]|0;
  $$phi$trans$insert28 = $$pre27&255;
  $$phi$trans$insert29 = (208 + ($$phi$trans$insert28)|0);
  $$pre30 = HEAP8[$$phi$trans$insert29>>0]|0;
  $$pre$phi37Z2D = 0;$23 = $$pre30;
 }
 else if ((label|0) == 9) {
  $$pre36 = $13&255;
  $$pre$phi37Z2D = $$pre36;$23 = $17;
 }
 else if ((label|0) == 12) {
  return ($$019|0);
 }
 $22 = $23&255;
 $24 = (($$pre$phi37Z2D) - ($22))|0;
 $$019 = $24;
 return ($$019|0);
}
function _unixOpen($0,$1,$2,$3,$4) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 $4 = $4|0;
 var $$ = 0, $$$i$i = 0, $$0 = 0, $$0$i$i = 0, $$0$i$i155 = 0, $$0$i37$i = 0, $$0$i38$i = 0, $$0124 = 0, $$0125 = 0, $$0126187 = 0, $$0129 = 0, $$0133185 = 0, $$0140179 = 0, $$01623$i$i = 0, $$01623$i$i$i = 0, $$0165 = 0, $$0168 = 0, $$019$i = 0, $$01930$i = 0, $$01931$i = 0;
 var $$027$i$ph$i = 0, $$029$i = 0, $$032$i$i = 0, $$034$i = 0, $$034$i$i = 0, $$034$lcssa$i = 0, $$03444$i = 0, $$0505661$i = 0, $$1 = 0, $$1$i$i = 0, $$1$i154 = 0, $$1145 = 0, $$1166 = 0, $$1169 = 0, $$2128 = 0, $$2131 = 0, $$2135 = 0, $$2164 = 0, $$2164175 = 0, $$2167 = 0;
 var $$2167174 = 0, $$2170 = 0, $$2170173 = 0, $$241 = 0, $$3 = 0, $$3132186 = 0, $$3139 = 0, $$3143$ph = 0, $$3147 = 0, $$5 = 0, $$in$i53$i = 0, $$pre = 0, $$pre$i = 0, $$pre$phi214Z2D = 0, $$pre$phiZ2D = 0, $$pre213 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0;
 var $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0;
 var $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0;
 var $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0;
 var $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0;
 var $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0;
 var $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0;
 var $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0;
 var $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0;
 var $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0;
 var $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0;
 var $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0;
 var $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0;
 var $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0;
 var $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0;
 var $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0;
 var $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $39 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0;
 var $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0;
 var $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0;
 var $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $cond$i$i = 0, $cond33$i$i = 0, $or$cond$i = 0, $or$cond3 = 0, $or$cond40$i = 0, $phitmp$i = 0;
 var $spec$select = 0, $spec$select$i = 0, $spec$select$i$i$i = 0, $spec$select$i$i54$i = 0, $spec$select151 = 0, $switch$split2D = 0, $switch$split332D = 0, $vararg_buffer = 0, $vararg_buffer10 = 0, $vararg_buffer17 = 0, $vararg_buffer20 = 0, $vararg_buffer3 = 0, $vararg_ptr1 = 0, $vararg_ptr13 = 0, $vararg_ptr14 = 0, $vararg_ptr15 = 0, $vararg_ptr16 = 0, $vararg_ptr2 = 0, $vararg_ptr23 = 0, $vararg_ptr24 = 0;
 var $vararg_ptr25 = 0, $vararg_ptr26 = 0, $vararg_ptr6 = 0, $vararg_ptr7 = 0, $vararg_ptr8 = 0, $vararg_ptr9 = 0, dest = 0, label = 0, sp = 0, stop = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 1232|0;
 $vararg_buffer20 = sp + 1200|0;
 $vararg_buffer17 = sp + 1192|0;
 $vararg_buffer10 = sp + 1168|0;
 $vararg_buffer3 = sp + 1144|0;
 $vararg_buffer = sp + 1128|0;
 $5 = sp + 1048|0;
 $6 = sp + 528|0;
 $7 = sp;
 $8 = $3 & -256;
 $9 = $3 & 16;
 $10 = $3 & 8;
 $11 = $3 & 4;
 $12 = $3 & 1;
 $13 = $3 & 2;
 $14 = ($11|0)==(0);
 L1: do {
  if ($14) {
   $210 = 0;
  } else {
   $switch$split2D = ($8|0)<(16384);
   L3: do {
    if ($switch$split2D) {
     switch ($8|0) {
     case 2048:  {
      $210 = 1;
      break L1;
      break;
     }
     default: {
     }
     }
    } else {
     $switch$split332D = ($8|0)<(524288);
     if ($switch$split332D) {
      switch ($8|0) {
      case 16384:  {
       $210 = 1;
       break L1;
       break;
      }
      default: {
       break L3;
      }
      }
     } else {
      switch ($8|0) {
      case 524288:  {
       $210 = 1;
       break L1;
       break;
      }
      default: {
       break L3;
      }
      }
     }
    }
   } while(0);
   $210 = 0;
  }
 } while(0);
 $15 = HEAP32[14796]|0;
 $16 = (_getpid()|0);
 $17 = ($15|0)==($16|0);
 if (!($17)) {
  $18 = (_getpid()|0);
  HEAP32[14796] = $18;
  $19 = (_sqlite3_initialize()|0);
  $20 = ($19|0)==(0);
  if ($20) {
   HEAP8[59955] = 0;
  }
 }
 dest=$2; stop=dest+52|0; do { HEAP32[dest>>2]=0|0; dest=dest+4|0; } while ((dest|0) < (stop|0));
 $21 = ($8|0)==(256);
 if ($21) {
  $22 = HEAP32[14795]|0;
  $23 = ($22|0)==(0|0);
  L17: do {
   if ($23) {
    label = 20;
   } else {
    $24 = HEAP32[(2388)>>2]|0;
    $25 = (FUNCTION_TABLE_iii[$24 & 255]($1,$5)|0);
    $26 = ($25|0)==(0);
    if ($26) {
     $$01930$i = HEAP32[14795]|0;
     $27 = ($$01930$i|0)==(0|0);
     if ($27) {
      label = 20;
     } else {
      $28 = HEAP32[$5>>2]|0;
      $29 = ((($5)) + 72|0);
      $30 = HEAP32[$29>>2]|0;
      $$01931$i = $$01930$i;
      while(1) {
       $31 = HEAP32[$$01931$i>>2]|0;
       $32 = ($31|0)==($28|0);
       if ($32) {
        $33 = ((($$01931$i)) + 8|0);
        $34 = $33;
        $35 = $34;
        $36 = HEAP32[$35>>2]|0;
        $37 = (($34) + 4)|0;
        $38 = $37;
        $39 = HEAP32[$38>>2]|0;
        $40 = ($36|0)==($30|0);
        $41 = ($39|0)==(0);
        $42 = $40 & $41;
        if ($42) {
         break;
        }
       }
       $43 = ((($$01931$i)) + 44|0);
       $$019$i = HEAP32[$43>>2]|0;
       $44 = ($$019$i|0)==(0|0);
       if ($44) {
        label = 20;
        break L17;
       } else {
        $$01931$i = $$019$i;
       }
      }
      $45 = ((($$01931$i)) + 32|0);
      $46 = HEAP32[$45>>2]|0;
      $47 = ($46|0)==(0|0);
      if ($47) {
       label = 20;
      } else {
       $$029$i = $45;$52 = $46;
       while(1) {
        $51 = ((($52)) + 4|0);
        $53 = HEAP32[$51>>2]|0;
        $54 = ($53|0)==($3|0);
        $49 = ((($52)) + 8|0);
        if ($54) {
         break;
        }
        $48 = HEAP32[$49>>2]|0;
        $50 = ($48|0)==(0|0);
        if ($50) {
         label = 20;
         break L17;
        } else {
         $$029$i = $49;$52 = $48;
        }
       }
       $55 = HEAP32[$49>>2]|0;
       HEAP32[$$029$i>>2] = $55;
       $56 = HEAP32[$52>>2]|0;
       $$0124 = $52;$$0129 = $56;
      }
     }
    } else {
     label = 20;
    }
   }
  } while(0);
  do {
   if ((label|0) == 20) {
    $57 = (_sqlite3_initialize()|0);
    $58 = ($57|0)==(0);
    if (!($58)) {
     $$3 = 7;
     STACKTOP = sp;return ($$3|0);
    }
    $59 = HEAP32[7324]|0;
    $60 = ($59|0)==(0);
    if ($60) {
     $99 = HEAP32[(29340)>>2]|0;
     $100 = (FUNCTION_TABLE_ii[$99 & 127](12)|0);
     $101 = ($100|0)==(0|0);
     if ($101) {
      $$3 = 7;
     } else {
      $$0124 = $100;$$0129 = -1;
      break;
     }
     STACKTOP = sp;return ($$3|0);
    }
    $61 = HEAP32[(29356)>>2]|0;
    $62 = (FUNCTION_TABLE_ii[$61 & 127](12)|0);
    $63 = HEAP32[14985]|0;
    $64 = ($63>>>0)<(12);
    if ($64) {
     HEAP32[14985] = 12;
    }
    $65 = (59064);
    $66 = $65;
    $67 = HEAP32[$66>>2]|0;
    $68 = (($65) + 4)|0;
    $69 = $68;
    $70 = HEAP32[$69>>2]|0;
    $71 = ($70|0)>(0);
    $72 = ($67>>>0)>(0);
    $73 = ($70|0)==(0);
    $74 = $73 & $72;
    $75 = $71 | $74;
    if ($75) {
     $76 = HEAP32[14978]|0;
     $77 = ($62|0)<(0);
     $78 = $77 << 31 >> 31;
     $79 = (_i64Subtract(($67|0),($70|0),($62|0),($78|0))|0);
     $80 = (getTempRet0() | 0);
     $81 = ($80|0)<(0);
     $82 = ($79>>>0)<=($76>>>0);
     $83 = ($80|0)==(0);
     $84 = $83 & $82;
     $85 = $81 | $84;
     $$ = $85&1;
     HEAP32[(59072)>>2] = $$;
    }
    $86 = HEAP32[(29340)>>2]|0;
    $87 = (FUNCTION_TABLE_ii[$86 & 127]($62)|0);
    $88 = ($87|0)==(0|0);
    if ($88) {
     $$3 = 7;
     STACKTOP = sp;return ($$3|0);
    }
    $89 = HEAP32[(29352)>>2]|0;
    $90 = (FUNCTION_TABLE_ii[$89 & 127]($87)|0);
    $91 = HEAP32[14978]|0;
    $92 = (($91) + ($90))|0;
    HEAP32[14978] = $92;
    $93 = HEAP32[14982]|0;
    $94 = ($92>>>0)>($93>>>0);
    if ($94) {
     HEAP32[14982] = $92;
    }
    $95 = HEAP32[14981]|0;
    $96 = (($95) + 1)|0;
    HEAP32[14981] = $96;
    $97 = HEAP32[14987]|0;
    $98 = ($96>>>0)>($97>>>0);
    if ($98) {
     HEAP32[14987] = $96;
    }
    $$0124 = $87;$$0129 = -1;
   }
  } while(0);
  $102 = ((($2)) + 28|0);
  HEAP32[$102>>2] = $$0124;
  $$0125 = $1;$$2131 = $$0129;
 } else {
  $103 = ($1|0)==(0|0);
  if ($103) {
   $104 = ((($0)) + 8|0);
   $105 = HEAP32[$104>>2]|0;
   $106 = (_unixGetTempname($105,$7)|0);
   $107 = ($106|0)==(0);
   if ($107) {
    $$0125 = $7;$$2131 = -1;
   } else {
    $$3 = $106;
    STACKTOP = sp;return ($$3|0);
   }
  } else {
   $$0125 = $1;$$2131 = -1;
  }
 }
 $108 = ($13|0)!=(0);
 $109 = $11 << 4;
 $110 = $109 | $13;
 $111 = ($9|0)==(0);
 $112 = $110 | 131200;
 $$3139 = $111 ? $110 : $112;
 $113 = ($$2131|0)<(0);
 L61: do {
  if ($113) {
   $114 = $3 & 526336;
   $115 = ($114|0)==(0);
   L63: do {
    if ($115) {
     $139 = ($10|0)==(0);
     if ($139) {
      $140 = $3 & 64;
      $141 = ($140|0)==(0);
      $142 = ($$0125|0)==(0|0);
      $or$cond40$i = $141 | $142;
      if ($or$cond40$i) {
       $$2164175 = 0;$$2167174 = 0;$$2170173 = 0;
      } else {
       $143 = (_strlen($$0125)|0);
       $144 = $143 & 1073741823;
       $145 = (($144) + 1)|0;
       $146 = (($$0125) + ($145)|0);
       $147 = HEAP8[$146>>0]|0;
       $148 = ($147<<24>>24)==(0);
       if ($148) {
        $$2164175 = 0;$$2167174 = 0;$$2170173 = 0;
       } else {
        $$01623$i$i = $146;
        while(1) {
         $149 = (_strcmp($$01623$i$i,32118)|0);
         $150 = (_strlen($$01623$i$i)|0);
         $151 = $150 & 1073741823;
         $152 = (($151) + 1)|0;
         $153 = (($$01623$i$i) + ($152)|0);
         $154 = ($149|0)==(0);
         if ($154) {
          break;
         }
         $155 = (_strlen($153)|0);
         $156 = $155 & 1073741823;
         $157 = (($156) + 1)|0;
         $158 = (($153) + ($157)|0);
         $159 = HEAP8[$158>>0]|0;
         $160 = ($159<<24>>24)==(0);
         if ($160) {
          $$2164175 = 0;$$2167174 = 0;$$2170173 = 0;
          break L63;
         } else {
          $$01623$i$i = $158;
         }
        }
        $161 = HEAP32[(2388)>>2]|0;
        $162 = (FUNCTION_TABLE_iii[$161 & 255]($153,$5)|0);
        $163 = ($162|0)==(0);
        if ($163) {
         $164 = ((($5)) + 12|0);
         $165 = HEAP32[$164>>2]|0;
         $166 = $165 & 511;
         $167 = ((($5)) + 20|0);
         $168 = HEAP32[$167>>2]|0;
         $169 = ((($5)) + 24|0);
         $170 = HEAP32[$169>>2]|0;
         $$0$i38$i = 0;$$1 = $170;$$1166 = $168;$$1169 = $166;
        } else {
         $$0$i38$i = 1802;$$1 = 0;$$1166 = 0;$$1169 = 0;
        }
        $$1$i154 = $$0$i38$i;$$2164 = $$1;$$2167 = $$1166;$$2170 = $$1169;
        label = 57;
       }
      }
     } else {
      $$2164175 = 0;$$2167174 = 0;$$2170173 = 384;
     }
    } else {
     $116 = ($$0125|0)==(0|0);
     if ($116) {
      $$0$i$i = -1;
     } else {
      $117 = (_strlen($$0125)|0);
      $118 = $117 & 1073741823;
      $phitmp$i = (($118) + -1)|0;
      $$0$i$i = $phitmp$i;
     }
     $119 = (($$0125) + ($$0$i$i)|0);
     $120 = HEAP8[$119>>0]|0;
     $121 = ($120<<24>>24)==(45);
     L79: do {
      if ($121) {
       $$034$lcssa$i = $$0$i$i;
      } else {
       $$03444$i = $$0$i$i;$127 = $120;
       while(1) {
        $125 = ($$03444$i|0)==(0);
        $126 = ($127<<24>>24)==(46);
        $or$cond$i = $126 | $125;
        if ($or$cond$i) {
         break;
        }
        $$034$i = (($$03444$i) + -1)|0;
        $122 = (($$0125) + ($$034$i)|0);
        $123 = HEAP8[$122>>0]|0;
        $124 = ($123<<24>>24)==(45);
        if ($124) {
         $$034$lcssa$i = $$034$i;
         break L79;
        } else {
         $$03444$i = $$034$i;$127 = $123;
        }
       }
       $$2164175 = 0;$$2167174 = 0;$$2170173 = 0;
       break L63;
      }
     } while(0);
     _memcpy(($6|0),($$0125|0),($$034$lcssa$i|0))|0;
     $128 = (($6) + ($$034$lcssa$i)|0);
     HEAP8[$128>>0] = 0;
     $129 = HEAP32[(2388)>>2]|0;
     $130 = (FUNCTION_TABLE_iii[$129 & 255]($6,$5)|0);
     $131 = ($130|0)==(0);
     if ($131) {
      $132 = ((($5)) + 12|0);
      $133 = HEAP32[$132>>2]|0;
      $134 = $133 & 511;
      $135 = ((($5)) + 20|0);
      $136 = HEAP32[$135>>2]|0;
      $137 = ((($5)) + 24|0);
      $138 = HEAP32[$137>>2]|0;
      $$0 = $138;$$0$i37$i = 0;$$0165 = $136;$$0168 = $134;
     } else {
      $$0 = 0;$$0$i37$i = 1802;$$0165 = 0;$$0168 = 0;
     }
     $$1$i154 = $$0$i37$i;$$2164 = $$0;$$2167 = $$0165;$$2170 = $$0168;
     label = 57;
    }
   } while(0);
   if ((label|0) == 57) {
    $171 = ($$1$i154|0)==(0);
    if ($171) {
     $$2164175 = $$2164;$$2167174 = $$2167;$$2170173 = $$2170;
    } else {
     $$3 = $$1$i154;
     STACKTOP = sp;return ($$3|0);
    }
   }
   $172 = (_robust_open($$0125,$$3139,$$2170173)|0);
   $173 = ($172|0)<(0);
   do {
    if ($173) {
     if ($210) {
      $174 = (___errno_location()|0);
      $175 = HEAP32[$174>>2]|0;
      $176 = ($175|0)==(13);
      if ($176) {
       $177 = HEAP32[(2364)>>2]|0;
       $178 = (FUNCTION_TABLE_iii[$177 & 255]($$0125,0)|0);
       $179 = ($178|0)==(0);
       if ($179) {
        label = 62;
       } else {
        $$0140179 = 1544;
       }
      } else {
       label = 62;
      }
     } else {
      label = 62;
     }
     if ((label|0) == 62) {
      $180 = (___errno_location()|0);
      $181 = HEAP32[$180>>2]|0;
      $182 = ($181|0)!=(21);
      $or$cond3 = $108 & $182;
      if ($or$cond3) {
       $183 = $$3139 & 131200;
       $184 = (_robust_open($$0125,$183,$$2170173)|0);
       $185 = ($184|0)<(0);
       if ($185) {
        $$0140179 = 0;
       } else {
        $186 = $3 & -8;
        $187 = $186 | 1;
        $$pre213 = $3 & 526336;
        $$0126187 = $187;$$0133185 = 1;$$3132186 = $184;$$pre$phi214Z2D = $$pre213;
        break;
       }
      } else {
       $$0140179 = 0;
      }
     }
     HEAP32[$vararg_buffer>>2] = 31984;
     $vararg_ptr1 = ((($vararg_buffer)) + 4|0);
     HEAP32[$vararg_ptr1>>2] = 38565;
     $vararg_ptr2 = ((($vararg_buffer)) + 8|0);
     HEAP32[$vararg_ptr2>>2] = (31517);
     _sqlite3_log(14,32001,$vararg_buffer);
     $193 = (___errno_location()|0);
     $194 = HEAP32[$193>>2]|0;
     $195 = (_strerror($194)|0);
     $196 = ($$0125|0)==(0|0);
     $spec$select$i = $196 ? 59952 : $$0125;
     HEAP32[$vararg_buffer3>>2] = 38565;
     $vararg_ptr6 = ((($vararg_buffer3)) + 4|0);
     HEAP32[$vararg_ptr6>>2] = $194;
     $vararg_ptr7 = ((($vararg_buffer3)) + 8|0);
     HEAP32[$vararg_ptr7>>2] = 31690;
     $vararg_ptr8 = ((($vararg_buffer3)) + 12|0);
     HEAP32[$vararg_ptr8>>2] = $spec$select$i;
     $vararg_ptr9 = ((($vararg_buffer3)) + 16|0);
     HEAP32[$vararg_ptr9>>2] = $195;
     _sqlite3_log(14,31953,$vararg_buffer3);
     $197 = ($$0140179|0)==(0);
     $spec$select = $197 ? 14 : $$0140179;
     $$pre = ((($2)) + 28|0);
     $$3143$ph = $spec$select;$$pre$phiZ2D = $$pre;
     break L61;
    } else {
     $$0126187 = $3;$$0133185 = $12;$$3132186 = $172;$$pre$phi214Z2D = $114;
    }
   } while(0);
   $188 = ($$pre$phi214Z2D|0)==(0);
   if ($188) {
    $$2128 = $$0126187;$$2135 = $$0133185;$$5 = $$3132186;
    label = 69;
   } else {
    $189 = HEAP32[(2592)>>2]|0;
    $190 = (FUNCTION_TABLE_i[$189 & 127]()|0);
    $191 = ($190|0)==(0);
    if ($191) {
     $192 = HEAP32[(2580)>>2]|0;
     (FUNCTION_TABLE_iiii[$192 & 127]($$3132186,$$2167174,$$2164175)|0);
     $$2128 = $$0126187;$$2135 = $$0133185;$$5 = $$3132186;
     label = 69;
    } else {
     $$2128 = $$0126187;$$2135 = $$0133185;$$5 = $$3132186;
     label = 69;
    }
   }
  } else {
   $$2128 = $3;$$2135 = $12;$$5 = $$2131;
   label = 69;
  }
 } while(0);
 L106: do {
  if ((label|0) == 69) {
   $198 = ($4|0)==(0|0);
   if (!($198)) {
    HEAP32[$4>>2] = $$2128;
   }
   $199 = ((($2)) + 28|0);
   $200 = HEAP32[$199>>2]|0;
   $201 = ($200|0)==(0|0);
   if (!($201)) {
    HEAP32[$200>>2] = $$5;
    $202 = ((($200)) + 4|0);
    HEAP32[$202>>2] = $$2128;
   }
   $203 = ($10|0)==(0);
   if (!($203)) {
    $204 = HEAP32[(2532)>>2]|0;
    (FUNCTION_TABLE_ii[$204 & 127]($$0125)|0);
   }
   $205 = $10 << 2;
   $206 = ($$2135|0)==(0);
   $207 = $205 | 2;
   $$1145 = $206 ? $205 : $207;
   $208 = $$1145 | 128;
   $spec$select151 = $21 ? $$1145 : $208;
   $209 = $spec$select151 | 8;
   $$3147 = $210 ? $209 : $spec$select151;
   $211 = $$2128 & 64;
   $212 = $$3147 | $211;
   $213 = ((($2)) + 12|0);
   HEAP32[$213>>2] = $$5;
   $214 = ((($2)) + 4|0);
   HEAP32[$214>>2] = $0;
   $215 = ((($2)) + 32|0);
   HEAP32[$215>>2] = $1;
   $216 = $212&65535;
   $217 = $216 & 255;
   $218 = ((($2)) + 18|0);
   HEAP16[$218>>1] = $217;
   $219 = $212 & 64;
   $220 = ($219|0)==(0);
   $221 = ($1|0)==(0|0);
   $222 = $221 | $220;
   L117: do {
    if ($222) {
     $244 = $217;
     label = 82;
    } else {
     $223 = (_strlen($1)|0);
     $224 = $223 & 1073741823;
     $225 = (($224) + 1)|0;
     $226 = (($1) + ($225)|0);
     $227 = HEAP8[$226>>0]|0;
     $228 = ($227<<24>>24)==(0);
     if ($228) {
      $244 = $217;
      label = 82;
     } else {
      $$01623$i$i$i = $226;
      while(1) {
       $229 = (_strcmp($$01623$i$i$i,32125)|0);
       $230 = (_strlen($$01623$i$i$i)|0);
       $231 = $230 & 1073741823;
       $232 = (($231) + 1)|0;
       $233 = (($$01623$i$i$i) + ($232)|0);
       $234 = ($229|0)==(0);
       if ($234) {
        break;
       }
       $235 = (_strlen($233)|0);
       $236 = $235 & 1073741823;
       $237 = (($236) + 1)|0;
       $238 = (($233) + ($237)|0);
       $239 = HEAP8[$238>>0]|0;
       $240 = ($239<<24>>24)==(0);
       if ($240) {
        $244 = $217;
        label = 82;
        break L117;
       } else {
        $$01623$i$i$i = $238;
       }
      }
      $241 = (_getSafetyLevel($233,1,1)|0);
      $242 = ($241<<24>>24)==(0);
      if (!($242)) {
       $$pre$i = HEAP16[$218>>1]|0;
       $244 = $$pre$i;
       label = 82;
      }
     }
    }
   } while(0);
   if ((label|0) == 82) {
    $243 = $244 | 16;
    HEAP16[$218>>1] = $243;
   }
   $245 = ((($0)) + 16|0);
   $246 = HEAP32[$245>>2]|0;
   $247 = (_strcmp($246,31213)|0);
   $248 = ($247|0)==(0);
   if ($248) {
    $249 = HEAP16[$218>>1]|0;
    $250 = $249 | 1;
    HEAP16[$218>>1] = $250;
   }
   $251 = $$3147 & 128;
   $252 = ($251|0)==(0);
   L131: do {
    if ($252) {
     $253 = ((($0)) + 20|0);
     $254 = HEAP32[$253>>2]|0;
     $255 = HEAP32[$254>>2]|0;
     $256 = (FUNCTION_TABLE_iii[$255 & 255]($1,$2)|0);
     $257 = ($256|0)==(29668|0);
     if (!($257)) {
      $350 = ($256|0)==(29820|0);
      if (!($350)) {
       $$0505661$i = $256;
       break;
      }
      $351 = (_strlen($1)|0);
      $352 = (($351) + 6)|0;
      $353 = ($352|0)<(0);
      $354 = $353 << 31 >> 31;
      $355 = (_sqlite3_malloc64($352,$354)|0);
      $356 = ($355|0)==(0|0);
      if (!($356)) {
       HEAP32[$vararg_buffer17>>2] = $1;
       (_sqlite3_snprintf($352,$355,32130,$vararg_buffer17)|0);
       $357 = ((($2)) + 24|0);
       HEAP32[$357>>2] = $355;
       $$0505661$i = 29820;
       break;
      }
      $359 = ((($2)) + 24|0);
      HEAP32[$359>>2] = 0;
      $360 = ((($2)) + 20|0);
      HEAP32[$360>>2] = 0;
      $361 = ($$5|0)>(-1);
      if (!($361)) {
       $$3143$ph = 7;$$pre$phiZ2D = $199;
       break L106;
      }
      $362 = HEAP32[(2352)>>2]|0;
      $363 = (FUNCTION_TABLE_ii[$362 & 127]($$5)|0);
      $364 = ($363|0)==(0);
      if ($364) {
       $$3143$ph = 7;$$pre$phiZ2D = $199;
       break L106;
      }
      $365 = HEAP32[$215>>2]|0;
      $366 = (___errno_location()|0);
      $367 = HEAP32[$366>>2]|0;
      $368 = (_strerror($367)|0);
      $369 = ($365|0)==(0|0);
      $spec$select$i$i54$i = $369 ? 59952 : $365;
      HEAP32[$vararg_buffer20>>2] = 38152;
      $vararg_ptr23 = ((($vararg_buffer20)) + 4|0);
      HEAP32[$vararg_ptr23>>2] = $367;
      $vararg_ptr24 = ((($vararg_buffer20)) + 8|0);
      HEAP32[$vararg_ptr24>>2] = 31695;
      $vararg_ptr25 = ((($vararg_buffer20)) + 12|0);
      HEAP32[$vararg_ptr25>>2] = $spec$select$i$i54$i;
      $vararg_ptr26 = ((($vararg_buffer20)) + 16|0);
      HEAP32[$vararg_ptr26>>2] = $368;
      _sqlite3_log(4106,31953,$vararg_buffer20);
      $$3143$ph = 7;$$pre$phiZ2D = $199;
      break L106;
     }
     $258 = ((($2)) + 8|0);
     $259 = HEAP32[$213>>2]|0;
     $260 = HEAP32[(2400)>>2]|0;
     $261 = (FUNCTION_TABLE_iii[$260 & 255]($259,$6)|0);
     $262 = ($261|0)==(0);
     L142: do {
      if ($262) {
       $267 = $5;
       $268 = $267;
       HEAP32[$268>>2] = 0;
       $269 = (($267) + 4)|0;
       $270 = $269;
       HEAP32[$270>>2] = 0;
       $271 = HEAP32[$6>>2]|0;
       HEAP32[$5>>2] = $271;
       $272 = ((($6)) + 72|0);
       $273 = HEAP32[$272>>2]|0;
       $274 = ((($5)) + 8|0);
       $275 = $274;
       $276 = $275;
       HEAP32[$276>>2] = $273;
       $277 = (($275) + 4)|0;
       $278 = $277;
       HEAP32[$278>>2] = 0;
       $$032$i$i = HEAP32[14795]|0;
       $cond33$i$i = ($$032$i$i|0)==(0|0);
       L144: do {
        if ($cond33$i$i) {
         label = 93;
        } else {
         $$034$i$i = $$032$i$i;
         while(1) {
          $279 = (_memcmp($5,$$034$i$i,16)|0);
          $280 = ($279|0)==(0);
          if ($280) {
           break;
          }
          $281 = ((($$034$i$i)) + 44|0);
          $$0$i$i155 = HEAP32[$281>>2]|0;
          $cond$i$i = ($$0$i$i155|0)==(0|0);
          if ($cond$i$i) {
           label = 93;
           break L144;
          } else {
           $$034$i$i = $$0$i$i155;
          }
         }
         $337 = ((($$034$i$i)) + 36|0);
         $338 = HEAP32[$337>>2]|0;
         $339 = (($338) + 1)|0;
         HEAP32[$337>>2] = $339;
         $$1$i$i = $$034$i$i;
        }
       } while(0);
       if ((label|0) == 93) {
        $282 = (_sqlite3_initialize()|0);
        $283 = ($282|0)==(0);
        if (!($283)) {
         $$027$i$ph$i = 7;
         break;
        }
        $284 = HEAP32[7324]|0;
        $285 = ($284|0)==(0);
        do {
         if ($285) {
          $324 = HEAP32[(29340)>>2]|0;
          $325 = (FUNCTION_TABLE_ii[$324 & 127](56)|0);
          $326 = ($325|0)==(0|0);
          if ($326) {
           $$027$i$ph$i = 7;
           break L142;
          } else {
           $$in$i53$i = $325;
          }
         } else {
          $286 = HEAP32[(29356)>>2]|0;
          $287 = (FUNCTION_TABLE_ii[$286 & 127](56)|0);
          $288 = HEAP32[14985]|0;
          $289 = ($288>>>0)<(56);
          if ($289) {
           HEAP32[14985] = 56;
          }
          $290 = (59064);
          $291 = $290;
          $292 = HEAP32[$291>>2]|0;
          $293 = (($290) + 4)|0;
          $294 = $293;
          $295 = HEAP32[$294>>2]|0;
          $296 = ($295|0)>(0);
          $297 = ($292>>>0)>(0);
          $298 = ($295|0)==(0);
          $299 = $298 & $297;
          $300 = $296 | $299;
          if ($300) {
           $301 = HEAP32[14978]|0;
           $302 = ($287|0)<(0);
           $303 = $302 << 31 >> 31;
           $304 = (_i64Subtract(($292|0),($295|0),($287|0),($303|0))|0);
           $305 = (getTempRet0() | 0);
           $306 = ($305|0)<(0);
           $307 = ($304>>>0)<=($301>>>0);
           $308 = ($305|0)==(0);
           $309 = $308 & $307;
           $310 = $306 | $309;
           $$241 = $310&1;
           HEAP32[(59072)>>2] = $$241;
          }
          $311 = HEAP32[(29340)>>2]|0;
          $312 = (FUNCTION_TABLE_ii[$311 & 127]($287)|0);
          $313 = ($312|0)==(0|0);
          if ($313) {
           $$027$i$ph$i = 7;
           break L142;
          }
          $314 = HEAP32[(29352)>>2]|0;
          $315 = (FUNCTION_TABLE_ii[$314 & 127]($312)|0);
          $316 = HEAP32[14978]|0;
          $317 = (($316) + ($315))|0;
          HEAP32[14978] = $317;
          $318 = HEAP32[14982]|0;
          $319 = ($317>>>0)>($318>>>0);
          if ($319) {
           HEAP32[14982] = $317;
          }
          $320 = HEAP32[14981]|0;
          $321 = (($320) + 1)|0;
          HEAP32[14981] = $321;
          $322 = HEAP32[14987]|0;
          $323 = ($321>>>0)>($322>>>0);
          if (!($323)) {
           $$in$i53$i = $312;
           break;
          }
          HEAP32[14987] = $321;
          $$in$i53$i = $312;
         }
        } while(0);
        $327 = ((($$in$i53$i)) + 16|0);
        dest=$327; stop=dest+40|0; do { HEAP32[dest>>2]=0|0; dest=dest+4|0; } while ((dest|0) < (stop|0));
        ;HEAP32[$$in$i53$i>>2]=HEAP32[$5>>2]|0;HEAP32[$$in$i53$i+4>>2]=HEAP32[$5+4>>2]|0;HEAP32[$$in$i53$i+8>>2]=HEAP32[$5+8>>2]|0;HEAP32[$$in$i53$i+12>>2]=HEAP32[$5+12>>2]|0;
        $328 = HEAP32[(29300)>>2]|0;
        $329 = ($328|0)==(0);
        if (!($329)) {
         HEAP32[$327>>2] = (8);
        }
        $330 = ((($$in$i53$i)) + 36|0);
        HEAP32[$330>>2] = 1;
        $331 = HEAP32[14795]|0;
        $332 = ((($$in$i53$i)) + 44|0);
        HEAP32[$332>>2] = $331;
        $333 = ((($$in$i53$i)) + 48|0);
        HEAP32[$333>>2] = 0;
        $334 = ($331|0)==(0);
        if (!($334)) {
         $335 = $331;
         $336 = ((($335)) + 48|0);
         HEAP32[$336>>2] = $$in$i53$i;
        }
        HEAP32[14795] = $$in$i53$i;
        $$1$i$i = $$in$i53$i;
       }
       HEAP32[$258>>2] = $$1$i$i;
       $$0505661$i = 29668;
       break L131;
      } else {
       $263 = (___errno_location()|0);
       $264 = HEAP32[$263>>2]|0;
       $265 = ((($2)) + 20|0);
       HEAP32[$265>>2] = $264;
       $266 = ($264|0)==(75);
       $$$i$i = $266 ? 22 : 10;
       $$027$i$ph$i = $$$i$i;
      }
     } while(0);
     $340 = HEAP32[(2352)>>2]|0;
     $341 = (FUNCTION_TABLE_ii[$340 & 127]($$5)|0);
     $342 = ($341|0)==(0);
     if (!($342)) {
      $343 = ($2|0)==(0|0);
      if ($343) {
       $349 = 0;
      } else {
       $344 = HEAP32[$215>>2]|0;
       $349 = $344;
      }
      $345 = (___errno_location()|0);
      $346 = HEAP32[$345>>2]|0;
      $347 = (_strerror($346)|0);
      $348 = ($349|0)==(0|0);
      $spec$select$i$i$i = $348 ? 59952 : $349;
      HEAP32[$vararg_buffer10>>2] = 38067;
      $vararg_ptr13 = ((($vararg_buffer10)) + 4|0);
      HEAP32[$vararg_ptr13>>2] = $346;
      $vararg_ptr14 = ((($vararg_buffer10)) + 8|0);
      HEAP32[$vararg_ptr14>>2] = 31695;
      $vararg_ptr15 = ((($vararg_buffer10)) + 12|0);
      HEAP32[$vararg_ptr15>>2] = $spec$select$i$i$i;
      $vararg_ptr16 = ((($vararg_buffer10)) + 16|0);
      HEAP32[$vararg_ptr16>>2] = $347;
      _sqlite3_log(4106,31953,$vararg_buffer10);
     }
     $358 = ((($2)) + 20|0);
     HEAP32[$358>>2] = 0;
     $$3143$ph = $$027$i$ph$i;$$pre$phiZ2D = $199;
     break L106;
    } else {
     $$0505661$i = 29744;
    }
   } while(0);
   $370 = ((($2)) + 20|0);
   HEAP32[$370>>2] = 0;
   HEAP32[$2>>2] = $$0505661$i;
   _verifyDbFile($2);
   $$3 = 0;
   STACKTOP = sp;return ($$3|0);
  }
 } while(0);
 $371 = HEAP32[$$pre$phiZ2D>>2]|0;
 $372 = ($371|0)==(0|0);
 if ($372) {
  $$3 = $$3143$ph;
  STACKTOP = sp;return ($$3|0);
 }
 $373 = HEAP32[7324]|0;
 $374 = ($373|0)==(0);
 if ($374) {
  $382 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$382 & 127]($371);
  $$3 = $$3143$ph;
  STACKTOP = sp;return ($$3|0);
 } else {
  $375 = HEAP32[(29352)>>2]|0;
  $376 = (FUNCTION_TABLE_ii[$375 & 127]($371)|0);
  $377 = HEAP32[14978]|0;
  $378 = (($377) - ($376))|0;
  HEAP32[14978] = $378;
  $379 = HEAP32[14981]|0;
  $380 = (($379) + -1)|0;
  HEAP32[14981] = $380;
  $381 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$381 & 127]($371);
  $$3 = $$3143$ph;
  STACKTOP = sp;return ($$3|0);
 }
 return (0)|0;
}
function _unixDelete($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$010 = 0, $$1 = 0, $$2 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $spec$select$i = 0, $spec$select$i11 = 0, $vararg_buffer = 0, $vararg_buffer12 = 0, $vararg_buffer5 = 0, $vararg_ptr1 = 0, $vararg_ptr10 = 0;
 var $vararg_ptr11 = 0, $vararg_ptr15 = 0, $vararg_ptr16 = 0, $vararg_ptr17 = 0, $vararg_ptr18 = 0, $vararg_ptr2 = 0, $vararg_ptr3 = 0, $vararg_ptr4 = 0, $vararg_ptr8 = 0, $vararg_ptr9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 80|0;
 $vararg_buffer12 = sp + 48|0;
 $vararg_buffer5 = sp + 24|0;
 $vararg_buffer = sp;
 $3 = sp + 68|0;
 $4 = HEAP32[(2532)>>2]|0;
 $5 = (FUNCTION_TABLE_ii[$4 & 127]($1)|0);
 $6 = ($5|0)==(-1);
 if ($6) {
  $7 = (___errno_location()|0);
  $8 = HEAP32[$7>>2]|0;
  $9 = ($8|0)==(2);
  if ($9) {
   $$010 = 5898;
   STACKTOP = sp;return ($$010|0);
  }
  $10 = ($1|0)==(0|0);
  $spec$select$i = $10 ? 59952 : $1;
  $11 = (___errno_location()|0);
  $12 = HEAP32[$11>>2]|0;
  $13 = (_strerror($12)|0);
  HEAP32[$vararg_buffer>>2] = 38695;
  $vararg_ptr1 = ((($vararg_buffer)) + 4|0);
  HEAP32[$vararg_ptr1>>2] = $12;
  $vararg_ptr2 = ((($vararg_buffer)) + 8|0);
  HEAP32[$vararg_ptr2>>2] = 31800;
  $vararg_ptr3 = ((($vararg_buffer)) + 12|0);
  HEAP32[$vararg_ptr3>>2] = $spec$select$i;
  $vararg_ptr4 = ((($vararg_buffer)) + 16|0);
  HEAP32[$vararg_ptr4>>2] = $13;
  _sqlite3_log(2570,31953,$vararg_buffer);
  $$010 = 2570;
  STACKTOP = sp;return ($$010|0);
 }
 $14 = $2 & 1;
 $15 = ($14|0)==(0);
 if ($15) {
  $$010 = 0;
  STACKTOP = sp;return ($$010|0);
 }
 $16 = HEAP32[(2544)>>2]|0;
 $17 = (FUNCTION_TABLE_iii[$16 & 255]($1,$3)|0);
 $18 = ($17|0)==(0);
 if ($18) {
  $19 = HEAP32[$3>>2]|0;
  $20 = (_fsync($19)|0);
  $21 = ($20|0)==(0);
  if ($21) {
   $$1 = 0;
  } else {
   $22 = ($1|0)==(0|0);
   $spec$select$i11 = $22 ? 59952 : $1;
   $23 = (___errno_location()|0);
   $24 = HEAP32[$23>>2]|0;
   $25 = (_strerror($24)|0);
   HEAP32[$vararg_buffer5>>2] = 38705;
   $vararg_ptr8 = ((($vararg_buffer5)) + 4|0);
   HEAP32[$vararg_ptr8>>2] = $24;
   $vararg_ptr9 = ((($vararg_buffer5)) + 8|0);
   HEAP32[$vararg_ptr9>>2] = 32138;
   $vararg_ptr10 = ((($vararg_buffer5)) + 12|0);
   HEAP32[$vararg_ptr10>>2] = $spec$select$i11;
   $vararg_ptr11 = ((($vararg_buffer5)) + 16|0);
   HEAP32[$vararg_ptr11>>2] = $25;
   _sqlite3_log(1290,31953,$vararg_buffer5);
   $$1 = 1290;
  }
  $26 = HEAP32[$3>>2]|0;
  $27 = HEAP32[(2352)>>2]|0;
  $28 = (FUNCTION_TABLE_ii[$27 & 127]($26)|0);
  $29 = ($28|0)==(0);
  if ($29) {
   $$2 = $$1;
  } else {
   $30 = (___errno_location()|0);
   $31 = HEAP32[$30>>2]|0;
   $32 = (_strerror($31)|0);
   HEAP32[$vararg_buffer12>>2] = 38707;
   $vararg_ptr15 = ((($vararg_buffer12)) + 4|0);
   HEAP32[$vararg_ptr15>>2] = $31;
   $vararg_ptr16 = ((($vararg_buffer12)) + 8|0);
   HEAP32[$vararg_ptr16>>2] = 31695;
   $vararg_ptr17 = ((($vararg_buffer12)) + 12|0);
   HEAP32[$vararg_ptr17>>2] = 59952;
   $vararg_ptr18 = ((($vararg_buffer12)) + 16|0);
   HEAP32[$vararg_ptr18>>2] = $32;
   _sqlite3_log(4106,31953,$vararg_buffer12);
   $$2 = $$1;
  }
 } else {
  $$2 = 0;
 }
 $$010 = $$2;
 STACKTOP = sp;return ($$010|0);
}
function _unixAccess($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 80|0;
 $4 = sp;
 $5 = ($2|0)==(0);
 if ($5) {
  $6 = HEAP32[(2388)>>2]|0;
  $7 = (FUNCTION_TABLE_iii[$6 & 255]($1,$4)|0);
  $8 = ($7|0)==(0);
  $9 = ((($4)) + 36|0);
  $10 = HEAP32[$9>>2]|0;
  $11 = ($10|0)>(0);
  $12 = $8 & $11;
  $13 = $12&1;
  HEAP32[$3>>2] = $13;
  STACKTOP = sp;return 0;
 } else {
  $14 = HEAP32[(2364)>>2]|0;
  $15 = (FUNCTION_TABLE_iii[$14 & 255]($1,6)|0);
  $16 = ($15|0)==(0);
  $17 = $16&1;
  HEAP32[$3>>2] = $17;
  STACKTOP = sp;return 0;
 }
 return (0)|0;
}
function _unixFullPathname($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $$0$i$i = 0, $$0$i22$i = 0, $$0$i77 = 0, $$061$lcssa = 0, $$061135 = 0, $$063 = 0, $$064 = 0, $$067 = 0, $$1$ph = 0, $$165 = 0, $$168$ph = 0, $$171 = 0, $$2 = 0, $$269 = 0, $$288$ph = 0, $$28892 = 0, $$28897 = 0, $$4 = 0, $$6 = 0, $$7$ph = 0;
 var $$7101 = 0, $$7102 = 0, $$793 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0;
 var $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0;
 var $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0;
 var $82 = 0, $83 = 0, $84 = 0, $9 = 0, $or$cond75 = 0, $spec$select$i = 0, $spec$select$i$i = 0, $spec$select$i76 = 0, $vararg_buffer = 0, $vararg_buffer10 = 0, $vararg_buffer15 = 0, $vararg_buffer20 = 0, $vararg_buffer27 = 0, $vararg_buffer3 = 0, $vararg_buffer32 = 0, $vararg_buffer37 = 0, $vararg_buffer44 = 0, $vararg_buffer49 = 0, $vararg_ptr1 = 0, $vararg_ptr13 = 0;
 var $vararg_ptr14 = 0, $vararg_ptr18 = 0, $vararg_ptr19 = 0, $vararg_ptr2 = 0, $vararg_ptr23 = 0, $vararg_ptr24 = 0, $vararg_ptr25 = 0, $vararg_ptr26 = 0, $vararg_ptr30 = 0, $vararg_ptr31 = 0, $vararg_ptr35 = 0, $vararg_ptr36 = 0, $vararg_ptr40 = 0, $vararg_ptr41 = 0, $vararg_ptr42 = 0, $vararg_ptr43 = 0, $vararg_ptr47 = 0, $vararg_ptr48 = 0, $vararg_ptr6 = 0, $vararg_ptr7 = 0;
 var $vararg_ptr8 = 0, $vararg_ptr9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 256|0;
 $vararg_buffer49 = sp + 168|0;
 $vararg_buffer44 = sp + 152|0;
 $vararg_buffer37 = sp + 128|0;
 $vararg_buffer32 = sp + 112|0;
 $vararg_buffer27 = sp + 96|0;
 $vararg_buffer20 = sp + 72|0;
 $vararg_buffer15 = sp + 56|0;
 $vararg_buffer10 = sp + 40|0;
 $vararg_buffer3 = sp + 16|0;
 $vararg_buffer = sp;
 $4 = sp + 172|0;
 $5 = ((($4)) + 12|0);
 $6 = (($2) + -1)|0;
 $7 = (($2) + -2)|0;
 $8 = ($3|0)==(0|0);
 $$063 = 0;$$064 = $1;$$067 = 1;
 while(1) {
  $9 = HEAP32[(2664)>>2]|0;
  $10 = (FUNCTION_TABLE_iii[$9 & 255]($$064,$4)|0);
  $11 = ($10|0)==(0);
  if ($11) {
   $19 = HEAP32[$5>>2]|0;
   $20 = $19 & 61440;
   $21 = ($20|0)==(40960);
   if ($21) {
    $22 = ($$063|0)==(0|0);
    if ($22) {
     $23 = (_sqlite3_malloc($2)|0);
     $24 = ($23|0)==(0|0);
     if ($24) {
      $$288$ph = 0;$$7$ph = 7;
      label = 34;
      break;
     } else {
      $$1$ph = $23;$$168$ph = $$067;
     }
    } else {
     $25 = (($$067) + 1)|0;
     $26 = ($$067|0)>(99);
     if ($26) {
      label = 9;
      break;
     } else {
      $$1$ph = $$063;$$168$ph = $25;
     }
    }
    $27 = HEAP32[(2652)>>2]|0;
    $28 = (FUNCTION_TABLE_iiii[$27 & 127]($$064,$$1$ph,$6)|0);
    $29 = ($28|0)<(0);
    if ($29) {
     label = 11;
     break;
    }
    $34 = HEAP8[$$1$ph>>0]|0;
    $35 = ($34<<24>>24)==(47);
    do {
     if ($35) {
      $$171 = $28;$$4 = 0;
     } else {
      $36 = ($$064|0)==(0|0);
      L13: do {
       if ($36) {
        $$061$lcssa = 0;
       } else {
        $37 = (_strlen($$064)|0);
        $38 = $37 & 1073741823;
        $39 = ($38|0)==(0);
        if ($39) {
         $$061$lcssa = 0;
        } else {
         $$061135 = $38;
         while(1) {
          $41 = (($$061135) + -1)|0;
          $42 = (($$064) + ($41)|0);
          $43 = HEAP8[$42>>0]|0;
          $44 = ($43<<24>>24)==(47);
          if ($44) {
           $$061$lcssa = $$061135;
           break L13;
          }
          $40 = ($$061135|0)>(1);
          if ($40) {
           $$061135 = $41;
          } else {
           $$061$lcssa = $41;
           break;
          }
         }
        }
       }
      } while(0);
      $45 = (($$061$lcssa) + ($28))|0;
      $46 = ($45|0)<($2|0);
      if ($46) {
       $47 = (($$1$ph) + ($$061$lcssa)|0);
       $48 = (($28) + 1)|0;
       _memmove(($47|0),($$1$ph|0),($48|0))|0;
       _memcpy(($$1$ph|0),($$064|0),($$061$lcssa|0))|0;
       $$171 = $45;$$4 = 0;
       break;
      } else {
       HEAP32[$vararg_buffer27>>2] = 31984;
       $vararg_ptr30 = ((($vararg_buffer27)) + 4|0);
       HEAP32[$vararg_ptr30>>2] = 38842;
       $vararg_ptr31 = ((($vararg_buffer27)) + 8|0);
       HEAP32[$vararg_ptr31>>2] = (31517);
       _sqlite3_log(14,32001,$vararg_buffer27);
       $$171 = $28;$$4 = 14;
       break;
      }
     }
    } while(0);
    $49 = (($$1$ph) + ($$171)|0);
    HEAP8[$49>>0] = 0;
    $$165 = $$1$ph;$$2 = $$1$ph;$$269 = $$168$ph;$$6 = $$4;$84 = 0;
   } else {
    $$165 = $$064;$$2 = $$063;$$269 = $$067;$$6 = 0;$84 = 1;
   }
  } else {
   $12 = (___errno_location()|0);
   $13 = HEAP32[$12>>2]|0;
   $14 = ($13|0)==(2);
   if ($14) {
    $$165 = $$064;$$2 = $$063;$$269 = $$067;$$6 = 0;$84 = 1;
   } else {
    label = 4;
    break;
   }
  }
  $50 = ($$6|0)!=(0);
  $51 = ($$165|0)==($3|0);
  $or$cond75 = $50 | $51;
  if ($or$cond75) {
   if ($84) {
    $$28897 = $$2;$$7102 = $$6;
    label = 36;
    break;
   } else {
    $$7101 = $$6;
   }
  } else {
   $52 = ($$165|0)==(0|0);
   if ($52) {
    $$0$i$i = 0;
   } else {
    $53 = (_strlen($$165)|0);
    $54 = $53 & 1073741823;
    $$0$i$i = $54;
   }
   $55 = HEAP8[$$165>>0]|0;
   $56 = ($55<<24>>24)==(47);
   if ($56) {
    $$0$i77 = 0;
   } else {
    $57 = HEAP32[(2376)>>2]|0;
    $58 = (FUNCTION_TABLE_iii[$57 & 255]($3,$7)|0);
    $59 = ($58|0)==(0|0);
    if ($59) {
     label = 27;
     break;
    }
    if ($8) {
     $$0$i22$i = 0;
    } else {
     $63 = (_strlen($3)|0);
     $64 = $63 & 1073741823;
     $$0$i22$i = $64;
    }
    $65 = (($$0$i22$i) + 1)|0;
    $66 = (($3) + ($$0$i22$i)|0);
    HEAP8[$66>>0] = 47;
    $$0$i77 = $65;
   }
   $67 = (($$0$i77) + ($$0$i$i))|0;
   $68 = ($67|0)<($2|0);
   if (!($68)) {
    label = 32;
    break;
   }
   $70 = (($2) - ($$0$i77))|0;
   $71 = (($3) + ($$0$i77)|0);
   HEAP32[$vararg_buffer49>>2] = $$165;
   (_sqlite3_snprintf($70,$71,31408,$vararg_buffer49)|0);
   if ($84) {
    $$28897 = $$2;$$7102 = 0;
    label = 36;
    break;
   } else {
    $$7101 = 0;
   }
  }
  $72 = ($$7101|0)==(0);
  if ($72) {
   $$063 = $$2;$$064 = $3;$$067 = $$269;
  } else {
   $$28892 = $$2;$$793 = $$7101;
   break;
  }
 }
 if ((label|0) == 4) {
  HEAP32[$vararg_buffer>>2] = 31984;
  $vararg_ptr1 = ((($vararg_buffer)) + 4|0);
  HEAP32[$vararg_ptr1>>2] = 38819;
  $vararg_ptr2 = ((($vararg_buffer)) + 8|0);
  HEAP32[$vararg_ptr2>>2] = (31517);
  _sqlite3_log(14,32001,$vararg_buffer);
  $15 = (___errno_location()|0);
  $16 = HEAP32[$15>>2]|0;
  $17 = (_strerror($16)|0);
  $18 = ($$064|0)==(0|0);
  $spec$select$i = $18 ? 59952 : $$064;
  HEAP32[$vararg_buffer3>>2] = 38819;
  $vararg_ptr6 = ((($vararg_buffer3)) + 4|0);
  HEAP32[$vararg_ptr6>>2] = $16;
  $vararg_ptr7 = ((($vararg_buffer3)) + 8|0);
  HEAP32[$vararg_ptr7>>2] = 31888;
  $vararg_ptr8 = ((($vararg_buffer3)) + 12|0);
  HEAP32[$vararg_ptr8>>2] = $spec$select$i;
  $vararg_ptr9 = ((($vararg_buffer3)) + 16|0);
  HEAP32[$vararg_ptr9>>2] = $17;
  _sqlite3_log(14,31953,$vararg_buffer3);
  $$28897 = $$063;$$7102 = 14;
  label = 36;
 }
 else if ((label|0) == 9) {
  HEAP32[$vararg_buffer10>>2] = 31984;
  $vararg_ptr13 = ((($vararg_buffer10)) + 4|0);
  HEAP32[$vararg_ptr13>>2] = 38830;
  $vararg_ptr14 = ((($vararg_buffer10)) + 8|0);
  HEAP32[$vararg_ptr14>>2] = (31517);
  _sqlite3_log(14,32001,$vararg_buffer10);
  $$288$ph = $$063;$$7$ph = 14;
  label = 34;
 }
 else if ((label|0) == 11) {
  HEAP32[$vararg_buffer15>>2] = 31984;
  $vararg_ptr18 = ((($vararg_buffer15)) + 4|0);
  HEAP32[$vararg_ptr18>>2] = 38836;
  $vararg_ptr19 = ((($vararg_buffer15)) + 8|0);
  HEAP32[$vararg_ptr19>>2] = (31517);
  _sqlite3_log(14,32001,$vararg_buffer15);
  $30 = (___errno_location()|0);
  $31 = HEAP32[$30>>2]|0;
  $32 = (_strerror($31)|0);
  $33 = ($$064|0)==(0|0);
  $spec$select$i76 = $33 ? 59952 : $$064;
  HEAP32[$vararg_buffer20>>2] = 38836;
  $vararg_ptr23 = ((($vararg_buffer20)) + 4|0);
  HEAP32[$vararg_ptr23>>2] = $31;
  $vararg_ptr24 = ((($vararg_buffer20)) + 8|0);
  HEAP32[$vararg_ptr24>>2] = 31879;
  $vararg_ptr25 = ((($vararg_buffer20)) + 12|0);
  HEAP32[$vararg_ptr25>>2] = $spec$select$i76;
  $vararg_ptr26 = ((($vararg_buffer20)) + 16|0);
  HEAP32[$vararg_ptr26>>2] = $32;
  _sqlite3_log(14,31953,$vararg_buffer20);
  $$288$ph = $$1$ph;$$7$ph = 14;
  label = 34;
 }
 else if ((label|0) == 27) {
  HEAP32[$vararg_buffer32>>2] = 31984;
  $vararg_ptr35 = ((($vararg_buffer32)) + 4|0);
  HEAP32[$vararg_ptr35>>2] = 38762;
  $vararg_ptr36 = ((($vararg_buffer32)) + 8|0);
  HEAP32[$vararg_ptr36>>2] = (31517);
  _sqlite3_log(14,32001,$vararg_buffer32);
  $60 = (___errno_location()|0);
  $61 = HEAP32[$60>>2]|0;
  $62 = (_strerror($61)|0);
  $spec$select$i$i = $52 ? 59952 : $$165;
  HEAP32[$vararg_buffer37>>2] = 38762;
  $vararg_ptr40 = ((($vararg_buffer37)) + 4|0);
  HEAP32[$vararg_ptr40>>2] = $61;
  $vararg_ptr41 = ((($vararg_buffer37)) + 8|0);
  HEAP32[$vararg_ptr41>>2] = 31708;
  $vararg_ptr42 = ((($vararg_buffer37)) + 12|0);
  HEAP32[$vararg_ptr42>>2] = $spec$select$i$i;
  $vararg_ptr43 = ((($vararg_buffer37)) + 16|0);
  HEAP32[$vararg_ptr43>>2] = $62;
  _sqlite3_log(14,31953,$vararg_buffer37);
  $$288$ph = $$2;$$7$ph = 14;
  label = 34;
 }
 else if ((label|0) == 32) {
  $69 = (($3) + ($$0$i77)|0);
  HEAP8[$69>>0] = 0;
  HEAP32[$vararg_buffer44>>2] = 31984;
  $vararg_ptr47 = ((($vararg_buffer44)) + 4|0);
  HEAP32[$vararg_ptr47>>2] = 38771;
  $vararg_ptr48 = ((($vararg_buffer44)) + 8|0);
  HEAP32[$vararg_ptr48>>2] = (31517);
  _sqlite3_log(14,32001,$vararg_buffer44);
  $$288$ph = $$2;$$7$ph = 14;
  label = 34;
 }
 if ((label|0) == 34) {
  $$28892 = $$288$ph;$$793 = $$7$ph;
 }
 else if ((label|0) == 36) {
  $$28892 = $$28897;$$793 = $$7102;
 }
 $73 = ($$28892|0)==(0|0);
 if ($73) {
  STACKTOP = sp;return ($$793|0);
 }
 $74 = HEAP32[7324]|0;
 $75 = ($74|0)==(0);
 if ($75) {
  $83 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$83 & 127]($$28892);
  STACKTOP = sp;return ($$793|0);
 } else {
  $76 = HEAP32[(29352)>>2]|0;
  $77 = (FUNCTION_TABLE_ii[$76 & 127]($$28892)|0);
  $78 = HEAP32[14978]|0;
  $79 = (($78) - ($77))|0;
  HEAP32[14978] = $79;
  $80 = HEAP32[14981]|0;
  $81 = (($80) + -1)|0;
  HEAP32[14981] = $81;
  $82 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$82 & 127]($$28892);
  STACKTOP = sp;return ($$793|0);
 }
 return (0)|0;
}
function _unixRandomness($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$0 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $vararg_buffer = 0, $vararg_buffer2 = 0, $vararg_ptr1 = 0, $vararg_ptr5 = 0, $vararg_ptr6 = 0, $vararg_ptr7 = 0, $vararg_ptr8 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $vararg_buffer2 = sp + 8|0;
 $vararg_buffer = sp;
 $3 = sp + 28|0;
 _memset(($2|0),0,($1|0))|0;
 $4 = (_getpid()|0);
 HEAP32[14796] = $4;
 while(1) {
  $5 = HEAP32[(2340)>>2]|0;
  $6 = (FUNCTION_TABLE_iiii[$5 & 127](32144,524288,420)|0);
  $7 = ($6|0)<(0);
  if ($7) {
   $8 = (___errno_location()|0);
   $9 = HEAP32[$8>>2]|0;
   $10 = ($9|0)==(4);
   if (!($10)) {
    label = 8;
    break;
   }
  } else {
   $11 = ($6|0)>(2);
   if ($11) {
    break;
   }
   $12 = HEAP32[(2352)>>2]|0;
   (FUNCTION_TABLE_ii[$12 & 127]($6)|0);
   HEAP32[$vararg_buffer>>2] = 32144;
   $vararg_ptr1 = ((($vararg_buffer)) + 4|0);
   HEAP32[$vararg_ptr1>>2] = $6;
   _sqlite3_log(28,31900,$vararg_buffer);
   $13 = HEAP32[(2340)>>2]|0;
   $14 = (FUNCTION_TABLE_iiii[$13 & 127](31943,0,0)|0);
   $15 = ($14|0)<(0);
   if ($15) {
    label = 8;
    break;
   }
  }
 }
 if ((label|0) == 8) {
  (_time(($3|0))|0);
  $16 = HEAP32[$3>>2]|0;
  HEAP8[$2>>0]=$16&255;HEAP8[$2+1>>0]=($16>>8)&255;HEAP8[$2+2>>0]=($16>>16)&255;HEAP8[$2+3>>0]=$16>>24;
  $17 = ((($2)) + 4|0);
  $18 = HEAP32[14796]|0;
  HEAP8[$17>>0]=$18&255;HEAP8[$17+1>>0]=($18>>8)&255;HEAP8[$17+2>>0]=($18>>16)&255;HEAP8[$17+3>>0]=$18>>24;
  $$0 = 8;
  STACKTOP = sp;return ($$0|0);
 }
 while(1) {
  $19 = HEAP32[(2436)>>2]|0;
  $20 = (FUNCTION_TABLE_iiii[$19 & 127]($6,$2,$1)|0);
  $21 = ($20|0)<(0);
  if (!($21)) {
   break;
  }
  $22 = (___errno_location()|0);
  $23 = HEAP32[$22>>2]|0;
  $24 = ($23|0)==(4);
  if (!($24)) {
   break;
  }
 }
 $25 = HEAP32[(2352)>>2]|0;
 $26 = (FUNCTION_TABLE_ii[$25 & 127]($6)|0);
 $27 = ($26|0)==(0);
 if ($27) {
  $$0 = $1;
  STACKTOP = sp;return ($$0|0);
 }
 $28 = (___errno_location()|0);
 $29 = HEAP32[$28>>2]|0;
 $30 = (_strerror($29)|0);
 HEAP32[$vararg_buffer2>>2] = 38966;
 $vararg_ptr5 = ((($vararg_buffer2)) + 4|0);
 HEAP32[$vararg_ptr5>>2] = $29;
 $vararg_ptr6 = ((($vararg_buffer2)) + 8|0);
 HEAP32[$vararg_ptr6>>2] = 31695;
 $vararg_ptr7 = ((($vararg_buffer2)) + 12|0);
 HEAP32[$vararg_ptr7>>2] = 59952;
 $vararg_ptr8 = ((($vararg_buffer2)) + 16|0);
 HEAP32[$vararg_ptr8>>2] = $30;
 _sqlite3_log(4106,31953,$vararg_buffer2);
 $$0 = $1;
 STACKTOP = sp;return ($$0|0);
}
function _unixSleep($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $2 = 0, $3 = 0, $4 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = (($1) + 999999)|0;
 $3 = (($2|0) / 1000000)&-1;
 (_sleep($3)|0);
 $4 = ($3*1000000)|0;
 return ($4|0);
}
function _unixCurrentTime($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0.0, $18 = 0.0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $2 = sp;
 (_gettimeofday(($2|0),(0|0))|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ($3|0)<(0);
 $5 = $4 << 31 >> 31;
 $6 = (___muldi3(($3|0),($5|0),1000,0)|0);
 $7 = (getTempRet0() | 0);
 $8 = (_i64Add(($6|0),($7|0),1045635584,49096)|0);
 $9 = (getTempRet0() | 0);
 $10 = ((($2)) + 4|0);
 $11 = HEAP32[$10>>2]|0;
 $12 = (($11|0) / 1000)&-1;
 $13 = ($12|0)<(0);
 $14 = $13 << 31 >> 31;
 $15 = (_i64Add(($8|0),($9|0),($12|0),($14|0))|0);
 $16 = (getTempRet0() | 0);
 $17 = (+($15>>>0)) + (4294967296.0*(+($16|0)));
 $18 = $17 / 8.64E+7;
 HEAPF64[$1>>3] = $18;
 STACKTOP = sp;return 0;
}
function _unixGetLastError($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $3 = 0, $4 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = (___errno_location()|0);
 $4 = HEAP32[$3>>2]|0;
 return ($4|0);
}
function _unixCurrentTimeInt64($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0;
 var sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $2 = sp;
 (_gettimeofday(($2|0),(0|0))|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ($3|0)<(0);
 $5 = $4 << 31 >> 31;
 $6 = (___muldi3(($3|0),($5|0),1000,0)|0);
 $7 = (getTempRet0() | 0);
 $8 = (_i64Add(($6|0),($7|0),1045635584,49096)|0);
 $9 = (getTempRet0() | 0);
 $10 = ((($2)) + 4|0);
 $11 = HEAP32[$10>>2]|0;
 $12 = (($11|0) / 1000)&-1;
 $13 = ($12|0)<(0);
 $14 = $13 << 31 >> 31;
 $15 = (_i64Add(($8|0),($9|0),($12|0),($14|0))|0);
 $16 = (getTempRet0() | 0);
 $17 = $1;
 $18 = $17;
 HEAP32[$18>>2] = $15;
 $19 = (($17) + 4)|0;
 $20 = $19;
 HEAP32[$20>>2] = $16;
 STACKTOP = sp;return 0;
}
function _unixSetSystemCall($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$0 = 0, $$125$lcssa = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0;
 var $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0;
 var $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0;
 var $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0;
 var $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0;
 var $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0;
 var $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $spec$select = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = ($1|0)==(0|0);
 if (!($3)) {
  $8 = HEAP32[584]|0;
  $9 = (_strcmp($1,$8)|0);
  $10 = ($9|0)==(0);
  do {
   if ($10) {
    $$125$lcssa = 0;
   } else {
    $20 = HEAP32[(2348)>>2]|0;
    $21 = (_strcmp($1,$20)|0);
    $22 = ($21|0)==(0);
    if ($22) {
     $$125$lcssa = 1;
    } else {
     $77 = HEAP32[(2360)>>2]|0;
     $78 = (_strcmp($1,$77)|0);
     $79 = ($78|0)==(0);
     if ($79) {
      $$125$lcssa = 2;
     } else {
      $80 = HEAP32[(2372)>>2]|0;
      $81 = (_strcmp($1,$80)|0);
      $82 = ($81|0)==(0);
      if ($82) {
       $$125$lcssa = 3;
      } else {
       $83 = HEAP32[(2384)>>2]|0;
       $84 = (_strcmp($1,$83)|0);
       $85 = ($84|0)==(0);
       if ($85) {
        $$125$lcssa = 4;
       } else {
        $86 = HEAP32[(2396)>>2]|0;
        $87 = (_strcmp($1,$86)|0);
        $88 = ($87|0)==(0);
        if ($88) {
         $$125$lcssa = 5;
        } else {
         $89 = HEAP32[(2408)>>2]|0;
         $90 = (_strcmp($1,$89)|0);
         $91 = ($90|0)==(0);
         if ($91) {
          $$125$lcssa = 6;
         } else {
          $92 = HEAP32[(2420)>>2]|0;
          $93 = (_strcmp($1,$92)|0);
          $94 = ($93|0)==(0);
          if ($94) {
           $$125$lcssa = 7;
          } else {
           $95 = HEAP32[(2432)>>2]|0;
           $96 = (_strcmp($1,$95)|0);
           $97 = ($96|0)==(0);
           if ($97) {
            $$125$lcssa = 8;
           } else {
            $98 = HEAP32[(2444)>>2]|0;
            $99 = (_strcmp($1,$98)|0);
            $100 = ($99|0)==(0);
            if ($100) {
             $$125$lcssa = 9;
            } else {
             $101 = HEAP32[(2456)>>2]|0;
             $102 = (_strcmp($1,$101)|0);
             $103 = ($102|0)==(0);
             if ($103) {
              $$125$lcssa = 10;
             } else {
              $104 = HEAP32[(2468)>>2]|0;
              $105 = (_strcmp($1,$104)|0);
              $106 = ($105|0)==(0);
              if ($106) {
               $$125$lcssa = 11;
              } else {
               $107 = HEAP32[(2480)>>2]|0;
               $108 = (_strcmp($1,$107)|0);
               $109 = ($108|0)==(0);
               if ($109) {
                $$125$lcssa = 12;
               } else {
                $110 = HEAP32[(2492)>>2]|0;
                $111 = (_strcmp($1,$110)|0);
                $112 = ($111|0)==(0);
                if ($112) {
                 $$125$lcssa = 13;
                } else {
                 $113 = HEAP32[(2504)>>2]|0;
                 $114 = (_strcmp($1,$113)|0);
                 $115 = ($114|0)==(0);
                 if ($115) {
                  $$125$lcssa = 14;
                 } else {
                  $116 = HEAP32[(2516)>>2]|0;
                  $117 = (_strcmp($1,$116)|0);
                  $118 = ($117|0)==(0);
                  if ($118) {
                   $$125$lcssa = 15;
                  } else {
                   $119 = HEAP32[(2528)>>2]|0;
                   $120 = (_strcmp($1,$119)|0);
                   $121 = ($120|0)==(0);
                   if ($121) {
                    $$125$lcssa = 16;
                   } else {
                    $122 = HEAP32[(2540)>>2]|0;
                    $123 = (_strcmp($1,$122)|0);
                    $124 = ($123|0)==(0);
                    if ($124) {
                     $$125$lcssa = 17;
                     break;
                    }
                    $125 = HEAP32[(2552)>>2]|0;
                    $126 = (_strcmp($1,$125)|0);
                    $127 = ($126|0)==(0);
                    if ($127) {
                     $$125$lcssa = 18;
                     break;
                    }
                    $128 = HEAP32[(2564)>>2]|0;
                    $129 = (_strcmp($1,$128)|0);
                    $130 = ($129|0)==(0);
                    if ($130) {
                     $$125$lcssa = 19;
                     break;
                    }
                    $131 = HEAP32[(2576)>>2]|0;
                    $132 = (_strcmp($1,$131)|0);
                    $133 = ($132|0)==(0);
                    if ($133) {
                     $$125$lcssa = 20;
                     break;
                    }
                    $134 = HEAP32[(2588)>>2]|0;
                    $135 = (_strcmp($1,$134)|0);
                    $136 = ($135|0)==(0);
                    if ($136) {
                     $$125$lcssa = 21;
                     break;
                    }
                    $137 = HEAP32[(2600)>>2]|0;
                    $138 = (_strcmp($1,$137)|0);
                    $139 = ($138|0)==(0);
                    if ($139) {
                     $$125$lcssa = 22;
                     break;
                    }
                    $140 = HEAP32[(2612)>>2]|0;
                    $141 = (_strcmp($1,$140)|0);
                    $142 = ($141|0)==(0);
                    if ($142) {
                     $$125$lcssa = 23;
                     break;
                    }
                    $143 = HEAP32[(2624)>>2]|0;
                    $144 = (_strcmp($1,$143)|0);
                    $145 = ($144|0)==(0);
                    if ($145) {
                     $$125$lcssa = 24;
                     break;
                    }
                    $146 = HEAP32[(2636)>>2]|0;
                    $147 = (_strcmp($1,$146)|0);
                    $148 = ($147|0)==(0);
                    if ($148) {
                     $$125$lcssa = 25;
                     break;
                    }
                    $149 = HEAP32[(2648)>>2]|0;
                    $150 = (_strcmp($1,$149)|0);
                    $151 = ($150|0)==(0);
                    if ($151) {
                     $$125$lcssa = 26;
                     break;
                    }
                    $152 = HEAP32[(2660)>>2]|0;
                    $153 = (_strcmp($1,$152)|0);
                    $154 = ($153|0)==(0);
                    if ($154) {
                     $$125$lcssa = 27;
                     break;
                    }
                    $155 = HEAP32[(2672)>>2]|0;
                    $156 = (_strcmp($1,$155)|0);
                    $157 = ($156|0)==(0);
                    if ($157) {
                     $$125$lcssa = 28;
                     break;
                    } else {
                     $$0 = 12;
                    }
                    return ($$0|0);
                   }
                  }
                 }
                }
               }
              }
             }
            }
           }
          }
         }
        }
       }
      }
     }
    }
   }
  } while(0);
  $11 = (((2336 + (($$125$lcssa*12)|0)|0)) + 8|0);
  $12 = HEAP32[$11>>2]|0;
  $13 = ($12|0)==(0|0);
  if ($13) {
   $14 = (((2336 + (($$125$lcssa*12)|0)|0)) + 4|0);
   $15 = HEAP32[$14>>2]|0;
   HEAP32[$11>>2] = $15;
   $16 = $15;
   $18 = $16;
  } else {
   $18 = $12;
  }
  $17 = ($2|0)==(0|0);
  $spec$select = $17 ? $18 : $2;
  $19 = (((2336 + (($$125$lcssa*12)|0)|0)) + 4|0);
  HEAP32[$19>>2] = $spec$select;
  $$0 = 0;
  return ($$0|0);
 }
 $4 = HEAP32[(2344)>>2]|0;
 $5 = ($4|0)==(0|0);
 if (!($5)) {
  HEAP32[(2340)>>2] = $4;
 }
 $6 = HEAP32[(2356)>>2]|0;
 $7 = ($6|0)==(0|0);
 if (!($7)) {
  HEAP32[(2352)>>2] = $6;
 }
 $23 = HEAP32[(2368)>>2]|0;
 $24 = ($23|0)==(0|0);
 if (!($24)) {
  HEAP32[(2364)>>2] = $23;
 }
 $25 = HEAP32[(2380)>>2]|0;
 $26 = ($25|0)==(0|0);
 if (!($26)) {
  HEAP32[(2376)>>2] = $25;
 }
 $27 = HEAP32[(2392)>>2]|0;
 $28 = ($27|0)==(0|0);
 if (!($28)) {
  HEAP32[(2388)>>2] = $27;
 }
 $29 = HEAP32[(2404)>>2]|0;
 $30 = ($29|0)==(0|0);
 if (!($30)) {
  HEAP32[(2400)>>2] = $29;
 }
 $31 = HEAP32[(2416)>>2]|0;
 $32 = ($31|0)==(0|0);
 if (!($32)) {
  HEAP32[(2412)>>2] = $31;
 }
 $33 = HEAP32[(2428)>>2]|0;
 $34 = ($33|0)==(0|0);
 if (!($34)) {
  HEAP32[(2424)>>2] = $33;
 }
 $35 = HEAP32[(2440)>>2]|0;
 $36 = ($35|0)==(0|0);
 if (!($36)) {
  HEAP32[(2436)>>2] = $35;
 }
 $37 = HEAP32[(2452)>>2]|0;
 $38 = ($37|0)==(0|0);
 if (!($38)) {
  HEAP32[(2448)>>2] = $37;
 }
 $39 = HEAP32[(2464)>>2]|0;
 $40 = ($39|0)==(0|0);
 if (!($40)) {
  HEAP32[(2460)>>2] = $39;
 }
 $41 = HEAP32[(2476)>>2]|0;
 $42 = ($41|0)==(0|0);
 if (!($42)) {
  HEAP32[(2472)>>2] = $41;
 }
 $43 = HEAP32[(2488)>>2]|0;
 $44 = ($43|0)==(0|0);
 if (!($44)) {
  HEAP32[(2484)>>2] = $43;
 }
 $45 = HEAP32[(2500)>>2]|0;
 $46 = ($45|0)==(0|0);
 if (!($46)) {
  HEAP32[(2496)>>2] = $45;
 }
 $47 = HEAP32[(2512)>>2]|0;
 $48 = ($47|0)==(0|0);
 if (!($48)) {
  HEAP32[(2508)>>2] = $47;
 }
 $49 = HEAP32[(2524)>>2]|0;
 $50 = ($49|0)==(0|0);
 if (!($50)) {
  HEAP32[(2520)>>2] = $49;
 }
 $51 = HEAP32[(2536)>>2]|0;
 $52 = ($51|0)==(0|0);
 if (!($52)) {
  HEAP32[(2532)>>2] = $51;
 }
 $53 = HEAP32[(2548)>>2]|0;
 $54 = ($53|0)==(0|0);
 if (!($54)) {
  HEAP32[(2544)>>2] = $53;
 }
 $55 = HEAP32[(2560)>>2]|0;
 $56 = ($55|0)==(0|0);
 if (!($56)) {
  HEAP32[(2556)>>2] = $55;
 }
 $57 = HEAP32[(2572)>>2]|0;
 $58 = ($57|0)==(0|0);
 if (!($58)) {
  HEAP32[(2568)>>2] = $57;
 }
 $59 = HEAP32[(2584)>>2]|0;
 $60 = ($59|0)==(0|0);
 if (!($60)) {
  HEAP32[(2580)>>2] = $59;
 }
 $61 = HEAP32[(2596)>>2]|0;
 $62 = ($61|0)==(0|0);
 if (!($62)) {
  HEAP32[(2592)>>2] = $61;
 }
 $63 = HEAP32[(2608)>>2]|0;
 $64 = ($63|0)==(0|0);
 if (!($64)) {
  HEAP32[(2604)>>2] = $63;
 }
 $65 = HEAP32[(2620)>>2]|0;
 $66 = ($65|0)==(0|0);
 if (!($66)) {
  HEAP32[(2616)>>2] = $65;
 }
 $67 = HEAP32[(2632)>>2]|0;
 $68 = ($67|0)==(0|0);
 if (!($68)) {
  HEAP32[(2628)>>2] = $67;
 }
 $69 = HEAP32[(2644)>>2]|0;
 $70 = ($69|0)==(0|0);
 if (!($70)) {
  HEAP32[(2640)>>2] = $69;
 }
 $71 = HEAP32[(2656)>>2]|0;
 $72 = ($71|0)==(0|0);
 if (!($72)) {
  HEAP32[(2652)>>2] = $71;
 }
 $73 = HEAP32[(2668)>>2]|0;
 $74 = ($73|0)==(0|0);
 if (!($74)) {
  HEAP32[(2664)>>2] = $73;
 }
 $75 = HEAP32[(2680)>>2]|0;
 $76 = ($75|0)==(0|0);
 if ($76) {
  $$0 = 0;
  return ($$0|0);
 }
 HEAP32[(2676)>>2] = $75;
 $$0 = 0;
 return ($$0|0);
}
function _unixGetSystemCall($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$06 = 0, $$08$lcssa = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0;
 var $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0;
 var $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0;
 var $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = HEAP32[584]|0;
 $3 = (_strcmp($1,$2)|0);
 $4 = ($3|0)==(0);
 do {
  if ($4) {
   $$08$lcssa = 0;
  } else {
   $7 = HEAP32[(2348)>>2]|0;
   $8 = (_strcmp($1,$7)|0);
   $9 = ($8|0)==(0);
   if ($9) {
    $$08$lcssa = 1;
   } else {
    $10 = HEAP32[(2360)>>2]|0;
    $11 = (_strcmp($1,$10)|0);
    $12 = ($11|0)==(0);
    if ($12) {
     $$08$lcssa = 2;
    } else {
     $13 = HEAP32[(2372)>>2]|0;
     $14 = (_strcmp($1,$13)|0);
     $15 = ($14|0)==(0);
     if ($15) {
      $$08$lcssa = 3;
     } else {
      $16 = HEAP32[(2384)>>2]|0;
      $17 = (_strcmp($1,$16)|0);
      $18 = ($17|0)==(0);
      if ($18) {
       $$08$lcssa = 4;
      } else {
       $19 = HEAP32[(2396)>>2]|0;
       $20 = (_strcmp($1,$19)|0);
       $21 = ($20|0)==(0);
       if ($21) {
        $$08$lcssa = 5;
       } else {
        $22 = HEAP32[(2408)>>2]|0;
        $23 = (_strcmp($1,$22)|0);
        $24 = ($23|0)==(0);
        if ($24) {
         $$08$lcssa = 6;
        } else {
         $25 = HEAP32[(2420)>>2]|0;
         $26 = (_strcmp($1,$25)|0);
         $27 = ($26|0)==(0);
         if ($27) {
          $$08$lcssa = 7;
         } else {
          $28 = HEAP32[(2432)>>2]|0;
          $29 = (_strcmp($1,$28)|0);
          $30 = ($29|0)==(0);
          if ($30) {
           $$08$lcssa = 8;
          } else {
           $31 = HEAP32[(2444)>>2]|0;
           $32 = (_strcmp($1,$31)|0);
           $33 = ($32|0)==(0);
           if ($33) {
            $$08$lcssa = 9;
           } else {
            $34 = HEAP32[(2456)>>2]|0;
            $35 = (_strcmp($1,$34)|0);
            $36 = ($35|0)==(0);
            if ($36) {
             $$08$lcssa = 10;
            } else {
             $37 = HEAP32[(2468)>>2]|0;
             $38 = (_strcmp($1,$37)|0);
             $39 = ($38|0)==(0);
             if ($39) {
              $$08$lcssa = 11;
             } else {
              $40 = HEAP32[(2480)>>2]|0;
              $41 = (_strcmp($1,$40)|0);
              $42 = ($41|0)==(0);
              if ($42) {
               $$08$lcssa = 12;
              } else {
               $43 = HEAP32[(2492)>>2]|0;
               $44 = (_strcmp($1,$43)|0);
               $45 = ($44|0)==(0);
               if ($45) {
                $$08$lcssa = 13;
               } else {
                $46 = HEAP32[(2504)>>2]|0;
                $47 = (_strcmp($1,$46)|0);
                $48 = ($47|0)==(0);
                if ($48) {
                 $$08$lcssa = 14;
                } else {
                 $49 = HEAP32[(2516)>>2]|0;
                 $50 = (_strcmp($1,$49)|0);
                 $51 = ($50|0)==(0);
                 if ($51) {
                  $$08$lcssa = 15;
                 } else {
                  $52 = HEAP32[(2528)>>2]|0;
                  $53 = (_strcmp($1,$52)|0);
                  $54 = ($53|0)==(0);
                  if ($54) {
                   $$08$lcssa = 16;
                  } else {
                   $55 = HEAP32[(2540)>>2]|0;
                   $56 = (_strcmp($1,$55)|0);
                   $57 = ($56|0)==(0);
                   if ($57) {
                    $$08$lcssa = 17;
                   } else {
                    $58 = HEAP32[(2552)>>2]|0;
                    $59 = (_strcmp($1,$58)|0);
                    $60 = ($59|0)==(0);
                    if ($60) {
                     $$08$lcssa = 18;
                    } else {
                     $61 = HEAP32[(2564)>>2]|0;
                     $62 = (_strcmp($1,$61)|0);
                     $63 = ($62|0)==(0);
                     if ($63) {
                      $$08$lcssa = 19;
                      break;
                     }
                     $64 = HEAP32[(2576)>>2]|0;
                     $65 = (_strcmp($1,$64)|0);
                     $66 = ($65|0)==(0);
                     if ($66) {
                      $$08$lcssa = 20;
                      break;
                     }
                     $67 = HEAP32[(2588)>>2]|0;
                     $68 = (_strcmp($1,$67)|0);
                     $69 = ($68|0)==(0);
                     if ($69) {
                      $$08$lcssa = 21;
                      break;
                     }
                     $70 = HEAP32[(2600)>>2]|0;
                     $71 = (_strcmp($1,$70)|0);
                     $72 = ($71|0)==(0);
                     if ($72) {
                      $$08$lcssa = 22;
                      break;
                     }
                     $73 = HEAP32[(2612)>>2]|0;
                     $74 = (_strcmp($1,$73)|0);
                     $75 = ($74|0)==(0);
                     if ($75) {
                      $$08$lcssa = 23;
                      break;
                     }
                     $76 = HEAP32[(2624)>>2]|0;
                     $77 = (_strcmp($1,$76)|0);
                     $78 = ($77|0)==(0);
                     if ($78) {
                      $$08$lcssa = 24;
                      break;
                     }
                     $79 = HEAP32[(2636)>>2]|0;
                     $80 = (_strcmp($1,$79)|0);
                     $81 = ($80|0)==(0);
                     if ($81) {
                      $$08$lcssa = 25;
                      break;
                     }
                     $82 = HEAP32[(2648)>>2]|0;
                     $83 = (_strcmp($1,$82)|0);
                     $84 = ($83|0)==(0);
                     if ($84) {
                      $$08$lcssa = 26;
                      break;
                     }
                     $85 = HEAP32[(2660)>>2]|0;
                     $86 = (_strcmp($1,$85)|0);
                     $87 = ($86|0)==(0);
                     if ($87) {
                      $$08$lcssa = 27;
                      break;
                     }
                     $88 = HEAP32[(2672)>>2]|0;
                     $89 = (_strcmp($1,$88)|0);
                     $90 = ($89|0)==(0);
                     if ($90) {
                      $$08$lcssa = 28;
                      break;
                     } else {
                      $$06 = 0;
                     }
                     return ($$06|0);
                    }
                   }
                  }
                 }
                }
               }
              }
             }
            }
           }
          }
         }
        }
       }
      }
     }
    }
   }
  }
 } while(0);
 $5 = (((2336 + (($$08$lcssa*12)|0)|0)) + 4|0);
 $6 = HEAP32[$5>>2]|0;
 $$06 = $6;
 return ($$06|0);
}
function _unixNextSystemCall($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$011 = 0, $$215 = 0, $$215$in = 0, $$215$in$ph = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0;
 var $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0;
 var $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0;
 var $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0;
 var $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($1|0)==(0|0);
 do {
  if ($2) {
   $$215$in$ph = -1;
  } else {
   $3 = HEAP32[584]|0;
   $4 = (_strcmp($1,$3)|0);
   $5 = ($4|0)==(0);
   if ($5) {
    $$215$in$ph = 0;
   } else {
    $6 = HEAP32[(2348)>>2]|0;
    $7 = (_strcmp($1,$6)|0);
    $8 = ($7|0)==(0);
    if ($8) {
     $$215$in$ph = 1;
    } else {
     $15 = HEAP32[(2360)>>2]|0;
     $16 = (_strcmp($1,$15)|0);
     $17 = ($16|0)==(0);
     if ($17) {
      $$215$in$ph = 2;
     } else {
      $18 = HEAP32[(2372)>>2]|0;
      $19 = (_strcmp($1,$18)|0);
      $20 = ($19|0)==(0);
      if ($20) {
       $$215$in$ph = 3;
      } else {
       $21 = HEAP32[(2384)>>2]|0;
       $22 = (_strcmp($1,$21)|0);
       $23 = ($22|0)==(0);
       if ($23) {
        $$215$in$ph = 4;
       } else {
        $24 = HEAP32[(2396)>>2]|0;
        $25 = (_strcmp($1,$24)|0);
        $26 = ($25|0)==(0);
        if ($26) {
         $$215$in$ph = 5;
        } else {
         $27 = HEAP32[(2408)>>2]|0;
         $28 = (_strcmp($1,$27)|0);
         $29 = ($28|0)==(0);
         if ($29) {
          $$215$in$ph = 6;
         } else {
          $30 = HEAP32[(2420)>>2]|0;
          $31 = (_strcmp($1,$30)|0);
          $32 = ($31|0)==(0);
          if ($32) {
           $$215$in$ph = 7;
          } else {
           $33 = HEAP32[(2432)>>2]|0;
           $34 = (_strcmp($1,$33)|0);
           $35 = ($34|0)==(0);
           if ($35) {
            $$215$in$ph = 8;
           } else {
            $36 = HEAP32[(2444)>>2]|0;
            $37 = (_strcmp($1,$36)|0);
            $38 = ($37|0)==(0);
            if ($38) {
             $$215$in$ph = 9;
            } else {
             $39 = HEAP32[(2456)>>2]|0;
             $40 = (_strcmp($1,$39)|0);
             $41 = ($40|0)==(0);
             if ($41) {
              $$215$in$ph = 10;
             } else {
              $42 = HEAP32[(2468)>>2]|0;
              $43 = (_strcmp($1,$42)|0);
              $44 = ($43|0)==(0);
              if ($44) {
               $$215$in$ph = 11;
              } else {
               $45 = HEAP32[(2480)>>2]|0;
               $46 = (_strcmp($1,$45)|0);
               $47 = ($46|0)==(0);
               if ($47) {
                $$215$in$ph = 12;
               } else {
                $48 = HEAP32[(2492)>>2]|0;
                $49 = (_strcmp($1,$48)|0);
                $50 = ($49|0)==(0);
                if ($50) {
                 $$215$in$ph = 13;
                } else {
                 $51 = HEAP32[(2504)>>2]|0;
                 $52 = (_strcmp($1,$51)|0);
                 $53 = ($52|0)==(0);
                 if ($53) {
                  $$215$in$ph = 14;
                 } else {
                  $54 = HEAP32[(2516)>>2]|0;
                  $55 = (_strcmp($1,$54)|0);
                  $56 = ($55|0)==(0);
                  if ($56) {
                   $$215$in$ph = 15;
                  } else {
                   $57 = HEAP32[(2528)>>2]|0;
                   $58 = (_strcmp($1,$57)|0);
                   $59 = ($58|0)==(0);
                   if ($59) {
                    $$215$in$ph = 16;
                   } else {
                    $60 = HEAP32[(2540)>>2]|0;
                    $61 = (_strcmp($1,$60)|0);
                    $62 = ($61|0)==(0);
                    if ($62) {
                     $$215$in$ph = 17;
                    } else {
                     $63 = HEAP32[(2552)>>2]|0;
                     $64 = (_strcmp($1,$63)|0);
                     $65 = ($64|0)==(0);
                     if ($65) {
                      $$215$in$ph = 18;
                      break;
                     }
                     $66 = HEAP32[(2564)>>2]|0;
                     $67 = (_strcmp($1,$66)|0);
                     $68 = ($67|0)==(0);
                     if ($68) {
                      $$215$in$ph = 19;
                      break;
                     }
                     $69 = HEAP32[(2576)>>2]|0;
                     $70 = (_strcmp($1,$69)|0);
                     $71 = ($70|0)==(0);
                     if ($71) {
                      $$215$in$ph = 20;
                      break;
                     }
                     $72 = HEAP32[(2588)>>2]|0;
                     $73 = (_strcmp($1,$72)|0);
                     $74 = ($73|0)==(0);
                     if ($74) {
                      $$215$in$ph = 21;
                      break;
                     }
                     $75 = HEAP32[(2600)>>2]|0;
                     $76 = (_strcmp($1,$75)|0);
                     $77 = ($76|0)==(0);
                     if ($77) {
                      $$215$in$ph = 22;
                      break;
                     }
                     $78 = HEAP32[(2612)>>2]|0;
                     $79 = (_strcmp($1,$78)|0);
                     $80 = ($79|0)==(0);
                     if ($80) {
                      $$215$in$ph = 23;
                      break;
                     }
                     $81 = HEAP32[(2624)>>2]|0;
                     $82 = (_strcmp($1,$81)|0);
                     $83 = ($82|0)==(0);
                     if ($83) {
                      $$215$in$ph = 24;
                      break;
                     }
                     $84 = HEAP32[(2636)>>2]|0;
                     $85 = (_strcmp($1,$84)|0);
                     $86 = ($85|0)==(0);
                     if ($86) {
                      $$215$in$ph = 25;
                      break;
                     }
                     $87 = HEAP32[(2648)>>2]|0;
                     $88 = (_strcmp($1,$87)|0);
                     $89 = ($88|0)==(0);
                     if ($89) {
                      $$215$in$ph = 26;
                      break;
                     }
                     $90 = HEAP32[(2660)>>2]|0;
                     $91 = (_strcmp($1,$90)|0);
                     $92 = ($91|0)==(0);
                     if ($92) {
                      $$215$in$ph = 27;
                      break;
                     } else {
                      $$011 = 0;
                     }
                     return ($$011|0);
                    }
                   }
                  }
                 }
                }
               }
              }
             }
            }
           }
          }
         }
        }
       }
      }
     }
    }
   }
  }
 } while(0);
 $$215$in = $$215$in$ph;
 while(1) {
  $$215 = (($$215$in) + 1)|0;
  $10 = (((2336 + (($$215*12)|0)|0)) + 4|0);
  $11 = HEAP32[$10>>2]|0;
  $12 = ($11|0)==(0|0);
  if (!($12)) {
   break;
  }
  $9 = ($$215$in|0)<(27);
  if ($9) {
   $$215$in = $$215;
  } else {
   $$011 = 0;
   label = 8;
   break;
  }
 }
 if ((label|0) == 8) {
  return ($$011|0);
 }
 $13 = (2336 + (($$215*12)|0)|0);
 $14 = HEAP32[$13>>2]|0;
 $$011 = $14;
 return ($$011|0);
}
function _sqlite3ErrorWithMsg($0,$1,$2,$varargs) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $varargs = $varargs|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp;
 $4 = ((($0)) + 64|0);
 HEAP32[$4>>2] = $1;
 _sqlite3SystemError($0,$1);
 $5 = ($2|0)==(0|0);
 if ($5) {
  _sqlite3Error($0,$1);
  STACKTOP = sp;return;
 }
 $6 = ((($0)) + 260|0);
 $7 = HEAP32[$6>>2]|0;
 $8 = ($7|0)==(0|0);
 if ($8) {
  $9 = (_sqlite3ValueNew($0)|0);
  HEAP32[$6>>2] = $9;
  $10 = ($9|0)==(0|0);
  if ($10) {
   STACKTOP = sp;return;
  }
 }
 HEAP32[$3>>2] = $varargs;
 $11 = (_sqlite3VMPrintf($0,$2,$3)|0);
 $12 = HEAP32[$6>>2]|0;
 $13 = ($12|0)==(0|0);
 if (!($13)) {
  (_sqlite3VdbeMemSetStr($12,$11,-1,1,116)|0);
 }
 STACKTOP = sp;return;
}
function _sqlite3MallocZero($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$ = 0, $$0$i5 = 0, $$0$i8 = 0, $$pre$phiZ2D = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0;
 var $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0;
 var $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = (_i64Add(($0|0),($1|0),-1,-1)|0);
 $3 = (getTempRet0() | 0);
 $4 = ($3>>>0)>(0);
 $5 = ($2>>>0)>(2147483390);
 $6 = ($3|0)==(0);
 $7 = $6 & $5;
 $8 = $4 | $7;
 if ($8) {
  $$0$i5 = 0;
  return ($$0$i5|0);
 }
 $9 = HEAP32[7324]|0;
 $10 = ($9|0)==(0);
 if ($10) {
  $49 = HEAP32[(29340)>>2]|0;
  $50 = (FUNCTION_TABLE_ii[$49 & 127]($0)|0);
  $51 = ($50|0)==(0|0);
  if ($51) {
   $$0$i5 = 0;
   return ($$0$i5|0);
  } else {
   $$0$i8 = $50;$$pre$phiZ2D = $0;
  }
 } else {
  $11 = HEAP32[(29356)>>2]|0;
  $12 = (FUNCTION_TABLE_ii[$11 & 127]($0)|0);
  $13 = HEAP32[14985]|0;
  $14 = ($13>>>0)<($0>>>0);
  if ($14) {
   HEAP32[14985] = $0;
  }
  $15 = (59064);
  $16 = $15;
  $17 = HEAP32[$16>>2]|0;
  $18 = (($15) + 4)|0;
  $19 = $18;
  $20 = HEAP32[$19>>2]|0;
  $21 = ($20|0)>(0);
  $22 = ($17>>>0)>(0);
  $23 = ($20|0)==(0);
  $24 = $23 & $22;
  $25 = $21 | $24;
  if ($25) {
   $26 = HEAP32[14978]|0;
   $27 = ($12|0)<(0);
   $28 = $27 << 31 >> 31;
   $29 = (_i64Subtract(($17|0),($20|0),($12|0),($28|0))|0);
   $30 = (getTempRet0() | 0);
   $31 = ($30|0)<(0);
   $32 = ($29>>>0)<=($26>>>0);
   $33 = ($30|0)==(0);
   $34 = $33 & $32;
   $35 = $31 | $34;
   $$ = $35&1;
   HEAP32[(59072)>>2] = $$;
  }
  $36 = HEAP32[(29340)>>2]|0;
  $37 = (FUNCTION_TABLE_ii[$36 & 127]($12)|0);
  $38 = ($37|0)==(0|0);
  if ($38) {
   $$0$i5 = 0;
   return ($$0$i5|0);
  }
  $39 = HEAP32[(29352)>>2]|0;
  $40 = (FUNCTION_TABLE_ii[$39 & 127]($37)|0);
  $41 = HEAP32[14978]|0;
  $42 = (($41) + ($40))|0;
  HEAP32[14978] = $42;
  $43 = HEAP32[14982]|0;
  $44 = ($42>>>0)>($43>>>0);
  if ($44) {
   HEAP32[14982] = $42;
  }
  $45 = HEAP32[14981]|0;
  $46 = (($45) + 1)|0;
  HEAP32[14981] = $46;
  $47 = HEAP32[14987]|0;
  $48 = ($46>>>0)>($47>>>0);
  if ($48) {
   HEAP32[14987] = $46;
   $$0$i8 = $37;$$pre$phiZ2D = $0;
  } else {
   $$0$i8 = $37;$$pre$phiZ2D = $0;
  }
 }
 _memset(($$0$i8|0),0,($$pre$phiZ2D|0))|0;
 $$0$i5 = $$0$i8;
 return ($$0$i5|0);
}
function _sqlite3Error($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($0)) + 64|0);
 HEAP32[$2>>2] = $1;
 $3 = ($1|0)==(0);
 if (!($3)) {
  _sqlite3ErrorFinish($0,$1);
  return;
 }
 $4 = ((($0)) + 260|0);
 $5 = HEAP32[$4>>2]|0;
 $6 = ($5|0)==(0|0);
 if ($6) {
  return;
 }
 _sqlite3ErrorFinish($0,0);
 return;
}
function _sqlite3_backup_step($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i204 = 0, $$0$i205 = 0, $$0$i206 = 0, $$0$i208 = 0, $$0178240 = 0, $$0179245 = 0, $$0180218 = 0, $$0181 = 0, $$1 = 0, $$1$i$ph = 0, $$12241 = 0, $$13 = 0, $$14 = 0, $$15 = 0, $$16$lcssa = 0, $$17 = 0, $$20 = 0, $$22 = 0, $$24 = 0, $$3 = 0;
 var $$4 = 0, $$5$lcssa = 0, $$5246 = 0, $$6 = 0, $$7 = 0, $$idx$i = 0, $$idx$val$i = 0, $$idx$val$idx$val = 0, $$idx$val$idx$val$i = 0, $$idx191 = 0, $$idx191$val = 0, $$idx191$val$idx$val = 0, $$idx192$val$idx$val = 0, $$idx193 = 0, $$idx193$val = 0, $$idx193$val$idx = 0, $$idx193$val$idx$val = 0, $$idx194 = 0, $$idx194$val = 0, $$idx194$val$idx = 0;
 var $$idx194$val$idx$val = 0, $$idx195 = 0, $$idx195$val = 0, $$idx196 = 0, $$idx196$val = 0, $$idx197$val$idx = 0, $$idx197$val$idx$val = 0, $$idx198 = 0, $$idx198$val = 0, $$idx199 = 0, $$idx199$val = 0, $$idx200 = 0, $$idx200$val = 0, $$idx201$val = 0, $$idx202$val = 0, $$idx202$val$idx = 0, $$idx202$val$idx$val = 0, $$idx202$val$idx$val$idx = 0, $$idx202$val$idx$val$idx$val = 0, $$idx203 = 0;
 var $$idx203$val = 0, $$lcssa = 0, $$not = 0, $$off$i$i = 0, $$phi$trans$insert = 0, $$phi$trans$insert257 = 0, $$pre = 0, $$pre253 = 0, $$pre255 = 0, $$pre256 = 0, $$pre258 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0;
 var $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0;
 var $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0;
 var $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0;
 var $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0;
 var $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0;
 var $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0;
 var $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0;
 var $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0;
 var $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0;
 var $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0;
 var $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0;
 var $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0;
 var $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0;
 var $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0;
 var $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0;
 var $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0;
 var $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0;
 var $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0;
 var $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0;
 var $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0;
 var $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0;
 var $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $or$cond = 0, $or$cond$i = 0, $or$cond185 = 0, $or$cond188 = 0, $or$cond189 = 0, $or$cond189237 = 0, $or$cond3 = 0, $or$cond3$not = 0, $phi$call$i$i$i$i = 0, $spec$select186 = 0, $spec$select190 = 0, $spec$store$select = 0, label = 0;
 var sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $2 = sp;
 $3 = ((($0)) + 24|0);
 $4 = HEAP32[$3>>2]|0;
 $5 = HEAP32[$4>>2]|0;
 $6 = ((($4)) + 4|0);
 $7 = HEAP32[$6>>2]|0;
 $8 = ((($7)) + 4|0);
 HEAP32[$8>>2] = $5;
 $9 = ((($0)) + 28|0);
 $10 = HEAP32[$9>>2]|0;
 switch ($10|0) {
 case 6: case 0: case 5:  {
  break;
 }
 default: {
  $$24 = $10;
  STACKTOP = sp;return ($$24|0);
 }
 }
 $$idx192$val$idx$val = HEAP32[$7>>2]|0;
 $11 = ((($0)) + 4|0);
 $12 = HEAP32[$11>>2]|0;
 $$idx191 = ((($12)) + 4|0);
 $$idx191$val = HEAP32[$$idx191>>2]|0;
 $$idx191$val$idx$val = HEAP32[$$idx191$val>>2]|0;
 $13 = HEAP32[$0>>2]|0;
 $14 = ($13|0)==(0|0);
 if ($14) {
  label = 4;
 } else {
  $15 = ((($7)) + 20|0);
  $16 = HEAP8[$15>>0]|0;
  $17 = ($16<<24>>24)==(2);
  if ($17) {
   $$0181 = 0;$$1 = 5;
  } else {
   label = 4;
  }
 }
 if ((label|0) == 4) {
  $$idx199 = ((($4)) + 8|0);
  $$idx199$val = HEAP8[$$idx199>>0]|0;
  $18 = ($$idx199$val<<24>>24)==(0);
  if ($18) {
   $19 = (_sqlite3BtreeBeginTrans($4,0,0)|0);
   $$0181 = 1;$$1 = $19;
  } else {
   $$0181 = 0;$$1 = 0;
  }
 }
 $20 = ((($0)) + 12|0);
 $21 = HEAP32[$20>>2]|0;
 $22 = $21 | $$1;
 $23 = ($22|0)==(0);
 if ($23) {
  $$idx201$val = HEAP32[$11>>2]|0;
  $$idx202$val = HEAP32[$3>>2]|0;
  $$idx202$val$idx = ((($$idx202$val)) + 4|0);
  $$idx202$val$idx$val = HEAP32[$$idx202$val$idx>>2]|0;
  $$idx202$val$idx$val$idx = ((($$idx202$val$idx$val)) + 32|0);
  $$idx202$val$idx$val$idx$val = HEAP32[$$idx202$val$idx$val$idx>>2]|0;
  $24 = ((($$idx201$val)) + 4|0);
  $25 = HEAP32[$24>>2]|0;
  $26 = HEAP32[$$idx201$val>>2]|0;
  $27 = ((($25)) + 4|0);
  HEAP32[$27>>2] = $26;
  $28 = ((($25)) + 22|0);
  $29 = HEAP16[$28>>1]|0;
  $30 = $29 & 2;
  $31 = ($30<<16>>16)==(0);
  if ($31) {
   $32 = ((($25)) + 32|0);
   $33 = HEAP32[$32>>2]|0;
   $34 = ((($25)) + 36|0);
   $35 = HEAP32[$34>>2]|0;
   $36 = (($33) - ($35))|0;
   $$off$i$i = (($$idx202$val$idx$val$idx$val) + -512)|0;
   $37 = ($$off$i$i>>>0)<(65025);
   if ($37) {
    $38 = (($$idx202$val$idx$val$idx$val) + -1)|0;
    $39 = $38 & $$idx202$val$idx$val$idx$val;
    $40 = ($39|0)==(0);
    if ($40) {
     HEAP32[$32>>2] = $$idx202$val$idx$val$idx$val;
     $41 = ((($25)) + 80|0);
     $42 = HEAP32[$41>>2]|0;
     $43 = ($42|0)==(0|0);
     if (!($43)) {
      $44 = ((($42)) + -4|0);
      HEAP32[$41>>2] = $44;
      $45 = $44;
      $46 = HEAP32[(59264)>>2]|0;
      $47 = ($46>>>0)>($45>>>0);
      do {
       if ($47) {
        $58 = HEAP32[(29352)>>2]|0;
        $59 = (FUNCTION_TABLE_ii[$58 & 127]($44)|0);
        $phi$call$i$i$i$i = $59;
        label = 16;
       } else {
        $48 = HEAP32[(59268)>>2]|0;
        $49 = ($48>>>0)>($45>>>0);
        if ($49) {
         $50 = HEAP32[14979]|0;
         $51 = (($50) + -1)|0;
         HEAP32[14979] = $51;
         $52 = HEAP32[(59276)>>2]|0;
         HEAP32[$44>>2] = $52;
         HEAP32[(59276)>>2] = $44;
         $53 = HEAP32[(59280)>>2]|0;
         $54 = (($53) + 1)|0;
         HEAP32[(59280)>>2] = $54;
         $55 = HEAP32[(59260)>>2]|0;
         $56 = ($54|0)<($55|0);
         $57 = $56&1;
         HEAP32[(59284)>>2] = $57;
         break;
        } else {
         $60 = HEAP32[(29352)>>2]|0;
         $61 = (FUNCTION_TABLE_ii[$60 & 127]($44)|0);
         $phi$call$i$i$i$i = $61;
         label = 16;
         break;
        }
       }
      } while(0);
      do {
       if ((label|0) == 16) {
        $62 = HEAP32[14980]|0;
        $63 = (($62) - ($phi$call$i$i$i$i))|0;
        HEAP32[14980] = $63;
        $64 = HEAP32[7324]|0;
        $65 = ($64|0)==(0);
        if ($65) {
         $73 = HEAP32[(29344)>>2]|0;
         FUNCTION_TABLE_vi[$73 & 127]($44);
         break;
        } else {
         $66 = HEAP32[(29352)>>2]|0;
         $67 = (FUNCTION_TABLE_ii[$66 & 127]($44)|0);
         $68 = HEAP32[14978]|0;
         $69 = (($68) - ($67))|0;
         HEAP32[14978] = $69;
         $70 = HEAP32[14981]|0;
         $71 = (($70) + -1)|0;
         HEAP32[14981] = $71;
         $72 = HEAP32[(29344)>>2]|0;
         FUNCTION_TABLE_vi[$72 & 127]($44);
         break;
        }
       }
      } while(0);
      HEAP32[$41>>2] = 0;
     }
    }
   }
   $74 = HEAP32[$25>>2]|0;
   $75 = (_sqlite3PagerSetPagesize($74,$32,$36)|0);
   $76 = HEAP32[$32>>2]|0;
   $77 = $36 & 65535;
   $78 = (($76) - ($77))|0;
   HEAP32[$34>>2] = $78;
   $79 = ($75|0)==(7);
   if ($79) {
    $$3 = 7;
   } else {
    label = 21;
   }
  } else {
   label = 21;
  }
 } else {
  label = 21;
 }
 if ((label|0) == 21) {
  $80 = ($$1|0)==(0);
  if ($80) {
   $81 = HEAP32[$20>>2]|0;
   $82 = ($81|0)==(0);
   if ($82) {
    $83 = HEAP32[$11>>2]|0;
    $84 = ((($0)) + 8|0);
    $85 = (_sqlite3BtreeBeginTrans($83,2,$84)|0);
    $86 = ($85|0)==(0);
    if ($86) {
     HEAP32[$20>>2] = 1;
     $$3 = 0;
    } else {
     $$3 = $85;
    }
   } else {
    $$3 = 0;
   }
  } else {
   $$3 = $$1;
  }
 }
 $87 = HEAP32[$3>>2]|0;
 $$idx194 = ((($87)) + 4|0);
 $$idx194$val = HEAP32[$$idx194>>2]|0;
 $$idx194$val$idx = ((($$idx194$val)) + 32|0);
 $$idx194$val$idx$val = HEAP32[$$idx194$val$idx>>2]|0;
 $88 = HEAP32[$11>>2]|0;
 $$idx193 = ((($88)) + 4|0);
 $$idx193$val = HEAP32[$$idx193>>2]|0;
 $$idx193$val$idx = ((($$idx193$val)) + 32|0);
 $$idx193$val$idx$val = HEAP32[$$idx193$val$idx>>2]|0;
 $$idx$val$idx$val = HEAP32[$$idx193$val>>2]|0;
 $$idx203 = ((($$idx$val$idx$val)) + 5|0);
 $$idx203$val = HEAP8[$$idx203>>0]|0;
 $89 = ($$3|0)==(0);
 $90 = ($$idx203$val<<24>>24)==(5);
 $or$cond3 = $89 & $90;
 $or$cond3$not = $or$cond3 ^ 1;
 $91 = ($$idx194$val$idx$val|0)==($$idx193$val$idx$val|0);
 $or$cond = $91 | $or$cond3$not;
 $$4 = $or$cond ? $$3 : 8;
 $$idx197$val$idx = ((($$idx194$val)) + 44|0);
 $$idx197$val$idx$val = HEAP32[$$idx197$val$idx>>2]|0;
 $92 = ($1|0)==(0);
 L36: do {
  if ($92) {
   $$5$lcssa = $$4;
  } else {
   $93 = ((($0)) + 16|0);
   $94 = ((($$idx192$val$idx$val)) + 204|0);
   $$pre = HEAP32[$93>>2]|0;
   $$0179245 = 0;$$5246 = $$4;$96 = $$pre;
   while(1) {
    $95 = ($96>>>0)<=($$idx197$val$idx$val>>>0);
    $97 = ($$5246|0)==(0);
    $or$cond185 = $97 & $95;
    if (!($or$cond185)) {
     $$5$lcssa = $$5246;
     break L36;
    }
    $98 = HEAP32[$3>>2]|0;
    $99 = ((($98)) + 4|0);
    $100 = HEAP32[$99>>2]|0;
    $101 = ((($100)) + 32|0);
    $102 = HEAP32[$101>>2]|0;
    $103 = (1073741824 / ($102>>>0))&-1;
    $104 = (($103) + 1)|0;
    $105 = ($96|0)==($104|0);
    if ($105) {
     $$7 = 0;$126 = $96;
    } else {
     $106 = HEAP32[$94>>2]|0;
     $107 = (FUNCTION_TABLE_iiiii[$106 & 127]($$idx192$val$idx$val,$96,$2,2)|0);
     $108 = ($107|0)==(0);
     do {
      if ($108) {
       $109 = HEAP32[$2>>2]|0;
       $$idx196 = ((($109)) + 4|0);
       $$idx196$val = HEAP32[$$idx196>>2]|0;
       $110 = (_backupOnePage($0,$96,$$idx196$val,0)|0);
       $111 = HEAP32[$2>>2]|0;
       $112 = ($111|0)==(0|0);
       if ($112) {
        $$6 = $110;
       } else {
        $113 = ((($111)) + 28|0);
        $114 = HEAP16[$113>>1]|0;
        $115 = $114 & 32;
        $116 = ($115<<16>>16)==(0);
        if ($116) {
         _sqlite3PcacheRelease($111);
         $$6 = $110;
         break;
        } else {
         $117 = ((($111)) + 20|0);
         $118 = HEAP32[$117>>2]|0;
         $119 = ((($118)) + 120|0);
         $120 = HEAP32[$119>>2]|0;
         $121 = (($120) + -1)|0;
         HEAP32[$119>>2] = $121;
         $122 = ((($118)) + 136|0);
         $123 = HEAP32[$122>>2]|0;
         $124 = ((($111)) + 16|0);
         HEAP32[$124>>2] = $123;
         HEAP32[$122>>2] = $111;
         $$6 = $110;
         break;
        }
       }
      } else {
       $$6 = $107;
      }
     } while(0);
     $$pre253 = HEAP32[$93>>2]|0;
     $$7 = $$6;$126 = $$pre253;
    }
    $125 = (($126) + 1)|0;
    HEAP32[$93>>2] = $125;
    $127 = (($$0179245) + 1)|0;
    $128 = ($127>>>0)<($1>>>0);
    if ($128) {
     $$0179245 = $127;$$5246 = $$7;$96 = $125;
    } else {
     $$5$lcssa = $$7;
     break;
    }
   }
  }
 } while(0);
 switch ($$5$lcssa|0) {
 case 0:  {
  $129 = ((($0)) + 36|0);
  HEAP32[$129>>2] = $$idx197$val$idx$val;
  $130 = (($$idx197$val$idx$val) + 1)|0;
  $131 = ((($0)) + 16|0);
  $132 = HEAP32[$131>>2]|0;
  $133 = (($130) - ($132))|0;
  $134 = ((($0)) + 32|0);
  HEAP32[$134>>2] = $133;
  $135 = ($132>>>0)>($$idx197$val$idx$val>>>0);
  if ($135) {
   label = 40;
  } else {
   $136 = ((($0)) + 40|0);
   $137 = HEAP32[$136>>2]|0;
   $138 = ($137|0)==(0);
   if ($138) {
    $139 = HEAP32[$3>>2]|0;
    $$idx$i = ((($139)) + 4|0);
    $$idx$val$i = HEAP32[$$idx$i>>2]|0;
    $$idx$val$idx$val$i = HEAP32[$$idx$val$i>>2]|0;
    $140 = ((($$idx$val$idx$val$i)) + 88|0);
    $141 = HEAP32[$140>>2]|0;
    $142 = ((($0)) + 44|0);
    HEAP32[$142>>2] = $141;
    HEAP32[$140>>2] = $0;
    HEAP32[$136>>2] = 1;
    $$22 = 0;
   } else {
    $$22 = 0;
   }
  }
  break;
 }
 case 101:  {
  label = 40;
  break;
 }
 default: {
  $$22 = $$5$lcssa;
 }
 }
 L56: do {
  if ((label|0) == 40) {
   $143 = ($$idx197$val$idx$val|0)==(0);
   if ($143) {
    $144 = HEAP32[$11>>2]|0;
    $145 = HEAP32[$144>>2]|0;
    $146 = ((($144)) + 4|0);
    $147 = HEAP32[$146>>2]|0;
    $148 = ((($147)) + 4|0);
    HEAP32[$148>>2] = $145;
    $149 = ((($147)) + 44|0);
    HEAP32[$149>>2] = 0;
    $150 = (_newDatabase($147)|0);
    switch ($150|0) {
    case 0: case 101:  {
     $$0180218 = 1;
     break;
    }
    default: {
     $$22 = $150;
     break L56;
    }
    }
   } else {
    $$0180218 = $$idx197$val$idx$val;
   }
   $151 = HEAP32[$11>>2]|0;
   $152 = ((($0)) + 8|0);
   $153 = HEAP32[$152>>2]|0;
   $154 = (($153) + 1)|0;
   $155 = ((($151)) + 4|0);
   $156 = HEAP32[$155>>2]|0;
   $157 = HEAP32[$151>>2]|0;
   $158 = ((($156)) + 4|0);
   HEAP32[$158>>2] = $157;
   $159 = ((($156)) + 12|0);
   $160 = HEAP32[$159>>2]|0;
   $161 = ((($160)) + 56|0);
   $162 = HEAP32[$161>>2]|0;
   $163 = ((($160)) + 72|0);
   $164 = HEAP32[$163>>2]|0;
   $165 = ((($164)) + 20|0);
   $166 = HEAP32[$165>>2]|0;
   $167 = ((($164)) + 28|0);
   $168 = HEAP16[$167>>1]|0;
   $169 = $168 & 4;
   $170 = ($169<<16>>16)==(0);
   if ($170) {
    label = 46;
   } else {
    $171 = ((($166)) + 24|0);
    $172 = HEAP32[$171>>2]|0;
    $173 = ((($164)) + 24|0);
    $174 = HEAP32[$173>>2]|0;
    $175 = ($172>>>0)<($174>>>0);
    if ($175) {
     label = 46;
    } else {
     $176 = ((($166)) + 96|0);
     $177 = HEAP32[$176>>2]|0;
     $178 = ($177|0)==(0);
     if (!($178)) {
      $179 = (_subjournalPageIfRequired($164)|0);
      $$0$i205 = $179;
      label = 50;
     }
    }
   }
   do {
    if ((label|0) == 46) {
     $180 = ((($166)) + 40|0);
     $181 = HEAP32[$180>>2]|0;
     $182 = ($181|0)==(0);
     if (!($182)) {
      $$22 = $181;
      break L56;
     }
     $183 = ((($166)) + 148|0);
     $184 = HEAP32[$183>>2]|0;
     $185 = ((($166)) + 152|0);
     $186 = HEAP32[$185>>2]|0;
     $187 = ($184>>>0)>($186>>>0);
     if ($187) {
      $188 = (_pagerWriteLargeSector($164)|0);
      $$0$i205 = $188;
      label = 50;
      break;
     } else {
      $189 = (_pager_write($164)|0);
      $$0$i205 = $189;
      label = 50;
      break;
     }
    }
   } while(0);
   if ((label|0) == 50) {
    $190 = ($$0$i205|0)==(0);
    if (!($190)) {
     $$22 = $$0$i205;
     break;
    }
   }
   $191 = ((($162)) + 40|0);
   $192 = $154 >>> 24;
   $193 = $192&255;
   HEAP8[$191>>0] = $193;
   $194 = $154 >>> 16;
   $195 = $194&255;
   $196 = ((($162)) + 41|0);
   HEAP8[$196>>0] = $195;
   $197 = $154 >>> 8;
   $198 = $197&255;
   $199 = ((($162)) + 42|0);
   HEAP8[$199>>0] = $198;
   $200 = $154&255;
   $201 = ((($162)) + 43|0);
   HEAP8[$201>>0] = $200;
   $202 = HEAP32[$0>>2]|0;
   $203 = ($202|0)==(0|0);
   if (!($203)) {
    _sqlite3ResetAllSchemasOfConnection($202);
   }
   L77: do {
    if ($90) {
     $204 = HEAP32[$11>>2]|0;
     $205 = ((($204)) + 4|0);
     $206 = HEAP32[$205>>2]|0;
     $207 = ((($206)) + 22|0);
     $208 = HEAP16[$207>>1]|0;
     $209 = $208 & -33;
     HEAP16[$207>>1] = $209;
     $210 = (_sqlite3BtreeBeginTrans($204,0,0)|0);
     $211 = ($210|0)==(0);
     L79: do {
      if ($211) {
       $212 = ((($206)) + 12|0);
       $213 = HEAP32[$212>>2]|0;
       $214 = ((($213)) + 56|0);
       $215 = HEAP32[$214>>2]|0;
       $216 = ((($215)) + 18|0);
       $217 = HEAP8[$216>>0]|0;
       $218 = ($217<<24>>24)==(2);
       if ($218) {
        $219 = ((($215)) + 19|0);
        $220 = HEAP8[$219>>0]|0;
        $221 = ($220<<24>>24)==(2);
        if (!($221)) {
         label = 57;
        }
       } else {
        label = 57;
       }
       if ((label|0) == 57) {
        $222 = (_sqlite3BtreeBeginTrans($204,2,0)|0);
        $223 = ($222|0)==(0);
        if (!($223)) {
         $$1$i$ph = $222;
         break;
        }
        $224 = HEAP32[$212>>2]|0;
        $225 = ((($224)) + 72|0);
        $226 = HEAP32[$225>>2]|0;
        $227 = ((($226)) + 20|0);
        $228 = HEAP32[$227>>2]|0;
        $229 = ((($226)) + 28|0);
        $230 = HEAP16[$229>>1]|0;
        $231 = $230 & 4;
        $232 = ($231<<16>>16)==(0);
        if ($232) {
         label = 62;
        } else {
         $233 = ((($228)) + 24|0);
         $234 = HEAP32[$233>>2]|0;
         $235 = ((($226)) + 24|0);
         $236 = HEAP32[$235>>2]|0;
         $237 = ($234>>>0)<($236>>>0);
         if ($237) {
          label = 62;
         } else {
          $238 = ((($228)) + 96|0);
          $239 = HEAP32[$238>>2]|0;
          $240 = ($239|0)==(0);
          if (!($240)) {
           $241 = (_subjournalPageIfRequired($226)|0);
           $$0$i206 = $241;
           label = 66;
          }
         }
        }
        do {
         if ((label|0) == 62) {
          $242 = ((($228)) + 40|0);
          $243 = HEAP32[$242>>2]|0;
          $244 = ($243|0)==(0);
          if (!($244)) {
           $$1$i$ph = $243;
           break L79;
          }
          $245 = ((($228)) + 148|0);
          $246 = HEAP32[$245>>2]|0;
          $247 = ((($228)) + 152|0);
          $248 = HEAP32[$247>>2]|0;
          $249 = ($246>>>0)>($248>>>0);
          if ($249) {
           $250 = (_pagerWriteLargeSector($226)|0);
           $$0$i206 = $250;
           label = 66;
           break;
          } else {
           $251 = (_pager_write($226)|0);
           $$0$i206 = $251;
           label = 66;
           break;
          }
         }
        } while(0);
        if ((label|0) == 66) {
         $252 = ($$0$i206|0)==(0);
         if (!($252)) {
          $$1$i$ph = $$0$i206;
          break;
         }
        }
        HEAP8[$216>>0] = 2;
        $253 = ((($215)) + 19|0);
        HEAP8[$253>>0] = 2;
       }
       $256 = HEAP16[$207>>1]|0;
       $257 = $256 & -33;
       HEAP16[$207>>1] = $257;
       break L77;
      } else {
       $$1$i$ph = $210;
      }
     } while(0);
     $254 = HEAP16[$207>>1]|0;
     $255 = $254 & -33;
     HEAP16[$207>>1] = $255;
     $$22 = $$1$i$ph;
     break L56;
    }
   } while(0);
   $258 = ($$idx194$val$idx$val|0)<($$idx193$val$idx$val|0);
   if ($258) {
    $259 = (($$idx193$val$idx$val|0) / ($$idx194$val$idx$val|0))&-1;
    $260 = (($$0180218) + -1)|0;
    $261 = (($260) + ($259))|0;
    $262 = (($261|0) / ($259|0))&-1;
    $263 = HEAP32[$11>>2]|0;
    $264 = ((($263)) + 4|0);
    $265 = HEAP32[$264>>2]|0;
    $266 = ((($265)) + 32|0);
    $267 = HEAP32[$266>>2]|0;
    $268 = (1073741824 / ($267>>>0))&-1;
    $269 = (($268) + 1)|0;
    $270 = ($262|0)==($269|0);
    $271 = $270 << 31 >> 31;
    $spec$select186 = (($262) + ($271))|0;
    $272 = ($$idx194$val$idx$val|0)<(0);
    $273 = $272 << 31 >> 31;
    $274 = ($$0180218|0)<(0);
    $275 = $274 << 31 >> 31;
    $276 = (___muldi3(($$0180218|0),($275|0),($$idx194$val$idx$val|0),($273|0))|0);
    $277 = (getTempRet0() | 0);
    $$idx198 = ((($$idx191$val$idx$val)) + 60|0);
    $$idx198$val = HEAP32[$$idx198>>2]|0;
    $$idx200 = ((($$idx191$val$idx$val)) + 24|0);
    $$idx200$val = HEAP32[$$idx200>>2]|0;
    $278 = ($spec$select186>>>0)>($$idx200$val>>>0);
    if ($278) {
     label = 91;
    } else {
     $279 = ((($$idx191$val$idx$val)) + 204|0);
     $$0178240 = $spec$select186;$$12241 = 0;$281 = 1073741824;$282 = $267;
     while(1) {
      $280 = (($281>>>0) / ($282>>>0))&-1;
      $283 = (($280) + 1)|0;
      $284 = ($$0178240|0)==($283|0);
      if ($284) {
       $$14 = $$12241;
      } else {
       $285 = HEAP32[$279>>2]|0;
       $286 = (FUNCTION_TABLE_iiiii[$285 & 127]($$idx191$val$idx$val,$$0178240,$2,0)|0);
       $287 = ($286|0)==(0);
       do {
        if ($287) {
         $288 = HEAP32[$2>>2]|0;
         $289 = ((($288)) + 20|0);
         $290 = HEAP32[$289>>2]|0;
         $291 = ((($288)) + 28|0);
         $292 = HEAP16[$291>>1]|0;
         $293 = $292 & 4;
         $294 = ($293<<16>>16)==(0);
         do {
          if ($294) {
           label = 79;
          } else {
           $295 = ((($290)) + 24|0);
           $296 = HEAP32[$295>>2]|0;
           $297 = ((($288)) + 24|0);
           $298 = HEAP32[$297>>2]|0;
           $299 = ($296>>>0)<($298>>>0);
           if ($299) {
            label = 79;
           } else {
            $300 = ((($290)) + 96|0);
            $301 = HEAP32[$300>>2]|0;
            $302 = ($301|0)==(0);
            if ($302) {
             $$0$i208 = 0;
             break;
            }
            $303 = (_subjournalPageIfRequired($288)|0);
            $$0$i208 = $303;
           }
          }
         } while(0);
         do {
          if ((label|0) == 79) {
           label = 0;
           $304 = ((($290)) + 40|0);
           $305 = HEAP32[$304>>2]|0;
           $306 = ($305|0)==(0);
           if ($306) {
            $307 = ((($290)) + 148|0);
            $308 = HEAP32[$307>>2]|0;
            $309 = ((($290)) + 152|0);
            $310 = HEAP32[$309>>2]|0;
            $311 = ($308>>>0)>($310>>>0);
            if ($311) {
             $312 = (_pagerWriteLargeSector($288)|0);
             $$0$i208 = $312;
             break;
            } else {
             $313 = (_pager_write($288)|0);
             $$0$i208 = $313;
             break;
            }
           } else {
            $$0$i208 = $305;
           }
          }
         } while(0);
         $314 = HEAP32[$2>>2]|0;
         $315 = ($314|0)==(0|0);
         if ($315) {
          $$13 = $$0$i208;
         } else {
          $316 = ((($314)) + 28|0);
          $317 = HEAP16[$316>>1]|0;
          $318 = $317 & 32;
          $319 = ($318<<16>>16)==(0);
          if ($319) {
           _sqlite3PcacheRelease($314);
           $$13 = $$0$i208;
           break;
          } else {
           $320 = ((($314)) + 20|0);
           $321 = HEAP32[$320>>2]|0;
           $322 = ((($321)) + 120|0);
           $323 = HEAP32[$322>>2]|0;
           $324 = (($323) + -1)|0;
           HEAP32[$322>>2] = $324;
           $325 = ((($321)) + 136|0);
           $326 = HEAP32[$325>>2]|0;
           $327 = ((($314)) + 16|0);
           HEAP32[$327>>2] = $326;
           HEAP32[$325>>2] = $314;
           $$13 = $$0$i208;
           break;
          }
         }
        } else {
         $$13 = $286;
        }
       } while(0);
       $$14 = $$13;
      }
      $328 = (($$0178240) + 1)|0;
      $329 = ($$14|0)==(0);
      $$not = $329 ^ 1;
      $330 = ($328>>>0)>($$idx200$val>>>0);
      $or$cond188 = $330 | $$not;
      if ($or$cond188) {
       break;
      }
      $$pre255 = HEAP32[$11>>2]|0;
      $$phi$trans$insert = ((($$pre255)) + 4|0);
      $$pre256 = HEAP32[$$phi$trans$insert>>2]|0;
      $$phi$trans$insert257 = ((($$pre256)) + 32|0);
      $$pre258 = HEAP32[$$phi$trans$insert257>>2]|0;
      $$0178240 = $328;$$12241 = $$14;$281 = 1073741824;$282 = $$pre258;
     }
     if ($329) {
      label = 91;
     } else {
      $$15 = $$14;
     }
    }
    if ((label|0) == 91) {
     $331 = (_sqlite3PagerCommitPhaseOne($$idx191$val$idx$val,0,1)|0);
     $$15 = $331;
    }
    $332 = (1073741824 + ($$idx193$val$idx$val))|0;
    $333 = ($332|0)<(0);
    $334 = $333 << 31 >> 31;
    $335 = ($277|0)>($334|0);
    $336 = ($276>>>0)>($332>>>0);
    $337 = ($277|0)==($334|0);
    $338 = $337 & $336;
    $339 = $335 | $338;
    $340 = $339 ? $332 : $276;
    $341 = $339 ? $334 : $277;
    $342 = (1073741824 + ($$idx194$val$idx$val))|0;
    $343 = ($342|0)<(0);
    $344 = $343 << 31 >> 31;
    $345 = ($$15|0)==(0);
    $346 = ($341|0)>($344|0);
    $347 = ($340>>>0)>($342>>>0);
    $348 = ($341|0)==($344|0);
    $349 = $348 & $347;
    $350 = $346 | $349;
    $or$cond189237 = $350 & $345;
    if ($or$cond189237) {
     $351 = ((($$idx192$val$idx$val)) + 204|0);
     $352 = $342;$353 = $344;
     while(1) {
      HEAP32[$2>>2] = 0;
      $354 = (___divdi3(($352|0),($353|0),($$idx194$val$idx$val|0),($273|0))|0);
      $355 = (getTempRet0() | 0);
      $356 = (($354) + 1)|0;
      $357 = HEAP32[$351>>2]|0;
      $358 = (FUNCTION_TABLE_iiiii[$357 & 127]($$idx192$val$idx$val,$356,$2,0)|0);
      $359 = ($358|0)==(0);
      if ($359) {
       $360 = HEAP32[$2>>2]|0;
       $$idx195 = ((($360)) + 4|0);
       $$idx195$val = HEAP32[$$idx195>>2]|0;
       $361 = HEAP32[$$idx198$val>>2]|0;
       $362 = ((($361)) + 12|0);
       $363 = HEAP32[$362>>2]|0;
       $364 = (FUNCTION_TABLE_iiiiii[$363 & 127]($$idx198$val,$$idx195$val,$$idx194$val$idx$val,$352,$353)|0);
       $$17 = $364;
      } else {
       $$17 = $358;
      }
      $365 = HEAP32[$2>>2]|0;
      $366 = ($365|0)==(0|0);
      do {
       if (!($366)) {
        $367 = ((($365)) + 28|0);
        $368 = HEAP16[$367>>1]|0;
        $369 = $368 & 32;
        $370 = ($369<<16>>16)==(0);
        if ($370) {
         _sqlite3PcacheRelease($365);
         break;
        } else {
         $371 = ((($365)) + 20|0);
         $372 = HEAP32[$371>>2]|0;
         $373 = ((($372)) + 120|0);
         $374 = HEAP32[$373>>2]|0;
         $375 = (($374) + -1)|0;
         HEAP32[$373>>2] = $375;
         $376 = ((($372)) + 136|0);
         $377 = HEAP32[$376>>2]|0;
         $378 = ((($365)) + 16|0);
         HEAP32[$378>>2] = $377;
         HEAP32[$376>>2] = $365;
         break;
        }
       }
      } while(0);
      $379 = (_i64Add(($352|0),($353|0),($$idx194$val$idx$val|0),($273|0))|0);
      $380 = (getTempRet0() | 0);
      $381 = ($$17|0)==(0);
      $382 = ($380|0)<($341|0);
      $383 = ($379>>>0)<($340>>>0);
      $384 = ($380|0)==($341|0);
      $385 = $384 & $383;
      $386 = $382 | $385;
      $or$cond189 = $386 & $381;
      if ($or$cond189) {
       $352 = $379;$353 = $380;
      } else {
       $$16$lcssa = $$17;$$lcssa = $381;
       break;
      }
     }
    } else {
     $$16$lcssa = $$15;$$lcssa = $345;
    }
    if (!($$lcssa)) {
     $$22 = $$16$lcssa;
     break;
    }
    $387 = HEAP32[$$idx198$val>>2]|0;
    $388 = ((($387)) + 24|0);
    $389 = HEAP32[$388>>2]|0;
    $390 = (FUNCTION_TABLE_iii[$389 & 255]($$idx198$val,$2)|0);
    $391 = ($390|0)==(0);
    $392 = $2;
    $393 = $392;
    $394 = HEAP32[$393>>2]|0;
    $395 = (($392) + 4)|0;
    $396 = $395;
    $397 = HEAP32[$396>>2]|0;
    $398 = ($397|0)>($277|0);
    $399 = ($394>>>0)>($276>>>0);
    $400 = ($397|0)==($277|0);
    $401 = $400 & $399;
    $402 = $398 | $401;
    $or$cond$i = $391 & $402;
    if ($or$cond$i) {
     $403 = HEAP32[$$idx198$val>>2]|0;
     $404 = ((($403)) + 16|0);
     $405 = HEAP32[$404>>2]|0;
     $406 = (FUNCTION_TABLE_iiii[$405 & 127]($$idx198$val,$276,$277)|0);
     $$0$i204 = $406;
    } else {
     $$0$i204 = $390;
    }
    $407 = ($$0$i204|0)==(0);
    if (!($407)) {
     $$22 = $$0$i204;
     break;
    }
    $408 = HEAP32[$$idx198>>2]|0;
    $409 = HEAP32[$408>>2]|0;
    $410 = ($409|0)==(0|0);
    if (!($410)) {
     $411 = ((($409)) + 40|0);
     $412 = HEAP32[$411>>2]|0;
     $413 = (FUNCTION_TABLE_iiii[$412 & 127]($408,21,0)|0);
     switch ($413|0) {
     case 0: case 12:  {
      break;
     }
     default: {
      $$22 = $413;
      break L56;
     }
     }
    }
    $414 = ((($$idx191$val$idx$val)) + 7|0);
    $415 = HEAP8[$414>>0]|0;
    $416 = ($415<<24>>24)==(0);
    if ($416) {
     $417 = HEAP32[$$idx198>>2]|0;
     $418 = ((($$idx191$val$idx$val)) + 10|0);
     $419 = HEAP8[$418>>0]|0;
     $420 = ($419<<24>>24)==(0);
     if (!($420)) {
      $421 = $419&255;
      $422 = HEAP32[$417>>2]|0;
      $423 = ((($422)) + 20|0);
      $424 = HEAP32[$423>>2]|0;
      $425 = (FUNCTION_TABLE_iii[$424 & 255]($417,$421)|0);
      $$20 = $425;
      label = 111;
     }
    }
   } else {
    $426 = (($$idx194$val$idx$val|0) / ($$idx193$val$idx$val|0))&-1;
    $427 = Math_imul($426, $$0180218)|0;
    $428 = ((($$idx191$val$idx$val)) + 24|0);
    HEAP32[$428>>2] = $427;
    $429 = (_sqlite3PagerCommitPhaseOne($$idx191$val$idx$val,0,0)|0);
    $$20 = $429;
    label = 111;
   }
   if ((label|0) == 111) {
    $430 = ($$20|0)==(0);
    if (!($430)) {
     $$22 = $$20;
     break;
    }
   }
   $431 = HEAP32[$11>>2]|0;
   $432 = (_sqlite3BtreeCommitPhaseTwo($431,0)|0);
   $433 = ($432|0)==(0);
   $spec$store$select = $433 ? 101 : $432;
   $$22 = $spec$store$select;
  }
 } while(0);
 $434 = ($$0181|0)==(0);
 if (!($434)) {
  $435 = HEAP32[$3>>2]|0;
  (_sqlite3BtreeCommitPhaseOne($435,0)|0);
  $436 = HEAP32[$3>>2]|0;
  (_sqlite3BtreeCommitPhaseTwo($436,0)|0);
 }
 $437 = ($$22|0)==(3082);
 $spec$select190 = $437 ? 7 : $$22;
 HEAP32[$9>>2] = $spec$select190;
 $$24 = $spec$select190;
 STACKTOP = sp;return ($$24|0);
}
function _sqlite3BtreeBeginTrans($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$0 = 0, $$0$i104 = 0, $$0$i105 = 0, $$017$i$i = 0, $$020$i = 0, $$021$i = 0, $$028$i = 0, $$086$i = 0, $$097 = 0, $$097172 = 0, $$097173 = 0, $$1 = 0, $$1100 = 0, $$2$lcssa = 0, $$285120$i = 0, $$4 = 0, $$4$ph = 0, $$4125 = 0, $$4127129 = 0, $$idx = 0;
 var $$idx$i = 0, $$idx$i$i$i = 0, $$idx$i$i$i$i = 0, $$idx$i$i$i100$i = 0, $$idx$i$i$i110 = 0, $$idx$i$i$i95$i = 0, $$idx$i108 = 0, $$idx$i111 = 0, $$idx$val = 0, $$idx$val$i = 0, $$idx$val$i$i$i = 0, $$idx$val$i$i$i$i = 0, $$idx$val$i$i$i101$i = 0, $$idx$val$i$i$i96$i = 0, $$idx$val$i109 = 0, $$idx$val$i112 = 0, $$idx$val15$i$i$i = 0, $$idx103 = 0, $$idx103$val = 0, $$idx14$i$i$i = 0;
 var $$idx14$val$i$i$i = 0, $$idx91$i = 0, $$idx91$val$i = 0, $$idx92$i = 0, $$idx92$val$i = 0, $$idx93$i = 0, $$idx93$val$i = 0, $$off$i = 0, $$old = 0, $$old136 = 0, $$phi$trans$insert$i = 0, $$phi$trans$insert$phi$trans$insert$i = 0, $$phi$trans$insert127$i = 0, $$pn = 0, $$pr = 0, $$pre = 0, $$pre$i = 0, $$pre$i32$i = 0, $$pre$phi195Z2D = 0, $$pre$phi197Z2D = 0;
 var $$pre$phi199Z2D = 0, $$pre$phi48$iZ2D = 0, $$pre$phi49$iZ2D = 0, $$pre$phiZ2D = 0, $$pre$pre$i = 0, $$pre128$i = 0, $$pre192 = 0, $$pre193 = 0, $$pre194 = 0, $$pre196 = 0, $$pre198 = 0, $$pre42$pre$i = 0, $$pre43$i = 0, $$pre43$pre$i = 0, $$pre44$i = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0;
 var $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0;
 var $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0;
 var $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0;
 var $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0;
 var $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0;
 var $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0;
 var $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0;
 var $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0;
 var $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0;
 var $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0;
 var $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0;
 var $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0;
 var $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0;
 var $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0;
 var $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0;
 var $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0;
 var $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0;
 var $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0;
 var $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0;
 var $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0;
 var $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0;
 var $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0, $5 = 0, $50 = 0, $500 = 0;
 var $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $509 = 0, $51 = 0, $510 = 0, $511 = 0, $512 = 0, $513 = 0, $514 = 0, $515 = 0, $516 = 0, $517 = 0, $518 = 0, $519 = 0;
 var $52 = 0, $520 = 0, $521 = 0, $522 = 0, $523 = 0, $524 = 0, $525 = 0, $526 = 0, $527 = 0, $528 = 0, $529 = 0, $53 = 0, $530 = 0, $531 = 0, $532 = 0, $533 = 0, $534 = 0, $535 = 0, $536 = 0, $537 = 0;
 var $538 = 0, $539 = 0, $54 = 0, $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $545 = 0, $546 = 0, $547 = 0, $548 = 0, $549 = 0, $55 = 0, $550 = 0, $551 = 0, $552 = 0, $553 = 0, $554 = 0, $555 = 0;
 var $556 = 0, $557 = 0, $558 = 0, $559 = 0, $56 = 0, $560 = 0, $561 = 0, $562 = 0, $563 = 0, $564 = 0, $565 = 0, $566 = 0, $567 = 0, $568 = 0, $569 = 0, $57 = 0, $570 = 0, $571 = 0, $572 = 0, $573 = 0;
 var $574 = 0, $575 = 0, $576 = 0, $577 = 0, $578 = 0, $579 = 0, $58 = 0, $580 = 0, $581 = 0, $582 = 0, $583 = 0, $584 = 0, $585 = 0, $586 = 0, $587 = 0, $588 = 0, $589 = 0, $59 = 0, $590 = 0, $591 = 0;
 var $592 = 0, $593 = 0, $594 = 0, $595 = 0, $596 = 0, $597 = 0, $598 = 0, $599 = 0, $6 = 0, $60 = 0, $600 = 0, $601 = 0, $602 = 0, $603 = 0, $604 = 0, $605 = 0, $606 = 0, $607 = 0, $608 = 0, $609 = 0;
 var $61 = 0, $610 = 0, $611 = 0, $612 = 0, $613 = 0, $614 = 0, $615 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0;
 var $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0;
 var $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $or$cond = 0, $or$cond$i = 0, $or$cond137 = 0, $or$cond140 = 0, $or$cond3 = 0, $or$cond5 = 0, $phi$call$i$i$i$i = 0, $spec$select = 0, $spec$select$i = 0, $spec$select102 = 0, $spec$select132$i = 0, $vararg_buffer = 0;
 var $vararg_ptr1 = 0, $vararg_ptr2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $vararg_buffer = sp;
 $3 = sp + 12|0;
 $4 = ((($0)) + 4|0);
 $5 = HEAP32[$4>>2]|0;
 $6 = HEAP32[$0>>2]|0;
 $7 = ((($5)) + 4|0);
 HEAP32[$7>>2] = $6;
 $8 = ((($0)) + 8|0);
 $9 = HEAP8[$8>>0]|0;
 $10 = ($9<<24>>24)==(2);
 $11 = $6;
 do {
  if (!($10)) {
   $12 = ($9<<24>>24)!=(1);
   $13 = ($1|0)!=(0);
   $or$cond = $13 | $12;
   if ($or$cond) {
    $14 = ((($11)) + 32|0);
    $15 = $14;
    $16 = $15;
    $17 = HEAP32[$16>>2]|0;
    $18 = (($15) + 4)|0;
    $19 = $18;
    $20 = HEAP32[$19>>2]|0;
    $21 = $17 & 33554432;
    $22 = ($21|0)==(0);
    $23 = (0)==(0);
    $24 = $22 & $23;
    if (!($24)) {
     $25 = HEAP32[$5>>2]|0;
     $$idx103 = ((($25)) + 14|0);
     $$idx103$val = HEAP8[$$idx103>>0]|0;
     $26 = ($$idx103$val<<24>>24)==(0);
     if ($26) {
      $27 = ((($5)) + 22|0);
      $28 = HEAP16[$27>>1]|0;
      $29 = $28 & -2;
      HEAP16[$27>>1] = $29;
     }
    }
    $30 = ((($5)) + 22|0);
    $31 = HEAP16[$30>>1]|0;
    $32 = $31 & 1;
    $33 = ($32<<16>>16)!=(0);
    $or$cond3 = $13 & $33;
    if ($or$cond3) {
     $$0 = 8;
     STACKTOP = sp;return ($$0|0);
    }
    L11: do {
     if ($13) {
      $34 = ((($5)) + 20|0);
      $35 = HEAP8[$34>>0]|0;
      $36 = ($35<<24>>24)!=(2);
      $37 = $31 & 128;
      $38 = ($37<<16>>16)==(0);
      $or$cond137 = $38 & $36;
      if ($or$cond137) {
       $41 = ($1|0)>(1);
       if ($41) {
        $42 = ((($5)) + 72|0);
        $$097172 = HEAP32[$42>>2]|0;
        $43 = ($$097172|0)==(0|0);
        if (!($43)) {
         $$097173 = $$097172;
         while(1) {
          $46 = HEAP32[$$097173>>2]|0;
          $47 = ($46|0)==($0|0);
          if (!($47)) {
           $$pn = $46;
           label = 16;
           break L11;
          }
          $44 = ((($$097173)) + 12|0);
          $$097 = HEAP32[$44>>2]|0;
          $45 = ($$097|0)==(0|0);
          if ($45) {
           break;
          } else {
           $$097173 = $$097;
          }
         }
        }
       }
      } else {
       label = 10;
      }
     } else {
      $$old = $31 & 128;
      $$old136 = ($$old<<16>>16)==(0);
      if (!($$old136)) {
       label = 10;
      }
     }
    } while(0);
    if ((label|0) == 10) {
     $39 = ((($5)) + 76|0);
     $40 = HEAP32[$39>>2]|0;
     $$pn = $40;
     label = 16;
    }
    if ((label|0) == 16) {
     $$1 = HEAP32[$$pn>>2]|0;
     $48 = ($$1|0)==(0|0);
     if (!($48)) {
      $$0 = 262;
      STACKTOP = sp;return ($$0|0);
     }
    }
    $49 = ((($0)) + 9|0);
    $50 = HEAP8[$49>>0]|0;
    $51 = ($50<<24>>24)==(0);
    L26: do {
     if (!($51)) {
      $52 = ((($5)) + 76|0);
      $53 = HEAP32[$52>>2]|0;
      $54 = ($53|0)==($0|0);
      $55 = $31 & 64;
      $56 = ($55<<16>>16)==(0);
      $or$cond140 = $56 | $54;
      if (!($or$cond140)) {
       $$0 = 262;
       STACKTOP = sp;return ($$0|0);
      }
      $57 = ((($5)) + 72|0);
      $$020$i = HEAP32[$57>>2]|0;
      $58 = ($$020$i|0)==(0|0);
      if (!($58)) {
       $$021$i = $$020$i;
       while(1) {
        $59 = HEAP32[$$021$i>>2]|0;
        $60 = ($59|0)==($0|0);
        if (!($60)) {
         $61 = ((($$021$i)) + 4|0);
         $62 = HEAP32[$61>>2]|0;
         $63 = ($62|0)==(1);
         if ($63) {
          $64 = ((($$021$i)) + 8|0);
          $65 = HEAP8[$64>>0]|0;
          $66 = ($65<<24>>24)==(1);
          if (!($66)) {
           $$0 = 262;
           break;
          }
         }
        }
        $67 = ((($$021$i)) + 12|0);
        $$0$i105 = HEAP32[$67>>2]|0;
        $68 = ($$0$i105|0)==(0|0);
        if ($68) {
         break L26;
        } else {
         $$021$i = $$0$i105;
        }
       }
       STACKTOP = sp;return ($$0|0);
      }
     }
    } while(0);
    $69 = $31 & -17;
    HEAP16[$30>>1] = $69;
    $70 = ((($5)) + 44|0);
    $71 = HEAP32[$70>>2]|0;
    $72 = ($71|0)==(0);
    if ($72) {
     $73 = $31 | 16;
     HEAP16[$30>>1] = $73;
    }
    $74 = ((($5)) + 12|0);
    $75 = ((($5)) + 20|0);
    $$phi$trans$insert127$i = ((($5)) + 36|0);
    $76 = ((($5)) + 24|0);
    $77 = ((($5)) + 26|0);
    $78 = ((($5)) + 28|0);
    $79 = ((($5)) + 30|0);
    $80 = ((($5)) + 21|0);
    $81 = ((($5)) + 32|0);
    $82 = ((($5)) + 80|0);
    $83 = ((($5)) + 17|0);
    $84 = ((($5)) + 18|0);
    $85 = ($1|0)<(2);
    $$1100 = 0;
    L43: while(1) {
     $86 = HEAP32[$74>>2]|0;
     $87 = ($86|0)==(0|0);
     L45: do {
      if ($87) {
       L47: while(1) {
        $88 = HEAP32[$5>>2]|0;
        $89 = (_sqlite3PagerSharedLock($88)|0);
        $90 = ($89|0)==(0);
        if (!($90)) {
         $$4$ph = $89;
         label = 135;
         break L45;
        }
        $91 = HEAP32[$5>>2]|0;
        $92 = ((($91)) + 204|0);
        $93 = HEAP32[$92>>2]|0;
        $94 = (FUNCTION_TABLE_iiiii[$93 & 127]($91,1,$3,0)|0);
        $95 = ($94|0)==(0);
        if (!($95)) {
         label = 35;
         break;
        }
        $96 = HEAP32[$3>>2]|0;
        $$idx14$i$i$i = ((($96)) + 8|0);
        $$idx14$val$i$i$i = HEAP32[$$idx14$i$i$i>>2]|0;
        $97 = ((($$idx14$val$i$i$i)) + 4|0);
        $98 = HEAP32[$97>>2]|0;
        $99 = ($98|0)==(1);
        if ($99) {
         $$phi$trans$insert$i = ((($$idx14$val$i$i$i)) + 56|0);
         $$pre$i = HEAP32[$$phi$trans$insert$i>>2]|0;
         $107 = $$pre$i;
        } else {
         $$idx$i$i$i = ((($96)) + 4|0);
         $$idx$val15$i$i$i = HEAP32[$$idx$i$i$i>>2]|0;
         $100 = ((($$idx14$val$i$i$i)) + 56|0);
         HEAP32[$100>>2] = $$idx$val15$i$i$i;
         $101 = ((($$idx14$val$i$i$i)) + 72|0);
         HEAP32[$101>>2] = $96;
         $102 = ((($$idx14$val$i$i$i)) + 52|0);
         HEAP32[$102>>2] = $5;
         HEAP32[$97>>2] = 1;
         $103 = ((($$idx14$val$i$i$i)) + 9|0);
         HEAP8[$103>>0] = 100;
         $104 = $$idx$val15$i$i$i;
         $107 = $104;
        }
        $105 = $$idx14$val$i$i$i;
        $106 = ((($107)) + 28|0);
        $108 = HEAP8[$106>>0]|0;
        $109 = $108&255;
        $110 = $109 << 24;
        $111 = ((($107)) + 29|0);
        $112 = HEAP8[$111>>0]|0;
        $113 = $112&255;
        $114 = $113 << 16;
        $115 = $114 | $110;
        $116 = ((($107)) + 30|0);
        $117 = HEAP8[$116>>0]|0;
        $118 = $117&255;
        $119 = $118 << 8;
        $120 = $115 | $119;
        $121 = ((($107)) + 31|0);
        $122 = HEAP8[$121>>0]|0;
        $123 = $122&255;
        $124 = $120 | $123;
        $125 = HEAP32[$5>>2]|0;
        $$idx$i = ((($125)) + 24|0);
        $$idx$val$i = HEAP32[$$idx$i>>2]|0;
        $126 = ($124|0)==(0);
        if ($126) {
         label = 38;
        } else {
         $127 = ((($107)) + 24|0);
         $128 = ((($107)) + 92|0);
         $129 = (_memcmp($127,$128,4)|0);
         $130 = ($129|0)==(0);
         if ($130) {
          $$086$i = $124;
         } else {
          label = 38;
         }
        }
        if ((label|0) == 38) {
         label = 0;
         $$086$i = $$idx$val$i;
        }
        $131 = HEAP32[$7>>2]|0;
        $132 = ((($131)) + 32|0);
        $133 = $132;
        $134 = $133;
        $135 = HEAP32[$134>>2]|0;
        $136 = (($133) + 4)|0;
        $137 = $136;
        $138 = HEAP32[$137>>2]|0;
        $139 = $135 & 33554432;
        $140 = ($139|0)==(0);
        $141 = (0)==(0);
        $142 = $140 & $141;
        $spec$select$i = $142 ? $$086$i : 0;
        $143 = ($spec$select$i|0)==(0);
        do {
         if ($143) {
          $$pre128$i = HEAP32[$$phi$trans$insert127$i>>2]|0;
          $312 = $$pre128$i;
          label = 85;
         } else {
          $144 = (_memcmp($107,2720,16)|0);
          $145 = ($144|0)==(0);
          if (!($145)) {
           $$285120$i = 26;
           break L47;
          }
          $146 = ((($107)) + 18|0);
          $147 = HEAP8[$146>>0]|0;
          $148 = ($147&255)>(2);
          if ($148) {
           $149 = HEAP16[$30>>1]|0;
           $150 = $149 | 1;
           HEAP16[$30>>1] = $150;
          }
          $151 = ((($107)) + 19|0);
          $152 = HEAP8[$151>>0]|0;
          $153 = ($152&255)>(2);
          if ($153) {
           $$285120$i = 26;
           break L47;
          }
          $154 = ($152<<24>>24)==(2);
          do {
           if ($154) {
            $155 = HEAP16[$30>>1]|0;
            $156 = $155 & 32;
            $157 = ($156<<16>>16)==(0);
            if (!($157)) {
             label = 62;
             break;
            }
            $158 = ((($125)) + 12|0);
            $159 = HEAP8[$158>>0]|0;
            $160 = ($159<<24>>24)==(0);
            if (!($160)) {
             label = 62;
             break;
            }
            $161 = ((($125)) + 216|0);
            $162 = HEAP32[$161>>2]|0;
            $163 = ($162|0)==(0|0);
            if (!($163)) {
             label = 62;
             break;
            }
            $164 = ((($125)) + 60|0);
            $165 = HEAP32[$164>>2]|0;
            $166 = HEAP32[$165>>2]|0;
            $167 = ((($125)) + 13|0);
            $168 = HEAP8[$167>>0]|0;
            $169 = ($168<<24>>24)==(0);
            if (!($169)) {
             $$285120$i = 14;
             break L47;
            }
            $170 = ((($125)) + 4|0);
            $171 = HEAP8[$170>>0]|0;
            $172 = ($171<<24>>24)==(0);
            if ($172) {
             $173 = HEAP32[$166>>2]|0;
             $174 = ($173|0)>(1);
             if (!($174)) {
              $$285120$i = 14;
              break L47;
             }
             $175 = ((($166)) + 52|0);
             $176 = HEAP32[$175>>2]|0;
             $177 = ($176|0)==(0|0);
             if ($177) {
              $$285120$i = 14;
              break L47;
             }
            }
            $178 = ((($125)) + 64|0);
            $179 = HEAP32[$178>>2]|0;
            $180 = HEAP32[$179>>2]|0;
            $181 = ($180|0)==(0|0);
            if (!($181)) {
             $182 = ((($180)) + 4|0);
             $183 = HEAP32[$182>>2]|0;
             (FUNCTION_TABLE_ii[$183 & 127]($179)|0);
             HEAP32[$179>>2] = 0;
            }
            $184 = (_pagerOpenWal($125)|0);
            $185 = ($184|0)==(0);
            if (!($185)) {
             $$285120$i = $184;
             break L47;
            }
            $186 = ((($125)) + 5|0);
            HEAP8[$186>>0] = 5;
            $187 = ((($125)) + 16|0);
            HEAP8[$187>>0] = 0;
            $$idx93$i = ((($$idx14$val$i$i$i)) + 72|0);
            $$idx93$val$i = HEAP32[$$idx93$i>>2]|0;
            $188 = ((($$idx93$val$i)) + 20|0);
            $189 = HEAP32[$188>>2]|0;
            _sqlite3PcacheRelease($$idx93$val$i);
            $190 = ((($189)) + 212|0);
            $191 = HEAP32[$190>>2]|0;
            $$idx$i$i$i95$i = ((($191)) + 12|0);
            $$idx$val$i$i$i96$i = HEAP32[$$idx$i$i$i95$i>>2]|0;
            $192 = ($$idx$val$i$i$i96$i|0)==(0);
            if (!($192)) {
             break;
            }
            $193 = ((($189)) + 16|0);
            $194 = HEAP8[$193>>0]|0;
            L83: do {
             switch ($194<<24>>24) {
             case 0: case 6:  {
              break;
             }
             case 1:  {
              $195 = ((($189)) + 4|0);
              $196 = HEAP8[$195>>0]|0;
              $197 = ($196<<24>>24)==(0);
              if (!($197)) {
               break L83;
              }
              (_pager_end_transaction($189,0,0)|0);
              break;
             }
             default: {
              (_sqlite3PagerRollback($189)|0);
             }
             }
            } while(0);
            _pager_unlock($189);
           } else {
            label = 62;
           }
          } while(0);
          if ((label|0) == 62) {
           label = 0;
           $198 = ((($107)) + 21|0);
           $199 = (_memcmp($198,32387,3)|0);
           $200 = ($199|0)==(0);
           if (!($200)) {
            $$285120$i = 26;
            break L47;
           }
           $201 = ((($107)) + 16|0);
           $202 = HEAP8[$201>>0]|0;
           $203 = $202&255;
           $204 = $203 << 8;
           $205 = ((($107)) + 17|0);
           $206 = HEAP8[$205>>0]|0;
           $207 = $206&255;
           $208 = $207 << 16;
           $209 = $208 | $204;
           $210 = (($209) + 16777215)|0;
           $211 = $210 & $209;
           $212 = ($211|0)!=(0);
           $$off$i = (($209) + -257)|0;
           $213 = ($$off$i>>>0)>(65279);
           $214 = $213 | $212;
           if ($214) {
            $$285120$i = 26;
            break L47;
           }
           $215 = HEAP16[$30>>1]|0;
           $216 = $215 | 2;
           HEAP16[$30>>1] = $216;
           $217 = ((($107)) + 20|0);
           $218 = HEAP8[$217>>0]|0;
           $219 = $218&255;
           $220 = (($209) - ($219))|0;
           $221 = HEAP32[$81>>2]|0;
           $222 = ($209|0)==($221|0);
           if ($222) {
            $265 = $135 & 268435457;
            $266 = ($265|0)!=(1);
            $267 = (0)!=(0);
            $268 = $266 | $267;
            $269 = ($spec$select$i>>>0)>($$idx$val$i>>>0);
            $or$cond$i = $268 & $269;
            if ($or$cond$i) {
             label = 82;
             break L47;
            }
            $270 = ($220>>>0)<(480);
            if ($270) {
             $$285120$i = 26;
             break L47;
            }
            HEAP32[$81>>2] = $209;
            HEAP32[$$phi$trans$insert127$i>>2] = $220;
            $271 = ((($107)) + 52|0);
            $272 = HEAP8[$271>>0]|0;
            $273 = $272&255;
            $274 = $273 << 24;
            $275 = ((($107)) + 53|0);
            $276 = HEAP8[$275>>0]|0;
            $277 = $276&255;
            $278 = $277 << 16;
            $279 = $278 | $274;
            $280 = ((($107)) + 54|0);
            $281 = HEAP8[$280>>0]|0;
            $282 = $281&255;
            $283 = $282 << 8;
            $284 = $279 | $283;
            $285 = ((($107)) + 55|0);
            $286 = HEAP8[$285>>0]|0;
            $287 = $286&255;
            $288 = $284 | $287;
            $289 = ($288|0)!=(0);
            $290 = $289&1;
            HEAP8[$83>>0] = $290;
            $291 = ((($107)) + 64|0);
            $292 = HEAP8[$291>>0]|0;
            $293 = $292&255;
            $294 = $293 << 24;
            $295 = ((($107)) + 65|0);
            $296 = HEAP8[$295>>0]|0;
            $297 = $296&255;
            $298 = $297 << 16;
            $299 = $298 | $294;
            $300 = ((($107)) + 66|0);
            $301 = HEAP8[$300>>0]|0;
            $302 = $301&255;
            $303 = $302 << 8;
            $304 = $299 | $303;
            $305 = ((($107)) + 67|0);
            $306 = HEAP8[$305>>0]|0;
            $307 = $306&255;
            $308 = $304 | $307;
            $309 = ($308|0)!=(0);
            $310 = $309&1;
            HEAP8[$84>>0] = $310;
            $312 = $220;
            label = 85;
            break;
           }
           $$idx92$i = ((($$idx14$val$i$i$i)) + 72|0);
           $$idx92$val$i = HEAP32[$$idx92$i>>2]|0;
           $223 = ((($$idx92$val$i)) + 20|0);
           $224 = HEAP32[$223>>2]|0;
           _sqlite3PcacheRelease($$idx92$val$i);
           $225 = ((($224)) + 212|0);
           $226 = HEAP32[$225>>2]|0;
           $$idx$i$i$i100$i = ((($226)) + 12|0);
           $$idx$val$i$i$i101$i = HEAP32[$$idx$i$i$i100$i>>2]|0;
           $227 = ($$idx$val$i$i$i101$i|0)==(0);
           if ($227) {
            $228 = ((($224)) + 16|0);
            $229 = HEAP8[$228>>0]|0;
            L99: do {
             switch ($229<<24>>24) {
             case 0: case 6:  {
              break;
             }
             case 1:  {
              $230 = ((($224)) + 4|0);
              $231 = HEAP8[$230>>0]|0;
              $232 = ($231<<24>>24)==(0);
              if (!($232)) {
               break L99;
              }
              (_pager_end_transaction($224,0,0)|0);
              break;
             }
             default: {
              (_sqlite3PagerRollback($224)|0);
             }
             }
            } while(0);
            _pager_unlock($224);
           }
           HEAP32[$$phi$trans$insert127$i>>2] = $220;
           HEAP32[$81>>2] = $209;
           $233 = HEAP32[$82>>2]|0;
           $234 = ($233|0)==(0|0);
           if (!($234)) {
            $235 = ((($233)) + -4|0);
            HEAP32[$82>>2] = $235;
            $236 = $235;
            $237 = HEAP32[(59264)>>2]|0;
            $238 = ($237>>>0)>($236>>>0);
            do {
             if ($238) {
              $249 = HEAP32[(29352)>>2]|0;
              $250 = (FUNCTION_TABLE_ii[$249 & 127]($235)|0);
              $phi$call$i$i$i$i = $250;
              label = 77;
             } else {
              $239 = HEAP32[(59268)>>2]|0;
              $240 = ($239>>>0)>($236>>>0);
              if ($240) {
               $241 = HEAP32[14979]|0;
               $242 = (($241) + -1)|0;
               HEAP32[14979] = $242;
               $243 = HEAP32[(59276)>>2]|0;
               HEAP32[$235>>2] = $243;
               HEAP32[(59276)>>2] = $235;
               $244 = HEAP32[(59280)>>2]|0;
               $245 = (($244) + 1)|0;
               HEAP32[(59280)>>2] = $245;
               $246 = HEAP32[(59260)>>2]|0;
               $247 = ($245|0)<($246|0);
               $248 = $247&1;
               HEAP32[(59284)>>2] = $248;
               break;
              } else {
               $251 = HEAP32[(29352)>>2]|0;
               $252 = (FUNCTION_TABLE_ii[$251 & 127]($235)|0);
               $phi$call$i$i$i$i = $252;
               label = 77;
               break;
              }
             }
            } while(0);
            do {
             if ((label|0) == 77) {
              label = 0;
              $253 = HEAP32[14980]|0;
              $254 = (($253) - ($phi$call$i$i$i$i))|0;
              HEAP32[14980] = $254;
              $255 = HEAP32[7324]|0;
              $256 = ($255|0)==(0);
              if ($256) {
               $264 = HEAP32[(29344)>>2]|0;
               FUNCTION_TABLE_vi[$264 & 127]($235);
               break;
              } else {
               $257 = HEAP32[(29352)>>2]|0;
               $258 = (FUNCTION_TABLE_ii[$257 & 127]($235)|0);
               $259 = HEAP32[14978]|0;
               $260 = (($259) - ($258))|0;
               HEAP32[14978] = $260;
               $261 = HEAP32[14981]|0;
               $262 = (($261) + -1)|0;
               HEAP32[14981] = $262;
               $263 = HEAP32[(29344)>>2]|0;
               FUNCTION_TABLE_vi[$263 & 127]($235);
               break;
              }
             }
            } while(0);
            HEAP32[$82>>2] = 0;
           }
           $336 = HEAP32[$5>>2]|0;
           $337 = (_sqlite3PagerSetPagesize($336,$81,$219)|0);
           $338 = ($337|0)==(0);
           if (!($338)) {
            $$4$ph = $337;
            label = 135;
            break L45;
           }
          }
          $$pr = HEAP32[$74>>2]|0;
          $340 = $$pr;
         }
        } while(0);
        if ((label|0) == 85) {
         label = 0;
         $311 = $312 << 6;
         $313 = (($311) + -768)|0;
         $314 = (($313>>>0) / 255)&-1;
         $315 = (($314) + 65513)|0;
         $316 = $315&65535;
         HEAP16[$76>>1] = $316;
         $317 = $312 << 5;
         $318 = (($317) + -384)|0;
         $319 = (($318>>>0) / 255)&-1;
         $320 = (($319) + 65513)|0;
         $321 = $320&65535;
         HEAP16[$77>>1] = $321;
         $322 = (($312) + 65501)|0;
         $323 = $322&65535;
         HEAP16[$78>>1] = $323;
         HEAP16[$79>>1] = $321;
         $324 = ($316&65535)>(127);
         $325 = $315&255;
         $spec$select132$i = $324 ? 127 : $325;
         HEAP8[$80>>0] = $spec$select132$i;
         HEAP32[$74>>2] = $105;
         HEAP32[$70>>2] = $spec$select$i;
         $340 = $$idx14$val$i$i$i;
        }
        $339 = ($340|0)==(0|0);
        if (!($339)) {
         $$2$lcssa = 0;
         label = 96;
         break L45;
        }
       }
       if ((label|0) == 35) {
        label = 0;
        $$4$ph = $94;
        label = 135;
        break;
       }
       else if ((label|0) == 82) {
        label = 0;
        HEAP32[$vararg_buffer>>2] = 32306;
        $vararg_ptr1 = ((($vararg_buffer)) + 4|0);
        HEAP32[$vararg_ptr1>>2] = 66365;
        $vararg_ptr2 = ((($vararg_buffer)) + 8|0);
        HEAP32[$vararg_ptr2>>2] = (31517);
        _sqlite3_log(11,32001,$vararg_buffer);
        $$285120$i = 11;
       }
       $$idx91$i = ((($$idx14$val$i$i$i)) + 72|0);
       $$idx91$val$i = HEAP32[$$idx91$i>>2]|0;
       $326 = ((($$idx91$val$i)) + 20|0);
       $327 = HEAP32[$326>>2]|0;
       _sqlite3PcacheRelease($$idx91$val$i);
       $328 = ((($327)) + 212|0);
       $329 = HEAP32[$328>>2]|0;
       $$idx$i$i$i$i = ((($329)) + 12|0);
       $$idx$val$i$i$i$i = HEAP32[$$idx$i$i$i$i>>2]|0;
       $330 = ($$idx$val$i$i$i$i|0)==(0);
       if ($330) {
        $331 = ((($327)) + 16|0);
        $332 = HEAP8[$331>>0]|0;
        switch ($332<<24>>24) {
        case 0: case 6:  {
         break;
        }
        case 1:  {
         $333 = ((($327)) + 4|0);
         $334 = HEAP8[$333>>0]|0;
         $335 = ($334<<24>>24)==(0);
         if ($335) {
          (_pager_end_transaction($327,0,0)|0);
         }
         break;
        }
        default: {
         (_sqlite3PagerRollback($327)|0);
        }
        }
        _pager_unlock($327);
       }
       HEAP32[$74>>2] = 0;
       $$4$ph = $$285120$i;
       label = 135;
      } else {
       $$2$lcssa = $$1100;
       label = 96;
      }
     } while(0);
     L136: do {
      if ((label|0) == 96) {
       label = 0;
       $341 = ($$2$lcssa|0)==(0);
       $or$cond5 = $13 & $341;
       L138: do {
        if ($or$cond5) {
         $342 = HEAP16[$30>>1]|0;
         $343 = $342 & 1;
         $344 = ($343<<16>>16)==(0);
         if (!($344)) {
          $$4$ph = 8;
          label = 135;
          break L136;
         }
         $345 = HEAP32[$5>>2]|0;
         $346 = ((($345)) + 40|0);
         $347 = HEAP32[$346>>2]|0;
         $348 = ($347|0)==(0);
         L141: do {
          if ($348) {
           $349 = HEAP32[$0>>2]|0;
           $$idx = ((($349)) + 80|0);
           $$idx$val = HEAP8[$$idx>>0]|0;
           $350 = ($$idx$val<<24>>24)==(2);
           $351 = $350&1;
           $352 = ((($345)) + 21|0);
           HEAP8[$352>>0] = $351;
           $353 = ((($345)) + 16|0);
           $354 = HEAP8[$353>>0]|0;
           $355 = ($354<<24>>24)==(1);
           if ($355) {
            $356 = ((($345)) + 216|0);
            $357 = HEAP32[$356>>2]|0;
            $358 = ($357|0)==(0|0);
            L144: do {
             if ($358) {
              $416 = ((($345)) + 17|0);
              $417 = HEAP8[$416>>0]|0;
              L146: do {
               switch ($417<<24>>24) {
               case 0: case 1: case 5:  {
                $418 = ((($345)) + 13|0);
                $419 = HEAP8[$418>>0]|0;
                $420 = ($419<<24>>24)==(0);
                if ($420) {
                 $421 = ((($345)) + 60|0);
                 $422 = HEAP32[$421>>2]|0;
                 $423 = HEAP32[$422>>2]|0;
                 $424 = ((($423)) + 28|0);
                 $425 = HEAP32[$424>>2]|0;
                 $426 = (FUNCTION_TABLE_iii[$425 & 255]($422,2)|0);
                 $427 = ($426|0)==(0);
                 if (!($427)) {
                  $$028$i = $426;
                  label = 132;
                  break L141;
                 }
                 $$pre$i32$i = HEAP8[$416>>0]|0;
                 $429 = $$pre$i32$i;
                } else {
                 $429 = $417;
                }
                $428 = ($429<<24>>24)==(5);
                if ($428) {
                 $614 = 5;
                 break L146;
                }
                HEAP8[$416>>0] = 2;
                $614 = 2;
                break;
               }
               default: {
                $614 = $417;
               }
               }
              } while(0);
              if (!($85)) {
               $430 = ((($345)) + 13|0);
               $431 = ((($345)) + 60|0);
               $432 = ((($345)) + 176|0);
               $433 = ((($345)) + 180|0);
               $434 = $614;
               L155: while(1) {
                switch ($434<<24>>24) {
                case 0: case 1: case 2: case 3: case 5:  {
                 break;
                }
                default: {
                 break L144;
                }
                }
                $435 = HEAP8[$430>>0]|0;
                $436 = ($435<<24>>24)==(0);
                if (!($436)) {
                 break;
                }
                $437 = HEAP32[$431>>2]|0;
                $438 = HEAP32[$437>>2]|0;
                $439 = ((($438)) + 28|0);
                $440 = HEAP32[$439>>2]|0;
                $441 = (FUNCTION_TABLE_iii[$440 & 255]($437,4)|0);
                switch ($441|0) {
                case 0:  {
                 break L155;
                 break;
                }
                case 5:  {
                 break;
                }
                default: {
                 $$028$i = $441;
                 label = 132;
                 break L141;
                }
                }
                $442 = HEAP32[$432>>2]|0;
                $443 = HEAP32[$433>>2]|0;
                $444 = (FUNCTION_TABLE_ii[$442 & 127]($443)|0);
                $445 = ($444|0)==(0);
                if ($445) {
                 $$4$ph = 5;
                 label = 135;
                 break L136;
                }
                $$pre44$i = HEAP8[$416>>0]|0;
                $434 = $$pre44$i;
               }
               HEAP8[$416>>0] = 4;
              }
             } else {
              $359 = ((($345)) + 4|0);
              $360 = HEAP8[$359>>0]|0;
              $361 = ($360<<24>>24)==(0);
              do {
               if ($361) {
                $389 = $357;
               } else {
                $362 = ((($357)) + 43|0);
                $363 = HEAP8[$362>>0]|0;
                $364 = ($363<<24>>24)==(0);
                if (!($364)) {
                 $389 = $357;
                 break;
                }
                $365 = ((($345)) + 17|0);
                $366 = HEAP8[$365>>0]|0;
                L166: do {
                 switch ($366<<24>>24) {
                 case 0: case 1: case 2: case 3: case 5:  {
                  $367 = ((($345)) + 13|0);
                  $368 = HEAP8[$367>>0]|0;
                  $369 = ($368<<24>>24)==(0);
                  if (!($369)) {
                   HEAP8[$365>>0] = 4;
                   $$pre$phi49$iZ2D = $362;$379 = $357;
                   label = 108;
                   break L166;
                  }
                  $370 = ((($345)) + 60|0);
                  $371 = HEAP32[$370>>2]|0;
                  $372 = HEAP32[$371>>2]|0;
                  $373 = ((($372)) + 28|0);
                  $374 = HEAP32[$373>>2]|0;
                  $375 = (FUNCTION_TABLE_iii[$374 & 255]($371,4)|0);
                  $376 = ($375|0)==(0);
                  if (!($376)) {
                   $$028$i = $375;
                   label = 132;
                   break L141;
                  }
                  $$pre$pre$i = HEAP32[$356>>2]|0;
                  $$phi$trans$insert$phi$trans$insert$i = ((($$pre$pre$i)) + 43|0);
                  $$pre42$pre$i = HEAP8[$$phi$trans$insert$phi$trans$insert$i>>0]|0;
                  HEAP8[$365>>0] = 4;
                  $377 = ($$pre42$pre$i<<24>>24)==(0);
                  if ($377) {
                   $$pre$phi49$iZ2D = $$phi$trans$insert$phi$trans$insert$i;$379 = $$pre$pre$i;
                   label = 108;
                  } else {
                   $$pre$phi48$iZ2D = $$phi$trans$insert$phi$trans$insert$i;$$pre43$i = $$pre$pre$i;
                  }
                  break;
                 }
                 default: {
                  $$pre$phi49$iZ2D = $362;$379 = $357;
                  label = 108;
                 }
                 }
                } while(0);
                if ((label|0) == 108) {
                 label = 0;
                 $378 = ((($379)) + 40|0);
                 $380 = HEAP16[$378>>1]|0;
                 $381 = $380 << 16 >> 16;
                 $382 = (($381) + 3)|0;
                 $383 = ((($379)) + 4|0);
                 $384 = HEAP32[$383>>2]|0;
                 $385 = HEAP32[$384>>2]|0;
                 $386 = ((($385)) + 56|0);
                 $387 = HEAP32[$386>>2]|0;
                 (FUNCTION_TABLE_iiiii[$387 & 127]($384,$382,1,5)|0);
                 $$pre43$pre$i = HEAP32[$356>>2]|0;
                 $$pre$phi48$iZ2D = $$pre$phi49$iZ2D;$$pre43$i = $$pre43$pre$i;
                }
                HEAP8[$$pre$phi48$iZ2D>>0] = 1;
                $389 = $$pre43$i;
               }
              } while(0);
              $388 = ((($389)) + 46|0);
              $390 = HEAP8[$388>>0]|0;
              $391 = ($390<<24>>24)==(0);
              if (!($391)) {
               $$4$ph = 8;
               label = 135;
               break L136;
              }
              $392 = ((($389)) + 43|0);
              $393 = HEAP8[$392>>0]|0;
              $394 = ($393<<24>>24)==(0);
              if ($394) {
               $395 = ((($389)) + 4|0);
               $396 = HEAP32[$395>>2]|0;
               $397 = HEAP32[$396>>2]|0;
               $398 = ((($397)) + 56|0);
               $399 = HEAP32[$398>>2]|0;
               $400 = (FUNCTION_TABLE_iiiii[$399 & 127]($396,0,1,10)|0);
               $401 = ($400|0)==(0);
               if (!($401)) {
                $$028$i = $400;
                label = 132;
                break L141;
               }
              }
              $402 = ((($389)) + 44|0);
              HEAP8[$402>>0] = 1;
              $403 = ((($389)) + 52|0);
              $404 = ((($389)) + 32|0);
              $405 = HEAP32[$404>>2]|0;
              $406 = HEAP32[$405>>2]|0;
              $407 = (_memcmp($403,$406,48)|0);
              $408 = ($407|0)==(0);
              if ($408) {
               break;
              }
              $409 = HEAP8[$392>>0]|0;
              $410 = ($409<<24>>24)==(0);
              if ($410) {
               $411 = ((($389)) + 4|0);
               $412 = HEAP32[$411>>2]|0;
               $413 = HEAP32[$412>>2]|0;
               $414 = ((($413)) + 56|0);
               $415 = HEAP32[$414>>2]|0;
               (FUNCTION_TABLE_iiiii[$415 & 127]($412,0,1,9)|0);
              }
              HEAP8[$402>>0] = 0;
              label = 134;
              break L141;
             }
            } while(0);
            HEAP8[$353>>0] = 2;
            $446 = ((($345)) + 24|0);
            $447 = HEAP32[$446>>2]|0;
            $448 = ((($345)) + 36|0);
            HEAP32[$448>>2] = $447;
            $449 = ((($345)) + 32|0);
            HEAP32[$449>>2] = $447;
            $450 = ((($345)) + 28|0);
            HEAP32[$450>>2] = $447;
            $451 = ((($345)) + 72|0);
            $452 = $451;
            $453 = $452;
            HEAP32[$453>>2] = 0;
            $454 = (($452) + 4)|0;
            $455 = $454;
            HEAP32[$455>>2] = 0;
            label = 133;
           } else {
            label = 133;
           }
          } else {
           $$028$i = $347;
           label = 132;
          }
         } while(0);
         if ((label|0) == 132) {
          label = 0;
          switch ($$028$i|0) {
          case 0:  {
           label = 133;
           break;
          }
          case 517:  {
           label = 134;
           break;
          }
          default: {
           $$4 = $$028$i;
           break L138;
          }
          }
         }
         if ((label|0) == 133) {
          label = 0;
          $456 = (_newDatabase($5)|0);
          $$4 = $456;
          break;
         }
         else if ((label|0) == 134) {
          label = 0;
          $457 = HEAP8[$75>>0]|0;
          $458 = ($457<<24>>24)==(0);
          $spec$select = $458 ? 5 : 517;
          $$4$ph = $spec$select;
          label = 135;
          break L136;
         }
        } else {
         $$4 = $$2$lcssa;
        }
       } while(0);
       $460 = ($$4|0)==(0);
       if ($460) {
        $$4127129 = 0;
        break L43;
       } else {
        $$4125 = $$4;$615 = 0;
       }
      }
     } while(0);
     if ((label|0) == 135) {
      label = 0;
      $459 = ($$4$ph|0)==(0);
      $$4125 = $$4$ph;$615 = $459;
     }
     $461 = HEAP8[$75>>0]|0;
     $462 = ($461<<24>>24)==(0);
     if ($462) {
      $463 = HEAP32[$74>>2]|0;
      $464 = ($463|0)==(0|0);
      if (!($464)) {
       HEAP32[$74>>2] = 0;
       $$idx$i108 = ((($463)) + 72|0);
       $$idx$val$i109 = HEAP32[$$idx$i108>>2]|0;
       $465 = ((($$idx$val$i109)) + 20|0);
       $466 = HEAP32[$465>>2]|0;
       _sqlite3PcacheRelease($$idx$val$i109);
       $467 = ((($466)) + 212|0);
       $468 = HEAP32[$467>>2]|0;
       $$idx$i$i$i110 = ((($468)) + 12|0);
       $$idx$val$i$i$i = HEAP32[$$idx$i$i$i110>>2]|0;
       $469 = ($$idx$val$i$i$i|0)==(0);
       if ($469) {
        $470 = ((($466)) + 16|0);
        $471 = HEAP8[$470>>0]|0;
        switch ($471<<24>>24) {
        case 0: case 6:  {
         break;
        }
        case 1:  {
         $472 = ((($466)) + 4|0);
         $473 = HEAP8[$472>>0]|0;
         $474 = ($473<<24>>24)==(0);
         if ($474) {
          (_pager_end_transaction($466,0,0)|0);
         }
         break;
        }
        default: {
         (_sqlite3PagerRollback($466)|0);
        }
        }
        _pager_unlock($466);
       }
      }
     }
     $475 = $$4125 & 255;
     $476 = ($475|0)==(5);
     if (!($476)) {
      label = 156;
      break;
     }
     $477 = HEAP8[$75>>0]|0;
     $478 = ($477<<24>>24)==(0);
     if (!($478)) {
      label = 156;
      break;
     }
     $479 = HEAP32[$7>>2]|0;
     $480 = HEAP32[$5>>2]|0;
     $$idx$i111 = ((($480)) + 60|0);
     $$idx$val$i112 = HEAP32[$$idx$i111>>2]|0;
     $481 = ((($479)) + 396|0);
     $482 = HEAP32[$481>>2]|0;
     $483 = ($482|0)==(0|0);
     if ($483) {
      label = 154;
      break;
     }
     $484 = ((($479)) + 404|0);
     $485 = HEAP32[$484>>2]|0;
     $486 = ($485|0)<(0);
     if ($486) {
      label = 154;
      break;
     }
     $487 = ((($479)) + 408|0);
     $488 = HEAP8[$487>>0]|0;
     $489 = ($488<<24>>24)==(0);
     if ($489) {
      $493 = ((($479)) + 400|0);
      $494 = HEAP32[$493>>2]|0;
      $495 = (FUNCTION_TABLE_iii[$482 & 255]($494,$485)|0);
      $$017$i$i = $495;
     } else {
      $490 = ((($479)) + 400|0);
      $491 = HEAP32[$490>>2]|0;
      $492 = (FUNCTION_TABLE_iiii[$482 & 127]($491,$485,$$idx$val$i112)|0);
      $$017$i$i = $492;
     }
     $496 = ($$017$i$i|0)==(0);
     if ($496) {
      label = 153;
      break;
     }
     $497 = HEAP32[$484>>2]|0;
     $498 = (($497) + 1)|0;
     HEAP32[$484>>2] = $498;
     $$1100 = $$4125;
    }
    if ((label|0) == 153) {
     HEAP32[$484>>2] = -1;
     if ($615) {
      $$4127129 = $$4125;
     } else {
      $$0 = $$4125;
      STACKTOP = sp;return ($$0|0);
     }
    }
    else if ((label|0) == 154) {
     if ($615) {
      $$4127129 = $$4125;
     } else {
      $$0 = $$4125;
      STACKTOP = sp;return ($$0|0);
     }
    }
    else if ((label|0) == 156) {
     if ($615) {
      $$4127129 = $$4125;
     } else {
      $$0 = $$4125;
      STACKTOP = sp;return ($$0|0);
     }
    }
    $499 = HEAP8[$8>>0]|0;
    $500 = ($499<<24>>24)==(0);
    if ($500) {
     $501 = ((($5)) + 40|0);
     $502 = HEAP32[$501>>2]|0;
     $503 = (($502) + 1)|0;
     HEAP32[$501>>2] = $503;
     $504 = HEAP8[$49>>0]|0;
     $505 = ($504<<24>>24)==(0);
     if (!($505)) {
      $506 = ((($0)) + 32|0);
      $507 = ((($0)) + 40|0);
      HEAP8[$507>>0] = 1;
      $508 = ((($5)) + 72|0);
      $509 = HEAP32[$508>>2]|0;
      $510 = ((($0)) + 44|0);
      HEAP32[$510>>2] = $509;
      HEAP32[$508>>2] = $506;
     }
    }
    $511 = $13 ? 2 : 1;
    HEAP8[$8>>0] = $511;
    $512 = HEAP8[$75>>0]|0;
    $513 = ($511&255)>($512&255);
    if ($513) {
     HEAP8[$75>>0] = $511;
    }
    if ($13) {
     $514 = HEAP32[$74>>2]|0;
     $515 = ((($5)) + 76|0);
     HEAP32[$515>>2] = $0;
     $516 = HEAP16[$30>>1]|0;
     $517 = $516 & -65;
     $518 = ($1|0)>(1);
     $519 = $516 | 64;
     $spec$select102 = $518 ? $519 : $517;
     HEAP16[$30>>1] = $spec$select102;
     $520 = HEAP32[$70>>2]|0;
     $521 = ((($514)) + 56|0);
     $522 = HEAP32[$521>>2]|0;
     $523 = ((($522)) + 28|0);
     $524 = HEAP8[$523>>0]|0;
     $525 = $524&255;
     $526 = $525 << 24;
     $527 = ((($522)) + 29|0);
     $528 = HEAP8[$527>>0]|0;
     $529 = $528&255;
     $530 = $529 << 16;
     $531 = $530 | $526;
     $532 = ((($522)) + 30|0);
     $533 = HEAP8[$532>>0]|0;
     $534 = $533&255;
     $535 = $534 << 8;
     $536 = $531 | $535;
     $537 = ((($522)) + 31|0);
     $538 = HEAP8[$537>>0]|0;
     $539 = $538&255;
     $540 = $536 | $539;
     $541 = ($520|0)==($540|0);
     if (!($541)) {
      $542 = ((($514)) + 72|0);
      $543 = HEAP32[$542>>2]|0;
      $544 = ((($543)) + 20|0);
      $545 = HEAP32[$544>>2]|0;
      $546 = ((($543)) + 28|0);
      $547 = HEAP16[$546>>1]|0;
      $548 = $547 & 4;
      $549 = ($548<<16>>16)==(0);
      if ($549) {
       label = 168;
      } else {
       $550 = ((($545)) + 24|0);
       $551 = HEAP32[$550>>2]|0;
       $552 = ((($543)) + 24|0);
       $553 = HEAP32[$552>>2]|0;
       $554 = ($551>>>0)<($553>>>0);
       if ($554) {
        label = 168;
       } else {
        $555 = ((($545)) + 96|0);
        $556 = HEAP32[$555>>2]|0;
        $557 = ($556|0)==(0);
        if ($557) {
         $$pre$phi195Z2D = $527;$$pre$phi197Z2D = $532;$$pre$phi199Z2D = $537;$$pre$phiZ2D = $523;$571 = $520;
        } else {
         $558 = (_subjournalPageIfRequired($543)|0);
         $$0$i104 = $558;
         label = 172;
        }
       }
      }
      do {
       if ((label|0) == 168) {
        $559 = ((($545)) + 40|0);
        $560 = HEAP32[$559>>2]|0;
        $561 = ($560|0)==(0);
        if (!($561)) {
         $$0 = $560;
         STACKTOP = sp;return ($$0|0);
        }
        $562 = ((($545)) + 148|0);
        $563 = HEAP32[$562>>2]|0;
        $564 = ((($545)) + 152|0);
        $565 = HEAP32[$564>>2]|0;
        $566 = ($563>>>0)>($565>>>0);
        if ($566) {
         $567 = (_pagerWriteLargeSector($543)|0);
         $$0$i104 = $567;
         label = 172;
         break;
        } else {
         $568 = (_pager_write($543)|0);
         $$0$i104 = $568;
         label = 172;
         break;
        }
       }
      } while(0);
      do {
       if ((label|0) == 172) {
        $569 = ($$0$i104|0)==(0);
        if ($569) {
         $$pre = HEAP32[$521>>2]|0;
         $$pre192 = HEAP32[$70>>2]|0;
         $$pre193 = ((($$pre)) + 28|0);
         $$pre194 = ((($$pre)) + 29|0);
         $$pre196 = ((($$pre)) + 30|0);
         $$pre198 = ((($$pre)) + 31|0);
         $$pre$phi195Z2D = $$pre194;$$pre$phi197Z2D = $$pre196;$$pre$phi199Z2D = $$pre198;$$pre$phiZ2D = $$pre193;$571 = $$pre192;
         break;
        } else {
         $$0 = $$0$i104;
         STACKTOP = sp;return ($$0|0);
        }
       }
      } while(0);
      $570 = $571 >>> 24;
      $572 = $570&255;
      HEAP8[$$pre$phiZ2D>>0] = $572;
      $573 = $571 >>> 16;
      $574 = $573&255;
      HEAP8[$$pre$phi195Z2D>>0] = $574;
      $575 = $571 >>> 8;
      $576 = $575&255;
      HEAP8[$$pre$phi197Z2D>>0] = $576;
      $577 = $571&255;
      HEAP8[$$pre$phi199Z2D>>0] = $577;
      break;
     }
    }
    $578 = ($$4127129|0)==(0);
    if (!($578)) {
     $$0 = $$4127129;
     STACKTOP = sp;return ($$0|0);
    }
   }
  }
 } while(0);
 $579 = ($2|0)==(0|0);
 if (!($579)) {
  $580 = ((($5)) + 12|0);
  $581 = HEAP32[$580>>2]|0;
  $582 = ((($581)) + 56|0);
  $583 = HEAP32[$582>>2]|0;
  $584 = ((($583)) + 40|0);
  $585 = HEAP8[$584>>0]|0;
  $586 = $585&255;
  $587 = $586 << 24;
  $588 = ((($583)) + 41|0);
  $589 = HEAP8[$588>>0]|0;
  $590 = $589&255;
  $591 = $590 << 16;
  $592 = $591 | $587;
  $593 = ((($583)) + 42|0);
  $594 = HEAP8[$593>>0]|0;
  $595 = $594&255;
  $596 = $595 << 8;
  $597 = $592 | $596;
  $598 = ((($583)) + 43|0);
  $599 = HEAP8[$598>>0]|0;
  $600 = $599&255;
  $601 = $597 | $600;
  HEAP32[$2>>2] = $601;
 }
 $602 = ($1|0)==(0);
 if ($602) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 $603 = HEAP32[$5>>2]|0;
 $604 = HEAP32[$0>>2]|0;
 $605 = ((($604)) + 452|0);
 $606 = HEAP32[$605>>2]|0;
 $607 = ((($603)) + 96|0);
 $608 = HEAP32[$607>>2]|0;
 $609 = ($608|0)<($606|0);
 if (!($609)) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 $610 = ((($603)) + 6|0);
 $611 = HEAP8[$610>>0]|0;
 $612 = ($611<<24>>24)==(0);
 if ($612) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 $613 = (_pagerOpenSavepoint($603,$606)|0);
 $$0 = $613;
 STACKTOP = sp;return ($$0|0);
}
function _backupOnePage($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $$0 = 0, $$0$i = 0, $$1$lcssa = 0, $$2 = 0, $$3 = 0, $$idx = 0, $$idx$val = 0, $$idx$val$idx$val = 0, $$idx64$val$idx = 0, $$idx64$val$idx$val = 0, $$idx65 = 0, $$idx65$val = 0, $$idx65$val$idx = 0, $$idx65$val$idx$val = 0, $$idx66 = 0, $$idx66$val = 0, $$idx67 = 0, $$idx67$val = 0, $$idx68 = 0, $$idx68$val = 0;
 var $$idx69 = 0, $$idx69$val = 0, $$idx69$val$idx = 0, $$idx69$val$idx$val = 0, $$phi$trans$insert = 0, $$phi$trans$insert75 = 0, $$pre = 0, $$pre74 = 0, $$pre76 = 0, $$pre77 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0;
 var $109 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0;
 var $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0;
 var $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0;
 var $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $or$cond = 0, $or$cond63 = 0, $or$cond6372 = 0, $spec$select = 0;
 var label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $4 = sp;
 $5 = ((($0)) + 4|0);
 $6 = HEAP32[$5>>2]|0;
 $$idx = ((($6)) + 4|0);
 $$idx$val = HEAP32[$$idx>>2]|0;
 $$idx$val$idx$val = HEAP32[$$idx$val>>2]|0;
 $7 = ((($0)) + 24|0);
 $8 = HEAP32[$7>>2]|0;
 $$idx65 = ((($8)) + 4|0);
 $$idx65$val = HEAP32[$$idx65>>2]|0;
 $$idx65$val$idx = ((($$idx65$val)) + 32|0);
 $$idx65$val$idx$val = HEAP32[$$idx65$val$idx>>2]|0;
 $$idx64$val$idx = ((($$idx$val)) + 32|0);
 $$idx64$val$idx$val = HEAP32[$$idx64$val$idx>>2]|0;
 $9 = ($$idx65$val$idx$val|0)<($$idx64$val$idx$val|0);
 $10 = $9 ? $$idx65$val$idx$val : $$idx64$val$idx$val;
 $11 = ($$idx65$val$idx$val|0)<(0);
 $12 = $11 << 31 >> 31;
 $13 = (___muldi3(($$idx65$val$idx$val|0),($12|0),($1|0),0)|0);
 $14 = (getTempRet0() | 0);
 $15 = ($$idx65$val$idx$val|0)==($$idx64$val$idx$val|0);
 if ($15) {
  $$0 = 0;
 } else {
  $$idx66 = ((($$idx$val$idx$val)) + 12|0);
  $$idx66$val = HEAP8[$$idx66>>0]|0;
  $16 = ($$idx66$val<<24>>24)==(0);
  $spec$select = $16 ? 0 : 8;
  $$0 = $spec$select;
 }
 $17 = ($$0|0)==(0);
 $18 = ($$idx65$val$idx$val|0)>(0);
 $or$cond6372 = $18 & $17;
 if (!($or$cond6372)) {
  $$1$lcssa = $$0;
  STACKTOP = sp;return ($$1$lcssa|0);
 }
 $19 = (_i64Subtract(($13|0),($14|0),($$idx65$val$idx$val|0),($12|0))|0);
 $20 = (getTempRet0() | 0);
 $21 = ($$idx64$val$idx$val|0)<(0);
 $22 = $21 << 31 >> 31;
 $23 = ((($$idx$val$idx$val)) + 204|0);
 $24 = ($3|0)==(0);
 $25 = $19;$26 = $20;$30 = $$idx64$val$idx$val;
 while(1) {
  HEAP32[$4>>2] = 0;
  $27 = (___divdi3(($25|0),($26|0),($$idx64$val$idx$val|0),($22|0))|0);
  $28 = (getTempRet0() | 0);
  $29 = (1073741824 / ($30>>>0))&-1;
  $31 = ($29|0)==($27|0);
  do {
   if ($31) {
    $$3 = 0;
   } else {
    $32 = (($27) + 1)|0;
    $33 = HEAP32[$23>>2]|0;
    $34 = (FUNCTION_TABLE_iiiii[$33 & 127]($$idx$val$idx$val,$32,$4,0)|0);
    $35 = ($34|0)==(0);
    L11: do {
     if ($35) {
      $36 = HEAP32[$4>>2]|0;
      $37 = ((($36)) + 20|0);
      $38 = HEAP32[$37>>2]|0;
      $39 = ((($36)) + 28|0);
      $40 = HEAP16[$39>>1]|0;
      $41 = $40 & 4;
      $42 = ($41<<16>>16)==(0);
      if ($42) {
       label = 11;
      } else {
       $43 = ((($38)) + 24|0);
       $44 = HEAP32[$43>>2]|0;
       $45 = ((($36)) + 24|0);
       $46 = HEAP32[$45>>2]|0;
       $47 = ($44>>>0)<($46>>>0);
       if ($47) {
        label = 11;
       } else {
        $48 = ((($38)) + 96|0);
        $49 = HEAP32[$48>>2]|0;
        $50 = ($49|0)==(0);
        if ($50) {
         $66 = $36;
        } else {
         $51 = (_subjournalPageIfRequired($36)|0);
         $$0$i = $51;
         label = 15;
        }
       }
      }
      do {
       if ((label|0) == 11) {
        label = 0;
        $52 = ((($38)) + 40|0);
        $53 = HEAP32[$52>>2]|0;
        $54 = ($53|0)==(0);
        if (!($54)) {
         $$2 = $53;
         break L11;
        }
        $55 = ((($38)) + 148|0);
        $56 = HEAP32[$55>>2]|0;
        $57 = ((($38)) + 152|0);
        $58 = HEAP32[$57>>2]|0;
        $59 = ($56>>>0)>($58>>>0);
        if ($59) {
         $60 = (_pagerWriteLargeSector($36)|0);
         $$0$i = $60;
         label = 15;
         break;
        } else {
         $61 = (_pager_write($36)|0);
         $$0$i = $61;
         label = 15;
         break;
        }
       }
      } while(0);
      if ((label|0) == 15) {
       label = 0;
       $62 = ($$0$i|0)==(0);
       if (!($62)) {
        $$2 = $$0$i;
        break;
       }
       $$pre77 = HEAP32[$4>>2]|0;
       $66 = $$pre77;
      }
      $63 = (___remdi3(($25|0),($26|0),($$idx65$val$idx$val|0),($12|0))|0);
      $64 = (getTempRet0() | 0);
      $65 = (($2) + ($63)|0);
      $$idx67 = ((($66)) + 4|0);
      $$idx67$val = HEAP32[$$idx67>>2]|0;
      $67 = (___muldi3(($27|0),($28|0),($$idx64$val$idx$val|0),($22|0))|0);
      $68 = (getTempRet0() | 0);
      $69 = (_i64Subtract(($25|0),($26|0),($67|0),($68|0))|0);
      $70 = (getTempRet0() | 0);
      $71 = (($$idx67$val) + ($69)|0);
      _memcpy(($71|0),($65|0),($10|0))|0;
      $72 = HEAP32[$4>>2]|0;
      $$idx68 = ((($72)) + 8|0);
      $$idx68$val = HEAP32[$$idx68>>2]|0;
      HEAP8[$$idx68$val>>0] = 0;
      $73 = ($25|0)==(0);
      $74 = ($26|0)==(0);
      $75 = $73 & $74;
      $or$cond = $24 & $75;
      if ($or$cond) {
       $76 = ((($71)) + 28|0);
       $77 = HEAP32[$7>>2]|0;
       $$idx69 = ((($77)) + 4|0);
       $$idx69$val = HEAP32[$$idx69>>2]|0;
       $$idx69$val$idx = ((($$idx69$val)) + 44|0);
       $$idx69$val$idx$val = HEAP32[$$idx69$val$idx>>2]|0;
       $78 = $$idx69$val$idx$val >>> 24;
       $79 = $78&255;
       HEAP8[$76>>0] = $79;
       $80 = $$idx69$val$idx$val >>> 16;
       $81 = $80&255;
       $82 = ((($76)) + 1|0);
       HEAP8[$82>>0] = $81;
       $83 = $$idx69$val$idx$val >>> 8;
       $84 = $83&255;
       $85 = ((($76)) + 2|0);
       HEAP8[$85>>0] = $84;
       $86 = $$idx69$val$idx$val&255;
       $87 = ((($76)) + 3|0);
       HEAP8[$87>>0] = $86;
       $$2 = 0;
      } else {
       $$2 = 0;
      }
     } else {
      $$2 = $34;
     }
    } while(0);
    $88 = HEAP32[$4>>2]|0;
    $89 = ($88|0)==(0|0);
    if ($89) {
     $$3 = $$2;
    } else {
     $90 = ((($88)) + 28|0);
     $91 = HEAP16[$90>>1]|0;
     $92 = $91 & 32;
     $93 = ($92<<16>>16)==(0);
     if ($93) {
      _sqlite3PcacheRelease($88);
      $$3 = $$2;
      break;
     } else {
      $94 = ((($88)) + 20|0);
      $95 = HEAP32[$94>>2]|0;
      $96 = ((($95)) + 120|0);
      $97 = HEAP32[$96>>2]|0;
      $98 = (($97) + -1)|0;
      HEAP32[$96>>2] = $98;
      $99 = ((($95)) + 136|0);
      $100 = HEAP32[$99>>2]|0;
      $101 = ((($88)) + 16|0);
      HEAP32[$101>>2] = $100;
      HEAP32[$99>>2] = $88;
      $$3 = $$2;
      break;
     }
    }
   }
  } while(0);
  $102 = (_i64Add(($25|0),($26|0),($$idx64$val$idx$val|0),($22|0))|0);
  $103 = (getTempRet0() | 0);
  $104 = ($$3|0)==(0);
  $105 = ($103|0)<($14|0);
  $106 = ($102>>>0)<($13>>>0);
  $107 = ($103|0)==($14|0);
  $108 = $107 & $106;
  $109 = $105 | $108;
  $or$cond63 = $109 & $104;
  if (!($or$cond63)) {
   $$1$lcssa = $$3;
   break;
  }
  $$pre = HEAP32[$5>>2]|0;
  $$phi$trans$insert = ((($$pre)) + 4|0);
  $$pre74 = HEAP32[$$phi$trans$insert>>2]|0;
  $$phi$trans$insert75 = ((($$pre74)) + 32|0);
  $$pre76 = HEAP32[$$phi$trans$insert75>>2]|0;
  $25 = $102;$26 = $103;$30 = $$pre76;
 }
 STACKTOP = sp;return ($$1$lcssa|0);
}
function _sqlite3ResetAllSchemasOfConnection($0) {
 $0 = $0|0;
 var $$010$i = 0, $$017 = 0, $$pre = 0, $$pre$phiZ2D = 0, $$pre18 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0;
 var $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $exitcond = 0, label = 0;
 var sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 20|0);
 $2 = HEAP32[$1>>2]|0;
 $3 = ($2|0)>(0);
 if ($3) {
  $4 = ((($0)) + 16|0);
  $5 = HEAP32[$4>>2]|0;
  $$010$i = 0;
  while(1) {
   $6 = (((($5) + ($$010$i<<4)|0)) + 4|0);
   $7 = HEAP32[$6>>2]|0;
   $8 = ($7|0)==(0|0);
   if (!($8)) {
    $9 = HEAP32[$7>>2]|0;
    $10 = ((($7)) + 4|0);
    $11 = HEAP32[$10>>2]|0;
    $12 = ((($11)) + 4|0);
    HEAP32[$12>>2] = $9;
   }
   $13 = (($$010$i) + 1)|0;
   $exitcond = ($13|0)==($2|0);
   if ($exitcond) {
    break;
   } else {
    $$010$i = $13;
   }
  }
  if ($3) {
   $14 = ((($0)) + 16|0);
   $15 = ((($0)) + 56|0);
   $$017 = 0;$33 = $2;
   while(1) {
    $16 = HEAP32[$14>>2]|0;
    $17 = (((($16) + ($$017<<4)|0)) + 12|0);
    $18 = HEAP32[$17>>2]|0;
    $19 = ($18|0)==(0|0);
    do {
     if ($19) {
      $27 = $33;
     } else {
      $20 = HEAP32[$15>>2]|0;
      $21 = ($20|0)==(0);
      if ($21) {
       _sqlite3SchemaClear($18);
       $$pre = HEAP32[$1>>2]|0;
       $27 = $$pre;
       break;
      } else {
       $22 = ((($18)) + 78|0);
       $23 = HEAP16[$22>>1]|0;
       $24 = $23 | 8;
       HEAP16[$22>>1] = $24;
       $27 = $33;
       break;
      }
     }
    } while(0);
    $25 = (($$017) + 1)|0;
    $26 = ($25|0)<($27|0);
    if ($26) {
     $$017 = $25;$33 = $27;
    } else {
     $$pre$phiZ2D = $15;
     break;
    }
   }
  } else {
   label = 7;
  }
 } else {
  label = 7;
 }
 if ((label|0) == 7) {
  $$pre18 = ((($0)) + 56|0);
  $$pre$phiZ2D = $$pre18;
 }
 $28 = ((($0)) + 24|0);
 $29 = HEAP32[$28>>2]|0;
 $30 = $29 & -18;
 HEAP32[$28>>2] = $30;
 _sqlite3VtabUnlockList($0);
 $31 = HEAP32[$$pre$phiZ2D>>2]|0;
 $32 = ($31|0)==(0);
 if (!($32)) {
  return;
 }
 _sqlite3CollapseDatabaseArray($0);
 return;
}
function _sqlite3PagerCommitPhaseOne($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$0 = 0, $$0$i$i = 0, $$0$i$i$i = 0, $$0$i56 = 0, $$0$i68 = 0, $$0$lcssa$i = 0, $$01$i = 0, $$01214$i$i = 0, $$013$i$i = 0, $$015$i$i = 0, $$047$i$ph = 0, $$048$lcssa$i = 0, $$04850$i = 0, $$05$i = 0, $$05072 = 0, $$051$i = 0, $$1$i70 = 0, $$15374 = 0, $$5 = 0, $$phi$trans$insert$i = 0;
 var $$phi$trans$insert$i58 = 0, $$pr = 0, $$pre$i = 0, $$pre33$i$i$i = 0, $$pre33$i$i$i63 = 0, $$pre4$i = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0;
 var $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0;
 var $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0;
 var $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0;
 var $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0;
 var $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0;
 var $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0;
 var $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0;
 var $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0;
 var $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0;
 var $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0;
 var $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0;
 var $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0;
 var $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0;
 var $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0;
 var $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0;
 var $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0;
 var $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0;
 var $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0;
 var $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0;
 var $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0;
 var $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0;
 var $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0, $5 = 0, $50 = 0, $500 = 0, $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $51 = 0;
 var $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0;
 var $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0;
 var $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $phitmp$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp + 8|0;
 $4 = sp;
 $5 = ((($0)) + 40|0);
 $6 = HEAP32[$5>>2]|0;
 $7 = ($6|0)==(0);
 if (!($7)) {
  $$0 = $6;
  STACKTOP = sp;return ($$0|0);
 }
 $8 = HEAP32[(29556)>>2]|0;
 $9 = ($8|0)==(0|0);
 if (!($9)) {
  $10 = (FUNCTION_TABLE_ii[$8 & 127](400)|0);
  $11 = ($10|0)==(0);
  if (!($11)) {
   $$0 = 10;
   STACKTOP = sp;return ($$0|0);
  }
 }
 $12 = ((($0)) + 16|0);
 $13 = HEAP8[$12>>0]|0;
 $14 = ($13&255)<(3);
 if ($14) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 $15 = ((($0)) + 12|0);
 $16 = HEAP8[$15>>0]|0;
 $17 = ($16<<24>>24)==(0);
 do {
  if ($17) {
   label = 18;
  } else {
   $18 = ((($0)) + 60|0);
   $19 = HEAP32[$18>>2]|0;
   $20 = HEAP32[$19>>2]|0;
   $21 = ($20|0)==(0|0);
   if (!($21)) {
    $22 = ((($0)) + 212|0);
    $23 = HEAP32[$22>>2]|0;
    $24 = ((($23)) + 16|0);
    $25 = HEAP32[$24>>2]|0;
    $26 = ($25|0)>(-1);
    if ($26) {
     $$0$i$i$i = $25;
    } else {
     $27 = ($25|0)<(0);
     $28 = $27 << 31 >> 31;
     $29 = (___muldi3(($25|0),($28|0),-1024,-1)|0);
     $30 = (getTempRet0() | 0);
     $31 = ((($23)) + 24|0);
     $32 = HEAP32[$31>>2]|0;
     $33 = ((($23)) + 28|0);
     $34 = HEAP32[$33>>2]|0;
     $35 = (($34) + ($32))|0;
     $36 = ($35|0)<(0);
     $37 = $36 << 31 >> 31;
     $38 = (___divdi3(($29|0),($30|0),($35|0),($37|0))|0);
     $39 = (getTempRet0() | 0);
     $$0$i$i$i = $38;
    }
    $$013$i$i = HEAP32[$23>>2]|0;
    $40 = ($$013$i$i|0)==(0|0);
    if ($40) {
     $49 = 0;$50 = 0;
    } else {
     $$01214$i$i = 0;$$015$i$i = $$013$i$i;
     while(1) {
      $41 = (($$01214$i$i) + 1)|0;
      $42 = ((($$015$i$i)) + 32|0);
      $$0$i$i = HEAP32[$42>>2]|0;
      $43 = ($$0$i$i|0)==(0|0);
      if ($43) {
       break;
      } else {
       $$01214$i$i = $41;$$015$i$i = $$0$i$i;
      }
     }
     $44 = (___muldi3(($41|0),0,100,0)|0);
     $45 = (getTempRet0() | 0);
     $49 = $44;$50 = $45;
    }
    $46 = ($$0$i$i$i|0)==(0);
    if (!($46)) {
     $47 = ($$0$i$i$i|0)<(0);
     $48 = $47 << 31 >> 31;
     $51 = (___divdi3(($49|0),($50|0),($$0$i$i$i|0),($48|0))|0);
     $52 = (getTempRet0() | 0);
     $phitmp$i = ($51|0)<(25);
     if (!($phitmp$i)) {
      label = 18;
      break;
     }
    }
   }
   $53 = ((($0)) + 88|0);
   $54 = HEAP32[$53>>2]|0;
   $55 = ($54|0)==(0|0);
   if (!($55)) {
    $$05$i = $54;
    while(1) {
     $56 = ((($$05$i)) + 16|0);
     HEAP32[$56>>2] = 1;
     $57 = ((($$05$i)) + 44|0);
     $58 = HEAP32[$57>>2]|0;
     $59 = ($58|0)==(0|0);
     if ($59) {
      break;
     } else {
      $$05$i = $58;
     }
    }
   }
  }
 } while(0);
 do {
  if ((label|0) == 18) {
   $60 = ((($0)) + 216|0);
   $61 = HEAP32[$60>>2]|0;
   $62 = ($61|0)==(0|0);
   if ($62) {
    $127 = ((($0)) + 18|0);
    $128 = HEAP8[$127>>0]|0;
    $129 = ($128<<24>>24)==(0);
    if ($129) {
     $130 = ((($0)) + 24|0);
     $131 = HEAP32[$130>>2]|0;
     $132 = ($131|0)==(0);
     if (!($132)) {
      $133 = ((($0)) + 204|0);
      $134 = HEAP32[$133>>2]|0;
      $135 = (FUNCTION_TABLE_iiiii[$134 & 127]($0,1,$3,0)|0);
      $136 = ($135|0)==(0);
      L36: do {
       if ($136) {
        $137 = HEAP32[$3>>2]|0;
        $138 = ((($137)) + 20|0);
        $139 = HEAP32[$138>>2]|0;
        $140 = ((($137)) + 28|0);
        $141 = HEAP16[$140>>1]|0;
        $142 = $141 & 4;
        $143 = ($142<<16>>16)==(0);
        if ($143) {
         label = 49;
        } else {
         $144 = ((($139)) + 24|0);
         $145 = HEAP32[$144>>2]|0;
         $146 = ((($137)) + 24|0);
         $147 = HEAP32[$146>>2]|0;
         $148 = ($145>>>0)<($147>>>0);
         if ($148) {
          label = 49;
         } else {
          $149 = ((($139)) + 96|0);
          $150 = HEAP32[$149>>2]|0;
          $151 = ($150|0)==(0);
          if ($151) {
           $165 = $139;$185 = $137;
          } else {
           $152 = (_subjournalPageIfRequired($137)|0);
           $$0$i56 = $152;
           label = 53;
          }
         }
        }
        do {
         if ((label|0) == 49) {
          $153 = ((($139)) + 40|0);
          $154 = HEAP32[$153>>2]|0;
          $155 = ($154|0)==(0);
          if (!($155)) {
           $$01$i = $154;
           break L36;
          }
          $156 = ((($139)) + 148|0);
          $157 = HEAP32[$156>>2]|0;
          $158 = ((($139)) + 152|0);
          $159 = HEAP32[$158>>2]|0;
          $160 = ($157>>>0)>($159>>>0);
          if ($160) {
           $161 = (_pagerWriteLargeSector($137)|0);
           $$0$i56 = $161;
           label = 53;
           break;
          } else {
           $162 = (_pager_write($137)|0);
           $$0$i56 = $162;
           label = 53;
           break;
          }
         }
        } while(0);
        if ((label|0) == 53) {
         $163 = ($$0$i56|0)==(0);
         if (!($163)) {
          $$01$i = $$0$i56;
          break;
         }
         $$pre$i = HEAP32[$3>>2]|0;
         $$phi$trans$insert$i = ((($$pre$i)) + 20|0);
         $$pre4$i = HEAP32[$$phi$trans$insert$i>>2]|0;
         $165 = $$pre4$i;$185 = $$pre$i;
        }
        $164 = ((($165)) + 104|0);
        $166 = HEAP8[$164>>0]|0;
        $167 = $166&255;
        $168 = $167 << 24;
        $169 = ((($165)) + 105|0);
        $170 = HEAP8[$169>>0]|0;
        $171 = $170&255;
        $172 = $171 << 16;
        $173 = $172 | $168;
        $174 = ((($165)) + 106|0);
        $175 = HEAP8[$174>>0]|0;
        $176 = $175&255;
        $177 = $176 << 8;
        $178 = $173 | $177;
        $179 = ((($165)) + 107|0);
        $180 = HEAP8[$179>>0]|0;
        $181 = $180&255;
        $182 = $178 | $181;
        $183 = (($182) + 1)|0;
        $184 = ((($185)) + 4|0);
        $186 = HEAP32[$184>>2]|0;
        $187 = ((($186)) + 24|0);
        $188 = $183 >>> 24;
        $189 = $188&255;
        HEAP8[$187>>0] = $189;
        $190 = $183 >>> 16;
        $191 = $190&255;
        $192 = ((($186)) + 25|0);
        HEAP8[$192>>0] = $191;
        $193 = $183 >>> 8;
        $194 = $193&255;
        $195 = ((($186)) + 26|0);
        HEAP8[$195>>0] = $194;
        $196 = $183&255;
        $197 = ((($186)) + 27|0);
        HEAP8[$197>>0] = $196;
        $198 = HEAP32[$184>>2]|0;
        $199 = ((($198)) + 92|0);
        HEAP8[$199>>0] = $189;
        $200 = ((($198)) + 93|0);
        HEAP8[$200>>0] = $191;
        $201 = ((($198)) + 94|0);
        HEAP8[$201>>0] = $194;
        $202 = ((($198)) + 95|0);
        HEAP8[$202>>0] = $196;
        $203 = HEAP32[$184>>2]|0;
        $204 = ((($203)) + 96|0);
        HEAP8[$204>>0] = 0;
        $205 = ((($203)) + 97|0);
        HEAP8[$205>>0] = 46;
        $206 = ((($203)) + 98|0);
        HEAP8[$206>>0] = 52;
        $207 = ((($203)) + 99|0);
        HEAP8[$207>>0] = 32;
        HEAP8[$127>>0] = 1;
        $$01$i = 0;
       } else {
        $$01$i = $135;
       }
      } while(0);
      $208 = HEAP32[$3>>2]|0;
      $209 = ($208|0)==(0|0);
      do {
       if (!($209)) {
        $210 = ((($208)) + 28|0);
        $211 = HEAP16[$210>>1]|0;
        $212 = $211 & 32;
        $213 = ($212<<16>>16)==(0);
        if ($213) {
         _sqlite3PcacheRelease($208);
         break;
        } else {
         $214 = ((($208)) + 20|0);
         $215 = HEAP32[$214>>2]|0;
         $216 = ((($215)) + 120|0);
         $217 = HEAP32[$216>>2]|0;
         $218 = (($217) + -1)|0;
         HEAP32[$216>>2] = $218;
         $219 = ((($215)) + 136|0);
         $220 = HEAP32[$219>>2]|0;
         $221 = ((($208)) + 16|0);
         HEAP32[$221>>2] = $220;
         HEAP32[$219>>2] = $208;
         break;
        }
       }
      } while(0);
      $222 = ($$01$i|0)==(0);
      if (!($222)) {
       $$0 = $$01$i;
       STACKTOP = sp;return ($$0|0);
      }
     }
    }
    $223 = ($1|0)==(0|0);
    L61: do {
     if ($223) {
      label = 79;
     } else {
      $224 = ((($0)) + 5|0);
      $225 = HEAP8[$224>>0]|0;
      $226 = ($225<<24>>24)==(4);
      if ($226) {
       label = 79;
      } else {
       $227 = ((($0)) + 64|0);
       $228 = HEAP32[$227>>2]|0;
       $229 = HEAP32[$228>>2]|0;
       $230 = ($229|0)==(0|0);
       if ($230) {
        label = 79;
       } else {
        $231 = ((($0)) + 19|0);
        HEAP8[$231>>0] = 1;
        $232 = HEAP8[$1>>0]|0;
        $233 = ($232<<24>>24)==(0);
        if ($233) {
         $$0$lcssa$i = 0;$$048$lcssa$i = 0;
        } else {
         $$04850$i = 0;$$051$i = 0;$235 = $232;
         while(1) {
          $234 = $235 << 24 >> 24;
          $236 = (($$051$i) + ($234))|0;
          $237 = (($$04850$i) + 1)|0;
          $238 = (($1) + ($237)|0);
          $239 = HEAP8[$238>>0]|0;
          $240 = ($239<<24>>24)==(0);
          if ($240) {
           $$0$lcssa$i = $236;$$048$lcssa$i = $237;
           break;
          } else {
           $$04850$i = $237;$$051$i = $236;$235 = $239;
          }
         }
        }
        $241 = ((($0)) + 8|0);
        $242 = HEAP8[$241>>0]|0;
        $243 = ($242<<24>>24)==(0);
        $$phi$trans$insert$i58 = ((($0)) + 72|0);
        $244 = $$phi$trans$insert$i58;
        $245 = $244;
        $246 = HEAP32[$245>>2]|0;
        $247 = (($244) + 4)|0;
        $248 = $247;
        $249 = HEAP32[$248>>2]|0;
        if ($243) {
         $286 = $246;$287 = $249;
        } else {
         $250 = ($246|0)==(0);
         $251 = ($249|0)==(0);
         $252 = $250 & $251;
         if ($252) {
          $265 = 0;$268 = 0;
         } else {
          $253 = (_i64Add(($246|0),($249|0),-1,-1)|0);
          $254 = (getTempRet0() | 0);
          $255 = ((($0)) + 148|0);
          $256 = HEAP32[$255>>2]|0;
          $257 = (___divdi3(($253|0),($254|0),($256|0),0)|0);
          $258 = (getTempRet0() | 0);
          $259 = (_i64Add(($257|0),($258|0),1,0)|0);
          $260 = (getTempRet0() | 0);
          $261 = (___muldi3(($259|0),($260|0),($256|0),0)|0);
          $262 = (getTempRet0() | 0);
          $265 = $261;$268 = $262;
         }
         $263 = $$phi$trans$insert$i58;
         $264 = $263;
         HEAP32[$264>>2] = $265;
         $266 = (($263) + 4)|0;
         $267 = $266;
         HEAP32[$267>>2] = $268;
         $286 = $265;$287 = $268;
        }
        $269 = ((($0)) + 152|0);
        $270 = HEAP32[$269>>2]|0;
        $271 = (1073741824 / ($270|0))&-1;
        $272 = (($271) + 1)|0;
        $273 = $272 >>> 24;
        $274 = $273&255;
        HEAP8[$3>>0] = $274;
        $275 = $272 >>> 16;
        $276 = $275&255;
        $277 = ((($3)) + 1|0);
        HEAP8[$277>>0] = $276;
        $278 = $272 >>> 8;
        $279 = $278&255;
        $280 = ((($3)) + 2|0);
        HEAP8[$280>>0] = $279;
        $281 = $272&255;
        $282 = ((($3)) + 3|0);
        HEAP8[$282>>0] = $281;
        $283 = HEAP32[$228>>2]|0;
        $284 = ((($283)) + 12|0);
        $285 = HEAP32[$284>>2]|0;
        $288 = (FUNCTION_TABLE_iiiiii[$285 & 127]($228,$3,4,$286,$287)|0);
        $289 = ($288|0)==(0);
        do {
         if ($289) {
          $290 = HEAP32[$227>>2]|0;
          $291 = (_i64Add(($286|0),($287|0),4,0)|0);
          $292 = (getTempRet0() | 0);
          $293 = HEAP32[$290>>2]|0;
          $294 = ((($293)) + 12|0);
          $295 = HEAP32[$294>>2]|0;
          $296 = (FUNCTION_TABLE_iiiiii[$295 & 127]($290,$1,$$048$lcssa$i,$291,$292)|0);
          $297 = ($296|0)==(0);
          if ($297) {
           $298 = HEAP32[$227>>2]|0;
           $299 = (_i64Add(($291|0),($292|0),($$048$lcssa$i|0),0)|0);
           $300 = (getTempRet0() | 0);
           $301 = $$048$lcssa$i >>> 24;
           $302 = $301&255;
           HEAP8[$3>>0] = $302;
           $303 = $$048$lcssa$i >>> 16;
           $304 = $303&255;
           HEAP8[$277>>0] = $304;
           $305 = $$048$lcssa$i >>> 8;
           $306 = $305&255;
           HEAP8[$280>>0] = $306;
           $307 = $$048$lcssa$i&255;
           HEAP8[$282>>0] = $307;
           $308 = HEAP32[$298>>2]|0;
           $309 = ((($308)) + 12|0);
           $310 = HEAP32[$309>>2]|0;
           $311 = (FUNCTION_TABLE_iiiiii[$310 & 127]($298,$3,4,$299,$300)|0);
           $312 = ($311|0)==(0);
           if ($312) {
            $313 = HEAP32[$227>>2]|0;
            $314 = (_i64Add(($299|0),($300|0),4,0)|0);
            $315 = (getTempRet0() | 0);
            $316 = $$0$lcssa$i >>> 24;
            $317 = $316&255;
            HEAP8[$3>>0] = $317;
            $318 = $$0$lcssa$i >>> 16;
            $319 = $318&255;
            HEAP8[$277>>0] = $319;
            $320 = $$0$lcssa$i >>> 8;
            $321 = $320&255;
            HEAP8[$280>>0] = $321;
            $322 = $$0$lcssa$i&255;
            HEAP8[$282>>0] = $322;
            $323 = HEAP32[$313>>2]|0;
            $324 = ((($323)) + 12|0);
            $325 = HEAP32[$324>>2]|0;
            $326 = (FUNCTION_TABLE_iiiiii[$325 & 127]($313,$3,4,$314,$315)|0);
            $327 = ($326|0)==(0);
            if ($327) {
             $328 = HEAP32[$227>>2]|0;
             $329 = (_i64Add(($299|0),($300|0),8,0)|0);
             $330 = (getTempRet0() | 0);
             $331 = HEAP32[$328>>2]|0;
             $332 = ((($331)) + 12|0);
             $333 = HEAP32[$332>>2]|0;
             $334 = (FUNCTION_TABLE_iiiiii[$333 & 127]($328,32352,8,$329,$330)|0);
             $335 = ($334|0)==(0);
             if ($335) {
              $336 = (($$048$lcssa$i) + 20)|0;
              $337 = $$phi$trans$insert$i58;
              $338 = $337;
              $339 = HEAP32[$338>>2]|0;
              $340 = (($337) + 4)|0;
              $341 = $340;
              $342 = HEAP32[$341>>2]|0;
              $343 = (_i64Add(($339|0),($342|0),($336|0),0)|0);
              $344 = (getTempRet0() | 0);
              $345 = $$phi$trans$insert$i58;
              $346 = $345;
              HEAP32[$346>>2] = $343;
              $347 = (($345) + 4)|0;
              $348 = $347;
              HEAP32[$348>>2] = $344;
              $349 = HEAP32[$227>>2]|0;
              $350 = HEAP32[$349>>2]|0;
              $351 = ((($350)) + 24|0);
              $352 = HEAP32[$351>>2]|0;
              $353 = (FUNCTION_TABLE_iii[$352 & 255]($349,$4)|0);
              $354 = ($353|0)==(0);
              if (!($354)) {
               $$047$i$ph = $353;
               break;
              }
              $355 = $4;
              $356 = $355;
              $357 = HEAP32[$356>>2]|0;
              $358 = (($355) + 4)|0;
              $359 = $358;
              $360 = HEAP32[$359>>2]|0;
              $361 = $$phi$trans$insert$i58;
              $362 = $361;
              $363 = HEAP32[$362>>2]|0;
              $364 = (($361) + 4)|0;
              $365 = $364;
              $366 = HEAP32[$365>>2]|0;
              $367 = ($360|0)>($366|0);
              $368 = ($357>>>0)>($363>>>0);
              $369 = ($360|0)==($366|0);
              $370 = $369 & $368;
              $371 = $367 | $370;
              if (!($371)) {
               label = 79;
               break L61;
              }
              $372 = HEAP32[$227>>2]|0;
              $373 = HEAP32[$372>>2]|0;
              $374 = ((($373)) + 16|0);
              $375 = HEAP32[$374>>2]|0;
              $376 = (FUNCTION_TABLE_iiii[$375 & 127]($372,$363,$366)|0);
              $377 = ($376|0)==(0);
              if ($377) {
               break L61;
              } else {
               $$0 = $376;
              }
              STACKTOP = sp;return ($$0|0);
             } else {
              $$047$i$ph = $334;
             }
            } else {
             $$047$i$ph = $326;
            }
           } else {
            $$047$i$ph = $311;
           }
          } else {
           $$047$i$ph = $296;
          }
         } else {
          $$047$i$ph = $288;
         }
        } while(0);
        $$0 = $$047$i$ph;
        STACKTOP = sp;return ($$0|0);
       }
      }
     }
    } while(0);
    if ((label|0) == 79) {
    }
    $378 = (_syncJournal($0,0)|0);
    $379 = ($378|0)==(0);
    if (!($379)) {
     $$0 = $378;
     STACKTOP = sp;return ($$0|0);
    }
    $380 = ((($0)) + 212|0);
    $381 = HEAP32[$380>>2]|0;
    $382 = (_sqlite3PcacheDirtyList($381)|0);
    $383 = (_pager_write_pagelist($0,$382)|0);
    $384 = ($383|0)==(0);
    if (!($384)) {
     $$0 = $383;
     STACKTOP = sp;return ($$0|0);
    }
    $385 = HEAP32[$380>>2]|0;
    $386 = HEAP32[$385>>2]|0;
    $387 = ($386|0)==(0|0);
    if (!($387)) {
     $389 = $386;
     while(1) {
      $388 = ((($389)) + 12|0);
      $390 = HEAP32[$388>>2]|0;
      $391 = ((($390)) + 8|0);
      $392 = HEAP32[$391>>2]|0;
      $393 = ($392|0)==($389|0);
      $394 = ((($389)) + 36|0);
      $395 = HEAP32[$394>>2]|0;
      if ($393) {
       HEAP32[$391>>2] = $395;
      }
      $396 = ((($389)) + 32|0);
      $397 = HEAP32[$396>>2]|0;
      $398 = ($397|0)==(0|0);
      if ($398) {
       $400 = $395;
       $401 = ((($390)) + 4|0);
       HEAP32[$401>>2] = $395;
       $403 = $400;
      } else {
       $399 = ((($397)) + 36|0);
       HEAP32[$399>>2] = $395;
       $$pre33$i$i$i63 = HEAP32[$394>>2]|0;
       $403 = $$pre33$i$i$i63;
      }
      $402 = ($403|0)==(0|0);
      if ($402) {
       HEAP32[$390>>2] = $397;
       if ($398) {
        $405 = ((($390)) + 33|0);
        HEAP8[$405>>0] = 2;
       }
      } else {
       $404 = ((($403)) + 32|0);
       HEAP32[$404>>2] = $397;
      }
      $406 = ((($389)) + 28|0);
      $407 = HEAP16[$406>>1]|0;
      $408 = $407 & -16;
      $409 = $408 | 1;
      HEAP16[$406>>1] = $409;
      $410 = ((($389)) + 30|0);
      $411 = HEAP16[$410>>1]|0;
      $412 = ($411<<16>>16)==(0);
      if ($412) {
       $413 = ((($390)) + 32|0);
       $414 = HEAP8[$413>>0]|0;
       $415 = ($414<<24>>24)==(0);
       if (!($415)) {
        $416 = HEAP32[(29440)>>2]|0;
        $417 = ((($390)) + 44|0);
        $418 = HEAP32[$417>>2]|0;
        $419 = HEAP32[$389>>2]|0;
        FUNCTION_TABLE_viii[$416 & 255]($418,$419,0);
       }
      }
      $420 = HEAP32[$385>>2]|0;
      $421 = ($420|0)==(0|0);
      if ($421) {
       break;
      } else {
       $389 = $420;
      }
     }
    }
    $422 = ((($0)) + 24|0);
    $423 = HEAP32[$422>>2]|0;
    $424 = ((($0)) + 32|0);
    $425 = HEAP32[$424>>2]|0;
    $426 = ($423>>>0)>($425>>>0);
    L117: do {
     if ($426) {
      $427 = ((($0)) + 152|0);
      $428 = HEAP32[$427>>2]|0;
      $429 = (1073741824 / ($428|0))&-1;
      $430 = (($429) + 1)|0;
      $431 = ($423|0)==($430|0);
      $432 = $431&1;
      $433 = (($423) - ($432))|0;
      $434 = ((($0)) + 60|0);
      $435 = HEAP32[$434>>2]|0;
      $436 = HEAP32[$435>>2]|0;
      $437 = ($436|0)==(0|0);
      if (!($437)) {
       $438 = HEAP8[$12>>0]|0;
       $439 = (($438) + -1)<<24>>24;
       $440 = ($439&255)>(2);
       if ($440) {
        $441 = ((($436)) + 24|0);
        $442 = HEAP32[$441>>2]|0;
        $443 = (FUNCTION_TABLE_iii[$442 & 255]($435,$3)|0);
        $444 = ($428|0)<(0);
        $445 = $444 << 31 >> 31;
        $446 = (___muldi3(($433|0),0,($428|0),($445|0))|0);
        $447 = (getTempRet0() | 0);
        $448 = ($443|0)==(0);
        do {
         if ($448) {
          $449 = $3;
          $450 = $449;
          $451 = HEAP32[$450>>2]|0;
          $452 = (($449) + 4)|0;
          $453 = $452;
          $454 = HEAP32[$453>>2]|0;
          $455 = ($451|0)==($446|0);
          $456 = ($454|0)==($447|0);
          $457 = $455 & $456;
          if (!($457)) {
           $458 = ($454|0)>($447|0);
           $459 = ($451>>>0)>($446>>>0);
           $460 = ($454|0)==($447|0);
           $461 = $460 & $459;
           $462 = $458 | $461;
           do {
            if ($462) {
             $463 = HEAP32[$434>>2]|0;
             $464 = HEAP32[$463>>2]|0;
             $465 = ((($464)) + 16|0);
             $466 = HEAP32[$465>>2]|0;
             $467 = (FUNCTION_TABLE_iiii[$466 & 127]($463,$446,$447)|0);
             $$0$i68 = $467;
             label = 107;
            } else {
             $468 = (_i64Add(($451|0),($454|0),($428|0),($445|0))|0);
             $469 = (getTempRet0() | 0);
             $470 = ($469|0)>($447|0);
             $471 = ($468>>>0)>($446>>>0);
             $472 = ($469|0)==($447|0);
             $473 = $472 & $471;
             $474 = $470 | $473;
             if ($474) {
              break;
             }
             $475 = ((($0)) + 208|0);
             $476 = HEAP32[$475>>2]|0;
             _memset(($476|0),0,($428|0))|0;
             $477 = HEAP32[$434>>2]|0;
             $478 = (_i64Subtract(($446|0),($447|0),($428|0),($445|0))|0);
             $479 = (getTempRet0() | 0);
             $480 = HEAP32[$477>>2]|0;
             $481 = ((($480)) + 12|0);
             $482 = HEAP32[$481>>2]|0;
             $483 = (FUNCTION_TABLE_iiiiii[$482 & 127]($477,$476,$428,$478,$479)|0);
             $$0$i68 = $483;
             label = 107;
            }
           } while(0);
           if ((label|0) == 107) {
            $484 = ($$0$i68|0)==(0);
            if (!($484)) {
             $$1$i70 = $$0$i68;
             break;
            }
           }
           HEAP32[$424>>2] = $433;
          }
          break L117;
         } else {
          $$1$i70 = $443;
         }
        } while(0);
        $$0 = $$1$i70;
        STACKTOP = sp;return ($$0|0);
       }
      }
     }
    } while(0);
    $485 = ($2|0)==(0);
    if (!($485)) {
     break;
    }
    $486 = ((($0)) + 60|0);
    $487 = HEAP32[$486>>2]|0;
    $488 = HEAP32[$487>>2]|0;
    $489 = ($488|0)==(0|0);
    L137: do {
     if (!($489)) {
      $490 = ((($488)) + 40|0);
      $491 = HEAP32[$490>>2]|0;
      $492 = (FUNCTION_TABLE_iiii[$491 & 127]($487,21,$1)|0);
      switch ($492|0) {
      case 0: case 12:  {
       break L137;
       break;
      }
      default: {
       $$0 = $492;
      }
      }
      STACKTOP = sp;return ($$0|0);
     }
    } while(0);
    $493 = ((($0)) + 7|0);
    $494 = HEAP8[$493>>0]|0;
    $495 = ($494<<24>>24)==(0);
    if (!($495)) {
     break;
    }
    $496 = HEAP32[$486>>2]|0;
    $497 = ((($0)) + 10|0);
    $498 = HEAP8[$497>>0]|0;
    $499 = ($498<<24>>24)==(0);
    if ($499) {
     break;
    }
    $500 = $498&255;
    $501 = HEAP32[$496>>2]|0;
    $502 = ((($501)) + 20|0);
    $503 = HEAP32[$502>>2]|0;
    $504 = (FUNCTION_TABLE_iii[$503 & 255]($496,$500)|0);
    $$5 = $504;
   } else {
    HEAP32[$3>>2] = 0;
    $63 = ((($0)) + 212|0);
    $64 = HEAP32[$63>>2]|0;
    $65 = (_sqlite3PcacheDirtyList($64)|0);
    $66 = ($65|0)==(0|0);
    if ($66) {
     $67 = ((($0)) + 204|0);
     $68 = HEAP32[$67>>2]|0;
     $69 = (FUNCTION_TABLE_iiiii[$68 & 127]($0,1,$3,0)|0);
     $70 = HEAP32[$3>>2]|0;
     $71 = ((($70)) + 16|0);
     HEAP32[$71>>2] = 0;
     $72 = ($70|0)==(0|0);
     if ($72) {
      $$15374 = $69;
     } else {
      $$05072 = $70;
      label = 21;
     }
    } else {
     $$05072 = $65;
     label = 21;
    }
    do {
     if ((label|0) == 21) {
      $73 = ((($0)) + 24|0);
      $74 = HEAP32[$73>>2]|0;
      $75 = (_pagerWalFrames($0,$$05072,$74,1)|0);
      $$pr = HEAP32[$3>>2]|0;
      $76 = ($$pr|0)==(0|0);
      if ($76) {
       $$15374 = $75;
      } else {
       $77 = ((($$pr)) + 28|0);
       $78 = HEAP16[$77>>1]|0;
       $79 = $78 & 32;
       $80 = ($79<<16>>16)==(0);
       if ($80) {
        _sqlite3PcacheRelease($$pr);
        $$15374 = $75;
        break;
       } else {
        $81 = ((($$pr)) + 20|0);
        $82 = HEAP32[$81>>2]|0;
        $83 = ((($82)) + 120|0);
        $84 = HEAP32[$83>>2]|0;
        $85 = (($84) + -1)|0;
        HEAP32[$83>>2] = $85;
        $86 = ((($82)) + 136|0);
        $87 = HEAP32[$86>>2]|0;
        $88 = ((($$pr)) + 16|0);
        HEAP32[$88>>2] = $87;
        HEAP32[$86>>2] = $$pr;
        $$15374 = $75;
        break;
       }
      }
     }
    } while(0);
    $89 = ($$15374|0)==(0);
    if ($89) {
     $90 = HEAP32[$63>>2]|0;
     $91 = HEAP32[$90>>2]|0;
     $92 = ($91|0)==(0|0);
     if (!($92)) {
      $94 = $91;
      while(1) {
       $93 = ((($94)) + 12|0);
       $95 = HEAP32[$93>>2]|0;
       $96 = ((($95)) + 8|0);
       $97 = HEAP32[$96>>2]|0;
       $98 = ($97|0)==($94|0);
       $99 = ((($94)) + 36|0);
       $100 = HEAP32[$99>>2]|0;
       if ($98) {
        HEAP32[$96>>2] = $100;
       }
       $101 = ((($94)) + 32|0);
       $102 = HEAP32[$101>>2]|0;
       $103 = ($102|0)==(0|0);
       if ($103) {
        $105 = $100;
        $106 = ((($95)) + 4|0);
        HEAP32[$106>>2] = $100;
        $108 = $105;
       } else {
        $104 = ((($102)) + 36|0);
        HEAP32[$104>>2] = $100;
        $$pre33$i$i$i = HEAP32[$99>>2]|0;
        $108 = $$pre33$i$i$i;
       }
       $107 = ($108|0)==(0|0);
       if ($107) {
        HEAP32[$95>>2] = $102;
        if ($103) {
         $110 = ((($95)) + 33|0);
         HEAP8[$110>>0] = 2;
        }
       } else {
        $109 = ((($108)) + 32|0);
        HEAP32[$109>>2] = $102;
       }
       $111 = ((($94)) + 28|0);
       $112 = HEAP16[$111>>1]|0;
       $113 = $112 & -16;
       $114 = $113 | 1;
       HEAP16[$111>>1] = $114;
       $115 = ((($94)) + 30|0);
       $116 = HEAP16[$115>>1]|0;
       $117 = ($116<<16>>16)==(0);
       if ($117) {
        $118 = ((($95)) + 32|0);
        $119 = HEAP8[$118>>0]|0;
        $120 = ($119<<24>>24)==(0);
        if (!($120)) {
         $121 = HEAP32[(29440)>>2]|0;
         $122 = ((($95)) + 44|0);
         $123 = HEAP32[$122>>2]|0;
         $124 = HEAP32[$94>>2]|0;
         FUNCTION_TABLE_viii[$121 & 255]($123,$124,0);
        }
       }
       $125 = HEAP32[$90>>2]|0;
       $126 = ($125|0)==(0|0);
       if ($126) {
        break;
       } else {
        $94 = $125;
       }
      }
     }
    }
    $$5 = $$15374;
   }
   $505 = ($$5|0)==(0);
   if (!($505)) {
    $$0 = $$5;
    STACKTOP = sp;return ($$0|0);
   }
  }
 } while(0);
 $506 = ((($0)) + 216|0);
 $507 = HEAP32[$506>>2]|0;
 $508 = ($507|0)==(0|0);
 if (!($508)) {
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
 }
 HEAP8[$12>>0] = 5;
 $$0 = 0;
 STACKTOP = sp;return ($$0|0);
}
function _sqlite3BtreeCommitPhaseTwo($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i = 0, $$1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0;
 var $getPageNormal$getPageError$i$i$i = 0, $or$cond = 0, $trunc$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($0)) + 8|0);
 $3 = HEAP8[$2>>0]|0;
 $4 = ($3<<24>>24)==(0);
 if ($4) {
  $$1 = 0;
  return ($$1|0);
 }
 $5 = HEAP32[$0>>2]|0;
 $6 = ((($0)) + 4|0);
 $7 = HEAP32[$6>>2]|0;
 $8 = ((($7)) + 4|0);
 HEAP32[$8>>2] = $5;
 $9 = ($3<<24>>24)==(2);
 if ($9) {
  $10 = HEAP32[$7>>2]|0;
  $11 = ((($10)) + 40|0);
  $12 = HEAP32[$11>>2]|0;
  $13 = ($12|0)==(0);
  L6: do {
   if ($13) {
    $14 = ((($10)) + 16|0);
    $15 = HEAP8[$14>>0]|0;
    $16 = ($15<<24>>24)==(2);
    if ($16) {
     $17 = ((($10)) + 4|0);
     $18 = HEAP8[$17>>0]|0;
     $19 = ($18<<24>>24)==(0);
     if (!($19)) {
      $20 = ((($10)) + 5|0);
      $21 = HEAP8[$20>>0]|0;
      $22 = ($21<<24>>24)==(1);
      if ($22) {
       HEAP8[$14>>0] = 1;
       break;
      }
     }
    }
    $23 = ((($10)) + 100|0);
    $24 = HEAP32[$23>>2]|0;
    $25 = (($24) + 1)|0;
    HEAP32[$23>>2] = $25;
    $26 = ((($10)) + 19|0);
    $27 = HEAP8[$26>>0]|0;
    $28 = $27&255;
    $29 = (_pager_end_transaction($10,$28,1)|0);
    $trunc$i$i = $29&255;
    switch ($trunc$i$i<<24>>24) {
    case 10: case 13:  {
     break;
    }
    default: {
     $$0$i = $29;
     label = 10;
     break L6;
    }
    }
    HEAP32[$11>>2] = $29;
    HEAP8[$14>>0] = 6;
    $30 = ($29|0)==(0);
    $31 = ((($10)) + 204|0);
    $getPageNormal$getPageError$i$i$i = $30 ? 74 : 75;
    HEAP32[$31>>2] = $getPageNormal$getPageError$i$i$i;
    $$0$i = $29;
    label = 10;
   } else {
    $$0$i = $12;
    label = 10;
   }
  } while(0);
  if ((label|0) == 10) {
   $32 = ($$0$i|0)!=(0);
   $33 = ($1|0)==(0);
   $or$cond = $33 & $32;
   if ($or$cond) {
    $$1 = $$0$i;
    return ($$1|0);
   }
  }
  $34 = ((($0)) + 20|0);
  $35 = HEAP32[$34>>2]|0;
  $36 = (($35) + -1)|0;
  HEAP32[$34>>2] = $36;
  $37 = ((($7)) + 20|0);
  HEAP8[$37>>0] = 1;
  $38 = ((($7)) + 60|0);
  $39 = HEAP32[$38>>2]|0;
  _sqlite3BitvecDestroy($39);
  HEAP32[$38>>2] = 0;
 }
 _btreeEndTransaction($0);
 $$1 = 0;
 return ($$1|0);
}
function _sqlite3BtreeCommitPhaseOne($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0, $$0$i59$i = 0, $$0$i60$i = 0, $$015$i$i = 0, $$015$i$i$i = 0, $$015$i27$i$i = 0, $$05$i$i = 0, $$05663$i = 0, $$057$i = 0, $$06$i$i = 0, $$1 = 0, $$1$i$i = 0, $$158$lcssa$i = 0, $$2$i = 0, $$idx$i = 0, $$idx$val$i = 0, $$lcssa$i = 0, $$pre = 0, $$pre34$i$i = 0, $10 = 0;
 var $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0;
 var $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0;
 var $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0;
 var $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0;
 var $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0;
 var $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0;
 var $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0;
 var $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $brmerge$i = 0;
 var $or$cond$i = 0, $or$cond$i$i = 0, $or$cond29$i$i = 0, $or$cond3$i = 0, $spec$select$i$i = 0, $spec$select$i$i$i = 0, $spec$select$i26$i$i = 0, $spec$select$v$i$i = 0, $spec$select$v$i$i$i = 0, $spec$select$v$i25$i$i = 0, $vararg_buffer = 0, $vararg_buffer3 = 0, $vararg_ptr1 = 0, $vararg_ptr2 = 0, $vararg_ptr6 = 0, $vararg_ptr7 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $vararg_buffer3 = sp + 16|0;
 $vararg_buffer = sp;
 $2 = ((($0)) + 8|0);
 $3 = HEAP8[$2>>0]|0;
 $4 = ($3<<24>>24)==(2);
 if (!($4)) {
  $$1 = 0;
  STACKTOP = sp;return ($$1|0);
 }
 $5 = ((($0)) + 4|0);
 $6 = HEAP32[$5>>2]|0;
 $7 = HEAP32[$0>>2]|0;
 $8 = ((($6)) + 4|0);
 HEAP32[$8>>2] = $7;
 $9 = ((($6)) + 17|0);
 $10 = HEAP8[$9>>0]|0;
 $11 = ($10<<24>>24)==(0);
 do {
  if (!($11)) {
   $12 = HEAP32[$6>>2]|0;
   $13 = ((($6)) + 8|0);
   $$05$i$i = HEAP32[$13>>2]|0;
   $14 = ($$05$i$i|0)==(0|0);
   if (!($14)) {
    $$06$i$i = $$05$i$i;
    while(1) {
     $15 = ((($$06$i$i)) + 1|0);
     $16 = HEAP8[$15>>0]|0;
     $17 = $16 & -5;
     HEAP8[$15>>0] = $17;
     $18 = ((($$06$i$i)) + 24|0);
     $$0$i$i = HEAP32[$18>>2]|0;
     $19 = ($$0$i$i|0)==(0|0);
     if ($19) {
      break;
     } else {
      $$06$i$i = $$0$i$i;
     }
    }
   }
   $20 = ((($6)) + 18|0);
   $21 = HEAP8[$20>>0]|0;
   $22 = ($21<<24>>24)==(0);
   if ($22) {
    $$idx$i = ((($6)) + 44|0);
    $$idx$val$i = HEAP32[$$idx$i>>2]|0;
    $23 = ($$idx$val$i>>>0)<(2);
    if ($23) {
     $$015$i$i = 0;
    } else {
     $24 = ((($6)) + 36|0);
     $25 = HEAP32[$24>>2]|0;
     $26 = (($25>>>0) / 5)&-1;
     $27 = (($26) + 1)|0;
     $28 = (($$idx$val$i) + -2)|0;
     $29 = (($28>>>0) % ($27>>>0))&-1;
     $30 = (($28) - ($29))|0;
     $31 = ((($6)) + 32|0);
     $32 = HEAP32[$31>>2]|0;
     $33 = (1073741824 / ($32>>>0))&-1;
     $34 = (($30) + 1)|0;
     $35 = ($34|0)==($33|0);
     $spec$select$v$i$i = $35 ? 3 : 2;
     $spec$select$i$i = (($spec$select$v$i$i) + ($30))|0;
     $$015$i$i = $spec$select$i$i;
    }
    $36 = ($$015$i$i|0)==($$idx$val$i|0);
    if (!($36)) {
     $37 = ((($6)) + 32|0);
     $38 = HEAP32[$37>>2]|0;
     $39 = (1073741824 / ($38>>>0))&-1;
     $40 = (($39) + 1)|0;
     $41 = ($$idx$val$i|0)==($40|0);
     if (!($41)) {
      $42 = ((($6)) + 12|0);
      $43 = HEAP32[$42>>2]|0;
      $44 = ((($43)) + 56|0);
      $45 = HEAP32[$44>>2]|0;
      $46 = ((($45)) + 36|0);
      $47 = HEAP8[$46>>0]|0;
      $48 = $47&255;
      $49 = $48 << 24;
      $50 = ((($45)) + 37|0);
      $51 = HEAP8[$50>>0]|0;
      $52 = $51&255;
      $53 = $52 << 16;
      $54 = $53 | $49;
      $55 = ((($45)) + 38|0);
      $56 = HEAP8[$55>>0]|0;
      $57 = $56&255;
      $58 = $57 << 8;
      $59 = $54 | $58;
      $60 = ((($45)) + 39|0);
      $61 = HEAP8[$60>>0]|0;
      $62 = $61&255;
      $63 = $59 | $62;
      $64 = ((($6)) + 36|0);
      $65 = HEAP32[$64>>2]|0;
      $66 = (($65>>>0) / 5)&-1;
      $$pre34$i$i = (($66) + 1)|0;
      if ($23) {
       $$015$i$i$i = 0;
      } else {
       $67 = (($$idx$val$i) + -2)|0;
       $68 = (($67>>>0) % ($$pre34$i$i>>>0))&-1;
       $69 = (($67) - ($68))|0;
       $70 = (($69) + 1)|0;
       $71 = ($70|0)==($39|0);
       $spec$select$v$i$i$i = $71 ? 3 : 2;
       $spec$select$i$i$i = (($spec$select$v$i$i$i) + ($69))|0;
       $$015$i$i$i = $spec$select$i$i$i;
      }
      $72 = (($66) - ($$idx$val$i))|0;
      $73 = (($72) + ($63))|0;
      $74 = (($73) + ($$015$i$i$i))|0;
      $75 = (($74>>>0) / ($66>>>0))&-1;
      $76 = (($$idx$val$i) - ($63))|0;
      $77 = (($76) - ($75))|0;
      $78 = ($40>>>0)<($$idx$val$i>>>0);
      $79 = ($77>>>0)<($40>>>0);
      $or$cond$i$i = $78 & $79;
      $80 = $or$cond$i$i << 31 >> 31;
      $$0$i60$i = (($77) + ($80))|0;
      $$1$i$i = $$0$i60$i;
      while(1) {
       $81 = ($$1$i$i>>>0)<(2);
       if ($81) {
        $$015$i27$i$i = 0;
       } else {
        $82 = (($$1$i$i) + -2)|0;
        $83 = (($82>>>0) % ($$pre34$i$i>>>0))&-1;
        $84 = (($82) - ($83))|0;
        $85 = (($84) + 1)|0;
        $86 = ($85|0)==($39|0);
        $spec$select$v$i25$i$i = $86 ? 3 : 2;
        $spec$select$i26$i$i = (($spec$select$v$i25$i$i) + ($84))|0;
        $$015$i27$i$i = $spec$select$i26$i$i;
       }
       $87 = ($$015$i27$i$i|0)==($$1$i$i|0);
       $88 = ($$1$i$i|0)==($40|0);
       $or$cond29$i$i = $88 | $87;
       $89 = (($$1$i$i) + -1)|0;
       if ($or$cond29$i$i) {
        $$1$i$i = $89;
       } else {
        break;
       }
      }
      $90 = ($$1$i$i>>>0)>($$idx$val$i>>>0);
      if ($90) {
       HEAP32[$vararg_buffer3>>2] = 32306;
       $vararg_ptr6 = ((($vararg_buffer3)) + 4|0);
       HEAP32[$vararg_ptr6>>2] = 67104;
       $vararg_ptr7 = ((($vararg_buffer3)) + 8|0);
       HEAP32[$vararg_ptr7>>2] = (31517);
       _sqlite3_log(11,32001,$vararg_buffer3);
       $$1 = 11;
       STACKTOP = sp;return ($$1|0);
      }
      $91 = ($$1$i$i>>>0)>=($$idx$val$i>>>0);
      $brmerge$i = $14 | $91;
      if ($brmerge$i) {
       $$057$i = 0;
      } else {
       $92 = (_saveCursorsOnList($$05$i$i,0,0)|0);
       $$057$i = $92;
      }
      $93 = ($$idx$val$i>>>0)>($$1$i$i>>>0);
      $94 = ($$057$i|0)==(0);
      $95 = $93 & $94;
      if ($95) {
       $$05663$i = $$idx$val$i;
       while(1) {
        $96 = (_incrVacuumStep($6,$$1$i$i,$$05663$i,1)|0);
        $97 = (($$05663$i) + -1)|0;
        $98 = ($97>>>0)>($$1$i$i>>>0);
        $99 = ($96|0)==(0);
        $100 = $99 & $98;
        if ($100) {
         $$05663$i = $97;
        } else {
         $$158$lcssa$i = $96;$$lcssa$i = $99;
         break;
        }
       }
      } else {
       $$158$lcssa$i = $$057$i;$$lcssa$i = $94;
      }
      $101 = ($$158$lcssa$i|0)==(101);
      $or$cond$i = $101 | $$lcssa$i;
      $102 = ($63|0)!=(0);
      $or$cond3$i = $102 & $or$cond$i;
      if ($or$cond3$i) {
       $103 = HEAP32[$42>>2]|0;
       $104 = ((($103)) + 72|0);
       $105 = HEAP32[$104>>2]|0;
       $106 = ((($105)) + 20|0);
       $107 = HEAP32[$106>>2]|0;
       $108 = ((($105)) + 28|0);
       $109 = HEAP16[$108>>1]|0;
       $110 = $109 & 4;
       $111 = ($110<<16>>16)==(0);
       if ($111) {
        label = 30;
       } else {
        $112 = ((($107)) + 24|0);
        $113 = HEAP32[$112>>2]|0;
        $114 = ((($105)) + 24|0);
        $115 = HEAP32[$114>>2]|0;
        $116 = ($113>>>0)<($115>>>0);
        if ($116) {
         label = 30;
        } else {
         $117 = ((($107)) + 96|0);
         $118 = HEAP32[$117>>2]|0;
         $119 = ($118|0)==(0);
         if ($119) {
          $$0$i59$i = 0;
         } else {
          $120 = (_subjournalPageIfRequired($105)|0);
          $$0$i59$i = $120;
         }
        }
       }
       do {
        if ((label|0) == 30) {
         $121 = ((($107)) + 40|0);
         $122 = HEAP32[$121>>2]|0;
         $123 = ($122|0)==(0);
         if ($123) {
          $124 = ((($107)) + 148|0);
          $125 = HEAP32[$124>>2]|0;
          $126 = ((($107)) + 152|0);
          $127 = HEAP32[$126>>2]|0;
          $128 = ($125>>>0)>($127>>>0);
          if ($128) {
           $129 = (_pagerWriteLargeSector($105)|0);
           $$0$i59$i = $129;
           break;
          } else {
           $130 = (_pager_write($105)|0);
           $$0$i59$i = $130;
           break;
          }
         } else {
          $$0$i59$i = $122;
         }
        }
       } while(0);
       $131 = HEAP32[$42>>2]|0;
       $132 = ((($131)) + 56|0);
       $133 = HEAP32[$132>>2]|0;
       $134 = ((($133)) + 32|0);
       HEAP8[$134>>0]=0&255;HEAP8[$134+1>>0]=(0>>8)&255;HEAP8[$134+2>>0]=(0>>16)&255;HEAP8[$134+3>>0]=0>>24;
       $135 = HEAP32[$42>>2]|0;
       $136 = ((($135)) + 56|0);
       $137 = HEAP32[$136>>2]|0;
       $138 = ((($137)) + 36|0);
       HEAP8[$138>>0]=0&255;HEAP8[$138+1>>0]=(0>>8)&255;HEAP8[$138+2>>0]=(0>>16)&255;HEAP8[$138+3>>0]=0>>24;
       $139 = HEAP32[$42>>2]|0;
       $140 = ((($139)) + 56|0);
       $141 = HEAP32[$140>>2]|0;
       $142 = ((($141)) + 28|0);
       $143 = $$1$i$i >>> 24;
       $144 = $143&255;
       HEAP8[$142>>0] = $144;
       $145 = $$1$i$i >>> 16;
       $146 = $145&255;
       $147 = ((($141)) + 29|0);
       HEAP8[$147>>0] = $146;
       $148 = $$1$i$i >>> 8;
       $149 = $148&255;
       $150 = ((($141)) + 30|0);
       HEAP8[$150>>0] = $149;
       $151 = $$1$i$i&255;
       $152 = ((($141)) + 31|0);
       HEAP8[$152>>0] = $151;
       $153 = ((($6)) + 19|0);
       HEAP8[$153>>0] = 1;
       HEAP32[$$idx$i>>2] = $$1$i$i;
       $$2$i = $$0$i59$i;
      } else {
       $$2$i = $$158$lcssa$i;
      }
      $154 = ($$2$i|0)==(0);
      if ($154) {
       break;
      }
      (_sqlite3PagerRollback($12)|0);
      $$1 = $$2$i;
      STACKTOP = sp;return ($$1|0);
     }
    }
    HEAP32[$vararg_buffer>>2] = 32306;
    $vararg_ptr1 = ((($vararg_buffer)) + 4|0);
    HEAP32[$vararg_ptr1>>2] = 67099;
    $vararg_ptr2 = ((($vararg_buffer)) + 8|0);
    HEAP32[$vararg_ptr2>>2] = (31517);
    _sqlite3_log(11,32001,$vararg_buffer);
    $$1 = 11;
    STACKTOP = sp;return ($$1|0);
   }
  }
 } while(0);
 $155 = ((($6)) + 19|0);
 $156 = HEAP8[$155>>0]|0;
 $157 = ($156<<24>>24)==(0);
 $$pre = HEAP32[$6>>2]|0;
 if (!($157)) {
  $158 = ((($6)) + 44|0);
  $159 = HEAP32[$158>>2]|0;
  $160 = ((($$pre)) + 24|0);
  HEAP32[$160>>2] = $159;
 }
 $161 = (_sqlite3PagerCommitPhaseOne($$pre,$1,0)|0);
 $$1 = $161;
 STACKTOP = sp;return ($$1|0);
}
function _sqlite3_backup_finish($0) {
 $0 = $0|0;
 var $$0 = 0, $$029 = 0, $$idx$val$idx$val = 0, $$pr = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0;
 var $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0;
 var $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $spec$select = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$0 = 0;
  return ($$0|0);
 }
 $2 = ((($0)) + 20|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ((($0)) + 24|0);
 $5 = HEAP32[$4>>2]|0;
 $6 = HEAP32[$5>>2]|0;
 $7 = ((($5)) + 4|0);
 $8 = HEAP32[$7>>2]|0;
 $9 = ((($8)) + 4|0);
 HEAP32[$9>>2] = $6;
 $10 = HEAP32[$0>>2]|0;
 $11 = ($10|0)==(0|0);
 if (!($11)) {
  $12 = ((($5)) + 16|0);
  $13 = HEAP32[$12>>2]|0;
  $14 = (($13) + -1)|0;
  HEAP32[$12>>2] = $14;
 }
 $15 = ((($0)) + 40|0);
 $16 = HEAP32[$15>>2]|0;
 $17 = ($16|0)==(0);
 if (!($17)) {
  $$idx$val$idx$val = HEAP32[$8>>2]|0;
  $18 = ((($$idx$val$idx$val)) + 88|0);
  $$029 = $18;
  while(1) {
   $19 = HEAP32[$$029>>2]|0;
   $20 = ($19|0)==($0|0);
   $21 = ((($19)) + 44|0);
   if ($20) {
    break;
   } else {
    $$029 = $21;
   }
  }
  $22 = ((($0)) + 44|0);
  $23 = HEAP32[$22>>2]|0;
  HEAP32[$$029>>2] = $23;
 }
 $24 = ((($0)) + 4|0);
 $25 = HEAP32[$24>>2]|0;
 _sqlite3BtreeRollback($25,0,0);
 $26 = ((($0)) + 28|0);
 $27 = HEAP32[$26>>2]|0;
 $28 = ($27|0)==(101);
 $spec$select = $28 ? 0 : $27;
 $29 = HEAP32[$0>>2]|0;
 $30 = ($29|0)==(0|0);
 do {
  if (!($30)) {
   $31 = ((($29)) + 64|0);
   HEAP32[$31>>2] = $spec$select;
   $32 = ($spec$select|0)==(0);
   if ($32) {
    $33 = ((($29)) + 260|0);
    $34 = HEAP32[$33>>2]|0;
    $35 = ($34|0)==(0|0);
    if (!($35)) {
     _sqlite3ErrorFinish($29,0);
    }
   } else {
    _sqlite3ErrorFinish($29,$spec$select);
   }
   $36 = HEAP32[$0>>2]|0;
   _sqlite3LeaveMutexAndCloseZombie($36);
   $$pr = HEAP32[$0>>2]|0;
   $37 = ($$pr|0)==(0|0);
   if (!($37)) {
    $38 = HEAP32[7324]|0;
    $39 = ($38|0)==(0);
    if ($39) {
     $47 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$47 & 127]($0);
     break;
    } else {
     $40 = HEAP32[(29352)>>2]|0;
     $41 = (FUNCTION_TABLE_ii[$40 & 127]($0)|0);
     $42 = HEAP32[14978]|0;
     $43 = (($42) - ($41))|0;
     HEAP32[14978] = $43;
     $44 = HEAP32[14981]|0;
     $45 = (($44) + -1)|0;
     HEAP32[14981] = $45;
     $46 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$46 & 127]($0);
     break;
    }
   }
  }
 } while(0);
 _sqlite3LeaveMutexAndCloseZombie($3);
 $$0 = $spec$select;
 return ($$0|0);
}
function _sqlite3BtreeRollback($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$0$ph = 0, $$017$i = 0, $$028$ph = 0, $$06 = 0, $$idx = 0, $$idx$i$i = 0, $$idx$i$i$i = 0, $$idx$val = 0, $$idx$val$i$i$i = 0, $$idx$val15$i$i = 0, $$idx1 = 0, $$idx1$val = 0, $$idx14$i$i = 0, $$idx14$val$i$i = 0, $$phi$trans$insert = 0, $$pre = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0;
 var $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0;
 var $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0;
 var $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0;
 var sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp;
 $4 = ((($0)) + 4|0);
 $5 = HEAP32[$4>>2]|0;
 $6 = HEAP32[$0>>2]|0;
 $7 = ((($5)) + 4|0);
 HEAP32[$7>>2] = $6;
 $8 = ($1|0)==(0);
 if ($8) {
  $9 = ((($5)) + 8|0);
  $$017$i = HEAP32[$9>>2]|0;
  $10 = ($$017$i|0)==(0|0);
  if (!($10)) {
   $11 = (_saveCursorsOnList($$017$i,0,0)|0);
   $12 = ($11|0)==(0);
   if (!($12)) {
    $$0$ph = $11;$$028$ph = 0;
    label = 4;
   }
  }
 } else {
  $$0$ph = $1;$$028$ph = $2;
  label = 4;
 }
 if ((label|0) == 4) {
  (_sqlite3BtreeTripAllCursors($0,$$0$ph,$$028$ph)|0);
 }
 $13 = ((($0)) + 8|0);
 $14 = HEAP8[$13>>0]|0;
 $15 = ($14<<24>>24)==(2);
 if (!($15)) {
  _btreeEndTransaction($0);
  STACKTOP = sp;return;
 }
 $16 = HEAP32[$5>>2]|0;
 (_sqlite3PagerRollback($16)|0);
 $17 = HEAP32[$5>>2]|0;
 $18 = ((($17)) + 204|0);
 $19 = HEAP32[$18>>2]|0;
 $20 = (FUNCTION_TABLE_iiiii[$19 & 127]($17,1,$3,0)|0);
 $21 = ($20|0)==(0);
 if ($21) {
  $22 = HEAP32[$3>>2]|0;
  $$idx14$i$i = ((($22)) + 8|0);
  $$idx14$val$i$i = HEAP32[$$idx14$i$i>>2]|0;
  $23 = ((($$idx14$val$i$i)) + 4|0);
  $24 = HEAP32[$23>>2]|0;
  $25 = ($24|0)==(1);
  if ($25) {
   $$phi$trans$insert = ((($$idx14$val$i$i)) + 56|0);
   $$pre = HEAP32[$$phi$trans$insert>>2]|0;
   $32 = $$pre;
  } else {
   $$idx$i$i = ((($22)) + 4|0);
   $$idx$val15$i$i = HEAP32[$$idx$i$i>>2]|0;
   $26 = ((($$idx14$val$i$i)) + 56|0);
   HEAP32[$26>>2] = $$idx$val15$i$i;
   $27 = ((($$idx14$val$i$i)) + 72|0);
   HEAP32[$27>>2] = $22;
   $28 = ((($$idx14$val$i$i)) + 52|0);
   HEAP32[$28>>2] = $5;
   HEAP32[$23>>2] = 1;
   $29 = ((($$idx14$val$i$i)) + 9|0);
   HEAP8[$29>>0] = 100;
   $30 = $$idx$val15$i$i;
   $32 = $30;
  }
  $31 = ((($32)) + 28|0);
  $33 = HEAP8[$31>>0]|0;
  $34 = $33&255;
  $35 = $34 << 24;
  $36 = ((($32)) + 29|0);
  $37 = HEAP8[$36>>0]|0;
  $38 = $37&255;
  $39 = $38 << 16;
  $40 = $39 | $35;
  $41 = ((($32)) + 30|0);
  $42 = HEAP8[$41>>0]|0;
  $43 = $42&255;
  $44 = $43 << 8;
  $45 = $40 | $44;
  $46 = ((($32)) + 31|0);
  $47 = HEAP8[$46>>0]|0;
  $48 = $47&255;
  $49 = $45 | $48;
  $50 = ($49|0)==(0);
  if ($50) {
   $51 = HEAP32[$5>>2]|0;
   $$idx = ((($51)) + 24|0);
   $$idx$val = HEAP32[$$idx>>2]|0;
   $$06 = $$idx$val;
  } else {
   $$06 = $49;
  }
  $52 = ((($5)) + 44|0);
  HEAP32[$52>>2] = $$06;
  $$idx1 = ((($$idx14$val$i$i)) + 72|0);
  $$idx1$val = HEAP32[$$idx1>>2]|0;
  $53 = ((($$idx1$val)) + 20|0);
  $54 = HEAP32[$53>>2]|0;
  _sqlite3PcacheRelease($$idx1$val);
  $55 = ((($54)) + 212|0);
  $56 = HEAP32[$55>>2]|0;
  $$idx$i$i$i = ((($56)) + 12|0);
  $$idx$val$i$i$i = HEAP32[$$idx$i$i$i>>2]|0;
  $57 = ($$idx$val$i$i$i|0)==(0);
  if ($57) {
   $58 = ((($54)) + 16|0);
   $59 = HEAP8[$58>>0]|0;
   switch ($59<<24>>24) {
   case 0: case 6:  {
    break;
   }
   case 1:  {
    $60 = ((($54)) + 4|0);
    $61 = HEAP8[$60>>0]|0;
    $62 = ($61<<24>>24)==(0);
    if ($62) {
     (_pager_end_transaction($54,0,0)|0);
    }
    break;
   }
   default: {
    (_sqlite3PagerRollback($54)|0);
   }
   }
   _pager_unlock($54);
  }
 } else {
 }
 $63 = ((($5)) + 20|0);
 HEAP8[$63>>0] = 1;
 $64 = ((($5)) + 60|0);
 $65 = HEAP32[$64>>2]|0;
 _sqlite3BitvecDestroy($65);
 HEAP32[$64>>2] = 0;
 _btreeEndTransaction($0);
 STACKTOP = sp;return;
}
function _sqlite3LeaveMutexAndCloseZombie($0) {
 $0 = $0|0;
 var $$0 = 0, $$0$i = 0, $$0$i$i = 0, $$0129 = 0, $$0130 = 0, $$01314$i = 0, $$03235$i = 0, $$03334$i = 0, $$05$i$i = 0, $$06$i$i = 0, $$078133 = 0, $$080 = 0, $$1 = 0, $$1$i = 0, $$1125 = 0, $$1126 = 0, $$2 = 0, $$2122 = 0, $$2123 = 0, $$idx$i = 0;
 var $$idx$val$i = 0, $$in$i = 0, $$in$i108 = 0, $$in$i99 = 0, $$lcssa = 0, $$pr = 0, $$pr142 = 0, $$pr143 = 0, $$pre = 0, $1 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0;
 var $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0;
 var $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0;
 var $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0;
 var $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0;
 var $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0;
 var $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0;
 var $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0;
 var $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0;
 var $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0;
 var $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0;
 var $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0;
 var $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0;
 var $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0;
 var $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0;
 var $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0;
 var $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0;
 var $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0;
 var $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0;
 var $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0;
 var $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0;
 var $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0;
 var $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0;
 var $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0;
 var $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0;
 var $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 96|0);
 $2 = HEAP32[$1>>2]|0;
 $3 = ($2|0)==(1691352191);
 if (!($3)) {
  return;
 }
 $4 = ((($0)) + 4|0);
 $5 = HEAP32[$4>>2]|0;
 $6 = ($5|0)==(0|0);
 if (!($6)) {
  return;
 }
 $7 = ((($0)) + 20|0);
 $8 = HEAP32[$7>>2]|0;
 $9 = ($8|0)>(0);
 L7: do {
  if ($9) {
   $10 = ((($0)) + 16|0);
   $11 = HEAP32[$10>>2]|0;
   $$01314$i = 0;
   while(1) {
    $12 = (((($11) + ($$01314$i<<4)|0)) + 4|0);
    $13 = HEAP32[$12>>2]|0;
    $14 = ($13|0)==(0|0);
    if (!($14)) {
     $$idx$i = ((($13)) + 16|0);
     $$idx$val$i = HEAP32[$$idx$i>>2]|0;
     $15 = ($$idx$val$i|0)==(0);
     if (!($15)) {
      break;
     }
    }
    $16 = (($$01314$i) + 1)|0;
    $17 = ($16|0)<($8|0);
    if ($17) {
     $$01314$i = $16;
    } else {
     break L7;
    }
   }
   return;
  }
 } while(0);
 _sqlite3RollbackAll($0,0);
 $18 = ((($0)) + 444|0);
 $19 = HEAP32[$18>>2]|0;
 $20 = ($19|0)==(0|0);
 if (!($20)) {
  $21 = ((($0)) + 480|0);
  $22 = ((($0)) + 304|0);
  $23 = ((($0)) + 308|0);
  $24 = ((($0)) + 300|0);
  $26 = $19;
  while(1) {
   $25 = ((($26)) + 24|0);
   $27 = HEAP32[$25>>2]|0;
   HEAP32[$18>>2] = $27;
   $28 = HEAP32[$21>>2]|0;
   $29 = ($28|0)==(0|0);
   do {
    if ($29) {
     $30 = $26;
     $31 = HEAP32[$22>>2]|0;
     $32 = ($31>>>0)>($30>>>0);
     if (!($32)) {
      $33 = HEAP32[$23>>2]|0;
      $34 = ($33>>>0)>($30>>>0);
      if ($34) {
       $35 = HEAP32[$24>>2]|0;
       HEAP32[$26>>2] = $35;
       HEAP32[$24>>2] = $26;
       break;
      }
     }
     $36 = HEAP32[7324]|0;
     $37 = ($36|0)==(0);
     if ($37) {
      $45 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$45 & 127]($26);
      break;
     } else {
      $38 = HEAP32[(29352)>>2]|0;
      $39 = (FUNCTION_TABLE_ii[$38 & 127]($26)|0);
      $40 = HEAP32[14978]|0;
      $41 = (($40) - ($39))|0;
      HEAP32[14978] = $41;
      $42 = HEAP32[14981]|0;
      $43 = (($42) + -1)|0;
      HEAP32[14981] = $43;
      $44 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$44 & 127]($26);
      break;
     }
    } else {
     _measureAllocationSize($0,$26);
    }
   } while(0);
   $46 = HEAP32[$18>>2]|0;
   $47 = ($46|0)==(0|0);
   if ($47) {
    break;
   } else {
    $26 = $46;
   }
  }
 }
 $48 = ((($0)) + 452|0);
 HEAP32[$48>>2] = 0;
 $49 = ((($0)) + 456|0);
 HEAP32[$49>>2] = 0;
 $50 = ((($0)) + 87|0);
 HEAP8[$50>>0] = 0;
 $51 = HEAP32[$7>>2]|0;
 $52 = ($51|0)>(0);
 $53 = ((($0)) + 16|0);
 $54 = HEAP32[$53>>2]|0;
 if ($52) {
  $$078133 = 0;$56 = $54;
  while(1) {
   $55 = (((($56) + ($$078133<<4)|0)) + 4|0);
   $57 = HEAP32[$55>>2]|0;
   $58 = ($57|0)==(0|0);
   if (!($58)) {
    _sqlite3BtreeClose($57);
    HEAP32[$55>>2] = 0;
    $59 = ($$078133|0)==(1);
    if (!($59)) {
     $60 = (((($56) + ($$078133<<4)|0)) + 12|0);
     HEAP32[$60>>2] = 0;
    }
   }
   $61 = (($$078133) + 1)|0;
   $62 = HEAP32[$7>>2]|0;
   $63 = ($61|0)<($62|0);
   $64 = HEAP32[$53>>2]|0;
   if ($63) {
    $$078133 = $61;$56 = $64;
   } else {
    $$lcssa = $64;
    break;
   }
  }
 } else {
  $$lcssa = $54;
 }
 $65 = ((($$lcssa)) + 28|0);
 $66 = HEAP32[$65>>2]|0;
 $67 = ($66|0)==(0|0);
 if (!($67)) {
  _sqlite3SchemaClear($66);
 }
 $68 = ((($0)) + 360|0);
 $69 = HEAP32[$68>>2]|0;
 HEAP32[$68>>2] = 0;
 $70 = ($69|0)==(0|0);
 if (!($70)) {
  $$05$i$i = HEAP32[$4>>2]|0;
  $71 = ($$05$i$i|0)==(0|0);
  if (!($71)) {
   $$06$i$i = $$05$i$i;
   while(1) {
    $72 = ((($$06$i$i)) + 149|0);
    $73 = HEAPU8[$72>>0]|(HEAPU8[$72+1>>0]<<8);
    $74 = $73 & -4;
    $75 = $74 | 1;
    HEAP8[$72>>0]=$75&255;HEAP8[$72+1>>0]=$75>>8;
    $76 = ((($$06$i$i)) + 8|0);
    $$0$i$i = HEAP32[$76>>2]|0;
    $77 = ($$0$i$i|0)==(0|0);
    if ($77) {
     break;
    } else {
     $$06$i$i = $$0$i$i;
    }
   }
  }
  $$0$i = $69;
  while(1) {
   $78 = ((($$0$i)) + 24|0);
   $79 = HEAP32[$78>>2]|0;
   $80 = HEAP32[$$0$i>>2]|0;
   $81 = ((($$0$i)) + 12|0);
   $82 = HEAP32[$81>>2]|0;
   $83 = (($82) + -1)|0;
   HEAP32[$81>>2] = $83;
   $84 = ($83|0)==(0);
   do {
    if ($84) {
     $85 = ((($$0$i)) + 8|0);
     $86 = HEAP32[$85>>2]|0;
     $87 = ($86|0)==(0|0);
     if (!($87)) {
      $88 = HEAP32[$86>>2]|0;
      $89 = ((($88)) + 16|0);
      $90 = HEAP32[$89>>2]|0;
      (FUNCTION_TABLE_ii[$90 & 127]($86)|0);
     }
     $91 = ($80|0)==(0|0);
     if (!($91)) {
      $92 = ((($80)) + 480|0);
      $93 = HEAP32[$92>>2]|0;
      $94 = ($93|0)==(0|0);
      if (!($94)) {
       _measureAllocationSize($80,$$0$i);
       break;
      }
      $95 = $$0$i;
      $96 = ((($80)) + 304|0);
      $97 = HEAP32[$96>>2]|0;
      $98 = ($97>>>0)>($95>>>0);
      if (!($98)) {
       $99 = ((($80)) + 308|0);
       $100 = HEAP32[$99>>2]|0;
       $101 = ($100>>>0)>($95>>>0);
       if ($101) {
        $102 = ((($80)) + 300|0);
        $103 = HEAP32[$102>>2]|0;
        HEAP32[$$0$i>>2] = $103;
        HEAP32[$102>>2] = $$0$i;
        break;
       }
      }
     }
     $104 = HEAP32[7324]|0;
     $105 = ($104|0)==(0);
     if ($105) {
      $113 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$113 & 127]($$0$i);
      break;
     } else {
      $106 = HEAP32[(29352)>>2]|0;
      $107 = (FUNCTION_TABLE_ii[$106 & 127]($$0$i)|0);
      $108 = HEAP32[14978]|0;
      $109 = (($108) - ($107))|0;
      HEAP32[14978] = $109;
      $110 = HEAP32[14981]|0;
      $111 = (($110) + -1)|0;
      HEAP32[14981] = $111;
      $112 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$112 & 127]($$0$i);
      break;
     }
    }
   } while(0);
   $114 = ($79|0)==(0|0);
   if ($114) {
    break;
   } else {
    $$0$i = $79;
   }
  }
 }
 $115 = HEAP32[$7>>2]|0;
 $116 = ($115|0)>(2);
 if ($116) {
  $117 = ((($0)) + 480|0);
  $118 = ((($0)) + 304|0);
  $119 = ((($0)) + 308|0);
  $120 = ((($0)) + 300|0);
  $$03235$i = 2;$$03334$i = 2;
  while(1) {
   $121 = HEAP32[$53>>2]|0;
   $122 = (($121) + ($$03235$i<<4)|0);
   $123 = (((($121) + ($$03235$i<<4)|0)) + 4|0);
   $124 = HEAP32[$123>>2]|0;
   $125 = ($124|0)==(0|0);
   if ($125) {
    $126 = HEAP32[$122>>2]|0;
    $127 = ($126|0)==(0|0);
    do {
     if (!($127)) {
      $128 = HEAP32[$117>>2]|0;
      $129 = ($128|0)==(0|0);
      if (!($129)) {
       _measureAllocationSize($0,$126);
       break;
      }
      $130 = $126;
      $131 = HEAP32[$118>>2]|0;
      $132 = ($131>>>0)>($130>>>0);
      if (!($132)) {
       $133 = HEAP32[$119>>2]|0;
       $134 = ($133>>>0)>($130>>>0);
       if ($134) {
        $135 = HEAP32[$120>>2]|0;
        HEAP32[$126>>2] = $135;
        HEAP32[$120>>2] = $126;
        break;
       }
      }
      $136 = HEAP32[7324]|0;
      $137 = ($136|0)==(0);
      if ($137) {
       $145 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$145 & 127]($126);
       break;
      } else {
       $138 = HEAP32[(29352)>>2]|0;
       $139 = (FUNCTION_TABLE_ii[$138 & 127]($126)|0);
       $140 = HEAP32[14978]|0;
       $141 = (($140) - ($139))|0;
       HEAP32[14978] = $141;
       $142 = HEAP32[14981]|0;
       $143 = (($142) + -1)|0;
       HEAP32[14981] = $143;
       $144 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$144 & 127]($126);
       break;
      }
     }
    } while(0);
    HEAP32[$122>>2] = 0;
    $$1$i = $$03334$i;
   } else {
    $146 = ($$03334$i|0)<($$03235$i|0);
    if ($146) {
     $147 = (($121) + ($$03334$i<<4)|0);
     ;HEAP32[$147>>2]=HEAP32[$122>>2]|0;HEAP32[$147+4>>2]=HEAP32[$122+4>>2]|0;HEAP32[$147+8>>2]=HEAP32[$122+8>>2]|0;HEAP32[$147+12>>2]=HEAP32[$122+12>>2]|0;
    }
    $148 = (($$03334$i) + 1)|0;
    $$1$i = $148;
   }
   $149 = (($$03235$i) + 1)|0;
   $150 = HEAP32[$7>>2]|0;
   $151 = ($149|0)<($150|0);
   if ($151) {
    $$03235$i = $149;$$03334$i = $$1$i;
   } else {
    break;
   }
  }
  HEAP32[$7>>2] = $$1$i;
  $152 = ($$1$i|0)<(3);
  if ($152) {
   label = 64;
  }
 } else {
  HEAP32[$7>>2] = 2;
  label = 64;
 }
 if ((label|0) == 64) {
  $153 = HEAP32[$53>>2]|0;
  $154 = ((($0)) + 412|0);
  $155 = ($153|0)==($154|0);
  if (!($155)) {
   ;HEAP32[$154>>2]=HEAP32[$153>>2]|0;HEAP32[$154+4>>2]=HEAP32[$153+4>>2]|0;HEAP32[$154+8>>2]=HEAP32[$153+8>>2]|0;HEAP32[$154+12>>2]=HEAP32[$153+12>>2]|0;HEAP32[$154+16>>2]=HEAP32[$153+16>>2]|0;HEAP32[$154+20>>2]=HEAP32[$153+20>>2]|0;HEAP32[$154+24>>2]=HEAP32[$153+24>>2]|0;HEAP32[$154+28>>2]=HEAP32[$153+28>>2]|0;
   $156 = ($153|0)==(0|0);
   do {
    if (!($156)) {
     $157 = ((($0)) + 480|0);
     $158 = HEAP32[$157>>2]|0;
     $159 = ($158|0)==(0|0);
     if (!($159)) {
      _measureAllocationSize($0,$153);
      break;
     }
     $160 = $153;
     $161 = ((($0)) + 304|0);
     $162 = HEAP32[$161>>2]|0;
     $163 = ($162>>>0)>($160>>>0);
     if (!($163)) {
      $164 = ((($0)) + 308|0);
      $165 = HEAP32[$164>>2]|0;
      $166 = ($165>>>0)>($160>>>0);
      if ($166) {
       $167 = ((($0)) + 300|0);
       $168 = HEAP32[$167>>2]|0;
       HEAP32[$153>>2] = $168;
       HEAP32[$167>>2] = $153;
       break;
      }
     }
     $169 = HEAP32[7324]|0;
     $170 = ($169|0)==(0);
     if ($170) {
      $178 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$178 & 127]($153);
      break;
     } else {
      $171 = HEAP32[(29352)>>2]|0;
      $172 = (FUNCTION_TABLE_ii[$171 & 127]($153)|0);
      $173 = HEAP32[14978]|0;
      $174 = (($173) - ($172))|0;
      HEAP32[14978] = $174;
      $175 = HEAP32[14981]|0;
      $176 = (($175) + -1)|0;
      HEAP32[14981] = $176;
      $177 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$177 & 127]($153);
      break;
     }
    }
   } while(0);
   HEAP32[$53>>2] = $154;
  }
 }
 $179 = ((($0)) + 372|0);
 $$0129 = HEAP32[$179>>2]|0;
 $180 = ($$0129|0)==(0|0);
 if ($180) {
  $253 = 0;
 } else {
  $181 = ((($0)) + 480|0);
  $182 = ((($0)) + 304|0);
  $183 = ((($0)) + 308|0);
  $184 = ((($0)) + 300|0);
  $$0130 = $$0129;
  while(1) {
   $185 = ((($$0130)) + 8|0);
   $186 = HEAP32[$185>>2]|0;
   $$080 = $186;
   while(1) {
    $187 = ((($$080)) + 36|0);
    $188 = HEAP32[$187>>2]|0;
    $189 = ($188|0)==(0|0);
    do {
     if (!($189)) {
      $190 = HEAP32[$188>>2]|0;
      $191 = (($190) + -1)|0;
      HEAP32[$188>>2] = $191;
      $192 = ($191|0)==(0);
      if ($192) {
       $193 = ((($188)) + 4|0);
       $194 = HEAP32[$193>>2]|0;
       $195 = ((($188)) + 8|0);
       $196 = HEAP32[$195>>2]|0;
       FUNCTION_TABLE_vi[$194 & 127]($196);
       $197 = HEAP32[$181>>2]|0;
       $198 = ($197|0)==(0|0);
       if (!($198)) {
        _measureAllocationSize($0,$188);
        break;
       }
       $199 = $188;
       $200 = HEAP32[$182>>2]|0;
       $201 = ($200>>>0)>($199>>>0);
       if (!($201)) {
        $202 = HEAP32[$183>>2]|0;
        $203 = ($202>>>0)>($199>>>0);
        if ($203) {
         $204 = HEAP32[$184>>2]|0;
         HEAP32[$188>>2] = $204;
         HEAP32[$184>>2] = $188;
         break;
        }
       }
       $205 = HEAP32[7324]|0;
       $206 = ($205|0)==(0);
       if ($206) {
        $214 = HEAP32[(29344)>>2]|0;
        FUNCTION_TABLE_vi[$214 & 127]($188);
        break;
       } else {
        $207 = HEAP32[(29352)>>2]|0;
        $208 = (FUNCTION_TABLE_ii[$207 & 127]($188)|0);
        $209 = HEAP32[14978]|0;
        $210 = (($209) - ($208))|0;
        HEAP32[14978] = $210;
        $211 = HEAP32[14981]|0;
        $212 = (($211) + -1)|0;
        HEAP32[14981] = $212;
        $213 = HEAP32[(29344)>>2]|0;
        FUNCTION_TABLE_vi[$213 & 127]($188);
        break;
       }
      }
     }
    } while(0);
    $215 = ((($$080)) + 12|0);
    $216 = HEAP32[$215>>2]|0;
    $217 = ($$080|0)==(0|0);
    do {
     if (!($217)) {
      $218 = HEAP32[$181>>2]|0;
      $219 = ($218|0)==(0|0);
      if (!($219)) {
       _measureAllocationSize($0,$$080);
       break;
      }
      $220 = $$080;
      $221 = HEAP32[$182>>2]|0;
      $222 = ($221>>>0)>($220>>>0);
      if (!($222)) {
       $223 = HEAP32[$183>>2]|0;
       $224 = ($223>>>0)>($220>>>0);
       if ($224) {
        $225 = HEAP32[$184>>2]|0;
        HEAP32[$$080>>2] = $225;
        HEAP32[$184>>2] = $$080;
        break;
       }
      }
      $226 = HEAP32[7324]|0;
      $227 = ($226|0)==(0);
      if ($227) {
       $235 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$235 & 127]($$080);
       break;
      } else {
       $228 = HEAP32[(29352)>>2]|0;
       $229 = (FUNCTION_TABLE_ii[$228 & 127]($$080)|0);
       $230 = HEAP32[14978]|0;
       $231 = (($230) - ($229))|0;
       HEAP32[14978] = $231;
       $232 = HEAP32[14981]|0;
       $233 = (($232) + -1)|0;
       HEAP32[14981] = $233;
       $234 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$234 & 127]($$080);
       break;
      }
     }
    } while(0);
    $236 = ($216|0)==(0|0);
    if ($236) {
     break;
    } else {
     $$080 = $216;
    }
   }
   $$0 = HEAP32[$$0130>>2]|0;
   $237 = ($$0|0)==(0|0);
   if ($237) {
    break;
   } else {
    $$0130 = $$0;
   }
  }
  $$pre = HEAP32[$179>>2]|0;
  $253 = $$pre;
 }
 HEAP32[$179>>2] = 0;
 $238 = ((($0)) + 376|0);
 $239 = HEAP32[$238>>2]|0;
 $240 = ($239|0)==(0|0);
 do {
  if (!($240)) {
   $241 = HEAP32[7324]|0;
   $242 = ($241|0)==(0);
   if ($242) {
    $250 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$250 & 127]($239);
    break;
   } else {
    $243 = HEAP32[(29352)>>2]|0;
    $244 = (FUNCTION_TABLE_ii[$243 & 127]($239)|0);
    $245 = HEAP32[14978]|0;
    $246 = (($245) - ($244))|0;
    HEAP32[14978] = $246;
    $247 = HEAP32[14981]|0;
    $248 = (($247) + -1)|0;
    HEAP32[14981] = $248;
    $249 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$249 & 127]($239);
    break;
   }
  }
 } while(0);
 HEAP32[$238>>2] = 0;
 $251 = ((($0)) + 364|0);
 HEAP32[$251>>2] = 0;
 $252 = ($253|0)==(0|0);
 if (!($252)) {
  $$in$i = $253;
  while(1) {
   $254 = HEAP32[$$in$i>>2]|0;
   $255 = HEAP32[7324]|0;
   $256 = ($255|0)==(0);
   if ($256) {
    $264 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$264 & 127]($$in$i);
   } else {
    $257 = HEAP32[(29352)>>2]|0;
    $258 = (FUNCTION_TABLE_ii[$257 & 127]($$in$i)|0);
    $259 = HEAP32[14978]|0;
    $260 = (($259) - ($258))|0;
    HEAP32[14978] = $260;
    $261 = HEAP32[14981]|0;
    $262 = (($261) + -1)|0;
    HEAP32[14981] = $262;
    $263 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$263 & 127]($$in$i);
   }
   $265 = ($254|0)==(0|0);
   if ($265) {
    break;
   } else {
    $$in$i = $254;
   }
  }
 }
 $266 = ((($0)) + 368|0);
 HEAP32[$266>>2] = 0;
 $267 = ((($0)) + 388|0);
 $$1125 = HEAP32[$267>>2]|0;
 $268 = ($$1125|0)==(0|0);
 if ($268) {
  $317 = 0;
 } else {
  $269 = ((($0)) + 480|0);
  $270 = ((($0)) + 304|0);
  $271 = ((($0)) + 308|0);
  $272 = ((($0)) + 300|0);
  $$1126 = $$1125;
  while(1) {
   $273 = ((($$1126)) + 8|0);
   $274 = HEAP32[$273>>2]|0;
   $275 = ((($274)) + 16|0);
   $276 = HEAP32[$275>>2]|0;
   $277 = ($276|0)==(0|0);
   if (!($277)) {
    $278 = ((($274)) + 8|0);
    $279 = HEAP32[$278>>2]|0;
    FUNCTION_TABLE_vi[$276 & 127]($279);
   }
   $280 = ((($274)) + 36|0);
   $281 = HEAP32[$280>>2]|0;
   $282 = ($281|0)==(0|0);
   if (!($282)) {
    $493 = ((($274)) + 28|0);
    $494 = HEAP32[$493>>2]|0;
    FUNCTION_TABLE_vi[$281 & 127]($494);
   }
   $495 = ((($274)) + 56|0);
   $496 = HEAP32[$495>>2]|0;
   $497 = ($496|0)==(0|0);
   if (!($497)) {
    $498 = ((($274)) + 48|0);
    $499 = HEAP32[$498>>2]|0;
    FUNCTION_TABLE_vi[$496 & 127]($499);
   }
   $283 = HEAP32[$269>>2]|0;
   $284 = ($283|0)==(0|0);
   L178: do {
    if ($284) {
     $285 = $274;
     $286 = HEAP32[$270>>2]|0;
     $287 = ($286>>>0)>($285>>>0);
     do {
      if (!($287)) {
       $288 = HEAP32[$271>>2]|0;
       $289 = ($288>>>0)>($285>>>0);
       if (!($289)) {
        break;
       }
       $290 = HEAP32[$272>>2]|0;
       HEAP32[$274>>2] = $290;
       HEAP32[$272>>2] = $274;
       break L178;
      }
     } while(0);
     $291 = HEAP32[7324]|0;
     $292 = ($291|0)==(0);
     if ($292) {
      $300 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$300 & 127]($274);
      break;
     } else {
      $293 = HEAP32[(29352)>>2]|0;
      $294 = (FUNCTION_TABLE_ii[$293 & 127]($274)|0);
      $295 = HEAP32[14978]|0;
      $296 = (($295) - ($294))|0;
      HEAP32[14978] = $296;
      $297 = HEAP32[14981]|0;
      $298 = (($297) + -1)|0;
      HEAP32[14981] = $298;
      $299 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$299 & 127]($274);
      break;
     }
    } else {
     _measureAllocationSize($0,$274);
    }
   } while(0);
   $$1 = HEAP32[$$1126>>2]|0;
   $301 = ($$1|0)==(0|0);
   if ($301) {
    break;
   } else {
    $$1126 = $$1;
   }
  }
  $$pr142 = HEAP32[$267>>2]|0;
  $317 = $$pr142;
 }
 HEAP32[$267>>2] = 0;
 $302 = ((($0)) + 392|0);
 $303 = HEAP32[$302>>2]|0;
 $304 = ($303|0)==(0|0);
 do {
  if (!($304)) {
   $305 = HEAP32[7324]|0;
   $306 = ($305|0)==(0);
   if ($306) {
    $314 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$314 & 127]($303);
    break;
   } else {
    $307 = HEAP32[(29352)>>2]|0;
    $308 = (FUNCTION_TABLE_ii[$307 & 127]($303)|0);
    $309 = HEAP32[14978]|0;
    $310 = (($309) - ($308))|0;
    HEAP32[14978] = $310;
    $311 = HEAP32[14981]|0;
    $312 = (($311) + -1)|0;
    HEAP32[14981] = $312;
    $313 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$313 & 127]($303);
    break;
   }
  }
 } while(0);
 HEAP32[$302>>2] = 0;
 $315 = ((($0)) + 380|0);
 HEAP32[$315>>2] = 0;
 $316 = ($317|0)==(0|0);
 if (!($316)) {
  $$in$i99 = $317;
  while(1) {
   $318 = HEAP32[$$in$i99>>2]|0;
   $319 = HEAP32[7324]|0;
   $320 = ($319|0)==(0);
   if ($320) {
    $328 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$328 & 127]($$in$i99);
   } else {
    $321 = HEAP32[(29352)>>2]|0;
    $322 = (FUNCTION_TABLE_ii[$321 & 127]($$in$i99)|0);
    $323 = HEAP32[14978]|0;
    $324 = (($323) - ($322))|0;
    HEAP32[14978] = $324;
    $325 = HEAP32[14981]|0;
    $326 = (($325) + -1)|0;
    HEAP32[14981] = $326;
    $327 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$327 & 127]($$in$i99);
   }
   $329 = ($318|0)==(0|0);
   if ($329) {
    break;
   } else {
    $$in$i99 = $318;
   }
  }
 }
 $330 = ((($0)) + 384|0);
 HEAP32[$330>>2] = 0;
 $331 = ((($0)) + 344|0);
 $$2122 = HEAP32[$331>>2]|0;
 $332 = ($$2122|0)==(0|0);
 if ($332) {
  $390 = 0;
 } else {
  $333 = ((($0)) + 480|0);
  $334 = ((($0)) + 304|0);
  $335 = ((($0)) + 308|0);
  $336 = ((($0)) + 300|0);
  $$2123 = $$2122;
  while(1) {
   $337 = ((($$2123)) + 8|0);
   $338 = HEAP32[$337>>2]|0;
   $339 = ((($338)) + 12|0);
   $340 = HEAP32[$339>>2]|0;
   $341 = ($340|0)==(0|0);
   if (!($341)) {
    $342 = ((($338)) + 8|0);
    $343 = HEAP32[$342>>2]|0;
    FUNCTION_TABLE_vi[$340 & 127]($343);
   }
   $344 = ((($338)) + 16|0);
   $345 = HEAP32[$344>>2]|0;
   $346 = ($345|0)==(0|0);
   if (!($346)) {
    $347 = ((($345)) + 36|0);
    $348 = HEAP32[$347>>2]|0;
    $349 = $348 | 2;
    HEAP32[$347>>2] = $349;
    $350 = HEAP32[$333>>2]|0;
    $351 = ($350|0)==(0|0);
    do {
     if ($351) {
      $352 = ((($345)) + 32|0);
      $353 = HEAP32[$352>>2]|0;
      $354 = (($353) + -1)|0;
      HEAP32[$352>>2] = $354;
      $355 = ($354|0)==(0);
      if (!($355)) {
       break;
      }
      _deleteTable($0,$345);
     } else {
      _deleteTable($0,$345);
     }
    } while(0);
    HEAP32[$344>>2] = 0;
   }
   $356 = HEAP32[$333>>2]|0;
   $357 = ($356|0)==(0|0);
   L221: do {
    if ($357) {
     $358 = $338;
     $359 = HEAP32[$334>>2]|0;
     $360 = ($359>>>0)>($358>>>0);
     do {
      if (!($360)) {
       $361 = HEAP32[$335>>2]|0;
       $362 = ($361>>>0)>($358>>>0);
       if (!($362)) {
        break;
       }
       $363 = HEAP32[$336>>2]|0;
       HEAP32[$338>>2] = $363;
       HEAP32[$336>>2] = $338;
       break L221;
      }
     } while(0);
     $364 = HEAP32[7324]|0;
     $365 = ($364|0)==(0);
     if ($365) {
      $373 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$373 & 127]($338);
      break;
     } else {
      $366 = HEAP32[(29352)>>2]|0;
      $367 = (FUNCTION_TABLE_ii[$366 & 127]($338)|0);
      $368 = HEAP32[14978]|0;
      $369 = (($368) - ($367))|0;
      HEAP32[14978] = $369;
      $370 = HEAP32[14981]|0;
      $371 = (($370) + -1)|0;
      HEAP32[14981] = $371;
      $372 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$372 & 127]($338);
      break;
     }
    } else {
     _measureAllocationSize($0,$338);
    }
   } while(0);
   $$2 = HEAP32[$$2123>>2]|0;
   $374 = ($$2|0)==(0|0);
   if ($374) {
    break;
   } else {
    $$2123 = $$2;
   }
  }
  $$pr143 = HEAP32[$331>>2]|0;
  $390 = $$pr143;
 }
 HEAP32[$331>>2] = 0;
 $375 = ((($0)) + 348|0);
 $376 = HEAP32[$375>>2]|0;
 $377 = ($376|0)==(0|0);
 do {
  if (!($377)) {
   $378 = HEAP32[7324]|0;
   $379 = ($378|0)==(0);
   if ($379) {
    $387 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$387 & 127]($376);
    break;
   } else {
    $380 = HEAP32[(29352)>>2]|0;
    $381 = (FUNCTION_TABLE_ii[$380 & 127]($376)|0);
    $382 = HEAP32[14978]|0;
    $383 = (($382) - ($381))|0;
    HEAP32[14978] = $383;
    $384 = HEAP32[14981]|0;
    $385 = (($384) + -1)|0;
    HEAP32[14981] = $385;
    $386 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$386 & 127]($376);
    break;
   }
  }
 } while(0);
 HEAP32[$375>>2] = 0;
 $388 = ((($0)) + 336|0);
 HEAP32[$388>>2] = 0;
 $389 = ($390|0)==(0|0);
 if (!($389)) {
  $$in$i108 = $390;
  while(1) {
   $391 = HEAP32[$$in$i108>>2]|0;
   $392 = HEAP32[7324]|0;
   $393 = ($392|0)==(0);
   if ($393) {
    $401 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$401 & 127]($$in$i108);
   } else {
    $394 = HEAP32[(29352)>>2]|0;
    $395 = (FUNCTION_TABLE_ii[$394 & 127]($$in$i108)|0);
    $396 = HEAP32[14978]|0;
    $397 = (($396) - ($395))|0;
    HEAP32[14978] = $397;
    $398 = HEAP32[14981]|0;
    $399 = (($398) + -1)|0;
    HEAP32[14981] = $399;
    $400 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$400 & 127]($$in$i108);
   }
   $402 = ($391|0)==(0|0);
   if ($402) {
    break;
   } else {
    $$in$i108 = $391;
   }
  }
 }
 $403 = ((($0)) + 340|0);
 HEAP32[$403>>2] = 0;
 $404 = ((($0)) + 64|0);
 HEAP32[$404>>2] = 0;
 $405 = ((($0)) + 260|0);
 $406 = HEAP32[$405>>2]|0;
 $407 = ($406|0)==(0|0);
 L249: do {
  if (!($407)) {
   _sqlite3ErrorFinish($0,0);
   $$pr = HEAP32[$405>>2]|0;
   $408 = ($$pr|0)==(0|0);
   if (!($408)) {
    $409 = ((($$pr)) + 8|0);
    $410 = HEAP16[$409>>1]|0;
    $411 = $410 & 9216;
    $412 = ($411<<16>>16)==(0);
    if ($412) {
     $413 = ((($$pr)) + 24|0);
     $414 = HEAP32[$413>>2]|0;
     $415 = ($414|0)==(0);
     if (!($415)) {
      label = 169;
     }
    } else {
     label = 169;
    }
    if ((label|0) == 169) {
     _vdbeMemClear($$pr);
    }
    $416 = ((($$pr)) + 32|0);
    $417 = HEAP32[$416>>2]|0;
    $418 = ($417|0)==(0|0);
    do {
     if (!($418)) {
      $419 = ((($417)) + 480|0);
      $420 = HEAP32[$419>>2]|0;
      $421 = ($420|0)==(0|0);
      if (!($421)) {
       _measureAllocationSize($417,$$pr);
       break L249;
      }
      $422 = $$pr;
      $423 = ((($417)) + 304|0);
      $424 = HEAP32[$423>>2]|0;
      $425 = ($424>>>0)>($422>>>0);
      if ($425) {
       break;
      }
      $426 = ((($417)) + 308|0);
      $427 = HEAP32[$426>>2]|0;
      $428 = ($427>>>0)>($422>>>0);
      if (!($428)) {
       break;
      }
      $429 = ((($417)) + 300|0);
      $430 = HEAP32[$429>>2]|0;
      HEAP32[$$pr>>2] = $430;
      HEAP32[$429>>2] = $$pr;
      break L249;
     }
    } while(0);
    $431 = HEAP32[7324]|0;
    $432 = ($431|0)==(0);
    if ($432) {
     $440 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$440 & 127]($$pr);
     break;
    } else {
     $433 = HEAP32[(29352)>>2]|0;
     $434 = (FUNCTION_TABLE_ii[$433 & 127]($$pr)|0);
     $435 = HEAP32[14978]|0;
     $436 = (($435) - ($434))|0;
     HEAP32[14978] = $436;
     $437 = HEAP32[14981]|0;
     $438 = (($437) + -1)|0;
     HEAP32[14981] = $438;
     $439 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$439 & 127]($$pr);
     break;
    }
   }
  }
 } while(0);
 HEAP32[$1>>2] = -1254786768;
 $441 = HEAP32[$53>>2]|0;
 $442 = ((($441)) + 28|0);
 $443 = HEAP32[$442>>2]|0;
 $444 = ($443|0)==(0|0);
 L269: do {
  if (!($444)) {
   $445 = ((($0)) + 480|0);
   $446 = HEAP32[$445>>2]|0;
   $447 = ($446|0)==(0|0);
   if (!($447)) {
    _measureAllocationSize($0,$443);
    break;
   }
   $448 = $443;
   $449 = ((($0)) + 304|0);
   $450 = HEAP32[$449>>2]|0;
   $451 = ($450>>>0)>($448>>>0);
   do {
    if (!($451)) {
     $452 = ((($0)) + 308|0);
     $453 = HEAP32[$452>>2]|0;
     $454 = ($453>>>0)>($448>>>0);
     if (!($454)) {
      break;
     }
     $455 = ((($0)) + 300|0);
     $456 = HEAP32[$455>>2]|0;
     HEAP32[$443>>2] = $456;
     HEAP32[$455>>2] = $443;
     break L269;
    }
   } while(0);
   $457 = HEAP32[7324]|0;
   $458 = ($457|0)==(0);
   if ($458) {
    $466 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$466 & 127]($443);
    break;
   } else {
    $459 = HEAP32[(29352)>>2]|0;
    $460 = (FUNCTION_TABLE_ii[$459 & 127]($443)|0);
    $461 = HEAP32[14978]|0;
    $462 = (($461) - ($460))|0;
    HEAP32[14978] = $462;
    $463 = HEAP32[14981]|0;
    $464 = (($463) + -1)|0;
    HEAP32[14981] = $464;
    $465 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$465 & 127]($443);
    break;
   }
  }
 } while(0);
 HEAP32[$1>>2] = -1623446221;
 $467 = ((($0)) + 278|0);
 $468 = HEAP8[$467>>0]|0;
 $469 = ($468<<24>>24)==(0);
 do {
  if (!($469)) {
   $470 = ((($0)) + 304|0);
   $471 = HEAP32[$470>>2]|0;
   $472 = ($471|0)==(0|0);
   if ($472) {
    break;
   }
   $473 = HEAP32[7324]|0;
   $474 = ($473|0)==(0);
   if ($474) {
    $482 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$482 & 127]($471);
    break;
   } else {
    $475 = HEAP32[(29352)>>2]|0;
    $476 = (FUNCTION_TABLE_ii[$475 & 127]($471)|0);
    $477 = HEAP32[14978]|0;
    $478 = (($477) - ($476))|0;
    HEAP32[14978] = $478;
    $479 = HEAP32[14981]|0;
    $480 = (($479) + -1)|0;
    HEAP32[14981] = $480;
    $481 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$481 & 127]($471);
    break;
   }
  }
 } while(0);
 $483 = HEAP32[7324]|0;
 $484 = ($483|0)==(0);
 if ($484) {
  $492 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$492 & 127]($0);
  return;
 } else {
  $485 = HEAP32[(29352)>>2]|0;
  $486 = (FUNCTION_TABLE_ii[$485 & 127]($0)|0);
  $487 = HEAP32[14978]|0;
  $488 = (($487) - ($486))|0;
  HEAP32[14978] = $488;
  $489 = HEAP32[14981]|0;
  $490 = (($489) + -1)|0;
  HEAP32[14981] = $490;
  $491 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$491 & 127]($0);
  return;
 }
}
function _sqlite3_finalize($0) {
 $0 = $0|0;
 var $$0$i21 = 0, $$0$i22 = 0, $$1 = 0, $$pre$i$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0;
 var $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0;
 var $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0;
 var $60 = 0, $61 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i = 0, $switch$split2D = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vararg_ptr3 = 0, $vararg_ptr4 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $vararg_buffer1 = sp + 8|0;
 $vararg_buffer = sp;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$1 = 0;
  STACKTOP = sp;return ($$1|0);
 }
 $2 = HEAP32[$0>>2]|0;
 $3 = ($2|0)==(0|0);
 if ($3) {
  _sqlite3_log(21,32391,$vararg_buffer);
  HEAP32[$vararg_buffer1>>2] = 58063;
  $vararg_ptr3 = ((($vararg_buffer1)) + 4|0);
  HEAP32[$vararg_ptr3>>2] = 81403;
  $vararg_ptr4 = ((($vararg_buffer1)) + 8|0);
  HEAP32[$vararg_ptr4>>2] = (31517);
  _sqlite3_log(21,32001,$vararg_buffer1);
  $$1 = 21;
  STACKTOP = sp;return ($$1|0);
 }
 $4 = ((($0)) + 136|0);
 $5 = $4;
 $6 = $5;
 $7 = HEAP32[$6>>2]|0;
 $8 = (($5) + 4)|0;
 $9 = $8;
 $10 = HEAP32[$9>>2]|0;
 $11 = ($10|0)>(0);
 $12 = ($7>>>0)>(0);
 $13 = ($10|0)==(0);
 $14 = $13 & $12;
 $15 = $11 | $14;
 if ($15) {
  _invokeProfileCallback($2,$0);
 }
 $16 = ((($0)) + 20|0);
 $17 = HEAP32[$16>>2]|0;
 $switch$split2D = ($17|0)<(832317811);
 if ($switch$split2D) {
  switch ($17|0) {
  case 770837923:  {
   label = 6;
   break;
  }
  default: {
   $$0$i21 = 0;
  }
  }
 } else {
  switch ($17|0) {
  case 832317811:  {
   label = 6;
   break;
  }
  default: {
   $$0$i21 = 0;
  }
  }
 }
 if ((label|0) == 6) {
  $18 = (_sqlite3VdbeReset($0)|0);
  $$0$i21 = $18;
 }
 $19 = HEAP32[$0>>2]|0;
 _sqlite3VdbeClearObject($19,$0);
 $20 = ((($0)) + 4|0);
 $21 = HEAP32[$20>>2]|0;
 $22 = ($21|0)==(0|0);
 $23 = ((($0)) + 8|0);
 $24 = HEAP32[$23>>2]|0;
 $25 = $21;
 if ($22) {
  $27 = $24;
  $28 = ((($19)) + 4|0);
  HEAP32[$28>>2] = $24;
  $30 = $27;
 } else {
  $26 = ((($21)) + 8|0);
  HEAP32[$26>>2] = $24;
  $$pre$i$i = HEAP32[$23>>2]|0;
  $30 = $$pre$i$i;
 }
 $29 = ($30|0)==(0|0);
 if (!($29)) {
  $31 = ((($30)) + 4|0);
  HEAP32[$31>>2] = $25;
 }
 HEAP32[$16>>2] = 1443283912;
 HEAP32[$0>>2] = 0;
 $32 = ($19|0)==(0|0);
 do {
  if ($32) {
   label = 18;
  } else {
   $33 = ((($19)) + 480|0);
   $34 = HEAP32[$33>>2]|0;
   $35 = ($34|0)==(0|0);
   if (!($35)) {
    _measureAllocationSize($19,$0);
    break;
   }
   $36 = $0;
   $37 = ((($19)) + 304|0);
   $38 = HEAP32[$37>>2]|0;
   $39 = ($38>>>0)>($36>>>0);
   if ($39) {
    label = 18;
   } else {
    $40 = ((($19)) + 308|0);
    $41 = HEAP32[$40>>2]|0;
    $42 = ($41>>>0)>($36>>>0);
    if ($42) {
     $43 = ((($19)) + 300|0);
     $44 = HEAP32[$43>>2]|0;
     HEAP32[$0>>2] = $44;
     HEAP32[$43>>2] = $0;
    } else {
     label = 18;
    }
   }
  }
 } while(0);
 do {
  if ((label|0) == 18) {
   $45 = HEAP32[7324]|0;
   $46 = ($45|0)==(0);
   if ($46) {
    $54 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$54 & 127]($0);
    break;
   } else {
    $47 = HEAP32[(29352)>>2]|0;
    $48 = (FUNCTION_TABLE_ii[$47 & 127]($0)|0);
    $49 = HEAP32[14978]|0;
    $50 = (($49) - ($48))|0;
    HEAP32[14978] = $50;
    $51 = HEAP32[14981]|0;
    $52 = (($51) + -1)|0;
    HEAP32[14981] = $52;
    $53 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$53 & 127]($0);
    break;
   }
  }
 } while(0);
 $55 = ((($2)) + 81|0);
 $56 = HEAP8[$55>>0]|0;
 $57 = ($56<<24>>24)!=(0);
 $58 = ($$0$i21|0)==(3082);
 $or$cond$i = $58 | $57;
 if ($or$cond$i) {
  _apiOomError($2);
  $$0$i22 = 7;
 } else {
  $59 = ((($2)) + 68|0);
  $60 = HEAP32[$59>>2]|0;
  $61 = $60 & $$0$i21;
  $$0$i22 = $61;
 }
 _sqlite3LeaveMutexAndCloseZombie($2);
 $$1 = $$0$i22;
 STACKTOP = sp;return ($$1|0);
}
function _invokeProfileCallback($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0.0, $2 = 0, $20 = 0.0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0;
 var $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $7 = 0, $8 = 0;
 var $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $2 = sp + 16|0;
 $3 = sp + 8|0;
 $4 = sp;
 $5 = HEAP32[$0>>2]|0;
 $6 = HEAP32[$5>>2]|0;
 $7 = ($6|0)>(1);
 if ($7) {
  $8 = ((($5)) + 72|0);
  $9 = HEAP32[$8>>2]|0;
  $10 = ($9|0)==(0|0);
  if ($10) {
   label = 4;
  } else {
   (FUNCTION_TABLE_iii[$9 & 255]($5,$3)|0);
   $11 = $3;
   $12 = $11;
   $13 = HEAP32[$12>>2]|0;
   $14 = (($11) + 4)|0;
   $15 = $14;
   $16 = HEAP32[$15>>2]|0;
   $34 = $13;$35 = $16;
  }
 } else {
  label = 4;
 }
 if ((label|0) == 4) {
  $17 = ((($5)) + 64|0);
  $18 = HEAP32[$17>>2]|0;
  (FUNCTION_TABLE_iii[$18 & 255]($5,$2)|0);
  $19 = +HEAPF64[$2>>3];
  $20 = $19 * 8.64E+7;
  $21 = (~~$20)>>>0;
  $22 = +Math_abs($20) >= 1.0 ? $20 > 0.0 ? (~~+Math_min(+Math_floor($20 / 4294967296.0), 4294967295.0)) >>> 0 : ~~+Math_ceil(($20 - +(~~$20 >>> 0)) / 4294967296.0) >>> 0 : 0;
  $23 = $3;
  $24 = $23;
  HEAP32[$24>>2] = $21;
  $25 = (($23) + 4)|0;
  $26 = $25;
  HEAP32[$26>>2] = $22;
  $34 = $21;$35 = $22;
 }
 $27 = ((($1)) + 136|0);
 $28 = $27;
 $29 = $28;
 $30 = HEAP32[$29>>2]|0;
 $31 = (($28) + 4)|0;
 $32 = $31;
 $33 = HEAP32[$32>>2]|0;
 $36 = (_i64Subtract(($34|0),($35|0),($30|0),($33|0))|0);
 $37 = (getTempRet0() | 0);
 $38 = (___muldi3(($36|0),($37|0),1000000,0)|0);
 $39 = (getTempRet0() | 0);
 $40 = $4;
 $41 = $40;
 HEAP32[$41>>2] = $38;
 $42 = (($40) + 4)|0;
 $43 = $42;
 HEAP32[$43>>2] = $39;
 $44 = ((($0)) + 204|0);
 $45 = HEAP32[$44>>2]|0;
 $46 = ($45|0)==(0|0);
 if (!($46)) {
  $47 = ((($0)) + 208|0);
  $48 = HEAP32[$47>>2]|0;
  $49 = ((($1)) + 188|0);
  $50 = HEAP32[$49>>2]|0;
  FUNCTION_TABLE_viiii[$45 & 127]($48,$50,$38,$39);
 }
 $51 = ((($0)) + 88|0);
 $52 = HEAP8[$51>>0]|0;
 $53 = $52 & 2;
 $54 = ($53<<24>>24)==(0);
 if ($54) {
  $59 = $27;
  $60 = $59;
  HEAP32[$60>>2] = 0;
  $61 = (($59) + 4)|0;
  $62 = $61;
  HEAP32[$62>>2] = 0;
  STACKTOP = sp;return;
 }
 $55 = ((($0)) + 196|0);
 $56 = HEAP32[$55>>2]|0;
 $57 = ((($0)) + 200|0);
 $58 = HEAP32[$57>>2]|0;
 (FUNCTION_TABLE_iiiii[$56 & 127](2,$58,$1,$4)|0);
 $59 = $27;
 $60 = $59;
 HEAP32[$60>>2] = 0;
 $61 = (($59) + 4)|0;
 $62 = $61;
 HEAP32[$62>>2] = 0;
 STACKTOP = sp;return;
}
function _sqlite3ApiExit($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $or$cond = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($0)) + 81|0);
 $3 = HEAP8[$2>>0]|0;
 $4 = ($3<<24>>24)!=(0);
 $5 = ($1|0)==(3082);
 $or$cond = $5 | $4;
 if ($or$cond) {
  _apiOomError($0);
  $$0 = 7;
  return ($$0|0);
 } else {
  $6 = ((($0)) + 68|0);
  $7 = HEAP32[$6>>2]|0;
  $8 = $7 & $1;
  $$0 = $8;
  return ($$0|0);
 }
 return (0)|0;
}
function _sqlite3_reset($0) {
 $0 = $0|0;
 var $$0 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$0 = 0;
  return ($$0|0);
 }
 $2 = HEAP32[$0>>2]|0;
 $3 = ((($0)) + 136|0);
 $4 = $3;
 $5 = $4;
 $6 = HEAP32[$5>>2]|0;
 $7 = (($4) + 4)|0;
 $8 = $7;
 $9 = HEAP32[$8>>2]|0;
 $10 = ($9|0)>(0);
 $11 = ($6>>>0)>(0);
 $12 = ($9|0)==(0);
 $13 = $12 & $11;
 $14 = $10 | $13;
 if ($14) {
  _invokeProfileCallback($2,$0);
 }
 $15 = (_sqlite3VdbeReset($0)|0);
 $16 = ((($0)) + 20|0);
 HEAP32[$16>>2] = 770837923;
 $17 = ((($0)) + 36|0);
 HEAP32[$17>>2] = -1;
 $18 = ((($0)) + 40|0);
 HEAP32[$18>>2] = 0;
 $19 = ((($0)) + 146|0);
 HEAP8[$19>>0] = 2;
 $20 = ((($0)) + 44|0);
 HEAP32[$20>>2] = 0;
 $21 = ((($0)) + 32|0);
 HEAP32[$21>>2] = 1;
 $22 = ((($0)) + 147|0);
 HEAP8[$22>>0] = -1;
 $23 = ((($0)) + 48|0);
 HEAP32[$23>>2] = 0;
 $24 = ((($0)) + 64|0);
 $25 = $24;
 $26 = $25;
 HEAP32[$26>>2] = 0;
 $27 = (($25) + 4)|0;
 $28 = $27;
 HEAP32[$28>>2] = 0;
 $29 = ((($2)) + 81|0);
 $30 = HEAP8[$29>>0]|0;
 $31 = ($30<<24>>24)!=(0);
 $32 = ($15|0)==(3082);
 $or$cond$i = $32 | $31;
 if ($or$cond$i) {
  _apiOomError($2);
  $$0 = 7;
  return ($$0|0);
 } else {
  $33 = ((($2)) + 68|0);
  $34 = HEAP32[$33>>2]|0;
  $35 = $34 & $15;
  $$0 = $35;
  return ($$0|0);
 }
 return (0)|0;
}
function _sqlite3VdbeReset($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0;
 var $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0;
 var $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $vararg_buffer = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $vararg_buffer = sp;
 $1 = HEAP32[$0>>2]|0;
 (_sqlite3VdbeHalt($0)|0);
 $2 = ((($0)) + 36|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ($3|0)>(-1);
 if ($4) {
  (_sqlite3VdbeTransferError($0)|0);
  $5 = ((($0)) + 149|0);
  $6 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
  $7 = $6 & 64;
  $8 = ($7<<16>>16)==(0);
  if (!($8)) {
   $9 = $6 & -4;
   $10 = $9 | 1;
   HEAP8[$5>>0]=$10&255;HEAP8[$5+1>>0]=$10>>8;
  }
 } else {
  $11 = ((($0)) + 40|0);
  $12 = HEAP32[$11>>2]|0;
  $13 = ($12|0)==(0);
  if (!($13)) {
   $14 = ((($0)) + 149|0);
   $15 = HEAPU8[$14>>0]|(HEAPU8[$14+1>>0]<<8);
   $16 = $15 & 3;
   $17 = ($16<<16>>16)==(0);
   if (!($17)) {
    $18 = ((($0)) + 124|0);
    $19 = HEAP32[$18>>2]|0;
    $20 = ($19|0)==(0|0);
    $21 = $20 ? 0 : 31408;
    HEAP32[$vararg_buffer>>2] = $19;
    _sqlite3ErrorWithMsg($1,$12,$21,$vararg_buffer);
   }
  }
 }
 $22 = ((($0)) + 124|0);
 $23 = HEAP32[$22>>2]|0;
 $24 = ($23|0)==(0|0);
 do {
  if (!($24)) {
   $25 = ($1|0)==(0|0);
   if (!($25)) {
    $26 = ((($1)) + 480|0);
    $27 = HEAP32[$26>>2]|0;
    $28 = ($27|0)==(0|0);
    if (!($28)) {
     _measureAllocationSize($1,$23);
     break;
    }
    $29 = $23;
    $30 = ((($1)) + 304|0);
    $31 = HEAP32[$30>>2]|0;
    $32 = ($31>>>0)>($29>>>0);
    if (!($32)) {
     $33 = ((($1)) + 308|0);
     $34 = HEAP32[$33>>2]|0;
     $35 = ($34>>>0)>($29>>>0);
     if ($35) {
      $36 = ((($1)) + 300|0);
      $37 = HEAP32[$36>>2]|0;
      HEAP32[$23>>2] = $37;
      HEAP32[$36>>2] = $23;
      break;
     }
    }
   }
   $38 = HEAP32[7324]|0;
   $39 = ($38|0)==(0);
   if ($39) {
    $47 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$47 & 127]($23);
    break;
   } else {
    $40 = HEAP32[(29352)>>2]|0;
    $41 = (FUNCTION_TABLE_ii[$40 & 127]($23)|0);
    $42 = HEAP32[14978]|0;
    $43 = (($42) - ($41))|0;
    HEAP32[14978] = $43;
    $44 = HEAP32[14981]|0;
    $45 = (($44) + -1)|0;
    HEAP32[14981] = $45;
    $46 = HEAP32[(29344)>>2]|0;
    FUNCTION_TABLE_vi[$46 & 127]($23);
    break;
   }
  }
 } while(0);
 HEAP32[$22>>2] = 0;
 $48 = ((($0)) + 120|0);
 HEAP32[$48>>2] = 0;
 $49 = ((($0)) + 20|0);
 HEAP32[$49>>2] = 1224384374;
 $50 = ((($0)) + 40|0);
 $51 = HEAP32[$50>>2]|0;
 $52 = ((($1)) + 68|0);
 $53 = HEAP32[$52>>2]|0;
 $54 = $53 & $51;
 STACKTOP = sp;return ($54|0);
}
function _sqlite3_clear_bindings($0) {
 $0 = $0|0;
 var $$014 = 0, $$pre = 0, $$pre$phiZ2D = 0, $$pre15 = 0, $$pre16 = 0, $$pre17 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0;
 var $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 16|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = ($2<<16>>16)>(0);
 if ($3) {
  $4 = ((($0)) + 100|0);
  $$pre = HEAP32[$4>>2]|0;
  $$014 = 0;$25 = $2;$6 = $$pre;
  while(1) {
   $5 = (($6) + (($$014*40)|0)|0);
   $7 = (((($6) + (($$014*40)|0)|0)) + 8|0);
   $8 = HEAP16[$7>>1]|0;
   $9 = $8 & 9216;
   $10 = ($9<<16>>16)==(0);
   if ($10) {
    $11 = (((($6) + (($$014*40)|0)|0)) + 24|0);
    $12 = HEAP32[$11>>2]|0;
    $13 = ($12|0)==(0);
    if ($13) {
     $$pre$phiZ2D = $7;$16 = $25;$26 = $6;
    } else {
     label = 5;
    }
   } else {
    label = 5;
   }
   if ((label|0) == 5) {
    label = 0;
    _vdbeMemClear($5);
    $$pre15 = HEAP32[$4>>2]|0;
    $$pre16 = HEAP16[$1>>1]|0;
    $$pre17 = (((($$pre15) + (($$014*40)|0)|0)) + 8|0);
    $$pre$phiZ2D = $$pre17;$16 = $$pre16;$26 = $$pre15;
   }
   HEAP16[$$pre$phiZ2D>>1] = 1;
   $14 = (($$014) + 1)|0;
   $15 = $16 << 16 >> 16;
   $17 = ($14|0)<($15|0);
   if ($17) {
    $$014 = $14;$25 = $16;$6 = $26;
   } else {
    break;
   }
  }
 }
 $18 = ((($0)) + 208|0);
 $19 = HEAP32[$18>>2]|0;
 $20 = ($19|0)==(0);
 if ($20) {
  return 0;
 }
 $21 = ((($0)) + 149|0);
 $22 = HEAPU8[$21>>0]|(HEAPU8[$21+1>>0]<<8);
 $23 = $22 & -4;
 $24 = $23 | 1;
 HEAP8[$21>>0]=$24&255;HEAP8[$21+1>>0]=$24>>8;
 return 0;
}
function _sqlite3_value_blob($0) {
 $0 = $0|0;
 var $$0 = 0, $$0$i = 0, $$phi$trans$insert = 0, $$pre = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0;
 var $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $5 = 0;
 var $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 8|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = $2&65535;
 $4 = $3 & 18;
 $5 = ($4|0)==(0);
 if ($5) {
  $31 = $2 & 514;
  $32 = ($31<<16>>16)==(514);
  if ($32) {
   $33 = ((($0)) + 10|0);
   $34 = HEAP8[$33>>0]|0;
   $35 = ($34<<24>>24)==(1);
   if ($35) {
    $36 = ((($0)) + 16|0);
    $37 = HEAP32[$36>>2]|0;
    $$0 = $37;
    return ($$0|0);
   }
  }
  $38 = $2 & 1;
  $39 = ($38<<16>>16)==(0);
  if (!($39)) {
   $$0 = 0;
   return ($$0|0);
  }
  $40 = (_valueToText($0,1)|0);
  $$0 = $40;
  return ($$0|0);
 }
 $6 = $3 & 16384;
 $7 = ($6|0)==(0);
 $$phi$trans$insert = ((($0)) + 12|0);
 $$pre = HEAP32[$$phi$trans$insert>>2]|0;
 do {
  if ($7) {
   $26 = $2;$28 = $$pre;
  } else {
   $8 = HEAP32[$0>>2]|0;
   $9 = (($8) + ($$pre))|0;
   $10 = ($9|0)<(1);
   if ($10) {
    $11 = $2 & 16;
    $12 = ($11<<16>>16)==(0);
    if ($12) {
     $26 = $2;$28 = $$pre;
     break;
    } else {
     $$0$i = 1;
    }
   } else {
    $$0$i = $9;
   }
   $13 = (_sqlite3VdbeMemGrow($0,$$0$i,1)|0);
   $14 = ($13|0)==(0);
   if ($14) {
    $15 = ((($0)) + 16|0);
    $16 = HEAP32[$15>>2]|0;
    $17 = HEAP32[$$phi$trans$insert>>2]|0;
    $18 = (($16) + ($17)|0);
    $19 = HEAP32[$0>>2]|0;
    _memset(($18|0),0,($19|0))|0;
    $20 = HEAP32[$0>>2]|0;
    $21 = HEAP32[$$phi$trans$insert>>2]|0;
    $22 = (($21) + ($20))|0;
    HEAP32[$$phi$trans$insert>>2] = $22;
    $23 = HEAP16[$1>>1]|0;
    $24 = $23 & -16897;
    HEAP16[$1>>1] = $24;
    $26 = $24;$28 = $22;
    break;
   } else {
    $$0 = 0;
    return ($$0|0);
   }
  }
 } while(0);
 $25 = $26 | 16;
 HEAP16[$1>>1] = $25;
 $27 = ($28|0)==(0);
 if ($27) {
  $$0 = 0;
  return ($$0|0);
 }
 $29 = ((($0)) + 16|0);
 $30 = HEAP32[$29>>2]|0;
 $$0 = $30;
 return ($$0|0);
}
function _sqlite3_value_text($0) {
 $0 = $0|0;
 var $$0$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$0$i = 0;
  return ($$0$i|0);
 }
 $2 = ((($0)) + 8|0);
 $3 = HEAP16[$2>>1]|0;
 $4 = $3 & 514;
 $5 = ($4<<16>>16)==(514);
 if ($5) {
  $6 = ((($0)) + 10|0);
  $7 = HEAP8[$6>>0]|0;
  $8 = ($7<<24>>24)==(1);
  if ($8) {
   $9 = ((($0)) + 16|0);
   $10 = HEAP32[$9>>2]|0;
   $$0$i = $10;
   return ($$0$i|0);
  }
 }
 $11 = $3 & 1;
 $12 = ($11<<16>>16)==(0);
 if (!($12)) {
  $$0$i = 0;
  return ($$0$i|0);
 }
 $13 = (_valueToText($0,1)|0);
 $$0$i = $13;
 return ($$0$i|0);
}
function _sqlite3_value_bytes($0) {
 $0 = $0|0;
 var $$0$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0;
 var $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 8|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = $2 & 2;
 $4 = ($3<<16>>16)==(0);
 if (!($4)) {
  $5 = ((($0)) + 10|0);
  $6 = HEAP8[$5>>0]|0;
  $7 = ($6<<24>>24)==(1);
  if ($7) {
   $8 = ((($0)) + 12|0);
   $9 = HEAP32[$8>>2]|0;
   $$0$i = $9;
   return ($$0$i|0);
  }
 }
 $10 = $2&65535;
 $11 = $10 & 16;
 $12 = ($11|0)==(0);
 if ($12) {
  $19 = $10 & 1;
  $20 = ($19|0)==(0);
  if (!($20)) {
   $$0$i = 0;
   return ($$0$i|0);
  }
  $21 = (_valueBytes($0,1)|0);
  $$0$i = $21;
  return ($$0$i|0);
 } else {
  $13 = $10 & 16384;
  $14 = ($13|0)==(0);
  $15 = ((($0)) + 12|0);
  $16 = HEAP32[$15>>2]|0;
  if ($14) {
   $$0$i = $16;
   return ($$0$i|0);
  }
  $17 = HEAP32[$0>>2]|0;
  $18 = (($17) + ($16))|0;
  $$0$i = $18;
  return ($$0$i|0);
 }
 return (0)|0;
}
function _sqlite3_value_double($0) {
 $0 = $0|0;
 var $$0$i = 0.0, $$idx$i = 0, $$idx$val$i = 0, $$idx5$i = 0, $$idx5$val$i = 0, $$idx6$i = 0, $$idx6$val$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0.0, $16 = 0, $17 = 0, $18 = 0.0, $2 = 0, $3 = 0, $4 = 0;
 var $5 = 0, $6 = 0.0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 8|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = $2&65535;
 $4 = $3 & 8;
 $5 = ($4|0)==(0);
 if (!($5)) {
  $6 = +HEAPF64[$0>>3];
  $$0$i = $6;
  return (+$$0$i);
 }
 $7 = $3 & 4;
 $8 = ($7|0)==(0);
 if (!($8)) {
  $9 = $0;
  $10 = $9;
  $11 = HEAP32[$10>>2]|0;
  $12 = (($9) + 4)|0;
  $13 = $12;
  $14 = HEAP32[$13>>2]|0;
  $15 = (+($11>>>0)) + (4294967296.0*(+($14|0)));
  $$0$i = $15;
  return (+$$0$i);
 }
 $16 = $3 & 18;
 $17 = ($16|0)==(0);
 if ($17) {
  $$0$i = 0.0;
  return (+$$0$i);
 }
 $$idx$i = ((($0)) + 10|0);
 $$idx$val$i = HEAP8[$$idx$i>>0]|0;
 $$idx5$i = ((($0)) + 12|0);
 $$idx5$val$i = HEAP32[$$idx5$i>>2]|0;
 $$idx6$i = ((($0)) + 16|0);
 $$idx6$val$i = HEAP32[$$idx6$i>>2]|0;
 $18 = (+_memRealValue($$idx$val$i,$$idx5$val$i,$$idx6$val$i));
 $$0$i = $18;
 return (+$$0$i);
}
function _sqlite3_value_int($0) {
 $0 = $0|0;
 var $$0$i$off0 = 0, $$idx$i = 0, $$idx$val$i = 0, $$idx8$i = 0, $$idx8$val$i = 0, $$idx9$i = 0, $$idx9$val$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0.0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0;
 var $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 8|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = $2&65535;
 $4 = $3 & 4;
 $5 = ($4|0)==(0);
 if (!($5)) {
  $6 = $0;
  $7 = $6;
  $8 = HEAP32[$7>>2]|0;
  $9 = (($6) + 4)|0;
  $10 = $9;
  $11 = HEAP32[$10>>2]|0;
  $$0$i$off0 = $8;
  return ($$0$i$off0|0);
 }
 $12 = $3 & 8;
 $13 = ($12|0)==(0);
 if (!($13)) {
  $14 = +HEAPF64[$0>>3];
  $15 = (_doubleToInt64($14)|0);
  $16 = (getTempRet0() | 0);
  $$0$i$off0 = $15;
  return ($$0$i$off0|0);
 }
 $17 = $3 & 18;
 $18 = ($17|0)==(0);
 if ($18) {
  $$0$i$off0 = 0;
  return ($$0$i$off0|0);
 }
 $$idx$i = ((($0)) + 10|0);
 $$idx$val$i = HEAP8[$$idx$i>>0]|0;
 $$idx8$i = ((($0)) + 12|0);
 $$idx8$val$i = HEAP32[$$idx8$i>>2]|0;
 $$idx9$i = ((($0)) + 16|0);
 $$idx9$val$i = HEAP32[$$idx9$i>>2]|0;
 $19 = (_memIntValue($$idx$val$i,$$idx8$val$i,$$idx9$val$i)|0);
 $20 = (getTempRet0() | 0);
 $$0$i$off0 = $19;
 return ($$0$i$off0|0);
}
function _sqlite3_value_int64($0) {
 $0 = $0|0;
 var $$idx$i = 0, $$idx$val$i = 0, $$idx8$i = 0, $$idx8$val$i = 0, $$idx9$i = 0, $$idx9$val$i = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0.0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0;
 var $22 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 8|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = $2&65535;
 $4 = $3 & 4;
 $5 = ($4|0)==(0);
 if (!($5)) {
  $6 = $0;
  $7 = $6;
  $8 = HEAP32[$7>>2]|0;
  $9 = (($6) + 4)|0;
  $10 = $9;
  $11 = HEAP32[$10>>2]|0;
  $21 = $11;$22 = $8;
  setTempRet0(($21) | 0);
  return ($22|0);
 }
 $12 = $3 & 8;
 $13 = ($12|0)==(0);
 if (!($13)) {
  $14 = +HEAPF64[$0>>3];
  $15 = (_doubleToInt64($14)|0);
  $16 = (getTempRet0() | 0);
  $21 = $16;$22 = $15;
  setTempRet0(($21) | 0);
  return ($22|0);
 }
 $17 = $3 & 18;
 $18 = ($17|0)==(0);
 if ($18) {
  $21 = 0;$22 = 0;
  setTempRet0(($21) | 0);
  return ($22|0);
 }
 $$idx$i = ((($0)) + 10|0);
 $$idx$val$i = HEAP8[$$idx$i>>0]|0;
 $$idx8$i = ((($0)) + 12|0);
 $$idx8$val$i = HEAP32[$$idx8$i>>2]|0;
 $$idx9$i = ((($0)) + 16|0);
 $$idx9$val$i = HEAP32[$$idx9$i>>2]|0;
 $19 = (_memIntValue($$idx$val$i,$$idx8$val$i,$$idx9$val$i)|0);
 $20 = (getTempRet0() | 0);
 $21 = $20;$22 = $19;
 setTempRet0(($21) | 0);
 return ($22|0);
}
function _sqlite3_value_type($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 8|0);
 $2 = HEAP16[$1>>1]|0;
 $3 = $2 & 31;
 $4 = $3&65535;
 $5 = (816 + ($4)|0);
 $6 = HEAP8[$5>>0]|0;
 $7 = $6&255;
 return ($7|0);
}
function _sqlite3_value_dup($0) {
 $0 = $0|0;
 var $$ = 0, $$0$i16 = 0, $$015 = 0, $$in = 0, $$phi$trans$insert = 0, $$pre = 0, $$pre22 = 0, $1 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0;
 var $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0;
 var $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0;
 var $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0;
 var $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0;
 var $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0;
 var $96 = 0, $97 = 0, $98 = 0, $99 = 0, $phitmp = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$015 = 0;
  return ($$015|0);
 }
 $2 = (_sqlite3_initialize()|0);
 $3 = ($2|0)==(0);
 if (!($3)) {
  $$015 = 0;
  return ($$015|0);
 }
 $4 = HEAP32[7324]|0;
 $5 = ($4|0)==(0);
 if ($5) {
  $44 = HEAP32[(29340)>>2]|0;
  $45 = (FUNCTION_TABLE_ii[$44 & 127](40)|0);
  $46 = ($45|0)==(0|0);
  if ($46) {
   $$015 = 0;
   return ($$015|0);
  } else {
   $$in = $45;
  }
 } else {
  $6 = HEAP32[(29356)>>2]|0;
  $7 = (FUNCTION_TABLE_ii[$6 & 127](40)|0);
  $8 = HEAP32[14985]|0;
  $9 = ($8>>>0)<(40);
  if ($9) {
   HEAP32[14985] = 40;
  }
  $10 = (59064);
  $11 = $10;
  $12 = HEAP32[$11>>2]|0;
  $13 = (($10) + 4)|0;
  $14 = $13;
  $15 = HEAP32[$14>>2]|0;
  $16 = ($15|0)>(0);
  $17 = ($12>>>0)>(0);
  $18 = ($15|0)==(0);
  $19 = $18 & $17;
  $20 = $16 | $19;
  if ($20) {
   $21 = HEAP32[14978]|0;
   $22 = ($7|0)<(0);
   $23 = $22 << 31 >> 31;
   $24 = (_i64Subtract(($12|0),($15|0),($7|0),($23|0))|0);
   $25 = (getTempRet0() | 0);
   $26 = ($25|0)<(0);
   $27 = ($24>>>0)<=($21>>>0);
   $28 = ($25|0)==(0);
   $29 = $28 & $27;
   $30 = $26 | $29;
   $$ = $30&1;
   HEAP32[(59072)>>2] = $$;
  }
  $31 = HEAP32[(29340)>>2]|0;
  $32 = (FUNCTION_TABLE_ii[$31 & 127]($7)|0);
  $33 = ($32|0)==(0|0);
  if ($33) {
   $$015 = 0;
   return ($$015|0);
  }
  $34 = HEAP32[(29352)>>2]|0;
  $35 = (FUNCTION_TABLE_ii[$34 & 127]($32)|0);
  $36 = HEAP32[14978]|0;
  $37 = (($36) + ($35))|0;
  HEAP32[14978] = $37;
  $38 = HEAP32[14982]|0;
  $39 = ($37>>>0)>($38>>>0);
  if ($39) {
   HEAP32[14982] = $37;
  }
  $40 = HEAP32[14981]|0;
  $41 = (($40) + 1)|0;
  HEAP32[14981] = $41;
  $42 = HEAP32[14987]|0;
  $43 = ($41>>>0)>($42>>>0);
  if ($43) {
   HEAP32[14987] = $41;
   $$in = $32;
  } else {
   $$in = $32;
  }
 }
 $47 = ((($$in)) + 20|0);
 ;HEAP32[$47>>2]=0|0;HEAP32[$47+4>>2]=0|0;HEAP32[$47+8>>2]=0|0;HEAP32[$47+12>>2]=0|0;HEAP32[$47+16>>2]=0|0;
 ;HEAP32[$$in>>2]=HEAP32[$0>>2]|0;HEAP32[$$in+4>>2]=HEAP32[$0+4>>2]|0;HEAP32[$$in+8>>2]=HEAP32[$0+8>>2]|0;HEAP32[$$in+12>>2]=HEAP32[$0+12>>2]|0;HEAP32[$$in+16>>2]=HEAP32[$0+16>>2]|0;
 $48 = ((($$in)) + 8|0);
 $49 = HEAP16[$48>>1]|0;
 $50 = $49 & -1025;
 HEAP16[$48>>1] = $50;
 $51 = ((($$in)) + 32|0);
 HEAP32[$51>>2] = 0;
 $52 = $50&65535;
 $53 = $52 & 18;
 $54 = ($53|0)==(0);
 if ($54) {
  $$015 = $$in;
  return ($$015|0);
 }
 $55 = $52 & 58367;
 $56 = $55 | 4096;
 $57 = $56&65535;
 HEAP16[$48>>1] = $57;
 $58 = $52 & 16384;
 $59 = ($58|0)==(0);
 do {
  if ($59) {
   label = 21;
  } else {
   $60 = ((($$in)) + 12|0);
   $61 = HEAP32[$60>>2]|0;
   $62 = HEAP32[$$in>>2]|0;
   $63 = (($62) + ($61))|0;
   $64 = ($63|0)<(1);
   if ($64) {
    $65 = $57 & 16;
    $66 = ($65<<16>>16)==(0);
    if ($66) {
     label = 21;
     break;
    } else {
     $$0$i16 = 1;
    }
   } else {
    $$0$i16 = $63;
   }
   $67 = (_sqlite3VdbeMemGrow($$in,$$0$i16,1)|0);
   $68 = ($67|0)==(0);
   if ($68) {
    $69 = ((($$in)) + 16|0);
    $70 = HEAP32[$69>>2]|0;
    $71 = HEAP32[$60>>2]|0;
    $72 = (($70) + ($71)|0);
    $73 = HEAP32[$$in>>2]|0;
    _memset(($72|0),0,($73|0))|0;
    $74 = HEAP32[$$in>>2]|0;
    $75 = HEAP32[$60>>2]|0;
    $76 = (($75) + ($74))|0;
    HEAP32[$60>>2] = $76;
    $77 = HEAP16[$48>>1]|0;
    $78 = $77 & -16897;
    HEAP16[$48>>1] = $78;
    $$phi$trans$insert = ((($$in)) + 24|0);
    $$pre = HEAP32[$$phi$trans$insert>>2]|0;
    $phitmp = ($$pre|0)==(0);
    if ($phitmp) {
     label = 21;
    } else {
     $79 = ((($$in)) + 16|0);
     $80 = HEAP32[$79>>2]|0;
     $81 = ((($$in)) + 20|0);
     $82 = HEAP32[$81>>2]|0;
     $83 = ($80|0)==($82|0);
     if ($83) {
      $87 = $78;
      label = 23;
     } else {
      label = 21;
     }
    }
   }
  }
 } while(0);
 if ((label|0) == 21) {
  $84 = (_vdbeMemAddTerminator($$in)|0);
  $85 = ($84|0)==(0);
  if ($85) {
   $$pre22 = HEAP16[$48>>1]|0;
   $87 = $$pre22;
   label = 23;
  }
 }
 if ((label|0) == 23) {
  $86 = $87 & -4097;
  HEAP16[$48>>1] = $86;
  $$015 = $$in;
  return ($$015|0);
 }
 $88 = HEAP16[$48>>1]|0;
 $89 = $88 & 9216;
 $90 = ($89<<16>>16)==(0);
 if ($90) {
  $91 = ((($$in)) + 24|0);
  $92 = HEAP32[$91>>2]|0;
  $93 = ($92|0)==(0);
  if (!($93)) {
   label = 26;
  }
 } else {
  label = 26;
 }
 if ((label|0) == 26) {
  _vdbeMemClear($$in);
 }
 $94 = HEAP32[$51>>2]|0;
 $95 = ($94|0)==(0|0);
 if (!($95)) {
  $96 = ((($94)) + 480|0);
  $97 = HEAP32[$96>>2]|0;
  $98 = ($97|0)==(0|0);
  if (!($98)) {
   _measureAllocationSize($94,$$in);
   $$015 = 0;
   return ($$015|0);
  }
  $99 = $$in;
  $100 = ((($94)) + 304|0);
  $101 = HEAP32[$100>>2]|0;
  $102 = ($101>>>0)>($99>>>0);
  if (!($102)) {
   $103 = ((($94)) + 308|0);
   $104 = HEAP32[$103>>2]|0;
   $105 = ($104>>>0)>($99>>>0);
   if ($105) {
    $106 = ((($94)) + 300|0);
    $107 = HEAP32[$106>>2]|0;
    HEAP32[$$in>>2] = $107;
    HEAP32[$106>>2] = $$in;
    $$015 = 0;
    return ($$015|0);
   }
  }
 }
 $108 = HEAP32[7324]|0;
 $109 = ($108|0)==(0);
 if ($109) {
  $117 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$117 & 127]($$in);
  $$015 = 0;
  return ($$015|0);
 } else {
  $110 = HEAP32[(29352)>>2]|0;
  $111 = (FUNCTION_TABLE_ii[$110 & 127]($$in)|0);
  $112 = HEAP32[14978]|0;
  $113 = (($112) - ($111))|0;
  HEAP32[14978] = $113;
  $114 = HEAP32[14981]|0;
  $115 = (($114) + -1)|0;
  HEAP32[14981] = $115;
  $116 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$116 & 127]($$in);
  $$015 = 0;
  return ($$015|0);
 }
 return (0)|0;
}
function _sqlite3_value_free($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0;
 var $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  return;
 }
 $2 = ((($0)) + 8|0);
 $3 = HEAP16[$2>>1]|0;
 $4 = $3 & 9216;
 $5 = ($4<<16>>16)==(0);
 if ($5) {
  $6 = ((($0)) + 24|0);
  $7 = HEAP32[$6>>2]|0;
  $8 = ($7|0)==(0);
  if (!($8)) {
   label = 4;
  }
 } else {
  label = 4;
 }
 if ((label|0) == 4) {
  _vdbeMemClear($0);
 }
 $9 = ((($0)) + 32|0);
 $10 = HEAP32[$9>>2]|0;
 $11 = ($10|0)==(0|0);
 if (!($11)) {
  $12 = ((($10)) + 480|0);
  $13 = HEAP32[$12>>2]|0;
  $14 = ($13|0)==(0|0);
  if (!($14)) {
   _measureAllocationSize($10,$0);
   return;
  }
  $15 = $0;
  $16 = ((($10)) + 304|0);
  $17 = HEAP32[$16>>2]|0;
  $18 = ($17>>>0)>($15>>>0);
  if (!($18)) {
   $19 = ((($10)) + 308|0);
   $20 = HEAP32[$19>>2]|0;
   $21 = ($20>>>0)>($15>>>0);
   if ($21) {
    $22 = ((($10)) + 300|0);
    $23 = HEAP32[$22>>2]|0;
    HEAP32[$0>>2] = $23;
    HEAP32[$22>>2] = $0;
    return;
   }
  }
 }
 $24 = HEAP32[7324]|0;
 $25 = ($24|0)==(0);
 if ($25) {
  $33 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$33 & 127]($0);
  return;
 } else {
  $26 = HEAP32[(29352)>>2]|0;
  $27 = (FUNCTION_TABLE_ii[$26 & 127]($0)|0);
  $28 = HEAP32[14978]|0;
  $29 = (($28) - ($27))|0;
  HEAP32[14978] = $29;
  $30 = HEAP32[14981]|0;
  $31 = (($30) + -1)|0;
  HEAP32[14981] = $31;
  $32 = HEAP32[(29344)>>2]|0;
  FUNCTION_TABLE_vi[$32 & 127]($0);
  return;
 }
}
function _sqlite3_result_blob($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $4 = HEAP32[$0>>2]|0;
 $5 = (_sqlite3VdbeMemSetStr($4,$1,$2,0,$3)|0);
 $6 = ($5|0)==(18);
 if (!($6)) {
  return;
 }
 $7 = ((($0)) + 20|0);
 HEAP32[$7>>2] = 18;
 $8 = HEAP32[$0>>2]|0;
 (_sqlite3VdbeMemSetStr($8,31223,-1,1,0)|0);
 return;
}
function _sqlite3_result_double($0,$1) {
 $0 = $0|0;
 $1 = +$1;
 var $$0$$0$$0$$0$$i$i = 0.0, $$0$$0$$0$$0$2$i$i = 0.0, $$0$$0$$0$$0$3$i$i = 0.0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $2 = sp + 8|0;
 $3 = sp;
 $4 = HEAP32[$0>>2]|0;
 $5 = ((($4)) + 8|0);
 $6 = HEAP16[$5>>1]|0;
 $7 = $6 & 9216;
 $8 = ($7<<16>>16)==(0);
 if ($8) {
  HEAP16[$5>>1] = 1;
 } else {
  _vdbeMemClearExternAndSetNull($4);
 }
 HEAPF64[$2>>3] = $1;
 $$0$$0$$0$$0$2$i$i = +HEAPF64[$2>>3];
 HEAPF64[$3>>3] = $$0$$0$$0$$0$2$i$i;
 $$0$$0$$0$$0$3$i$i = +HEAPF64[$2>>3];
 $$0$$0$$0$$0$$i$i = +HEAPF64[$3>>3];
 $9 = $$0$$0$$0$$0$3$i$i == $$0$$0$$0$$0$$i$i;
 if (!($9)) {
  STACKTOP = sp;return;
 }
 HEAPF64[$4>>3] = $1;
 HEAP16[$5>>1] = 8;
 STACKTOP = sp;return;
}
function _sqlite3_result_error($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $3 = 0, $4 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = ((($0)) + 20|0);
 HEAP32[$3>>2] = 1;
 $4 = HEAP32[$0>>2]|0;
 (_sqlite3VdbeMemSetStr($4,$1,$2,1,(-1))|0);
 return;
}
function _sqlite3VdbeMemSetStr($0,$1,$2,$3,$4) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 $4 = $4|0;
 var $$ = 0, $$0$i$i = 0, $$0$i69 = 0, $$062 = 0, $$063 = 0, $$06470 = 0, $$1 = 0, $$165 = 0, $$167 = 0, $$2 = 0, $$225$i = 0, $$phi$trans$insert = 0, $$pre = 0, $$pre$i = 0, $$pre$phi74Z2D = 0, $$pre26$i = 0, $$pre27$i = 0, $$pre73$pre$phiZZ2D = 0, $$pre75 = 0, $10 = 0;
 var $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0;
 var $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0;
 var $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $16 = 0;
 var $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0;
 var $37 = 0, $38 = 0, $39 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0;
 var $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0;
 var $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0;
 var $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $or$cond = 0, $or$cond$i = 0, $or$cond23$i = 0, $spec$select = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $5 = ($1|0)==(0|0);
 if ($5) {
  $6 = ((($0)) + 8|0);
  $7 = HEAP16[$6>>1]|0;
  $8 = $7 & 9216;
  $9 = ($8<<16>>16)==(0);
  if ($9) {
   HEAP16[$6>>1] = 1;
   $$167 = 0;
   return ($$167|0);
  } else {
   _vdbeMemClearExternAndSetNull($0);
   $$167 = 0;
   return ($$167|0);
  }
 }
 $10 = ((($0)) + 32|0);
 $11 = HEAP32[$10>>2]|0;
 $12 = ($11|0)==(0|0);
 if ($12) {
  $$063 = 1000000000;
 } else {
  $13 = ((($11)) + 108|0);
  $14 = HEAP32[$13>>2]|0;
  $$063 = $14;
 }
 $15 = ($3<<24>>24)==(0);
 $16 = $15 ? 16 : 2;
 $17 = ($2|0)<(0);
 if ($17) {
  $18 = ($3<<24>>24)==(1);
  L14: do {
   if ($18) {
    $19 = (_strlen($1)|0);
    $20 = $19 & 2147483647;
    $$165 = $20;
   } else {
    $21 = ($$063|0)<(0);
    if ($21) {
     $$165 = 0;
    } else {
     $$06470 = 0;
     while(1) {
      $22 = (($1) + ($$06470)|0);
      $23 = HEAP8[$22>>0]|0;
      $24 = $$06470 | 1;
      $25 = (($1) + ($24)|0);
      $26 = HEAP8[$25>>0]|0;
      $27 = $26 | $23;
      $28 = ($27<<24>>24)==(0);
      if ($28) {
       $$165 = $$06470;
       break L14;
      }
      $29 = (($$06470) + 2)|0;
      $30 = ($29|0)>($$063|0);
      if ($30) {
       $$165 = $29;
       break;
      } else {
       $$06470 = $29;
      }
     }
    }
   }
  } while(0);
  $31 = $16 | 512;
  $$062 = $31;$$2 = $$165;
 } else {
  $$062 = $16;$$2 = $2;
 }
 $32 = ($4|0)==((-1)|0);
 do {
  if ($32) {
   $33 = $$062 & 512;
   $34 = ($33<<16>>16)==(0);
   $35 = ($3<<24>>24)==(1);
   $36 = $35 ? 1 : 2;
   $37 = $34 ? 0 : $36;
   $spec$select = (($37) + ($$2))|0;
   $38 = ($$2|0)>($$063|0);
   if ($38) {
    if ($12) {
     $$167 = 18;
     return ($$167|0);
    }
    $39 = ((($11)) + 236|0);
    $40 = HEAP32[$39>>2]|0;
    $41 = ($40|0)==(0|0);
    if ($41) {
     $$167 = 18;
     return ($$167|0);
    }
    $42 = ((($40)) + 12|0);
    HEAP32[$42>>2] = 18;
    $43 = ((($40)) + 36|0);
    $44 = HEAP32[$43>>2]|0;
    $45 = (($44) + 1)|0;
    HEAP32[$43>>2] = $45;
    $$167 = 18;
    return ($$167|0);
   } else {
    $46 = ($spec$select>>>0)>(32);
    $47 = $46 ? $spec$select : 32;
    $48 = ((($0)) + 24|0);
    $49 = HEAP32[$48>>2]|0;
    $50 = ($49|0)<($47|0);
    do {
     if ($50) {
      $58 = (_sqlite3VdbeMemGrow($0,$47,0)|0);
      $59 = ($58|0)==(0);
      if ($59) {
       $$phi$trans$insert = ((($0)) + 16|0);
       $$pre = HEAP32[$$phi$trans$insert>>2]|0;
       $$pre75 = ((($0)) + 8|0);
       $$pre73$pre$phiZZ2D = $$pre75;$60 = $$pre;
       break;
      } else {
       $$167 = 7;
       return ($$167|0);
      }
     } else {
      $51 = ((($0)) + 20|0);
      $52 = HEAP32[$51>>2]|0;
      $53 = ((($0)) + 16|0);
      HEAP32[$53>>2] = $52;
      $54 = ((($0)) + 8|0);
      $55 = HEAP16[$54>>1]|0;
      $56 = $55 & 13;
      HEAP16[$54>>1] = $56;
      $57 = $52;
      $$pre73$pre$phiZZ2D = $54;$60 = $57;
     }
    } while(0);
    _memcpy(($60|0),($1|0),($spec$select|0))|0;
    $$1 = $$062;$$pre$phi74Z2D = $$pre73$pre$phiZZ2D;
    break;
   }
  } else {
   $61 = ((($0)) + 8|0);
   $62 = HEAP16[$61>>1]|0;
   $63 = $62 & 9216;
   $64 = ($63<<16>>16)==(0);
   if ($64) {
    $65 = ((($0)) + 24|0);
    $66 = HEAP32[$65>>2]|0;
    $67 = ($66|0)==(0);
    if (!($67)) {
     label = 27;
    }
   } else {
    label = 27;
   }
   if ((label|0) == 27) {
    _vdbeMemClear($0);
   }
   $68 = ((($0)) + 16|0);
   HEAP32[$68>>2] = $1;
   $69 = ($4|0)==(116|0);
   if (!($69)) {
    $86 = ((($0)) + 36|0);
    HEAP32[$86>>2] = $4;
    $87 = ($4|0)==(0|0);
    $88 = $87 ? 2048 : 1024;
    $89 = $$062&65535;
    $90 = $88 | $89;
    $91 = $90&65535;
    $$1 = $91;$$pre$phi74Z2D = $61;
    break;
   }
   $70 = ((($0)) + 20|0);
   HEAP32[$70>>2] = $1;
   $71 = HEAP32[$10>>2]|0;
   $72 = ($71|0)==(0|0);
   if ($72) {
    label = 32;
   } else {
    $73 = $1;
    $74 = ((($71)) + 304|0);
    $75 = HEAP32[$74>>2]|0;
    $76 = ($75>>>0)>($73>>>0);
    if ($76) {
     label = 32;
    } else {
     $77 = ((($71)) + 308|0);
     $78 = HEAP32[$77>>2]|0;
     $79 = ($78>>>0)>($73>>>0);
     if ($79) {
      $82 = ((($71)) + 276|0);
      $83 = HEAP16[$82>>1]|0;
      $84 = $83&65535;
      $$0$i69 = $84;
     } else {
      label = 32;
     }
    }
   }
   if ((label|0) == 32) {
    $80 = HEAP32[(29352)>>2]|0;
    $81 = (FUNCTION_TABLE_ii[$80 & 127]($1)|0);
    $$0$i69 = $81;
   }
   $85 = ((($0)) + 24|0);
   HEAP32[$85>>2] = $$0$i69;
   $$1 = $$062;$$pre$phi74Z2D = $61;
  }
 } while(0);
 $92 = ((($0)) + 12|0);
 HEAP32[$92>>2] = $$2;
 HEAP16[$$pre$phi74Z2D>>1] = $$1;
 $93 = $15 ? 1 : $3;
 $94 = ((($0)) + 10|0);
 HEAP8[$94>>0] = $93;
 $95 = ($93<<24>>24)!=(1);
 $96 = ($$2|0)>(1);
 $or$cond = $95 & $96;
 do {
  if ($or$cond) {
   $97 = ((($0)) + 16|0);
   $98 = HEAP32[$97>>2]|0;
   $99 = HEAP8[$98>>0]|0;
   $100 = ((($98)) + 1|0);
   $101 = HEAP8[$100>>0]|0;
   $102 = ($99<<24>>24)==(-1);
   $103 = ($101<<24>>24)==(-2);
   $or$cond23$i = $102 & $103;
   if ($or$cond23$i) {
    $$225$i = 2;
   } else {
    $104 = ($101<<24>>24)==(-1);
    $105 = ($99<<24>>24)==(-2);
    $or$cond$i = $105 & $104;
    if ($or$cond$i) {
     $$225$i = 3;
    } else {
     break;
    }
   }
   $106 = $$1&65535;
   $107 = $106 & 18;
   $108 = ($107|0)==(0);
   do {
    if ($108) {
     $137 = $$1;$139 = $$2;$141 = $98;
    } else {
     $109 = $106 & 16384;
     $110 = ($109|0)==(0);
     do {
      if ($110) {
       $152 = $$2;$153 = $$1;
      } else {
       $111 = HEAP32[$0>>2]|0;
       $112 = (($111) + ($$2))|0;
       $113 = ($112|0)<(1);
       if ($113) {
        $114 = $$1 & 16;
        $115 = ($114<<16>>16)==(0);
        if ($115) {
         $152 = $$2;$153 = $$1;
         break;
        } else {
         $$0$i$i = 1;
        }
       } else {
        $$0$i$i = $112;
       }
       $116 = (_sqlite3VdbeMemGrow($0,$$0$i$i,1)|0);
       $117 = ($116|0)==(0);
       if ($117) {
        $118 = HEAP32[$97>>2]|0;
        $119 = HEAP32[$92>>2]|0;
        $120 = (($118) + ($119)|0);
        $121 = HEAP32[$0>>2]|0;
        _memset(($120|0),0,($121|0))|0;
        $122 = HEAP32[$0>>2]|0;
        $123 = HEAP32[$92>>2]|0;
        $124 = (($123) + ($122))|0;
        HEAP32[$92>>2] = $124;
        $125 = HEAP16[$$pre$phi74Z2D>>1]|0;
        $126 = $125 & -16897;
        HEAP16[$$pre$phi74Z2D>>1] = $126;
        $152 = $124;$153 = $126;
        break;
       } else {
        $$167 = 7;
        return ($$167|0);
       }
      }
     } while(0);
     $127 = ((($0)) + 24|0);
     $128 = HEAP32[$127>>2]|0;
     $129 = ($128|0)==(0);
     if (!($129)) {
      $130 = HEAP32[$97>>2]|0;
      $131 = ((($0)) + 20|0);
      $132 = HEAP32[$131>>2]|0;
      $133 = ($130|0)==($132|0);
      if ($133) {
       $137 = $153;$139 = $152;$141 = $130;
       break;
      }
     }
     $134 = (_vdbeMemAddTerminator($0)|0);
     $135 = ($134|0)==(0);
     if ($135) {
      $$pre$i = HEAP16[$$pre$phi74Z2D>>1]|0;
      $$pre26$i = HEAP32[$92>>2]|0;
      $$pre27$i = HEAP32[$97>>2]|0;
      $137 = $$pre$i;$139 = $$pre26$i;$141 = $$pre27$i;
      break;
     } else {
      $$167 = 7;
      return ($$167|0);
     }
    }
   } while(0);
   $136 = $137 & -4097;
   HEAP16[$$pre$phi74Z2D>>1] = $136;
   $138 = (($139) + -2)|0;
   HEAP32[$92>>2] = $138;
   $140 = ((($141)) + 2|0);
   _memmove(($141|0),($140|0),($138|0))|0;
   $142 = HEAP32[$97>>2]|0;
   $143 = HEAP32[$92>>2]|0;
   $144 = (($142) + ($143)|0);
   HEAP8[$144>>0] = 0;
   $145 = HEAP32[$97>>2]|0;
   $146 = HEAP32[$92>>2]|0;
   $147 = (($146) + 1)|0;
   $148 = (($145) + ($147)|0);
   HEAP8[$148>>0] = 0;
   $149 = HEAP16[$$pre$phi74Z2D>>1]|0;
   $150 = $149 | 512;
   HEAP16[$$pre$phi74Z2D>>1] = $150;
   HEAP8[$94>>0] = $$225$i;
  }
 } while(0);
 $151 = ($$2|0)>($$063|0);
 $$ = $151 ? 18 : 0;
 $$167 = $$;
 return ($$167|0);
}
function _sqlite3_result_int($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = HEAP32[$0>>2]|0;
 $3 = ($1|0)<(0);
 $4 = $3 << 31 >> 31;
 $5 = ((($2)) + 8|0);
 $6 = HEAP16[$5>>1]|0;
 $7 = $6 & 9216;
 $8 = ($7<<16>>16)==(0);
 if ($8) {
  $9 = $2;
  $10 = $9;
  HEAP32[$10>>2] = $1;
  $11 = (($9) + 4)|0;
  $12 = $11;
  HEAP32[$12>>2] = $4;
  HEAP16[$5>>1] = 4;
  return;
 } else {
  _vdbeReleaseAndSetInt64($2,$1,$4);
  return;
 }
}
function _sqlite3_result_int64($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = HEAP32[$0>>2]|0;
 $4 = ((($3)) + 8|0);
 $5 = HEAP16[$4>>1]|0;
 $6 = $5 & 9216;
 $7 = ($6<<16>>16)==(0);
 if ($7) {
  $8 = $3;
  $9 = $8;
  HEAP32[$9>>2] = $1;
  $10 = (($8) + 4)|0;
  $11 = $10;
  HEAP32[$11>>2] = $2;
  HEAP16[$4>>1] = 4;
  return;
 } else {
  _vdbeReleaseAndSetInt64($3,$1,$2);
  return;
 }
}
function _sqlite3_result_null($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = HEAP32[$0>>2]|0;
 $2 = ((($1)) + 8|0);
 $3 = HEAP16[$2>>1]|0;
 $4 = $3 & 9216;
 $5 = ($4<<16>>16)==(0);
 if ($5) {
  HEAP16[$2>>1] = 1;
  return;
 } else {
  _vdbeMemClearExternAndSetNull($1);
  return;
 }
}
function _sqlite3_result_text($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $4 = HEAP32[$0>>2]|0;
 $5 = (_sqlite3VdbeMemSetStr($4,$1,$2,1,$3)|0);
 $6 = ($5|0)==(18);
 if (!($6)) {
  return;
 }
 $7 = ((($0)) + 20|0);
 HEAP32[$7>>2] = 18;
 $8 = HEAP32[$0>>2]|0;
 (_sqlite3VdbeMemSetStr($8,31223,-1,1,0)|0);
 return;
}
function _sqlite3_result_value($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i2 = 0, $$pre = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0;
 var $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = HEAP32[$0>>2]|0;
 $3 = ((($2)) + 8|0);
 $4 = HEAP16[$3>>1]|0;
 $5 = $4 & 9216;
 $6 = ($5<<16>>16)==(0);
 if (!($6)) {
  _vdbeMemClearExternAndSetNull($2);
 }
 ;HEAP32[$2>>2]=HEAP32[$1>>2]|0;HEAP32[$2+4>>2]=HEAP32[$1+4>>2]|0;HEAP32[$2+8>>2]=HEAP32[$1+8>>2]|0;HEAP32[$2+12>>2]=HEAP32[$1+12>>2]|0;HEAP32[$2+16>>2]=HEAP32[$1+16>>2]|0;
 $7 = HEAP16[$3>>1]|0;
 $8 = $7 & -1025;
 HEAP16[$3>>1] = $8;
 $9 = $8&65535;
 $10 = $9 & 18;
 $11 = ($10|0)==(0);
 if ($11) {
  return;
 }
 $12 = ((($1)) + 8|0);
 $13 = HEAP16[$12>>1]|0;
 $14 = $13 & 2048;
 $15 = ($14<<16>>16)==(0);
 if (!($15)) {
  return;
 }
 $16 = $9 | 4096;
 $17 = $16&65535;
 HEAP16[$3>>1] = $17;
 $18 = $9 & 16384;
 $19 = ($18|0)==(0);
 do {
  if ($19) {
   $51 = $17;
  } else {
   $20 = ((($2)) + 12|0);
   $21 = HEAP32[$20>>2]|0;
   $22 = HEAP32[$2>>2]|0;
   $23 = (($22) + ($21))|0;
   $24 = ($23|0)<(1);
   if ($24) {
    $25 = $17 & 16;
    $26 = ($25<<16>>16)==(0);
    if ($26) {
     $51 = $17;
     break;
    } else {
     $$0$i2 = 1;
    }
   } else {
    $$0$i2 = $23;
   }
   $27 = (_sqlite3VdbeMemGrow($2,$$0$i2,1)|0);
   $28 = ($27|0)==(0);
   if ($28) {
    $29 = ((($2)) + 16|0);
    $30 = HEAP32[$29>>2]|0;
    $31 = HEAP32[$20>>2]|0;
    $32 = (($30) + ($31)|0);
    $33 = HEAP32[$2>>2]|0;
    _memset(($32|0),0,($33|0))|0;
    $34 = HEAP32[$2>>2]|0;
    $35 = HEAP32[$20>>2]|0;
    $36 = (($35) + ($34))|0;
    HEAP32[$20>>2] = $36;
    $37 = HEAP16[$3>>1]|0;
    $38 = $37 & -16897;
    HEAP16[$3>>1] = $38;
    $51 = $38;
    break;
   } else {
    return;
   }
  }
 } while(0);
 $39 = ((($2)) + 24|0);
 $40 = HEAP32[$39>>2]|0;
 $41 = ($40|0)==(0);
 if ($41) {
  label = 12;
 } else {
  $42 = ((($2)) + 16|0);
  $43 = HEAP32[$42>>2]|0;
  $44 = ((($2)) + 20|0);
  $45 = HEAP32[$44>>2]|0;
  $46 = ($43|0)==($45|0);
  if ($46) {
   $50 = $51;
  } else {
   label = 12;
  }
 }
 do {
  if ((label|0) == 12) {
   $47 = (_vdbeMemAddTerminator($2)|0);
   $48 = ($47|0)==(0);
   if ($48) {
    $$pre = HEAP16[$3>>1]|0;
    $50 = $$pre;
    break;
   } else {
    return;
   }
  }
 } while(0);
 $49 = $50 & -4097;
 HEAP16[$3>>1] = $49;
 return;
}
function _sqlite3_result_error_nomem($0) {
 $0 = $0|0;
 var $$pre = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = HEAP32[$0>>2]|0;
 $2 = ((($1)) + 8|0);
 $3 = HEAP16[$2>>1]|0;
 $4 = $3 & 9216;
 $5 = ($4<<16>>16)==(0);
 if ($5) {
  HEAP16[$2>>1] = 1;
  $8 = $1;
 } else {
  _vdbeMemClearExternAndSetNull($1);
  $$pre = HEAP32[$0>>2]|0;
  $8 = $$pre;
 }
 $6 = ((($0)) + 20|0);
 HEAP32[$6>>2] = 7;
 $7 = ((($8)) + 32|0);
 $9 = HEAP32[$7>>2]|0;
 $10 = ((($9)) + 81|0);
 $11 = HEAP8[$10>>0]|0;
 $12 = ($11<<24>>24)==(0);
 if (!($12)) {
  return;
 }
 $13 = ((($9)) + 82|0);
 $14 = HEAP8[$13>>0]|0;
 $15 = ($14<<24>>24)==(0);
 if (!($15)) {
  return;
 }
 HEAP8[$10>>0] = 1;
 $16 = ((($9)) + 180|0);
 $17 = HEAP32[$16>>2]|0;
 $18 = ($17|0)>(0);
 if ($18) {
  $19 = ((($9)) + 264|0);
  HEAP32[$19>>2] = 1;
 }
 $20 = ((($9)) + 272|0);
 $21 = HEAP32[$20>>2]|0;
 $22 = (($21) + 1)|0;
 HEAP32[$20>>2] = $22;
 $23 = ((($9)) + 236|0);
 $24 = HEAP32[$23>>2]|0;
 $25 = ($24|0)==(0|0);
 if ($25) {
  return;
 }
 $26 = ((($24)) + 12|0);
 HEAP32[$26>>2] = 7;
 return;
}
function _sqlite3OomFault($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ((($0)) + 81|0);
 $2 = HEAP8[$1>>0]|0;
 $3 = ($2<<24>>24)==(0);
 if (!($3)) {
  return;
 }
 $4 = ((($0)) + 82|0);
 $5 = HEAP8[$4>>0]|0;
 $6 = ($5<<24>>24)==(0);
 if (!($6)) {
  return;
 }
 HEAP8[$1>>0] = 1;
 $7 = ((($0)) + 180|0);
 $8 = HEAP32[$7>>2]|0;
 $9 = ($8|0)>(0);
 if ($9) {
  $10 = ((($0)) + 264|0);
  HEAP32[$10>>2] = 1;
 }
 $11 = ((($0)) + 272|0);
 $12 = HEAP32[$11>>2]|0;
 $13 = (($12) + 1)|0;
 HEAP32[$11>>2] = $13;
 $14 = ((($0)) + 236|0);
 $15 = HEAP32[$14>>2]|0;
 $16 = ($15|0)==(0|0);
 if ($16) {
  return;
 }
 $17 = ((($15)) + 12|0);
 HEAP32[$17>>2] = 7;
 return;
}
function _sqlite3_step($0) {
 $0 = $0|0;
 var $$$i$i$i = 0, $$0 = 0, $$0$i$add$i$i = 0, $$0$i$i = 0, $$0$i$idx$i$i = 0, $$0$i190$i$i = 0, $$0$i40 = 0, $$0$i41 = 0, $$0$i4355 = 0, $$0$i5161 = 0, $$0$i52 = 0, $$01$i$i = 0, $$0153$lcssa$us$i$i = 0, $$0153209$us$i$i = 0, $$0154223$i$i = 0, $$0156207$i$i = 0, $$0156207$i$i$us = 0, $$0156207$us$i$i = 0, $$0168$i$i = 0, $$0175222$i$i = 0;
 var $$0185$i$i = 0, $$02527$i$i = 0, $$028$i$i = 0, $$039 = 0, $$06973$i$i$i = 0, $$07074$i$i$i = 0, $$1$i = 0, $$1$i$add$i$i = 0, $$1$i$add$us$i$i = 0, $$1$i$idx$i$i = 0, $$1$i$idx$us$i$i = 0, $$1$i$ptr$i$i = 0, $$1$i$ptr$us$i$i = 0, $$1$i189$i$i = 0, $$1155$i$i = 0, $$1160$i$i = 0, $$1160$lcssa204$i$i = 0, $$1160$us$i$i = 0, $$1163$i$i = 0, $$1176$lcssa$i$i = 0;
 var $$1176$lcssa$i$i$us = 0, $$1176$lcssa$us$i$i = 0, $$1176206$i$i = 0, $$1176206$i$i$us = 0, $$1176206$us$i$i = 0, $$1180$i$i = 0, $$2$i$i = 0, $$2$i57 = 0, $$2164217$us$i$i = 0, $$2177$i$i = 0, $$2177$lcssa205$i$i = 0, $$2177$us$i$i = 0, $$2181216$us$i$i = 0, $$2218$us$i$i = 0, $$3$i = 0, $$5$us$i$i = 0, $$5167$us$i$i = 0, $$5184$us$i$i = 0, $$idx$val$i = 0, $$idx$val$i47 = 0;
 var $$idx$val$idx$val$i$i = 0, $$idx18$val$i = 0, $$idx26$i$i = 0, $$idx26$val$i$i = 0, $$lcssa$i$i = 0, $$lcssa$i$i$us = 0, $$lcssa$us$i$i = 0, $$phi$trans$insert$i$i = 0, $$phi$trans$insert253$i$i = 0, $$pre = 0, $$pre$i$i = 0, $$pre$i$i$i = 0, $$pre$i$i$i50 = 0, $$pre$i52$i = 0, $$pre$phi$i$iZ2D = 0, $$pre$phi257$i$iZ2D = 0, $$pre$phi259$i$iZ2D = 0, $$pre$phi262$i$iZ2D = 0, $$pre250$i$i = 0, $$pre251$i$i = 0;
 var $$pre252$i$i = 0, $$pre254$i$i = 0, $$pre255$i$i = 0, $$pre256$i$i = 0, $$pre258$i$i = 0, $$pre261$i$i = 0, $$pre78$i$i$i = 0, $$ptr201$ptr$i$i = 0, $$sink = 0, $1 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0;
 var $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0;
 var $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0;
 var $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0;
 var $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0;
 var $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0;
 var $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0;
 var $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0;
 var $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0;
 var $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0;
 var $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0;
 var $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0;
 var $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0;
 var $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0;
 var $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $351$phi = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0;
 var $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0;
 var $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0;
 var $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0;
 var $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0;
 var $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0;
 var $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0;
 var $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0;
 var $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0, $5 = 0, $50 = 0, $500 = 0.0, $501 = 0, $502 = 0, $503 = 0;
 var $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $509 = 0, $51 = 0, $510 = 0, $511 = 0, $512 = 0, $513 = 0, $514 = 0, $515 = 0, $516 = 0, $517 = 0, $518 = 0, $519 = 0, $52 = 0, $520 = 0, $521 = 0;
 var $522 = 0, $523 = 0, $524 = 0.0, $525 = 0, $526 = 0, $527 = 0, $528 = 0, $529 = 0, $53 = 0, $530 = 0, $531 = 0, $532 = 0, $533 = 0, $534 = 0, $535 = 0, $536 = 0, $537 = 0, $538 = 0, $539 = 0, $54 = 0;
 var $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $545 = 0, $546 = 0, $547 = 0, $548 = 0, $549 = 0, $55 = 0, $550 = 0, $551 = 0, $552 = 0, $553 = 0, $554 = 0, $555 = 0, $556 = 0, $557 = 0, $558 = 0;
 var $559 = 0, $56 = 0, $560 = 0, $561 = 0, $562 = 0, $563 = 0, $564 = 0, $565 = 0, $566 = 0, $567 = 0, $568 = 0, $569 = 0, $57 = 0, $570 = 0, $571 = 0, $572 = 0, $573 = 0, $574 = 0, $575 = 0, $576 = 0;
 var $577 = 0, $578 = 0, $579 = 0, $58 = 0, $580 = 0, $581 = 0, $582 = 0, $583 = 0, $584 = 0, $585 = 0, $586 = 0, $587 = 0, $588 = 0, $589 = 0, $59 = 0, $590 = 0, $591 = 0, $592 = 0, $593 = 0, $594 = 0;
 var $595 = 0, $596 = 0, $597 = 0, $598 = 0, $599 = 0, $6 = 0, $60 = 0, $600 = 0, $601 = 0, $602 = 0, $603 = 0, $604 = 0, $605 = 0, $606 = 0, $607 = 0, $608 = 0, $609 = 0, $61 = 0, $610 = 0, $611 = 0;
 var $612 = 0, $613 = 0, $614 = 0, $615 = 0, $616 = 0, $617 = 0, $618 = 0, $619 = 0, $62 = 0, $620 = 0, $621 = 0, $622 = 0, $623 = 0, $624 = 0, $625 = 0, $626 = 0, $627 = 0, $628 = 0, $629 = 0, $63 = 0;
 var $630 = 0, $631 = 0, $632 = 0, $633 = 0, $634 = 0, $635 = 0, $636 = 0, $637 = 0, $638 = 0, $639 = 0, $64 = 0, $640 = 0, $641 = 0, $642 = 0, $643 = 0, $644 = 0, $645 = 0, $646 = 0, $647 = 0, $648 = 0;
 var $649 = 0, $65 = 0, $650 = 0, $651 = 0, $652 = 0, $653 = 0, $654 = 0, $655 = 0, $656 = 0, $657 = 0, $658 = 0, $659 = 0, $66 = 0, $660 = 0, $661 = 0, $662 = 0, $663 = 0, $664 = 0, $665 = 0, $666 = 0;
 var $667 = 0, $668 = 0, $669 = 0, $67 = 0, $670 = 0, $671 = 0, $672 = 0, $673 = 0, $674 = 0, $675 = 0, $676 = 0, $677 = 0, $678 = 0, $679 = 0, $68 = 0, $680 = 0, $681 = 0, $682 = 0, $683 = 0, $684 = 0;
 var $685 = 0, $686 = 0, $687 = 0, $688 = 0, $689 = 0, $69 = 0, $690 = 0, $691 = 0, $692 = 0, $693 = 0, $694 = 0, $695 = 0, $696 = 0, $697 = 0, $698 = 0, $699 = 0, $7 = 0, $70 = 0, $700 = 0, $701 = 0;
 var $702 = 0, $703 = 0, $704 = 0, $705 = 0, $706 = 0, $707 = 0, $708 = 0, $709 = 0, $71 = 0, $710 = 0, $711 = 0, $712 = 0, $713 = 0, $714 = 0, $715 = 0, $716 = 0, $717 = 0, $718 = 0, $719 = 0, $72 = 0;
 var $720 = 0, $721 = 0, $722 = 0, $723 = 0, $724 = 0, $725 = 0, $726 = 0, $727 = 0, $728 = 0, $729 = 0, $73 = 0, $730 = 0, $731 = 0, $732 = 0, $733 = 0, $734 = 0, $735 = 0, $736 = 0, $737 = 0, $738 = 0;
 var $739 = 0, $74 = 0, $740 = 0, $741 = 0, $742 = 0, $743 = 0, $744 = 0, $745 = 0, $746 = 0, $747 = 0, $748 = 0, $749 = 0, $75 = 0, $750 = 0, $751 = 0, $752 = 0, $753 = 0, $754 = 0, $755 = 0, $756 = 0;
 var $757 = 0, $758 = 0, $759 = 0, $76 = 0, $760 = 0, $761 = 0, $762 = 0, $763 = 0, $764 = 0, $765 = 0, $766 = 0, $767 = 0, $768 = 0, $769 = 0, $77 = 0, $770 = 0, $771 = 0, $772 = 0, $773 = 0, $774 = 0;
 var $775 = 0, $776 = 0, $777 = 0, $778 = 0, $779 = 0, $78 = 0, $780 = 0, $781 = 0, $782 = 0, $783 = 0, $784 = 0, $785 = 0, $786 = 0, $787 = 0, $788 = 0, $789 = 0, $79 = 0, $790 = 0, $791 = 0, $792 = 0;
 var $793 = 0, $794 = 0, $795 = 0, $796 = 0, $797 = 0, $798 = 0, $799 = 0, $8 = 0, $80 = 0, $800 = 0, $801 = 0, $802 = 0, $803 = 0, $804 = 0, $805 = 0, $806 = 0, $807 = 0, $808 = 0, $809 = 0, $81 = 0;
 var $810 = 0, $811 = 0, $812 = 0, $813 = 0, $814 = 0, $815 = 0, $816 = 0, $817 = 0, $818 = 0, $819 = 0, $82 = 0, $820 = 0, $821 = 0, $822 = 0, $823 = 0, $824 = 0, $825 = 0, $826 = 0, $827 = 0, $828 = 0;
 var $829 = 0, $83 = 0, $830 = 0, $831 = 0, $832 = 0, $833 = 0, $834 = 0, $835 = 0, $836 = 0, $837 = 0, $838 = 0, $839 = 0, $84 = 0, $840 = 0, $841 = 0, $842 = 0, $843 = 0, $844 = 0, $845 = 0, $846 = 0;
 var $847 = 0, $848 = 0, $849 = 0, $85 = 0, $850 = 0, $851 = 0, $852 = 0, $853 = 0, $854 = 0, $855 = 0, $856 = 0, $857 = 0, $858 = 0, $859 = 0, $86 = 0, $860 = 0, $861 = 0, $862 = 0, $863 = 0, $864 = 0;
 var $865 = 0, $866 = 0, $867 = 0, $868 = 0, $869 = 0, $87 = 0, $870 = 0, $871 = 0, $872 = 0, $873 = 0, $874 = 0, $875 = 0, $876 = 0, $877 = 0, $878 = 0, $879 = 0, $88 = 0, $880 = 0, $881 = 0, $882 = 0;
 var $883 = 0, $884 = 0, $885 = 0, $886 = 0, $887 = 0, $888 = 0, $889 = 0, $89 = 0.0, $890 = 0, $891 = 0, $892 = 0, $893 = 0, $894 = 0, $895 = 0, $896 = 0, $897 = 0, $898 = 0, $899 = 0, $9 = 0, $90 = 0.0;
 var $900 = 0, $901 = 0, $902 = 0, $903 = 0, $904 = 0, $905 = 0, $906 = 0, $907 = 0, $908 = 0, $909 = 0, $91 = 0, $910 = 0, $911 = 0, $912 = 0, $913 = 0, $914 = 0, $915 = 0, $916 = 0, $917 = 0, $918 = 0;
 var $919 = 0, $92 = 0, $920 = 0, $921 = 0, $922 = 0, $923 = 0, $924 = 0, $925 = 0, $926 = 0, $927 = 0, $928 = 0, $929 = 0, $93 = 0, $930 = 0, $931 = 0, $932 = 0, $933 = 0, $934 = 0, $935 = 0, $936 = 0;
 var $937 = 0, $938 = 0, $939 = 0, $94 = 0, $940 = 0, $941 = 0, $942 = 0, $943 = 0, $944 = 0, $945 = 0, $946 = 0, $947 = 0, $948 = 0, $949 = 0, $95 = 0, $950 = 0, $951 = 0, $952 = 0, $953 = 0, $954 = 0;
 var $955 = 0, $956 = 0, $957 = 0, $958 = 0, $959 = 0, $96 = 0, $960 = 0, $961 = 0, $962 = 0, $963 = 0, $964 = 0, $965 = 0, $966 = 0, $967 = 0, $968 = 0, $969 = 0, $97 = 0, $970 = 0, $971 = 0, $972 = 0;
 var $973 = 0, $974 = 0, $975 = 0, $976 = 0, $977 = 0, $978 = 0, $979 = 0, $98 = 0, $980 = 0, $981 = 0, $99 = 0, $exitcond$i$i = 0, $exitcond$i$i$i = 0, $exitcond244$i$i = 0, $exitcond245$i$i = 0, $exitcond247$i$i = 0, $or$cond$i = 0, $or$cond$i$i = 0, $or$cond$i$i$i = 0, $or$cond$i$i44 = 0;
 var $or$cond$i51$i = 0, $spec$select$i$i$i = 0, $switch$split2D = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vararg_buffer12 = 0, $vararg_buffer16 = 0, $vararg_buffer19 = 0, $vararg_buffer23 = 0, $vararg_buffer26 = 0, $vararg_buffer29 = 0, $vararg_buffer3 = 0, $vararg_buffer32 = 0, $vararg_buffer35 = 0, $vararg_buffer38 = 0, $vararg_buffer41 = 0, $vararg_buffer44 = 0, $vararg_buffer46 = 0, $vararg_buffer49 = 0, $vararg_buffer7 = 0;
 var $vararg_buffer9 = 0, $vararg_ptr15 = 0, $vararg_ptr22 = 0, $vararg_ptr5 = 0, $vararg_ptr6 = 0, dest = 0, label = 0, sp = 0, src = 0, stop = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 384|0;
 $vararg_buffer49 = sp + 368|0;
 $vararg_buffer46 = sp + 360|0;
 $vararg_buffer44 = sp + 352|0;
 $vararg_buffer41 = sp + 344|0;
 $vararg_buffer38 = sp + 336|0;
 $vararg_buffer35 = sp + 328|0;
 $vararg_buffer32 = sp + 320|0;
 $vararg_buffer29 = sp + 312|0;
 $vararg_buffer26 = sp + 304|0;
 $vararg_buffer23 = sp + 296|0;
 $vararg_buffer19 = sp + 288|0;
 $vararg_buffer16 = sp + 280|0;
 $vararg_buffer12 = sp + 272|0;
 $vararg_buffer9 = sp + 264|0;
 $vararg_buffer7 = sp + 256|0;
 $vararg_buffer3 = sp + 240|0;
 $vararg_buffer1 = sp + 232|0;
 $vararg_buffer = sp + 224|0;
 $1 = sp;
 $2 = sp + 372|0;
 $3 = ($0|0)==(0|0);
 do {
  if ($3) {
   _sqlite3_log(21,32555,$vararg_buffer);
  } else {
   $$idx$val$i = HEAP32[$0>>2]|0;
   $4 = ($$idx$val$i|0)==(0|0);
   if ($4) {
    _sqlite3_log(21,32391,$vararg_buffer1);
    break;
   }
   $5 = ((($0)) + 149|0);
   $6 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
   $7 = $6 & -17;
   HEAP8[$5>>0]=$7&255;HEAP8[$5+1>>0]=$7>>8;
   $8 = ((($0)) + 20|0);
   $9 = ((($0)) + 136|0);
   $10 = ((($0)) + 36|0);
   $11 = ((($0)) + 40|0);
   $12 = ((($0)) + 146|0);
   $13 = ((($0)) + 44|0);
   $14 = ((($0)) + 32|0);
   $15 = ((($0)) + 147|0);
   $16 = ((($0)) + 48|0);
   $17 = ((($0)) + 64|0);
   $18 = ((($0)) + 148|0);
   $19 = ((($0)) + 88|0);
   $20 = ((($0)) + 188|0);
   $21 = ((($0)) + 120|0);
   $22 = ((($0)) + 8|0);
   $23 = ((($0)) + 4|0);
   $24 = ((($0)) + 208|0);
   $25 = ((($0)) + 160|0);
   $26 = ((($0)) + 180|0);
   $27 = ((($0)) + 108|0);
   $28 = ((($0)) + 100|0);
   $29 = ((($0)) + 104|0);
   $30 = ((($1)) + 4|0);
   $31 = ((($1)) + 8|0);
   $32 = ((($1)) + 12|0);
   $33 = ((($1)) + 16|0);
   $34 = ((($1)) + 21|0);
   $35 = ((($0)) + 144|0);
   $$039 = 0;
   L7: while(1) {
    $36 = HEAP32[$8>>2]|0;
    $37 = ($36|0)==(770837923);
    if (!($37)) {
     $38 = HEAP32[$0>>2]|0;
     $39 = $9;
     $40 = $39;
     $41 = HEAP32[$40>>2]|0;
     $42 = (($39) + 4)|0;
     $43 = $42;
     $44 = HEAP32[$43>>2]|0;
     $45 = ($44|0)>(0);
     $46 = ($41>>>0)>(0);
     $47 = ($44|0)==(0);
     $48 = $47 & $46;
     $49 = $45 | $48;
     if ($49) {
      _invokeProfileCallback($38,$0);
     }
     $50 = (_sqlite3VdbeReset($0)|0);
     HEAP32[$8>>2] = 770837923;
     HEAP32[$10>>2] = -1;
     HEAP32[$11>>2] = 0;
     HEAP8[$12>>0] = 2;
     HEAP32[$13>>2] = 0;
     HEAP32[$14>>2] = 1;
     HEAP8[$15>>0] = -1;
     HEAP32[$16>>2] = 0;
     $51 = $17;
     $52 = $51;
     HEAP32[$52>>2] = 0;
     $53 = (($51) + 4)|0;
     $54 = $53;
     HEAP32[$54>>2] = 0;
     $55 = ((($38)) + 81|0);
     $56 = HEAP8[$55>>0]|0;
     $57 = ($56<<24>>24)!=(0);
     $58 = ($50|0)==(3082);
     $or$cond$i$i$i = $58 | $57;
     if ($or$cond$i$i$i) {
      _apiOomError($38);
     }
    }
    $59 = HEAP32[$0>>2]|0;
    $60 = ((($59)) + 81|0);
    $61 = HEAP8[$60>>0]|0;
    $62 = ($61<<24>>24)==(0);
    if (!($62)) {
     label = 13;
     break;
    }
    $63 = HEAP32[$10>>2]|0;
    $64 = ($63|0)<(0);
    do {
     if ($64) {
      $65 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
      $66 = $65 & 3;
      $67 = ($66<<16>>16)==(0);
      if (!($67)) {
       HEAP32[$11>>2] = 17;
       $$2$i57 = 1;
       label = 246;
       break;
      }
      $68 = ((($59)) + 168|0);
      $69 = HEAP32[$68>>2]|0;
      $70 = ($69|0)==(0);
      if ($70) {
       $71 = ((($59)) + 264|0);
       HEAP32[$71>>2] = 0;
      }
      $72 = ((($59)) + 88|0);
      $73 = HEAP8[$72>>0]|0;
      $74 = $73 & -126;
      $75 = ($74<<24>>24)==(0);
      do {
       if (!($75)) {
        $76 = ((($59)) + 165|0);
        $77 = HEAP8[$76>>0]|0;
        $78 = ($77<<24>>24)==(0);
        if ($78) {
         $79 = HEAP32[$20>>2]|0;
         $80 = ($79|0)==(0|0);
         if (!($80)) {
          $81 = HEAP32[$59>>2]|0;
          $82 = HEAP32[$81>>2]|0;
          $83 = ($82|0)>(1);
          if ($83) {
           $84 = ((($81)) + 72|0);
           $85 = HEAP32[$84>>2]|0;
           $86 = ($85|0)==(0|0);
           if (!($86)) {
            (FUNCTION_TABLE_iii[$85 & 255]($81,$9)|0);
            break;
           }
          }
          $87 = ((($81)) + 64|0);
          $88 = HEAP32[$87>>2]|0;
          (FUNCTION_TABLE_iii[$88 & 255]($81,$1)|0);
          $89 = +HEAPF64[$1>>3];
          $90 = $89 * 8.64E+7;
          $91 = (~~$90)>>>0;
          $92 = +Math_abs($90) >= 1.0 ? $90 > 0.0 ? (~~+Math_min(+Math_floor($90 / 4294967296.0), 4294967295.0)) >>> 0 : ~~+Math_ceil(($90 - +(~~$90 >>> 0)) / 4294967296.0) >>> 0 : 0;
          $93 = $9;
          $94 = $93;
          HEAP32[$94>>2] = $91;
          $95 = (($93) + 4)|0;
          $96 = $95;
          HEAP32[$96>>2] = $92;
         }
        }
       }
      } while(0);
      $97 = HEAP32[$68>>2]|0;
      $98 = (($97) + 1)|0;
      HEAP32[$68>>2] = $98;
      $99 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
      $100 = $99 & 256;
      $101 = ($100<<16>>16)==(0);
      if ($101) {
       $102 = ((($59)) + 176|0);
       $103 = HEAP32[$102>>2]|0;
       $104 = (($103) + 1)|0;
       HEAP32[$102>>2] = $104;
       $$pre = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
       $106 = $$pre;
      } else {
       $106 = $99;
      }
      $105 = $106 & 512;
      $107 = ($105<<16>>16)==(0);
      if (!($107)) {
       $108 = ((($59)) + 172|0);
       $109 = HEAP32[$108>>2]|0;
       $110 = (($109) + 1)|0;
       HEAP32[$108>>2] = $110;
      }
      HEAP32[$10>>2] = 0;
      label = 31;
     } else {
      label = 31;
     }
    } while(0);
    if ((label|0) == 31) {
     label = 0;
     $111 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
     $112 = $111 & 12;
     $113 = ($112<<16>>16)==(0);
     L42: do {
      if ($113) {
       $704 = ((($59)) + 180|0);
       $705 = HEAP32[$704>>2]|0;
       $706 = (($705) + 1)|0;
       HEAP32[$704>>2] = $706;
       $707 = (_sqlite3VdbeExec($0)|0);
       $708 = HEAP32[$704>>2]|0;
       $709 = (($708) + -1)|0;
       HEAP32[$704>>2] = $709;
       $710 = ($707|0)==(100);
       if ($710) {
        $$1$i = 100;
       } else {
        $$0$i4355 = $707;
        label = 226;
       }
      } else {
       $114 = HEAP32[$0>>2]|0;
       $115 = HEAP32[$19>>2]|0;
       $$ptr201$ptr$i$i = ((($115)) + 40|0);
       $116 = ($112<<16>>16)==(4);
       if ($116) {
        $980 = 1;
       } else {
        $117 = ((($114)) + 32|0);
        $118 = $117;
        $119 = $118;
        $120 = HEAP32[$119>>2]|0;
        $121 = (($118) + 4)|0;
        $122 = $121;
        $123 = HEAP32[$122>>2]|0;
        $124 = $120 & 16777216;
        $125 = ($124|0)!=(0);
        $126 = (0)!=(0);
        $127 = $125 | $126;
        $980 = $127;
       }
       $128 = ((($115)) + 72|0);
       $129 = HEAP32[$128>>2]|0;
       $130 = ((($129)) + 480|0);
       $131 = HEAP32[$130>>2]|0;
       $132 = ($131|0)==(0|0);
       $133 = ($129|0)==(0|0);
       $134 = ((($129)) + 304|0);
       $135 = ((($129)) + 308|0);
       $136 = ((($129)) + 300|0);
       L48: do {
        if ($132) {
         if ($133) {
          $$1$i$idx$us$i$i = 1;
          while(1) {
           $180 = (((($115) + (($$1$i$idx$us$i$i*40)|0)|0)) + 8|0);
           $181 = HEAP16[$180>>1]|0;
           $182 = $181 & 9216;
           $183 = ($182<<16>>16)==(0);
           if ($183) {
            $184 = (((($115) + (($$1$i$idx$us$i$i*40)|0)|0)) + 24|0);
            $185 = HEAP32[$184>>2]|0;
            $186 = ($185|0)==(0);
            if (!($186)) {
             $187 = (((($115) + (($$1$i$idx$us$i$i*40)|0)|0)) + 20|0);
             $188 = HEAP32[$187>>2]|0;
             $189 = ($188|0)==(0|0);
             do {
              if (!($189)) {
               $190 = HEAP32[7324]|0;
               $191 = ($190|0)==(0);
               if ($191) {
                $199 = HEAP32[(29344)>>2]|0;
                FUNCTION_TABLE_vi[$199 & 127]($188);
                break;
               } else {
                $192 = HEAP32[(29352)>>2]|0;
                $193 = (FUNCTION_TABLE_ii[$192 & 127]($188)|0);
                $194 = HEAP32[14978]|0;
                $195 = (($194) - ($193))|0;
                HEAP32[14978] = $195;
                $196 = HEAP32[14981]|0;
                $197 = (($196) + -1)|0;
                HEAP32[14981] = $197;
                $198 = HEAP32[(29344)>>2]|0;
                FUNCTION_TABLE_vi[$198 & 127]($188);
                break;
               }
              }
             } while(0);
             HEAP32[$184>>2] = 0;
            }
           } else {
            $$1$i$ptr$us$i$i = (($115) + (($$1$i$idx$us$i$i*40)|0)|0);
            _vdbeMemClear($$1$i$ptr$us$i$i);
           }
           HEAP16[$180>>1] = 128;
           $$1$i$add$us$i$i = (($$1$i$idx$us$i$i) + 1)|0;
           $exitcond244$i$i = ($$1$i$add$us$i$i|0)==(9);
           if ($exitcond244$i$i) {
            break L48;
           } else {
            $$1$i$idx$us$i$i = $$1$i$add$us$i$i;
           }
          }
         }
         $$1$i$idx$i$i = 1;
         while(1) {
          $200 = (((($115) + (($$1$i$idx$i$i*40)|0)|0)) + 8|0);
          $201 = HEAP16[$200>>1]|0;
          $202 = $201 & 9216;
          $203 = ($202<<16>>16)==(0);
          if ($203) {
           $204 = (((($115) + (($$1$i$idx$i$i*40)|0)|0)) + 24|0);
           $205 = HEAP32[$204>>2]|0;
           $206 = ($205|0)==(0);
           if (!($206)) {
            $207 = (((($115) + (($$1$i$idx$i$i*40)|0)|0)) + 20|0);
            $208 = HEAP32[$207>>2]|0;
            $209 = HEAP32[$130>>2]|0;
            $210 = ($209|0)==(0|0);
            L71: do {
             if ($210) {
              $211 = $208;
              $212 = HEAP32[$134>>2]|0;
              $213 = ($212>>>0)>($211>>>0);
              do {
               if (!($213)) {
                $214 = HEAP32[$135>>2]|0;
                $215 = ($214>>>0)>($211>>>0);
                if (!($215)) {
                 break;
                }
                $216 = HEAP32[$136>>2]|0;
                HEAP32[$208>>2] = $216;
                HEAP32[$136>>2] = $208;
                break L71;
               }
              } while(0);
              $217 = ($208|0)==(0|0);
              if ($217) {
               break;
              }
              $218 = HEAP32[7324]|0;
              $219 = ($218|0)==(0);
              if ($219) {
               $227 = HEAP32[(29344)>>2]|0;
               FUNCTION_TABLE_vi[$227 & 127]($208);
               break;
              } else {
               $220 = HEAP32[(29352)>>2]|0;
               $221 = (FUNCTION_TABLE_ii[$220 & 127]($208)|0);
               $222 = HEAP32[14978]|0;
               $223 = (($222) - ($221))|0;
               HEAP32[14978] = $223;
               $224 = HEAP32[14981]|0;
               $225 = (($224) + -1)|0;
               HEAP32[14981] = $225;
               $226 = HEAP32[(29344)>>2]|0;
               FUNCTION_TABLE_vi[$226 & 127]($208);
               break;
              }
             } else {
              _measureAllocationSize($129,$208);
             }
            } while(0);
            HEAP32[$204>>2] = 0;
           }
          } else {
           $$1$i$ptr$i$i = (($115) + (($$1$i$idx$i$i*40)|0)|0);
           _vdbeMemClear($$1$i$ptr$i$i);
          }
          HEAP16[$200>>1] = 128;
          $$1$i$add$i$i = (($$1$i$idx$i$i) + 1)|0;
          $exitcond245$i$i = ($$1$i$add$i$i|0)==(9);
          if ($exitcond245$i$i) {
           break;
          } else {
           $$1$i$idx$i$i = $$1$i$add$i$i;
          }
         }
        } else {
         if (!($133)) {
          $$0$i$idx$i$i = 1;
          while(1) {
           $156 = (((($115) + (($$0$i$idx$i$i*40)|0)|0)) + 24|0);
           $157 = HEAP32[$156>>2]|0;
           $158 = ($157|0)==(0);
           L90: do {
            if (!($158)) {
             $159 = (((($115) + (($$0$i$idx$i$i*40)|0)|0)) + 20|0);
             $160 = HEAP32[$159>>2]|0;
             $161 = ($160|0)==(0|0);
             if (!($161)) {
              $162 = HEAP32[$130>>2]|0;
              $163 = ($162|0)==(0|0);
              if (!($163)) {
               _measureAllocationSize($129,$160);
               break;
              }
              $164 = $160;
              $165 = HEAP32[$134>>2]|0;
              $166 = ($165>>>0)>($164>>>0);
              do {
               if (!($166)) {
                $167 = HEAP32[$135>>2]|0;
                $168 = ($167>>>0)>($164>>>0);
                if (!($168)) {
                 break;
                }
                $169 = HEAP32[$136>>2]|0;
                HEAP32[$160>>2] = $169;
                HEAP32[$136>>2] = $160;
                break L90;
               }
              } while(0);
              $170 = HEAP32[7324]|0;
              $171 = ($170|0)==(0);
              if ($171) {
               $179 = HEAP32[(29344)>>2]|0;
               FUNCTION_TABLE_vi[$179 & 127]($160);
               break;
              } else {
               $172 = HEAP32[(29352)>>2]|0;
               $173 = (FUNCTION_TABLE_ii[$172 & 127]($160)|0);
               $174 = HEAP32[14978]|0;
               $175 = (($174) - ($173))|0;
               HEAP32[14978] = $175;
               $176 = HEAP32[14981]|0;
               $177 = (($176) + -1)|0;
               HEAP32[14981] = $177;
               $178 = HEAP32[(29344)>>2]|0;
               FUNCTION_TABLE_vi[$178 & 127]($160);
               break;
              }
             }
            }
           } while(0);
           $$0$i$add$i$i = (($$0$i$idx$i$i) + 1)|0;
           $exitcond247$i$i = ($$0$i$add$i$i|0)==(9);
           if ($exitcond247$i$i) {
            break L48;
           } else {
            $$0$i$idx$i$i = $$0$i$add$i$i;
           }
          }
         }
         $137 = ((($115)) + 64|0);
         $138 = HEAP32[$137>>2]|0;
         $139 = ($138|0)==(0);
         do {
          if (!($139)) {
           $140 = ((($115)) + 60|0);
           $141 = HEAP32[$140>>2]|0;
           $142 = ($141|0)==(0|0);
           if (!($142)) {
            $143 = HEAP32[7324]|0;
            $144 = ($143|0)==(0);
            if ($144) {
             $152 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$152 & 127]($141);
             break;
            } else {
             $145 = HEAP32[(29352)>>2]|0;
             $146 = (FUNCTION_TABLE_ii[$145 & 127]($141)|0);
             $147 = HEAP32[14978]|0;
             $148 = (($147) - ($146))|0;
             HEAP32[14978] = $148;
             $149 = HEAP32[14981]|0;
             $150 = (($149) + -1)|0;
             HEAP32[14981] = $150;
             $151 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$151 & 127]($141);
             break;
            }
           }
          }
         } while(0);
         $153 = ((($115)) + 104|0);
         $154 = HEAP32[$153>>2]|0;
         $155 = ($154|0)==(0);
         do {
          if (!($155)) {
           $595 = ((($115)) + 100|0);
           $596 = HEAP32[$595>>2]|0;
           $597 = ($596|0)==(0|0);
           if (!($597)) {
            $598 = HEAP32[7324]|0;
            $599 = ($598|0)==(0);
            if ($599) {
             $607 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$607 & 127]($596);
             break;
            } else {
             $600 = HEAP32[(29352)>>2]|0;
             $601 = (FUNCTION_TABLE_ii[$600 & 127]($596)|0);
             $602 = HEAP32[14978]|0;
             $603 = (($602) - ($601))|0;
             HEAP32[14978] = $603;
             $604 = HEAP32[14981]|0;
             $605 = (($604) + -1)|0;
             HEAP32[14981] = $605;
             $606 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$606 & 127]($596);
             break;
            }
           }
          }
         } while(0);
         $608 = ((($115)) + 144|0);
         $609 = HEAP32[$608>>2]|0;
         $610 = ($609|0)==(0);
         do {
          if (!($610)) {
           $611 = ((($115)) + 140|0);
           $612 = HEAP32[$611>>2]|0;
           $613 = ($612|0)==(0|0);
           if (!($613)) {
            $614 = HEAP32[7324]|0;
            $615 = ($614|0)==(0);
            if ($615) {
             $623 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$623 & 127]($612);
             break;
            } else {
             $616 = HEAP32[(29352)>>2]|0;
             $617 = (FUNCTION_TABLE_ii[$616 & 127]($612)|0);
             $618 = HEAP32[14978]|0;
             $619 = (($618) - ($617))|0;
             HEAP32[14978] = $619;
             $620 = HEAP32[14981]|0;
             $621 = (($620) + -1)|0;
             HEAP32[14981] = $621;
             $622 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$622 & 127]($612);
             break;
            }
           }
          }
         } while(0);
         $624 = ((($115)) + 184|0);
         $625 = HEAP32[$624>>2]|0;
         $626 = ($625|0)==(0);
         do {
          if (!($626)) {
           $627 = ((($115)) + 180|0);
           $628 = HEAP32[$627>>2]|0;
           $629 = ($628|0)==(0|0);
           if (!($629)) {
            $630 = HEAP32[7324]|0;
            $631 = ($630|0)==(0);
            if ($631) {
             $639 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$639 & 127]($628);
             break;
            } else {
             $632 = HEAP32[(29352)>>2]|0;
             $633 = (FUNCTION_TABLE_ii[$632 & 127]($628)|0);
             $634 = HEAP32[14978]|0;
             $635 = (($634) - ($633))|0;
             HEAP32[14978] = $635;
             $636 = HEAP32[14981]|0;
             $637 = (($636) + -1)|0;
             HEAP32[14981] = $637;
             $638 = HEAP32[(29344)>>2]|0;
             FUNCTION_TABLE_vi[$638 & 127]($628);
             break;
            }
           }
          }
         } while(0);
         $640 = ((($115)) + 224|0);
         $641 = HEAP32[$640>>2]|0;
         $642 = ($641|0)==(0);
         do {
          if (!($642)) {
           $643 = ((($115)) + 220|0);
           $644 = HEAP32[$643>>2]|0;
           $645 = ($644|0)==(0|0);
           if ($645) {
            break;
           }
           $646 = HEAP32[7324]|0;
           $647 = ($646|0)==(0);
           if ($647) {
            $655 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$655 & 127]($644);
            break;
           } else {
            $648 = HEAP32[(29352)>>2]|0;
            $649 = (FUNCTION_TABLE_ii[$648 & 127]($644)|0);
            $650 = HEAP32[14978]|0;
            $651 = (($650) - ($649))|0;
            HEAP32[14978] = $651;
            $652 = HEAP32[14981]|0;
            $653 = (($652) + -1)|0;
            HEAP32[14981] = $653;
            $654 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$654 & 127]($644);
            break;
           }
          }
         } while(0);
         $656 = ((($115)) + 264|0);
         $657 = HEAP32[$656>>2]|0;
         $658 = ($657|0)==(0);
         do {
          if (!($658)) {
           $659 = ((($115)) + 260|0);
           $660 = HEAP32[$659>>2]|0;
           $661 = ($660|0)==(0|0);
           if ($661) {
            break;
           }
           $662 = HEAP32[7324]|0;
           $663 = ($662|0)==(0);
           if ($663) {
            $671 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$671 & 127]($660);
            break;
           } else {
            $664 = HEAP32[(29352)>>2]|0;
            $665 = (FUNCTION_TABLE_ii[$664 & 127]($660)|0);
            $666 = HEAP32[14978]|0;
            $667 = (($666) - ($665))|0;
            HEAP32[14978] = $667;
            $668 = HEAP32[14981]|0;
            $669 = (($668) + -1)|0;
            HEAP32[14981] = $669;
            $670 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$670 & 127]($660);
            break;
           }
          }
         } while(0);
         $672 = ((($115)) + 304|0);
         $673 = HEAP32[$672>>2]|0;
         $674 = ($673|0)==(0);
         do {
          if (!($674)) {
           $675 = ((($115)) + 300|0);
           $676 = HEAP32[$675>>2]|0;
           $677 = ($676|0)==(0|0);
           if ($677) {
            break;
           }
           $678 = HEAP32[7324]|0;
           $679 = ($678|0)==(0);
           if ($679) {
            $687 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$687 & 127]($676);
            break;
           } else {
            $680 = HEAP32[(29352)>>2]|0;
            $681 = (FUNCTION_TABLE_ii[$680 & 127]($676)|0);
            $682 = HEAP32[14978]|0;
            $683 = (($682) - ($681))|0;
            HEAP32[14978] = $683;
            $684 = HEAP32[14981]|0;
            $685 = (($684) + -1)|0;
            HEAP32[14981] = $685;
            $686 = HEAP32[(29344)>>2]|0;
            FUNCTION_TABLE_vi[$686 & 127]($676);
            break;
           }
          }
         } while(0);
         $688 = ((($115)) + 344|0);
         $689 = HEAP32[$688>>2]|0;
         $690 = ($689|0)==(0);
         if ($690) {
          break;
         }
         $691 = ((($115)) + 340|0);
         $692 = HEAP32[$691>>2]|0;
         $693 = ($692|0)==(0|0);
         if ($693) {
          break;
         }
         $694 = HEAP32[7324]|0;
         $695 = ($694|0)==(0);
         if ($695) {
          $703 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$703 & 127]($692);
          break;
         } else {
          $696 = HEAP32[(29352)>>2]|0;
          $697 = (FUNCTION_TABLE_ii[$696 & 127]($692)|0);
          $698 = HEAP32[14978]|0;
          $699 = (($698) - ($697))|0;
          HEAP32[14978] = $699;
          $700 = HEAP32[14981]|0;
          $701 = (($700) + -1)|0;
          HEAP32[14981] = $701;
          $702 = HEAP32[(29344)>>2]|0;
          FUNCTION_TABLE_vi[$702 & 127]($692);
          break;
         }
        }
       } while(0);
       HEAP32[$21>>2] = 0;
       $228 = HEAP32[$11>>2]|0;
       $229 = ($228|0)==(7);
       if ($229) {
        $230 = ((($114)) + 81|0);
        $231 = HEAP8[$230>>0]|0;
        $232 = ($231<<24>>24)==(0);
        if (!($232)) {
         $$0$i4355 = 1;
         label = 226;
         break;
        }
        $233 = ((($114)) + 82|0);
        $234 = HEAP8[$233>>0]|0;
        $235 = ($234<<24>>24)==(0);
        if (!($235)) {
         $$0$i4355 = 1;
         label = 226;
         break;
        }
        HEAP8[$230>>0] = 1;
        $236 = ((($114)) + 180|0);
        $237 = HEAP32[$236>>2]|0;
        $238 = ($237|0)>(0);
        if ($238) {
         $239 = ((($114)) + 264|0);
         HEAP32[$239>>2] = 1;
        }
        $240 = ((($114)) + 272|0);
        $241 = HEAP32[$240>>2]|0;
        $242 = (($241) + 1)|0;
        HEAP32[$240>>2] = $242;
        $243 = ((($114)) + 236|0);
        $244 = HEAP32[$243>>2]|0;
        $245 = ($244|0)==(0|0);
        if ($245) {
         $$0$i4355 = 1;
         label = 226;
         break;
        }
        $246 = ((($244)) + 12|0);
        HEAP32[$246>>2] = 7;
        $$0$i4355 = 1;
        label = 226;
        break;
       }
       $247 = HEAP32[$27>>2]|0;
       if ($980) {
        $248 = HEAP32[$19>>2]|0;
        $249 = ((($248)) + 360|0);
        $250 = ((($248)) + 368|0);
        $251 = HEAP16[$250>>1]|0;
        $252 = $251 & 16;
        $253 = ($252<<16>>16)==(0);
        if ($253) {
         $$0185$i$i = $249;$$1155$i$i = $247;$$1163$i$i = 0;$$1180$i$i = 0;
        } else {
         $254 = ((($248)) + 372|0);
         $255 = HEAP32[$254>>2]|0;
         $256 = $255 >>> 2;
         $257 = ((($248)) + 376|0);
         $258 = HEAP32[$257>>2]|0;
         $259 = ($256|0)==(0);
         if ($259) {
          $$0185$i$i = $249;$$1155$i$i = $247;$$1163$i$i = 0;$$1180$i$i = $258;
         } else {
          $$0154223$i$i = $247;$$0175222$i$i = 0;
          while(1) {
           $260 = (($258) + ($$0175222$i$i<<2)|0);
           $261 = HEAP32[$260>>2]|0;
           $262 = ((($261)) + 4|0);
           $263 = HEAP32[$262>>2]|0;
           $264 = (($263) + ($$0154223$i$i))|0;
           $265 = (($$0175222$i$i) + 1)|0;
           $exitcond$i$i = ($265|0)==($256|0);
           if ($exitcond$i$i) {
            $$0185$i$i = $249;$$1155$i$i = $264;$$1163$i$i = $256;$$1180$i$i = $258;
            break;
           } else {
            $$0154223$i$i = $264;$$0175222$i$i = $265;
           }
          }
         }
        }
       } else {
        $$0185$i$i = 0;$$1155$i$i = $247;$$1163$i$i = 0;$$1180$i$i = 0;
       }
       $266 = HEAP32[$10>>2]|0;
       $267 = (($266) + 1)|0;
       HEAP32[$10>>2] = $267;
       $268 = ($266|0)<($$1155$i$i|0);
       L176: do {
        if ($268) {
         $269 = ((($$0185$i$i)) + 16|0);
         $270 = ((($$0185$i$i)) + 8|0);
         $271 = ((($$0185$i$i)) + 12|0);
         L178: do {
          if ($980) {
           $$2164217$us$i$i = $$1163$i$i;$$2181216$us$i$i = $$1180$i$i;$$2218$us$i$i = $$1155$i$i;$273 = $266;$274 = $247;
           while(1) {
            $272 = ($273|0)<($274|0);
            if ($272) {
             $290 = HEAP32[$29>>2]|0;
             $291 = (($290) + (($273*20)|0)|0);
             $$1160$us$i$i = $291;$$2177$us$i$i = $273;
            } else {
             $275 = (($273) - ($274))|0;
             $276 = HEAP32[$$2181216$us$i$i>>2]|0;
             $277 = ((($276)) + 4|0);
             $278 = HEAP32[$277>>2]|0;
             $279 = ($275|0)<($278|0);
             if ($279) {
              $$1176$lcssa$us$i$i = $275;$$lcssa$us$i$i = $276;
             } else {
              $$0156207$us$i$i = 0;$$1176206$us$i$i = $275;$281 = $278;
              while(1) {
               $280 = (($$1176206$us$i$i) - ($281))|0;
               $282 = (($$0156207$us$i$i) + 1)|0;
               $283 = (($$2181216$us$i$i) + ($282<<2)|0);
               $284 = HEAP32[$283>>2]|0;
               $285 = ((($284)) + 4|0);
               $286 = HEAP32[$285>>2]|0;
               $287 = ($280|0)<($286|0);
               if ($287) {
                $$1176$lcssa$us$i$i = $280;$$lcssa$us$i$i = $284;
                break;
               } else {
                $$0156207$us$i$i = $282;$$1176206$us$i$i = $280;$281 = $286;
               }
              }
             }
             $288 = HEAP32[$$lcssa$us$i$i>>2]|0;
             $289 = (($288) + (($$1176$lcssa$us$i$i*20)|0)|0);
             $$1160$us$i$i = $289;$$2177$us$i$i = $$1176$lcssa$us$i$i;
            }
            $292 = ((($$1160$us$i$i)) + 1|0);
            $293 = HEAP8[$292>>0]|0;
            $294 = ($293<<24>>24)==(-4);
            do {
             if ($294) {
              $295 = (($$2164217$us$i$i) + 1)|0;
              $296 = $295 << 2;
              $297 = ($$2164217$us$i$i|0)>(0);
              L193: do {
               if ($297) {
                $328 = ((($$1160$us$i$i)) + 16|0);
                $301 = HEAP32[$328>>2]|0;
                $$0153209$us$i$i = 0;
                while(1) {
                 $298 = (($$2181216$us$i$i) + ($$0153209$us$i$i<<2)|0);
                 $299 = HEAP32[$298>>2]|0;
                 $300 = ($299|0)==($301|0);
                 if ($300) {
                  $$0153$lcssa$us$i$i = $$0153209$us$i$i;
                  break L193;
                 }
                 $302 = (($$0153209$us$i$i) + 1)|0;
                 $303 = ($302|0)<($$2164217$us$i$i|0);
                 if ($303) {
                  $$0153209$us$i$i = $302;
                 } else {
                  $$0153$lcssa$us$i$i = $302;
                  break;
                 }
                }
               } else {
                $$0153$lcssa$us$i$i = 0;
               }
              } while(0);
              $304 = ($$0153$lcssa$us$i$i|0)==($$2164217$us$i$i|0);
              if (!($304)) {
               $$5$us$i$i = $$2218$us$i$i;$$5167$us$i$i = $$2164217$us$i$i;$$5184$us$i$i = $$2181216$us$i$i;
               break;
              }
              $305 = ($$2164217$us$i$i|0)!=(0);
              $306 = $305&1;
              $307 = (_sqlite3VdbeMemGrow($$0185$i$i,$296,$306)|0);
              HEAP32[$11>>2] = $307;
              $308 = ($307|0)==(0);
              if (!($308)) {
               $$0$i4355 = 1;
               label = 226;
               break L42;
              }
              $309 = HEAP32[$269>>2]|0;
              $310 = ((($$1160$us$i$i)) + 16|0);
              $311 = HEAP32[$310>>2]|0;
              $312 = (($309) + ($$2164217$us$i$i<<2)|0);
              HEAP32[$312>>2] = $311;
              $313 = HEAP16[$270>>1]|0;
              $314 = $313 | 16;
              HEAP16[$270>>1] = $314;
              HEAP32[$271>>2] = $296;
              $315 = HEAP32[$310>>2]|0;
              $316 = ((($315)) + 4|0);
              $317 = HEAP32[$316>>2]|0;
              $318 = (($317) + ($$2218$us$i$i))|0;
              $$5$us$i$i = $318;$$5167$us$i$i = $295;$$5184$us$i$i = $309;
             } else {
              $$5$us$i$i = $$2218$us$i$i;$$5167$us$i$i = $$2164217$us$i$i;$$5184$us$i$i = $$2181216$us$i$i;
             }
            } while(0);
            $319 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
            $320 = $319 & 8;
            $321 = ($320<<16>>16)==(0);
            if ($321) {
             $$1160$lcssa204$i$i = $$1160$us$i$i;$$2177$lcssa205$i$i = $$2177$us$i$i;$378 = $319;
             break L178;
            }
            $322 = HEAP8[$$1160$us$i$i>>0]|0;
            switch ($322<<24>>24) {
            case -85:  {
             $$1160$lcssa204$i$i = $$1160$us$i$i;$$2177$lcssa205$i$i = $$2177$us$i$i;$378 = $319;
             break L178;
             break;
            }
            case 61:  {
             $323 = HEAP32[$10>>2]|0;
             $324 = ($323|0)>(1);
             if ($324) {
              $$1160$lcssa204$i$i = $$1160$us$i$i;$$2177$lcssa205$i$i = $$2177$us$i$i;$378 = $319;
              break L178;
             } else {
              $326 = $323;
             }
             break;
            }
            default: {
             $$pre250$i$i = HEAP32[$10>>2]|0;
             $326 = $$pre250$i$i;
            }
            }
            $325 = (($326) + 1)|0;
            HEAP32[$10>>2] = $325;
            $327 = ($326|0)<($$5$us$i$i|0);
            if (!($327)) {
             break L176;
            }
            $$pre$i52$i = HEAP32[$27>>2]|0;
            $$2164217$us$i$i = $$5167$us$i$i;$$2181216$us$i$i = $$5184$us$i$i;$$2218$us$i$i = $$5$us$i$i;$273 = $326;$274 = $$pre$i52$i;
           }
          } else {
           $329 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
           $330 = $329 & 8;
           $331 = ($330<<16>>16)==(0);
           if ($331) {
            $332 = ($266|0)<($247|0);
            if ($332) {
             $348 = HEAP32[$29>>2]|0;
             $349 = (($348) + (($266*20)|0)|0);
             $$1160$lcssa204$i$i = $349;$$2177$lcssa205$i$i = $266;$378 = $329;
             break;
            }
            $333 = (($266) - ($247))|0;
            $334 = HEAP32[$$1180$i$i>>2]|0;
            $335 = ((($334)) + 4|0);
            $336 = HEAP32[$335>>2]|0;
            $337 = ($333|0)<($336|0);
            if ($337) {
             $$1176$lcssa$i$i$us = $333;$$lcssa$i$i$us = $334;
            } else {
             $$0156207$i$i$us = 0;$$1176206$i$i$us = $333;$339 = $336;
             while(1) {
              $338 = (($$1176206$i$i$us) - ($339))|0;
              $340 = (($$0156207$i$i$us) + 1)|0;
              $341 = (($$1180$i$i) + ($340<<2)|0);
              $342 = HEAP32[$341>>2]|0;
              $343 = ((($342)) + 4|0);
              $344 = HEAP32[$343>>2]|0;
              $345 = ($338|0)<($344|0);
              if ($345) {
               $$1176$lcssa$i$i$us = $338;$$lcssa$i$i$us = $342;
               break;
              } else {
               $$0156207$i$i$us = $340;$$1176206$i$i$us = $338;$339 = $344;
              }
             }
            }
            $346 = HEAP32[$$lcssa$i$i$us>>2]|0;
            $347 = (($346) + (($$1176$lcssa$i$i$us*20)|0)|0);
            $$1160$lcssa204$i$i = $347;$$2177$lcssa205$i$i = $$1176$lcssa$i$i$us;$378 = $329;
            break;
           }
           $351 = $266;$371 = $267;
           while(1) {
            $350 = ($351|0)<($247|0);
            if ($350) {
             $352 = HEAP32[$29>>2]|0;
             $353 = (($352) + (($351*20)|0)|0);
             $$1160$i$i = $353;$$2177$i$i = $351;
            } else {
             $354 = (($351) - ($247))|0;
             $355 = HEAP32[$$1180$i$i>>2]|0;
             $356 = ((($355)) + 4|0);
             $357 = HEAP32[$356>>2]|0;
             $358 = ($354|0)<($357|0);
             if ($358) {
              $$1176$lcssa$i$i = $354;$$lcssa$i$i = $355;
             } else {
              $$0156207$i$i = 0;$$1176206$i$i = $354;$360 = $357;
              while(1) {
               $359 = (($$1176206$i$i) - ($360))|0;
               $361 = (($$0156207$i$i) + 1)|0;
               $362 = (($$1180$i$i) + ($361<<2)|0);
               $363 = HEAP32[$362>>2]|0;
               $364 = ((($363)) + 4|0);
               $365 = HEAP32[$364>>2]|0;
               $366 = ($359|0)<($365|0);
               if ($366) {
                $$1176$lcssa$i$i = $359;$$lcssa$i$i = $363;
                break;
               } else {
                $$0156207$i$i = $361;$$1176206$i$i = $359;$360 = $365;
               }
              }
             }
             $367 = HEAP32[$$lcssa$i$i>>2]|0;
             $368 = (($367) + (($$1176$lcssa$i$i*20)|0)|0);
             $$1160$i$i = $368;$$2177$i$i = $$1176$lcssa$i$i;
            }
            $369 = HEAP8[$$1160$i$i>>0]|0;
            switch ($369<<24>>24) {
            case -85:  {
             $$1160$lcssa204$i$i = $$1160$i$i;$$2177$lcssa205$i$i = $$2177$i$i;$378 = $329;
             break L178;
             break;
            }
            case 61:  {
             $370 = ($371|0)>(1);
             if ($370) {
              $$1160$lcssa204$i$i = $$1160$i$i;$$2177$lcssa205$i$i = $$2177$i$i;$378 = $329;
              break L178;
             }
             break;
            }
            default: {
            }
            }
            $372 = (($371) + 1)|0;
            HEAP32[$10>>2] = $372;
            $373 = ($371|0)<($$1155$i$i|0);
            if ($373) {
             $351$phi = $371;$371 = $372;$351 = $351$phi;
            } else {
             break L176;
            }
           }
          }
         } while(0);
         $374 = ((($114)) + 264|0);
         $375 = HEAP32[$374>>2]|0;
         $376 = ($375|0)==(0);
         if (!($376)) {
          HEAP32[$11>>2] = 9;
          _sqlite3VdbeError($0,50367,$vararg_buffer7);
          $$0$i4355 = 1;
          label = 226;
          break L42;
         }
         $377 = $378 & 12;
         $379 = ($377<<16>>16)==(4);
         if ($379) {
          $380 = ((($115)) + 48|0);
          HEAP16[$380>>1] = 4;
          $381 = ($$2177$lcssa205$i$i|0)<(0);
          $382 = $381 << 31 >> 31;
          $383 = $$ptr201$ptr$i$i;
          $384 = $383;
          HEAP32[$384>>2] = $$2177$lcssa205$i$i;
          $385 = (($383) + 4)|0;
          $386 = $385;
          HEAP32[$386>>2] = $382;
          $387 = ((($115)) + 88|0);
          HEAP16[$387>>1] = 2562;
          $388 = HEAP8[$$1160$lcssa204$i$i>>0]|0;
          $389 = $388&255;
          $390 = (2800 + ($389<<2)|0);
          $391 = HEAP32[$390>>2]|0;
          $392 = ((($115)) + 96|0);
          HEAP32[$392>>2] = $391;
          $393 = (_strlen($391)|0);
          $394 = $393 & 1073741823;
          $395 = ((($115)) + 92|0);
          HEAP32[$395>>2] = $394;
          $396 = ((($115)) + 90|0);
          HEAP8[$396>>0] = 1;
          $397 = ((($115)) + 120|0);
          $$0168$i$i = $397;
         } else {
          $$0168$i$i = $$ptr201$ptr$i$i;
         }
         $398 = ((($$0168$i$i)) + 8|0);
         HEAP16[$398>>1] = 4;
         $399 = ((($$1160$lcssa204$i$i)) + 4|0);
         $400 = HEAP32[$399>>2]|0;
         $401 = ($400|0)<(0);
         $402 = $401 << 31 >> 31;
         $403 = $$0168$i$i;
         $404 = $403;
         HEAP32[$404>>2] = $400;
         $405 = (($403) + 4)|0;
         $406 = $405;
         HEAP32[$406>>2] = $402;
         $407 = ((($$0168$i$i)) + 40|0);
         $408 = ((($$0168$i$i)) + 48|0);
         HEAP16[$408>>1] = 4;
         $409 = ((($$1160$lcssa204$i$i)) + 8|0);
         $410 = HEAP32[$409>>2]|0;
         $411 = ($410|0)<(0);
         $412 = $411 << 31 >> 31;
         $413 = $407;
         $414 = $413;
         HEAP32[$414>>2] = $410;
         $415 = (($413) + 4)|0;
         $416 = $415;
         HEAP32[$416>>2] = $412;
         $417 = ((($$0168$i$i)) + 80|0);
         $418 = ((($$0168$i$i)) + 88|0);
         HEAP16[$418>>1] = 4;
         $419 = ((($$1160$lcssa204$i$i)) + 12|0);
         $420 = HEAP32[$419>>2]|0;
         $421 = ($420|0)<(0);
         $422 = $421 << 31 >> 31;
         $423 = $417;
         $424 = $423;
         HEAP32[$424>>2] = $420;
         $425 = (($423) + 4)|0;
         $426 = $425;
         HEAP32[$426>>2] = $422;
         $427 = ((($$0168$i$i)) + 120|0);
         $428 = ((($$0168$i$i)) + 144|0);
         $429 = HEAP32[$428>>2]|0;
         $430 = ($429|0)<(100);
         if ($430) {
          $438 = (_sqlite3VdbeMemGrow($427,100,0)|0);
          $439 = ($438|0)==(0);
          if (!($439)) {
           $$0$i4355 = 1;
           label = 226;
           break L42;
          }
          $$phi$trans$insert$i$i = ((($$0168$i$i)) + 136|0);
          $$pre251$i$i = HEAP32[$$phi$trans$insert$i$i>>2]|0;
          $$pre252$i$i = HEAP32[$428>>2]|0;
          $$pre256$i$i = ((($$0168$i$i)) + 128|0);
          $$pre$phi$i$iZ2D = $$pre256$i$i;$$pre$phi257$i$iZ2D = $$phi$trans$insert$i$i;$440 = $$pre251$i$i;$441 = $$pre252$i$i;
         } else {
          $431 = ((($$0168$i$i)) + 140|0);
          $432 = HEAP32[$431>>2]|0;
          $433 = ((($$0168$i$i)) + 136|0);
          HEAP32[$433>>2] = $432;
          $434 = ((($$0168$i$i)) + 128|0);
          $435 = HEAP16[$434>>1]|0;
          $436 = $435 & 13;
          HEAP16[$434>>1] = $436;
          $437 = $432;
          $$pre$phi$i$iZ2D = $434;$$pre$phi257$i$iZ2D = $433;$440 = $437;$441 = $429;
         }
         HEAP16[$$pre$phi$i$iZ2D>>1] = 514;
         HEAP32[$30>>2] = $440;
         HEAP32[$1>>2] = 0;
         HEAP32[$31>>2] = $441;
         $442 = ((($$1160$lcssa204$i$i)) + 1|0);
         ;HEAP32[$32>>2]=0|0;HEAP32[$32+4>>2]=0|0;HEAP16[$32+8>>1]=0|0;
         $443 = HEAP8[$442>>0]|0;
         $444 = $443 << 24 >> 24;
         L246: do {
          switch ($444|0) {
          case -9:  {
           $445 = ((($$1160$lcssa204$i$i)) + 16|0);
           $446 = HEAP32[$445>>2]|0;
           $447 = ((($446)) + 6|0);
           $448 = HEAP16[$447>>1]|0;
           $449 = $448&65535;
           HEAP32[$vararg_buffer9>>2] = $449;
           _sqlite3_str_appendf($1,34044,$vararg_buffer9);
           $450 = HEAP16[$447>>1]|0;
           $451 = ($450<<16>>16)==(0);
           if (!($451)) {
            $452 = ((($446)) + 16|0);
            $$06973$i$i$i = 0;
            while(1) {
             $453 = (((($446)) + 20|0) + ($$06973$i$i$i<<2)|0);
             $454 = HEAP32[$453>>2]|0;
             $455 = ($454|0)==(0|0);
             if ($455) {
              $457 = 59952;
             } else {
              $456 = HEAP32[$454>>2]|0;
              $457 = $456;
             }
             $458 = (_strcmp($457,34049)|0);
             $459 = ($458|0)==(0);
             $spec$select$i$i$i = $459 ? 34056 : $457;
             $460 = HEAP32[$452>>2]|0;
             $461 = (($460) + ($$06973$i$i$i)|0);
             $462 = HEAP8[$461>>0]|0;
             $463 = ($462<<24>>24)==(0);
             $464 = $463 ? 59952 : 34064;
             HEAP32[$vararg_buffer12>>2] = $464;
             $vararg_ptr15 = ((($vararg_buffer12)) + 4|0);
             HEAP32[$vararg_ptr15>>2] = $spec$select$i$i$i;
             _sqlite3_str_appendf($1,34058,$vararg_buffer12);
             $465 = (($$06973$i$i$i) + 1)|0;
             $466 = HEAP16[$447>>1]|0;
             $467 = $466&65535;
             $468 = ($465>>>0)<($467>>>0);
             if ($468) {
              $$06973$i$i$i = $465;
             } else {
              break;
             }
            }
           }
           $469 = HEAP32[$33>>2]|0;
           $470 = (($469) + 1)|0;
           $471 = HEAP32[$31>>2]|0;
           $472 = ($470>>>0)<($471>>>0);
           if ($472) {
            HEAP32[$33>>2] = $470;
            $473 = HEAP32[$30>>2]|0;
            $474 = (($473) + ($469)|0);
            HEAP8[$474>>0] = 41;
            $$1$i189$i$i = $440;
            break L246;
           } else {
            _enlargeAndAppend($1,34066,1);
            $$1$i189$i$i = $440;
            break L246;
           }
           break;
          }
          case -2:  {
           $475 = ((($$1160$lcssa204$i$i)) + 16|0);
           $476 = HEAP32[$475>>2]|0;
           $477 = HEAP32[$476>>2]|0;
           HEAP32[$vararg_buffer16>>2] = $477;
           _sqlite3_str_appendf($1,34068,$vararg_buffer16);
           $$1$i189$i$i = $440;
           break;
          }
          case -8:  {
           $478 = ((($$1160$lcssa204$i$i)) + 16|0);
           $479 = HEAP32[$478>>2]|0;
           $480 = ((($479)) + 32|0);
           $481 = HEAP32[$480>>2]|0;
           $482 = HEAP8[$479>>0]|0;
           $483 = $482 << 24 >> 24;
           HEAP32[$vararg_buffer19>>2] = $481;
           $vararg_ptr22 = ((($vararg_buffer19)) + 4|0);
           HEAP32[$vararg_ptr22>>2] = $483;
           _sqlite3_str_appendf($1,34076,$vararg_buffer19);
           $$1$i189$i$i = $440;
           break;
          }
          case -14:  {
           $484 = ((($$1160$lcssa204$i$i)) + 16|0);
           $485 = HEAP32[$484>>2]|0;
           $486 = $485;
           $487 = $486;
           $488 = HEAP32[$487>>2]|0;
           $489 = (($486) + 4)|0;
           $490 = $489;
           $491 = HEAP32[$490>>2]|0;
           $492 = $vararg_buffer23;
           $493 = $492;
           HEAP32[$493>>2] = $488;
           $494 = (($492) + 4)|0;
           $495 = $494;
           HEAP32[$495>>2] = $491;
           _sqlite3_str_appendf($1,32524,$vararg_buffer23);
           $$1$i189$i$i = $440;
           break;
          }
          case -3:  {
           $496 = ((($$1160$lcssa204$i$i)) + 16|0);
           $497 = HEAP32[$496>>2]|0;
           HEAP32[$vararg_buffer26>>2] = $497;
           _sqlite3_str_appendf($1,34083,$vararg_buffer26);
           $$1$i189$i$i = $440;
           break;
          }
          case -13:  {
           $498 = ((($$1160$lcssa204$i$i)) + 16|0);
           $499 = HEAP32[$498>>2]|0;
           $500 = +HEAPF64[$499>>3];
           HEAPF64[$vararg_buffer29>>3] = $500;
           _sqlite3_str_appendf($1,34086,$vararg_buffer29);
           $$1$i189$i$i = $440;
           break;
          }
          case -11:  {
           $501 = ((($$1160$lcssa204$i$i)) + 16|0);
           $502 = HEAP32[$501>>2]|0;
           $503 = ((($502)) + 8|0);
           $504 = HEAP16[$503>>1]|0;
           $505 = $504&65535;
           $506 = $505 & 2;
           $507 = ($506|0)==(0);
           if (!($507)) {
            $508 = ((($502)) + 16|0);
            $509 = HEAP32[$508>>2]|0;
            $$1$i189$i$i = $509;
            break L246;
           }
           $510 = $505 & 4;
           $511 = ($510|0)==(0);
           if (!($511)) {
            $512 = $502;
            $513 = $512;
            $514 = HEAP32[$513>>2]|0;
            $515 = (($512) + 4)|0;
            $516 = $515;
            $517 = HEAP32[$516>>2]|0;
            $518 = $vararg_buffer32;
            $519 = $518;
            HEAP32[$519>>2] = $514;
            $520 = (($518) + 4)|0;
            $521 = $520;
            HEAP32[$521>>2] = $517;
            _sqlite3_str_appendf($1,32524,$vararg_buffer32);
            $$1$i189$i$i = $440;
            break L246;
           }
           $522 = $505 & 8;
           $523 = ($522|0)==(0);
           if ($523) {
            $525 = $505 & 1;
            $526 = ($525|0)==(0);
            $$$i$i$i = $526 ? 34092 : 31171;
            $$1$i189$i$i = $$$i$i$i;
            break L246;
           } else {
            $524 = +HEAPF64[$502>>3];
            HEAPF64[$vararg_buffer35>>3] = $524;
            _sqlite3_str_appendf($1,34086,$vararg_buffer35);
            $$1$i189$i$i = $440;
            break L246;
           }
           break;
          }
          case -12:  {
           $527 = ((($$1160$lcssa204$i$i)) + 16|0);
           $528 = HEAP32[$527>>2]|0;
           $529 = ((($528)) + 8|0);
           $530 = HEAP32[$529>>2]|0;
           HEAP32[$vararg_buffer38>>2] = $530;
           _sqlite3_str_appendf($1,34099,$vararg_buffer38);
           $$1$i189$i$i = $440;
           break;
          }
          case -15:  {
           $531 = ((($$1160$lcssa204$i$i)) + 16|0);
           $532 = HEAP32[$531>>2]|0;
           $533 = HEAP32[$532>>2]|0;
           $534 = ($533|0)<(1);
           if ($534) {
            $539 = 0;$541 = $441;
           } else {
            $$07074$i$i$i = 1;
            while(1) {
             $535 = (($532) + ($$07074$i$i$i<<2)|0);
             $536 = HEAP32[$535>>2]|0;
             HEAP32[$vararg_buffer41>>2] = $536;
             _sqlite3_str_appendf($1,34107,$vararg_buffer41);
             $537 = (($$07074$i$i$i) + 1)|0;
             $exitcond$i$i$i = ($$07074$i$i$i|0)==($533|0);
             if ($exitcond$i$i$i) {
              break;
             } else {
              $$07074$i$i$i = $537;
             }
            }
            $$pre$i$i$i = HEAP32[$33>>2]|0;
            $$pre78$i$i$i = HEAP32[$31>>2]|0;
            $539 = $$pre$i$i$i;$541 = $$pre78$i$i$i;
           }
           HEAP8[$440>>0] = 91;
           $538 = (($539) + 1)|0;
           $540 = ($538>>>0)<($541>>>0);
           if ($540) {
            HEAP32[$33>>2] = $538;
            $542 = HEAP32[$30>>2]|0;
            $543 = (($542) + ($539)|0);
            HEAP8[$543>>0] = 93;
            $$1$i189$i$i = $440;
            break L246;
           } else {
            _enlargeAndAppend($1,34111,1);
            $$1$i189$i$i = $440;
            break L246;
           }
           break;
          }
          case -4:  {
           _sqlite3_str_appendf($1,34113,$vararg_buffer44);
           $$1$i189$i$i = $440;
           break;
          }
          case -5: case -17:  {
           HEAP8[$440>>0] = 0;
           $$1$i189$i$i = $440;
           break;
          }
          case -6:  {
           $544 = ((($$1160$lcssa204$i$i)) + 16|0);
           $545 = HEAP32[$544>>2]|0;
           $546 = HEAP32[$545>>2]|0;
           HEAP32[$vararg_buffer46>>2] = $546;
           _sqlite3_str_appendf($1,31408,$vararg_buffer46);
           $$1$i189$i$i = $440;
           break;
          }
          default: {
           $547 = ((($$1160$lcssa204$i$i)) + 16|0);
           $548 = HEAP32[$547>>2]|0;
           $549 = ($548|0)==(0|0);
           if (!($549)) {
            $$1$i189$i$i = $548;
            break L246;
           }
           HEAP8[$440>>0] = 0;
           $$1$i189$i$i = $440;
          }
          }
         } while(0);
         $550 = HEAP32[$30>>2]|0;
         $551 = ($550|0)==(0|0);
         do {
          if (!($551)) {
           $552 = HEAP32[$33>>2]|0;
           $553 = (($550) + ($552)|0);
           HEAP8[$553>>0] = 0;
           $554 = HEAP32[$32>>2]|0;
           $555 = ($554|0)==(0);
           if ($555) {
            break;
           }
           $556 = HEAP8[$34>>0]|0;
           $557 = $556 & 4;
           $558 = ($557<<24>>24)==(0);
           if (!($558)) {
            break;
           }
           (_strAccumFinishRealloc($1)|0);
          }
         } while(0);
         $559 = HEAP32[$$pre$phi257$i$iZ2D>>2]|0;
         $560 = ($$1$i189$i$i|0)==($559|0);
         if ($560) {
          $562 = ($$1$i189$i$i|0)==(0|0);
          if ($562) {
           $$0$i190$i$i = 0;
          } else {
           $563 = (_strlen($$1$i189$i$i)|0);
           $564 = $563 & 1073741823;
           $$0$i190$i$i = $564;
          }
          $565 = ((($$0168$i$i)) + 132|0);
          HEAP32[$565>>2] = $$0$i190$i$i;
          $566 = ((($$0168$i$i)) + 130|0);
          HEAP8[$566>>0] = 1;
         } else {
          $561 = ((($$0168$i$i)) + 132|0);
          HEAP32[$561>>2] = 0;
          (_sqlite3VdbeMemSetStr($427,$$1$i189$i$i,-1,1,0)|0);
         }
         $567 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
         $568 = $567 & 12;
         $569 = ($568<<16>>16)==(4);
         if ($569) {
          $570 = ((($$0168$i$i)) + 184|0);
          $571 = HEAP32[$570>>2]|0;
          $572 = ($571|0)<(4);
          if ($572) {
           $580 = ((($$0168$i$i)) + 160|0);
           $581 = (_sqlite3VdbeMemGrow($580,4,0)|0);
           $582 = ($581|0)==(0);
           if (!($582)) {
            $$0$i4355 = 1;
            label = 226;
            break L42;
           }
           $$phi$trans$insert253$i$i = ((($$0168$i$i)) + 176|0);
           $$pre254$i$i = HEAP32[$$phi$trans$insert253$i$i>>2]|0;
           $$pre258$i$i = ((($$0168$i$i)) + 168|0);
           $$pre$phi259$i$iZ2D = $$pre258$i$i;$587 = $$pre254$i$i;
          } else {
           $573 = ((($$0168$i$i)) + 180|0);
           $574 = HEAP32[$573>>2]|0;
           $575 = ((($$0168$i$i)) + 176|0);
           HEAP32[$575>>2] = $574;
           $576 = ((($$0168$i$i)) + 168|0);
           $577 = HEAP16[$576>>1]|0;
           $578 = $577 & 13;
           HEAP16[$576>>1] = $578;
           $579 = $574;
           $$pre$phi259$i$iZ2D = $576;$587 = $579;
          }
          HEAP16[$$pre$phi259$i$iZ2D>>1] = 514;
          $583 = ((($$0168$i$i)) + 172|0);
          HEAP32[$583>>2] = 2;
          $584 = ((($$1160$lcssa204$i$i)) + 2|0);
          $585 = HEAP16[$584>>1]|0;
          $586 = $585&65535;
          HEAP32[$vararg_buffer49>>2] = $586;
          (_sqlite3_snprintf(3,$587,32595,$vararg_buffer49)|0);
          $588 = ((($$0168$i$i)) + 170|0);
          HEAP8[$588>>0] = 1;
          $589 = ((($$0168$i$i)) + 208|0);
          HEAP16[$589>>1] = 1;
          $$pre255$i$i = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
          $$pre261$i$i = $$pre255$i$i & 12;
          $$pre$phi262$i$iZ2D = $$pre261$i$i;
         } else {
          $$pre$phi262$i$iZ2D = $568;
         }
         $590 = $$pre$phi262$i$iZ2D&65535;
         $591 = (12 - ($590))|0;
         $592 = $591&65535;
         HEAP16[$35>>1] = $592;
         $593 = HEAP32[$19>>2]|0;
         $594 = ((($593)) + 40|0);
         HEAP32[$21>>2] = $594;
         HEAP32[$11>>2] = 0;
         $$1$i = 100;
         break L42;
        }
       } while(0);
       HEAP32[$11>>2] = 0;
       $$0$i4355 = 101;
       label = 226;
      }
     } while(0);
     do {
      if ((label|0) == 226) {
       label = 0;
       $711 = $9;
       $712 = $711;
       $713 = HEAP32[$712>>2]|0;
       $714 = (($711) + 4)|0;
       $715 = $714;
       $716 = HEAP32[$715>>2]|0;
       $717 = ($716|0)>(0);
       $718 = ($713>>>0)>(0);
       $719 = ($716|0)==(0);
       $720 = $719 & $718;
       $721 = $717 | $720;
       if ($721) {
        _invokeProfileCallback($59,$0);
       }
       $722 = ($$0$i4355|0)==(101);
       if ($722) {
        $723 = ((($59)) + 79|0);
        $724 = HEAP8[$723>>0]|0;
        $725 = ($724<<24>>24)==(0);
        if ($725) {
         $$1$i = 101;
        } else {
         $726 = ((($59)) + 20|0);
         $727 = HEAP32[$726>>2]|0;
         $728 = ($727|0)>(0);
         if ($728) {
          $729 = ((($59)) + 16|0);
          $730 = ((($59)) + 240|0);
          $731 = ((($59)) + 244|0);
          $$02527$i$i = 0;$$028$i$i = 0;$981 = $727;
          while(1) {
           $732 = HEAP32[$729>>2]|0;
           $733 = (((($732) + ($$02527$i$i<<4)|0)) + 4|0);
           $734 = HEAP32[$733>>2]|0;
           $735 = ($734|0)==(0|0);
           do {
            if ($735) {
             $$2$i$i = $$028$i$i;$753 = $981;
            } else {
             $736 = HEAP32[$734>>2]|0;
             $737 = ((($734)) + 4|0);
             $738 = HEAP32[$737>>2]|0;
             $739 = ((($738)) + 4|0);
             HEAP32[$739>>2] = $736;
             $$idx$val$idx$val$i$i = HEAP32[$738>>2]|0;
             $$idx26$i$i = ((($$idx$val$idx$val$i$i)) + 216|0);
             $$idx26$val$i$i = HEAP32[$$idx26$i$i>>2]|0;
             $740 = ($$idx26$val$i$i|0)==(0|0);
             if ($740) {
              $$2$i$i = $$028$i$i;$753 = $981;
             } else {
              $741 = ((($$idx26$val$i$i)) + 12|0);
              $742 = HEAP32[$741>>2]|0;
              HEAP32[$741>>2] = 0;
              $743 = ($742|0)>(0);
              if ($743) {
               $744 = HEAP32[$730>>2]|0;
               $745 = ($744|0)!=(0|0);
               $746 = ($$028$i$i|0)==(0);
               $or$cond$i51$i = $746 & $745;
               if (!($or$cond$i51$i)) {
                $$2$i$i = $$028$i$i;$753 = $981;
                break;
               }
               $747 = (($732) + ($$02527$i$i<<4)|0);
               $748 = HEAP32[$731>>2]|0;
               $749 = HEAP32[$747>>2]|0;
               $750 = (FUNCTION_TABLE_iiiii[$744 & 127]($748,$59,$749,$742)|0);
               $$pre$i$i = HEAP32[$726>>2]|0;
               $$2$i$i = $750;$753 = $$pre$i$i;
              } else {
               $$2$i$i = $$028$i$i;$753 = $981;
              }
             }
            }
           } while(0);
           $751 = (($$02527$i$i) + 1)|0;
           $752 = ($751|0)<($753|0);
           if ($752) {
            $$02527$i$i = $751;$$028$i$i = $$2$i$i;$981 = $753;
           } else {
            break;
           }
          }
          HEAP32[$11>>2] = $$2$i$i;
          $754 = ($$2$i$i|0)==(0);
          if (!($754)) {
           $$1$i = 1;
           break;
          }
         } else {
          HEAP32[$11>>2] = 0;
         }
         $$1$i = 101;
        }
       } else {
        $$1$i = $$0$i4355;
       }
      }
     } while(0);
     $755 = ((($59)) + 64|0);
     HEAP32[$755>>2] = $$1$i;
     $756 = HEAP32[$0>>2]|0;
     $757 = HEAP32[$11>>2]|0;
     $758 = ((($756)) + 81|0);
     $759 = HEAP8[$758>>0]|0;
     $760 = ($759<<24>>24)!=(0);
     $761 = ($757|0)==(3082);
     $or$cond$i$i44 = $761 | $760;
     if ($or$cond$i$i44) {
      _apiOomError($756);
      label = 244;
     } else {
      $762 = ((($756)) + 68|0);
      $763 = HEAP32[$762>>2]|0;
      $764 = $763 & $757;
      $765 = ($764|0)==(7);
      if ($765) {
       label = 244;
      }
     }
     if ((label|0) == 244) {
      label = 0;
      HEAP32[$11>>2] = 7;
     }
     $766 = $$1$i | 1;
     $767 = ($766|0)==(101);
     if ($767) {
      $$3$i = $$1$i;
     } else {
      $$2$i57 = $$1$i;
      label = 246;
     }
    }
    if ((label|0) == 246) {
     label = 0;
     $768 = HEAP8[$18>>0]|0;
     $769 = ($768<<24>>24)<(0);
     if ($769) {
      $770 = (_sqlite3VdbeTransferError($0)|0);
      $$3$i = $770;
     } else {
      $$3$i = $$2$i57;
     }
    }
    $771 = ((($59)) + 68|0);
    $772 = HEAP32[$771>>2]|0;
    $773 = $772 & $$3$i;
    $774 = ($773|0)==(17);
    if (!($774)) {
     $$0 = $773;
     label = 322;
     break;
    }
    $775 = (($$039) + 1)|0;
    $776 = ($$039>>>0)<(50);
    if (!($776)) {
     $$0 = 17;
     label = 322;
     break;
    }
    $777 = HEAP32[$10>>2]|0;
    $778 = HEAP32[$20>>2]|0;
    $$idx$val$i47 = HEAP32[$0>>2]|0;
    $$idx18$val$i = HEAP8[$18>>0]|0;
    $779 = $$idx18$val$i&255;
    $780 = (_sqlite3LockAndPrepare($$idx$val$i47,$778,-1,$779,$0,$2,0)|0);
    switch ($780|0) {
    case 7:  {
     label = 251;
     break L7;
     break;
    }
    case 0:  {
     break;
    }
    default: {
     label = 281;
     break L7;
    }
    }
    $798 = HEAP32[$2>>2]|0;
    _memcpy(($1|0),($798|0),224)|0;
    _memcpy(($798|0),($0|0),224)|0;
    _memcpy(($0|0),($1|0),224)|0;
    $799 = ((($798)) + 8|0);
    $800 = HEAP32[$799>>2]|0;
    $801 = HEAP32[$22>>2]|0;
    HEAP32[$799>>2] = $801;
    HEAP32[$22>>2] = $800;
    $802 = ((($798)) + 4|0);
    $803 = HEAP32[$802>>2]|0;
    $804 = HEAP32[$23>>2]|0;
    HEAP32[$802>>2] = $804;
    HEAP32[$23>>2] = $803;
    $805 = ((($798)) + 188|0);
    $806 = HEAP32[$805>>2]|0;
    $807 = HEAP32[$20>>2]|0;
    HEAP32[$805>>2] = $807;
    HEAP32[$20>>2] = $806;
    $808 = ((($798)) + 208|0);
    $809 = HEAP32[$808>>2]|0;
    HEAP32[$24>>2] = $809;
    $810 = ((($798)) + 148|0);
    $811 = HEAP8[$810>>0]|0;
    HEAP8[$18>>0] = $811;
    $812 = ((($798)) + 160|0);
    ;HEAP32[$25>>2]=HEAP32[$812>>2]|0;HEAP32[$25+4>>2]=HEAP32[$812+4>>2]|0;HEAP32[$25+8>>2]=HEAP32[$812+8>>2]|0;HEAP32[$25+12>>2]=HEAP32[$812+12>>2]|0;HEAP32[$25+16>>2]=HEAP32[$812+16>>2]|0;HEAP32[$25+20>>2]=HEAP32[$812+20>>2]|0;HEAP32[$25+24>>2]=HEAP32[$812+24>>2]|0;
    $813 = HEAP32[$26>>2]|0;
    $814 = (($813) + 1)|0;
    HEAP32[$26>>2] = $814;
    $815 = ((($798)) + 16|0);
    $816 = HEAP16[$815>>1]|0;
    $817 = ($816<<16>>16)>(0);
    if ($817) {
     $818 = ((($798)) + 100|0);
     $$01$i$i = 0;
     while(1) {
      $819 = HEAP32[$28>>2]|0;
      $820 = (($819) + (($$01$i$i*40)|0)|0);
      $821 = HEAP32[$818>>2]|0;
      $822 = (($821) + (($$01$i$i*40)|0)|0);
      $823 = (((($819) + (($$01$i$i*40)|0)|0)) + 8|0);
      $824 = HEAP16[$823>>1]|0;
      $825 = $824 & 9216;
      $826 = ($825<<16>>16)==(0);
      if ($826) {
       $827 = (((($819) + (($$01$i$i*40)|0)|0)) + 24|0);
       $828 = HEAP32[$827>>2]|0;
       $829 = ($828|0)==(0);
       if (!($829)) {
        label = 261;
       }
      } else {
       label = 261;
      }
      if ((label|0) == 261) {
       label = 0;
       _vdbeMemClear($820);
      }
      dest=$820; src=$822; stop=dest+40|0; do { HEAP32[dest>>2]=HEAP32[src>>2]|0; dest=dest+4|0; src=src+4|0; } while ((dest|0) < (stop|0));
      $830 = (((($821) + (($$01$i$i*40)|0)|0)) + 8|0);
      HEAP16[$830>>1] = 1;
      $831 = (((($821) + (($$01$i$i*40)|0)|0)) + 24|0);
      HEAP32[$831>>2] = 0;
      $832 = (($$01$i$i) + 1)|0;
      $833 = HEAP16[$815>>1]|0;
      $834 = $833 << 16 >> 16;
      $835 = ($832|0)<($834|0);
      if ($835) {
       $$01$i$i = $832;
      } else {
       break;
      }
     }
    }
    $836 = ((($798)) + 40|0);
    HEAP32[$836>>2] = 0;
    $837 = ((($798)) + 20|0);
    $838 = HEAP32[$837>>2]|0;
    $switch$split2D = ($838|0)<(832317811);
    if ($switch$split2D) {
     switch ($838|0) {
     case 770837923:  {
      label = 264;
      break;
     }
     default: {
     }
     }
    } else {
     switch ($838|0) {
     case 832317811:  {
      label = 264;
      break;
     }
     default: {
     }
     }
    }
    if ((label|0) == 264) {
     label = 0;
     (_sqlite3VdbeReset($798)|0);
    }
    $839 = HEAP32[$798>>2]|0;
    _sqlite3VdbeClearObject($839,$798);
    $840 = HEAP32[$802>>2]|0;
    $841 = ($840|0)==(0|0);
    $842 = HEAP32[$799>>2]|0;
    $843 = $840;
    if ($841) {
     $845 = $842;
     $846 = ((($839)) + 4|0);
     HEAP32[$846>>2] = $842;
     $848 = $845;
    } else {
     $844 = ((($840)) + 8|0);
     HEAP32[$844>>2] = $842;
     $$pre$i$i$i50 = HEAP32[$799>>2]|0;
     $848 = $$pre$i$i$i50;
    }
    $847 = ($848|0)==(0|0);
    if (!($847)) {
     $849 = ((($848)) + 4|0);
     HEAP32[$849>>2] = $843;
    }
    HEAP32[$837>>2] = 1443283912;
    HEAP32[$798>>2] = 0;
    $850 = ($839|0)==(0|0);
    do {
     if ($850) {
      label = 276;
     } else {
      $851 = ((($839)) + 480|0);
      $852 = HEAP32[$851>>2]|0;
      $853 = ($852|0)==(0|0);
      if (!($853)) {
       _measureAllocationSize($839,$798);
       break;
      }
      $854 = $798;
      $855 = ((($839)) + 304|0);
      $856 = HEAP32[$855>>2]|0;
      $857 = ($856>>>0)>($854>>>0);
      if ($857) {
       label = 276;
      } else {
       $858 = ((($839)) + 308|0);
       $859 = HEAP32[$858>>2]|0;
       $860 = ($859>>>0)>($854>>>0);
       if ($860) {
        $861 = ((($839)) + 300|0);
        $862 = HEAP32[$861>>2]|0;
        HEAP32[$798>>2] = $862;
        HEAP32[$861>>2] = $798;
       } else {
        label = 276;
       }
      }
     }
    } while(0);
    do {
     if ((label|0) == 276) {
      label = 0;
      $863 = HEAP32[7324]|0;
      $864 = ($863|0)==(0);
      if ($864) {
       $872 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$872 & 127]($798);
       break;
      } else {
       $865 = HEAP32[(29352)>>2]|0;
       $866 = (FUNCTION_TABLE_ii[$865 & 127]($798)|0);
       $867 = HEAP32[14978]|0;
       $868 = (($867) - ($866))|0;
       HEAP32[14978] = $868;
       $869 = HEAP32[14981]|0;
       $870 = (($869) + -1)|0;
       HEAP32[14981] = $870;
       $871 = HEAP32[(29344)>>2]|0;
       FUNCTION_TABLE_vi[$871 & 127]($798);
       break;
      }
     }
    } while(0);
    $873 = HEAP32[$0>>2]|0;
    $874 = $9;
    $875 = $874;
    $876 = HEAP32[$875>>2]|0;
    $877 = (($874) + 4)|0;
    $878 = $877;
    $879 = HEAP32[$878>>2]|0;
    $880 = ($879|0)>(0);
    $881 = ($876>>>0)>(0);
    $882 = ($879|0)==(0);
    $883 = $882 & $881;
    $884 = $880 | $883;
    if ($884) {
     _invokeProfileCallback($873,$0);
    }
    $968 = (_sqlite3VdbeReset($0)|0);
    HEAP32[$8>>2] = 770837923;
    HEAP32[$10>>2] = -1;
    HEAP32[$11>>2] = 0;
    HEAP8[$12>>0] = 2;
    HEAP32[$13>>2] = 0;
    HEAP32[$14>>2] = 1;
    HEAP8[$15>>0] = -1;
    HEAP32[$16>>2] = 0;
    $969 = $17;
    $970 = $969;
    HEAP32[$970>>2] = 0;
    $971 = (($969) + 4)|0;
    $972 = $971;
    HEAP32[$972>>2] = 0;
    $973 = ((($873)) + 81|0);
    $974 = HEAP8[$973>>0]|0;
    $975 = ($974<<24>>24)!=(0);
    $976 = ($968|0)==(3082);
    $or$cond$i$i = $976 | $975;
    if ($or$cond$i$i) {
     _apiOomError($873);
    }
    $977 = ($777|0)>(-1);
    if ($977) {
     $978 = HEAPU8[$5>>0]|(HEAPU8[$5+1>>0]<<8);
     $979 = $978 | 16;
     HEAP8[$5>>0]=$979&255;HEAP8[$5+1>>0]=$979>>8;
    }
    $$039 = $775;
   }
   if ((label|0) == 13) {
    HEAP32[$11>>2] = 7;
    $$0 = 7;
    STACKTOP = sp;return ($$0|0);
   }
   else if ((label|0) == 251) {
    $781 = ((($$idx$val$i47)) + 81|0);
    $782 = HEAP8[$781>>0]|0;
    $783 = ($782<<24>>24)==(0);
    if ($783) {
     $784 = ((($$idx$val$i47)) + 82|0);
     $785 = HEAP8[$784>>0]|0;
     $786 = ($785<<24>>24)==(0);
     if ($786) {
      HEAP8[$781>>0] = 1;
      $787 = ((($$idx$val$i47)) + 180|0);
      $788 = HEAP32[$787>>2]|0;
      $789 = ($788|0)>(0);
      if ($789) {
       $790 = ((($$idx$val$i47)) + 264|0);
       HEAP32[$790>>2] = 1;
      }
      $791 = ((($$idx$val$i47)) + 272|0);
      $792 = HEAP32[$791>>2]|0;
      $793 = (($792) + 1)|0;
      HEAP32[$791>>2] = $793;
      $794 = ((($$idx$val$i47)) + 236|0);
      $795 = HEAP32[$794>>2]|0;
      $796 = ($795|0)==(0|0);
      if (!($796)) {
       $797 = ((($795)) + 12|0);
       HEAP32[$797>>2] = 7;
      }
     }
    }
    $$0$i5161 = 7;
   }
   else if ((label|0) == 281) {
    $$0$i5161 = $780;
   }
   else if ((label|0) == 322) {
    STACKTOP = sp;return ($$0|0);
   }
   $885 = ((($$idx$val$i)) + 260|0);
   $886 = HEAP32[$885>>2]|0;
   $887 = ($886|0)==(0|0);
   do {
    if ($887) {
     $$0$i$i = 0;
    } else {
     $888 = ((($886)) + 8|0);
     $889 = HEAP16[$888>>1]|0;
     $890 = $889 & 514;
     $891 = ($890<<16>>16)==(514);
     if ($891) {
      $892 = ((($886)) + 10|0);
      $893 = HEAP8[$892>>0]|0;
      $894 = ($893<<24>>24)==(1);
      if ($894) {
       $895 = ((($886)) + 16|0);
       $896 = HEAP32[$895>>2]|0;
       $$0$i$i = $896;
       break;
      }
     }
     $897 = $889 & 1;
     $898 = ($897<<16>>16)==(0);
     if ($898) {
      $899 = (_valueToText($886,1)|0);
      $$0$i$i = $899;
     } else {
      $$0$i$i = 0;
     }
    }
   } while(0);
   $900 = ((($0)) + 124|0);
   $901 = HEAP32[$900>>2]|0;
   $902 = ($901|0)==(0|0);
   do {
    if (!($902)) {
     $903 = ((($$idx$val$i)) + 480|0);
     $904 = HEAP32[$903>>2]|0;
     $905 = ($904|0)==(0|0);
     if (!($905)) {
      _measureAllocationSize($$idx$val$i,$901);
      break;
     }
     $906 = $901;
     $907 = ((($$idx$val$i)) + 304|0);
     $908 = HEAP32[$907>>2]|0;
     $909 = ($908>>>0)>($906>>>0);
     if (!($909)) {
      $910 = ((($$idx$val$i)) + 308|0);
      $911 = HEAP32[$910>>2]|0;
      $912 = ($911>>>0)>($906>>>0);
      if ($912) {
       $913 = ((($$idx$val$i)) + 300|0);
       $914 = HEAP32[$913>>2]|0;
       HEAP32[$901>>2] = $914;
       HEAP32[$913>>2] = $901;
       break;
      }
     }
     $915 = HEAP32[7324]|0;
     $916 = ($915|0)==(0);
     if ($916) {
      $924 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$924 & 127]($901);
      break;
     } else {
      $917 = HEAP32[(29352)>>2]|0;
      $918 = (FUNCTION_TABLE_ii[$917 & 127]($901)|0);
      $919 = HEAP32[14978]|0;
      $920 = (($919) - ($918))|0;
      HEAP32[14978] = $920;
      $921 = HEAP32[14981]|0;
      $922 = (($921) + -1)|0;
      HEAP32[14981] = $922;
      $923 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$923 & 127]($901);
      break;
     }
    }
   } while(0);
   $925 = ((($$idx$val$i)) + 81|0);
   $926 = HEAP8[$925>>0]|0;
   $927 = ($926<<24>>24)==(0);
   if (!($927)) {
    HEAP32[$900>>2] = 0;
    HEAP32[$11>>2] = 7;
    $$0 = 7;
    STACKTOP = sp;return ($$0|0);
   }
   $928 = ($$0$i$i|0)==(0|0);
   if ($928) {
    $$0$i41 = 0;
   } else {
    $929 = (_strlen($$0$i$i)|0);
    $930 = (($929) + 1)|0;
    $931 = ((($$idx$val$i)) + 272|0);
    $932 = HEAP32[$931>>2]|0;
    $933 = ($932|0)==(0);
    L434: do {
     if ($933) {
      $934 = ((($$idx$val$i)) + 276|0);
      $935 = HEAP16[$934>>1]|0;
      $936 = $935&65535;
      $937 = (0)<(0);
      $938 = ($936>>>0)<($930>>>0);
      $939 = (0)==(0);
      $940 = $939 & $938;
      $941 = $937 | $940;
      do {
       if ($941) {
        $942 = ((($$idx$val$i)) + 288|0);
        $$sink = $942;
       } else {
        $943 = ((($$idx$val$i)) + 300|0);
        $944 = HEAP32[$943>>2]|0;
        $945 = ($944|0)==(0|0);
        if (!($945)) {
         $946 = HEAP32[$944>>2]|0;
         HEAP32[$943>>2] = $946;
         $947 = ((($$idx$val$i)) + 284|0);
         $948 = HEAP32[$947>>2]|0;
         $949 = (($948) + 1)|0;
         HEAP32[$947>>2] = $949;
         $$0$i52 = $944;
         break L434;
        }
        $950 = ((($$idx$val$i)) + 296|0);
        $951 = HEAP32[$950>>2]|0;
        $952 = ($951|0)==(0|0);
        if ($952) {
         $957 = ((($$idx$val$i)) + 292|0);
         $$sink = $957;
         break;
        } else {
         $953 = HEAP32[$951>>2]|0;
         HEAP32[$950>>2] = $953;
         $954 = ((($$idx$val$i)) + 284|0);
         $955 = HEAP32[$954>>2]|0;
         $956 = (($955) + 1)|0;
         HEAP32[$954>>2] = $956;
         $$0$i52 = $951;
         break L434;
        }
       }
      } while(0);
      $958 = HEAP32[$$sink>>2]|0;
      $959 = (($958) + 1)|0;
      HEAP32[$$sink>>2] = $959;
      label = 308;
     } else {
      label = 308;
     }
    } while(0);
    if ((label|0) == 308) {
     $960 = (_dbMallocRawFinish($$idx$val$i,$930,0)|0);
     $$0$i52 = $960;
    }
    $961 = ($$0$i52|0)==(0|0);
    if ($961) {
     $$0$i41 = 0;
    } else {
     _memcpy(($$0$i52|0),($$0$i$i|0),($930|0))|0;
     $$0$i41 = $$0$i52;
    }
   }
   HEAP32[$900>>2] = $$0$i41;
   $962 = HEAP8[$925>>0]|0;
   $963 = ($962<<24>>24)!=(0);
   $964 = ($$0$i5161|0)==(3082);
   $or$cond$i = $964 | $963;
   if ($or$cond$i) {
    _apiOomError($$idx$val$i);
    $$0$i40 = 7;
   } else {
    $965 = ((($$idx$val$i)) + 68|0);
    $966 = HEAP32[$965>>2]|0;
    $967 = $966 & $$0$i5161;
    $$0$i40 = $967;
   }
   HEAP32[$11>>2] = $$0$i40;
   $$0 = $$0$i40;
   STACKTOP = sp;return ($$0|0);
  }
 } while(0);
 HEAP32[$vararg_buffer3>>2] = 58063;
 $vararg_ptr5 = ((($vararg_buffer3)) + 4|0);
 HEAP32[$vararg_ptr5>>2] = 81990;
 $vararg_ptr6 = ((($vararg_buffer3)) + 8|0);
 HEAP32[$vararg_ptr6>>2] = (31517);
 _sqlite3_log(21,32001,$vararg_buffer3);
 $$0 = 21;
 STACKTOP = sp;return ($$0|0);
}
function _sqlite3_aggregate_context($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0 = 0, $10 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($0)) + 8|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ((($3)) + 8|0);
 $5 = HEAP16[$4>>1]|0;
 $6 = $5 & 8192;
 $7 = ($6<<16>>16)==(0);
 if ($7) {
  $8 = (_createAggContext($0,$1)|0);
  $$0 = $8;
  return ($$0|0);
 } else {
  $9 = ((($3)) + 16|0);
  $10 = HEAP32[$9>>2]|0;
  $$0 = $10;
  return ($$0|0);
 }
 return (0)|0;
}
function _createAggContext($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$phi$trans$insert = 0, $$pre = 0, $$pre$phi15Z2D = 0, $$pre$phi16Z2D = 0, $$pre13 = 0, $$pre14 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0;
 var $23 = 0, $24 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ((($0)) + 8|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ($1|0)<(1);
 if ($4) {
  $5 = ((($3)) + 8|0);
  $6 = HEAP16[$5>>1]|0;
  $7 = $6 & 9216;
  $8 = ($7<<16>>16)==(0);
  if ($8) {
   HEAP16[$5>>1] = 1;
  } else {
   _vdbeMemClearExternAndSetNull($3);
  }
  $9 = ((($3)) + 16|0);
  HEAP32[$9>>2] = 0;
  $24 = 0;
  return ($24|0);
 }
 $10 = ((($3)) + 24|0);
 $11 = HEAP32[$10>>2]|0;
 $12 = ($11|0)<($1|0);
 if ($12) {
  (_sqlite3VdbeMemGrow($3,$1,0)|0);
  $$phi$trans$insert = ((($3)) + 16|0);
  $$pre = HEAP32[$$phi$trans$insert>>2]|0;
  $$pre14 = ((($3)) + 8|0);
  $$pre$phi15Z2D = $$pre14;$$pre$phi16Z2D = $$phi$trans$insert;$23 = $$pre;
 } else {
  $13 = ((($3)) + 20|0);
  $14 = HEAP32[$13>>2]|0;
  $15 = ((($3)) + 16|0);
  HEAP32[$15>>2] = $14;
  $16 = ((($3)) + 8|0);
  $17 = HEAP16[$16>>1]|0;
  $18 = $17 & 13;
  HEAP16[$16>>1] = $18;
  $19 = $14;
  $$pre$phi15Z2D = $16;$$pre$phi16Z2D = $15;$23 = $19;
 }
 HEAP16[$$pre$phi15Z2D>>1] = 8192;
 $20 = ((($0)) + 4|0);
 $21 = HEAP32[$20>>2]|0;
 HEAP32[$3>>2] = $21;
 $22 = ($23|0)==(0|0);
 if ($22) {
  $24 = $23;
  return ($24|0);
 }
 _memset(($23|0),0,($1|0))|0;
 $$pre13 = HEAP32[$$pre$phi16Z2D>>2]|0;
 $24 = $$pre13;
 return ($24|0);
}
function _sqlite3_data_count($0) {
 $0 = $0|0;
 var $$0 = 0, $1 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $1 = ($0|0)==(0|0);
 if ($1) {
  $$0 = 0;
  return ($$0|0);
 }
 $2 = ((($0)) + 120|0);
 $3 = HEAP32[$2>>2]|0;
 $4 = ($3|0)==(0|0);
 if ($4) {
  $$0 = 0;
  return ($$0|0);
 }
 $5 = ((($0)) + 144|0);
 $6 = HEAP16[$5>>1]|0;
 $7 = $6&65535;
 $$0 = $7;
 return ($$0|0);
}
function _sqlite3_column_blob($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0, $$014$i$ph = 0, $$pre$phiZ2D = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 if ($2) {
  $13 = (_sqlite3_value_blob(29576)|0);
  $25 = $13;
  return ($25|0);
 }
 $3 = ((($0)) + 120|0);
 $4 = HEAP32[$3>>2]|0;
 $5 = ($4|0)==(0|0);
 if ($5) {
  label = 5;
 } else {
  $6 = ((($0)) + 144|0);
  $7 = HEAP16[$6>>1]|0;
  $8 = $7&65535;
  $9 = ($8>>>0)>($1>>>0);
  if ($9) {
   $10 = (($4) + (($1*40)|0)|0);
   $$014$i$ph = $10;$$pre$phiZ2D = $0;
  } else {
   label = 5;
  }
 }
 if ((label|0) == 5) {
  $11 = HEAP32[$0>>2]|0;
  $12 = ((($11)) + 64|0);
  HEAP32[$12>>2] = 25;
  _sqlite3ErrorFinish($11,25);
  $$014$i$ph = 29576;$$pre$phiZ2D = $0;
 }
 $14 = (_sqlite3_value_blob($$014$i$ph)|0);
 $15 = HEAP32[$$pre$phiZ2D>>2]|0;
 $16 = ((($0)) + 40|0);
 $17 = HEAP32[$16>>2]|0;
 $18 = ((($15)) + 81|0);
 $19 = HEAP8[$18>>0]|0;
 $20 = ($19<<24>>24)!=(0);
 $21 = ($17|0)==(3082);
 $or$cond$i$i = $21 | $20;
 if ($or$cond$i$i) {
  _apiOomError($15);
  $$0$i$i = 7;
 } else {
  $22 = ((($15)) + 68|0);
  $23 = HEAP32[$22>>2]|0;
  $24 = $23 & $17;
  $$0$i$i = $24;
 }
 HEAP32[$16>>2] = $$0$i$i;
 $25 = $14;
 return ($25|0);
}
function _sqlite3_column_bytes($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0, $$0$i$i4 = 0, $$014$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0;
 var $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   $$014$i = 29576;
  } else {
   $3 = ((($0)) + 120|0);
   $4 = HEAP32[$3>>2]|0;
   $5 = ($4|0)==(0|0);
   if (!($5)) {
    $6 = ((($0)) + 144|0);
    $7 = HEAP16[$6>>1]|0;
    $8 = $7&65535;
    $9 = ($8>>>0)>($1>>>0);
    if ($9) {
     $10 = (($4) + (($1*40)|0)|0);
     $$014$i = $10;
     break;
    }
   }
   $11 = HEAP32[$0>>2]|0;
   $12 = ((($11)) + 64|0);
   HEAP32[$12>>2] = 25;
   _sqlite3ErrorFinish($11,25);
   $$014$i = 29576;
  }
 } while(0);
 $13 = ((($$014$i)) + 8|0);
 $14 = HEAP16[$13>>1]|0;
 $15 = $14 & 2;
 $16 = ($15<<16>>16)==(0);
 if ($16) {
  label = 9;
 } else {
  $17 = ((($$014$i)) + 10|0);
  $18 = HEAP8[$17>>0]|0;
  $19 = ($18<<24>>24)==(1);
  if ($19) {
   $20 = ((($$014$i)) + 12|0);
   $21 = HEAP32[$20>>2]|0;
   $$0$i$i = $21;
  } else {
   label = 9;
  }
 }
 do {
  if ((label|0) == 9) {
   $22 = $14&65535;
   $23 = $22 & 16;
   $24 = ($23|0)==(0);
   if ($24) {
    $31 = $22 & 1;
    $32 = ($31|0)==(0);
    if (!($32)) {
     $$0$i$i = 0;
     break;
    }
    $33 = (_valueBytes($$014$i,1)|0);
    $$0$i$i = $33;
    break;
   } else {
    $25 = $22 & 16384;
    $26 = ($25|0)==(0);
    $27 = ((($$014$i)) + 12|0);
    $28 = HEAP32[$27>>2]|0;
    if ($26) {
     $$0$i$i = $28;
     break;
    }
    $29 = HEAP32[$$014$i>>2]|0;
    $30 = (($29) + ($28))|0;
    $$0$i$i = $30;
    break;
   }
  }
 } while(0);
 if ($2) {
  return ($$0$i$i|0);
 }
 $34 = HEAP32[$0>>2]|0;
 $35 = ((($0)) + 40|0);
 $36 = HEAP32[$35>>2]|0;
 $37 = ((($34)) + 81|0);
 $38 = HEAP8[$37>>0]|0;
 $39 = ($38<<24>>24)!=(0);
 $40 = ($36|0)==(3082);
 $or$cond$i$i = $40 | $39;
 if ($or$cond$i$i) {
  _apiOomError($34);
  $$0$i$i4 = 7;
 } else {
  $41 = ((($34)) + 68|0);
  $42 = HEAP32[$41>>2]|0;
  $43 = $42 & $36;
  $$0$i$i4 = $43;
 }
 HEAP32[$35>>2] = $$0$i$i4;
 return ($$0$i$i|0);
}
function _sqlite3_column_double($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0.0, $$0$i$i4 = 0, $$014$i = 0, $$idx$i$i = 0, $$idx$val$i$i = 0, $$idx5$i$i = 0, $$idx5$val$i$i = 0, $$idx6$i$i = 0, $$idx6$val$i$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0.0, $19 = 0, $2 = 0;
 var $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0.0, $28 = 0, $29 = 0, $3 = 0, $30 = 0.0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0;
 var $39 = 0, $4 = 0, $40 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   $$014$i = 29576;
  } else {
   $3 = ((($0)) + 120|0);
   $4 = HEAP32[$3>>2]|0;
   $5 = ($4|0)==(0|0);
   if (!($5)) {
    $6 = ((($0)) + 144|0);
    $7 = HEAP16[$6>>1]|0;
    $8 = $7&65535;
    $9 = ($8>>>0)>($1>>>0);
    if ($9) {
     $10 = (($4) + (($1*40)|0)|0);
     $$014$i = $10;
     break;
    }
   }
   $11 = HEAP32[$0>>2]|0;
   $12 = ((($11)) + 64|0);
   HEAP32[$12>>2] = 25;
   _sqlite3ErrorFinish($11,25);
   $$014$i = 29576;
  }
 } while(0);
 $13 = ((($$014$i)) + 8|0);
 $14 = HEAP16[$13>>1]|0;
 $15 = $14&65535;
 $16 = $15 & 8;
 $17 = ($16|0)==(0);
 do {
  if ($17) {
   $19 = $15 & 4;
   $20 = ($19|0)==(0);
   if (!($20)) {
    $21 = $$014$i;
    $22 = $21;
    $23 = HEAP32[$22>>2]|0;
    $24 = (($21) + 4)|0;
    $25 = $24;
    $26 = HEAP32[$25>>2]|0;
    $27 = (+($23>>>0)) + (4294967296.0*(+($26|0)));
    $$0$i$i = $27;
    break;
   }
   $28 = $15 & 18;
   $29 = ($28|0)==(0);
   if ($29) {
    $$0$i$i = 0.0;
   } else {
    $$idx$i$i = ((($$014$i)) + 10|0);
    $$idx$val$i$i = HEAP8[$$idx$i$i>>0]|0;
    $$idx5$i$i = ((($$014$i)) + 12|0);
    $$idx5$val$i$i = HEAP32[$$idx5$i$i>>2]|0;
    $$idx6$i$i = ((($$014$i)) + 16|0);
    $$idx6$val$i$i = HEAP32[$$idx6$i$i>>2]|0;
    $30 = (+_memRealValue($$idx$val$i$i,$$idx5$val$i$i,$$idx6$val$i$i));
    $$0$i$i = $30;
   }
  } else {
   $18 = +HEAPF64[$$014$i>>3];
   $$0$i$i = $18;
  }
 } while(0);
 if ($2) {
  return (+$$0$i$i);
 }
 $31 = HEAP32[$0>>2]|0;
 $32 = ((($0)) + 40|0);
 $33 = HEAP32[$32>>2]|0;
 $34 = ((($31)) + 81|0);
 $35 = HEAP8[$34>>0]|0;
 $36 = ($35<<24>>24)!=(0);
 $37 = ($33|0)==(3082);
 $or$cond$i$i = $37 | $36;
 if ($or$cond$i$i) {
  _apiOomError($31);
  $$0$i$i4 = 7;
 } else {
  $38 = ((($31)) + 68|0);
  $39 = HEAP32[$38>>2]|0;
  $40 = $39 & $33;
  $$0$i$i4 = $40;
 }
 HEAP32[$32>>2] = $$0$i$i4;
 return (+$$0$i$i);
}
function _sqlite3_column_int($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0, $$0$i$off0$i = 0, $$014$i = 0, $$idx$i$i = 0, $$idx$val$i$i = 0, $$idx8$i$i = 0, $$idx8$val$i$i = 0, $$idx9$i$i = 0, $$idx9$val$i$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0;
 var $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0.0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0;
 var $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   $$014$i = 29576;
  } else {
   $3 = ((($0)) + 120|0);
   $4 = HEAP32[$3>>2]|0;
   $5 = ($4|0)==(0|0);
   if (!($5)) {
    $6 = ((($0)) + 144|0);
    $7 = HEAP16[$6>>1]|0;
    $8 = $7&65535;
    $9 = ($8>>>0)>($1>>>0);
    if ($9) {
     $10 = (($4) + (($1*40)|0)|0);
     $$014$i = $10;
     break;
    }
   }
   $11 = HEAP32[$0>>2]|0;
   $12 = ((($11)) + 64|0);
   HEAP32[$12>>2] = 25;
   _sqlite3ErrorFinish($11,25);
   $$014$i = 29576;
  }
 } while(0);
 $13 = ((($$014$i)) + 8|0);
 $14 = HEAP16[$13>>1]|0;
 $15 = $14&65535;
 $16 = $15 & 4;
 $17 = ($16|0)==(0);
 do {
  if ($17) {
   $24 = $15 & 8;
   $25 = ($24|0)==(0);
   if (!($25)) {
    $26 = +HEAPF64[$$014$i>>3];
    $27 = (_doubleToInt64($26)|0);
    $28 = (getTempRet0() | 0);
    $$0$i$off0$i = $27;
    break;
   }
   $29 = $15 & 18;
   $30 = ($29|0)==(0);
   if ($30) {
    $$0$i$off0$i = 0;
   } else {
    $$idx$i$i = ((($$014$i)) + 10|0);
    $$idx$val$i$i = HEAP8[$$idx$i$i>>0]|0;
    $$idx8$i$i = ((($$014$i)) + 12|0);
    $$idx8$val$i$i = HEAP32[$$idx8$i$i>>2]|0;
    $$idx9$i$i = ((($$014$i)) + 16|0);
    $$idx9$val$i$i = HEAP32[$$idx9$i$i>>2]|0;
    $31 = (_memIntValue($$idx$val$i$i,$$idx8$val$i$i,$$idx9$val$i$i)|0);
    $32 = (getTempRet0() | 0);
    $$0$i$off0$i = $31;
   }
  } else {
   $18 = $$014$i;
   $19 = $18;
   $20 = HEAP32[$19>>2]|0;
   $21 = (($18) + 4)|0;
   $22 = $21;
   $23 = HEAP32[$22>>2]|0;
   $$0$i$off0$i = $20;
  }
 } while(0);
 if ($2) {
  return ($$0$i$off0$i|0);
 }
 $33 = HEAP32[$0>>2]|0;
 $34 = ((($0)) + 40|0);
 $35 = HEAP32[$34>>2]|0;
 $36 = ((($33)) + 81|0);
 $37 = HEAP8[$36>>0]|0;
 $38 = ($37<<24>>24)!=(0);
 $39 = ($35|0)==(3082);
 $or$cond$i$i = $39 | $38;
 if ($or$cond$i$i) {
  _apiOomError($33);
  $$0$i$i = 7;
 } else {
  $40 = ((($33)) + 68|0);
  $41 = HEAP32[$40>>2]|0;
  $42 = $41 & $35;
  $$0$i$i = $42;
 }
 HEAP32[$34>>2] = $$0$i$i;
 return ($$0$i$off0$i|0);
}
function _sqlite3_column_int64($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i4 = 0, $$014$i = 0, $$idx$i$i = 0, $$idx$val$i$i = 0, $$idx8$i$i = 0, $$idx8$val$i$i = 0, $$idx9$i$i = 0, $$idx9$val$i$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0;
 var $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0.0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0;
 var $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   $$014$i = 29576;
  } else {
   $3 = ((($0)) + 120|0);
   $4 = HEAP32[$3>>2]|0;
   $5 = ($4|0)==(0|0);
   if (!($5)) {
    $6 = ((($0)) + 144|0);
    $7 = HEAP16[$6>>1]|0;
    $8 = $7&65535;
    $9 = ($8>>>0)>($1>>>0);
    if ($9) {
     $10 = (($4) + (($1*40)|0)|0);
     $$014$i = $10;
     break;
    }
   }
   $11 = HEAP32[$0>>2]|0;
   $12 = ((($11)) + 64|0);
   HEAP32[$12>>2] = 25;
   _sqlite3ErrorFinish($11,25);
   $$014$i = 29576;
  }
 } while(0);
 $13 = ((($$014$i)) + 8|0);
 $14 = HEAP16[$13>>1]|0;
 $15 = $14&65535;
 $16 = $15 & 4;
 $17 = ($16|0)==(0);
 do {
  if ($17) {
   $24 = $15 & 8;
   $25 = ($24|0)==(0);
   if (!($25)) {
    $26 = +HEAPF64[$$014$i>>3];
    $27 = (_doubleToInt64($26)|0);
    $28 = (getTempRet0() | 0);
    $43 = $28;$44 = $27;
    break;
   }
   $29 = $15 & 18;
   $30 = ($29|0)==(0);
   if ($30) {
    $43 = 0;$44 = 0;
   } else {
    $$idx$i$i = ((($$014$i)) + 10|0);
    $$idx$val$i$i = HEAP8[$$idx$i$i>>0]|0;
    $$idx8$i$i = ((($$014$i)) + 12|0);
    $$idx8$val$i$i = HEAP32[$$idx8$i$i>>2]|0;
    $$idx9$i$i = ((($$014$i)) + 16|0);
    $$idx9$val$i$i = HEAP32[$$idx9$i$i>>2]|0;
    $31 = (_memIntValue($$idx$val$i$i,$$idx8$val$i$i,$$idx9$val$i$i)|0);
    $32 = (getTempRet0() | 0);
    $43 = $32;$44 = $31;
   }
  } else {
   $18 = $$014$i;
   $19 = $18;
   $20 = HEAP32[$19>>2]|0;
   $21 = (($18) + 4)|0;
   $22 = $21;
   $23 = HEAP32[$22>>2]|0;
   $43 = $23;$44 = $20;
  }
 } while(0);
 if ($2) {
  setTempRet0(($43) | 0);
  return ($44|0);
 }
 $33 = HEAP32[$0>>2]|0;
 $34 = ((($0)) + 40|0);
 $35 = HEAP32[$34>>2]|0;
 $36 = ((($33)) + 81|0);
 $37 = HEAP8[$36>>0]|0;
 $38 = ($37<<24>>24)!=(0);
 $39 = ($35|0)==(3082);
 $or$cond$i$i = $39 | $38;
 if ($or$cond$i$i) {
  _apiOomError($33);
  $$0$i$i4 = 7;
 } else {
  $40 = ((($33)) + 68|0);
  $41 = HEAP32[$40>>2]|0;
  $42 = $41 & $35;
  $$0$i$i4 = $42;
 }
 HEAP32[$34>>2] = $$0$i$i4;
 setTempRet0(($43) | 0);
 return ($44|0);
}
function _sqlite3_column_text($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0, $$0$i$i4 = 0, $$014$i5 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   $$014$i5 = 29576;
  } else {
   $3 = ((($0)) + 120|0);
   $4 = HEAP32[$3>>2]|0;
   $5 = ($4|0)==(0|0);
   if (!($5)) {
    $6 = ((($0)) + 144|0);
    $7 = HEAP16[$6>>1]|0;
    $8 = $7&65535;
    $9 = ($8>>>0)>($1>>>0);
    if ($9) {
     $12 = (($4) + (($1*40)|0)|0);
     $$014$i5 = $12;
     break;
    }
   }
   $10 = HEAP32[$0>>2]|0;
   $11 = ((($10)) + 64|0);
   HEAP32[$11>>2] = 25;
   _sqlite3ErrorFinish($10,25);
   $$014$i5 = 29576;
  }
 } while(0);
 $13 = ((($$014$i5)) + 8|0);
 $14 = HEAP16[$13>>1]|0;
 $15 = $14 & 514;
 $16 = ($15<<16>>16)==(514);
 if ($16) {
  $17 = ((($$014$i5)) + 10|0);
  $18 = HEAP8[$17>>0]|0;
  $19 = ($18<<24>>24)==(1);
  if ($19) {
   $20 = ((($$014$i5)) + 16|0);
   $21 = HEAP32[$20>>2]|0;
   $$0$i$i = $21;
  } else {
   label = 9;
  }
 } else {
  label = 9;
 }
 if ((label|0) == 9) {
  $22 = $14 & 1;
  $23 = ($22<<16>>16)==(0);
  if ($23) {
   $24 = (_valueToText($$014$i5,1)|0);
   $$0$i$i = $24;
  } else {
   $$0$i$i = 0;
  }
 }
 if ($2) {
  return ($$0$i$i|0);
 }
 $25 = HEAP32[$0>>2]|0;
 $26 = ((($0)) + 40|0);
 $27 = HEAP32[$26>>2]|0;
 $28 = ((($25)) + 81|0);
 $29 = HEAP8[$28>>0]|0;
 $30 = ($29<<24>>24)!=(0);
 $31 = ($27|0)==(3082);
 $or$cond$i$i = $31 | $30;
 if ($or$cond$i$i) {
  _apiOomError($25);
  $$0$i$i4 = 7;
 } else {
  $32 = ((($25)) + 68|0);
  $33 = HEAP32[$32>>2]|0;
  $34 = $33 & $27;
  $$0$i$i4 = $34;
 }
 HEAP32[$26>>2] = $$0$i$i4;
 return ($$0$i$i|0);
}
function _sqlite3_column_type($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i$i = 0, $$014$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   $$014$i = 29576;
  } else {
   $3 = ((($0)) + 120|0);
   $4 = HEAP32[$3>>2]|0;
   $5 = ($4|0)==(0|0);
   if (!($5)) {
    $6 = ((($0)) + 144|0);
    $7 = HEAP16[$6>>1]|0;
    $8 = $7&65535;
    $9 = ($8>>>0)>($1>>>0);
    if ($9) {
     $10 = (($4) + (($1*40)|0)|0);
     $$014$i = $10;
     break;
    }
   }
   $11 = HEAP32[$0>>2]|0;
   $12 = ((($11)) + 64|0);
   HEAP32[$12>>2] = 25;
   _sqlite3ErrorFinish($11,25);
   $$014$i = 29576;
  }
 } while(0);
 $13 = ((($$014$i)) + 8|0);
 $14 = HEAP16[$13>>1]|0;
 $15 = $14 & 31;
 $16 = $15&65535;
 $17 = (816 + ($16)|0);
 $18 = HEAP8[$17>>0]|0;
 $19 = $18&255;
 if ($2) {
  return ($19|0);
 }
 $20 = HEAP32[$0>>2]|0;
 $21 = ((($0)) + 40|0);
 $22 = HEAP32[$21>>2]|0;
 $23 = ((($20)) + 81|0);
 $24 = HEAP8[$23>>0]|0;
 $25 = ($24<<24>>24)!=(0);
 $26 = ($22|0)==(3082);
 $or$cond$i$i = $26 | $25;
 if ($or$cond$i$i) {
  _apiOomError($20);
  $$0$i$i = 7;
 } else {
  $27 = ((($20)) + 68|0);
  $28 = HEAP32[$27>>2]|0;
  $29 = $28 & $22;
  $$0$i$i = $29;
 }
 HEAP32[$21>>2] = $$0$i$i;
 return ($19|0);
}
function _sqlite3_column_name($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i = 0, $$1$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = HEAP32[$0>>2]|0;
 $3 = ($0|0)==(0|0);
 if ($3) {
  $$1$i = 0;
  return ($$1$i|0);
 }
 $4 = ((($0)) + 144|0);
 $5 = HEAP16[$4>>1]|0;
 $6 = $5&65535;
 $7 = ($6>>>0)>($1>>>0);
 if (!($7)) {
  $$1$i = 0;
  return ($$1$i|0);
 }
 $8 = ((($0)) + 116|0);
 $9 = HEAP32[$8>>2]|0;
 $10 = (($9) + (($1*40)|0)|0);
 $11 = ($10|0)==(0|0);
 do {
  if ($11) {
   $$0$i = 0;
  } else {
   $12 = (((($9) + (($1*40)|0)|0)) + 8|0);
   $13 = HEAP16[$12>>1]|0;
   $14 = $13 & 514;
   $15 = ($14<<16>>16)==(514);
   if ($15) {
    $16 = (((($9) + (($1*40)|0)|0)) + 10|0);
    $17 = HEAP8[$16>>0]|0;
    $18 = ($17<<24>>24)==(1);
    if ($18) {
     $19 = (((($9) + (($1*40)|0)|0)) + 16|0);
     $20 = HEAP32[$19>>2]|0;
     $$0$i = $20;
     break;
    }
   }
   $21 = $13 & 1;
   $22 = ($21<<16>>16)==(0);
   if ($22) {
    $23 = (_valueToText($10,1)|0);
    $$0$i = $23;
   } else {
    $$0$i = 0;
   }
  }
 } while(0);
 $24 = ((($2)) + 81|0);
 $25 = HEAP8[$24>>0]|0;
 $26 = ($25<<24>>24)==(0);
 if ($26) {
  $$1$i = $$0$i;
  return ($$1$i|0);
 }
 $27 = ((($2)) + 180|0);
 $28 = HEAP32[$27>>2]|0;
 $29 = ($28|0)==(0);
 if (!($29)) {
  $$1$i = 0;
  return ($$1$i|0);
 }
 HEAP8[$24>>0] = 0;
 $30 = ((($2)) + 264|0);
 HEAP32[$30>>2] = 0;
 $31 = ((($2)) + 272|0);
 $32 = HEAP32[$31>>2]|0;
 $33 = (($32) + -1)|0;
 HEAP32[$31>>2] = $33;
 $$1$i = 0;
 return ($$1$i|0);
}
function _sqlite3_bind_blob($0,$1,$2,$3,$4) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 $4 = $4|0;
 var $$1$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0;
 var $magicptr$i = 0, $or$cond$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $5 = (_vdbeUnbind($0,$1)|0);
 $6 = ($5|0)==(0);
 if (!($6)) {
  $magicptr$i = $4;
  switch ($magicptr$i|0) {
  case 0: case -1:  {
   $$1$i = $5;
   return ($$1$i|0);
   break;
  }
  default: {
  }
  }
  FUNCTION_TABLE_vi[$4 & 127]($2);
  $$1$i = $5;
  return ($$1$i|0);
 }
 $7 = ($2|0)==(0|0);
 if ($7) {
  $$1$i = 0;
  return ($$1$i|0);
 }
 $8 = ((($0)) + 100|0);
 $9 = HEAP32[$8>>2]|0;
 $10 = (($1) + -1)|0;
 $11 = (($9) + (($10*40)|0)|0);
 $12 = (_sqlite3VdbeMemSetStr($11,$2,$3,0,$4)|0);
 $13 = ($12|0)==(0);
 if ($13) {
  $$1$i = 0;
  return ($$1$i|0);
 }
 $14 = HEAP32[$0>>2]|0;
 $15 = ((($14)) + 64|0);
 HEAP32[$15>>2] = $12;
 _sqlite3ErrorFinish($14,$12);
 $16 = HEAP32[$0>>2]|0;
 $17 = ((($16)) + 81|0);
 $18 = HEAP8[$17>>0]|0;
 $19 = ($18<<24>>24)!=(0);
 $20 = ($12|0)==(3082);
 $or$cond$i$i = $20 | $19;
 if ($or$cond$i$i) {
  _apiOomError($16);
  $$1$i = 7;
  return ($$1$i|0);
 } else {
  $21 = ((($16)) + 68|0);
  $22 = HEAP32[$21>>2]|0;
  $23 = $22 & $12;
  $$1$i = $23;
  return ($$1$i|0);
 }
 return (0)|0;
}
function _bindText($0,$1,$2,$3,$4,$5) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 $4 = $4|0;
 $5 = $5|0;
 var $$0 = 0, $$1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0;
 var $28 = 0, $29 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $magicptr = 0, $or$cond$i = 0, $or$cond31 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $6 = (_vdbeUnbind($0,$1)|0);
 $7 = ($6|0)==(0);
 if (!($7)) {
  $magicptr = $4;
  switch ($magicptr|0) {
  case 0: case -1:  {
   $$1 = $6;
   return ($$1|0);
   break;
  }
  default: {
  }
  }
  FUNCTION_TABLE_vi[$4 & 127]($2);
  $$1 = $6;
  return ($$1|0);
 }
 $8 = ($2|0)==(0|0);
 if ($8) {
  $$1 = 0;
  return ($$1|0);
 }
 $9 = ((($0)) + 100|0);
 $10 = HEAP32[$9>>2]|0;
 $11 = (($1) + -1)|0;
 $12 = (($10) + (($11*40)|0)|0);
 $13 = (_sqlite3VdbeMemSetStr($12,$2,$3,$5,$4)|0);
 $14 = ($13|0)!=(0);
 $15 = ($5<<24>>24)==(0);
 $or$cond31 = $15 | $14;
 do {
  if ($or$cond31) {
   $$0 = $13;
  } else {
   $16 = HEAP32[$0>>2]|0;
   $17 = ((($16)) + 78|0);
   $18 = HEAP8[$17>>0]|0;
   $19 = (((($10) + (($11*40)|0)|0)) + 8|0);
   $20 = HEAP16[$19>>1]|0;
   $21 = $20 & 2;
   $22 = ($21<<16>>16)==(0);
   if ($22) {
    $$1 = 0;
    return ($$1|0);
   }
   $23 = (((($10) + (($11*40)|0)|0)) + 10|0);
   $24 = HEAP8[$23>>0]|0;
   $25 = ($24<<24>>24)==($18<<24>>24);
   if ($25) {
    $$1 = 0;
    return ($$1|0);
   } else {
    $26 = (_sqlite3VdbeMemTranslate($12,$18)|0);
    $$0 = $26;
    break;
   }
  }
 } while(0);
 $27 = ($$0|0)==(0);
 if ($27) {
  $$1 = 0;
  return ($$1|0);
 }
 $28 = HEAP32[$0>>2]|0;
 $29 = ((($28)) + 64|0);
 HEAP32[$29>>2] = $$0;
 _sqlite3ErrorFinish($28,$$0);
 $30 = HEAP32[$0>>2]|0;
 $31 = ((($30)) + 81|0);
 $32 = HEAP8[$31>>0]|0;
 $33 = ($32<<24>>24)!=(0);
 $34 = ($$0|0)==(3082);
 $or$cond$i = $34 | $33;
 if ($or$cond$i) {
  _apiOomError($30);
  $$1 = 7;
  return ($$1|0);
 } else {
  $35 = ((($30)) + 68|0);
  $36 = HEAP32[$35>>2]|0;
  $37 = $36 & $$0;
  $$1 = $37;
  return ($$1|0);
 }
 return (0)|0;
}
function _sqlite3_bind_double($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = +$2;
 var $$0$$0$$0$$0$$i$i = 0.0, $$0$$0$$0$$0$2$i$i = 0.0, $$0$$0$$0$$0$3$i$i = 0.0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp + 8|0;
 $4 = sp;
 $5 = (_vdbeUnbind($0,$1)|0);
 $6 = ($5|0)==(0);
 if (!($6)) {
  STACKTOP = sp;return ($5|0);
 }
 $7 = ((($0)) + 100|0);
 $8 = HEAP32[$7>>2]|0;
 $9 = (($1) + -1)|0;
 $10 = (($8) + (($9*40)|0)|0);
 $11 = (((($8) + (($9*40)|0)|0)) + 8|0);
 $12 = HEAP16[$11>>1]|0;
 $13 = $12 & 9216;
 $14 = ($13<<16>>16)==(0);
 if ($14) {
  HEAP16[$11>>1] = 1;
 } else {
  _vdbeMemClearExternAndSetNull($10);
 }
 HEAPF64[$3>>3] = $2;
 $$0$$0$$0$$0$2$i$i = +HEAPF64[$3>>3];
 HEAPF64[$4>>3] = $$0$$0$$0$$0$2$i$i;
 $$0$$0$$0$$0$3$i$i = +HEAPF64[$3>>3];
 $$0$$0$$0$$0$$i$i = +HEAPF64[$4>>3];
 $15 = $$0$$0$$0$$0$3$i$i == $$0$$0$$0$$0$$i$i;
 if (!($15)) {
  STACKTOP = sp;return ($5|0);
 }
 HEAPF64[$10>>3] = $2;
 HEAP16[$11>>1] = 8;
 STACKTOP = sp;return ($5|0);
}
function _vdbeUnbind($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0 = 0, $$idx$val$i = 0, $$pre = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0;
 var $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0;
 var $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vararg_buffer10 = 0, $vararg_buffer3 = 0, $vararg_buffer7 = 0, $vararg_ptr13 = 0, $vararg_ptr14 = 0, $vararg_ptr5 = 0, $vararg_ptr6 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 64|0;
 $vararg_buffer10 = sp + 40|0;
 $vararg_buffer7 = sp + 32|0;
 $vararg_buffer3 = sp + 16|0;
 $vararg_buffer1 = sp + 8|0;
 $vararg_buffer = sp;
 $2 = ($0|0)==(0|0);
 do {
  if ($2) {
   _sqlite3_log(21,32555,$vararg_buffer);
  } else {
   $$idx$val$i = HEAP32[$0>>2]|0;
   $3 = ($$idx$val$i|0)==(0|0);
   if ($3) {
    _sqlite3_log(21,32391,$vararg_buffer1);
    break;
   }
   $4 = ((($0)) + 20|0);
   $5 = HEAP32[$4>>2]|0;
   $6 = ($5|0)==(770837923);
   if ($6) {
    $7 = ((($0)) + 36|0);
    $8 = HEAP32[$7>>2]|0;
    $9 = ($8|0)>(-1);
    if (!($9)) {
     $13 = ($1|0)<(1);
     if (!($13)) {
      $14 = ((($0)) + 16|0);
      $15 = HEAP16[$14>>1]|0;
      $16 = $15 << 16 >> 16;
      $17 = ($16|0)<($1|0);
      if (!($17)) {
       $19 = (($1) + -1)|0;
       $20 = ((($0)) + 100|0);
       $21 = HEAP32[$20>>2]|0;
       $22 = (($21) + (($19*40)|0)|0);
       $23 = (((($21) + (($19*40)|0)|0)) + 8|0);
       $24 = HEAP16[$23>>1]|0;
       $25 = $24 & 9216;
       $26 = ($25<<16>>16)==(0);
       if ($26) {
        $27 = (((($21) + (($19*40)|0)|0)) + 24|0);
        $28 = HEAP32[$27>>2]|0;
        $29 = ($28|0)==(0);
        if ($29) {
         $31 = $$idx$val$i;
        } else {
         label = 14;
        }
       } else {
        label = 14;
       }
       if ((label|0) == 14) {
        _vdbeMemClear($22);
        $$pre = HEAP32[$0>>2]|0;
        $31 = $$pre;
       }
       HEAP16[$23>>1] = 1;
       $30 = ((($31)) + 64|0);
       HEAP32[$30>>2] = 0;
       $32 = ((($0)) + 208|0);
       $33 = HEAP32[$32>>2]|0;
       $34 = ($33|0)==(0);
       if ($34) {
        $$0 = 0;
        STACKTOP = sp;return ($$0|0);
       }
       $35 = ($1|0)>(31);
       $36 = 1 << $19;
       $37 = $35 ? -2147483648 : $36;
       $38 = $33 & $37;
       $39 = ($38|0)==(0);
       if ($39) {
        $$0 = 0;
        STACKTOP = sp;return ($$0|0);
       }
       $40 = ((($0)) + 149|0);
       $41 = HEAPU8[$40>>0]|(HEAPU8[$40+1>>0]<<8);
       $42 = $41 & -4;
       $43 = $42 | 1;
       HEAP8[$40>>0]=$43&255;HEAP8[$40+1>>0]=$43>>8;
       $$0 = 0;
       STACKTOP = sp;return ($$0|0);
      }
     }
     $18 = ((($$idx$val$i)) + 64|0);
     HEAP32[$18>>2] = 25;
     _sqlite3ErrorFinish($$idx$val$i,25);
     $$0 = 25;
     STACKTOP = sp;return ($$0|0);
    }
   }
   $10 = ((($$idx$val$i)) + 64|0);
   HEAP32[$10>>2] = 21;
   _sqlite3ErrorFinish($$idx$val$i,21);
   $11 = ((($0)) + 188|0);
   $12 = HEAP32[$11>>2]|0;
   HEAP32[$vararg_buffer7>>2] = $12;
   _sqlite3_log(21,36834,$vararg_buffer7);
   HEAP32[$vararg_buffer10>>2] = 58063;
   $vararg_ptr13 = ((($vararg_buffer10)) + 4|0);
   HEAP32[$vararg_ptr13>>2] = 82575;
   $vararg_ptr14 = ((($vararg_buffer10)) + 8|0);
   HEAP32[$vararg_ptr14>>2] = (31517);
   _sqlite3_log(21,32001,$vararg_buffer10);
   $$0 = 21;
   STACKTOP = sp;return ($$0|0);
  }
 } while(0);
 HEAP32[$vararg_buffer3>>2] = 58063;
 $vararg_ptr5 = ((($vararg_buffer3)) + 4|0);
 HEAP32[$vararg_ptr5>>2] = 82567;
 $vararg_ptr6 = ((($vararg_buffer3)) + 8|0);
 HEAP32[$vararg_ptr6>>2] = (31517);
 _sqlite3_log(21,32001,$vararg_buffer3);
 $$0 = 21;
 STACKTOP = sp;return ($$0|0);
}
function _sqlite3_bind_int($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $3 = ($2|0)<(0);
 $4 = $3 << 31 >> 31;
 $5 = (_vdbeUnbind($0,$1)|0);
 $6 = ($5|0)==(0);
 if (!($6)) {
  return ($5|0);
 }
 $7 = ((($0)) + 100|0);
 $8 = HEAP32[$7>>2]|0;
 $9 = (($1) + -1)|0;
 $10 = (($8) + (($9*40)|0)|0);
 $11 = (((($8) + (($9*40)|0)|0)) + 8|0);
 $12 = HEAP16[$11>>1]|0;
 $13 = $12 & 9216;
 $14 = ($13<<16>>16)==(0);
 if ($14) {
  $15 = $10;
  $16 = $15;
  HEAP32[$16>>2] = $2;
  $17 = (($15) + 4)|0;
  $18 = $17;
  HEAP32[$18>>2] = $4;
  HEAP16[$11>>1] = 4;
  return ($5|0);
 } else {
  _vdbeReleaseAndSetInt64($10,$2,$4);
  return ($5|0);
 }
 return (0)|0;
}
function _sqlite3_bind_text($0,$1,$2,$3,$4) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 $4 = $4|0;
 var $5 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $5 = (_bindText($0,$1,$2,$3,$4,1)|0);
 return ($5|0);
}
function _sqlite3_bind_value($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$0 = 0, $$0$$0$$0$$0$$0$$i$i$i = 0.0, $$0$$0$$0$$0$$0$2$i$i$i = 0.0, $$0$$0$$0$$0$$0$3$i$i$i = 0.0, $$062$i = 0, $$063$i = 0, $$06470$i = 0, $$167$i$ph = 0, $$2$i = 0, $$phi$trans$insert$i = 0, $$pre$i = 0, $$pre73$pre$phi$iZZ2D = 0, $$pre75$i = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0;
 var $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0;
 var $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0;
 var $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0.0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0;
 var $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0;
 var $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0;
 var $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0;
 var $96 = 0, $97 = 0, $98 = 0, $99 = 0, $spec$select$i = 0, $spec$select$i$i = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp + 8|0;
 $4 = sp;
 $5 = ((($2)) + 8|0);
 $6 = HEAP16[$5>>1]|0;
 $7 = $6 & 31;
 $8 = $7&65535;
 $9 = (816 + ($8)|0);
 $10 = HEAP8[$9>>0]|0;
 switch ($10<<24>>24) {
 case 1:  {
  $11 = $2;
  $12 = $11;
  $13 = HEAP32[$12>>2]|0;
  $14 = (($11) + 4)|0;
  $15 = $14;
  $16 = HEAP32[$15>>2]|0;
  $17 = (_vdbeUnbind($0,$1)|0);
  $18 = ($17|0)==(0);
  if (!($18)) {
   $$0 = $17;
   STACKTOP = sp;return ($$0|0);
  }
  $19 = ((($0)) + 100|0);
  $20 = HEAP32[$19>>2]|0;
  $21 = (($1) + -1)|0;
  $22 = (($20) + (($21*40)|0)|0);
  $23 = (((($20) + (($21*40)|0)|0)) + 8|0);
  $24 = HEAP16[$23>>1]|0;
  $25 = $24 & 9216;
  $26 = ($25<<16>>16)==(0);
  if ($26) {
   $27 = $22;
   $28 = $27;
   HEAP32[$28>>2] = $13;
   $29 = (($27) + 4)|0;
   $30 = $29;
   HEAP32[$30>>2] = $16;
   HEAP16[$23>>1] = 4;
   $$0 = 0;
   STACKTOP = sp;return ($$0|0);
  } else {
   _vdbeReleaseAndSetInt64($22,$13,$16);
   $$0 = 0;
   STACKTOP = sp;return ($$0|0);
  }
  break;
 }
 case 2:  {
  $31 = +HEAPF64[$2>>3];
  $32 = (_vdbeUnbind($0,$1)|0);
  $33 = ($32|0)==(0);
  if (!($33)) {
   $$0 = $32;
   STACKTOP = sp;return ($$0|0);
  }
  $34 = ((($0)) + 100|0);
  $35 = HEAP32[$34>>2]|0;
  $36 = (($1) + -1)|0;
  $37 = (($35) + (($36*40)|0)|0);
  $38 = (((($35) + (($36*40)|0)|0)) + 8|0);
  $39 = HEAP16[$38>>1]|0;
  $40 = $39 & 9216;
  $41 = ($40<<16>>16)==(0);
  if ($41) {
   HEAP16[$38>>1] = 1;
  } else {
   _vdbeMemClearExternAndSetNull($37);
  }
  HEAPF64[$3>>3] = $31;
  $$0$$0$$0$$0$$0$2$i$i$i = +HEAPF64[$3>>3];
  HEAPF64[$4>>3] = $$0$$0$$0$$0$$0$2$i$i$i;
  $$0$$0$$0$$0$$0$3$i$i$i = +HEAPF64[$3>>3];
  $$0$$0$$0$$0$$0$$i$i$i = +HEAPF64[$4>>3];
  $42 = $$0$$0$$0$$0$$0$3$i$i$i == $$0$$0$$0$$0$$0$$i$i$i;
  if (!($42)) {
   $$0 = 0;
   STACKTOP = sp;return ($$0|0);
  }
  HEAPF64[$37>>3] = $31;
  HEAP16[$38>>1] = 8;
  $$0 = 0;
  STACKTOP = sp;return ($$0|0);
  break;
 }
 case 4:  {
  $43 = $6 & 16384;
  $44 = ($43<<16>>16)==(0);
  if (!($44)) {
   $45 = HEAP32[$2>>2]|0;
   $46 = (_vdbeUnbind($0,$1)|0);
   $47 = ($46|0)==(0);
   if (!($47)) {
    $$0 = $46;
    STACKTOP = sp;return ($$0|0);
   }
   $48 = ((($0)) + 100|0);
   $49 = HEAP32[$48>>2]|0;
   $50 = (($1) + -1)|0;
   $51 = (($49) + (($50*40)|0)|0);
   $52 = (((($49) + (($50*40)|0)|0)) + 8|0);
   $53 = HEAP16[$52>>1]|0;
   $54 = $53 & 9216;
   $55 = ($54<<16>>16)==(0);
   if ($55) {
    $56 = (((($49) + (($50*40)|0)|0)) + 24|0);
    $57 = HEAP32[$56>>2]|0;
    $58 = ($57|0)==(0);
    if (!($58)) {
     label = 16;
    }
   } else {
    label = 16;
   }
   if ((label|0) == 16) {
    _vdbeMemClear($51);
   }
   HEAP16[$52>>1] = 16400;
   $59 = (((($49) + (($50*40)|0)|0)) + 12|0);
   HEAP32[$59>>2] = 0;
   $60 = ($45|0)>(0);
   $spec$select$i$i = $60 ? $45 : 0;
   HEAP32[$51>>2] = $spec$select$i$i;
   $61 = (((($49) + (($50*40)|0)|0)) + 10|0);
   HEAP8[$61>>0] = 1;
   $62 = (((($49) + (($50*40)|0)|0)) + 16|0);
   HEAP32[$62>>2] = 0;
   $$0 = 0;
   STACKTOP = sp;return ($$0|0);
  }
  $63 = ((($2)) + 16|0);
  $64 = HEAP32[$63>>2]|0;
  $65 = ((($2)) + 12|0);
  $66 = HEAP32[$65>>2]|0;
  $67 = (_vdbeUnbind($0,$1)|0);
  $68 = ($67|0)==(0);
  if (!($68)) {
   $$0 = $67;
   STACKTOP = sp;return ($$0|0);
  }
  $69 = ($64|0)==(0|0);
  if ($69) {
   $$0 = 0;
   STACKTOP = sp;return ($$0|0);
  }
  $70 = ((($0)) + 100|0);
  $71 = HEAP32[$70>>2]|0;
  $72 = (($1) + -1)|0;
  $73 = (($71) + (($72*40)|0)|0);
  $74 = (((($71) + (($72*40)|0)|0)) + 32|0);
  $75 = HEAP32[$74>>2]|0;
  $76 = ($75|0)==(0|0);
  if ($76) {
   $$063$i = 1000000000;
  } else {
   $77 = ((($75)) + 108|0);
   $78 = HEAP32[$77>>2]|0;
   $$063$i = $78;
  }
  $79 = ($66|0)<(0);
  L47: do {
   if ($79) {
    $80 = ($$063$i|0)<(0);
    if ($80) {
     $$062$i = 528;$$2$i = 0;
    } else {
     $$06470$i = 0;
     while(1) {
      $81 = (($64) + ($$06470$i)|0);
      $82 = HEAP8[$81>>0]|0;
      $83 = $$06470$i | 1;
      $84 = (($64) + ($83)|0);
      $85 = HEAP8[$84>>0]|0;
      $86 = $85 | $82;
      $87 = ($86<<24>>24)==(0);
      if ($87) {
       $$062$i = 528;$$2$i = $$06470$i;
       break L47;
      }
      $88 = (($$06470$i) + 2)|0;
      $89 = ($88|0)>($$063$i|0);
      if ($89) {
       $$062$i = 528;$$2$i = $88;
       break;
      } else {
       $$06470$i = $88;
      }
     }
    }
   } else {
    $$062$i = 16;$$2$i = $66;
   }
  } while(0);
  $90 = ($$062$i&65535) >>> 8;
  $91 = $90&65535;
  $spec$select$i = (($$2$i) + ($91))|0;
  $92 = ($$2$i|0)>($$063$i|0);
  do {
   if ($92) {
    if ($76) {
     $$167$i$ph = 18;
    } else {
     $93 = ((($75)) + 236|0);
     $94 = HEAP32[$93>>2]|0;
     $95 = ($94|0)==(0|0);
     if ($95) {
      $$167$i$ph = 18;
     } else {
      $96 = ((($94)) + 12|0);
      HEAP32[$96>>2] = 18;
      $97 = ((($94)) + 36|0);
      $98 = HEAP32[$97>>2]|0;
      $99 = (($98) + 1)|0;
      HEAP32[$97>>2] = $99;
      $$167$i$ph = 18;
     }
    }
   } else {
    $100 = ($spec$select$i>>>0)>(32);
    $101 = $100 ? $spec$select$i : 32;
    $102 = (((($71) + (($72*40)|0)|0)) + 24|0);
    $103 = HEAP32[$102>>2]|0;
    $104 = ($103|0)<($101|0);
    if ($104) {
     $112 = (_sqlite3VdbeMemGrow($73,$101,0)|0);
     $113 = ($112|0)==(0);
     if (!($113)) {
      $$167$i$ph = 7;
      break;
     }
     $$phi$trans$insert$i = (((($71) + (($72*40)|0)|0)) + 16|0);
     $$pre$i = HEAP32[$$phi$trans$insert$i>>2]|0;
     $$pre75$i = (((($71) + (($72*40)|0)|0)) + 8|0);
     $$pre73$pre$phi$iZZ2D = $$pre75$i;$114 = $$pre$i;
    } else {
     $105 = (((($71) + (($72*40)|0)|0)) + 20|0);
     $106 = HEAP32[$105>>2]|0;
     $107 = (((($71) + (($72*40)|0)|0)) + 16|0);
     HEAP32[$107>>2] = $106;
     $108 = (((($71) + (($72*40)|0)|0)) + 8|0);
     $109 = HEAP16[$108>>1]|0;
     $110 = $109 & 13;
     HEAP16[$108>>1] = $110;
     $111 = $106;
     $$pre73$pre$phi$iZZ2D = $108;$114 = $111;
    }
    _memcpy(($114|0),($64|0),($spec$select$i|0))|0;
    $115 = (((($71) + (($72*40)|0)|0)) + 12|0);
    HEAP32[$115>>2] = $$2$i;
    HEAP16[$$pre73$pre$phi$iZZ2D>>1] = $$062$i;
    $116 = (((($71) + (($72*40)|0)|0)) + 10|0);
    HEAP8[$116>>0] = 1;
    $$0 = 0;
    STACKTOP = sp;return ($$0|0);
   }
  } while(0);
  $117 = HEAP32[$0>>2]|0;
  $118 = ((($117)) + 64|0);
  HEAP32[$118>>2] = $$167$i$ph;
  _sqlite3ErrorFinish($117,$$167$i$ph);
  $119 = HEAP32[$0>>2]|0;
  $120 = ((($119)) + 81|0);
  $121 = HEAP8[$120>>0]|0;
  $122 = ($121<<24>>24)==(0);
  if ($122) {
   $123 = ((($119)) + 68|0);
   $124 = HEAP32[$123>>2]|0;
   $125 = $124 & $$167$i$ph;
   $$0 = $125;
   STACKTOP = sp;return ($$0|0);
  } else {
   _apiOomError($119);
   $$0 = 7;
   STACKTOP = sp;return ($$0|0);
  }
  break;
 }
 case 3:  {
  $126 = ((($2)) + 16|0);
  $127 = HEAP32[$126>>2]|0;
  $128 = ((($2)) + 12|0);
  $129 = HEAP32[$128>>2]|0;
  $130 = ((($2)) + 10|0);
  $131 = HEAP8[$130>>0]|0;
  $132 = (_bindText($0,$1,$127,$129,(-1),$131)|0);
  $$0 = $132;
  STACKTOP = sp;return ($$0|0);
  break;
 }
 default: {
  $133 = (_vdbeUnbind($0,$1)|0);
  $$0 = $133;
  STACKTOP = sp;return ($$0|0);
 }
 }
 return (0)|0;
}
function _sqlite3_bind_parameter_index($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i3 = 0, $$023$i$i = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $3 = 0, $4 = 0;
 var $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 $2 = ($1|0)==(0|0);
 if ($2) {
  $$0$i3 = 0;
  return ($$0$i3|0);
 }
 $3 = (_strlen($1)|0);
 $4 = $3 & 1073741823;
 $5 = ($0|0)==(0|0);
 if ($5) {
  $$0$i3 = 0;
  return ($$0$i3|0);
 }
 $6 = ((($0)) + 128|0);
 $7 = HEAP32[$6>>2]|0;
 $8 = ($7|0)==(0|0);
 if ($8) {
  $$0$i3 = 0;
  return ($$0$i3|0);
 }
 $9 = ((($7)) + 4|0);
 $10 = HEAP32[$9>>2]|0;
 $$023$i$i = 2;
 while(1) {
  $11 = (($$023$i$i) + 2)|0;
  $12 = (($7) + ($11<<2)|0);
  $13 = (_strncmp($12,$1,$4)|0);
  $14 = ($13|0)==(0);
  if ($14) {
   $15 = (($12) + ($4)|0);
   $16 = HEAP8[$15>>0]|0;
   $17 = ($16<<24>>24)==(0);
   if ($17) {
    break;
   }
  }
  $20 = (($$023$i$i) + 1)|0;
  $21 = (($7) + ($20<<2)|0);
  $22 = HEAP32[$21>>2]|0;
  $23 = (($22) + ($$023$i$i))|0;
  $24 = ($23|0)<($10|0);
  if ($24) {
   $$023$i$i = $23;
  } else {
   $$0$i3 = 0;
   label = 9;
   break;
  }
 }
 if ((label|0) == 9) {
  return ($$0$i3|0);
 }
 $18 = (($7) + ($$023$i$i<<2)|0);
 $19 = HEAP32[$18>>2]|0;
 $$0$i3 = $19;
 return ($$0$i3|0);
}
function _sqlite3VdbeExpandSql($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $$0$i = 0, $$0$i70 = 0, $$01113$i = 0, $$01114$i = 0, $$015$i = 0, $$023$i$i = 0, $$06287 = 0, $$06385 = 0, $$077 = 0, $$083 = 0, $$288 = 0, $$lcssa = 0, $$pre = 0, $$pre94 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0;
 var $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0;
 var $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0.0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0;
 var $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0;
 var $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0;
 var $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0;
 var $196 = 0, $197 = 0, $198 = 0, $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0;
 var $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0;
 var $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0;
 var $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0;
 var $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0;
 var $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0;
 var $97 = 0, $98 = 0, $99 = 0, $exitcond = 0, $or$cond = 0, $or$cond91 = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vararg_buffer12 = 0, $vararg_buffer15 = 0, $vararg_buffer4 = 0, $vararg_buffer8 = 0, $vararg_ptr11 = 0, $vararg_ptr7 = 0, dest = 0, label = 0, sp = 0, stop = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 224|0;
 $vararg_buffer15 = sp + 184|0;
 $vararg_buffer12 = sp + 176|0;
 $vararg_buffer8 = sp + 168|0;
 $vararg_buffer4 = sp + 160|0;
 $vararg_buffer1 = sp + 152|0;
 $vararg_buffer = sp + 144|0;
 $2 = sp + 220|0;
 $3 = sp + 216|0;
 $4 = sp + 192|0;
 $5 = sp + 104|0;
 $6 = sp;
 HEAP32[$3>>2] = 0;
 $7 = HEAP32[$0>>2]|0;
 $8 = ((($7)) + 108|0);
 $9 = HEAP32[$8>>2]|0;
 $10 = ((($4)) + 4|0);
 HEAP32[$10>>2] = $6;
 HEAP32[$4>>2] = 0;
 $11 = ((($4)) + 8|0);
 HEAP32[$11>>2] = 100;
 $12 = ((($4)) + 12|0);
 HEAP32[$12>>2] = $9;
 $13 = ((($4)) + 16|0);
 HEAP32[$13>>2] = 0;
 $14 = ((($4)) + 20|0);
 HEAP8[$14>>0] = 0;
 $15 = ((($4)) + 21|0);
 HEAP8[$15>>0] = 0;
 $16 = ((($7)) + 180|0);
 $17 = HEAP32[$16>>2]|0;
 $18 = ($17|0)>(1);
 L1: do {
  if ($18) {
   $19 = HEAP8[$1>>0]|0;
   $20 = ($19<<24>>24)==(0);
   if (!($20)) {
    $$083 = $1;$23 = $19;
    while(1) {
     $21 = ((($$083)) + 1|0);
     $22 = ($23<<24>>24)==(10);
     L6: do {
      if ($22) {
       $$lcssa = $21;
      } else {
       $25 = $21;
       while(1) {
        $27 = HEAP8[$25>>0]|0;
        $28 = ($27<<24>>24)==(0);
        if ($28) {
         $$lcssa = $25;
         break L6;
        }
        $24 = ((($25)) + 1|0);
        $26 = ($27<<24>>24)==(10);
        if ($26) {
         $$lcssa = $24;
         break;
        } else {
         $25 = $24;
        }
       }
      }
     } while(0);
     $29 = HEAP32[$13>>2]|0;
     $30 = (($29) + 3)|0;
     $31 = HEAP32[$11>>2]|0;
     $32 = ($30>>>0)<($31>>>0);
     if ($32) {
      HEAP32[$13>>2] = $30;
      $33 = HEAP32[$10>>2]|0;
      $34 = (($33) + ($29)|0);
      ;HEAP8[$34>>0]=HEAP8[36874>>0]|0;HEAP8[$34+1>>0]=HEAP8[36874+1>>0]|0;HEAP8[$34+2>>0]=HEAP8[36874+2>>0]|0;
     } else {
      _enlargeAndAppend($4,36874,3);
     }
     $35 = $$lcssa;
     $36 = $$083;
     $37 = (($35) - ($36))|0;
     $38 = HEAP32[$13>>2]|0;
     $39 = (($38) + ($37))|0;
     $40 = HEAP32[$11>>2]|0;
     $41 = ($39>>>0)<($40>>>0);
     if ($41) {
      $42 = ($37|0)==(0);
      if (!($42)) {
       HEAP32[$13>>2] = $39;
       $43 = HEAP32[$10>>2]|0;
       $44 = (($43) + ($38)|0);
       _memcpy(($44|0),($$083|0),($37|0))|0;
      }
     } else {
      _enlargeAndAppend($4,$$083,$37);
     }
     $45 = HEAP8[$$lcssa>>0]|0;
     $46 = ($45<<24>>24)==(0);
     if ($46) {
      break;
     } else {
      $$083 = $$lcssa;$23 = $45;
     }
    }
   }
  } else {
   $47 = ((($0)) + 16|0);
   $48 = HEAP16[$47>>1]|0;
   $49 = ($48<<16>>16)==(0);
   if ($49) {
    $50 = ($1|0)==(0|0);
    if ($50) {
     break;
    }
    $51 = (_strlen($1)|0);
    $52 = $51 & 1073741823;
    $53 = ($52>>>0)<(100);
    if (!($53)) {
     _enlargeAndAppend($4,$1,$52);
     break;
    }
    $54 = ($52|0)==(0);
    if ($54) {
     break;
    }
    HEAP32[$13>>2] = $52;
    _memcpy(($6|0),($1|0),($52|0))|0;
    break;
   }
   $55 = HEAP8[$1>>0]|0;
   $56 = ($55<<24>>24)==(0);
   if (!($56)) {
    $57 = ((($0)) + 100|0);
    $58 = ((($0)) + 128|0);
    $59 = ((($7)) + 78|0);
    $60 = ((($5)) + 32|0);
    $61 = ((($5)) + 8|0);
    $62 = ((($5)) + 12|0);
    $63 = ((($5)) + 16|0);
    $64 = ((($5)) + 10|0);
    $65 = ((($5)) + 24|0);
    $$06287 = 1;$$288 = $1;
    while(1) {
     $$01114$i = 0;$$015$i = $$288;
     while(1) {
      $66 = (_sqlite3GetToken($$015$i,$2)|0);
      $67 = HEAP32[$2>>2]|0;
      $68 = ($67|0)==(148);
      if ($68) {
       $$01113$i = $$01114$i;$$077 = $66;
       break;
      }
      $69 = (($66) + ($$01114$i))|0;
      $70 = (($$015$i) + ($66)|0);
      $71 = HEAP8[$70>>0]|0;
      $72 = ($71<<24>>24)==(0);
      if ($72) {
       $$01113$i = $69;$$077 = 0;
       break;
      } else {
       $$01114$i = $69;$$015$i = $70;
      }
     }
     $73 = HEAP32[$13>>2]|0;
     $74 = (($73) + ($$01113$i))|0;
     $75 = HEAP32[$11>>2]|0;
     $76 = ($74>>>0)<($75>>>0);
     if ($76) {
      $77 = ($$01113$i|0)==(0);
      if (!($77)) {
       HEAP32[$13>>2] = $74;
       $78 = HEAP32[$10>>2]|0;
       $79 = (($78) + ($73)|0);
       _memcpy(($79|0),($$288|0),($$01113$i|0))|0;
      }
     } else {
      _enlargeAndAppend($4,$$288,$$01113$i);
     }
     $80 = (($$288) + ($$01113$i)|0);
     $81 = ($$077|0)==(0);
     if ($81) {
      break L1;
     }
     $82 = HEAP8[$80>>0]|0;
     $83 = ($82<<24>>24)==(63);
     do {
      if ($83) {
       $84 = ($$077|0)>(1);
       if ($84) {
        $85 = ((($80)) + 1|0);
        (_sqlite3GetInt32($85,$3)|0);
        $$pre = HEAP32[$3>>2]|0;
        $106 = $$pre;
        break;
       } else {
        HEAP32[$3>>2] = $$06287;
        $106 = $$06287;
        break;
       }
      } else {
       $86 = HEAP32[$58>>2]|0;
       $87 = ($86|0)==(0|0);
       L49: do {
        if ($87) {
         $$0$i70 = 0;
        } else {
         $88 = ((($86)) + 4|0);
         $89 = HEAP32[$88>>2]|0;
         $$023$i$i = 2;
         while(1) {
          $90 = (($$023$i$i) + 2)|0;
          $91 = (($86) + ($90<<2)|0);
          $92 = (_strncmp($91,$80,$$077)|0);
          $93 = ($92|0)==(0);
          if ($93) {
           $94 = (($91) + ($$077)|0);
           $95 = HEAP8[$94>>0]|0;
           $96 = ($95<<24>>24)==(0);
           if ($96) {
            break;
           }
          }
          $99 = (($$023$i$i) + 1)|0;
          $100 = (($86) + ($99<<2)|0);
          $101 = HEAP32[$100>>2]|0;
          $102 = (($101) + ($$023$i$i))|0;
          $103 = ($102|0)<($89|0);
          if ($103) {
           $$023$i$i = $102;
          } else {
           $$0$i70 = 0;
           break L49;
          }
         }
         $97 = (($86) + ($$023$i$i<<2)|0);
         $98 = HEAP32[$97>>2]|0;
         $$0$i70 = $98;
        }
       } while(0);
       HEAP32[$3>>2] = $$0$i70;
       $106 = $$0$i70;
      }
     } while(0);
     $104 = (($80) + ($$077)|0);
     $105 = (($106) + 1)|0;
     $107 = HEAP32[$57>>2]|0;
     $108 = (($106) + -1)|0;
     $109 = (($107) + (($108*40)|0)|0);
     $110 = (((($107) + (($108*40)|0)|0)) + 8|0);
     $111 = HEAP16[$110>>1]|0;
     $112 = $111&65535;
     $113 = $112 & 1;
     $114 = ($113|0)==(0);
     do {
      if ($114) {
       $121 = $112 & 4;
       $122 = ($121|0)==(0);
       if (!($122)) {
        $123 = $109;
        $124 = $123;
        $125 = HEAP32[$124>>2]|0;
        $126 = (($123) + 4)|0;
        $127 = $126;
        $128 = HEAP32[$127>>2]|0;
        $129 = $vararg_buffer;
        $130 = $129;
        HEAP32[$130>>2] = $125;
        $131 = (($129) + 4)|0;
        $132 = $131;
        HEAP32[$132>>2] = $128;
        _sqlite3_str_appendf($4,32524,$vararg_buffer);
        break;
       }
       $133 = $112 & 8;
       $134 = ($133|0)==(0);
       if (!($134)) {
        $135 = +HEAPF64[$109>>3];
        HEAPF64[$vararg_buffer1>>3] = $135;
        _sqlite3_str_appendf($4,32529,$vararg_buffer1);
        break;
       }
       $136 = $112 & 2;
       $137 = ($136|0)==(0);
       if (!($137)) {
        $138 = HEAP8[$59>>0]|0;
        $139 = ($138<<24>>24)==(1);
        if ($139) {
         $158 = (((($107) + (($108*40)|0)|0)) + 12|0);
         $159 = HEAP32[$158>>2]|0;
         $160 = (((($107) + (($108*40)|0)|0)) + 16|0);
         $161 = HEAP32[$160>>2]|0;
         HEAP32[$vararg_buffer8>>2] = $159;
         $vararg_ptr11 = ((($vararg_buffer8)) + 4|0);
         HEAP32[$vararg_ptr11>>2] = $161;
         _sqlite3_str_appendf($4,36878,$vararg_buffer8);
         break;
        }
        dest=$5; stop=dest+40|0; do { HEAP32[dest>>2]=0|0; dest=dest+4|0; } while ((dest|0) < (stop|0));
        HEAP32[$60>>2] = $7;
        $140 = (((($107) + (($108*40)|0)|0)) + 16|0);
        $141 = HEAP32[$140>>2]|0;
        $142 = (((($107) + (($108*40)|0)|0)) + 12|0);
        $143 = HEAP32[$142>>2]|0;
        (_sqlite3VdbeMemSetStr($5,$141,$143,$138,0)|0);
        $144 = HEAP16[$61>>1]|0;
        $145 = $144 & 2;
        $146 = ($145<<16>>16)==(0);
        $147 = HEAP8[$64>>0]|0;
        $148 = ($147<<24>>24)==(1);
        $or$cond = $146 | $148;
        if (!($or$cond)) {
         $149 = (_sqlite3VdbeMemTranslate($5,1)|0);
         $150 = ($149|0)==(7);
         if ($150) {
          HEAP8[$14>>0] = 7;
          HEAP32[$11>>2] = 0;
         }
        }
        $151 = HEAP32[$62>>2]|0;
        $152 = HEAP32[$63>>2]|0;
        HEAP32[$vararg_buffer4>>2] = $151;
        $vararg_ptr7 = ((($vararg_buffer4)) + 4|0);
        HEAP32[$vararg_ptr7>>2] = $152;
        _sqlite3_str_appendf($4,36878,$vararg_buffer4);
        $153 = HEAP16[$61>>1]|0;
        $154 = $153 & 9216;
        $155 = ($154<<16>>16)==(0);
        $156 = HEAP32[$65>>2]|0;
        $157 = ($156|0)==(0);
        $or$cond91 = $155 & $157;
        if ($or$cond91) {
         break;
        }
        _vdbeMemClear($5);
        break;
       }
       $162 = $112 & 16384;
       $163 = ($162|0)==(0);
       if (!($163)) {
        $164 = HEAP32[$109>>2]|0;
        HEAP32[$vararg_buffer12>>2] = $164;
        _sqlite3_str_appendf($4,36885,$vararg_buffer12);
        break;
       }
       $165 = HEAP32[$13>>2]|0;
       $166 = (($165) + 2)|0;
       $167 = HEAP32[$11>>2]|0;
       $168 = ($166>>>0)<($167>>>0);
       if ($168) {
        HEAP32[$13>>2] = $166;
        $169 = HEAP32[$10>>2]|0;
        $170 = (($169) + ($165)|0);
        HEAP8[$170>>0]=10104&255;HEAP8[$170+1>>0]=10104>>8;
       } else {
        _enlargeAndAppend($4,36898,2);
       }
       $171 = (((($107) + (($108*40)|0)|0)) + 12|0);
       $172 = HEAP32[$171>>2]|0;
       $173 = ($172|0)>(0);
       if ($173) {
        $174 = (((($107) + (($108*40)|0)|0)) + 16|0);
        $$06385 = 0;
        while(1) {
         $175 = HEAP32[$174>>2]|0;
         $176 = (($175) + ($$06385)|0);
         $177 = HEAP8[$176>>0]|0;
         $178 = $177&255;
         HEAP32[$vararg_buffer15>>2] = $178;
         _sqlite3_str_appendf($4,36901,$vararg_buffer15);
         $179 = (($$06385) + 1)|0;
         $exitcond = ($179|0)==($172|0);
         if ($exitcond) {
          break;
         } else {
          $$06385 = $179;
         }
        }
       }
       $180 = HEAP32[$13>>2]|0;
       $181 = (($180) + 1)|0;
       $182 = HEAP32[$11>>2]|0;
       $183 = ($181>>>0)<($182>>>0);
       if ($183) {
        HEAP32[$13>>2] = $181;
        $184 = HEAP32[$10>>2]|0;
        $185 = (($184) + ($180)|0);
        HEAP8[$185>>0] = 39;
        break;
       } else {
        _enlargeAndAppend($4,36906,1);
        break;
       }
      } else {
       $115 = HEAP32[$13>>2]|0;
       $116 = (($115) + 4)|0;
       $117 = HEAP32[$11>>2]|0;
       $118 = ($116>>>0)<($117>>>0);
       if ($118) {
        HEAP32[$13>>2] = $116;
        $119 = HEAP32[$10>>2]|0;
        $120 = (($119) + ($115)|0);
        HEAP8[$120>>0]=1280070990&255;HEAP8[$120+1>>0]=(1280070990>>8)&255;HEAP8[$120+2>>0]=(1280070990>>16)&255;HEAP8[$120+3>>0]=1280070990>>24;
        break;
       } else {
        _enlargeAndAppend($4,31171,4);
        break;
       }
      }
     } while(0);
     $186 = HEAP8[$104>>0]|0;
     $187 = ($186<<24>>24)==(0);
     if ($187) {
      break;
     } else {
      $$06287 = $105;$$288 = $104;
     }
    }
   }
  }
 } while(0);
 $188 = HEAP8[$14>>0]|0;
 $189 = ($188<<24>>24)==(0);
 if ($189) {
  $$pre94 = HEAP32[$10>>2]|0;
  $221 = ($$pre94|0)==(0|0);
  if (!($221)) {
   $222 = HEAP32[$13>>2]|0;
   $223 = (($$pre94) + ($222)|0);
   HEAP8[$223>>0] = 0;
   $224 = HEAP32[$12>>2]|0;
   $225 = ($224|0)==(0);
   if (!($225)) {
    $226 = HEAP8[$15>>0]|0;
    $227 = $226 & 4;
    $228 = ($227<<24>>24)==(0);
    if ($228) {
     $229 = (_strAccumFinishRealloc($4)|0);
     $$0$i = $229;
     STACKTOP = sp;return ($$0$i|0);
    }
   }
  }
 } else {
  $190 = HEAP8[$15>>0]|0;
  $191 = $190 & 4;
  $192 = ($191<<24>>24)==(0);
  if (!($192)) {
   $193 = HEAP32[$4>>2]|0;
   $194 = HEAP32[$10>>2]|0;
   $195 = ($194|0)==(0|0);
   do {
    if (!($195)) {
     $196 = ($193|0)==(0|0);
     if (!($196)) {
      $197 = ((($193)) + 480|0);
      $198 = HEAP32[$197>>2]|0;
      $199 = ($198|0)==(0|0);
      if (!($199)) {
       _measureAllocationSize($193,$194);
       break;
      }
      $200 = $194;
      $201 = ((($193)) + 304|0);
      $202 = HEAP32[$201>>2]|0;
      $203 = ($202>>>0)>($200>>>0);
      if (!($203)) {
       $204 = ((($193)) + 308|0);
       $205 = HEAP32[$204>>2]|0;
       $206 = ($205>>>0)>($200>>>0);
       if ($206) {
        $207 = ((($193)) + 300|0);
        $208 = HEAP32[$207>>2]|0;
        HEAP32[$194>>2] = $208;
        HEAP32[$207>>2] = $194;
        break;
       }
      }
     }
     $209 = HEAP32[7324]|0;
     $210 = ($209|0)==(0);
     if ($210) {
      $218 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$218 & 127]($194);
      break;
     } else {
      $211 = HEAP32[(29352)>>2]|0;
      $212 = (FUNCTION_TABLE_ii[$211 & 127]($194)|0);
      $213 = HEAP32[14978]|0;
      $214 = (($213) - ($212))|0;
      HEAP32[14978] = $214;
      $215 = HEAP32[14981]|0;
      $216 = (($215) + -1)|0;
      HEAP32[14981] = $216;
      $217 = HEAP32[(29344)>>2]|0;
      FUNCTION_TABLE_vi[$217 & 127]($194);
      break;
     }
    }
   } while(0);
   $219 = HEAP8[$15>>0]|0;
   $220 = $219 & -5;
   HEAP8[$15>>0] = $220;
  }
  HEAP32[$11>>2] = 0;
  HEAP32[$13>>2] = 0;
  HEAP32[$10>>2] = 0;
 }
 $230 = HEAP32[$10>>2]|0;
 $$0$i = $230;
 STACKTOP = sp;return ($$0$i|0);
}
function _sqlite3_value_numeric_type($0) {
 $0 = $0|0;
 var $$0 = 0, $$masked = 0, $$pre = 0, $$sink8 = 0, $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0;
 var $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0.0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0;
 var $42 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp + 8|0;
 $2 = sp;
 $3 = ((($0)) + 8|0);
 $4 = HEAP16[$3>>1]|0;
 $5 = $4 & 31;
 $6 = $5&65535;
 $7 = (816 + ($6)|0);
 $8 = HEAP8[$7>>0]|0;
 $9 = $8&255;
 $10 = $6 | 16;
 $11 = ($10|0)==(18);
 if (!($11)) {
  $$0 = $9;
  STACKTOP = sp;return ($$0|0);
 }
 $12 = ((($0)) + 10|0);
 $13 = HEAP8[$12>>0]|0;
 $14 = ((($0)) + 16|0);
 $15 = HEAP32[$14>>2]|0;
 $16 = ((($0)) + 12|0);
 $17 = HEAP32[$16>>2]|0;
 $18 = (_sqlite3AtoF($15,$1,$17,$13)|0);
 $19 = ($18|0)==(0);
 if ($19) {
  $$pre = HEAP16[$3>>1]|0;
  $38 = $$pre;
 } else {
  $20 = HEAP32[$14>>2]|0;
  $21 = HEAP32[$16>>2]|0;
  $22 = (_sqlite3Atoi64($20,$2,$21,$13)|0);
  $23 = ($22|0)==(0);
  if ($23) {
   $24 = $2;
   $25 = $24;
   $26 = HEAP32[$25>>2]|0;
   $27 = (($24) + 4)|0;
   $28 = $27;
   $29 = HEAP32[$28>>2]|0;
   $30 = $0;
   $31 = $30;
   HEAP32[$31>>2] = $26;
   $32 = (($30) + 4)|0;
   $33 = $32;
   HEAP32[$33>>2] = $29;
   $$sink8 = 4;
  } else {
   $34 = +HEAPF64[$1>>3];
   HEAPF64[$0>>3] = $34;
   $$sink8 = 8;
  }
  $35 = HEAP16[$3>>1]|0;
  $$masked = $35 & -3;
  $36 = $$masked | $$sink8;
  HEAP16[$3>>1] = $36;
  $38 = $36;
 }
 $37 = $38 & 31;
 $39 = $37&65535;
 $40 = (816 + ($39)|0);
 $41 = HEAP8[$40>>0]|0;
 $42 = $41&255;
 $$0 = $42;
 STACKTOP = sp;return ($$0|0);
}
function _sqlite3_blob_open($0,$1,$2,$3,$4,$5,$6,$7) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 $4 = $4|0;
 $5 = $5|0;
 $6 = $6|0;
 $7 = $7|0;
 var $$ = 0, $$0$i = 0, $$0$i$i = 0, $$0$i$i179 = 0, $$0$i$i194 = 0, $$0$i175 = 0, $$0$i178 = 0, $$0$i189 = 0, $$0$i6$i = 0, $$010$i = 0, $$01315$i = 0, $$0149 = 0, $$0151239 = 0, $$0152226 = 0, $$0153 = 0, $$0153231 = 0, $$0153233 = 0, $$0154 = 0, $$0154244 = 0, $$0154246 = 0;
 var $$0155232 = 0, $$0159 = 0, $$016$i = 0, $$0163$lcssa = 0, $$0163218 = 0, $$1$i = 0, $$1$i176 = 0, $$1$lcssa = 0, $$1225 = 0, $$3 = 0, $$3162 = 0, $$3162206 = 0, $$3162207 = 0, $$4$lcssa = 0, $$4245 = 0, $$5$lcssa = 0, $$5238 = 0, $$6 = 0, $$idx$i = 0, $$idx$val$i = 0;
 var $$lcssa$i = 0, $$pr$pre = 0, $$pre$i = 0, $$pre$i$i = 0, $$sink = 0, $$sink266 = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0;
 var $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0;
 var $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0;
 var $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0;
 var $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0;
 var $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0;
 var $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0;
 var $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0;
 var $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0;
 var $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0;
 var $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0;
 var $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0;
 var $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0;
 var $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0;
 var $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0;
 var $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0;
 var $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0;
 var $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0;
 var $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0;
 var $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0;
 var $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0;
 var $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $50 = 0, $51 = 0;
 var $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $70 = 0, $71 = 0;
 var $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0;
 var $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, $exitcond = 0, $or$cond = 0, $or$cond$i = 0, $or$cond$i188 = 0, $or$cond14$i = 0, $spec$select = 0, $switch$split2D = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vararg_buffer10 = 0;
 var $vararg_buffer13 = 0, $vararg_buffer4 = 0, $vararg_buffer7 = 0, dest = 0, label = 0, sp = 0, stop = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 336|0;
 $vararg_buffer13 = sp + 40|0;
 $vararg_buffer10 = sp + 32|0;
 $vararg_buffer7 = sp + 24|0;
 $vararg_buffer4 = sp + 16|0;
 $vararg_buffer1 = sp + 8|0;
 $vararg_buffer = sp;
 $8 = sp + 320|0;
 $9 = sp + 44|0;
 HEAP32[$8>>2] = 0;
 HEAP32[$7>>2] = 0;
 $10 = ($6|0)!=(0);
 $11 = $10&1;
 $12 = ($0|0)==(0|0);
 L1: do {
  if ($12) {
   $41 = HEAP32[7324]|0;
   $42 = ($41|0)==(0);
   if ($42) {
    $81 = HEAP32[(29340)>>2]|0;
    $82 = (FUNCTION_TABLE_ii[$81 & 127](32)|0);
    $$0$i$i = $82;
    label = 24;
    break;
   }
   $43 = HEAP32[(29356)>>2]|0;
   $44 = (FUNCTION_TABLE_ii[$43 & 127](32)|0);
   $45 = HEAP32[14985]|0;
   $46 = ($45>>>0)<(32);
   if ($46) {
    HEAP32[14985] = 32;
   }
   $47 = (59064);
   $48 = $47;
   $49 = HEAP32[$48>>2]|0;
   $50 = (($47) + 4)|0;
   $51 = $50;
   $52 = HEAP32[$51>>2]|0;
   $53 = ($52|0)>(0);
   $54 = ($49>>>0)>(0);
   $55 = ($52|0)==(0);
   $56 = $55 & $54;
   $57 = $53 | $56;
   if ($57) {
    $58 = HEAP32[14978]|0;
    $59 = ($44|0)<(0);
    $60 = $59 << 31 >> 31;
    $61 = (_i64Subtract(($49|0),($52|0),($44|0),($60|0))|0);
    $62 = (getTempRet0() | 0);
    $63 = ($62|0)<(0);
    $64 = ($61>>>0)<=($58>>>0);
    $65 = ($62|0)==(0);
    $66 = $65 & $64;
    $67 = $63 | $66;
    $$ = $67&1;
    HEAP32[(59072)>>2] = $$;
   }
   $68 = HEAP32[(29340)>>2]|0;
   $69 = (FUNCTION_TABLE_ii[$68 & 127]($44)|0);
   $70 = ($69|0)==(0|0);
   if ($70) {
    $$0$i6$i = 0;
   } else {
    $71 = HEAP32[(29352)>>2]|0;
    $72 = (FUNCTION_TABLE_ii[$71 & 127]($69)|0);
    $73 = HEAP32[14978]|0;
    $74 = (($73) + ($72))|0;
    HEAP32[14978] = $74;
    $75 = HEAP32[14982]|0;
    $76 = ($74>>>0)>($75>>>0);
    if ($76) {
     HEAP32[14982] = $74;
    }
    $77 = HEAP32[14981]|0;
    $78 = (($77) + 1)|0;
    HEAP32[14981] = $78;
    $79 = HEAP32[14987]|0;
    $80 = ($78>>>0)>($79>>>0);
    if ($80) {
     HEAP32[14987] = $78;
     $$0$i$i194 = $69;
     label = 25;
    } else {
     $$0$i$i194 = $69;
     label = 25;
    }
   }
  } else {
   $13 = ((($0)) + 272|0);
   $14 = HEAP32[$13>>2]|0;
   $15 = ($14|0)==(0);
   if ($15) {
    $16 = ((($0)) + 276|0);
    $17 = HEAP16[$16>>1]|0;
    $18 = ($17&65535)<(32);
    do {
     if ($18) {
      $19 = ((($0)) + 288|0);
      $$sink = $19;
     } else {
      $20 = ((($0)) + 300|0);
      $21 = HEAP32[$20>>2]|0;
      $22 = ($21|0)==(0|0);
      if (!($22)) {
       $23 = HEAP32[$21>>2]|0;
       HEAP32[$20>>2] = $23;
       $24 = ((($0)) + 284|0);
       $25 = HEAP32[$24>>2]|0;
       $26 = (($25) + 1)|0;
       HEAP32[$24>>2] = $26;
       $$0$i$i = $21;
       label = 24;
       break L1;
      }
      $27 = ((($0)) + 296|0);
      $28 = HEAP32[$27>>2]|0;
      $29 = ($28|0)==(0|0);
      if ($29) {
       $34 = ((($0)) + 292|0);
       $$sink = $34;
       break;
      } else {
       $30 = HEAP32[$28>>2]|0;
       HEAP32[$27>>2] = $30;
       $31 = ((($0)) + 284|0);
       $32 = HEAP32[$31>>2]|0;
       $33 = (($32) + 1)|0;
       HEAP32[$31>>2] = $33;
       $$0$i$i = $28;
       label = 24;
       break L1;
      }
     }
    } while(0);
    $38 = HEAP32[$$sink>>2]|0;
    $39 = (($38) + 1)|0;
    HEAP32[$$sink>>2] = $39;
   } else {
    $35 = ((($0)) + 81|0);
    $36 = HEAP8[$35>>0]|0;
    $37 = ($36<<24>>24)==(0);
    if (!($37)) {
     $$0$i6$i = 0;
     break;
    }
   }
   $40 = (_dbMallocRawFinish($0,32,0)|0);
   $$0$i$i = $40;
   label = 24;
  }
 } while(0);
 if ((label|0) == 24) {
  $83 = ($$0$i$i|0)==(0|0);
  if ($83) {
   $$0$i6$i = 0;
  } else {
   $$0$i$i194 = $$0$i$i;
   label = 25;
  }
 }
 if ((label|0) == 25) {
  dest=$$0$i$i194; stop=dest+32|0; do { HEAP8[dest>>0]=0|0; dest=dest+1|0; } while ((dest|0) < (stop|0));
  $$0$i6$i = $$0$i$i194;
 }
 $84 = ($$0$i6$i|0)!=(0|0);
 $85 = ((($0)) + 20|0);
 $86 = ((($0)) + 16|0);
 $87 = ((($0)) + 480|0);
 $88 = ((($0)) + 304|0);
 $89 = ((($0)) + 308|0);
 $90 = ((($0)) + 300|0);
 $91 = ((($$0$i6$i)) + 28|0);
 $92 = ((($$0$i6$i)) + 24|0);
 $93 = ((($0)) + 32|0);
 $94 = ((($$0$i6$i)) + 16|0);
 $95 = ((($$0$i6$i)) + 8|0);
 $96 = ((($$0$i6$i)) + 20|0);
 $97 = ((($0)) + 81|0);
 $98 = ((($9)) + 8|0);
 $99 = ((($9)) + 196|0);
 $100 = ((($9)) + 44|0);
 $101 = ((($9)) + 40|0);
 $$0149 = 0;$$0159 = 0;
 while(1) {
  _memset(($9|0),0,276)|0;
  if (!($84)) {
   label = 117;
   break;
  }
  HEAP32[$9>>2] = $0;
  $102 = HEAP32[$8>>2]|0;
  $103 = ($102|0)==(0|0);
  do {
   if (!($103)) {
    if (!($12)) {
     $104 = HEAP32[$87>>2]|0;
     $105 = ($104|0)==(0|0);
     if (!($105)) {
      _measureAllocationSize($0,$102);
      break;
     }
     $106 = $102;
     $107 = HEAP32[$88>>2]|0;
     $108 = ($107>>>0)>($106>>>0);
     if (!($108)) {
      $109 = HEAP32[$89>>2]|0;
      $110 = ($109>>>0)>($106>>>0);
      if ($110) {
       $111 = HEAP32[$90>>2]|0;
       HEAP32[$102>>2] = $111;
       HEAP32[$90>>2] = $102;
       break;
      }
     }
    }
    $112 = HEAP32[7324]|0;
    $113 = ($112|0)==(0);
    if ($113) {
     $121 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$121 & 127]($102);
     break;
    } else {
     $114 = HEAP32[(29352)>>2]|0;
     $115 = (FUNCTION_TABLE_ii[$114 & 127]($102)|0);
     $116 = HEAP32[14978]|0;
     $117 = (($116) - ($115))|0;
     HEAP32[14978] = $117;
     $118 = HEAP32[14981]|0;
     $119 = (($118) + -1)|0;
     HEAP32[14981] = $119;
     $120 = HEAP32[(29344)>>2]|0;
     FUNCTION_TABLE_vi[$120 & 127]($102);
     break;
    }
   }
  } while(0);
  HEAP32[$8>>2] = 0;
  $122 = HEAP32[$85>>2]|0;
  $123 = ($122|0)>(0);
  if ($123) {
   $124 = HEAP32[$86>>2]|0;
   $$010$i = 0;
   while(1) {
    $125 = (((($124) + ($$010$i<<4)|0)) + 4|0);
    $126 = HEAP32[$125>>2]|0;
    $127 = ($126|0)==(0|0);
    if (!($127)) {
     $128 = HEAP32[$126>>2]|0;
     $129 = ((($126)) + 4|0);
     $130 = HEAP32[$129>>2]|0;
     $131 = ((($130)) + 4|0);
     HEAP32[$131>>2] = $128;
    }
    $132 = (($$010$i) + 1)|0;
    $exitcond = ($132|0)==($122|0);
    if ($exitcond) {
     break;
    } else {
     $$010$i = $132;
    }
   }
  }
  $133 = (_sqlite3LocateTable($9,0,$2,$1)|0);
  $134 = ($133|0)==(0|0);
  if ($134) {
   label = 50;
   break;
  }
  $135 = ((($133)) + 56|0);
  $136 = HEAP32[$135>>2]|0;
  $137 = ($136|0)==(0);
  if (!($137)) {
   label = 45;
   break;
  }
  $138 = ((($133)) + 36|0);
  $139 = HEAP32[$138>>2]|0;
  $140 = $139 & 32;
  $141 = ($140|0)==(0);
  if (!($141)) {
   label = 47;
   break;
  }
  $142 = ((($133)) + 12|0);
  $143 = HEAP32[$142>>2]|0;
  $144 = ($143|0)==(0|0);
  if (!($144)) {
   label = 49;
   break;
  }
  HEAP32[$91>>2] = $133;
  $149 = HEAP32[$86>>2]|0;
  $150 = ((($133)) + 72|0);
  $151 = HEAP32[$150>>2]|0;
  $152 = ($151|0)==(0|0);
  if ($152) {
   $$1$i = -1000000;
  } else {
   $$0$i = 0;
   while(1) {
    $153 = (((($149) + ($$0$i<<4)|0)) + 12|0);
    $154 = HEAP32[$153>>2]|0;
    $155 = ($154|0)==($151|0);
    $156 = (($$0$i) + 1)|0;
    if ($155) {
     $$1$i = $$0$i;
     break;
    } else {
     $$0$i = $156;
    }
   }
  }
  $157 = (($149) + ($$1$i<<4)|0);
  $158 = HEAP32[$157>>2]|0;
  HEAP32[$92>>2] = $158;
  $159 = ((($133)) + 42|0);
  $160 = HEAP16[$159>>1]|0;
  $161 = $160 << 16 >> 16;
  $162 = ($160<<16>>16)>(0);
  L71: do {
   if ($162) {
    $163 = ((($133)) + 4|0);
    $164 = HEAP32[$163>>2]|0;
    $165 = HEAP8[$3>>0]|0;
    $166 = $165&255;
    $167 = (208 + ($166)|0);
    $168 = HEAP8[$167>>0]|0;
    $169 = $168&255;
    $$0163218 = 0;
    while(1) {
     $170 = (($164) + ($$0163218<<4)|0);
     $171 = HEAP32[$170>>2]|0;
     $172 = HEAP8[$171>>0]|0;
     $173 = $172&255;
     $174 = (208 + ($173)|0);
     $175 = HEAP8[$174>>0]|0;
     $176 = $175&255;
     $177 = (($176) - ($169))|0;
     $178 = ($177|0)!=(0);
     $179 = ($172<<24>>24)==(0);
     $or$cond14$i = $179 | $178;
     if ($or$cond14$i) {
      $$lcssa$i = $177;
     } else {
      $$01315$i = $3;$$016$i = $171;
      while(1) {
       $180 = ((($$016$i)) + 1|0);
       $181 = ((($$01315$i)) + 1|0);
       $182 = HEAP8[$180>>0]|0;
       $183 = $182&255;
       $184 = (208 + ($183)|0);
       $185 = HEAP8[$184>>0]|0;
       $186 = $185&255;
       $187 = HEAP8[$181>>0]|0;
       $188 = $187&255;
       $189 = (208 + ($188)|0);
       $190 = HEAP8[$189>>0]|0;
       $191 = $190&255;
       $192 = (($186) - ($191))|0;
       $193 = ($192|0)!=(0);
       $194 = ($182<<24>>24)==(0);
       $or$cond$i = $194 | $193;
       if ($or$cond$i) {
        $$lcssa$i = $192;
        break;
       } else {
        $$01315$i = $181;$$016$i = $180;
       }
      }
     }
     $195 = ($$lcssa$i|0)==(0);
     if ($195) {
      $$0163$lcssa = $$0163218;
      break L71;
     }
     $196 = (($$0163218) + 1)|0;
     $197 = ($196|0)<($161|0);
     if ($197) {
      $$0163218 = $196;
     } else {
      $$0163$lcssa = $196;
      break;
     }
    }
   } else {
    $$0163$lcssa = 0;
   }
  } while(0);
  $198 = ($$0163$lcssa|0)==($161|0);
  if ($198) {
   label = 63;
   break;
  }
  if ($10) {
   $200 = $93;
   $201 = $200;
   $202 = HEAP32[$201>>2]|0;
   $203 = (($200) + 4)|0;
   $204 = $203;
   $205 = HEAP32[$204>>2]|0;
   $206 = $202 & 16384;
   $207 = ($206|0)==(0);
   $208 = (0)==(0);
   $209 = $207 & $208;
   if ($209) {
    $$3 = 0;
   } else {
    $210 = ((($133)) + 16|0);
    $$0153231 = HEAP32[$210>>2]|0;
    $211 = ($$0153231|0)==(0|0);
    if ($211) {
     $$3 = 0;
    } else {
     $$0153233 = $$0153231;$$0155232 = 0;
     while(1) {
      $212 = ((($$0153233)) + 20|0);
      $213 = HEAP32[$212>>2]|0;
      $214 = ($213|0)>(0);
      if ($214) {
       $$0152226 = 0;$$1225 = $$0155232;
       while(1) {
        $215 = (((($$0153233)) + 36|0) + ($$0152226<<3)|0);
        $216 = HEAP32[$215>>2]|0;
        $217 = ($216|0)==($$0163$lcssa|0);
        $spec$select = $217 ? 31354 : $$1225;
        $218 = (($$0152226) + 1)|0;
        $219 = ($218|0)<($213|0);
        if ($219) {
         $$0152226 = $218;$$1225 = $spec$select;
        } else {
         $$1$lcssa = $spec$select;
         break;
        }
       }
      } else {
       $$1$lcssa = $$0155232;
      }
      $220 = ((($$0153233)) + 4|0);
      $$0153 = HEAP32[$220>>2]|0;
      $221 = ($$0153|0)==(0|0);
      if ($221) {
       $$3 = $$1$lcssa;
       break;
      } else {
       $$0153233 = $$0153;$$0155232 = $$1$lcssa;
      }
     }
    }
   }
   $222 = ((($133)) + 8|0);
   $$0154244 = HEAP32[$222>>2]|0;
   $223 = ($$0154244|0)==(0|0);
   if ($223) {
    $$4$lcssa = $$3;
   } else {
    $$0154246 = $$0154244;$$4245 = $$3;
    while(1) {
     $224 = ((($$0154246)) + 50|0);
     $225 = HEAP16[$224>>1]|0;
     $226 = ($225<<16>>16)==(0);
     if ($226) {
      $$5$lcssa = $$4245;
     } else {
      $227 = ((($$0154246)) + 4|0);
      $228 = HEAP32[$227>>2]|0;
      $229 = $225&65535;
      $$0151239 = 0;$$5238 = $$4245;
      while(1) {
       $230 = (($228) + ($$0151239<<1)|0);
       $231 = HEAP16[$230>>1]|0;
       $232 = $231 << 16 >> 16;
       $233 = ($$0163$lcssa|0)==($232|0);
       $234 = ($231<<16>>16)==(-2);
       $or$cond = $234 | $233;
       $$6 = $or$cond ? 31366 : $$5238;
       $235 = (($$0151239) + 1)|0;
       $236 = ($235>>>0)<($229>>>0);
       if ($236) {
        $$0151239 = $235;$$5238 = $$6;
       } else {
        $$5$lcssa = $$6;
        break;
       }
      }
     }
     $237 = ((($$0154246)) + 20|0);
     $$0154 = HEAP32[$237>>2]|0;
     $238 = ($$0154|0)==(0|0);
     if ($238) {
      $$4$lcssa = $$5$lcssa;
      break;
     } else {
      $$0154246 = $$0154;$$4245 = $$5$lcssa;
     }
    }
   }
   $239 = ($$4$lcssa|0)==(0|0);
   if (!($239)) {
    label = 79;
    break;
   }
  }
  $241 = HEAP32[$9>>2]|0;
  $242 = ((($241)) + 272|0);
  $243 = HEAP32[$242>>2]|0;
  $244 = ($243|0)==(0);
  L107: do {
   if ($244) {
    $245 = ((($241)) + 276|0);
    $246 = HEAP16[$245>>1]|0;
    $247 = ($246&65535)<(224);
    do {
     if ($247) {
      $248 = ((($241)) + 288|0);
      $$sink266 = $248;
     } else {
      $249 = ((($241)) + 300|0);
      $250 = HEAP32[$249>>2]|0;
      $251 = ($250|0)==(0|0);
      if (!($251)) {
       $252 = HEAP32[$250>>2]|0;
       HEAP32[$249>>2] = $252;
       $253 = ((($241)) + 284|0);
       $254 = HEAP32[$253>>2]|0;
       $255 = (($254) + 1)|0;
       HEAP32[$253>>2] = $255;
       $$0$i178 = $250;
       label = 91;
       break L107;
      }
      $256 = ((($241)) + 296|0);
      $257 = HEAP32[$256>>2]|0;
      $258 = ($257|0)==(0|0);
      if ($258) {
       $263 = ((($241)) + 292|0);
       $$sink266 = $263;
       break;
      } else {
       $259 = HEAP32[$257>>2]|0;
       HEAP32[$256>>2] = $259;
       $260 = ((($241)) + 284|0);
       $261 = HEAP32[$260>>2]|0;
       $262 = (($261) + 1)|0;
       HEAP32[$260>>2] = $262;
       $$0$i178 = $257;
       label = 91;
       break L107;
      }
     }
    } while(0);
    $267 = HEAP32[$$sink266>>2]|0;
    $268 = (($267) + 1)|0;
    HEAP32[$$sink266>>2] = $268;
    label = 90;
   } else {
    $264 = ((($241)) + 81|0);
    $265 = HEAP8[$264>>0]|0;
    $266 = ($265<<24>>24)==(0);
    if ($266) {
     label = 90;
    } else {
     label = 94;
    }
   }
  } while(0);
  if ((label|0) == 90) {
   label = 0;
   $269 = (_dbMallocRawFinish($241,224,0)|0);
   $$0$i178 = $269;
   label = 91;
  }
  do {
   if ((label|0) == 91) {
    label = 0;
    $270 = ($$0$i178|0)==(0|0);
    if ($270) {
     label = 94;
    } else {
     $271 = ((($$0$i178)) + 104|0);
     dest=$271; stop=dest+120|0; do { HEAP32[dest>>2]=0|0; dest=dest+4|0; } while ((dest|0) < (stop|0));
     HEAP32[$$0$i178>>2] = $241;
     $272 = ((($241)) + 4|0);
     $273 = HEAP32[$272>>2]|0;
     $274 = ($273|0)==(0|0);
     $275 = $273;
     if (!($274)) {
      $276 = ((($273)) + 4|0);
      HEAP32[$276>>2] = $$0$i178;
     }
     $277 = ((($$0$i178)) + 8|0);
     HEAP32[$277>>2] = $275;
     $278 = ((($$0$i178)) + 4|0);
     HEAP32[$278>>2] = 0;
     HEAP32[$272>>2] = $$0$i178;
     $279 = ((($$0$i178)) + 20|0);
     HEAP32[$279>>2] = 381479589;
     $280 = ((($$0$i178)) + 12|0);
     HEAP32[$280>>2] = $9;
     HEAP32[$98>>2] = $$0$i178;
     (_growOp3($$0$i178,61,0,1,0)|0);
     HEAP32[$94>>2] = $$0$i178;
     $281 = HEAP32[$150>>2]|0;
     $282 = ($281|0)==(0|0);
     if ($282) {
      $$1$i176 = -1000000;
     } else {
      $283 = HEAP32[$86>>2]|0;
      $$0$i175 = 0;
      while(1) {
       $284 = (((($283) + ($$0$i175<<4)|0)) + 12|0);
       $285 = HEAP32[$284>>2]|0;
       $286 = ($285|0)==($281|0);
       $287 = (($$0$i175) + 1)|0;
       if ($286) {
        $$1$i176 = $$0$i175;
        break;
       } else {
        $$0$i175 = $287;
       }
      }
     }
     $288 = HEAP32[$281>>2]|0;
     $289 = ((($281)) + 4|0);
     $290 = HEAP32[$289>>2]|0;
     $291 = ((($$0$i178)) + 108|0);
     $292 = HEAP32[$291>>2]|0;
     $293 = ((($$0$i178)) + 112|0);
     $294 = HEAP32[$293>>2]|0;
     $295 = ($294|0)>($292|0);
     if ($295) {
      $297 = (($292) + 1)|0;
      HEAP32[$291>>2] = $297;
      $298 = HEAP32[$271>>2]|0;
      $299 = (($298) + (($292*20)|0)|0);
      HEAP8[$299>>0] = 2;
      $300 = (((($298) + (($292*20)|0)|0)) + 2|0);
      HEAP16[$300>>1] = 0;
      $301 = (((($298) + (($292*20)|0)|0)) + 4|0);
      HEAP32[$301>>2] = $$1$i176;
      $