var events = require('events');
var utils = require('./utils');
var wrap = exports.wrap = {
  /**
   * Wrap an anonymous or named function to notify an Emitter and
   * return the wrapper function.
   * @param {events.EventEmitter} emitter The emitter to notify
   * @param {Function} fn The function to wrap
   * @param {String} name The optional name
   */
  fn: function(emitter, fn, strict, name, scope) {
    var wrapped = function() {
      var result;
      utils.emitEvents(emitter, 'before', name, [arguments, this, name]);

      try {
        result = fn.apply(scope || this, arguments);
      } catch (e) {
        utils.emitEvents(emitter, 'error', name, [ e, arguments, this, name ]);
        throw e;
      }

      utils.emitEvents(emitter, 'after', name, [ result, arguments, this, name ]);
      return result;
    };

    if (strict) {
      eval('wrapped = ' + utils.addArgs(wrapped.toString(), fn.length));
    }

    return wrapped;
  },
  /**
   * Wrap an anonymous or named function that calls a callback asynchronously
   * to notify an Emitter and return the wrapper function.
   * @param {events.EventEmitter} emitter The emitter to notify
   * @param {Function} fn The function to wrap
   * @param {Integer} position The position of the callback in the arguments
   * array (defaults to 0). Set to -1 if the callback is the last argument.
   * @param {String} name The optional name
   */
  async: function(emitter, fn, position, strict, name, scope) {
    var wrapped = function() {
      var pos = position == -1 ? arguments.length - 1 : (position || 0);
      var callback = arguments[pos];
      var context = this;
      var methodArgs = arguments;
      var callbackWrapper = function() {
        try {
          callback.apply(context, arguments);
        } catch (e) {
          utils.emitEvents(emitter, 'error', name, [ e, methodArgs, context, name ]);
          throw e;
        }
        var eventType = arguments[0] instanceof Error ? 'error' : 'after';
        utils.emitEvents(emitter, eventType, name, [ arguments, methodArgs, context, name ]);
      };

      utils.emitEvents(emitter, 'before', name, [ methodArgs, this, name ]);
      methodArgs[pos] = callbackWrapper;

      try {
        return fn.apply(scope || this, methodArgs);
      } catch (e) {
        utils.emitEvents(emitter, 'error', name, [ e, methodArgs, context, name ]);
        throw e;
      }
    };

    if (strict) {
      eval('wrapped = ' + utils.addArgs(wrapped.toString(), fn.length));
    }

    return wrapped;
  }
};

var Emitter = exports.Emitter = function(obj) {
  this.obj = obj;
};

Emitter.prototype = Object.create(events.EventEmitter.prototype);

/**
 * Punch a method with the given name, with
 * @param {String | Array} method The name of the method or a list of
 * method names.
 * @param {Integer} position The optional position of the asynchronous callback
 * in the arguments list.
 */
Emitter.prototype.punch = function(method, position, strict) {
  if (Array.isArray(method)) {
    var self = this;
    method.forEach(function(method) {
      self.punch(method, position, strict);
    });
  } else {
    var old = this.obj[method];
    if (typeof old == 'function') {
      this.obj[method] = (!position && position !== 0) ?
        wrap.fn(this, old, strict, method) :
        wrap.async(this, old, position, strict, method);
    }
  }
  return this;
};

exports.emitter = function(obj) {
  return new Emitter(obj);
};
