"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Identifies the kind of a CommandLineParameter.
 * @public
 */
var CommandLineParameterKind;
(function (CommandLineParameterKind) {
    /** Indicates a CommandLineChoiceParameter */
    CommandLineParameterKind[CommandLineParameterKind["Choice"] = 0] = "Choice";
    /** Indicates a CommandLineFlagParameter */
    CommandLineParameterKind[CommandLineParameterKind["Flag"] = 1] = "Flag";
    /** Indicates a CommandLineIntegerParameter */
    CommandLineParameterKind[CommandLineParameterKind["Integer"] = 2] = "Integer";
    /** Indicates a CommandLineStringParameter */
    CommandLineParameterKind[CommandLineParameterKind["String"] = 3] = "String";
    /** Indicates a CommandLineStringListParameter */
    CommandLineParameterKind[CommandLineParameterKind["StringList"] = 4] = "StringList";
})(CommandLineParameterKind = exports.CommandLineParameterKind || (exports.CommandLineParameterKind = {}));
/**
 * The base class for the various command-line parameter types.
 * @public
 */
class CommandLineParameter {
    /** @internal */
    constructor(definition) {
        this.longName = definition.parameterLongName;
        this.shortName = definition.parameterShortName;
        this.description = definition.description;
        this.required = !!definition.required;
        this.environmentVariable = definition.environmentVariable;
        if (!CommandLineParameter._longNameRegExp.test(this.longName)) {
            throw new Error(`Invalid name: "${this.longName}". The parameter long name must be`
                + ` lower-case and use dash delimiters (e.g. "--do-a-thing")`);
        }
        if (this.shortName) {
            if (!CommandLineParameter._shortNameRegExp.test(this.shortName)) {
                throw new Error(`Invalid name: "${this.shortName}". The parameter short name must be`
                    + ` a dash followed by a single upper-case or lower-case letter (e.g. "-a")`);
            }
        }
        if (this.environmentVariable) {
            if (this.required) {
                throw new Error(`An "environmentVariable" cannot be specified for "${this.longName}"`
                    + ` because it is a required parameter`);
            }
            if (!CommandLineParameter._environmentVariableRegExp.test(this.environmentVariable)) {
                throw new Error(`Invalid environment variable name: "${this.environmentVariable}". The name must`
                    + ` consist only of upper-case letters, numbers, and underscores. It may not start with a number.`);
            }
        }
    }
    /**
     * Returns additional text used by the help formatter.
     * @internal
     */
    _getSupplementaryNotes(supplementaryNotes) {
        if (this.environmentVariable !== undefined) {
            supplementaryNotes.push('This parameter may alternatively specified via the ' + this.environmentVariable
                + ' environment variable.');
        }
    }
    /**
     * Internal usage only.  Used to report unexpected output from the argparse library.
     */
    reportInvalidData(data) {
        throw new Error(`Unexpected data object for parameter "${this.longName}": `
            + JSON.stringify(data));
    }
    validateDefaultValue(hasDefaultValue) {
        if (this.required && hasDefaultValue) {
            // If a parameter is "required", then the user understands that they always need to
            // specify a value for this parameter (either via the command line or via an environment variable).
            // It would be confusing to allow a default value that sometimes allows the "required" parameter
            // to be omitted.  If you sometimes don't have a suitable default value, then the better approach
            // is to throw a custom error explaining why the parameter is required in that case.
            throw new Error(`A default value cannot be specified for "${this.longName}"`
                + ` because it is a "required" parameter`);
        }
    }
}
// Example: "--do-something"
CommandLineParameter._longNameRegExp = /^-(-[a-z0-9]+)+$/;
// Example: "-d"
CommandLineParameter._shortNameRegExp = /^-[a-zA-Z]$/;
// "Environment variable names used by the utilities in the Shell and Utilities volume of
// IEEE Std 1003.1-2001 consist solely of uppercase letters, digits, and the '_' (underscore)
// from the characters defined in Portable Character Set and do not begin with a digit."
// Example: "THE_SETTING"
CommandLineParameter._environmentVariableRegExp = /^[A-Z_][A-Z0-9_]*$/;
exports.CommandLineParameter = CommandLineParameter;
/**
 * The common base class for parameters types that receive an argument.
 *
 * @remarks
 * An argument is an accompanying command-line token, such as "123" in the
 * example "--max-count 123".
 * @public
 */
class CommandLineParameterWithArgument extends CommandLineParameter {
    /** @internal */
    constructor(definition) {
        super(definition);
        if (definition.argumentName === '') {
            throw new Error('The argument name cannot be an empty string. (For the default name, specify undefined.)');
        }
        if (definition.argumentName.toUpperCase() !== definition.argumentName) {
            throw new Error(`Invalid name: "${definition.argumentName}". The argument name must be all upper case.`);
        }
        const match = definition.argumentName.match(CommandLineParameterWithArgument._invalidArgumentNameRegExp);
        if (match) {
            throw new Error(`The argument name "${definition.argumentName}" contains an invalid character "${match[0]}".`
                + ` Only upper-case letters, numbers, and underscores are allowed.`);
        }
        this.argumentName = definition.argumentName;
    }
}
// Matches the first character that *isn't* part of a valid upper-case argument name such as "URL_2"
CommandLineParameterWithArgument._invalidArgumentNameRegExp = /[^A-Z_0-9]/;
exports.CommandLineParameterWithArgument = CommandLineParameterWithArgument;
/**
 * The data type returned by {@link CommandLineParameterProvider.defineChoiceParameter}.
 * @public
 */
class CommandLineChoiceParameter extends CommandLineParameter {
    /** @internal */
    constructor(definition) {
        super(definition);
        this._value = undefined;
        if (definition.alternatives.length < 1) {
            throw new Error(`When defining a choice parameter, the alternatives list must contain at least one value.`);
        }
        if (definition.defaultValue && definition.alternatives.indexOf(definition.defaultValue) === -1) {
            throw new Error(`The specified default value "${definition.defaultValue}"`
                + ` is not one of the available options: ${definition.alternatives.toString()}`);
        }
        this.alternatives = definition.alternatives;
        this.defaultValue = definition.defaultValue;
        this.validateDefaultValue(!!this.defaultValue);
    }
    /** {@inheritDoc CommandLineParameter.kind} */
    get kind() {
        return CommandLineParameterKind.Choice;
    }
    /**
     * {@inheritDoc CommandLineParameter._setValue}
     * @internal
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    _setValue(data) {
        if (data !== null && data !== undefined) {
            if (typeof data !== 'string') {
                this.reportInvalidData(data);
            }
            this._value = data;
            return;
        }
        if (this.environmentVariable !== undefined) {
            // Try reading the environment variable
            const environmentValue = process.env[this.environmentVariable];
            if (environmentValue !== undefined && environmentValue !== '') {
                if (this.alternatives.indexOf(environmentValue) < 0) {
                    const choices = '"' + this.alternatives.join('", "') + '"';
                    throw new Error(`Invalid value "${environmentValue}" for the environment variable`
                        + ` ${this.environmentVariable}.  Valid choices are: ${choices}`);
                }
                this._value = environmentValue;
                return;
            }
        }
        if (this.defaultValue !== undefined) {
            this._value = this.defaultValue;
            return;
        }
        this._value = undefined;
    }
    /**
     * {@inheritDoc CommandLineParameter._getSupplementaryNotes}
     * @internal
     */
    _getSupplementaryNotes(supplementaryNotes) {
        super._getSupplementaryNotes(supplementaryNotes);
        if (this.defaultValue !== undefined) {
            supplementaryNotes.push(`The default value is "${this.defaultValue}".`);
        }
    }
    /**
     * Returns the argument value for a choice parameter that was parsed from the command line.
     *
     * @remarks
     * The return value will be `undefined` if the command-line has not been parsed yet,
     * or if the parameter was omitted and has no default value.
     */
    get value() {
        return this._value;
    }
    /** {@inheritDoc CommandLineParameter.appendToArgList} @override */
    appendToArgList(argList) {
        if (this.value !== undefined) {
            argList.push(this.longName);
            argList.push(this.value);
        }
    }
}
exports.CommandLineChoiceParameter = CommandLineChoiceParameter;
/**
 * The data type returned by {@link CommandLineParameterProvider.defineFlagParameter}.
 * @public
 */
class CommandLineFlagParameter extends CommandLineParameter {
    /** @internal */
    constructor(definition) {
        super(definition);
        this._value = false;
    }
    /** {@inheritDoc CommandLineParameter.kind} */
    get kind() {
        return CommandLineParameterKind.Flag;
    }
    /**
     * {@inheritDoc CommandLineParameter._setValue}
     * @internal
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    _setValue(data) {
        if (data !== null && data !== undefined) {
            if (typeof data !== 'boolean') {
                this.reportInvalidData(data);
            }
            this._value = data;
            return;
        }
        if (this.environmentVariable !== undefined) {
            // Try reading the environment variable
            const environmentValue = process.env[this.environmentVariable];
            if (environmentValue !== undefined && environmentValue !== '') {
                if (environmentValue !== '0' && environmentValue !== '1') {
                    throw new Error(`Invalid value "${environmentValue}" for the environment variable`
                        + ` ${this.environmentVariable}.  Valid choices are 0 or 1.`);
                }
                this._value = environmentValue === '1';
                return;
            }
        }
        this._value = false;
    }
    /**
     * Returns a boolean indicating whether the parameter was included in the command line.
     *
     * @remarks
     * The return value will be false if the command-line has not been parsed yet,
     * or if the flag was not used.
     */
    get value() {
        return this._value;
    }
    /** {@inheritDoc CommandLineParameter.appendToArgList} @override */
    appendToArgList(argList) {
        if (this.value) {
            argList.push(this.longName);
        }
    }
}
exports.CommandLineFlagParameter = CommandLineFlagParameter;
/**
 * The data type returned by {@link CommandLineParameterProvider.defineIntegerParameter}.
 * @public
 */
class CommandLineIntegerParameter extends CommandLineParameterWithArgument {
    /** @internal */
    constructor(definition) {
        super(definition);
        this._value = undefined;
        this.defaultValue = definition.defaultValue;
        this.validateDefaultValue(!!this.defaultValue);
    }
    /** {@inheritDoc CommandLineParameter.kind} */
    get kind() {
        return CommandLineParameterKind.Integer;
    }
    /**
     * {@inheritDoc CommandLineParameter._setValue}
     * @internal
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    _setValue(data) {
        if (data !== null && data !== undefined) {
            if (typeof data !== 'number') {
                this.reportInvalidData(data);
            }
            this._value = data;
            return;
        }
        if (this.environmentVariable !== undefined) {
            // Try reading the environment variable
            const environmentValue = process.env[this.environmentVariable];
            if (environmentValue !== undefined && environmentValue !== '') {
                const parsed = parseInt(environmentValue, 10);
                if (isNaN(parsed) || environmentValue.indexOf('.') >= 0) {
                    throw new Error(`Invalid value "${environmentValue}" for the environment variable`
                        + ` ${this.environmentVariable}.  It must be an integer value.`);
                }
                this._value = parsed;
                return;
            }
        }
        if (this.defaultValue !== undefined) {
            this._value = this.defaultValue;
            return;
        }
        this._value = undefined;
    }
    /**
     * {@inheritDoc CommandLineParameter._getSupplementaryNotes}
     * @internal
     */
    _getSupplementaryNotes(supplementaryNotes) {
        super._getSupplementaryNotes(supplementaryNotes);
        if (this.defaultValue !== undefined) {
            supplementaryNotes.push(`The default value is ${this.defaultValue}.`);
        }
    }
    /**
     * Returns the argument value for an integer parameter that was parsed from the command line.
     *
     * @remarks
     * The return value will be undefined if the command-line has not been parsed yet,
     * or if the parameter was omitted and has no default value.
     */
    get value() {
        return this._value;
    }
    /** {@inheritDoc CommandLineParameter.appendToArgList} @override */
    appendToArgList(argList) {
        if (this.value !== undefined) {
            argList.push(this.longName);
            argList.push(this.value.toString());
        }
    }
}
exports.CommandLineIntegerParameter = CommandLineIntegerParameter;
/**
 * The data type returned by {@link CommandLineParameterProvider.defineStringParameter}.
 * @public
 */
class CommandLineStringParameter extends CommandLineParameterWithArgument {
    /** @internal */
    constructor(definition) {
        super(definition);
        this._value = undefined;
        this.defaultValue = definition.defaultValue;
        this.validateDefaultValue(!!this.defaultValue);
    }
    /** {@inheritDoc CommandLineParameter.kind} */
    get kind() {
        return CommandLineParameterKind.String;
    }
    /**
     * {@inheritDoc CommandLineParameter._setValue}
     * @internal
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    _setValue(data) {
        if (data !== null && data !== undefined) {
            if (typeof data !== 'string') {
                this.reportInvalidData(data);
            }
            this._value = data;
            return;
        }
        if (this.environmentVariable !== undefined) {
            // Try reading the environment variable
            const environmentValue = process.env[this.environmentVariable];
            if (environmentValue !== undefined) {
                // NOTE: If the environment variable is defined as an empty string,
                // here we will accept the empty string as our value.  (For number/flag we don't do that.)
                this._value = environmentValue;
                return;
            }
        }
        if (this.defaultValue !== undefined) {
            this._value = this.defaultValue;
            return;
        }
        this._value = undefined;
    }
    /**
     * {@inheritDoc CommandLineParameter._getSupplementaryNotes}
     * @internal
     */
    _getSupplementaryNotes(supplementaryNotes) {
        super._getSupplementaryNotes(supplementaryNotes);
        if (this.defaultValue !== undefined) {
            if (this.defaultValue.length < 160) {
                supplementaryNotes.push(`The default value is ${JSON.stringify(this.defaultValue)}.`);
            }
        }
    }
    /**
     * Returns the argument value for a string parameter that was parsed from the command line.
     *
     * @remarks
     * The return value will be undefined if the command-line has not been parsed yet,
     * or if the parameter was omitted and has no default value.
     */
    get value() {
        return this._value;
    }
    /** {@inheritDoc CommandLineParameter.appendToArgList} @override */
    appendToArgList(argList) {
        if (this.value !== undefined) {
            argList.push(this.longName);
            argList.push(this.value);
        }
    }
}
exports.CommandLineStringParameter = CommandLineStringParameter;
/**
 * The data type returned by {@link CommandLineParameterProvider.defineStringListParameter}.
 * @public
 */
class CommandLineStringListParameter extends CommandLineParameterWithArgument {
    /** @internal */
    constructor(definition) {
        super(definition);
        this._values = [];
    }
    /** {@inheritDoc CommandLineParameter.kind} */
    get kind() {
        return CommandLineParameterKind.StringList;
    }
    /**
     * {@inheritDoc CommandLineParameter._setValue}
     * @internal
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    _setValue(data) {
        if (data !== null && data !== undefined) {
            if (!Array.isArray(data)) {
                this.reportInvalidData(data);
            }
            for (const arrayItem of data) {
                if (typeof (arrayItem) !== 'string') {
                    this.reportInvalidData(data);
                }
            }
            this._values = data;
            return;
        }
        if (this.environmentVariable !== undefined) {
            // Try reading the environment variable
            const environmentValue = process.env[this.environmentVariable];
            if (environmentValue !== undefined) {
                // NOTE: If the environment variable is defined as an empty string,
                // here we will accept the empty string as our value.  (For number/flag we don't do that.)
                // In the current implementation, the environment variable for a "string list" can only
                // store a single item.  If we wanted to allow multiple items (and still have a conventional-seeming
                // environment), we would ask the caller to provide an appropriate delimiter.  Getting involved
                // with escaping here seems unwise, since there are so many shell escaping mechanisms that could
                // potentially confuse the experience.
                this._values = [environmentValue];
                return;
            }
        }
        // (No default value for string lists)
        this._values = [];
    }
    /**
     * Returns the string arguments for a string list parameter that was parsed from the command line.
     *
     * @remarks
     * The array will be empty if the command-line has not been parsed yet,
     * or if the parameter was omitted and has no default value.
     */
    get values() {
        return this._values;
    }
    /** {@inheritDoc CommandLineParameter.appendToArgList} @override */
    appendToArgList(argList) {
        if (this.values.length > 0) {
            for (const value of this.values) {
                argList.push(this.longName);
                argList.push(value);
            }
        }
    }
}
exports.CommandLineStringListParameter = CommandLineStringListParameter;
//# sourceMappingURL=CommandLineParameter.js.map