import * as argparse from 'argparse';
import { ICommandLineFlagDefinition, ICommandLineStringDefinition, ICommandLineStringListDefinition, ICommandLineIntegerDefinition, ICommandLineChoiceDefinition } from './CommandLineDefinition';
import { CommandLineParameter, CommandLineFlagParameter, CommandLineStringParameter, CommandLineStringListParameter, CommandLineIntegerParameter, CommandLineChoiceParameter } from './CommandLineParameter';
/**
 * This is the argparse result data object
 * @internal
 */
export interface ICommandLineParserData {
    action: string;
    [key: string]: any;
}
/**
 * This is the common base class for CommandLineAction and CommandLineParser
 * that provides functionality for defining command-line parameters.
 *
 * @public
 */
export declare abstract class CommandLineParameterProvider {
    private static _keyCounter;
    private _parameters;
    private _parametersByLongName;
    /** @internal */
    constructor();
    /**
     * Returns a collection of the parameters that were defined for this object.
     */
    readonly parameters: ReadonlyArray<CommandLineParameter>;
    /**
     * Defines a command-line parameter whose value must be a string from a fixed set of
     * allowable choices (similar to an enum).
     *
     * @remarks
     * Example:  example-tool --log-level warn
     */
    defineChoiceParameter(definition: ICommandLineChoiceDefinition): CommandLineChoiceParameter;
    /**
     * Returns the CommandLineChoiceParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getChoiceParameter(parameterLongName: string): CommandLineChoiceParameter;
    /**
     * Defines a command-line switch whose boolean value is true if the switch is provided,
     * and false otherwise.
     *
     * @remarks
     * Example:  example-tool --debug
     */
    defineFlagParameter(definition: ICommandLineFlagDefinition): CommandLineFlagParameter;
    /**
     * Returns the CommandLineFlagParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getFlagParameter(parameterLongName: string): CommandLineFlagParameter;
    /**
     * Defines a command-line parameter whose value is an integer.
     *
     * @remarks
     * Example:  example-tool --max-attempts 5
     */
    defineIntegerParameter(definition: ICommandLineIntegerDefinition): CommandLineIntegerParameter;
    /**
     * Returns the CommandLineIntegerParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getIntegerParameter(parameterLongName: string): CommandLineIntegerParameter;
    /**
     * Defines a command-line parameter whose value is a single text string.
     *
     * @remarks
     * Example:  example-tool --message "Hello, world!"
     */
    defineStringParameter(definition: ICommandLineStringDefinition): CommandLineStringParameter;
    /**
     * Returns the CommandLineStringParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getStringParameter(parameterLongName: string): CommandLineStringParameter;
    /**
     * Defines a command-line parameter whose value is one or more text strings.
     *
     * @remarks
     * Example:  example-tool --add file1.txt --add file2.txt --add file3.txt
     */
    defineStringListParameter(definition: ICommandLineStringListDefinition): CommandLineStringListParameter;
    /**
     * Returns the CommandLineStringListParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getStringListParameter(parameterLongName: string): CommandLineStringListParameter;
    /**
     * Generates the command-line help text.
     */
    renderHelpText(): string;
    /**
     * The child class should implement this hook to define its command-line parameters,
     * e.g. by calling defineFlagParameter().
     */
    protected abstract onDefineParameters(): void;
    /**
     * Retrieves the argparse object.
     * @internal
     */
    protected abstract _getArgumentParser(): argparse.ArgumentParser;
    /** @internal */
    protected _processParsedData(data: ICommandLineParserData): void;
    private _generateKey;
    private _getParameter;
    private _defineParameter;
}
//# sourceMappingURL=CommandLineParameterProvider.d.ts.map