"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
const CommandLineParameter_1 = require("./CommandLineParameter");
/**
 * This is the common base class for CommandLineAction and CommandLineParser
 * that provides functionality for defining command-line parameters.
 *
 * @public
 */
class CommandLineParameterProvider {
    /** @internal */
    // Third party code should not inherit subclasses or call this constructor
    constructor() {
        this._parameters = [];
        this._parametersByLongName = new Map();
    }
    /**
     * Returns a collection of the parameters that were defined for this object.
     */
    get parameters() {
        return this._parameters;
    }
    /**
     * Defines a command-line parameter whose value must be a string from a fixed set of
     * allowable choices (similar to an enum).
     *
     * @remarks
     * Example:  example-tool --log-level warn
     */
    defineChoiceParameter(definition) {
        const parameter = new CommandLineParameter_1.CommandLineChoiceParameter(definition);
        this._defineParameter(parameter);
        return parameter;
    }
    /**
     * Returns the CommandLineChoiceParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getChoiceParameter(parameterLongName) {
        return this._getParameter(parameterLongName, CommandLineParameter_1.CommandLineParameterKind.Choice);
    }
    /**
     * Defines a command-line switch whose boolean value is true if the switch is provided,
     * and false otherwise.
     *
     * @remarks
     * Example:  example-tool --debug
     */
    defineFlagParameter(definition) {
        const parameter = new CommandLineParameter_1.CommandLineFlagParameter(definition);
        this._defineParameter(parameter);
        return parameter;
    }
    /**
     * Returns the CommandLineFlagParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getFlagParameter(parameterLongName) {
        return this._getParameter(parameterLongName, CommandLineParameter_1.CommandLineParameterKind.Flag);
    }
    /**
     * Defines a command-line parameter whose value is an integer.
     *
     * @remarks
     * Example:  example-tool --max-attempts 5
     */
    defineIntegerParameter(definition) {
        const parameter = new CommandLineParameter_1.CommandLineIntegerParameter(definition);
        this._defineParameter(parameter);
        return parameter;
    }
    /**
     * Returns the CommandLineIntegerParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getIntegerParameter(parameterLongName) {
        return this._getParameter(parameterLongName, CommandLineParameter_1.CommandLineParameterKind.Integer);
    }
    /**
     * Defines a command-line parameter whose value is a single text string.
     *
     * @remarks
     * Example:  example-tool --message "Hello, world!"
     */
    defineStringParameter(definition) {
        const parameter = new CommandLineParameter_1.CommandLineStringParameter(definition);
        this._defineParameter(parameter);
        return parameter;
    }
    /**
     * Returns the CommandLineStringParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getStringParameter(parameterLongName) {
        return this._getParameter(parameterLongName, CommandLineParameter_1.CommandLineParameterKind.String);
    }
    /**
     * Defines a command-line parameter whose value is one or more text strings.
     *
     * @remarks
     * Example:  example-tool --add file1.txt --add file2.txt --add file3.txt
     */
    defineStringListParameter(definition) {
        const parameter = new CommandLineParameter_1.CommandLineStringListParameter(definition);
        this._defineParameter(parameter);
        return parameter;
    }
    /**
     * Returns the CommandLineStringListParameter with the specified long name.
     * @remarks
     * This method throws an exception if the parameter is not defined.
     */
    getStringListParameter(parameterLongName) {
        return this._getParameter(parameterLongName, CommandLineParameter_1.CommandLineParameterKind.StringList);
    }
    /**
     * Generates the command-line help text.
     */
    renderHelpText() {
        return this._getArgumentParser().formatHelp();
    }
    /** @internal */
    _processParsedData(data) {
        // Fill in the values for the parameters
        for (const parameter of this._parameters) {
            const value = data[parameter._parserKey]; // eslint-disable-line @typescript-eslint/no-explicit-any
            parameter._setValue(value);
        }
    }
    _generateKey() {
        return 'key_' + (CommandLineParameterProvider._keyCounter++).toString();
    }
    _getParameter(parameterLongName, expectedKind) {
        const parameter = this._parametersByLongName.get(parameterLongName);
        if (!parameter) {
            throw new Error(`The parameter "${parameterLongName}" is not defined`);
        }
        if (parameter.kind !== expectedKind) {
            throw new Error(`The parameter "${parameterLongName}" is of type "${CommandLineParameter_1.CommandLineParameterKind[parameter.kind]}"`
                + ` whereas the caller was expecting "${CommandLineParameter_1.CommandLineParameterKind[expectedKind]}".`);
        }
        return parameter;
    }
    _defineParameter(parameter) {
        const names = [];
        if (parameter.shortName) {
            names.push(parameter.shortName);
        }
        names.push(parameter.longName);
        parameter._parserKey = this._generateKey();
        let finalDescription = parameter.description;
        const supplementaryNotes = [];
        parameter._getSupplementaryNotes(supplementaryNotes);
        if (supplementaryNotes.length > 0) {
            // If they left the period off the end of their sentence, then add one.
            if (finalDescription.match(/[a-z0-9]\s*$/i)) {
                finalDescription = finalDescription.trimRight() + '.';
            }
            // Append the supplementary text
            finalDescription += ' ' + supplementaryNotes.join(' ');
        }
        // NOTE: Our "environmentVariable" feature takes precedence over argparse's "defaultValue",
        // so we have to reimplement that feature.
        const argparseOptions = {
            help: finalDescription,
            dest: parameter._parserKey,
            metavar: parameter.argumentName || undefined,
            required: parameter.required
        };
        switch (parameter.kind) {
            case CommandLineParameter_1.CommandLineParameterKind.Choice:
                const choiceParameter = parameter;
                argparseOptions.choices = choiceParameter.alternatives;
                break;
            case CommandLineParameter_1.CommandLineParameterKind.Flag:
                argparseOptions.action = 'storeTrue';
                break;
            case CommandLineParameter_1.CommandLineParameterKind.Integer:
                argparseOptions.type = 'int';
                break;
            case CommandLineParameter_1.CommandLineParameterKind.String:
                break;
            case CommandLineParameter_1.CommandLineParameterKind.StringList:
                argparseOptions.action = 'append';
                break;
        }
        this._getArgumentParser().addArgument(names, argparseOptions);
        this._parameters.push(parameter);
        this._parametersByLongName.set(parameter.longName, parameter);
    }
}
CommandLineParameterProvider._keyCounter = 0;
exports.CommandLineParameterProvider = CommandLineParameterProvider;
//# sourceMappingURL=CommandLineParameterProvider.js.map