import { Subject, AnonymousSubject } from '../../Subject';
import { Subscriber } from '../../Subscriber';
import { Observable } from '../../Observable';
import { Subscription } from '../../Subscription';
import { root } from '../../util/root';
import { ReplaySubject } from '../../ReplaySubject';
import { tryCatch } from '../../util/tryCatch';
import { errorObject } from '../../util/errorObject';
import { assign } from '../../util/assign';
/**
 * We need this JSDoc comment for affecting ESDoc.
 * @extends {Ignored}
 * @hide true
 */
export class WebSocketSubject extends AnonymousSubject {
    constructor(urlConfigOrSource, destination) {
        super();
        this._output = new Subject();
        this.WebSocketCtor = root.WebSocket;
        if (typeof urlConfigOrSource === 'string') {
            this.url = urlConfigOrSource;
        }
        else {
            // WARNING: config object could override important members here.
            assign(this, urlConfigOrSource);
        }
        if (!this.WebSocketCtor) {
            throw new Error('no WebSocket constructor can be found');
        }
        this.destination = new ReplaySubject();
    }
    resultSelector(e) {
        return JSON.parse(e.data);
    }
    /**
     * @param urlConfigOrSource
     * @return {WebSocketSubject}
     * @static true
     * @name webSocket
     * @owner Observable
     */
    static create(urlConfigOrSource) {
        return new WebSocketSubject(urlConfigOrSource);
    }
    lift(operator) {
        const sock = new WebSocketSubject(this, this.destination);
        sock.operator = operator;
        return sock;
    }
    // TODO: factor this out to be a proper Operator/Subscriber implementation and eliminate closures
    multiplex(subMsg, unsubMsg, messageFilter) {
        const self = this;
        return new Observable((observer) => {
            const result = tryCatch(subMsg)();
            if (result === errorObject) {
                observer.error(errorObject.e);
            }
            else {
                self.next(result);
            }
            let subscription = self.subscribe(x => {
                const result = tryCatch(messageFilter)(x);
                if (result === errorObject) {
                    observer.error(errorObject.e);
                }
                else if (result) {
                    observer.next(x);
                }
            }, err => observer.error(err), () => observer.complete());
            return () => {
                const result = tryCatch(unsubMsg)();
                if (result === errorObject) {
                    observer.error(errorObject.e);
                }
                else {
                    self.next(result);
                }
                subscription.unsubscribe();
            };
        });
    }
    _connectSocket() {
        const socket = this.protocol ? new WebSocket(this.url, this.protocol) : new WebSocket(this.url);
        this.socket = socket;
        const subscription = new Subscription(() => {
            this.socket = null;
            if (socket && socket.readyState === 1) {
                socket.close();
            }
        });
        const observer = this._output;
        socket.onopen = (e) => {
            const openObserver = this.openObserver;
            if (openObserver) {
                openObserver.next(e);
            }
            const queue = this.destination;
            this.destination = Subscriber.create((x) => socket.readyState === 1 && socket.send(x), (e) => {
                const closingObserver = this.closingObserver;
                if (closingObserver) {
                    closingObserver.next(undefined);
                }
                if (e && e.code) {
                    socket.close(e.code, e.reason);
                }
                else {
                    observer.error(new TypeError('WebSocketSubject.error must be called with an object with an error code, ' +
                        'and an optional reason: { code: number, reason: string }'));
                }
                this.destination = new ReplaySubject();
                this.socket = null;
            }, () => {
                const closingObserver = this.closingObserver;
                if (closingObserver) {
                    closingObserver.next(undefined);
                }
                socket.close();
                this.destination = new ReplaySubject();
                this.socket = null;
            });
            if (queue && queue instanceof ReplaySubject) {
                subscription.add(queue.subscribe(this.destination));
            }
        };
        socket.onerror = (e) => observer.error(e);
        socket.onclose = (e) => {
            const closeObserver = this.closeObserver;
            if (closeObserver) {
                closeObserver.next(e);
            }
            if (e.wasClean) {
                observer.complete();
            }
            else {
                observer.error(e);
            }
        };
        socket.onmessage = (e) => {
            const result = tryCatch(this.resultSelector)(e);
            if (result === errorObject) {
                observer.error(errorObject.e);
            }
            else {
                observer.next(result);
            }
        };
    }
    _subscribe(subscriber) {
        if (!this.socket) {
            this._connectSocket();
        }
        let subscription = new Subscription();
        subscription.add(this._output.subscribe(subscriber));
        subscription.add(() => {
            const { socket } = this;
            if (socket && socket.readyState === 1) {
                socket.close();
                this.socket = null;
            }
        });
        return subscription;
    }
    unsubscribe() {
        const { socket } = this;
        if (socket && socket.readyState === 1) {
            socket.close();
            this.socket = null;
        }
        super.unsubscribe();
        this.destination = new ReplaySubject();
    }
}
//# sourceMappingURL=WebSocketSubject.js.map