import { Observer } from './types';
import { Subscription } from './Subscription';
/**
 * Implements the {@link Observer} interface and extends the
 * {@link Subscription} class. While the {@link Observer} is the public API for
 * consuming the values of an {@link Observable}, all Observers get converted to
 * a Subscriber, in order to provide Subscription-like capabilities such as
 * `unsubscribe`. Subscriber is a common type in RxJS, and crucial for
 * implementing operators, but it is rarely used as a public API.
 *
 * @class Subscriber<T>
 */
export declare class Subscriber<T> extends Subscription implements Observer<T> {
    /** @deprecated Internal implementation detail, do not use directly. Will be made internal in v8. */
    protected isStopped: boolean;
    /** @deprecated Internal implementation detail, do not use directly. Will be made internal in v8. */
    protected destination: Subscriber<T> | Observer<T>;
    /**
     * Creates an instance of an RxJS Subscriber. This is the workhorse of the library.
     *
     * If another instance of Subscriber is passed in, it will automatically wire up unsubscription
     * between this instnace and the passed in instance.
     *
     * If a partial or full observer is passed in, it will be wrapped and appropriate safeguards will be applied.
     *
     * If a next-handler function is passed in, it will be wrapped and appropriate safeguards will be applied.
     *
     * @param destination A subscriber, partial observer, or function that receives the next value.
     */
    constructor(destination?: Subscriber<T> | Partial<Observer<T>> | ((value: T) => void) | null);
    /**
     * The {@link Observer} callback to receive notifications of type `next` from
     * the Observable, with a value. The Observable may call this method 0 or more
     * times.
     * @param {T} [value] The `next` value.
     * @return {void}
     */
    next(value?: T): void;
    /**
     * The {@link Observer} callback to receive notifications of type `error` from
     * the Observable, with an attached `Error`. Notifies the Observer that
     * the Observable has experienced an error condition.
     * @param {any} [err] The `error` exception.
     * @return {void}
     */
    error(err?: any): void;
    /**
     * The {@link Observer} callback to receive a valueless notification of type
     * `complete` from the Observable. Notifies the Observer that the Observable
     * has finished sending push-based notifications.
     * @return {void}
     */
    complete(): void;
    unsubscribe(): void;
    protected _next(value: T): void;
    protected _error(err: any): void;
    protected _complete(): void;
}
/**
 * The observer used as a stub for subscriptions where the user did not
 * pass any arguments to `subscribe`. Comes with the default error handling
 * behavior.
 */
export declare const EMPTY_OBSERVER: Readonly<Observer<any>> & {
    closed: true;
};
export declare function isSubscriber<T>(value: any): value is Subscriber<T>;
//# sourceMappingURL=Subscriber.d.ts.map