(function (global, factory) {
    if (typeof define === "function" && define.amd) {
        define(['exports', 'san', '../common/util/cx', '../common/util/css', '../babelHelpers'], factory);
    } else if (typeof exports !== "undefined") {
        factory(exports, require('san'), require('../common/util/cx'), require('../common/util/css'), require('../babelHelpers'));
    } else {
        var mod = {
            exports: {}
        };
        factory(mod.exports, global.san, global.cx, global.css, global.babelHelpers);
        global.CircularProgress = mod.exports;
    }
})(this, function (exports, _san, _cx, _css, babelHelpers) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var _san2 = babelHelpers.interopRequireDefault(_san);

    var _css2 = babelHelpers.interopRequireDefault(_css);

    var cx = (0, _cx.create)('circular-progress');

    var CIRCLE_SVG_ANIMATION_MAP = {

        0: {
            strokeLengthPercent: 0,
            strokeOffsetPercent: 0,
            transitionDuration: 0
        },

        1: {
            strokeLengthPercent: 0.7,
            strokeOffsetPercent: -0.3,
            transitionDuration: 750
        },

        2: {
            strokeLengthPercent: 0.7,
            strokeOffsetPercent: -1,
            transitionDuration: 850
        }

    };

    function getCircleSVGStyle(step, size, strokeWidth) {
        var _CIRCLE_SVG_ANIMATION = CIRCLE_SVG_ANIMATION_MAP[step],
            strokeLengthPercent = _CIRCLE_SVG_ANIMATION.strokeLengthPercent,
            strokeOffsetPercent = _CIRCLE_SVG_ANIMATION.strokeOffsetPercent,
            transitionDuration = _CIRCLE_SVG_ANIMATION.transitionDuration;


        var total = Math.PI * (size - strokeWidth);
        var length = total * strokeLengthPercent;

        var result = {
            strokeDashoffset: (total * strokeOffsetPercent).toFixed(2),
            strokeDasharray: length.toFixed(2) + ', ' + total.toFixed(2),
            transitionDuration: transitionDuration + 'ms'
        };

        return result;
    }

    exports.default = _san2.default.defineComponent({

        template: '\n        <div class="{{computedClassName}}" style="{{mainStyle}}">\n            <div class="' + cx.getPartClassName('content') + '">\n                <svg class="' + cx.getPartClassName('svg') + '">\n                    <circle\n                        cx="{{size / 2}}"\n                        cy="{{size / 2}}"\n                        r="{{(size - strokeWidth) / 2}}"\n                        class="' + cx.getPartClassName('path') + '"\n                        style="{{circleStyle}}"\n                        fill="none"\n                        stroke-width="{{strokeWidth}}"\n                        stroke-miterlimit="10" />\n                </svg>\n            </div>\n        </div>\n    ',

        dataTypes: {
            mode: _san.DataTypes.oneOf(['indeterminate', 'determinate']),
            value: _san.DataTypes.number,
            max: _san.DataTypes.number,
            min: _san.DataTypes.number,
            step: _san.DataTypes.number,
            strokeWidth: _san.DataTypes.number
        },

        computed: {
            computedClassName: function computedClassName() {
                var mode = this.data.get('mode');
                return cx(this).addVariants(mode).build();
            },
            mainStyle: function mainStyle() {
                var size = this.data.get('size');
                return (0, _css2.default)({
                    width: size + 'px',
                    height: size + 'px'
                });
            },
            circleStyle: function circleStyle() {

                var mode = this.data.get('mode');
                var size = this.data.get('size');

                if (mode === 'determinate') {
                    var value = this.data.get('value');
                    var min = this.data.get('min');
                    var max = this.data.get('max');

                    var total = max - min;
                    var percent = value - min;

                    var totalLength = Math.PI * size;
                    var percentLength = percent / total * totalLength;

                    return (0, _css2.default)({
                        strokeDasharray: percentLength + ', ' + totalLength
                    });
                }

                var step = this.data.get('step');
                var strokeWidth = this.data.get('strokeWidth');
                return (0, _css2.default)(getCircleSVGStyle(step, size, strokeWidth));
            }
        },

        initData: function initData() {
            return {
                step: 2,
                size: 32,
                strokeWidth: 2,
                mode: 'indeterminate',
                max: 100,
                min: 0,
                value: 0
            };
        },
        inited: function inited() {
            this.animate = this.animate.bind(this);
        },
        attached: function attached() {

            var mode = this.data.get('mode');

            if (mode === 'indeterminate') {
                this.startAnimation();
            }
        },
        startAnimation: function startAnimation() {
            if (!this.animateTimer) {
                this.animateTimer = setTimeout(this.animate, CIRCLE_SVG_ANIMATION_MAP[0].transitionDuration);
            }
        },
        stopAnimation: function stopAnimation() {
            if (this.animateTimer) {
                clearTimeout(this.animateTimer);
                this.animateTimer = null;
            }
        },
        animate: function animate() {

            var step = this.data.get('step');
            var nextStep = (step + 1) % 3;

            this.data.set('step', nextStep);

            this.animateTimer = setTimeout(this.animate, step ? 750 : 250);
        },
        detached: function detached() {
            this.stopAnimation();
        }
    });
});
//# sourceMappingURL=CircularProgress.js.map
