(function (global, factory) {
    if (typeof define === "function" && define.amd) {
        define(['exports'], factory);
    } else if (typeof exports !== "undefined") {
        factory(exports);
    } else {
        var mod = {
            exports: {}
        };
        factory(mod.exports);
        global.highlight = mod.exports;
    }
})(this, function (exports) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });


    var getTextNodes = function getTextNodes(node) {
        var textNodes = [];
        if (node.nodeType === Node.TEXT_NODE) {
            textNodes.push(node);
        } else {
            var children = node.childNodes;
            for (var i = 0, len = children.length; i < len; ++i) {
                textNodes.push.apply(textNodes, getTextNodes(children[i]));
            }
        }
        return textNodes;
    };

    var setSelectionRange = function setSelectionRange(el, start, end) {
        if (document.createRange && window.getSelection) {
            var range = document.createRange();
            range.selectNodeContents(el);
            var textNodes = getTextNodes(el);
            var foundStart = false;
            var charCount = 0;
            var endCharCount = void 0;

            for (var i = 0, textNode; textNode = textNodes[i++];) {
                endCharCount = charCount + textNode.length;
                if (!foundStart && start >= charCount && (start < endCharCount || start === endCharCount && i <= textNodes.length)) {
                    range.setStart(textNode, start - charCount);
                    foundStart = true;
                }
                if (foundStart && end <= endCharCount) {
                    range.setEnd(textNode, end - charCount);
                    break;
                }
                charCount = endCharCount;
            }

            var sel = window.getSelection();
            sel.removeAllRanges();
            sel.addRange(range);
        } else if (document.selection && document.body.createTextRange) {
            var textRange = document.body.createTextRange();
            textRange.moveToElementText(el);
            textRange.collapse(true);
            textRange.moveEnd('character', end);
            textRange.moveStart('character', start);
            textRange.select();
        }
    };

    var makeEditableAndHighlight = function makeEditableAndHighlight(backColor, foreColor) {
        var sel = window.getSelection();
        var range = void 0;
        if (sel.rangeCount && sel.getRangeAt) {
            range = sel.getRangeAt(0);
        }
        document.designMode = 'on';
        if (range) {
            sel.removeAllRanges();
            sel.addRange(range);
        }
        if (!document.execCommand('HiliteColor', false, backColor)) {
            document.execCommand('BackColor', false, backColor);
        }
        document.execCommand('ForeColor', false, foreColor);
        document.designMode = 'off';
        sel.removeAllRanges();
    };

    var makeEditableAndUnhighlight = function makeEditableAndUnhighlight() {
        var sel = window.getSelection();
        var range = void 0;
        if (sel.rangeCount && sel.getRangeAt) {
            range = sel.getRangeAt(0);
        }
        document.designMode = 'on';
        if (range) {
            sel.removeAllRanges();
            sel.addRange(range);
        }
        document.execCommand('removeFormat', false);
        document.designMode = 'off';
        if (range && range.startContainer && range.startContainer instanceof HTMLElement) {
            range.startContainer.normalize();
        }
        sel.removeAllRanges();
    };

    var doHighlight = function doHighlight(backColor, foreColor) {
        var range = void 0;
        if (window.getSelection) {
            try {
                if (!document.execCommand('BackColor', false, backColor)) {
                    makeEditableAndHighlight(backColor, foreColor);
                }
            } catch (ex) {
                makeEditableAndHighlight(backColor, foreColor);
            }
        } else if (document.selection && document.selection.createRange) {
            range = document.selection.createRange();
            range.execCommand('BackColor', false, backColor);
        }
    };

    var doUnhighlight = function doUnhighlight() {
        var range = void 0;
        if (window.getSelection) {
            try {
                if (!document.execCommand('BackColor', false)) {
                    makeEditableAndUnhighlight();
                }
            } catch (ex) {
                makeEditableAndUnhighlight();
            }
        } else if (document.selection && document.selection.createRange) {
            range = document.selection.createRange();
            range.execCommand('removeFormat', false);
        }
    };

    var highlightFunc = function highlightFunc(el, keyword, input, backColor, foreColor) {
        if (!el || !el.innerText || !keyword) {
            return;
        }
        var text = el.innerText.toLowerCase();
        keyword = keyword.toLowerCase();

        var len = keyword.length;
        var start = text.indexOf(keyword);
        while (start > -1) {
            var end = start + len;
            setSelectionRange(el, start, end);
            doHighlight(backColor, foreColor);
            start = text.indexOf(keyword, start + 1);
        }
        input && input.focus();
    };

    var unhighlightFunc = function unhighlightFunc(el, input) {
        if (!el || !el.innerText) {
            return;
        }
        setSelectionRange(el, 0, el.innerText.length);
        doUnhighlight();
        input && input.focus();
    };

    var Highlight = exports.Highlight = {
        highlight: highlightFunc,
        unhighlight: unhighlightFunc
    };
});
//# sourceMappingURL=highlight.js.map
