import { type ClientLayout } from '../hooks/useLayout';
import { type WorkspaceStore } from '../store/store';
import type { SecurityScheme } from '@scalar/oas-utils/entities/spec';
import { type Workspace } from '@scalar/oas-utils/entities/workspace';
import type { Path, PathValue } from '@scalar/object-utils/nested';
import type { OpenAPI, ReferenceConfiguration, SpecConfiguration } from '@scalar/types/legacy';
import { type Component } from 'vue';
import type { Router } from 'vue-router';
/** Configuration options for the Scalar API client */
export type ClientConfiguration = {
    proxyUrl?: ReferenceConfiguration['proxyUrl'];
    themeId?: ReferenceConfiguration['theme'];
} & Pick<ReferenceConfiguration, 'spec' | 'showSidebar' | 'servers' | 'searchHotKey' | 'authentication' | 'baseServerURL' | 'hideClientButton' | '_integration'>;
export type OpenClientPayload = ({
    path: string;
    method: OpenAPI.HttpMethod | Lowercase<OpenAPI.HttpMethod>;
    requestUid?: never;
} | {
    path?: never;
    method?: never;
    requestUid: string;
}) & {
    _source?: 'api-reference' | 'gitbook';
};
export type CreateApiClientParams = {
    /** Element to mount the references to */
    el: HTMLElement | null;
    /** Main vue app component to create the vue app */
    appComponent: Component;
    /** Configuration object for API client */
    configuration?: ClientConfiguration;
    /** Read only version of the client app */
    isReadOnly?: boolean;
    /** Persist the workspace to localStoragfe */
    persistData?: boolean;
    /**
     * Will attempt to mount the references immediately
     * For SSR this may need to be blocked and done client side
     */
    mountOnInitialize?: boolean;
    /** Instance of a vue router */
    router: Router;
    /** In case the store has been instantiated beforehand */
    store?: WorkspaceStore | undefined;
    /**
     * The layout of the client
     * @see {@link ClientLayout}
     */
    layout?: ClientLayout;
};
/**
 * ApiClient type
 *
 * We need to do this due to some typescript type propogation errors
 * This is pretty much add properties as they are needed
 */
export type ApiClient = Omit<Awaited<ReturnType<typeof createApiClient>>, 'app' | 'store'> & {
    /** Add properties as they are needed, see above */
    app: {
        unmount: () => void;
    };
    /**
     * The main workspace store from the client
     * These refs don't wanna play nice with typescript, if we need them we can de-reference them
     */
    store: Omit<WorkspaceStore, 'router' | 'events' | 'sidebarWidth' | 'proxyUrl' | 'requestHistory'>;
};
/**
 * Sync method to create the api client vue app and store
 *
 * This method will NOT import the spec, just create the modal so you must use update/updateConfig before opening
 */
export declare const createApiClient: ({ el, appComponent, configuration, isReadOnly, store: _store, persistData, mountOnInitialize, layout, router, }: CreateApiClientParams) => {
    /** The vue app instance for the modal, be careful with this */
    app: import("vue").App<Element>;
    updateSpec: (spec: SpecConfiguration) => Promise<void>;
    /**
     * Update the API client config
     *
     * Deletes the current store before importing again for now, in the future will Diff
     */
    updateConfig(newConfig: ClientConfiguration, mergeConfigs?: boolean): void;
    /** Update the currently selected server via URL */
    updateServer: (serverUrl: string) => void;
    /** Update the currently selected server via URL */
    onUpdateServer: (callback: (url: string) => void) => void;
    /**
     * Update the auth values, we currently don't change the auth selection
     */
    updateAuth: <P extends Path<SecurityScheme>>({ nameKey, propertyKey, value, }: {
        nameKey: string;
        propertyKey: P;
        value: NonNullable<PathValue<SecurityScheme, P>>;
    }) => void;
    /** Route to a method + path */
    route: (payload?: OpenClientPayload) => void;
    /** Open the API client modal and optionally route to a request */
    open: (payload?: OpenClientPayload) => void;
    /** Mount the references to a given element */
    mount: (mountingEl?: HTMLElement | null) => void;
    /** State for controlling the modal */
    modalState: {
        open: boolean;
        show: () => void;
        hide: () => void;
    };
    store: {
        workspaces: Record<string, {
            uid: string & import("zod").BRAND<"workspace">;
            name: string;
            description: string;
            collections: (string & import("zod").BRAND<"collection">)[];
            environments: Record<string, string>;
            activeEnvironmentId: string;
            cookies: (string & import("zod").BRAND<"cookie">)[];
            themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
            selectedHttpClient: {
                targetKey: string;
                clientKey: string;
            };
            hotKeyConfig?: {
                modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                    event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                    modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                }>> | undefined;
            } | undefined;
            proxyUrl?: string | undefined;
        }>;
        collections: Record<string, {
            type: "collection";
            uid: string & import("zod").BRAND<"collection">;
            children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
            openapi: string;
            security: Record<string, string[]>[];
            'x-scalar-icon': string;
            securitySchemes: string[];
            selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
            servers: (string & import("zod").BRAND<"server">)[];
            requests: (string & import("zod").BRAND<"operation">)[];
            tags: (string & import("zod").BRAND<"tag">)[];
            watchMode: boolean;
            watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
            externalDocs?: {
                url: string;
                description?: string | undefined;
            } | undefined;
            jsonSchemaDialect?: string | undefined;
            info?: {
                title: string;
                version: string;
                description?: string | undefined;
                summary?: string | undefined;
                termsOfService?: string | undefined;
                contact?: {
                    name?: string | undefined;
                    url?: string | undefined;
                    email?: string | undefined;
                } | undefined;
                license?: {
                    name: string;
                    identifier?: string | undefined;
                    url?: string | undefined;
                } | undefined;
            } | undefined;
            components?: Record<string, unknown> | undefined;
            webhooks?: Record<string, unknown> | undefined;
            'x-scalar-active-environment'?: string | undefined;
            'x-scalar-environments'?: Record<string, {
                variables: Record<string, string | {
                    default: string;
                    description?: string | undefined;
                }>;
                description?: string | undefined;
                color?: string | undefined;
            }> | undefined;
            'x-scalar-secrets'?: Record<string, {
                example?: string | undefined;
                description?: string | undefined;
            }> | undefined;
            selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
            documentUrl?: string | undefined;
            integration?: string | null | undefined;
        }>;
        tags: Record<string, {
            type: "tag";
            uid: string & import("zod").BRAND<"tag">;
            name: string;
            children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
            description?: string | undefined;
            externalDocs?: {
                url: string;
                description?: string | undefined;
            } | undefined;
            'x-scalar-children'?: {
                tagName: string;
            }[] | undefined;
            'x-internal'?: boolean | undefined;
            'x-scalar-ignore'?: boolean | undefined;
        }>;
        cookies: Record<string, {
            value: string;
            uid: string & import("zod").BRAND<"cookie">;
            name: string;
            path?: string | undefined;
            domain?: string | undefined;
        }>;
        environments: Record<string, {
            value: string;
            uid: string & import("zod").BRAND<"environment">;
            name: string;
            color: string;
            isDefault?: boolean | undefined;
        }>;
        requestExamples: Record<string, {
            type: "requestExample";
            uid: string & import("zod").BRAND<"example">;
            name: string;
            parameters: {
                path: {
                    value: string;
                    key: string;
                    enabled: boolean;
                    minimum?: number | undefined;
                    type?: string | string[] | undefined;
                    maximum?: number | undefined;
                    default?: any;
                    required?: boolean | undefined;
                    description?: string | undefined;
                    enum?: string[] | undefined;
                    examples?: string[] | undefined;
                    format?: string | undefined;
                    file?: any;
                    nullable?: boolean | undefined;
                }[];
                cookies: {
                    value: string;
                    key: string;
                    enabled: boolean;
                    minimum?: number | undefined;
                    type?: string | string[] | undefined;
                    maximum?: number | undefined;
                    default?: any;
                    required?: boolean | undefined;
                    description?: string | undefined;
                    enum?: string[] | undefined;
                    examples?: string[] | undefined;
                    format?: string | undefined;
                    file?: any;
                    nullable?: boolean | undefined;
                }[];
                query: {
                    value: string;
                    key: string;
                    enabled: boolean;
                    minimum?: number | undefined;
                    type?: string | string[] | undefined;
                    maximum?: number | undefined;
                    default?: any;
                    required?: boolean | undefined;
                    description?: string | undefined;
                    enum?: string[] | undefined;
                    examples?: string[] | undefined;
                    format?: string | undefined;
                    file?: any;
                    nullable?: boolean | undefined;
                }[];
                headers: {
                    value: string;
                    key: string;
                    enabled: boolean;
                    minimum?: number | undefined;
                    type?: string | string[] | undefined;
                    maximum?: number | undefined;
                    default?: any;
                    required?: boolean | undefined;
                    description?: string | undefined;
                    enum?: string[] | undefined;
                    examples?: string[] | undefined;
                    format?: string | undefined;
                    file?: any;
                    nullable?: boolean | undefined;
                }[];
            };
            body: {
                activeBody: "binary" | "formData" | "raw";
                binary?: Blob | undefined;
                formData?: {
                    value: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    encoding: "form-data" | "urlencoded";
                } | undefined;
                raw?: {
                    value: string;
                    encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                } | undefined;
            };
            requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
            serverVariables?: Record<string, string[]> | undefined;
        }>;
        requests: Record<string, {
            path: string;
            type: "request";
            uid: string & import("zod").BRAND<"operation">;
            selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
            selectedServerUid: (string & import("zod").BRAND<"server">) | null;
            servers: (string & import("zod").BRAND<"server">)[];
            examples: (string & import("zod").BRAND<"example">)[];
            method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
            description?: string | undefined;
            summary?: string | undefined;
            externalDocs?: {
                url: string;
                description?: string | undefined;
            } | undefined;
            'x-internal'?: boolean | undefined;
            'x-scalar-ignore'?: boolean | undefined;
            security?: Record<string, string[]>[] | undefined;
            tags?: string[] | undefined;
            deprecated?: boolean | undefined;
            operationId?: string | undefined;
            parameters?: {
                required: boolean;
                name: string;
                in: "path" | "cookie" | "query" | "header";
                deprecated: boolean;
                example?: unknown;
                description?: string | undefined;
                schema?: unknown;
                content?: unknown;
                style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                examples?: Record<string, {
                    value?: unknown;
                    summary?: string | undefined;
                }> | undefined;
            }[] | undefined;
            requestBody?: any;
            responses?: Record<string, any> | undefined;
        }>;
        servers: Record<string, {
            uid: string & import("zod").BRAND<"server">;
            url: string;
            description?: string | undefined;
            variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                enum?: [string, ...string[]];
                value?: string;
            }> | undefined;
        }>;
        securitySchemes: Record<string, {
            type: "apiKey";
            value: string;
            uid: string & import("zod").BRAND<"securityScheme">;
            name: string;
            nameKey: string;
            in: "cookie" | "query" | "header";
            description?: string | undefined;
        } | {
            type: "http";
            uid: string & import("zod").BRAND<"securityScheme">;
            nameKey: string;
            scheme: "basic" | "bearer";
            bearerFormat: string;
            username: string;
            password: string;
            token: string;
            description?: string | undefined;
        } | {
            type: "openIdConnect";
            uid: string & import("zod").BRAND<"securityScheme">;
            nameKey: string;
            openIdConnectUrl: string;
            description?: string | undefined;
        } | {
            type: "oauth2";
            uid: string & import("zod").BRAND<"securityScheme">;
            nameKey: string;
            flows: {
                password?: {
                    type: "password";
                    username: string;
                    password: string;
                    token: string;
                    refreshUrl: string;
                    scopes: Record<string, string>;
                    selectedScopes: string[];
                    "x-scalar-client-id": string;
                    tokenUrl: string;
                    clientSecret: string;
                } | undefined;
                implicit?: {
                    type: "implicit";
                    token: string;
                    refreshUrl: string;
                    scopes: Record<string, string>;
                    selectedScopes: string[];
                    "x-scalar-client-id": string;
                    authorizationUrl: string;
                    "x-scalar-redirect-uri": string;
                } | undefined;
                clientCredentials?: {
                    type: "clientCredentials";
                    token: string;
                    refreshUrl: string;
                    scopes: Record<string, string>;
                    selectedScopes: string[];
                    "x-scalar-client-id": string;
                    tokenUrl: string;
                    clientSecret: string;
                } | undefined;
                authorizationCode?: {
                    type: "authorizationCode";
                    token: string;
                    refreshUrl: string;
                    scopes: Record<string, string>;
                    selectedScopes: string[];
                    "x-scalar-client-id": string;
                    authorizationUrl: string;
                    "x-scalar-redirect-uri": string;
                    tokenUrl: string;
                    clientSecret: string;
                    "x-usePkce": "SHA-256" | "plain" | "no";
                } | undefined;
            };
            description?: string | undefined;
        }>;
        modalState: {
            open: boolean;
            show: () => void;
            hide: () => void;
        };
        events: {
            executeRequest: import("./event-bus").EventBus<{
                requestUid?: string;
            }>;
            cancelRequest: import("./event-bus").EventBus<any>;
            requestStatus: import("./event-bus").EventBus<import("./send-request").RequestStatus>;
            commandPalette: import("./event-bus").EventBus<import("../components/CommandPalette/TheCommandPalette.vue").CommandPaletteEvent>;
            hotKeys: import("./event-bus").EventBus<Partial<Record<"addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar", KeyboardEvent>>>;
        };
        sidebarWidth: import("vue").Ref<string, string>;
        setSidebarWidth: (width: string) => void;
        proxyUrl: string | undefined;
        hideClientButton: boolean;
        showSidebar: boolean;
        integration: "elysiajs" | "fastify" | "vue" | "html" | "adonisjs" | "docusaurus" | "dotnet" | "express" | "fastapi" | "go" | "hono" | "laravel" | "litestar" | "nestjs" | "nextjs" | "nitro" | "nuxt" | "platformatic" | "react" | "rust" | null | undefined;
        importSpecFile: (_spec: string | Record<string, any>, workspaceUid: string, options?: Pick<{
            type?: "collection" | undefined;
            uid?: string | undefined;
            externalDocs?: {
                description?: string | undefined;
                url?: string | undefined;
            } | undefined;
            children?: string[] | undefined;
            openapi?: string | undefined;
            jsonSchemaDialect?: string | undefined;
            info?: {
                description?: string | undefined;
                title?: string | undefined;
                summary?: string | undefined;
                termsOfService?: string | undefined;
                contact?: {
                    name?: string | undefined;
                    url?: string | undefined;
                    email?: string | undefined;
                } | undefined;
                license?: {
                    name?: string | undefined;
                    identifier?: string | undefined;
                    url?: string | undefined;
                } | undefined;
                version?: string | undefined;
            } | undefined;
            security?: Record<string, string[] | undefined>[] | undefined;
            components?: Record<string, unknown> | undefined;
            webhooks?: Record<string, unknown> | undefined;
            'x-scalar-icon'?: string | undefined;
            'x-scalar-active-environment'?: string | undefined;
            'x-scalar-environments'?: Record<string, {
                variables: Record<string, string | {
                    default?: string | undefined;
                    description?: string | undefined;
                }>;
                description?: string | undefined;
                color?: string | undefined;
            }> | undefined;
            'x-scalar-secrets'?: Record<string, {
                example?: string | undefined;
                description?: string | undefined;
            }> | undefined;
            securitySchemes?: string[] | undefined;
            selectedSecuritySchemeUids?: (string | string[])[] | undefined;
            selectedServerUid?: string | undefined;
            servers?: string[] | undefined;
            requests?: string[] | undefined;
            tags?: string[] | undefined;
            documentUrl?: string | undefined;
            watchMode?: boolean | undefined;
            integration?: string | null | undefined;
            watchModeStatus?: "IDLE" | "WATCHING" | "ERROR" | undefined;
        }, "watchMode" | "documentUrl"> & Pick<ReferenceConfiguration, "servers" | "authentication" | "baseServerURL"> & {
            setCollectionSecurity?: boolean;
            shouldLoad?: boolean;
        } & Pick<ReferenceConfiguration, "servers">) => Promise<{
            error: false;
            collection: import("@scalar/oas-utils/entities/spec").Collection;
            requests: import("@scalar/oas-utils/entities/spec").Request[];
            schema: import("@scalar/openapi-types").OpenAPIV3_1.Document;
            examples: import("@scalar/oas-utils/entities/spec").RequestExample[];
            servers: import("@scalar/oas-utils/entities/spec").Server[];
            tags: import("@scalar/oas-utils/entities/spec").Tag[];
            securitySchemes: SecurityScheme[];
        } | undefined>;
        importSpecFromUrl: (url: string, workspaceUid: string, { proxyUrl, ...options }?: Omit<Pick<{
            type?: "collection" | undefined;
            uid?: string | undefined;
            externalDocs?: {
                description?: string | undefined;
                url?: string | undefined;
            } | undefined;
            children?: string[] | undefined;
            openapi?: string | undefined;
            jsonSchemaDialect?: string | undefined;
            info?: {
                description?: string | undefined;
                title?: string | undefined;
                summary?: string | undefined;
                termsOfService?: string | undefined;
                contact?: {
                    name?: string | undefined;
                    url?: string | undefined;
                    email?: string | undefined;
                } | undefined;
                license?: {
                    name?: string | undefined;
                    identifier?: string | undefined;
                    url?: string | undefined;
                } | undefined;
                version?: string | undefined;
            } | undefined;
            security?: Record<string, string[] | undefined>[] | undefined;
            components?: Record<string, unknown> | undefined;
            webhooks?: Record<string, unknown> | undefined;
            'x-scalar-icon'?: string | undefined;
            'x-scalar-active-environment'?: string | undefined;
            'x-scalar-environments'?: Record<string, {
                variables: Record<string, string | {
                    default?: string | undefined;
                    description?: string | undefined;
                }>;
                description?: string | undefined;
                color?: string | undefined;
            }> | undefined;
            'x-scalar-secrets'?: Record<string, {
                example?: string | undefined;
                description?: string | undefined;
            }> | undefined;
            securitySchemes?: string[] | undefined;
            selectedSecuritySchemeUids?: (string | string[])[] | undefined;
            selectedServerUid?: string | undefined;
            servers?: string[] | undefined;
            requests?: string[] | undefined;
            tags?: string[] | undefined;
            documentUrl?: string | undefined;
            watchMode?: boolean | undefined;
            integration?: string | null | undefined;
            watchModeStatus?: "IDLE" | "WATCHING" | "ERROR" | undefined;
        }, "watchMode" | "documentUrl"> & Pick<ReferenceConfiguration, "servers" | "authentication" | "baseServerURL"> & {
            setCollectionSecurity?: boolean;
            shouldLoad?: boolean;
        } & Pick<ReferenceConfiguration, "servers">, "documentUrl"> & Pick<ReferenceConfiguration, "proxyUrl">) => Promise<import("./errors").ErrorResponse<Awaited<ReturnType<(_spec: string | Record<string, any>, workspaceUid: string, options?: Pick<{
            type?: "collection" | undefined;
            uid?: string | undefined;
            externalDocs?: {
                description?: string | undefined;
                url?: string | undefined;
            } | undefined;
            children?: string[] | undefined;
            openapi?: string | undefined;
            jsonSchemaDialect?: string | undefined;
            info?: {
                description?: string | undefined;
                title?: string | undefined;
                summary?: string | undefined;
                termsOfService?: string | undefined;
                contact?: {
                    name?: string | undefined;
                    url?: string | undefined;
                    email?: string | undefined;
                } | undefined;
                license?: {
                    name?: string | undefined;
                    identifier?: string | undefined;
                    url?: string | undefined;
                } | undefined;
                version?: string | undefined;
            } | undefined;
            security?: Record<string, string[] | undefined>[] | undefined;
            components?: Record<string, unknown> | undefined;
            webhooks?: Record<string, unknown> | undefined;
            'x-scalar-icon'?: string | undefined;
            'x-scalar-active-environment'?: string | undefined;
            'x-scalar-environments'?: Record<string, {
                variables: Record<string, string | {
                    default?: string | undefined;
                    description?: string | undefined;
                }>;
                description?: string | undefined;
                color?: string | undefined;
            }> | undefined;
            'x-scalar-secrets'?: Record<string, {
                example?: string | undefined;
                description?: string | undefined;
            }> | undefined;
            securitySchemes?: string[] | undefined;
            selectedSecuritySchemeUids?: (string | string[])[] | undefined;
            selectedServerUid?: string | undefined;
            servers?: string[] | undefined;
            requests?: string[] | undefined;
            tags?: string[] | undefined;
            documentUrl?: string | undefined;
            watchMode?: boolean | undefined;
            integration?: string | null | undefined;
            watchModeStatus?: "IDLE" | "WATCHING" | "ERROR" | undefined;
        }, "watchMode" | "documentUrl"> & Pick<ReferenceConfiguration, "servers" | "authentication" | "baseServerURL"> & {
            setCollectionSecurity?: boolean;
            shouldLoad?: boolean;
        } & Pick<ReferenceConfiguration, "servers">) => Promise<{
            error: false;
            collection: import("@scalar/oas-utils/entities/spec").Collection;
            requests: import("@scalar/oas-utils/entities/spec").Request[];
            schema: import("@scalar/openapi-types").OpenAPIV3_1.Document;
            examples: import("@scalar/oas-utils/entities/spec").RequestExample[];
            servers: import("@scalar/oas-utils/entities/spec").Server[];
            tags: import("@scalar/oas-utils/entities/spec").Tag[];
            securitySchemes: SecurityScheme[];
        } | undefined>>>>>;
        cookieMutators: {
            add: (item: {
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }) => void;
            delete: (uid: (string & import("zod").BRAND<"cookie">) | null | undefined) => void;
            set: (item: {
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }) => void;
            edit: <P extends "name" | "uid" | "path" | "value" | "domain">(uid: (string & import("zod").BRAND<"cookie">) | null | undefined, path: P, value: P extends "name" | "uid" | "path" | "value" | "domain" ? {
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "name" | "uid" | "path" | "value" | "domain" ? R extends Path<{
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }[K]> ? PathValue<{
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "name" | "uid" | "path" | "value" | "domain">(uid: string & import("zod").BRAND<"cookie">, path: P, value: P extends "name" | "uid" | "path" | "value" | "domain" ? {
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "name" | "uid" | "path" | "value" | "domain" ? R extends Path<{
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }[K]> ? PathValue<{
                value: string;
                uid: string & import("zod").BRAND<"cookie">;
                name: string;
                path?: string | undefined;
                domain?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"cookie">) => void;
            redo: (uid: string & import("zod").BRAND<"cookie">) => void;
            reset: () => void;
        };
        collectionMutators: {
            rawAdd: (item: {
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }) => void;
            add: (payload: import("@scalar/oas-utils/entities/spec").CollectionPayload, workspaceUid: Workspace["uid"]) => {
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            };
            delete: (collection: import("@scalar/oas-utils/entities/spec").Collection, workspace: Workspace) => void;
            addEnvironment: (environmentName: string, environment: import("@scalar/oas-utils/entities/spec").XScalarEnvironment, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
            removeEnvironment: (environmentName: string, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
            set: (item: {
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }) => void;
            edit: <P extends "servers" | "tags" | "security" | "externalDocs" | "type" | "uid" | "requests" | "x-scalar-environments" | "children" | "openapi" | "x-scalar-icon" | "securitySchemes" | "selectedSecuritySchemeUids" | "watchMode" | "watchModeStatus" | "jsonSchemaDialect" | "info" | "components" | "webhooks" | "x-scalar-active-environment" | "x-scalar-secrets" | "selectedServerUid" | "documentUrl" | "integration" | `servers.${number}` | `tags.${number}` | `security.${number}` | `security.${number}.${string}` | "externalDocs.description" | "externalDocs.url" | `requests.${number}` | `x-scalar-environments.${string}` | `children.${number}` | `securitySchemes.${number}` | `selectedSecuritySchemeUids.${number}` | `selectedSecuritySchemeUids.${number}.${number}` | "info.summary" | "info.description" | "info.title" | "info.version" | "info.termsOfService" | "info.contact" | "info.license" | "info.contact.name" | "info.contact.url" | "info.contact.email" | "info.license.name" | "info.license.url" | "info.license.identifier" | `components.${string}` | `webhooks.${string}` | `x-scalar-secrets.${string}`>(uid: (string & import("zod").BRAND<"collection">) | null | undefined, path: P, value: P extends "servers" | "tags" | "security" | "externalDocs" | "type" | "uid" | "requests" | "x-scalar-environments" | "children" | "openapi" | "x-scalar-icon" | "securitySchemes" | "selectedSecuritySchemeUids" | "watchMode" | "watchModeStatus" | "jsonSchemaDialect" | "info" | "components" | "webhooks" | "x-scalar-active-environment" | "x-scalar-secrets" | "selectedServerUid" | "documentUrl" | "integration" ? {
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "servers" | "tags" | "security" | "externalDocs" | "type" | "uid" | "requests" | "x-scalar-environments" | "children" | "openapi" | "x-scalar-icon" | "securitySchemes" | "selectedSecuritySchemeUids" | "watchMode" | "watchModeStatus" | "jsonSchemaDialect" | "info" | "components" | "webhooks" | "x-scalar-active-environment" | "x-scalar-secrets" | "selectedServerUid" | "documentUrl" | "integration" ? R extends Path<{
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }[K]> ? PathValue<{
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "servers" | "tags" | "security" | "externalDocs" | "type" | "uid" | "requests" | "x-scalar-environments" | "children" | "openapi" | "x-scalar-icon" | "securitySchemes" | "selectedSecuritySchemeUids" | "watchMode" | "watchModeStatus" | "jsonSchemaDialect" | "info" | "components" | "webhooks" | "x-scalar-active-environment" | "x-scalar-secrets" | "selectedServerUid" | "documentUrl" | "integration" | `servers.${number}` | `tags.${number}` | `security.${number}` | `security.${number}.${string}` | "externalDocs.description" | "externalDocs.url" | `requests.${number}` | `x-scalar-environments.${string}` | `children.${number}` | `securitySchemes.${number}` | `selectedSecuritySchemeUids.${number}` | `selectedSecuritySchemeUids.${number}.${number}` | "info.summary" | "info.description" | "info.title" | "info.version" | "info.termsOfService" | "info.contact" | "info.license" | "info.contact.name" | "info.contact.url" | "info.contact.email" | "info.license.name" | "info.license.url" | "info.license.identifier" | `components.${string}` | `webhooks.${string}` | `x-scalar-secrets.${string}`>(uid: string & import("zod").BRAND<"collection">, path: P, value: P extends "servers" | "tags" | "security" | "externalDocs" | "type" | "uid" | "requests" | "x-scalar-environments" | "children" | "openapi" | "x-scalar-icon" | "securitySchemes" | "selectedSecuritySchemeUids" | "watchMode" | "watchModeStatus" | "jsonSchemaDialect" | "info" | "components" | "webhooks" | "x-scalar-active-environment" | "x-scalar-secrets" | "selectedServerUid" | "documentUrl" | "integration" ? {
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "servers" | "tags" | "security" | "externalDocs" | "type" | "uid" | "requests" | "x-scalar-environments" | "children" | "openapi" | "x-scalar-icon" | "securitySchemes" | "selectedSecuritySchemeUids" | "watchMode" | "watchModeStatus" | "jsonSchemaDialect" | "info" | "components" | "webhooks" | "x-scalar-active-environment" | "x-scalar-secrets" | "selectedServerUid" | "documentUrl" | "integration" ? R extends Path<{
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }[K]> ? PathValue<{
                type: "collection";
                uid: string & import("zod").BRAND<"collection">;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                openapi: string;
                security: Record<string, string[]>[];
                'x-scalar-icon': string;
                securitySchemes: string[];
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                servers: (string & import("zod").BRAND<"server">)[];
                requests: (string & import("zod").BRAND<"operation">)[];
                tags: (string & import("zod").BRAND<"tag">)[];
                watchMode: boolean;
                watchModeStatus: "IDLE" | "WATCHING" | "ERROR";
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                jsonSchemaDialect?: string | undefined;
                info?: {
                    title: string;
                    version: string;
                    description?: string | undefined;
                    summary?: string | undefined;
                    termsOfService?: string | undefined;
                    contact?: {
                        name?: string | undefined;
                        url?: string | undefined;
                        email?: string | undefined;
                    } | undefined;
                    license?: {
                        name: string;
                        identifier?: string | undefined;
                        url?: string | undefined;
                    } | undefined;
                } | undefined;
                components?: Record<string, unknown> | undefined;
                webhooks?: Record<string, unknown> | undefined;
                'x-scalar-active-environment'?: string | undefined;
                'x-scalar-environments'?: Record<string, {
                    variables: Record<string, string | {
                        default: string;
                        description?: string | undefined;
                    }>;
                    description?: string | undefined;
                    color?: string | undefined;
                }> | undefined;
                'x-scalar-secrets'?: Record<string, {
                    example?: string | undefined;
                    description?: string | undefined;
                }> | undefined;
                selectedServerUid?: (string & import("zod").BRAND<"server">) | undefined;
                documentUrl?: string | undefined;
                integration?: string | null | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"collection">) => void;
            redo: (uid: string & import("zod").BRAND<"collection">) => void;
            reset: () => void;
        };
        environmentMutators: {
            delete: (uid: import("@scalar/oas-utils/entities/environment").Environment["uid"]) => void;
            add: (item: {
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }) => void;
            set: (item: {
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }) => void;
            edit: <P extends "name" | "uid" | "color" | "value" | "isDefault">(uid: (string & import("zod").BRAND<"environment">) | null | undefined, path: P, value: P extends "name" | "uid" | "color" | "value" | "isDefault" ? {
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "name" | "uid" | "color" | "value" | "isDefault" ? R extends Path<{
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }[K]> ? PathValue<{
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "name" | "uid" | "color" | "value" | "isDefault">(uid: string & import("zod").BRAND<"environment">, path: P, value: P extends "name" | "uid" | "color" | "value" | "isDefault" ? {
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "name" | "uid" | "color" | "value" | "isDefault" ? R extends Path<{
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }[K]> ? PathValue<{
                value: string;
                uid: string & import("zod").BRAND<"environment">;
                name: string;
                color: string;
                isDefault?: boolean | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"environment">) => void;
            redo: (uid: string & import("zod").BRAND<"environment">) => void;
            reset: () => void;
        };
        requestMutators: {
            rawAdd: (item: {
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }) => void;
            add: (payload: import("@scalar/oas-utils/entities/spec").RequestPayload, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void | {
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            };
            delete: (request: import("@scalar/oas-utils/entities/spec").Request, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
            set: (item: {
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }) => void;
            edit: <P extends "servers" | "tags" | "summary" | "description" | "operationId" | "security" | "requestBody" | "parameters" | "externalDocs" | "deprecated" | "responses" | "x-internal" | "x-scalar-ignore" | "type" | "uid" | "selectedSecuritySchemeUids" | "selectedServerUid" | `servers.${number}` | `tags.${number}` | `security.${number}` | `security.${number}.${string}` | "externalDocs.description" | "externalDocs.url" | `selectedSecuritySchemeUids.${number}` | `selectedSecuritySchemeUids.${number}.${number}` | "path" | "method" | "examples" | `requestBody.${string}` | `parameters.${number}` | `parameters.${number}.description` | `parameters.${number}.deprecated` | `parameters.${number}.name` | `parameters.${number}.in` | `parameters.${number}.example` | `parameters.${number}.style` | `parameters.${number}.required` | `parameters.${number}.examples` | `parameters.${number}.content` | `parameters.${number}.schema` | `parameters.${number}.examples.${string}` | `responses.${string}` | `examples.${number}`>(uid: (string & import("zod").BRAND<"operation">) | null | undefined, path: P, value: P extends "servers" | "tags" | "summary" | "description" | "operationId" | "security" | "requestBody" | "parameters" | "externalDocs" | "deprecated" | "responses" | "x-internal" | "x-scalar-ignore" | "type" | "uid" | "selectedSecuritySchemeUids" | "selectedServerUid" | "path" | "method" | "examples" ? {
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "servers" | "tags" | "summary" | "description" | "operationId" | "security" | "requestBody" | "parameters" | "externalDocs" | "deprecated" | "responses" | "x-internal" | "x-scalar-ignore" | "type" | "uid" | "selectedSecuritySchemeUids" | "selectedServerUid" | "path" | "method" | "examples" ? R extends Path<{
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }[K]> ? PathValue<{
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "servers" | "tags" | "summary" | "description" | "operationId" | "security" | "requestBody" | "parameters" | "externalDocs" | "deprecated" | "responses" | "x-internal" | "x-scalar-ignore" | "type" | "uid" | "selectedSecuritySchemeUids" | "selectedServerUid" | `servers.${number}` | `tags.${number}` | `security.${number}` | `security.${number}.${string}` | "externalDocs.description" | "externalDocs.url" | `selectedSecuritySchemeUids.${number}` | `selectedSecuritySchemeUids.${number}.${number}` | "path" | "method" | "examples" | `requestBody.${string}` | `parameters.${number}` | `parameters.${number}.description` | `parameters.${number}.deprecated` | `parameters.${number}.name` | `parameters.${number}.in` | `parameters.${number}.example` | `parameters.${number}.style` | `parameters.${number}.required` | `parameters.${number}.examples` | `parameters.${number}.content` | `parameters.${number}.schema` | `parameters.${number}.examples.${string}` | `responses.${string}` | `examples.${number}`>(uid: string & import("zod").BRAND<"operation">, path: P, value: P extends "servers" | "tags" | "summary" | "description" | "operationId" | "security" | "requestBody" | "parameters" | "externalDocs" | "deprecated" | "responses" | "x-internal" | "x-scalar-ignore" | "type" | "uid" | "selectedSecuritySchemeUids" | "selectedServerUid" | "path" | "method" | "examples" ? {
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "servers" | "tags" | "summary" | "description" | "operationId" | "security" | "requestBody" | "parameters" | "externalDocs" | "deprecated" | "responses" | "x-internal" | "x-scalar-ignore" | "type" | "uid" | "selectedSecuritySchemeUids" | "selectedServerUid" | "path" | "method" | "examples" ? R extends Path<{
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }[K]> ? PathValue<{
                path: string;
                type: "request";
                uid: string & import("zod").BRAND<"operation">;
                selectedSecuritySchemeUids: ((string & import("zod").BRAND<"securityScheme">) | (string & import("zod").BRAND<"securityScheme">)[])[];
                selectedServerUid: (string & import("zod").BRAND<"server">) | null;
                servers: (string & import("zod").BRAND<"server">)[];
                examples: (string & import("zod").BRAND<"example">)[];
                method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
                description?: string | undefined;
                summary?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
                security?: Record<string, string[]>[] | undefined;
                tags?: string[] | undefined;
                deprecated?: boolean | undefined;
                operationId?: string | undefined;
                parameters?: {
                    required: boolean;
                    name: string;
                    in: "path" | "cookie" | "query" | "header";
                    deprecated: boolean;
                    example?: unknown;
                    description?: string | undefined;
                    schema?: unknown;
                    content?: unknown;
                    style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
                    examples?: Record<string, {
                        value?: unknown;
                        summary?: string | undefined;
                    }> | undefined;
                }[] | undefined;
                requestBody?: any;
                responses?: Record<string, any> | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"operation">) => void;
            redo: (uid: string & import("zod").BRAND<"operation">) => void;
            reset: () => void;
        };
        findRequestParents: (r: import("@scalar/oas-utils/entities/spec").Request) => string[];
        requestExampleMutators: {
            rawAdd: (item: {
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }) => void;
            add: (request: import("@scalar/oas-utils/entities/spec").Request, _name?: string) => {
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            };
            delete: (requestExample: import("@scalar/oas-utils/entities/spec").RequestExample) => void;
            set: (item: {
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }) => void;
            edit: <P extends "parameters" | "type" | "name" | "uid" | "body" | "requestUid" | "serverVariables" | "body.formData.value" | "parameters.cookies" | `parameters.cookies.${number}` | "parameters.path" | "parameters.headers" | "parameters.query" | `parameters.cookies.${number}.description` | `parameters.cookies.${number}.required` | `parameters.cookies.${number}.examples` | `parameters.cookies.${number}.type` | `parameters.cookies.${number}.default` | `parameters.cookies.${number}.key` | `parameters.cookies.${number}.file` | `parameters.cookies.${number}.value` | `parameters.cookies.${number}.enum` | `parameters.cookies.${number}.nullable` | `parameters.cookies.${number}.format` | `parameters.cookies.${number}.maximum` | `parameters.cookies.${number}.minimum` | `parameters.cookies.${number}.examples.${number}` | `parameters.cookies.${number}.enum.${number}` | `parameters.cookies.${number}.enabled` | `parameters.cookies.${number}.type.${number}` | `parameters.cookies.${number}.default.${string}` | `parameters.cookies.${number}.file.${string}` | `parameters.path.${number}` | `parameters.path.${number}.description` | `parameters.path.${number}.required` | `parameters.path.${number}.examples` | `parameters.path.${number}.type` | `parameters.path.${number}.default` | `parameters.path.${number}.key` | `parameters.path.${number}.file` | `parameters.path.${number}.value` | `parameters.path.${number}.enum` | `parameters.path.${number}.nullable` | `parameters.path.${number}.format` | `parameters.path.${number}.maximum` | `parameters.path.${number}.minimum` | `parameters.path.${number}.examples.${number}` | `parameters.path.${number}.enum.${number}` | `parameters.path.${number}.enabled` | `parameters.path.${number}.type.${number}` | `parameters.path.${number}.default.${string}` | `parameters.path.${number}.file.${string}` | `parameters.headers.${number}` | `parameters.headers.${number}.description` | `parameters.headers.${number}.required` | `parameters.headers.${number}.examples` | `parameters.headers.${number}.type` | `parameters.headers.${number}.default` | `parameters.headers.${number}.key` | `parameters.headers.${number}.file` | `parameters.headers.${number}.value` | `parameters.headers.${number}.enum` | `parameters.headers.${number}.nullable` | `parameters.headers.${number}.format` | `parameters.headers.${number}.maximum` | `parameters.headers.${number}.minimum` | `parameters.headers.${number}.examples.${number}` | `parameters.headers.${number}.enum.${number}` | `parameters.headers.${number}.enabled` | `parameters.headers.${number}.type.${number}` | `parameters.headers.${number}.default.${string}` | `parameters.headers.${number}.file.${string}` | `parameters.query.${number}` | `parameters.query.${number}.description` | `parameters.query.${number}.required` | `parameters.query.${number}.examples` | `parameters.query.${number}.type` | `parameters.query.${number}.default` | `parameters.query.${number}.key` | `parameters.query.${number}.file` | `parameters.query.${number}.value` | `parameters.query.${number}.enum` | `parameters.query.${number}.nullable` | `parameters.query.${number}.format` | `parameters.query.${number}.maximum` | `parameters.query.${number}.minimum` | `parameters.query.${number}.examples.${number}` | `parameters.query.${number}.enum.${number}` | `parameters.query.${number}.enabled` | `parameters.query.${number}.type.${number}` | `parameters.query.${number}.default.${string}` | `parameters.query.${number}.file.${string}` | "body.binary" | "body.formData" | "body.raw" | "body.activeBody" | "body.binary.type" | "body.binary.size" | "body.binary.slice" | "body.binary.text" | "body.binary.arrayBuffer" | "body.binary.stream" | "body.formData.encoding" | `body.formData.value.${number}` | `body.formData.value.${number}.description` | `body.formData.value.${number}.required` | `body.formData.value.${number}.examples` | `body.formData.value.${number}.type` | `body.formData.value.${number}.default` | `body.formData.value.${number}.key` | `body.formData.value.${number}.file` | `body.formData.value.${number}.value` | `body.formData.value.${number}.enum` | `body.formData.value.${number}.nullable` | `body.formData.value.${number}.format` | `body.formData.value.${number}.maximum` | `body.formData.value.${number}.minimum` | `body.formData.value.${number}.examples.${number}` | `body.formData.value.${number}.enum.${number}` | `body.formData.value.${number}.enabled` | `body.formData.value.${number}.type.${number}` | `body.formData.value.${number}.default.${string}` | `body.formData.value.${number}.file.${string}` | "body.raw.value" | "body.raw.encoding" | `serverVariables.${string}`>(uid: (string & import("zod").BRAND<"example">) | null | undefined, path: P, value: P extends "parameters" | "type" | "name" | "uid" | "body" | "requestUid" | "serverVariables" ? {
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "parameters" | "type" | "name" | "uid" | "body" | "requestUid" | "serverVariables" ? R extends Path<{
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }[K]> ? PathValue<{
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "parameters" | "type" | "name" | "uid" | "body" | "requestUid" | "serverVariables" | "body.formData.value" | "parameters.cookies" | `parameters.cookies.${number}` | "parameters.path" | "parameters.headers" | "parameters.query" | `parameters.cookies.${number}.description` | `parameters.cookies.${number}.required` | `parameters.cookies.${number}.examples` | `parameters.cookies.${number}.type` | `parameters.cookies.${number}.default` | `parameters.cookies.${number}.key` | `parameters.cookies.${number}.file` | `parameters.cookies.${number}.value` | `parameters.cookies.${number}.enum` | `parameters.cookies.${number}.nullable` | `parameters.cookies.${number}.format` | `parameters.cookies.${number}.maximum` | `parameters.cookies.${number}.minimum` | `parameters.cookies.${number}.examples.${number}` | `parameters.cookies.${number}.enum.${number}` | `parameters.cookies.${number}.enabled` | `parameters.cookies.${number}.type.${number}` | `parameters.cookies.${number}.default.${string}` | `parameters.cookies.${number}.file.${string}` | `parameters.path.${number}` | `parameters.path.${number}.description` | `parameters.path.${number}.required` | `parameters.path.${number}.examples` | `parameters.path.${number}.type` | `parameters.path.${number}.default` | `parameters.path.${number}.key` | `parameters.path.${number}.file` | `parameters.path.${number}.value` | `parameters.path.${number}.enum` | `parameters.path.${number}.nullable` | `parameters.path.${number}.format` | `parameters.path.${number}.maximum` | `parameters.path.${number}.minimum` | `parameters.path.${number}.examples.${number}` | `parameters.path.${number}.enum.${number}` | `parameters.path.${number}.enabled` | `parameters.path.${number}.type.${number}` | `parameters.path.${number}.default.${string}` | `parameters.path.${number}.file.${string}` | `parameters.headers.${number}` | `parameters.headers.${number}.description` | `parameters.headers.${number}.required` | `parameters.headers.${number}.examples` | `parameters.headers.${number}.type` | `parameters.headers.${number}.default` | `parameters.headers.${number}.key` | `parameters.headers.${number}.file` | `parameters.headers.${number}.value` | `parameters.headers.${number}.enum` | `parameters.headers.${number}.nullable` | `parameters.headers.${number}.format` | `parameters.headers.${number}.maximum` | `parameters.headers.${number}.minimum` | `parameters.headers.${number}.examples.${number}` | `parameters.headers.${number}.enum.${number}` | `parameters.headers.${number}.enabled` | `parameters.headers.${number}.type.${number}` | `parameters.headers.${number}.default.${string}` | `parameters.headers.${number}.file.${string}` | `parameters.query.${number}` | `parameters.query.${number}.description` | `parameters.query.${number}.required` | `parameters.query.${number}.examples` | `parameters.query.${number}.type` | `parameters.query.${number}.default` | `parameters.query.${number}.key` | `parameters.query.${number}.file` | `parameters.query.${number}.value` | `parameters.query.${number}.enum` | `parameters.query.${number}.nullable` | `parameters.query.${number}.format` | `parameters.query.${number}.maximum` | `parameters.query.${number}.minimum` | `parameters.query.${number}.examples.${number}` | `parameters.query.${number}.enum.${number}` | `parameters.query.${number}.enabled` | `parameters.query.${number}.type.${number}` | `parameters.query.${number}.default.${string}` | `parameters.query.${number}.file.${string}` | "body.binary" | "body.formData" | "body.raw" | "body.activeBody" | "body.binary.type" | "body.binary.size" | "body.binary.slice" | "body.binary.text" | "body.binary.arrayBuffer" | "body.binary.stream" | "body.formData.encoding" | `body.formData.value.${number}` | `body.formData.value.${number}.description` | `body.formData.value.${number}.required` | `body.formData.value.${number}.examples` | `body.formData.value.${number}.type` | `body.formData.value.${number}.default` | `body.formData.value.${number}.key` | `body.formData.value.${number}.file` | `body.formData.value.${number}.value` | `body.formData.value.${number}.enum` | `body.formData.value.${number}.nullable` | `body.formData.value.${number}.format` | `body.formData.value.${number}.maximum` | `body.formData.value.${number}.minimum` | `body.formData.value.${number}.examples.${number}` | `body.formData.value.${number}.enum.${number}` | `body.formData.value.${number}.enabled` | `body.formData.value.${number}.type.${number}` | `body.formData.value.${number}.default.${string}` | `body.formData.value.${number}.file.${string}` | "body.raw.value" | "body.raw.encoding" | `serverVariables.${string}`>(uid: string & import("zod").BRAND<"example">, path: P, value: P extends "parameters" | "type" | "name" | "uid" | "body" | "requestUid" | "serverVariables" ? {
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "parameters" | "type" | "name" | "uid" | "body" | "requestUid" | "serverVariables" ? R extends Path<{
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }[K]> ? PathValue<{
                type: "requestExample";
                uid: string & import("zod").BRAND<"example">;
                name: string;
                parameters: {
                    path: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    cookies: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    query: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                    headers: {
                        value: string;
                        key: string;
                        enabled: boolean;
                        minimum?: number | undefined;
                        type?: string | string[] | undefined;
                        maximum?: number | undefined;
                        default?: any;
                        required?: boolean | undefined;
                        description?: string | undefined;
                        enum?: string[] | undefined;
                        examples?: string[] | undefined;
                        format?: string | undefined;
                        file?: any;
                        nullable?: boolean | undefined;
                    }[];
                };
                body: {
                    activeBody: "binary" | "formData" | "raw";
                    binary?: Blob | undefined;
                    formData?: {
                        value: {
                            value: string;
                            key: string;
                            enabled: boolean;
                            minimum?: number | undefined;
                            type?: string | string[] | undefined;
                            maximum?: number | undefined;
                            default?: any;
                            required?: boolean | undefined;
                            description?: string | undefined;
                            enum?: string[] | undefined;
                            examples?: string[] | undefined;
                            format?: string | undefined;
                            file?: any;
                            nullable?: boolean | undefined;
                        }[];
                        encoding: "form-data" | "urlencoded";
                    } | undefined;
                    raw?: {
                        value: string;
                        encoding: "xml" | "text" | "json" | "html" | "javascript" | "yaml" | "edn";
                    } | undefined;
                };
                requestUid?: (string & import("zod").BRAND<"operation">) | undefined;
                serverVariables?: Record<string, string[]> | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"example">) => void;
            redo: (uid: string & import("zod").BRAND<"example">) => void;
            reset: () => void;
        };
        requestHistory: import("vue").Reactive<import("@scalar/oas-utils/entities/spec").RequestEvent[]>;
        securitySchemeMutators: {
            rawAdd: (item: {
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            } | {
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            } | {
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            } | {
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }) => void;
            add: (payload: import("@scalar/oas-utils/entities/spec").SecuritySchemePayload, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => {
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            } | {
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            } | {
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            } | {
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            };
            delete: (schemeUid: SecurityScheme["uid"]) => void;
            set: (item: {
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            } | {
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            } | {
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            } | {
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }) => void;
            edit: <P extends "description" | "type" | "name" | "in" | "uid" | "nameKey" | "scheme" | "bearerFormat" | "openIdConnectUrl" | "flows" | "password" | "username" | "value" | "token" | "flows.password" | "flows.implicit" | "flows.clientCredentials" | "flows.authorizationCode" | "flows.password.type" | "flows.password.password" | "flows.password.username" | "flows.password.token" | "flows.password.refreshUrl" | "flows.password.scopes" | "flows.password.selectedScopes" | "flows.password.x-scalar-client-id" | "flows.password.tokenUrl" | "flows.password.clientSecret" | `flows.password.scopes.${string}` | `flows.password.selectedScopes.${number}` | "flows.implicit.type" | "flows.implicit.token" | "flows.implicit.refreshUrl" | "flows.implicit.scopes" | "flows.implicit.selectedScopes" | "flows.implicit.x-scalar-client-id" | `flows.implicit.scopes.${string}` | `flows.implicit.selectedScopes.${number}` | "flows.implicit.authorizationUrl" | "flows.implicit.x-scalar-redirect-uri" | "flows.clientCredentials.type" | "flows.clientCredentials.token" | "flows.clientCredentials.refreshUrl" | "flows.clientCredentials.scopes" | "flows.clientCredentials.selectedScopes" | "flows.clientCredentials.x-scalar-client-id" | "flows.clientCredentials.tokenUrl" | "flows.clientCredentials.clientSecret" | `flows.clientCredentials.scopes.${string}` | `flows.clientCredentials.selectedScopes.${number}` | "flows.authorizationCode.type" | "flows.authorizationCode.token" | "flows.authorizationCode.refreshUrl" | "flows.authorizationCode.scopes" | "flows.authorizationCode.selectedScopes" | "flows.authorizationCode.x-scalar-client-id" | "flows.authorizationCode.tokenUrl" | "flows.authorizationCode.clientSecret" | `flows.authorizationCode.scopes.${string}` | `flows.authorizationCode.selectedScopes.${number}` | "flows.authorizationCode.authorizationUrl" | "flows.authorizationCode.x-scalar-redirect-uri" | "flows.authorizationCode.x-usePkce">(uid: (string & import("zod").BRAND<"securityScheme">) | null | undefined, path: P, value: (P extends "description" | "type" | "name" | "in" | "uid" | "nameKey" | "value" ? {
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "name" | "in" | "uid" | "nameKey" | "value" ? R extends Path<{
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) | (P extends "description" | "type" | "uid" | "nameKey" | "scheme" | "bearerFormat" | "password" | "username" | "token" ? {
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "uid" | "nameKey" | "scheme" | "bearerFormat" | "password" | "username" | "token" ? R extends Path<{
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) | (P extends "description" | "type" | "uid" | "nameKey" | "openIdConnectUrl" ? {
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "uid" | "nameKey" | "openIdConnectUrl" ? R extends Path<{
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) | (P extends "description" | "type" | "uid" | "nameKey" | "flows" ? {
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "uid" | "nameKey" | "flows" ? R extends Path<{
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never)) => void;
            untrackedEdit: <P extends "description" | "type" | "name" | "in" | "uid" | "nameKey" | "scheme" | "bearerFormat" | "openIdConnectUrl" | "flows" | "password" | "username" | "value" | "token" | "flows.password" | "flows.implicit" | "flows.clientCredentials" | "flows.authorizationCode" | "flows.password.type" | "flows.password.password" | "flows.password.username" | "flows.password.token" | "flows.password.refreshUrl" | "flows.password.scopes" | "flows.password.selectedScopes" | "flows.password.x-scalar-client-id" | "flows.password.tokenUrl" | "flows.password.clientSecret" | `flows.password.scopes.${string}` | `flows.password.selectedScopes.${number}` | "flows.implicit.type" | "flows.implicit.token" | "flows.implicit.refreshUrl" | "flows.implicit.scopes" | "flows.implicit.selectedScopes" | "flows.implicit.x-scalar-client-id" | `flows.implicit.scopes.${string}` | `flows.implicit.selectedScopes.${number}` | "flows.implicit.authorizationUrl" | "flows.implicit.x-scalar-redirect-uri" | "flows.clientCredentials.type" | "flows.clientCredentials.token" | "flows.clientCredentials.refreshUrl" | "flows.clientCredentials.scopes" | "flows.clientCredentials.selectedScopes" | "flows.clientCredentials.x-scalar-client-id" | "flows.clientCredentials.tokenUrl" | "flows.clientCredentials.clientSecret" | `flows.clientCredentials.scopes.${string}` | `flows.clientCredentials.selectedScopes.${number}` | "flows.authorizationCode.type" | "flows.authorizationCode.token" | "flows.authorizationCode.refreshUrl" | "flows.authorizationCode.scopes" | "flows.authorizationCode.selectedScopes" | "flows.authorizationCode.x-scalar-client-id" | "flows.authorizationCode.tokenUrl" | "flows.authorizationCode.clientSecret" | `flows.authorizationCode.scopes.${string}` | `flows.authorizationCode.selectedScopes.${number}` | "flows.authorizationCode.authorizationUrl" | "flows.authorizationCode.x-scalar-redirect-uri" | "flows.authorizationCode.x-usePkce">(uid: string & import("zod").BRAND<"securityScheme">, path: P, value: (P extends "description" | "type" | "name" | "in" | "uid" | "nameKey" | "value" ? {
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "name" | "in" | "uid" | "nameKey" | "value" ? R extends Path<{
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "apiKey";
                value: string;
                uid: string & import("zod").BRAND<"securityScheme">;
                name: string;
                nameKey: string;
                in: "cookie" | "query" | "header";
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) | (P extends "description" | "type" | "uid" | "nameKey" | "scheme" | "bearerFormat" | "password" | "username" | "token" ? {
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "uid" | "nameKey" | "scheme" | "bearerFormat" | "password" | "username" | "token" ? R extends Path<{
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "http";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                scheme: "basic" | "bearer";
                bearerFormat: string;
                username: string;
                password: string;
                token: string;
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) | (P extends "description" | "type" | "uid" | "nameKey" | "openIdConnectUrl" ? {
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "uid" | "nameKey" | "openIdConnectUrl" ? R extends Path<{
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "openIdConnect";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                openIdConnectUrl: string;
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) | (P extends "description" | "type" | "uid" | "nameKey" | "flows" ? {
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "type" | "uid" | "nameKey" | "flows" ? R extends Path<{
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }[K]> ? PathValue<{
                type: "oauth2";
                uid: string & import("zod").BRAND<"securityScheme">;
                nameKey: string;
                flows: {
                    password?: {
                        type: "password";
                        username: string;
                        password: string;
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    implicit?: {
                        type: "implicit";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                    } | undefined;
                    clientCredentials?: {
                        type: "clientCredentials";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        tokenUrl: string;
                        clientSecret: string;
                    } | undefined;
                    authorizationCode?: {
                        type: "authorizationCode";
                        token: string;
                        refreshUrl: string;
                        scopes: Record<string, string>;
                        selectedScopes: string[];
                        "x-scalar-client-id": string;
                        authorizationUrl: string;
                        "x-scalar-redirect-uri": string;
                        tokenUrl: string;
                        clientSecret: string;
                        "x-usePkce": "SHA-256" | "plain" | "no";
                    } | undefined;
                };
                description?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never)) => void;
            undo: (uid: string & import("zod").BRAND<"securityScheme">) => void;
            redo: (uid: string & import("zod").BRAND<"securityScheme">) => void;
            reset: () => void;
        };
        serverMutators: {
            rawAdd: (item: {
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }) => void;
            add: (payload: import("@scalar/oas-utils/entities/spec").ServerPayload, parentUid: string) => {
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            };
            delete: (serverUid: import("@scalar/oas-utils/entities/spec").Server["uid"], collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
            set: (item: {
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }) => void;
            edit: <P extends "description" | "uid" | "url" | "variables" | `variables.${string}`>(uid: (string & import("zod").BRAND<"server">) | null | undefined, path: P, value: P extends "description" | "uid" | "url" | "variables" ? {
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "uid" | "url" | "variables" ? R extends Path<{
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }[K]> ? PathValue<{
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "description" | "uid" | "url" | "variables" | `variables.${string}`>(uid: string & import("zod").BRAND<"server">, path: P, value: P extends "description" | "uid" | "url" | "variables" ? {
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "uid" | "url" | "variables" ? R extends Path<{
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }[K]> ? PathValue<{
                uid: string & import("zod").BRAND<"server">;
                url: string;
                description?: string | undefined;
                variables?: Record<string, Omit<import("@scalar/openapi-types").OpenAPIV3_1.ServerVariableObject, "enum"> & {
                    enum?: [string, ...string[]];
                    value?: string;
                }> | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"server">) => void;
            redo: (uid: string & import("zod").BRAND<"server">) => void;
            reset: () => void;
        };
        tagMutators: {
            rawAdd: (item: {
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }) => void;
            add: (payload: import("@scalar/oas-utils/entities/spec").TagPayload, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void | {
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            };
            delete: (tag: import("@scalar/oas-utils/entities/spec").Tag, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
            set: (item: {
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }) => void;
            edit: <P extends "description" | "externalDocs" | "x-internal" | "x-scalar-ignore" | "type" | "name" | "uid" | "children" | "externalDocs.description" | "externalDocs.url" | `children.${number}` | "x-scalar-children" | `x-scalar-children.${number}` | `x-scalar-children.${number}.tagName`>(uid: (string & import("zod").BRAND<"tag">) | null | undefined, path: P, value: P extends "description" | "externalDocs" | "x-internal" | "x-scalar-ignore" | "type" | "name" | "uid" | "children" | "x-scalar-children" ? {
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "externalDocs" | "x-internal" | "x-scalar-ignore" | "type" | "name" | "uid" | "children" | "x-scalar-children" ? R extends Path<{
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }[K]> ? PathValue<{
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "description" | "externalDocs" | "x-internal" | "x-scalar-ignore" | "type" | "name" | "uid" | "children" | "externalDocs.description" | "externalDocs.url" | `children.${number}` | "x-scalar-children" | `x-scalar-children.${number}` | `x-scalar-children.${number}.tagName`>(uid: string & import("zod").BRAND<"tag">, path: P, value: P extends "description" | "externalDocs" | "x-internal" | "x-scalar-ignore" | "type" | "name" | "uid" | "children" | "x-scalar-children" ? {
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "description" | "externalDocs" | "x-internal" | "x-scalar-ignore" | "type" | "name" | "uid" | "children" | "x-scalar-children" ? R extends Path<{
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }[K]> ? PathValue<{
                type: "tag";
                uid: string & import("zod").BRAND<"tag">;
                name: string;
                children: ((string & import("zod").BRAND<"tag">) | (string & import("zod").BRAND<"operation">))[];
                description?: string | undefined;
                externalDocs?: {
                    url: string;
                    description?: string | undefined;
                } | undefined;
                'x-scalar-children'?: {
                    tagName: string;
                }[] | undefined;
                'x-internal'?: boolean | undefined;
                'x-scalar-ignore'?: boolean | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"tag">) => void;
            redo: (uid: string & import("zod").BRAND<"tag">) => void;
            reset: () => void;
        };
        workspaceMutators: {
            rawAdd: (item: {
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }) => void;
            add: (payload?: Partial<Workspace>) => {
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            };
            delete: (uid: Workspace["uid"]) => void;
            set: (item: {
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }) => void;
            edit: <P extends "cookies" | "description" | "name" | "uid" | "collections" | "environments" | "activeEnvironmentId" | "themeId" | "selectedHttpClient" | "hotKeyConfig" | "proxyUrl" | `cookies.${number}` | `collections.${number}` | `environments.${string}` | "selectedHttpClient.targetKey" | "selectedHttpClient.clientKey" | "hotKeyConfig.modifiers" | "hotKeyConfig.hotKeys" | `hotKeyConfig.modifiers.${number}` | "hotKeyConfig.hotKeys." | "hotKeyConfig.hotKeys./" | "hotKeyConfig.hotKeys.Space" | "hotKeyConfig.hotKeys.Backspace" | "hotKeyConfig.hotKeys.Tab" | "hotKeyConfig.hotKeys.Enter" | "hotKeyConfig.hotKeys.Escape" | "hotKeyConfig.hotKeys.ArrowDown" | "hotKeyConfig.hotKeys.ArrowLeft" | "hotKeyConfig.hotKeys.ArrowRight" | "hotKeyConfig.hotKeys.ArrowUp" | "hotKeyConfig.hotKeys.End" | "hotKeyConfig.hotKeys.Home" | "hotKeyConfig.hotKeys.PageDown" | "hotKeyConfig.hotKeys.PageUp" | "hotKeyConfig.hotKeys.Delete" | "hotKeyConfig.hotKeys.0" | "hotKeyConfig.hotKeys.1" | "hotKeyConfig.hotKeys.2" | "hotKeyConfig.hotKeys.3" | "hotKeyConfig.hotKeys.4" | "hotKeyConfig.hotKeys.5" | "hotKeyConfig.hotKeys.6" | "hotKeyConfig.hotKeys.7" | "hotKeyConfig.hotKeys.8" | "hotKeyConfig.hotKeys.9" | "hotKeyConfig.hotKeys.a" | "hotKeyConfig.hotKeys.b" | "hotKeyConfig.hotKeys.c" | "hotKeyConfig.hotKeys.d" | "hotKeyConfig.hotKeys.e" | "hotKeyConfig.hotKeys.f" | "hotKeyConfig.hotKeys.g" | "hotKeyConfig.hotKeys.h" | "hotKeyConfig.hotKeys.i" | "hotKeyConfig.hotKeys.j" | "hotKeyConfig.hotKeys.k" | "hotKeyConfig.hotKeys.l" | "hotKeyConfig.hotKeys.m" | "hotKeyConfig.hotKeys.n" | "hotKeyConfig.hotKeys.o" | "hotKeyConfig.hotKeys.p" | "hotKeyConfig.hotKeys.q" | "hotKeyConfig.hotKeys.r" | "hotKeyConfig.hotKeys.s" | "hotKeyConfig.hotKeys.t" | "hotKeyConfig.hotKeys.u" | "hotKeyConfig.hotKeys.v" | "hotKeyConfig.hotKeys.w" | "hotKeyConfig.hotKeys.x" | "hotKeyConfig.hotKeys.y" | "hotKeyConfig.hotKeys.z" | "hotKeyConfig.hotKeys.*" | "hotKeyConfig.hotKeys.+" | "hotKeyConfig.hotKeys.-" | "hotKeyConfig.hotKeys.." | "hotKeyConfig.hotKeys.F1" | "hotKeyConfig.hotKeys.F2" | "hotKeyConfig.hotKeys.F3" | "hotKeyConfig.hotKeys.F4" | "hotKeyConfig.hotKeys.F5" | "hotKeyConfig.hotKeys.F6" | "hotKeyConfig.hotKeys.F7" | "hotKeyConfig.hotKeys.F8" | "hotKeyConfig.hotKeys.F9" | "hotKeyConfig.hotKeys.F10" | "hotKeyConfig.hotKeys.F11" | "hotKeyConfig.hotKeys.F12" | "hotKeyConfig.hotKeys.;" | "hotKeyConfig.hotKeys.=" | "hotKeyConfig.hotKeys.," | "hotKeyConfig.hotKeys.`" | "hotKeyConfig.hotKeys.[" | "hotKeyConfig.hotKeys.\\" | "hotKeyConfig.hotKeys.]" | "hotKeyConfig.hotKeys..modifiers" | `hotKeyConfig.hotKeys..modifiers.${number}` | "hotKeyConfig.hotKeys..event" | "hotKeyConfig.hotKeys./.modifiers" | `hotKeyConfig.hotKeys./.modifiers.${number}` | "hotKeyConfig.hotKeys./.event" | "hotKeyConfig.hotKeys.Space.modifiers" | `hotKeyConfig.hotKeys.Space.modifiers.${number}` | "hotKeyConfig.hotKeys.Space.event" | "hotKeyConfig.hotKeys.Backspace.modifiers" | `hotKeyConfig.hotKeys.Backspace.modifiers.${number}` | "hotKeyConfig.hotKeys.Backspace.event" | "hotKeyConfig.hotKeys.Tab.modifiers" | `hotKeyConfig.hotKeys.Tab.modifiers.${number}` | "hotKeyConfig.hotKeys.Tab.event" | "hotKeyConfig.hotKeys.Enter.modifiers" | `hotKeyConfig.hotKeys.Enter.modifiers.${number}` | "hotKeyConfig.hotKeys.Enter.event" | "hotKeyConfig.hotKeys.Escape.modifiers" | `hotKeyConfig.hotKeys.Escape.modifiers.${number}` | "hotKeyConfig.hotKeys.Escape.event" | "hotKeyConfig.hotKeys.ArrowDown.modifiers" | `hotKeyConfig.hotKeys.ArrowDown.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowDown.event" | "hotKeyConfig.hotKeys.ArrowLeft.modifiers" | `hotKeyConfig.hotKeys.ArrowLeft.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowLeft.event" | "hotKeyConfig.hotKeys.ArrowRight.modifiers" | `hotKeyConfig.hotKeys.ArrowRight.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowRight.event" | "hotKeyConfig.hotKeys.ArrowUp.modifiers" | `hotKeyConfig.hotKeys.ArrowUp.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowUp.event" | "hotKeyConfig.hotKeys.End.modifiers" | `hotKeyConfig.hotKeys.End.modifiers.${number}` | "hotKeyConfig.hotKeys.End.event" | "hotKeyConfig.hotKeys.Home.modifiers" | `hotKeyConfig.hotKeys.Home.modifiers.${number}` | "hotKeyConfig.hotKeys.Home.event" | "hotKeyConfig.hotKeys.PageDown.modifiers" | `hotKeyConfig.hotKeys.PageDown.modifiers.${number}` | "hotKeyConfig.hotKeys.PageDown.event" | "hotKeyConfig.hotKeys.PageUp.modifiers" | `hotKeyConfig.hotKeys.PageUp.modifiers.${number}` | "hotKeyConfig.hotKeys.PageUp.event" | "hotKeyConfig.hotKeys.Delete.modifiers" | `hotKeyConfig.hotKeys.Delete.modifiers.${number}` | "hotKeyConfig.hotKeys.Delete.event" | "hotKeyConfig.hotKeys.0.modifiers" | `hotKeyConfig.hotKeys.0.modifiers.${number}` | "hotKeyConfig.hotKeys.0.event" | "hotKeyConfig.hotKeys.1.modifiers" | `hotKeyConfig.hotKeys.1.modifiers.${number}` | "hotKeyConfig.hotKeys.1.event" | "hotKeyConfig.hotKeys.2.modifiers" | `hotKeyConfig.hotKeys.2.modifiers.${number}` | "hotKeyConfig.hotKeys.2.event" | "hotKeyConfig.hotKeys.3.modifiers" | `hotKeyConfig.hotKeys.3.modifiers.${number}` | "hotKeyConfig.hotKeys.3.event" | "hotKeyConfig.hotKeys.4.modifiers" | `hotKeyConfig.hotKeys.4.modifiers.${number}` | "hotKeyConfig.hotKeys.4.event" | "hotKeyConfig.hotKeys.5.modifiers" | `hotKeyConfig.hotKeys.5.modifiers.${number}` | "hotKeyConfig.hotKeys.5.event" | "hotKeyConfig.hotKeys.6.modifiers" | `hotKeyConfig.hotKeys.6.modifiers.${number}` | "hotKeyConfig.hotKeys.6.event" | "hotKeyConfig.hotKeys.7.modifiers" | `hotKeyConfig.hotKeys.7.modifiers.${number}` | "hotKeyConfig.hotKeys.7.event" | "hotKeyConfig.hotKeys.8.modifiers" | `hotKeyConfig.hotKeys.8.modifiers.${number}` | "hotKeyConfig.hotKeys.8.event" | "hotKeyConfig.hotKeys.9.modifiers" | `hotKeyConfig.hotKeys.9.modifiers.${number}` | "hotKeyConfig.hotKeys.9.event" | "hotKeyConfig.hotKeys.a.modifiers" | `hotKeyConfig.hotKeys.a.modifiers.${number}` | "hotKeyConfig.hotKeys.a.event" | "hotKeyConfig.hotKeys.b.modifiers" | `hotKeyConfig.hotKeys.b.modifiers.${number}` | "hotKeyConfig.hotKeys.b.event" | "hotKeyConfig.hotKeys.c.modifiers" | `hotKeyConfig.hotKeys.c.modifiers.${number}` | "hotKeyConfig.hotKeys.c.event" | "hotKeyConfig.hotKeys.d.modifiers" | `hotKeyConfig.hotKeys.d.modifiers.${number}` | "hotKeyConfig.hotKeys.d.event" | "hotKeyConfig.hotKeys.e.modifiers" | `hotKeyConfig.hotKeys.e.modifiers.${number}` | "hotKeyConfig.hotKeys.e.event" | "hotKeyConfig.hotKeys.f.modifiers" | `hotKeyConfig.hotKeys.f.modifiers.${number}` | "hotKeyConfig.hotKeys.f.event" | "hotKeyConfig.hotKeys.g.modifiers" | `hotKeyConfig.hotKeys.g.modifiers.${number}` | "hotKeyConfig.hotKeys.g.event" | "hotKeyConfig.hotKeys.h.modifiers" | `hotKeyConfig.hotKeys.h.modifiers.${number}` | "hotKeyConfig.hotKeys.h.event" | "hotKeyConfig.hotKeys.i.modifiers" | `hotKeyConfig.hotKeys.i.modifiers.${number}` | "hotKeyConfig.hotKeys.i.event" | "hotKeyConfig.hotKeys.j.modifiers" | `hotKeyConfig.hotKeys.j.modifiers.${number}` | "hotKeyConfig.hotKeys.j.event" | "hotKeyConfig.hotKeys.k.modifiers" | `hotKeyConfig.hotKeys.k.modifiers.${number}` | "hotKeyConfig.hotKeys.k.event" | "hotKeyConfig.hotKeys.l.modifiers" | `hotKeyConfig.hotKeys.l.modifiers.${number}` | "hotKeyConfig.hotKeys.l.event" | "hotKeyConfig.hotKeys.m.modifiers" | `hotKeyConfig.hotKeys.m.modifiers.${number}` | "hotKeyConfig.hotKeys.m.event" | "hotKeyConfig.hotKeys.n.modifiers" | `hotKeyConfig.hotKeys.n.modifiers.${number}` | "hotKeyConfig.hotKeys.n.event" | "hotKeyConfig.hotKeys.o.modifiers" | `hotKeyConfig.hotKeys.o.modifiers.${number}` | "hotKeyConfig.hotKeys.o.event" | "hotKeyConfig.hotKeys.p.modifiers" | `hotKeyConfig.hotKeys.p.modifiers.${number}` | "hotKeyConfig.hotKeys.p.event" | "hotKeyConfig.hotKeys.q.modifiers" | `hotKeyConfig.hotKeys.q.modifiers.${number}` | "hotKeyConfig.hotKeys.q.event" | "hotKeyConfig.hotKeys.r.modifiers" | `hotKeyConfig.hotKeys.r.modifiers.${number}` | "hotKeyConfig.hotKeys.r.event" | "hotKeyConfig.hotKeys.s.modifiers" | `hotKeyConfig.hotKeys.s.modifiers.${number}` | "hotKeyConfig.hotKeys.s.event" | "hotKeyConfig.hotKeys.t.modifiers" | `hotKeyConfig.hotKeys.t.modifiers.${number}` | "hotKeyConfig.hotKeys.t.event" | "hotKeyConfig.hotKeys.u.modifiers" | `hotKeyConfig.hotKeys.u.modifiers.${number}` | "hotKeyConfig.hotKeys.u.event" | "hotKeyConfig.hotKeys.v.modifiers" | `hotKeyConfig.hotKeys.v.modifiers.${number}` | "hotKeyConfig.hotKeys.v.event" | "hotKeyConfig.hotKeys.w.modifiers" | `hotKeyConfig.hotKeys.w.modifiers.${number}` | "hotKeyConfig.hotKeys.w.event" | "hotKeyConfig.hotKeys.x.modifiers" | `hotKeyConfig.hotKeys.x.modifiers.${number}` | "hotKeyConfig.hotKeys.x.event" | "hotKeyConfig.hotKeys.y.modifiers" | `hotKeyConfig.hotKeys.y.modifiers.${number}` | "hotKeyConfig.hotKeys.y.event" | "hotKeyConfig.hotKeys.z.modifiers" | `hotKeyConfig.hotKeys.z.modifiers.${number}` | "hotKeyConfig.hotKeys.z.event" | "hotKeyConfig.hotKeys.*.modifiers" | `hotKeyConfig.hotKeys.*.modifiers.${number}` | "hotKeyConfig.hotKeys.*.event" | "hotKeyConfig.hotKeys.+.modifiers" | `hotKeyConfig.hotKeys.+.modifiers.${number}` | "hotKeyConfig.hotKeys.+.event" | "hotKeyConfig.hotKeys.-.modifiers" | `hotKeyConfig.hotKeys.-.modifiers.${number}` | "hotKeyConfig.hotKeys.-.event" | "hotKeyConfig.hotKeys...modifiers" | `hotKeyConfig.hotKeys...modifiers.${number}` | "hotKeyConfig.hotKeys...event" | "hotKeyConfig.hotKeys.F1.modifiers" | `hotKeyConfig.hotKeys.F1.modifiers.${number}` | "hotKeyConfig.hotKeys.F1.event" | "hotKeyConfig.hotKeys.F2.modifiers" | `hotKeyConfig.hotKeys.F2.modifiers.${number}` | "hotKeyConfig.hotKeys.F2.event" | "hotKeyConfig.hotKeys.F3.modifiers" | `hotKeyConfig.hotKeys.F3.modifiers.${number}` | "hotKeyConfig.hotKeys.F3.event" | "hotKeyConfig.hotKeys.F4.modifiers" | `hotKeyConfig.hotKeys.F4.modifiers.${number}` | "hotKeyConfig.hotKeys.F4.event" | "hotKeyConfig.hotKeys.F5.modifiers" | `hotKeyConfig.hotKeys.F5.modifiers.${number}` | "hotKeyConfig.hotKeys.F5.event" | "hotKeyConfig.hotKeys.F6.modifiers" | `hotKeyConfig.hotKeys.F6.modifiers.${number}` | "hotKeyConfig.hotKeys.F6.event" | "hotKeyConfig.hotKeys.F7.modifiers" | `hotKeyConfig.hotKeys.F7.modifiers.${number}` | "hotKeyConfig.hotKeys.F7.event" | "hotKeyConfig.hotKeys.F8.modifiers" | `hotKeyConfig.hotKeys.F8.modifiers.${number}` | "hotKeyConfig.hotKeys.F8.event" | "hotKeyConfig.hotKeys.F9.modifiers" | `hotKeyConfig.hotKeys.F9.modifiers.${number}` | "hotKeyConfig.hotKeys.F9.event" | "hotKeyConfig.hotKeys.F10.modifiers" | `hotKeyConfig.hotKeys.F10.modifiers.${number}` | "hotKeyConfig.hotKeys.F10.event" | "hotKeyConfig.hotKeys.F11.modifiers" | `hotKeyConfig.hotKeys.F11.modifiers.${number}` | "hotKeyConfig.hotKeys.F11.event" | "hotKeyConfig.hotKeys.F12.modifiers" | `hotKeyConfig.hotKeys.F12.modifiers.${number}` | "hotKeyConfig.hotKeys.F12.event" | "hotKeyConfig.hotKeys.;.modifiers" | `hotKeyConfig.hotKeys.;.modifiers.${number}` | "hotKeyConfig.hotKeys.;.event" | "hotKeyConfig.hotKeys.=.modifiers" | `hotKeyConfig.hotKeys.=.modifiers.${number}` | "hotKeyConfig.hotKeys.=.event" | "hotKeyConfig.hotKeys.,.modifiers" | `hotKeyConfig.hotKeys.,.modifiers.${number}` | "hotKeyConfig.hotKeys.,.event" | "hotKeyConfig.hotKeys.`.modifiers" | `hotKeyConfig.hotKeys.\`.modifiers.${number}` | "hotKeyConfig.hotKeys.`.event" | "hotKeyConfig.hotKeys.[.modifiers" | `hotKeyConfig.hotKeys.[.modifiers.${number}` | "hotKeyConfig.hotKeys.[.event" | "hotKeyConfig.hotKeys.\\.modifiers" | `hotKeyConfig.hotKeys.\\.modifiers.${number}` | "hotKeyConfig.hotKeys.\\.event" | "hotKeyConfig.hotKeys.].modifiers" | `hotKeyConfig.hotKeys.].modifiers.${number}` | "hotKeyConfig.hotKeys.].event">(uid: (string & import("zod").BRAND<"workspace">) | null | undefined, path: P, value: P extends "cookies" | "description" | "name" | "uid" | "collections" | "environments" | "activeEnvironmentId" | "themeId" | "selectedHttpClient" | "hotKeyConfig" | "proxyUrl" ? {
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "cookies" | "description" | "name" | "uid" | "collections" | "environments" | "activeEnvironmentId" | "themeId" | "selectedHttpClient" | "hotKeyConfig" | "proxyUrl" ? R extends Path<{
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }[K]> ? PathValue<{
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            untrackedEdit: <P extends "cookies" | "description" | "name" | "uid" | "collections" | "environments" | "activeEnvironmentId" | "themeId" | "selectedHttpClient" | "hotKeyConfig" | "proxyUrl" | `cookies.${number}` | `collections.${number}` | `environments.${string}` | "selectedHttpClient.targetKey" | "selectedHttpClient.clientKey" | "hotKeyConfig.modifiers" | "hotKeyConfig.hotKeys" | `hotKeyConfig.modifiers.${number}` | "hotKeyConfig.hotKeys." | "hotKeyConfig.hotKeys./" | "hotKeyConfig.hotKeys.Space" | "hotKeyConfig.hotKeys.Backspace" | "hotKeyConfig.hotKeys.Tab" | "hotKeyConfig.hotKeys.Enter" | "hotKeyConfig.hotKeys.Escape" | "hotKeyConfig.hotKeys.ArrowDown" | "hotKeyConfig.hotKeys.ArrowLeft" | "hotKeyConfig.hotKeys.ArrowRight" | "hotKeyConfig.hotKeys.ArrowUp" | "hotKeyConfig.hotKeys.End" | "hotKeyConfig.hotKeys.Home" | "hotKeyConfig.hotKeys.PageDown" | "hotKeyConfig.hotKeys.PageUp" | "hotKeyConfig.hotKeys.Delete" | "hotKeyConfig.hotKeys.0" | "hotKeyConfig.hotKeys.1" | "hotKeyConfig.hotKeys.2" | "hotKeyConfig.hotKeys.3" | "hotKeyConfig.hotKeys.4" | "hotKeyConfig.hotKeys.5" | "hotKeyConfig.hotKeys.6" | "hotKeyConfig.hotKeys.7" | "hotKeyConfig.hotKeys.8" | "hotKeyConfig.hotKeys.9" | "hotKeyConfig.hotKeys.a" | "hotKeyConfig.hotKeys.b" | "hotKeyConfig.hotKeys.c" | "hotKeyConfig.hotKeys.d" | "hotKeyConfig.hotKeys.e" | "hotKeyConfig.hotKeys.f" | "hotKeyConfig.hotKeys.g" | "hotKeyConfig.hotKeys.h" | "hotKeyConfig.hotKeys.i" | "hotKeyConfig.hotKeys.j" | "hotKeyConfig.hotKeys.k" | "hotKeyConfig.hotKeys.l" | "hotKeyConfig.hotKeys.m" | "hotKeyConfig.hotKeys.n" | "hotKeyConfig.hotKeys.o" | "hotKeyConfig.hotKeys.p" | "hotKeyConfig.hotKeys.q" | "hotKeyConfig.hotKeys.r" | "hotKeyConfig.hotKeys.s" | "hotKeyConfig.hotKeys.t" | "hotKeyConfig.hotKeys.u" | "hotKeyConfig.hotKeys.v" | "hotKeyConfig.hotKeys.w" | "hotKeyConfig.hotKeys.x" | "hotKeyConfig.hotKeys.y" | "hotKeyConfig.hotKeys.z" | "hotKeyConfig.hotKeys.*" | "hotKeyConfig.hotKeys.+" | "hotKeyConfig.hotKeys.-" | "hotKeyConfig.hotKeys.." | "hotKeyConfig.hotKeys.F1" | "hotKeyConfig.hotKeys.F2" | "hotKeyConfig.hotKeys.F3" | "hotKeyConfig.hotKeys.F4" | "hotKeyConfig.hotKeys.F5" | "hotKeyConfig.hotKeys.F6" | "hotKeyConfig.hotKeys.F7" | "hotKeyConfig.hotKeys.F8" | "hotKeyConfig.hotKeys.F9" | "hotKeyConfig.hotKeys.F10" | "hotKeyConfig.hotKeys.F11" | "hotKeyConfig.hotKeys.F12" | "hotKeyConfig.hotKeys.;" | "hotKeyConfig.hotKeys.=" | "hotKeyConfig.hotKeys.," | "hotKeyConfig.hotKeys.`" | "hotKeyConfig.hotKeys.[" | "hotKeyConfig.hotKeys.\\" | "hotKeyConfig.hotKeys.]" | "hotKeyConfig.hotKeys..modifiers" | `hotKeyConfig.hotKeys..modifiers.${number}` | "hotKeyConfig.hotKeys..event" | "hotKeyConfig.hotKeys./.modifiers" | `hotKeyConfig.hotKeys./.modifiers.${number}` | "hotKeyConfig.hotKeys./.event" | "hotKeyConfig.hotKeys.Space.modifiers" | `hotKeyConfig.hotKeys.Space.modifiers.${number}` | "hotKeyConfig.hotKeys.Space.event" | "hotKeyConfig.hotKeys.Backspace.modifiers" | `hotKeyConfig.hotKeys.Backspace.modifiers.${number}` | "hotKeyConfig.hotKeys.Backspace.event" | "hotKeyConfig.hotKeys.Tab.modifiers" | `hotKeyConfig.hotKeys.Tab.modifiers.${number}` | "hotKeyConfig.hotKeys.Tab.event" | "hotKeyConfig.hotKeys.Enter.modifiers" | `hotKeyConfig.hotKeys.Enter.modifiers.${number}` | "hotKeyConfig.hotKeys.Enter.event" | "hotKeyConfig.hotKeys.Escape.modifiers" | `hotKeyConfig.hotKeys.Escape.modifiers.${number}` | "hotKeyConfig.hotKeys.Escape.event" | "hotKeyConfig.hotKeys.ArrowDown.modifiers" | `hotKeyConfig.hotKeys.ArrowDown.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowDown.event" | "hotKeyConfig.hotKeys.ArrowLeft.modifiers" | `hotKeyConfig.hotKeys.ArrowLeft.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowLeft.event" | "hotKeyConfig.hotKeys.ArrowRight.modifiers" | `hotKeyConfig.hotKeys.ArrowRight.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowRight.event" | "hotKeyConfig.hotKeys.ArrowUp.modifiers" | `hotKeyConfig.hotKeys.ArrowUp.modifiers.${number}` | "hotKeyConfig.hotKeys.ArrowUp.event" | "hotKeyConfig.hotKeys.End.modifiers" | `hotKeyConfig.hotKeys.End.modifiers.${number}` | "hotKeyConfig.hotKeys.End.event" | "hotKeyConfig.hotKeys.Home.modifiers" | `hotKeyConfig.hotKeys.Home.modifiers.${number}` | "hotKeyConfig.hotKeys.Home.event" | "hotKeyConfig.hotKeys.PageDown.modifiers" | `hotKeyConfig.hotKeys.PageDown.modifiers.${number}` | "hotKeyConfig.hotKeys.PageDown.event" | "hotKeyConfig.hotKeys.PageUp.modifiers" | `hotKeyConfig.hotKeys.PageUp.modifiers.${number}` | "hotKeyConfig.hotKeys.PageUp.event" | "hotKeyConfig.hotKeys.Delete.modifiers" | `hotKeyConfig.hotKeys.Delete.modifiers.${number}` | "hotKeyConfig.hotKeys.Delete.event" | "hotKeyConfig.hotKeys.0.modifiers" | `hotKeyConfig.hotKeys.0.modifiers.${number}` | "hotKeyConfig.hotKeys.0.event" | "hotKeyConfig.hotKeys.1.modifiers" | `hotKeyConfig.hotKeys.1.modifiers.${number}` | "hotKeyConfig.hotKeys.1.event" | "hotKeyConfig.hotKeys.2.modifiers" | `hotKeyConfig.hotKeys.2.modifiers.${number}` | "hotKeyConfig.hotKeys.2.event" | "hotKeyConfig.hotKeys.3.modifiers" | `hotKeyConfig.hotKeys.3.modifiers.${number}` | "hotKeyConfig.hotKeys.3.event" | "hotKeyConfig.hotKeys.4.modifiers" | `hotKeyConfig.hotKeys.4.modifiers.${number}` | "hotKeyConfig.hotKeys.4.event" | "hotKeyConfig.hotKeys.5.modifiers" | `hotKeyConfig.hotKeys.5.modifiers.${number}` | "hotKeyConfig.hotKeys.5.event" | "hotKeyConfig.hotKeys.6.modifiers" | `hotKeyConfig.hotKeys.6.modifiers.${number}` | "hotKeyConfig.hotKeys.6.event" | "hotKeyConfig.hotKeys.7.modifiers" | `hotKeyConfig.hotKeys.7.modifiers.${number}` | "hotKeyConfig.hotKeys.7.event" | "hotKeyConfig.hotKeys.8.modifiers" | `hotKeyConfig.hotKeys.8.modifiers.${number}` | "hotKeyConfig.hotKeys.8.event" | "hotKeyConfig.hotKeys.9.modifiers" | `hotKeyConfig.hotKeys.9.modifiers.${number}` | "hotKeyConfig.hotKeys.9.event" | "hotKeyConfig.hotKeys.a.modifiers" | `hotKeyConfig.hotKeys.a.modifiers.${number}` | "hotKeyConfig.hotKeys.a.event" | "hotKeyConfig.hotKeys.b.modifiers" | `hotKeyConfig.hotKeys.b.modifiers.${number}` | "hotKeyConfig.hotKeys.b.event" | "hotKeyConfig.hotKeys.c.modifiers" | `hotKeyConfig.hotKeys.c.modifiers.${number}` | "hotKeyConfig.hotKeys.c.event" | "hotKeyConfig.hotKeys.d.modifiers" | `hotKeyConfig.hotKeys.d.modifiers.${number}` | "hotKeyConfig.hotKeys.d.event" | "hotKeyConfig.hotKeys.e.modifiers" | `hotKeyConfig.hotKeys.e.modifiers.${number}` | "hotKeyConfig.hotKeys.e.event" | "hotKeyConfig.hotKeys.f.modifiers" | `hotKeyConfig.hotKeys.f.modifiers.${number}` | "hotKeyConfig.hotKeys.f.event" | "hotKeyConfig.hotKeys.g.modifiers" | `hotKeyConfig.hotKeys.g.modifiers.${number}` | "hotKeyConfig.hotKeys.g.event" | "hotKeyConfig.hotKeys.h.modifiers" | `hotKeyConfig.hotKeys.h.modifiers.${number}` | "hotKeyConfig.hotKeys.h.event" | "hotKeyConfig.hotKeys.i.modifiers" | `hotKeyConfig.hotKeys.i.modifiers.${number}` | "hotKeyConfig.hotKeys.i.event" | "hotKeyConfig.hotKeys.j.modifiers" | `hotKeyConfig.hotKeys.j.modifiers.${number}` | "hotKeyConfig.hotKeys.j.event" | "hotKeyConfig.hotKeys.k.modifiers" | `hotKeyConfig.hotKeys.k.modifiers.${number}` | "hotKeyConfig.hotKeys.k.event" | "hotKeyConfig.hotKeys.l.modifiers" | `hotKeyConfig.hotKeys.l.modifiers.${number}` | "hotKeyConfig.hotKeys.l.event" | "hotKeyConfig.hotKeys.m.modifiers" | `hotKeyConfig.hotKeys.m.modifiers.${number}` | "hotKeyConfig.hotKeys.m.event" | "hotKeyConfig.hotKeys.n.modifiers" | `hotKeyConfig.hotKeys.n.modifiers.${number}` | "hotKeyConfig.hotKeys.n.event" | "hotKeyConfig.hotKeys.o.modifiers" | `hotKeyConfig.hotKeys.o.modifiers.${number}` | "hotKeyConfig.hotKeys.o.event" | "hotKeyConfig.hotKeys.p.modifiers" | `hotKeyConfig.hotKeys.p.modifiers.${number}` | "hotKeyConfig.hotKeys.p.event" | "hotKeyConfig.hotKeys.q.modifiers" | `hotKeyConfig.hotKeys.q.modifiers.${number}` | "hotKeyConfig.hotKeys.q.event" | "hotKeyConfig.hotKeys.r.modifiers" | `hotKeyConfig.hotKeys.r.modifiers.${number}` | "hotKeyConfig.hotKeys.r.event" | "hotKeyConfig.hotKeys.s.modifiers" | `hotKeyConfig.hotKeys.s.modifiers.${number}` | "hotKeyConfig.hotKeys.s.event" | "hotKeyConfig.hotKeys.t.modifiers" | `hotKeyConfig.hotKeys.t.modifiers.${number}` | "hotKeyConfig.hotKeys.t.event" | "hotKeyConfig.hotKeys.u.modifiers" | `hotKeyConfig.hotKeys.u.modifiers.${number}` | "hotKeyConfig.hotKeys.u.event" | "hotKeyConfig.hotKeys.v.modifiers" | `hotKeyConfig.hotKeys.v.modifiers.${number}` | "hotKeyConfig.hotKeys.v.event" | "hotKeyConfig.hotKeys.w.modifiers" | `hotKeyConfig.hotKeys.w.modifiers.${number}` | "hotKeyConfig.hotKeys.w.event" | "hotKeyConfig.hotKeys.x.modifiers" | `hotKeyConfig.hotKeys.x.modifiers.${number}` | "hotKeyConfig.hotKeys.x.event" | "hotKeyConfig.hotKeys.y.modifiers" | `hotKeyConfig.hotKeys.y.modifiers.${number}` | "hotKeyConfig.hotKeys.y.event" | "hotKeyConfig.hotKeys.z.modifiers" | `hotKeyConfig.hotKeys.z.modifiers.${number}` | "hotKeyConfig.hotKeys.z.event" | "hotKeyConfig.hotKeys.*.modifiers" | `hotKeyConfig.hotKeys.*.modifiers.${number}` | "hotKeyConfig.hotKeys.*.event" | "hotKeyConfig.hotKeys.+.modifiers" | `hotKeyConfig.hotKeys.+.modifiers.${number}` | "hotKeyConfig.hotKeys.+.event" | "hotKeyConfig.hotKeys.-.modifiers" | `hotKeyConfig.hotKeys.-.modifiers.${number}` | "hotKeyConfig.hotKeys.-.event" | "hotKeyConfig.hotKeys...modifiers" | `hotKeyConfig.hotKeys...modifiers.${number}` | "hotKeyConfig.hotKeys...event" | "hotKeyConfig.hotKeys.F1.modifiers" | `hotKeyConfig.hotKeys.F1.modifiers.${number}` | "hotKeyConfig.hotKeys.F1.event" | "hotKeyConfig.hotKeys.F2.modifiers" | `hotKeyConfig.hotKeys.F2.modifiers.${number}` | "hotKeyConfig.hotKeys.F2.event" | "hotKeyConfig.hotKeys.F3.modifiers" | `hotKeyConfig.hotKeys.F3.modifiers.${number}` | "hotKeyConfig.hotKeys.F3.event" | "hotKeyConfig.hotKeys.F4.modifiers" | `hotKeyConfig.hotKeys.F4.modifiers.${number}` | "hotKeyConfig.hotKeys.F4.event" | "hotKeyConfig.hotKeys.F5.modifiers" | `hotKeyConfig.hotKeys.F5.modifiers.${number}` | "hotKeyConfig.hotKeys.F5.event" | "hotKeyConfig.hotKeys.F6.modifiers" | `hotKeyConfig.hotKeys.F6.modifiers.${number}` | "hotKeyConfig.hotKeys.F6.event" | "hotKeyConfig.hotKeys.F7.modifiers" | `hotKeyConfig.hotKeys.F7.modifiers.${number}` | "hotKeyConfig.hotKeys.F7.event" | "hotKeyConfig.hotKeys.F8.modifiers" | `hotKeyConfig.hotKeys.F8.modifiers.${number}` | "hotKeyConfig.hotKeys.F8.event" | "hotKeyConfig.hotKeys.F9.modifiers" | `hotKeyConfig.hotKeys.F9.modifiers.${number}` | "hotKeyConfig.hotKeys.F9.event" | "hotKeyConfig.hotKeys.F10.modifiers" | `hotKeyConfig.hotKeys.F10.modifiers.${number}` | "hotKeyConfig.hotKeys.F10.event" | "hotKeyConfig.hotKeys.F11.modifiers" | `hotKeyConfig.hotKeys.F11.modifiers.${number}` | "hotKeyConfig.hotKeys.F11.event" | "hotKeyConfig.hotKeys.F12.modifiers" | `hotKeyConfig.hotKeys.F12.modifiers.${number}` | "hotKeyConfig.hotKeys.F12.event" | "hotKeyConfig.hotKeys.;.modifiers" | `hotKeyConfig.hotKeys.;.modifiers.${number}` | "hotKeyConfig.hotKeys.;.event" | "hotKeyConfig.hotKeys.=.modifiers" | `hotKeyConfig.hotKeys.=.modifiers.${number}` | "hotKeyConfig.hotKeys.=.event" | "hotKeyConfig.hotKeys.,.modifiers" | `hotKeyConfig.hotKeys.,.modifiers.${number}` | "hotKeyConfig.hotKeys.,.event" | "hotKeyConfig.hotKeys.`.modifiers" | `hotKeyConfig.hotKeys.\`.modifiers.${number}` | "hotKeyConfig.hotKeys.`.event" | "hotKeyConfig.hotKeys.[.modifiers" | `hotKeyConfig.hotKeys.[.modifiers.${number}` | "hotKeyConfig.hotKeys.[.event" | "hotKeyConfig.hotKeys.\\.modifiers" | `hotKeyConfig.hotKeys.\\.modifiers.${number}` | "hotKeyConfig.hotKeys.\\.event" | "hotKeyConfig.hotKeys.].modifiers" | `hotKeyConfig.hotKeys.].modifiers.${number}` | "hotKeyConfig.hotKeys.].event">(uid: string & import("zod").BRAND<"workspace">, path: P, value: P extends "cookies" | "description" | "name" | "uid" | "collections" | "environments" | "activeEnvironmentId" | "themeId" | "selectedHttpClient" | "hotKeyConfig" | "proxyUrl" ? {
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }[P] : P extends `${infer K}.${infer R}` ? K extends "cookies" | "description" | "name" | "uid" | "collections" | "environments" | "activeEnvironmentId" | "themeId" | "selectedHttpClient" | "hotKeyConfig" | "proxyUrl" ? R extends Path<{
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }[K]> ? PathValue<{
                uid: string & import("zod").BRAND<"workspace">;
                name: string;
                description: string;
                collections: (string & import("zod").BRAND<"collection">)[];
                environments: Record<string, string>;
                activeEnvironmentId: string;
                cookies: (string & import("zod").BRAND<"cookie">)[];
                themeId: "default" | "alternate" | "moon" | "purple" | "solarized" | "bluePlanet" | "deepSpace" | "saturn" | "kepler" | "elysiajs" | "fastify" | "mars" | "none";
                selectedHttpClient: {
                    targetKey: string;
                    clientKey: string;
                };
                hotKeyConfig?: {
                    modifiers: ("Meta" | "Control" | "Shift" | "Alt" | "default")[];
                    hotKeys?: Partial<Record<"" | "/" | "Space" | "Backspace" | "Tab" | "Enter" | "Escape" | "ArrowDown" | "ArrowLeft" | "ArrowRight" | "ArrowUp" | "End" | "Home" | "PageDown" | "PageUp" | "Delete" | "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9" | "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z" | "*" | "+" | "-" | "." | "F1" | "F2" | "F3" | "F4" | "F5" | "F6" | "F7" | "F8" | "F9" | "F10" | "F11" | "F12" | ";" | "=" | "," | "`" | "[" | "\\" | "]", {
                        event: "addTopNav" | "closeModal" | "closeTopNav" | "createNew" | "executeRequest" | "focusAddressBar" | "focusRequestSearch" | "jumpToLastTab" | "jumpToTab" | "navigateSearchResultsDown" | "navigateSearchResultsUp" | "navigateTopNavLeft" | "navigateTopNavRight" | "openCommandPalette" | "selectSearchResult" | "toggleSidebar";
                        modifiers?: ("Meta" | "Control" | "Shift" | "Alt" | "default")[] | undefined;
                    }>> | undefined;
                } | undefined;
                proxyUrl?: string | undefined;
            }[K], R> : never : K extends `${number}` ? never : never : P extends `${number}` ? never : never) => void;
            undo: (uid: string & import("zod").BRAND<"workspace">) => void;
            redo: (uid: string & import("zod").BRAND<"workspace">) => void;
            reset: () => void;
        };
        addCollectionEnvironment: (environmentName: string, environment: import("@scalar/oas-utils/entities/spec").XScalarEnvironment, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
        removeCollectionEnvironment: (environmentName: string, collectionUid: import("@scalar/oas-utils/entities/spec").Collection["uid"]) => void;
    };
    /** Update the currently selected example */
    updateExample: (exampleKey: string, operationId: string) => void;
};
//# sourceMappingURL=create-client.d.ts.map