import { makeUrlAbsolute } from "@scalar/oas-utils/helpers";
import { parse } from "yaml";
async function resolve(value, options) {
  if (value?.startsWith("http://") || value?.startsWith("https://")) {
    const githubRawUrl = transformGitHubUrl(value);
    if (githubRawUrl) {
      return githubRawUrl;
    }
    if (value?.toLowerCase().endsWith(".json")) {
      return value;
    }
    if (value?.toLowerCase().endsWith(".yaml") || value?.toLowerCase().endsWith(".yml")) {
      return value;
    }
    const sandboxUrl = value.match(/https:\/\/sandbox\.scalar\.com\/(p|e)\/([a-zA-Z0-9]+)/);
    if (sandboxUrl?.[2]) {
      return `https://sandbox.scalar.com/files/${sandboxUrl[2]}/openapi.yaml`;
    }
    try {
      let result = await (options?.fetch ? options.fetch(value) : fetch(value, {
        cache: "no-cache"
      }));
      if (!result.ok && options?.fetch) {
        result = await fetch(value, {
          cache: "no-cache"
        });
      }
      if (result.ok) {
        const content = await result.text();
        const forwardedHost = result.headers.get("X-Forwarded-Host");
        try {
          const jsonContent = JSON.parse(content);
          if (jsonContent.openapi || jsonContent.swagger) {
            return value;
          }
        } catch {
          try {
            const yamlContent = parse(content);
            if (yamlContent?.openapi || yamlContent?.swagger) {
              return value;
            }
          } catch {
          }
        }
        const urlOrPathOrDocument = parseHtml(content);
        if (typeof urlOrPathOrDocument === "string") {
          try {
            return JSON.parse(urlOrPathOrDocument);
          } catch {
            try {
              const yaml = parse(urlOrPathOrDocument);
              if (typeof yaml === "object") {
                return yaml;
              }
            } catch {
            }
          }
        }
        if (typeof urlOrPathOrDocument === "object") {
          return urlOrPathOrDocument;
        }
        if (urlOrPathOrDocument) {
          return makeUrlAbsolute(urlOrPathOrDocument, forwardedHost || value);
        }
        const configUrl = getConfigurationAttributeUrl(content);
        if (configUrl) {
          return makeUrlAbsolute(configUrl, forwardedHost || value);
        }
        const embeddedSpec = parseEmbeddedOpenApi(content);
        if (embeddedSpec) {
          return embeddedSpec;
        }
      } else {
        console.warn(`[@scalar/import] Failed to fetch ${value}`);
      }
    } catch (error) {
      console.error(`[@scalar/import] Failed to fetch ${value}`, error);
    }
  }
  return void 0;
}
function parseHtml(html) {
  if (!html?.includes("<")) {
    return void 0;
  }
  const dataUrlMatch = html.match(/id=["']api-reference["'][\s\S]*?data-url=["']([^"']+)["']/)?.slice(1).find(Boolean);
  if (dataUrlMatch) {
    return dataUrlMatch;
  }
  const specUrlMatch = html.match(/spec-url=["']([^"']+)["']/);
  if (specUrlMatch?.[1]) {
    return specUrlMatch[1];
  }
  const redocInit = html.match(/Redoc.init\(["']([^"']+)["']/);
  if (redocInit?.[1]) {
    return redocInit[1];
  }
  const doubleEncodedConfigurationUrl = html.match(/&amp;quot;url&amp;quot;:&amp;quot;([^;]+)&amp;quot;/);
  if (doubleEncodedConfigurationUrl?.[1]) {
    return doubleEncodedConfigurationUrl[1];
  }
  const encodedConfigurationUrl = html.match(/&quot;url&quot;:&quot;([^;]+)&quot;/);
  if (encodedConfigurationUrl?.[1]) {
    return encodedConfigurationUrl[1];
  }
  const scriptContent = parseScriptContent(html);
  if (scriptContent) {
    return scriptContent;
  }
  const scriptConfigMatch = html.match(/url:\s*["']([^"']+)["']/i);
  if (scriptConfigMatch?.[1]) {
    return scriptConfigMatch?.[1];
  }
  const linkMatch = html.match(/<a[^>]*href=["']([^"']+\.(?:yaml|yml|json))["'][^>]*>/i);
  if (linkMatch?.[1]) {
    return linkMatch[1];
  }
  const escapedJsonMatch = html.match(/\\"spec\\":\{.*?\\"url\\":\\"([^"\\]+)\\"/);
  if (escapedJsonMatch?.[1]) {
    return escapedJsonMatch[1];
  }
  return void 0;
}
function getContentOfScriptTag(html) {
  const patterns = [
    // Double quotes
    /<script[^>]*id="api-reference[\s\S]*?">([\s\S]*?)<\/script>/i,
    // Single quotes
    /<script[^>]*id='api-reference[\s\S]*?'>([\s\S]*?)<\/script>/i
  ];
  for (const pattern of patterns) {
    const match = html.match(pattern);
    if (match?.[1]) {
      return match[1].trim();
    }
  }
  return void 0;
}
function parseScriptContent(html) {
  const content = getContentOfScriptTag(html);
  try {
    if (content) {
      try {
        return JSON.parse(content);
      } catch {
        try {
          const sanitizedContent = content.replace(/\\s/g, "\\\\s");
          return JSON.parse(sanitizedContent);
        } catch {
          return parse(content);
        }
      }
    }
  } catch (error) {
    console.error("[@scalar/import] Failed to parse script content:", error);
  }
  return void 0;
}
function getConfigurationAttribute(html) {
  const patterns = [
    // Double quotes
    /<script[^>]*id="api-reference"[^>]*data-configuration=["]([^"]+)["][^>]*>(.*?)<\/script>/s,
    // Single quotes
    /<script[^>]*id='api-reference'[^>]*data-configuration=[']([^']+)['][^>]*>(.*?)<\/script>/s,
    // Mix quote single first
    /<script[^>]*id='api-reference'[^>]*data-configuration=["]([^"]+)["][^>]*>(.*?)<\/script>/s,
    // Mix quote double first
    /<script[^>]*id="api-reference"[^>]*data-configuration=[']([^']+)['][^>]*>(.*?)<\/script>/s
  ];
  for (const pattern of patterns) {
    const match = html.match(pattern);
    if (match?.[1]) {
      return match[1];
    }
  }
  return void 0;
}
const getConfigurationAttributeUrl = (html) => {
  const configString = getConfigurationAttribute(html);
  if (!configString) {
    return void 0;
  }
  const config = JSON.parse(decodeHtmlEntities(configString));
  return config.url || config.spec?.url;
};
function parseEmbeddedOpenApi(html) {
  const configString = getConfigurationAttribute(html);
  if (!configString) {
    return void 0;
  }
  try {
    const config = JSON.parse(decodeHtmlEntities(configString));
    const content = config.content || config.spec?.content;
    if (content) {
      if (typeof content === "string") {
        return parse(content);
      }
      return content;
    }
  } catch (error) {
    console.error("[@scalar/import] Failed to parse embedded OpenAPI document:", error);
  }
  return void 0;
}
function decodeHtmlEntities(text) {
  const entities = {
    "&amp;": "&",
    "&lt;": "<",
    "&gt;": ">",
    "&quot;": '"',
    "&#39;": "'"
  };
  const updatedText = text.replace(new RegExp(Object.keys(entities).join("|"), "g"), (match) => entities[match]).replace(/\n/g, "\\n").trim();
  if (updatedText.startsWith("{&quot;")) {
    return decodeHtmlEntities(updatedText);
  }
  return updatedText;
}
function transformGitHubUrl(url) {
  const githubRegex = /^https:\/\/github\.com\/([^/]+)\/([^/]+)\/blob\/([^/]+)\/(.+)$/;
  const match = url.match(githubRegex);
  if (match) {
    const [, owner, repo, branch, path] = match;
    return `https://raw.githubusercontent.com/${owner}/${repo}/refs/heads/${branch}/${path}`;
  }
  return void 0;
}
export {
  getConfigurationAttribute,
  getConfigurationAttributeUrl,
  getContentOfScriptTag,
  resolve
};
//# sourceMappingURL=resolve.js.map
