import { z } from 'zod';
import { type RequestExample } from './request-examples.js';
export declare const requestMethods: readonly ["connect", "delete", "get", "head", "options", "patch", "post", "put", "trace"];
export type RequestMethod = (typeof requestMethods)[number];
/** A single set of populated values for a sent request */
export type ResponseInstance = Omit<Response, 'headers'> & {
    /** Store headers as an object to match what we had with axios */
    headers: Record<string, string>;
    /** Keys of headers which set cookies */
    cookieHeaderKeys: string[];
    /** Time in ms the request took */
    duration: number;
    /** The response data */
    data: string | Blob;
    /** The response size in bytes */
    size: number;
    /** The response status */
    status: number;
    /** The response method */
    method: RequestMethod;
    /** The request path */
    path: string;
};
/** A single request/response set to save to the history stack */
export type RequestEvent = {
    request: RequestExample;
    response: ResponseInstance;
    timestamp: number;
};
/** Open API Compliant Request Validator */
export declare const oasRequestSchema: z.ZodObject<{
    /**
     * A list of tags for API documentation control. Tags can be used for logical
     * grouping of operations by resources or any other qualifier.
     *
     * These tags are the openapi spec tag names, not uids
     */
    tags: z.ZodOptional<z.ZodArray<z.ZodString, "many">>;
    /** A short summary of what the operation does. */
    summary: z.ZodOptional<z.ZodString>;
    /** A verbose explanation of the operation behavior. CommonMark syntax MAY be used for rich text representation. */
    description: z.ZodOptional<z.ZodString>;
    /**
     * Unique string used to identify the operation. The id MUST be unique among all operations described in the API.
     * The operationId value is case-sensitive. Tools and libraries MAY use the operationId to uniquely identify an
     * operation, therefore, it is RECOMMENDED to follow bin common programming naming conventions./
     */
    operationId: z.ZodOptional<z.ZodString>;
    /**
     * A declaration of which security mechanisms can be used across the API. The list of
     * values includes alternative security requirement objects that can be used. Only
     * one of the security requirement objects need to be satisfied to authorize a request.
     * Individual operations can override this definition. To make security optional, an empty
     * security requirement ({}) can be included in the array.
     */
    security: z.ZodOptional<z.ZodArray<z.ZodRecord<z.ZodString, z.ZodDefault<z.ZodOptional<z.ZodArray<z.ZodString, "many">>>>, "many">>;
    /**
     * The request body applicable for this operation. The requestBody is fully supported in HTTP methods where the
     * HTTP 1.1 specification [RFC7231] has explicitly defined semantics for request bodies. In other cases where the
     * HTTP spec is vague (such as GET, HEAD and DELETE), requestBody is permitted but does not have well-defined
     * semantics and SHOULD be avoided if possible.
     */
    requestBody: z.ZodOptional<z.ZodAny>;
    /**
     * Request parameters
     */
    parameters: z.ZodOptional<z.ZodArray<z.ZodObject<{
        in: z.ZodEnum<["path", "query", "header", "cookie"]>;
        name: z.ZodString;
        description: z.ZodOptional<z.ZodString>;
        required: z.ZodDefault<z.ZodOptional<z.ZodBoolean>>;
        deprecated: z.ZodDefault<z.ZodOptional<z.ZodBoolean>>;
        schema: z.ZodOptional<z.ZodUnknown>;
        content: z.ZodOptional<z.ZodUnknown>;
        style: z.ZodOptional<z.ZodEnum<["matrix", "simple", "form", "label", "spaceDelimited", "pipeDelimited", "deepObject"]>>;
        example: z.ZodOptional<z.ZodUnknown>;
        examples: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodObject<{
            value: z.ZodUnknown;
            summary: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            value?: unknown;
            summary?: string | undefined;
        }, {
            value?: unknown;
            summary?: string | undefined;
        }>>>;
    }, "strip", z.ZodTypeAny, {
        required: boolean;
        name: string;
        in: "path" | "cookie" | "query" | "header";
        deprecated: boolean;
        example?: unknown;
        description?: string | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }, {
        name: string;
        in: "path" | "cookie" | "query" | "header";
        example?: unknown;
        required?: boolean | undefined;
        description?: string | undefined;
        deprecated?: boolean | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }>, "many">>;
    /**
     * External documentation object
     */
    externalDocs: z.ZodOptional<z.ZodObject<{
        description: z.ZodOptional<z.ZodString>;
        url: z.ZodDefault<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        url: string;
        description?: string | undefined;
    }, {
        description?: string | undefined;
        url?: string | undefined;
    }>>;
    deprecated: z.ZodOptional<z.ZodBoolean>;
    /** Response formats */
    responses: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodAny>>;
    /** xScalar examples */
    'x-scalar-examples': z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodObject<{
        name: z.ZodOptional<z.ZodString>;
        body: z.ZodOptional<z.ZodObject<{
            encoding: z.ZodEnum<["application/json", "text/plain", "text/html", "application/javascript", "application/xml", "application/yaml", "application/edn", "application/octet-stream", "application/x-www-form-urlencoded", "multipart/form-data", "binary"]>;
            content: z.ZodUnion<[z.ZodRecord<z.ZodString, z.ZodAny>, z.ZodString]>;
            file: z.ZodOptional<z.ZodNullable<z.ZodObject<{
                url: z.ZodString;
                base64: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                url: string;
                base64?: string | undefined;
            }, {
                url: string;
                base64?: string | undefined;
            }>>>;
        }, "strip", z.ZodTypeAny, {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        }, {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        }>>;
        parameters: z.ZodObject<{
            path: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
            query: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
            headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
            cookies: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
        }, "strip", z.ZodTypeAny, {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        }, {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        }>;
    }, "strip", z.ZodTypeAny, {
        parameters: {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        };
        name?: string | undefined;
        body?: {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        } | undefined;
    }, {
        parameters: {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        };
        name?: string | undefined;
        body?: {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        } | undefined;
    }>>>;
    /** Hide operations */
    'x-internal': z.ZodOptional<z.ZodBoolean>;
    'x-scalar-ignore': z.ZodOptional<z.ZodBoolean>;
}, "strip", z.ZodTypeAny, {
    description?: string | undefined;
    summary?: string | undefined;
    externalDocs?: {
        url: string;
        description?: string | undefined;
    } | undefined;
    'x-internal'?: boolean | undefined;
    'x-scalar-ignore'?: boolean | undefined;
    security?: Record<string, string[]>[] | undefined;
    tags?: string[] | undefined;
    deprecated?: boolean | undefined;
    operationId?: string | undefined;
    parameters?: {
        required: boolean;
        name: string;
        in: "path" | "cookie" | "query" | "header";
        deprecated: boolean;
        example?: unknown;
        description?: string | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }[] | undefined;
    requestBody?: any;
    responses?: Record<string, any> | undefined;
    'x-scalar-examples'?: Record<string, {
        parameters: {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        };
        name?: string | undefined;
        body?: {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        } | undefined;
    }> | undefined;
}, {
    description?: string | undefined;
    summary?: string | undefined;
    externalDocs?: {
        description?: string | undefined;
        url?: string | undefined;
    } | undefined;
    'x-internal'?: boolean | undefined;
    'x-scalar-ignore'?: boolean | undefined;
    security?: Record<string, string[] | undefined>[] | undefined;
    tags?: string[] | undefined;
    deprecated?: boolean | undefined;
    operationId?: string | undefined;
    parameters?: {
        name: string;
        in: "path" | "cookie" | "query" | "header";
        example?: unknown;
        required?: boolean | undefined;
        description?: string | undefined;
        deprecated?: boolean | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }[] | undefined;
    requestBody?: any;
    responses?: Record<string, any> | undefined;
    'x-scalar-examples'?: Record<string, {
        parameters: {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        };
        name?: string | undefined;
        body?: {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        } | undefined;
    }> | undefined;
}>;
/** Unified request schema for client usage */
export declare const requestSchema: z.ZodObject<z.objectUtil.extendShape<Omit<{
    /**
     * A list of tags for API documentation control. Tags can be used for logical
     * grouping of operations by resources or any other qualifier.
     *
     * These tags are the openapi spec tag names, not uids
     */
    tags: z.ZodOptional<z.ZodArray<z.ZodString, "many">>;
    /** A short summary of what the operation does. */
    summary: z.ZodOptional<z.ZodString>;
    /** A verbose explanation of the operation behavior. CommonMark syntax MAY be used for rich text representation. */
    description: z.ZodOptional<z.ZodString>;
    /**
     * Unique string used to identify the operation. The id MUST be unique among all operations described in the API.
     * The operationId value is case-sensitive. Tools and libraries MAY use the operationId to uniquely identify an
     * operation, therefore, it is RECOMMENDED to follow bin common programming naming conventions./
     */
    operationId: z.ZodOptional<z.ZodString>;
    /**
     * A declaration of which security mechanisms can be used across the API. The list of
     * values includes alternative security requirement objects that can be used. Only
     * one of the security requirement objects need to be satisfied to authorize a request.
     * Individual operations can override this definition. To make security optional, an empty
     * security requirement ({}) can be included in the array.
     */
    security: z.ZodOptional<z.ZodArray<z.ZodRecord<z.ZodString, z.ZodDefault<z.ZodOptional<z.ZodArray<z.ZodString, "many">>>>, "many">>;
    /**
     * The request body applicable for this operation. The requestBody is fully supported in HTTP methods where the
     * HTTP 1.1 specification [RFC7231] has explicitly defined semantics for request bodies. In other cases where the
     * HTTP spec is vague (such as GET, HEAD and DELETE), requestBody is permitted but does not have well-defined
     * semantics and SHOULD be avoided if possible.
     */
    requestBody: z.ZodOptional<z.ZodAny>;
    /**
     * Request parameters
     */
    parameters: z.ZodOptional<z.ZodArray<z.ZodObject<{
        in: z.ZodEnum<["path", "query", "header", "cookie"]>;
        name: z.ZodString;
        description: z.ZodOptional<z.ZodString>;
        required: z.ZodDefault<z.ZodOptional<z.ZodBoolean>>;
        deprecated: z.ZodDefault<z.ZodOptional<z.ZodBoolean>>;
        schema: z.ZodOptional<z.ZodUnknown>;
        content: z.ZodOptional<z.ZodUnknown>;
        style: z.ZodOptional<z.ZodEnum<["matrix", "simple", "form", "label", "spaceDelimited", "pipeDelimited", "deepObject"]>>;
        example: z.ZodOptional<z.ZodUnknown>;
        examples: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodObject<{
            value: z.ZodUnknown;
            summary: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            value?: unknown;
            summary?: string | undefined;
        }, {
            value?: unknown;
            summary?: string | undefined;
        }>>>;
    }, "strip", z.ZodTypeAny, {
        required: boolean;
        name: string;
        in: "path" | "cookie" | "query" | "header";
        deprecated: boolean;
        example?: unknown;
        description?: string | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }, {
        name: string;
        in: "path" | "cookie" | "query" | "header";
        example?: unknown;
        required?: boolean | undefined;
        description?: string | undefined;
        deprecated?: boolean | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }>, "many">>;
    /**
     * External documentation object
     */
    externalDocs: z.ZodOptional<z.ZodObject<{
        description: z.ZodOptional<z.ZodString>;
        url: z.ZodDefault<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        url: string;
        description?: string | undefined;
    }, {
        description?: string | undefined;
        url?: string | undefined;
    }>>;
    deprecated: z.ZodOptional<z.ZodBoolean>;
    /** Response formats */
    responses: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodAny>>;
    /** xScalar examples */
    'x-scalar-examples': z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodObject<{
        name: z.ZodOptional<z.ZodString>;
        body: z.ZodOptional<z.ZodObject<{
            encoding: z.ZodEnum<["application/json", "text/plain", "text/html", "application/javascript", "application/xml", "application/yaml", "application/edn", "application/octet-stream", "application/x-www-form-urlencoded", "multipart/form-data", "binary"]>;
            content: z.ZodUnion<[z.ZodRecord<z.ZodString, z.ZodAny>, z.ZodString]>;
            file: z.ZodOptional<z.ZodNullable<z.ZodObject<{
                url: z.ZodString;
                base64: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                url: string;
                base64?: string | undefined;
            }, {
                url: string;
                base64?: string | undefined;
            }>>>;
        }, "strip", z.ZodTypeAny, {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        }, {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        }>>;
        parameters: z.ZodObject<{
            path: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
            query: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
            headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
            cookies: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
        }, "strip", z.ZodTypeAny, {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        }, {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        }>;
    }, "strip", z.ZodTypeAny, {
        parameters: {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        };
        name?: string | undefined;
        body?: {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        } | undefined;
    }, {
        parameters: {
            path?: Record<string, string> | undefined;
            cookies?: Record<string, string> | undefined;
            query?: Record<string, string> | undefined;
            headers?: Record<string, string> | undefined;
        };
        name?: string | undefined;
        body?: {
            content: string | Record<string, any>;
            encoding: "application/json" | "text/plain" | "text/html" | "application/javascript" | "application/xml" | "application/yaml" | "application/edn" | "application/octet-stream" | "application/x-www-form-urlencoded" | "multipart/form-data" | "binary";
            file?: {
                url: string;
                base64?: string | undefined;
            } | null | undefined;
        } | undefined;
    }>>>;
    /** Hide operations */
    'x-internal': z.ZodOptional<z.ZodBoolean>;
    'x-scalar-ignore': z.ZodOptional<z.ZodBoolean>;
}, "x-scalar-examples">, {
    type: z.ZodDefault<z.ZodOptional<z.ZodLiteral<"request">>>;
    uid: z.ZodBranded<z.ZodDefault<z.ZodOptional<z.ZodString>>, "operation">;
    /** Path Key */
    path: z.ZodDefault<z.ZodOptional<z.ZodString>>;
    /** Request Method */
    method: z.ZodDefault<z.ZodEnum<["connect", "delete", "get", "head", "options", "patch", "post", "put", "trace"]>>;
    /** List of server UIDs specific to the request */
    servers: z.ZodDefault<z.ZodArray<z.ZodBranded<z.ZodString, "server">, "many">>;
    /** The currently selected server */
    selectedServerUid: z.ZodDefault<z.ZodNullable<z.ZodOptional<z.ZodBranded<z.ZodString, "server">>>>;
    /** List of example UIDs associated with the request */
    examples: z.ZodDefault<z.ZodArray<z.ZodBranded<z.ZodString, "example">, "many">>;
    /** List of security scheme UIDs associated with the request */
    selectedSecuritySchemeUids: z.ZodDefault<z.ZodArray<z.ZodUnion<[z.ZodBranded<z.ZodString, "securityScheme">, z.ZodArray<z.ZodBranded<z.ZodString, "securityScheme">, "many">]>, "many">>;
}>, "strip", z.ZodTypeAny, {
    path: string;
    type: "request";
    uid: string & z.BRAND<"operation">;
    selectedSecuritySchemeUids: ((string & z.BRAND<"securityScheme">) | (string & z.BRAND<"securityScheme">)[])[];
    selectedServerUid: (string & z.BRAND<"server">) | null;
    servers: (string & z.BRAND<"server">)[];
    examples: (string & z.BRAND<"example">)[];
    method: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace";
    description?: string | undefined;
    summary?: string | undefined;
    externalDocs?: {
        url: string;
        description?: string | undefined;
    } | undefined;
    'x-internal'?: boolean | undefined;
    'x-scalar-ignore'?: boolean | undefined;
    security?: Record<string, string[]>[] | undefined;
    tags?: string[] | undefined;
    deprecated?: boolean | undefined;
    operationId?: string | undefined;
    parameters?: {
        required: boolean;
        name: string;
        in: "path" | "cookie" | "query" | "header";
        deprecated: boolean;
        example?: unknown;
        description?: string | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }[] | undefined;
    requestBody?: any;
    responses?: Record<string, any> | undefined;
}, {
    path?: string | undefined;
    type?: "request" | undefined;
    uid?: string | undefined;
    description?: string | undefined;
    summary?: string | undefined;
    externalDocs?: {
        description?: string | undefined;
        url?: string | undefined;
    } | undefined;
    'x-internal'?: boolean | undefined;
    'x-scalar-ignore'?: boolean | undefined;
    security?: Record<string, string[] | undefined>[] | undefined;
    selectedSecuritySchemeUids?: (string | string[])[] | undefined;
    selectedServerUid?: string | null | undefined;
    servers?: string[] | undefined;
    tags?: string[] | undefined;
    deprecated?: boolean | undefined;
    examples?: string[] | undefined;
    operationId?: string | undefined;
    parameters?: {
        name: string;
        in: "path" | "cookie" | "query" | "header";
        example?: unknown;
        required?: boolean | undefined;
        description?: string | undefined;
        deprecated?: boolean | undefined;
        schema?: unknown;
        content?: unknown;
        style?: "matrix" | "simple" | "form" | "label" | "spaceDelimited" | "pipeDelimited" | "deepObject" | undefined;
        examples?: Record<string, {
            value?: unknown;
            summary?: string | undefined;
        }> | undefined;
    }[] | undefined;
    requestBody?: any;
    responses?: Record<string, any> | undefined;
    method?: "options" | "connect" | "delete" | "get" | "head" | "patch" | "post" | "put" | "trace" | undefined;
}>;
export type Request = z.infer<typeof requestSchema>;
export type RequestPayload = z.input<typeof requestSchema>;
//# sourceMappingURL=requests.d.ts.map