import { REGEX } from './regexHelpers.js';
import { isLocalUrl } from './isLocalUrl.js';

/** Redirects the request to a proxy server with a given URL. */
function redirectToProxy(proxy, url) {
    if (!shouldUseProxy(proxy, url)) {
        return url ?? '';
    }
    // Create new URL object from url
    const newUrl = new URL(url);
    // Rewrite the URL with the proxy
    newUrl.href = proxy;
    // Add the original URL as a query parameter
    newUrl.searchParams.append('scalar_url', url);
    return newUrl.toString();
}
/** Check if the URL is relative or if it's a domain without protocol */
const isRelativePath = (url) => {
    // Allow http:// https:// and other protocols such as file://
    if (REGEX.PROTOCOL.test(url)) {
        return false;
    }
    // Check if it looks like a domain (contains dots and no spaces)
    // This catches cases like "galaxy.scalar.com/planets"
    if (/^[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+(\/|$)/.test(url)) {
        return false;
    }
    return true;
};
/** Returns false for requests to localhost, relative URLs, if no proxy is defined … */
function shouldUseProxy(proxy, url) {
    // No proxy or url
    if (!proxy || !url)
        return false;
    // Relative URLs
    if (isRelativePath(url))
        return false;
    // Requests to localhost
    if (isLocalUrl(url))
        return false;
    return true;
}

export { isRelativePath, redirectToProxy, shouldUseProxy };
