import { parse, stringify } from "yaml";
const yaml = {
  /** Parse and throw if the return value is not an object */
  parse: (val) => {
    const yamlObject = parse(val);
    if (typeof yamlObject !== "object") {
      throw Error("Invalid YAML object");
    }
    return yamlObject;
  },
  /** Parse and return a fallback on failure */
  parseSafe(val, fallback) {
    try {
      return yaml.parse(val);
    } catch (err) {
      return typeof fallback === "function" ? fallback(err) : fallback;
    }
  },
  stringify
};
const json = {
  /** Parse and throw if the return value is not an object */
  parse: (val) => {
    const jsonObject = JSON.parse(val);
    if (typeof jsonObject !== "object") {
      throw Error("Invalid JSON object");
    }
    return jsonObject;
  },
  /** Parse and return a fallback on failure */
  parseSafe(val, fallback) {
    try {
      return json.parse(val);
    } catch (err) {
      return typeof fallback === "function" ? fallback(err) : fallback;
    }
  },
  stringify: (val) => JSON.stringify(val)
};
const isJsonString = (value) => {
  if (typeof value !== "string") {
    return false;
  }
  return !!json.parseSafe(value, false);
};
const transformToJson = (value) => {
  return JSON.stringify(json.parseSafe(value, yaml.parseSafe(value, value)));
};
function formatJsonOrYamlString(value) {
  const trimmed = value.trim();
  if (trimmed[0] !== "{" && trimmed[0] !== "[") {
    return value;
  }
  try {
    return JSON.stringify(JSON.parse(value), null, 2);
  } catch {
    return value;
  }
}
const parseJsonOrYaml = (value) => {
  if (typeof value !== "string") {
    return value;
  }
  const jsonObject = json.parseSafe(value, null);
  if (jsonObject) {
    return jsonObject;
  }
  if (value.length > 0 && ["{", "["].includes(value[0] ?? "")) {
    throw Error("Invalid JSON or YAML");
  }
  return yaml.parseSafe(value, (err) => {
    throw Error(err);
  });
};
export {
  formatJsonOrYamlString,
  isJsonString,
  json,
  parseJsonOrYaml,
  transformToJson,
  yaml
};
//# sourceMappingURL=parse.js.map
