import { json2xml } from "@scalar/helpers/file/json2xml";
import { normalizeMimeTypeObject } from "../helpers/normalize-mime-type-object.js";
import { prettyPrintJson } from "../helpers/pretty-print-json.js";
import { getExampleFromSchema } from "./get-example-from-schema.js";
import { getParametersFromOperation } from "./get-parameters-from-operation.js";
function getParamsFromObject(obj, nested = false, field) {
  return Object.entries(obj).flatMap(([key, value]) => {
    const name = field ?? key;
    if (Array.isArray(value) && !nested) {
      return getParamsFromObject(value, true, key);
    }
    if (typeof value === "object" && !(value instanceof File) && value !== null) {
      value = JSON.stringify(value);
    }
    return [{ name, value }];
  });
}
const standardMimeTypes = [
  "application/json",
  "application/octet-stream",
  "application/x-www-form-urlencoded",
  "application/xml",
  "multipart/form-data",
  "text/plain"
];
function getRequestBodyFromOperation(operation, selectedExampleKey, omitEmptyAndOptionalProperties) {
  const originalContent = operation.information?.requestBody?.content;
  const content = normalizeMimeTypeObject(originalContent);
  const mimeType = standardMimeTypes.find((currentMimeType) => !!content?.[currentMimeType]) ?? (Object.keys(content ?? {})[0] || "application/json");
  const isJsonLike = mimeType.includes("json") || mimeType.endsWith("+json");
  const examples = content?.[mimeType]?.examples ?? content?.["application/json"]?.examples;
  const selectedExample = examples?.[selectedExampleKey ?? Object.keys(examples ?? {})[0] ?? ""];
  if (selectedExample) {
    return {
      mimeType,
      text: prettyPrintJson(selectedExample?.value)
    };
  }
  const bodyParameters = getParametersFromOperation(operation, "body", false);
  if (bodyParameters.length > 0) {
    return {
      mimeType: "application/json",
      text: prettyPrintJson(bodyParameters[0]?.value ?? "")
    };
  }
  const formDataParameters = getParametersFromOperation(operation, "formData", false);
  if (formDataParameters.length > 0) {
    return {
      mimeType: "application/x-www-form-urlencoded",
      params: formDataParameters.map((parameter) => ({
        name: parameter.name,
        /**
         * TODO: This value MUST be a string
         * Figure out why this is not always a string
         *
         * JSON.stringify is a TEMPORARY fix
         */
        value: typeof parameter.value === "string" ? parameter.value : JSON.stringify(parameter.value)
      }))
    };
  }
  if (!mimeType) {
    return null;
  }
  const requestBodyObject = content?.[mimeType];
  const example = requestBodyObject?.example ? requestBodyObject?.example : void 0;
  if (isJsonLike) {
    const exampleFromSchema = requestBodyObject?.schema ? getExampleFromSchema(requestBodyObject?.schema, {
      mode: "write",
      omitEmptyAndOptionalProperties: omitEmptyAndOptionalProperties ?? false
    }) : null;
    const body = example ?? exampleFromSchema;
    return {
      mimeType,
      text: body ? typeof body === "string" ? body : JSON.stringify(body, null, 2) : void 0
    };
  }
  if (mimeType === "application/xml") {
    const exampleFromSchema = requestBodyObject?.schema ? getExampleFromSchema(requestBodyObject?.schema, {
      xml: true,
      mode: "write"
    }) : null;
    return {
      mimeType,
      text: example ?? json2xml(exampleFromSchema, "  ")
    };
  }
  if (mimeType === "application/octet-stream") {
    return {
      mimeType,
      text: "BINARY"
    };
  }
  if (mimeType === "text/plain") {
    const exampleFromSchema = requestBodyObject?.schema ? getExampleFromSchema(requestBodyObject?.schema, {
      xml: true,
      mode: "write"
    }) : null;
    return {
      mimeType,
      text: example ?? exampleFromSchema ?? ""
    };
  }
  if (mimeType === "multipart/form-data" || mimeType === "application/x-www-form-urlencoded") {
    const exampleFromSchema = requestBodyObject?.schema ? getExampleFromSchema(requestBodyObject?.schema, {
      xml: true,
      mode: "write"
    }) : null;
    return {
      mimeType,
      params: getParamsFromObject(example ?? exampleFromSchema ?? {})
    };
  }
  return null;
}
export {
  getRequestBodyFromOperation
};
//# sourceMappingURL=get-request-body-from-operation.js.map
