import {
  concatAll,
  getChildren,
  getErrors,
  getSiblings,
  isAnyOfError,
  isEnumError,
  isRequiredError,
  notUndefined
} from "./utils.js";
import {
  AdditionalPropValidationError,
  DefaultValidationError,
  EnumValidationError,
  PatternValidationError,
  RequiredValidationError,
  UnevaluatedPropValidationError
} from "./validation-errors/index.js";
const JSON_POINTERS_REGEX = /\/[\w_-]+(\/\d+)?/g;
function makeTree(ajvErrors = []) {
  const root = { children: {} };
  ajvErrors.forEach((ajvError) => {
    const instancePath = typeof ajvError.instancePath !== "undefined" ? ajvError.instancePath : ajvError.dataPath;
    const paths = instancePath === "" ? [""] : instancePath.match(JSON_POINTERS_REGEX);
    if (paths) {
      paths.reduce((obj, path, i) => {
        obj.children[path] = obj.children[path] || { children: {}, errors: [] };
        if (i === paths.length - 1) {
          obj.children[path].errors.push(ajvError);
        }
        return obj.children[path];
      }, root);
    }
  });
  return root;
}
function filterRedundantErrors(root, parent, key) {
  getErrors(root).forEach((error) => {
    if (isRequiredError(error)) {
      root.errors = [error];
      root.children = {};
    }
  });
  if (getErrors(root).some(isAnyOfError)) {
    if (Object.keys(root.children).length > 0) {
      delete root.errors;
    }
  }
  if (root.errors?.length && getErrors(root).every(isEnumError)) {
    if (getSiblings(parent)(root).filter(notUndefined).some(getErrors)) {
      delete parent.children[key];
    }
  }
  Object.entries(root.children).forEach(([k, child]) => filterRedundantErrors(child, root, k));
}
function createErrorInstances(root, options) {
  const errors = getErrors(root);
  if (errors.length && errors.every(isEnumError)) {
    const uniqueValues = new Set(concatAll([])(errors.map((e) => e.params.allowedValues)));
    const allowedValues = [...uniqueValues];
    const error = errors[0];
    return [
      new EnumValidationError(
        {
          ...error,
          params: { allowedValues }
        },
        options
      )
    ];
  }
  return concatAll(
    errors.reduce((ret, error) => {
      switch (error.keyword) {
        case "additionalProperties":
          return ret.concat(new AdditionalPropValidationError(error, options));
        case "pattern":
          return ret.concat(new PatternValidationError(error, options));
        case "required":
          return ret.concat(new RequiredValidationError(error, options));
        case "unevaluatedProperties":
          return ret.concat(new UnevaluatedPropValidationError(error, options));
        default:
          return ret.concat(new DefaultValidationError(error, options));
      }
    }, [])
  )(getChildren(root).map((child) => createErrorInstances(child, options)));
}
function prettify(ajvErrors, options) {
  const tree = makeTree(ajvErrors || []);
  filterRedundantErrors(tree);
  return createErrorInstances(tree, options);
}
export {
  createErrorInstances,
  prettify as default,
  filterRedundantErrors,
  makeTree
};
//# sourceMappingURL=helpers.js.map
