import type { UnknownObject } from '../../types/index.js';
/**
 * Checks if a string is a remote URL (starts with http:// or https://)
 * @param value - The URL string to check
 * @returns true if the string is a remote URL, false otherwise
 * @example
 * ```ts
 * isRemoteUrl('https://example.com/schema.json') // true
 * isRemoteUrl('http://api.example.com/schemas/user.json') // true
 * isRemoteUrl('#/components/schemas/User') // false
 * isRemoteUrl('./local-schema.json') // false
 * ```
 */
export declare function isRemoteUrl(value: string): boolean;
/**
 * Checks if a string is a local reference (starts with #)
 * @param value - The reference string to check
 * @returns true if the string is a local reference, false otherwise
 * @example
 * ```ts
 * isLocalRef('#/components/schemas/User') // true
 * isLocalRef('https://example.com/schema.json') // false
 * isLocalRef('./local-schema.json') // false
 * ```
 */
export declare function isLocalRef(value: string): boolean;
export type ResolveResult = {
    ok: true;
    data: unknown;
} | {
    ok: false;
};
/**
 * Retrieves a nested value from an object using an array of property segments.
 * @param target - The target object to traverse
 * @param segments - Array of property names representing the path to the desired value
 * @returns The value at the specified path, or undefined if the path doesn't exist
 * @example
 * const obj = { foo: { bar: { baz: 42 } } };
 * getNestedValue(obj, ['foo', 'bar', 'baz']); // returns 42
 */
export declare function getNestedValue(target: Record<string, any>, segments: string[]): any;
/**
 * Sets a value at a specified path in an object, creating intermediate objects/arrays as needed.
 * This function traverses the object structure and creates any missing intermediate objects
 * or arrays based on the path segments. If the next segment is a numeric string, it creates
 * an array instead of an object.
 *
 * @param obj - The target object to set the value in
 * @param path - The JSON pointer path where the value should be set
 * @param value - The value to set at the specified path
 * @throws {Error} If attempting to set a value at the root path ('')
 *
 * @example
 * const obj = {}
 * setValueAtPath(obj, '/foo/bar/0', 'value')
 * // Result:
 * // {
 * //   foo: {
 * //     bar: ['value']
 * //   }
 * // }
 *
 * @example
 * const obj = { existing: { path: 'old' } }
 * setValueAtPath(obj, '/existing/path', 'new')
 * // Result:
 * // {
 * //   existing: {
 * //     path: 'new'
 * //   }
 * // }
 */
export declare function setValueAtPath(obj: any, path: string, value: any): void;
/**
 * Prefixes an internal JSON reference with a given path prefix.
 * Takes a local reference (starting with #) and prepends the provided prefix segments.
 *
 * @param input - The internal reference string to prefix (must start with #)
 * @param prefix - Array of path segments to prepend to the reference
 * @returns The prefixed reference string
 * @throws Error if input is not a local reference
 * @example
 * prefixInternalRef('#/components/schemas/User', ['definitions'])
 * // Returns: '#/definitions/components/schemas/User'
 */
export declare function prefixInternalRef(input: string, prefix: string[]): string;
/**
 * Updates internal references in an object by adding a prefix to their paths.
 * Recursively traverses the input object and modifies any local $ref references
 * by prepending the given prefix to their paths. This is used when embedding external
 * documents to maintain correct reference paths relative to the main document.
 *
 * @param input - The object to update references in
 * @param prefix - Array of path segments to prepend to internal reference paths
 * @returns void
 * @example
 * ```ts
 * const input = {
 *   foo: {
 *     $ref: '#/components/schemas/User'
 *   }
 * }
 * prefixInternalRefRecursive(input, ['definitions'])
 * // Result:
 * // {
 * //   foo: {
 * //     $ref: '#/definitions/components/schemas/User'
 * //   }
 * // }
 * ```
 */
export declare function prefixInternalRefRecursive(input: unknown, prefix: string[]): string;
/**
 * Generates a short SHA-1 hash from a string value.
 * This function is used to create unique identifiers for external references
 * while keeping the hash length manageable. It uses the Web Crypto API to
 * generate a SHA-1 hash and returns the first 7 characters of the hex string.
 *
 * @param value - The string to hash
 * @returns A 7-character hexadecimal hash
 * @example
 * // Returns "a1b2c3d"
 * getHash("https://example.com/schema.json")
 */
export declare function getHash(value: string): Promise<string>;
/**
 * Represents a plugin that handles resolving references from external sources.
 * Plugins are responsible for fetching and processing data from different sources
 * like URLs or the filesystem. Each plugin must implement validation to determine
 * if it can handle a specific reference, and an execution function to perform
 * the actual resolution.
 *
 * @property validate - Determines if this plugin can handle the given reference
 * @property exec - Fetches and processes the reference, returning the resolved data
 */
export type Plugin = {
    validate: (value: string) => boolean;
    exec: (value: string) => Promise<ResolveResult>;
};
type Config = {
    plugins: Plugin[];
    root?: UnknownObject;
    cache?: Map<string, Promise<ResolveResult>>;
    treeShake: boolean;
    urlMap?: boolean;
};
/**
 * Bundles an OpenAPI specification by resolving all external references.
 * This function traverses the input object recursively and embeds external $ref
 * references into an x-ext section. External references can be URLs or local files.
 * The original $refs are updated to point to their embedded content in the x-ext section.
 * If the input is an object, it will be modified in place by adding an x-ext
 * property to store resolved external references.
 *
 * @param input - The OpenAPI specification object or string to bundle. If a string is provided,
 *                it should be a URL or file path that points to an OpenAPI specification.
 *                The string will be resolved using the provided plugins before bundling.
 * @param config - Configuration object containing plugins for resolving references
 * @returns A promise that resolves to the bundled specification with all references embedded
 * @example
 * // Example with object input
 * const spec = {
 *   paths: {
 *     '/users': {
 *       $ref: 'https://example.com/schemas/users.yaml'
 *     }
 *   }
 * }
 *
 * const bundled = await bundle(spec, { plugins: [fetchUrls()] })
 * // Result:
 * // {
 * //   paths: {
 * //     '/users': {
 * //       $ref: '#/x-ext/https~1~1example.com~1schemas~1users.yaml'
 * //     }
 * //   },
 * //   'x-ext': {
 * //     'https~1~1example.com~1schemas~1users.yaml': {
 * //       // Resolved content from users.yaml
 * //     }
 * //   }
 * // }
 *
 * // Example with URL input
 * const bundledFromUrl = await bundle('https://example.com/openapi.yaml', { plugins: [fetchUrls()] })
 * // The function will first fetch the OpenAPI spec from the URL,
 * // then bundle all its external references into the x-ext section
 */
export declare function bundle(input: UnknownObject | string, config: Config): Promise<unknown>;
export {};
//# sourceMappingURL=bundle.d.ts.map