import { escapeJsonPointer } from "../../utils/escape-json-pointer.js";
import path from "../../polyfills/path.js";
import { getSegmentsFromPath } from "../../utils/get-segments-from-path.js";
import { isObject } from "../../utils/is-object.js";
function isRemoteUrl(value) {
  return value.startsWith("http://") || value.startsWith("https://");
}
function isLocalRef(value) {
  return value.startsWith("#");
}
async function resolveContents(value, plugins) {
  const plugin = plugins.find((p) => p.validate(value));
  if (plugin) {
    return plugin.exec(value);
  }
  return {
    ok: false
  };
}
function getNestedValue(target, segments) {
  return segments.reduce((acc, key) => {
    if (acc === void 0) {
      return void 0;
    }
    return acc[key];
  }, target);
}
function setValueAtPath(obj, path2, value) {
  if (path2 === "") {
    throw new Error("Cannot set value at root ('') pointer");
  }
  const parts = getSegmentsFromPath(path2);
  let current = obj;
  for (let i = 0; i < parts.length; i++) {
    const key = parts[i];
    const isLast = i === parts.length - 1;
    const nextKey = parts[i + 1];
    const shouldBeArray = /^\d+$/.test(nextKey ?? "");
    if (isLast) {
      current[key] = value;
    } else {
      if (!(key in current) || typeof current[key] !== "object") {
        current[key] = shouldBeArray ? [] : {};
      }
      current = current[key];
    }
  }
}
function resolveReferencePath(base, relativePath) {
  if (isRemoteUrl(relativePath)) {
    return relativePath;
  }
  if (isRemoteUrl(base)) {
    const url = new URL(base);
    const mergedPath = path.join(path.dirname(url.pathname), relativePath);
    return new URL(mergedPath, base).toString();
  }
  return path.join(path.dirname(base), relativePath);
}
function prefixInternalRef(input, prefix) {
  if (!isLocalRef(input)) {
    throw "Please provide an internal ref";
  }
  return `#/${prefix.map(escapeJsonPointer).join("/")}${input.substring(1)}`;
}
function prefixInternalRefRecursive(input, prefix) {
  if (!isObject(input)) {
    return;
  }
  Object.values(input).forEach((el) => prefixInternalRefRecursive(el, prefix));
  if (typeof input === "object" && "$ref" in input && typeof input["$ref"] === "string") {
    const ref = input["$ref"];
    if (!isLocalRef(ref)) {
      return;
    }
    return input["$ref"] = prefixInternalRef(ref, prefix);
  }
}
const resolveAndCopyReferences = (targetDocument, sourceDocument, referencePath, externalRefsKey, documentKey, processedNodes = /* @__PURE__ */ new Set()) => {
  const referencedValue = getNestedValue(sourceDocument, getSegmentsFromPath(referencePath));
  if (processedNodes.has(referencedValue)) {
    return;
  }
  processedNodes.add(referencedValue);
  setValueAtPath(targetDocument, referencePath, referencedValue);
  const traverse = (node) => {
    if (!node || typeof node !== "object") {
      return;
    }
    if ("$ref" in node && typeof node["$ref"] === "string") {
      if (node["$ref"].startsWith(`#/${externalRefsKey}/${escapeJsonPointer(documentKey)}`)) {
        resolveAndCopyReferences(
          targetDocument,
          sourceDocument,
          node["$ref"].substring(1),
          documentKey,
          externalRefsKey,
          processedNodes
        );
      }
    }
    for (const value of Object.values(node)) {
      traverse(value);
    }
  };
  traverse(referencedValue);
};
async function getHash(value) {
  const encoder = new TextEncoder();
  const data = encoder.encode(value);
  const hashBuffer = await crypto.subtle.digest("SHA-1", data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  const hashHex = hashArray.map((b) => b.toString(16).padStart(2, "0")).join("");
  return hashHex.slice(0, 7);
}
async function bundle(input, config) {
  const cache = config.cache ?? /* @__PURE__ */ new Map();
  const resolveInput = async () => {
    if (typeof input !== "string") {
      return input;
    }
    const result = await resolveContents(input, config.plugins);
    if (result.ok) {
      return result.data;
    }
    throw "Please provide a valid string value or pass a loader to process the input";
  };
  const rawSpecification = await resolveInput();
  const documentRoot = config.root ?? rawSpecification;
  const EXTERNAL_KEY = "x-ext";
  const EXTERNAL_URL_MAPPING = "x-ext-urls";
  const bundler = async (root, origin = typeof input === "string" ? input : "") => {
    if (!isObject(root) && !Array.isArray(root)) {
      return;
    }
    if (typeof root === "object" && "$ref" in root && typeof root["$ref"] === "string") {
      const ref = root["$ref"];
      if (isLocalRef(ref)) {
        return;
      }
      const [prefix, path2 = ""] = ref.split("#", 2);
      const resolvedPath = resolveReferencePath(origin, prefix);
      const hashPath = await getHash(resolvedPath);
      const seen = cache.has(resolvedPath);
      if (!seen) {
        cache.set(resolvedPath, resolveContents(resolvedPath, config.plugins));
      }
      const result = await cache.get(resolvedPath);
      if (result.ok) {
        if (!seen) {
          prefixInternalRefRecursive(result.data, [EXTERNAL_KEY, hashPath]);
          await bundler(result.data, resolvedPath);
          if (config.urlMap) {
            setValueAtPath(documentRoot, `/${EXTERNAL_URL_MAPPING}/${escapeJsonPointer(resolvedPath)}`, hashPath);
          }
        }
        if (config.treeShake === true) {
          resolveAndCopyReferences(
            documentRoot,
            { [EXTERNAL_KEY]: { [hashPath]: result.data } },
            prefixInternalRef(`#${path2}`, [EXTERNAL_KEY, hashPath]).substring(1),
            EXTERNAL_KEY,
            hashPath
          );
        } else if (!seen) {
          setValueAtPath(documentRoot, `/${EXTERNAL_KEY}/${hashPath}`, result.data);
        }
        root.$ref = prefixInternalRef(`#${path2}`, [EXTERNAL_KEY, hashPath]);
        return;
      }
      return console.warn(
        `Failed to resolve external reference "${prefix}". The reference may be invalid, inaccessible, or missing a loader for this type of reference.`
      );
    }
    await Promise.all(
      Object.entries(root).map(async ([key, value]) => {
        if (key === EXTERNAL_KEY) {
          return;
        }
        await bundler(value, origin);
      })
    );
  };
  await bundler(rawSpecification);
  return rawSpecification;
}
export {
  bundle,
  getHash,
  getNestedValue,
  isLocalRef,
  isRemoteUrl,
  prefixInternalRef,
  prefixInternalRefRecursive,
  setValueAtPath
};
//# sourceMappingURL=bundle.js.map
