import { traverse } from "./traverse.js";
function upgradeFromTwoToThree(originalSpecification) {
  let specification = originalSpecification;
  if (specification !== null && typeof specification === "object" && typeof specification.swagger === "string" && specification.swagger?.startsWith("2.0")) {
    specification.openapi = "3.0.4";
    delete specification.swagger;
  } else {
    return specification;
  }
  if (specification.host) {
    const schemes = Array.isArray(specification.schemes) && specification.schemes?.length ? specification.schemes : ["http"];
    specification.servers = schemes.map((scheme) => ({
      url: `${scheme}://${specification.host}${specification.basePath ?? ""}`
    }));
    delete specification.basePath;
    delete specification.schemes;
    delete specification.host;
  } else if (specification.basePath) {
    specification.servers = [{ url: specification.basePath }];
    delete specification.basePath;
  }
  if (specification.definitions) {
    specification.components = Object.assign({}, specification.components, {
      schemas: specification.definitions
    });
    delete specification.definitions;
    specification = traverse(specification, (schema) => {
      if (typeof schema.$ref === "string" && schema.$ref.startsWith("#/definitions/")) {
        schema.$ref = schema.$ref.replace(/^#\/definitions\//, "#/components/schemas/");
      }
      return schema;
    });
  }
  specification = traverse(specification, (schema) => {
    if (schema.type === "file") {
      schema.type = "string";
      schema.format = "binary";
    }
    return schema;
  });
  if (typeof specification.paths === "object") {
    for (const path in specification.paths) {
      if (Object.hasOwn(specification.paths, path)) {
        const pathItem = specification.paths[path];
        for (const method in pathItem) {
          if (Object.hasOwn(pathItem, method)) {
            const operationItem = pathItem[method];
            if (operationItem.parameters) {
              const bodyParameter = structuredClone(
                operationItem.parameters.find((parameter) => parameter.in === "body") ?? {}
              );
              if (bodyParameter && Object.keys(bodyParameter).length) {
                delete bodyParameter.name;
                delete bodyParameter.in;
                const consumes = specification.consumes ?? operationItem.consumes ?? ["application/json"];
                if (typeof operationItem.requestBody !== "object") {
                  operationItem.requestBody = {};
                }
                if (typeof operationItem.requestBody.content !== "object") {
                  operationItem.requestBody.content = {};
                }
                const { schema, ...requestBody } = bodyParameter;
                operationItem.requestBody = {
                  ...operationItem.requestBody,
                  ...requestBody
                };
                for (const type of consumes) {
                  operationItem.requestBody.content[type] = {
                    schema
                  };
                }
              }
              operationItem.parameters = operationItem.parameters.filter(
                (parameter) => parameter.in !== "body"
              );
              delete operationItem.consumes;
              const formDataParameters = operationItem.parameters.filter(
                (parameter) => parameter.in === "formData"
              );
              if (formDataParameters.length > 0) {
                if (typeof operationItem.requestBody !== "object") {
                  operationItem.requestBody = {};
                }
                if (typeof operationItem.requestBody.content !== "object") {
                  operationItem.requestBody.content = {};
                }
                operationItem.requestBody.content["application/x-www-form-urlencoded"] = {
                  schema: {
                    type: "object",
                    properties: {},
                    required: []
                    // Initialize required array
                  }
                };
                for (const param of formDataParameters) {
                  operationItem.requestBody.content["application/x-www-form-urlencoded"].schema.properties[param.name] = {
                    type: param.type,
                    description: param.description
                  };
                  if (param.required) {
                    operationItem.requestBody.content["application/x-www-form-urlencoded"].schema.required.push(
                      param.name
                    );
                  }
                }
                operationItem.parameters = operationItem.parameters.filter(
                  (parameter) => parameter.in !== "formData"
                );
              }
              operationItem.parameters = operationItem.parameters.map(
                (parameter) => transformParameterObject(parameter)
              );
            }
            if (operationItem.responses) {
              for (const response in operationItem.responses) {
                if (Object.hasOwn(operationItem.responses, response)) {
                  const responseItem = operationItem.responses[response];
                  if (responseItem.headers) {
                    responseItem.headers = Object.entries(responseItem.headers).reduce((acc, [name, header]) => {
                      return {
                        [name]: transformParameterObject(header),
                        ...acc
                      };
                    }, {});
                  }
                  if (responseItem.schema) {
                    const produces = specification.produces ?? operationItem.produces ?? ["application/json"];
                    if (typeof responseItem.content !== "object") {
                      responseItem.content = {};
                    }
                    for (const type of produces) {
                      responseItem.content[type] = {
                        schema: responseItem.schema
                      };
                    }
                    delete responseItem.schema;
                  }
                }
              }
            }
            delete operationItem.produces;
            if (operationItem.parameters?.length === 0) {
              delete operationItem.parameters;
            }
          }
        }
      }
    }
  }
  if (specification.securityDefinitions) {
    if (typeof specification.components !== "object") {
      specification.components = {};
    }
    specification.components = specification.components;
    Object.assign(specification.components, { securitySchemes: {} });
    for (const [key, securityScheme] of Object.entries(specification.securityDefinitions)) {
      if (typeof securityScheme === "object") {
        if ("type" in securityScheme && securityScheme.type === "oauth2") {
          const { flow, authorizationUrl, tokenUrl, scopes } = securityScheme;
          Object.assign(specification.components.securitySchemes, {
            [key]: {
              type: "oauth2",
              flows: {
                [flow]: Object.assign(
                  {},
                  authorizationUrl && { authorizationUrl },
                  tokenUrl && { tokenUrl },
                  scopes && { scopes }
                )
              }
            }
          });
        } else if ("type" in securityScheme && securityScheme.type === "basic") {
          Object.assign(specification.components.securitySchemes, {
            [key]: {
              type: "http",
              scheme: "basic"
            }
          });
        } else {
          Object.assign(specification.components.securitySchemes, {
            [key]: securityScheme
          });
        }
      }
    }
    delete specification.securityDefinitions;
  }
  return specification;
}
function transformItemsObject(obj) {
  const schemaProperties = [
    "type",
    "format",
    "items",
    "maximum",
    "exclusiveMaximum",
    "minimum",
    "exclusiveMinimum",
    "maxLength",
    "minLength",
    "pattern",
    "maxItems",
    "minItems",
    "uniqueItems",
    "enum",
    "multipleOf"
  ];
  return schemaProperties.reduce((acc, property) => {
    if (Object.hasOwn(obj, property)) {
      acc[property] = obj[property];
      delete obj[property];
    }
    return acc;
  }, {});
}
function transformParameterObject(parameter) {
  delete parameter.collectionFormat;
  delete parameter.default;
  return {
    schema: transformItemsObject(parameter),
    ...parameter
  };
}
export {
  upgradeFromTwoToThree
};
//# sourceMappingURL=upgrade-from-two-to-three.js.map
