const phpGuzzle = {
  target: "php",
  client: "guzzle",
  title: "Guzzle",
  generate(request, configuration) {
    if (!request) {
      return "";
    }
    const options = {};
    const method = (request.method || "GET").toUpperCase();
    const url = request.url || "";
    if (request.headers && Array.isArray(request.headers) && request.headers.length > 0) {
      const headers = {};
      request.headers.forEach((header) => {
        if (headers[header.name] === void 0) {
          headers[header.name] = header.value;
        } else if (Array.isArray(headers[header.name])) {
          headers[header.name].push(header.value);
        } else {
          headers[header.name] = [headers[header.name], header.value];
        }
      });
      options.headers = headers;
    }
    if (request.queryString && request.queryString.length > 0) {
      const query = {};
      request.queryString.forEach((param) => {
        query[param.name] = param.value;
      });
      options.query = query;
    }
    if (request.cookies && request.cookies.length > 0) {
      const cookies = {};
      request.cookies.forEach((cookie) => {
        cookies[cookie.name] = cookie.value;
      });
      options.cookies = cookies;
    }
    if (configuration?.auth?.username && configuration.auth.password) {
      options.auth = [configuration.auth.username, configuration.auth.password];
    }
    if (request.postData) {
      if (request.postData.mimeType === "application/json") {
        try {
          options.json = JSON.parse(request.postData.text || "{}");
        } catch (e) {
          options.body = request.postData.text;
        }
      } else if (request.postData.mimeType === "multipart/form-data") {
        if (request.postData.params) {
          options.multipart = request.postData.params.map((param) => ({
            name: param.name,
            contents: param.fileName ? `fopen('${param.fileName}', 'r')` : param.value || ""
          }));
        } else if (request.postData.text) {
          try {
            options.form_params = JSON.parse(request.postData.text);
          } catch (e) {
            options.body = request.postData.text;
          }
        }
      } else if (request.postData.mimeType === "application/x-www-form-urlencoded") {
        if (request.postData.params) {
          const formParams = {};
          request.postData.params.forEach((param) => {
            formParams[param.name] = param.value || "";
          });
          options.form_params = formParams;
        }
      } else {
        options.body = request.postData.text;
      }
    }
    if (request.headers && Array.isArray(request.headers) && request.headers.some((h) => h.name === "Accept-Encoding" && h.value.includes("gzip"))) {
      options.decode_content = true;
    }
    let code = "$client = new GuzzleHttp\\Client();\n\n";
    if (Object.keys(options).length > 0) {
      const formattedOptions = formatOptionsArray(options);
      code += `$response = $client->request('${method}', '${url}', ${formattedOptions});`;
    } else {
      code += `$response = $client->request('${method}', '${url}');`;
    }
    return code;
  }
};
function formatOptionsArray(options, indent = 0) {
  if (Object.keys(options).length === 0) return "[]";
  const spaces = " ".repeat(4);
  let result = "[\n";
  for (const [key, value] of Object.entries(options)) {
    const formattedValue = formatValue(value, indent + 1);
    result += `${spaces.repeat(indent + 1)}'${key}' => ${formattedValue},
`;
  }
  result += `${spaces.repeat(indent)}]`;
  return result;
}
function formatValue(value, indent) {
  if (value === null) return "null";
  if (typeof value === "boolean") return value ? "true" : "false";
  if (typeof value === "string" && value.startsWith("fopen(")) return value;
  if (typeof value === "string") return `'${value}'`;
  if (typeof value === "number") return value.toString();
  if (Array.isArray(value)) {
    if (value.length === 0) return "[]";
    const spaces = " ".repeat(4);
    let result = "[\n";
    value.forEach((item) => {
      const formattedItem = formatValue(item, indent + 1);
      result += `${spaces.repeat(indent + 1)}${formattedItem},
`;
    });
    result += `${spaces.repeat(indent)}]`;
    return result;
  }
  if (typeof value === "object") {
    return formatOptionsArray(value, indent);
  }
  return `'${value}'`;
}
export {
  phpGuzzle
};
//# sourceMappingURL=guzzle.js.map
