const LENGTH_CONSIDERED_AS_SHORT = 40;
const pythonRequests = {
  target: "python",
  client: "requests",
  title: "Requests",
  generate(request, configuration) {
    const normalizedRequest = {
      url: "https://example.com",
      method: "get",
      ...request
    };
    const method = normalizedRequest.method.toLowerCase();
    const options = {};
    if (normalizedRequest.headers?.length) {
      options.headers = normalizedRequest.headers.reduce(
        (acc, header) => {
          if (!(header.name in acc)) {
            acc[header.name] = header.value;
          }
          return acc;
        },
        {}
      );
    }
    if (normalizedRequest.queryString?.length) {
      options.params = Object.fromEntries(normalizedRequest.queryString.map((q) => [q.name, q.value]));
    }
    if (normalizedRequest.cookies?.length) {
      options.cookies = Object.fromEntries(normalizedRequest.cookies.map((c) => [c.name, c.value]));
    }
    if (configuration?.auth?.username && configuration?.auth?.password) {
      options.auth = [configuration.auth.username, configuration.auth.password];
    }
    if (normalizedRequest.postData) {
      const { mimeType, text, params } = normalizedRequest.postData;
      if (mimeType === "application/json" && text) {
        try {
          options.json = JSON.parse(text);
        } catch {
          options.data = text;
        }
      } else if (mimeType === "application/octet-stream" && text) {
        options.data = text;
      } else if (mimeType === "multipart/form-data" && params) {
        const files = [];
        const formData = {};
        params.forEach((param) => {
          if (param.fileName !== void 0) {
            files.push({ key: param.name, file: `open("${param.fileName}", "rb")` });
          } else if (param.value !== void 0) {
            formData[param.name] = param.value;
          }
        });
        if (Object.keys(files).length) {
          options.files = files;
        }
        if (Object.keys(formData).length) {
          options.data = formData;
        }
      } else if (mimeType === "application/x-www-form-urlencoded" && params) {
        options.data = Object.fromEntries(params.map((p) => [p.name, p.value]));
      }
    }
    const formattedParams = [];
    const urlParam = `"${normalizedRequest.url}"`;
    if (normalizedRequest.url.length > LENGTH_CONSIDERED_AS_SHORT) {
      formattedParams.push(urlParam);
    } else {
      formattedParams.push("");
    }
    for (const [key, value] of Object.entries(options)) {
      if (key === "auth") {
        formattedParams.push(`${key}=(${JSON.stringify(value[0])}, ${JSON.stringify(value[1])})`);
      } else if (key === "files") {
        const filesTuples = value.map(({ key: key2, file }) => `      ("${key2}", ${file})`);
        const filesStr = "[\n" + filesTuples.join(",\n") + "\n    ]";
        formattedParams.push(`${key}=${filesStr}`);
      } else if (key === "json") {
        const jsonString = JSON.stringify(value, null, 2).split("\n").map((line, i) => i === 0 ? line : "    " + line).join("\n");
        formattedParams.push(`${key}=${jsonString}`);
      } else if (key === "data" && normalizedRequest.postData?.mimeType === "application/octet-stream") {
        formattedParams.push(`${key}=b"${value}"`);
      } else {
        const str = JSON.stringify(value, null, 2).split("\n").map((line, i) => i === 0 ? line : "    " + line).join("\n");
        formattedParams.push(`${key}=${str}`);
      }
    }
    if (normalizedRequest.url.length > LENGTH_CONSIDERED_AS_SHORT) {
      return `requests.${method}(
    ${formattedParams.join(",\n    ")}
)`;
    }
    if (formattedParams.length <= 1) {
      return `requests.${method}(${urlParam})`;
    }
    return `requests.${method}(${urlParam}${formattedParams.length > 1 ? "," : ""}
    ${formattedParams.slice(1).join(",\n    ")}
)`;
  }
};
export {
  pythonRequests
};
//# sourceMappingURL=requests.js.map
