const shellCurl = {
  target: "shell",
  client: "curl",
  title: "Curl",
  generate(request, configuration) {
    const normalizedRequest = {
      method: "GET",
      ...request
    };
    normalizedRequest.method = normalizedRequest.method.toUpperCase();
    const parts = ["curl"];
    const queryString = normalizedRequest.queryString?.length ? "?" + normalizedRequest.queryString.map((param) => {
      const encodedName = encodeURIComponent(param.name);
      const encodedValue = encodeURIComponent(param.value);
      return `${encodedName}=${encodedValue}`;
    }).join("&") : "";
    const url = `${normalizedRequest.url}${queryString}`;
    const hasSpecialChars = /[\s<>[\]{}|\\^%]/.test(url);
    const urlPart = queryString || hasSpecialChars ? `'${url}'` : url;
    parts[0] = `curl ${urlPart}`;
    if (normalizedRequest.method !== "GET") {
      parts.push(`--request ${normalizedRequest.method}`);
    }
    if (configuration?.auth?.username && configuration?.auth?.password) {
      parts.push(`--user '${configuration.auth.username}:${configuration.auth.password}'`);
    }
    if (normalizedRequest.headers?.length) {
      normalizedRequest.headers.forEach((header) => {
        parts.push(`--header '${header.name}: ${header.value}'`);
      });
      const acceptEncoding = normalizedRequest.headers.find((header) => header.name.toLowerCase() === "accept-encoding");
      if (acceptEncoding && /gzip|deflate/.test(acceptEncoding.value)) {
        parts.push("--compressed");
      }
    }
    if (normalizedRequest.cookies?.length) {
      const cookieString = normalizedRequest.cookies.map((cookie) => {
        const encodedName = encodeURIComponent(cookie.name);
        const encodedValue = encodeURIComponent(cookie.value);
        return `${encodedName}=${encodedValue}`;
      }).join("; ");
      parts.push(`--cookie '${cookieString}'`);
    }
    if (normalizedRequest.postData) {
      if (normalizedRequest.postData.mimeType === "application/json") {
        if (normalizedRequest.postData.text) {
          try {
            const jsonData = JSON.parse(normalizedRequest.postData.text);
            const prettyJson = JSON.stringify(jsonData, null, 2);
            parts.push(`--data '${prettyJson}'`);
          } catch {
            parts.push(`--data '${normalizedRequest.postData.text}'`);
          }
        }
      } else if (normalizedRequest.postData.mimeType === "application/octet-stream") {
        parts.push(`--data-binary '${normalizedRequest.postData.text}'`);
      } else if (normalizedRequest.postData.mimeType === "application/x-www-form-urlencoded" && normalizedRequest.postData.params) {
        normalizedRequest.postData.params.forEach((param) => {
          parts.push(`--data-urlencode '${encodeURIComponent(param.name)}=${param.value}'`);
        });
      } else if (normalizedRequest.postData.mimeType === "multipart/form-data" && normalizedRequest.postData.params) {
        normalizedRequest.postData.params.forEach((param) => {
          if (param.fileName !== void 0) {
            parts.push(`--form '${param.name}=@${param.fileName}'`);
          } else {
            parts.push(`--form '${param.name}=${param.value}'`);
          }
        });
      } else {
        try {
          const jsonData = JSON.parse(normalizedRequest.postData.text ?? "");
          const prettyJson = JSON.stringify(jsonData, null, 2);
          parts.push(`--data '${prettyJson}'`);
        } catch {
          parts.push(`--data '${normalizedRequest.postData.text}'`);
        }
      }
    }
    return parts.join(" \\\n  ");
  }
};
export {
  shellCurl
};
//# sourceMappingURL=curl.js.map
