import { describe, expect, it } from "vitest";
import { swiftNsurlsession } from "./nsurlsession.js";
describe("swiftNsurlsession", () => {
  it("returns a basic request", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com"
    });
    expect(result).toBe(
      `import Foundation

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "GET"

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("returns a POST request", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      method: "post"
    });
    expect(result).toBe(
      `import Foundation

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "POST"

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("has headers", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      headers: [
        {
          name: "Content-Type",
          value: "application/json"
        }
      ]
    });
    expect(result).toBe(
      `import Foundation

let headers = ["Content-Type": "application/json"]

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "GET"
request.allHTTPHeaderFields = headers

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles multipart form data with files", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      method: "POST",
      postData: {
        mimeType: "multipart/form-data",
        params: [
          {
            name: "file",
            fileName: "test.txt"
          },
          {
            name: "field",
            value: "value"
          }
        ]
      }
    });
    expect(result).toBe(
      `import Foundation
let parameters = [
  [
    "name": "file",
    "fileName": "test.txt"
  ],
  [
    "name": "field",
    "value": "value"
  ]
]

let boundary = "undefined"

var body = ""
var error: NSError? = nil
for param in parameters {
  let paramName = param["name"]!
  body += "--\\(boundary)\\r\\n"
  body += "Content-Disposition:form-data; name=\\"\\(paramName)\\""
  if let filename = param["fileName"] {
    let contentType = param["content-type"]!
    let fileContent = String(contentsOfFile: filename, encoding: String.Encoding.utf8)
    if (error != nil) {
      print(error as Any)
    }
    body += "; filename=\\"\\(filename)\\"\\r\\n"
    body += "Content-Type: \\(contentType)\\r\\n\\r\\n"
    body += fileContent
  } else if let paramValue = param["value"] {
    body += "\\r\\n\\r\\n\\(paramValue)"
  }
}

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "POST"
request.httpBody = postData as Data

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles url-encoded form data with special characters", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      method: "POST",
      postData: {
        mimeType: "application/x-www-form-urlencoded",
        params: [
          {
            name: "special chars!@#",
            value: "value"
          }
        ]
      }
    });
    expect(result).toBe(
      `import Foundation

let postData = NSMutableData(data: "special chars!@#=value".data(using: String.Encoding.utf8)!)

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "POST"
request.httpBody = postData as Data

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles binary data", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      method: "POST",
      postData: {
        mimeType: "application/octet-stream",
        text: "binary content"
      }
    });
    expect(result).toBe(
      `import Foundation

let postData = NSData(data: "binary content".data(using: String.Encoding.utf8)!)

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "POST"
request.httpBody = postData as Data

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles special characters in URL", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com/path with spaces/[brackets]"
    });
    expect(result).toBe(
      `import Foundation

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com/path%20with%20spaces/[brackets]")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "GET"

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles multiple headers with same name", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      headers: [
        { name: "X-Custom", value: "value1" },
        { name: "X-Custom", value: "value2" }
      ]
    });
    expect(result).toBe(
      `import Foundation

let headers = ["X-Custom": "value2"]

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "GET"
request.allHTTPHeaderFields = headers

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles headers with empty values", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com",
      headers: [{ name: "X-Empty", value: "" }]
    });
    expect(result).toBe(
      `import Foundation

let headers = ["X-Empty": ""]

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "GET"
request.allHTTPHeaderFields = headers

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
  it("handles query string parameters", () => {
    const result = swiftNsurlsession.generate({
      url: "https://example.com/api?param1=value1&param2=special value&param3=123"
    });
    expect(result).toBe(
      `import Foundation

let request = NSMutableURLRequest(url: NSURL(string: "https://example.com/api?param1=value1&param2=special%20value&param3=123")! as URL,
                                        cachePolicy: .useProtocolCachePolicy,
                                    timeoutInterval: 10.0)
request.httpMethod = "GET"

let session = URLSession.shared
let dataTask = session.dataTask(with: request as URLRequest, completionHandler: { (data, response, error) -> Void in
  if (error != nil) {
    print(error as Any)
  } else {
    let httpResponse = response as? HTTPURLResponse
    print(httpResponse)
  }
})

dataTask.resume()`
    );
  });
});
//# sourceMappingURL=nsurlsession.test.js.map
