import { computed, onMounted, onUnmounted, ref, watch } from "vue";
import { z } from "zod";
const colorMode = ref("dark");
const colorModeSchema = z.enum(["dark", "light", "system"]).optional().catch(void 0);
function useColorMode(opts = {}) {
  const { initialColorMode = "system", overrideColorMode } = opts;
  function toggleColorMode() {
    colorMode.value = darkLightMode.value === "dark" ? "light" : "dark";
    if (typeof window === "undefined") {
      return;
    }
    window?.localStorage?.setItem("colorMode", colorMode.value);
  }
  function setColorMode(value) {
    colorMode.value = value;
    if (typeof window === "undefined") {
      return;
    }
    window?.localStorage?.setItem("colorMode", colorMode.value);
  }
  function getSystemModePreference() {
    if (typeof window === "undefined") {
      return "light";
    }
    if (typeof window?.matchMedia !== "function") {
      return "dark";
    }
    return window?.matchMedia("(prefers-color-scheme: dark)")?.matches ? "dark" : "light";
  }
  const darkLightMode = computed({
    get: () => colorMode.value === "system" ? getSystemModePreference() : colorMode.value,
    set: setColorMode
  });
  const isDarkMode = computed({
    get: () => darkLightMode.value === "dark",
    set: (value) => setColorMode(value ? "dark" : "light")
  });
  function applyColorMode(mode) {
    if (typeof document === "undefined" || typeof window === "undefined") {
      return;
    }
    const classMode = overrideColorMode ?? (mode === "system" ? getSystemModePreference() : mode);
    if (classMode === "dark") {
      document.body.classList.add("dark-mode");
      document.body.classList.remove("light-mode");
    } else {
      document.body.classList.add("light-mode");
      document.body.classList.remove("dark-mode");
    }
  }
  const savedColorMode = colorModeSchema.parse(
    typeof window !== "undefined" ? window?.localStorage?.getItem("colorMode") : "system"
  );
  colorMode.value = savedColorMode ?? initialColorMode;
  watch(colorMode, applyColorMode, { immediate: true });
  onMounted(() => {
    if (typeof window !== "undefined" && typeof window?.matchMedia === "function") {
      const mediaQuery = window.matchMedia("(prefers-color-scheme: dark)");
      const handleChange = () => colorMode.value === "system" && applyColorMode("system");
      mediaQuery.addEventListener("change", handleChange);
      onUnmounted(() => {
        mediaQuery.removeEventListener("change", handleChange);
      });
    }
  });
  return {
    /** The current color mode (writable). */
    colorMode: computed({
      get: () => colorMode.value,
      set: setColorMode
    }),
    /** The computed dark/light mode (writable). */
    darkLightMode,
    /** Whether the current color mode is dark (writable). */
    isDarkMode,
    /** Toggles the color mode between light and dark. */
    toggleColorMode,
    /** Sets the color mode to the specified value. */
    setColorMode,
    /** Gets the system mode preference. */
    getSystemModePreference
  };
}
export {
  useColorMode
};
//# sourceMappingURL=useColorMode.js.map
