"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const schematics_1 = require("@angular-devkit/schematics");
const ts = require("typescript");
const ast_utils_1 = require("../utility/ast-utils");
const config_1 = require("../utility/config");
const ng_ast_utils_1 = require("../utility/ng-ast-utils");
const route_utils_1 = require("../utility/route-utils");
// Helper functions. (possible refactors to utils)
function formatMissingAppMsg(label, nameOrIndex) {
    const nameOrIndexText = nameOrIndex ? ` (${nameOrIndex})` : '';
    return `${label} app ${nameOrIndexText} not found.`;
}
function getSourceFile(host, path) {
    const buffer = host.read(path);
    if (!buffer) {
        throw new schematics_1.SchematicsException(`Could not find ${path}.`);
    }
    const content = buffer.toString();
    const source = ts.createSourceFile(path, content, ts.ScriptTarget.Latest, true);
    return source;
}
function getServerModulePath(host, project, architect) {
    const mainPath = architect.server.options.main;
    const mainSource = getSourceFile(host, mainPath);
    const allNodes = ast_utils_1.getSourceNodes(mainSource);
    const expNode = allNodes.filter(node => node.kind === ts.SyntaxKind.ExportDeclaration)[0];
    if (!expNode) {
        return null;
    }
    const relativePath = expNode.moduleSpecifier;
    const modulePath = core_1.normalize(`/${project.root}/src/${relativePath.text}.ts`);
    return modulePath;
}
function getComponentTemplateInfo(host, componentPath) {
    const compSource = getSourceFile(host, componentPath);
    const compMetadata = ast_utils_1.getDecoratorMetadata(compSource, 'Component', '@angular/core')[0];
    return {
        templateProp: getMetadataProperty(compMetadata, 'template'),
        templateUrlProp: getMetadataProperty(compMetadata, 'templateUrl'),
    };
}
function getComponentTemplate(host, compPath, tmplInfo) {
    let template = '';
    if (tmplInfo.templateProp) {
        template = tmplInfo.templateProp.getFullText();
    }
    else if (tmplInfo.templateUrlProp) {
        const templateUrl = tmplInfo.templateUrlProp.initializer.text;
        const dir = core_1.dirname(core_1.normalize(compPath));
        const templatePath = core_1.join(dir, templateUrl);
        const buffer = host.read(templatePath);
        if (buffer) {
            template = buffer.toString();
        }
    }
    return template;
}
function getBootstrapComponentPath(host, project) {
    if (!project.architect) {
        throw new Error('Project architect not found.');
    }
    const mainPath = project.architect.build.options.main;
    const modulePath = ng_ast_utils_1.getAppModulePath(host, mainPath);
    const moduleSource = getSourceFile(host, modulePath);
    const metadataNode = ast_utils_1.getDecoratorMetadata(moduleSource, 'NgModule', '@angular/core')[0];
    const bootstrapProperty = getMetadataProperty(metadataNode, 'bootstrap');
    const arrLiteral = bootstrapProperty
        .initializer;
    const componentSymbol = arrLiteral.elements[0].getText();
    const relativePath = ast_utils_1.getSourceNodes(moduleSource)
        .filter(node => node.kind === ts.SyntaxKind.ImportDeclaration)
        .filter(imp => {
        return ast_utils_1.findNode(imp, ts.SyntaxKind.Identifier, componentSymbol);
    })
        .map((imp) => {
        const pathStringLiteral = imp.moduleSpecifier;
        return pathStringLiteral.text;
    })[0];
    return core_1.join(core_1.dirname(core_1.normalize(modulePath)), relativePath + '.ts');
}
// end helper functions.
function validateProject(options) {
    return (host, context) => {
        const routerOutletCheckRegex = /<router\-outlet.*?>([\s\S]*?)<\/router\-outlet>/;
        const clientProject = getClientProject(host, options);
        if (clientProject.projectType !== 'application') {
            throw new schematics_1.SchematicsException(`App shell requires a project type of "application".`);
        }
        const componentPath = getBootstrapComponentPath(host, clientProject);
        const tmpl = getComponentTemplateInfo(host, componentPath);
        const template = getComponentTemplate(host, componentPath, tmpl);
        if (!routerOutletCheckRegex.test(template)) {
            const errorMsg = `Prerequisite for app shell is to define a router-outlet in your root component.`;
            context.logger.error(errorMsg);
            throw new schematics_1.SchematicsException(errorMsg);
        }
    };
}
function addUniversalTarget(options) {
    return (host, context) => {
        const architect = getClientArchitect(host, options);
        if (architect !== null) {
            if (architect.server !== undefined) {
                return host;
            }
        }
        // Copy options.
        const universalOptions = Object.assign({}, options, { name: options.universalProject });
        // Delete non-universal options.
        delete universalOptions.universalProject;
        delete universalOptions.route;
        return schematics_1.schematic('universal', universalOptions)(host, context);
    };
}
function addAppShellConfigToWorkspace(options) {
    return (host) => {
        if (!options.route) {
            throw new schematics_1.SchematicsException(`Route is not defined`);
        }
        const workspace = config_1.getWorkspace(host);
        const workspacePath = config_1.getWorkspacePath(host);
        const appShellTarget = {
            builder: '@angular-devkit/build-angular:app-shell',
            options: {
                browserTarget: `${options.clientProject}:build`,
                serverTarget: `${options.clientProject}:server`,
                route: options.route,
            },
        };
        if (!workspace.projects[options.clientProject]) {
            throw new schematics_1.SchematicsException(`Client app ${options.clientProject} not found.`);
        }
        const clientProject = workspace.projects[options.clientProject];
        if (!clientProject.architect) {
            throw new Error('Client project architect not found.');
        }
        clientProject.architect['app-shell'] = appShellTarget;
        host.overwrite(workspacePath, JSON.stringify(workspace, null, 2));
        return host;
    };
}
function addRouterModule(options) {
    return (host) => {
        const clientArchitect = getClientArchitect(host, options);
        const mainPath = clientArchitect.build.options.main;
        const modulePath = ng_ast_utils_1.getAppModulePath(host, mainPath);
        const moduleSource = getSourceFile(host, modulePath);
        const changes = ast_utils_1.addImportToModule(moduleSource, modulePath, 'RouterModule', '@angular/router');
        const recorder = host.beginUpdate(modulePath);
        changes.forEach((change) => {
            recorder.insertLeft(change.pos, change.toAdd);
        });
        host.commitUpdate(recorder);
        return host;
    };
}
function getMetadataProperty(metadata, propertyName) {
    const properties = metadata.properties;
    const property = properties
        .filter(prop => prop.kind === ts.SyntaxKind.PropertyAssignment)
        .filter((prop) => {
        const name = prop.name;
        switch (name.kind) {
            case ts.SyntaxKind.Identifier:
                return name.getText() === propertyName;
            case ts.SyntaxKind.StringLiteral:
                return name.text === propertyName;
        }
        return false;
    })[0];
    return property;
}
function addServerRoutes(options) {
    return (host) => {
        const clientProject = getClientProject(host, options);
        const architect = getClientArchitect(host, options);
        // const mainPath = universalArchitect.build.options.main;
        const modulePath = getServerModulePath(host, clientProject, architect);
        if (modulePath === null) {
            throw new schematics_1.SchematicsException('Universal/server module not found.');
        }
        let moduleSource = getSourceFile(host, modulePath);
        if (!ast_utils_1.isImported(moduleSource, 'Routes', '@angular/router')) {
            const recorder = host.beginUpdate(modulePath);
            const routesChange = route_utils_1.insertImport(moduleSource, modulePath, 'Routes', '@angular/router');
            if (routesChange.toAdd) {
                recorder.insertLeft(routesChange.pos, routesChange.toAdd);
            }
            const imports = ast_utils_1.getSourceNodes(moduleSource)
                .filter(node => node.kind === ts.SyntaxKind.ImportDeclaration)
                .sort((a, b) => a.getStart() - b.getStart());
            const insertPosition = imports[imports.length - 1].getEnd();
            const routeText = `\n\nconst routes: Routes = [ { path: '${options.route}', component: AppShellComponent }];`;
            recorder.insertRight(insertPosition, routeText);
            host.commitUpdate(recorder);
        }
        moduleSource = getSourceFile(host, modulePath);
        if (!ast_utils_1.isImported(moduleSource, 'RouterModule', '@angular/router')) {
            const recorder = host.beginUpdate(modulePath);
            const routerModuleChange = route_utils_1.insertImport(moduleSource, modulePath, 'RouterModule', '@angular/router');
            if (routerModuleChange.toAdd) {
                recorder.insertLeft(routerModuleChange.pos, routerModuleChange.toAdd);
            }
            const metadataChange = ast_utils_1.addSymbolToNgModuleMetadata(moduleSource, modulePath, 'imports', 'RouterModule.forRoot(routes)');
            if (metadataChange) {
                metadataChange.forEach((change) => {
                    recorder.insertRight(change.pos, change.toAdd);
                });
            }
            host.commitUpdate(recorder);
        }
        return host;
    };
}
function addShellComponent(options) {
    const componentOptions = {
        name: 'app-shell',
        module: options.rootModuleFileName,
        project: options.clientProject,
    };
    return schematics_1.schematic('component', componentOptions);
}
function getClientProject(host, options) {
    const workspace = config_1.getWorkspace(host);
    const clientProject = workspace.projects[options.clientProject];
    if (!clientProject) {
        throw new schematics_1.SchematicsException(formatMissingAppMsg('Client', options.clientProject));
    }
    return clientProject;
}
function getClientArchitect(host, options) {
    const clientArchitect = getClientProject(host, options).architect;
    if (!clientArchitect) {
        throw new Error('Client project architect not found.');
    }
    return clientArchitect;
}
function default_1(options) {
    return schematics_1.chain([
        validateProject(options),
        addUniversalTarget(options),
        addAppShellConfigToWorkspace(options),
        addRouterModule(options),
        addServerRoutes(options),
        addShellComponent(options),
    ]);
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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