"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const schematics_1 = require("@angular-devkit/schematics");
const tasks_1 = require("@angular-devkit/schematics/tasks");
const ts = require("typescript");
const ast_utils_1 = require("../utility/ast-utils");
const change_1 = require("../utility/change");
const config_1 = require("../utility/config");
const ng_ast_utils_1 = require("../utility/ng-ast-utils");
function getWorkspacePath(host) {
    const possibleFiles = ['/angular.json', '/.angular.json'];
    return possibleFiles.filter(path => host.exists(path))[0];
}
function getClientProject(host, options) {
    const workspace = config_1.getWorkspace(host);
    const clientProject = workspace.projects[options.clientProject];
    if (!clientProject) {
        throw new schematics_1.SchematicsException(`Client app ${options.clientProject} not found.`);
    }
    return clientProject;
}
function getClientArchitect(host, options) {
    const clientArchitect = getClientProject(host, options).architect;
    if (!clientArchitect) {
        throw new Error('Client project architect not found.');
    }
    return clientArchitect;
}
function updateConfigFile(options, tsConfigDirectory) {
    return (host) => {
        const workspace = config_1.getWorkspace(host);
        if (!workspace.projects[options.clientProject]) {
            throw new schematics_1.SchematicsException(`Client app ${options.clientProject} not found.`);
        }
        const clientProject = workspace.projects[options.clientProject];
        if (!clientProject.architect) {
            throw new Error('Client project architect not found.');
        }
        const builderOptions = {
            outputPath: `dist/${options.clientProject}-server`,
            main: `${clientProject.root}src/main.server.ts`,
            tsConfig: core_1.join(tsConfigDirectory, `${options.tsconfigFileName}.json`),
        };
        const serverTarget = {
            builder: '@angular-devkit/build-angular:server',
            options: builderOptions,
        };
        clientProject.architect.server = serverTarget;
        const workspacePath = getWorkspacePath(host);
        host.overwrite(workspacePath, JSON.stringify(workspace, null, 2));
        return host;
    };
}
function findBrowserModuleImport(host, modulePath) {
    const moduleBuffer = host.read(modulePath);
    if (!moduleBuffer) {
        throw new schematics_1.SchematicsException(`Module file (${modulePath}) not found`);
    }
    const moduleFileText = moduleBuffer.toString('utf-8');
    const source = ts.createSourceFile(modulePath, moduleFileText, ts.ScriptTarget.Latest, true);
    const decoratorMetadata = ast_utils_1.getDecoratorMetadata(source, 'NgModule', '@angular/core')[0];
    const browserModuleNode = ast_utils_1.findNode(decoratorMetadata, ts.SyntaxKind.Identifier, 'BrowserModule');
    if (browserModuleNode === null) {
        throw new schematics_1.SchematicsException(`Cannot find BrowserModule import in ${modulePath}`);
    }
    return browserModuleNode;
}
function wrapBootstrapCall(options) {
    return (host) => {
        const clientArchitect = getClientArchitect(host, options);
        const mainPath = core_1.normalize('/' + clientArchitect.build.options.main);
        let bootstrapCall = ng_ast_utils_1.findBootstrapModuleCall(host, mainPath);
        if (bootstrapCall === null) {
            throw new schematics_1.SchematicsException('Bootstrap module not found.');
        }
        let bootstrapCallExpression = null;
        let currentCall = bootstrapCall;
        while (bootstrapCallExpression === null && currentCall.parent) {
            currentCall = currentCall.parent;
            if (currentCall.kind === ts.SyntaxKind.ExpressionStatement) {
                bootstrapCallExpression = currentCall;
            }
        }
        bootstrapCall = currentCall;
        const recorder = host.beginUpdate(mainPath);
        const beforeText = `document.addEventListener('DOMContentLoaded', () => {\n  `;
        const afterText = `\n});`;
        recorder.insertLeft(bootstrapCall.getStart(), beforeText);
        recorder.insertRight(bootstrapCall.getEnd(), afterText);
        host.commitUpdate(recorder);
    };
}
function addServerTransition(options) {
    return (host) => {
        const clientProject = getClientProject(host, options);
        const clientArchitect = getClientArchitect(host, options);
        const mainPath = core_1.normalize('/' + clientArchitect.build.options.main);
        const bootstrapModuleRelativePath = ng_ast_utils_1.findBootstrapModulePath(host, mainPath);
        const bootstrapModulePath = core_1.normalize(`/${clientProject.root}/src/${bootstrapModuleRelativePath}.ts`);
        const browserModuleImport = findBrowserModuleImport(host, bootstrapModulePath);
        const appId = options.appId;
        const transitionCall = `.withServerTransition({ appId: '${appId}' })`;
        const position = browserModuleImport.pos + browserModuleImport.getFullText().length;
        const transitionCallChange = new change_1.InsertChange(bootstrapModulePath, position, transitionCall);
        const transitionCallRecorder = host.beginUpdate(bootstrapModulePath);
        transitionCallRecorder.insertLeft(transitionCallChange.pos, transitionCallChange.toAdd);
        host.commitUpdate(transitionCallRecorder);
    };
}
function addDependencies() {
    return (host) => {
        const pkgPath = '/package.json';
        const buffer = host.read(pkgPath);
        if (buffer === null) {
            throw new schematics_1.SchematicsException('Could not find package.json');
        }
        const pkg = JSON.parse(buffer.toString());
        const ngCoreVersion = pkg.dependencies['@angular/core'];
        pkg.dependencies['@angular/platform-server'] = ngCoreVersion;
        host.overwrite(pkgPath, JSON.stringify(pkg, null, 2));
        return host;
    };
}
function getTsConfigOutDir(host, architect) {
    const tsConfigPath = architect.build.options.tsConfig;
    const tsConfigBuffer = host.read(tsConfigPath);
    if (!tsConfigBuffer) {
        throw new schematics_1.SchematicsException(`Could not read ${tsConfigPath}`);
    }
    const tsConfigContent = tsConfigBuffer.toString();
    const tsConfig = core_1.parseJson(tsConfigContent);
    if (tsConfig === null || typeof tsConfig !== 'object' || Array.isArray(tsConfig) ||
        tsConfig.compilerOptions === null || typeof tsConfig.compilerOptions !== 'object' ||
        Array.isArray(tsConfig.compilerOptions)) {
        throw new schematics_1.SchematicsException(`Invalid tsconfig - ${tsConfigPath}`);
    }
    const outDir = tsConfig.compilerOptions.outDir;
    return outDir;
}
function default_1(options) {
    return (host, context) => {
        const clientProject = getClientProject(host, options);
        if (clientProject.projectType !== 'application') {
            throw new schematics_1.SchematicsException(`Universal requires a project type of "application".`);
        }
        const clientArchitect = getClientArchitect(host, options);
        const outDir = getTsConfigOutDir(host, clientArchitect);
        const tsConfigExtends = core_1.basename(clientArchitect.build.options.tsConfig);
        const rootInSrc = clientProject.root === '';
        const tsConfigDirectory = core_1.join(core_1.normalize(clientProject.root), rootInSrc ? 'src' : '');
        if (!options.skipInstall) {
            context.addTask(new tasks_1.NodePackageInstallTask());
        }
        const templateSource = schematics_1.apply(schematics_1.url('./files/src'), [
            schematics_1.template(Object.assign({}, core_1.strings, options, { stripTsExtension: (s) => { return s.replace(/\.ts$/, ''); } })),
            schematics_1.move(core_1.join(core_1.normalize(clientProject.root), 'src')),
        ]);
        const rootSource = schematics_1.apply(schematics_1.url('./files/root'), [
            schematics_1.template(Object.assign({}, core_1.strings, options, { stripTsExtension: (s) => { return s.replace(/\.ts$/, ''); }, outDir,
                tsConfigExtends,
                rootInSrc })),
            schematics_1.move(tsConfigDirectory),
        ]);
        return schematics_1.chain([
            schematics_1.mergeWith(templateSource),
            schematics_1.mergeWith(rootSource),
            addDependencies(),
            updateConfigFile(options, tsConfigDirectory),
            wrapBootstrapCall(options),
            addServerTransition(options),
        ])(host, context);
    };
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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