"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const url = require("url");
const RegistryClient = require('npm-registry-client');
const npmPackageJsonCache = new Map();
const npmConfigOptionCache = new Map();
function getNpmConfigOption(option, scope, tryWithoutScope) {
    if (scope && tryWithoutScope) {
        return rxjs_1.concat(getNpmConfigOption(option, scope), getNpmConfigOption(option)).pipe(operators_1.filter(result => !!result), operators_1.defaultIfEmpty(), operators_1.first());
    }
    const fullOption = `${scope ? scope + ':' : ''}${option}`;
    let value = npmConfigOptionCache.get(fullOption);
    if (value) {
        return value;
    }
    const subject = new rxjs_1.ReplaySubject(1);
    try {
        child_process_1.exec(`npm get ${fullOption}`, (error, data) => {
            if (error) {
                subject.next();
            }
            else {
                data = data.trim();
                if (!data || data === 'undefined' || data === 'null') {
                    subject.next();
                }
                else {
                    subject.next(data);
                }
            }
            subject.complete();
        });
    }
    catch (_a) {
        subject.next();
        subject.complete();
    }
    value = subject.asObservable();
    npmConfigOptionCache.set(fullOption, value);
    return value;
}
/**
 * Get the NPM repository's package.json for a package. This is p
 * @param {string} packageName The package name to fetch.
 * @param {string} registryUrl The NPM Registry URL to use.
 * @param {LoggerApi} logger A logger instance to log debug information.
 * @returns An observable that will put the pacakge.json content.
 * @private
 */
function getNpmPackageJson(packageName, registryUrl, logger) {
    const scope = packageName.startsWith('@') ? packageName.split('/')[0] : undefined;
    return (registryUrl ? rxjs_1.of(registryUrl) : getNpmConfigOption('registry', scope, true)).pipe(operators_1.map(partialUrl => {
        if (!partialUrl) {
            partialUrl = 'https://registry.npmjs.org/';
        }
        const partial = url.parse(partialUrl);
        let fullUrl = new url.URL(`http://${partial.host}/${packageName.replace(/\//g, '%2F')}`);
        try {
            const registry = new url.URL(partialUrl);
            registry.pathname = (registry.pathname || '')
                .replace(/\/?$/, '/' + packageName.replace(/\//g, '%2F'));
            fullUrl = new url.URL(url.format(registry));
        }
        catch (_a) { }
        logger.debug(`Getting package.json from '${packageName}' (url: ${JSON.stringify(fullUrl)})...`);
        return fullUrl.toString();
    }), operators_1.concatMap(fullUrl => {
        let maybeRequest = npmPackageJsonCache.get(fullUrl);
        if (maybeRequest) {
            return maybeRequest;
        }
        return rxjs_1.concat(getNpmConfigOption('proxy'), getNpmConfigOption('https-proxy'), getNpmConfigOption('strict-ssl')).pipe(operators_1.toArray(), operators_1.concatMap(options => {
            const subject = new rxjs_1.ReplaySubject(1);
            const client = new RegistryClient({
                proxy: {
                    http: options[0],
                    https: options[1],
                },
                ssl: Object.assign({}, (options[2] === 'false'
                    ? { strict: false }
                    : (options[2] === 'true'
                        ? { strict: true }
                        : {}))),
            });
            client.log.level = 'silent';
            const params = {
                timeout: 30000,
            };
            client.get(fullUrl, params, (error, data) => {
                if (error) {
                    subject.error(error);
                }
                subject.next(data);
                subject.complete();
            });
            maybeRequest = subject.asObservable();
            npmPackageJsonCache.set(fullUrl.toString(), maybeRequest);
            return maybeRequest;
        }));
    }));
}
exports.getNpmPackageJson = getNpmPackageJson;
//# sourceMappingURL=data:application/json;base64,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