"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const url = require("url");
const RegistryClient = require('npm-registry-client');
const npmPackageJsonCache = new Map();
const npmConfigOptionCache = new Map();
function _readNpmRc() {
    return new rxjs_1.Observable(subject => {
        // TODO: have a way to read options without using fs directly.
        const path = require('path');
        const fs = require('fs');
        let npmrc = '';
        if (process.platform === 'win32') {
            if (process.env.LOCALAPPDATA) {
                npmrc = fs.readFileSync(path.join(process.env.LOCALAPPDATA, '.npmrc')).toString('utf-8');
            }
        }
        else {
            if (process.env.HOME) {
                npmrc = fs.readFileSync(path.join(process.env.HOME, '.npmrc')).toString('utf-8');
            }
        }
        const allOptionsArr = npmrc.split(/\r?\n/).map(x => x.trim());
        const allOptions = {};
        allOptionsArr.forEach(x => {
            const [key, ...value] = x.split('=');
            allOptions[key] = value.join('=');
        });
        subject.next(allOptions);
        subject.complete();
    }).pipe(operators_1.catchError(() => rxjs_1.of({})), operators_1.shareReplay());
}
function getOptionFromNpmRc(option) {
    return _readNpmRc().pipe(operators_1.map(options => options[option]));
}
function getOptionFromNpmCli(option) {
    return new rxjs_1.Observable(subject => {
        child_process_1.exec(`npm get ${option}`, (error, data) => {
            if (error) {
                throw error;
            }
            else {
                data = data.trim();
                if (!data || data === 'undefined' || data === 'null') {
                    subject.next();
                }
                else {
                    subject.next(data);
                }
            }
            subject.complete();
        });
    }).pipe(operators_1.catchError(() => rxjs_1.of(undefined)), operators_1.shareReplay());
}
function getNpmConfigOption(option, scope, tryWithoutScope) {
    if (scope && tryWithoutScope) {
        return rxjs_1.concat(getNpmConfigOption(option, scope), getNpmConfigOption(option)).pipe(operators_1.filter(result => !!result), operators_1.defaultIfEmpty(), operators_1.first());
    }
    const fullOption = `${scope ? scope + ':' : ''}${option}`;
    let value = npmConfigOptionCache.get(fullOption);
    if (value) {
        return value;
    }
    value = option.startsWith('_')
        ? getOptionFromNpmRc(fullOption)
        : getOptionFromNpmCli(fullOption);
    npmConfigOptionCache.set(fullOption, value);
    return value;
}
function getNpmClientSslOptions(strictSsl, cafile) {
    const sslOptions = {};
    if (strictSsl === 'false') {
        sslOptions.strict = false;
    }
    else if (strictSsl === 'true') {
        sslOptions.strict = true;
    }
    if (cafile) {
        sslOptions.ca = fs_1.readFileSync(cafile);
    }
    return sslOptions;
}
/**
 * Get the NPM repository's package.json for a package. This is p
 * @param {string} packageName The package name to fetch.
 * @param {string} registryUrl The NPM Registry URL to use.
 * @param {LoggerApi} logger A logger instance to log debug information.
 * @returns An observable that will put the pacakge.json content.
 * @private
 */
function getNpmPackageJson(packageName, registryUrl, logger) {
    const scope = packageName.startsWith('@') ? packageName.split('/')[0] : undefined;
    return (registryUrl ? rxjs_1.of(registryUrl) : getNpmConfigOption('registry', scope, true)).pipe(operators_1.map(partialUrl => {
        if (!partialUrl) {
            partialUrl = 'https://registry.npmjs.org/';
        }
        const partial = url.parse(partialUrl);
        let fullUrl = new url.URL(`http://${partial.host}/${packageName.replace(/\//g, '%2F')}`);
        try {
            const registry = new url.URL(partialUrl);
            registry.pathname = (registry.pathname || '')
                .replace(/\/?$/, '/' + packageName.replace(/\//g, '%2F'));
            fullUrl = new url.URL(url.format(registry));
        }
        catch (_a) { }
        logger.debug(`Getting package.json from '${packageName}' (url: ${JSON.stringify(fullUrl)})...`);
        return fullUrl;
    }), operators_1.concatMap(fullUrl => {
        let maybeRequest = npmPackageJsonCache.get(fullUrl.toString());
        if (maybeRequest) {
            return maybeRequest;
        }
        const registryKey = `//${fullUrl.host}/`;
        return rxjs_1.concat(getNpmConfigOption('proxy'), getNpmConfigOption('https-proxy'), getNpmConfigOption('strict-ssl'), getNpmConfigOption('cafile'), getNpmConfigOption('_auth'), getNpmConfigOption('_authToken', registryKey), getNpmConfigOption('username', registryKey, true), getNpmConfigOption('password', registryKey, true), getNpmConfigOption('alwaysAuth', registryKey, true)).pipe(operators_1.toArray(), operators_1.concatMap(options => {
            const [http, https, strictSsl, cafile, token, authToken, username, password, alwaysAuth,] = options;
            const subject = new rxjs_1.ReplaySubject(1);
            const sslOptions = getNpmClientSslOptions(strictSsl, cafile);
            const auth = {};
            if (alwaysAuth !== undefined) {
                auth.alwaysAuth = alwaysAuth === 'false' ? false : !!alwaysAuth;
            }
            if (authToken) {
                auth.token = authToken;
            }
            else if (token) {
                auth.token = token;
            }
            else if (username) {
                auth.username = username;
                auth.password = password;
            }
            const client = new RegistryClient({
                proxy: { http, https },
                ssl: sslOptions,
            });
            client.log.level = 'silent';
            const params = {
                timeout: 30000,
                auth,
            };
            client.get(fullUrl.toString(), params, (error, data) => {
                if (error) {
                    subject.error(error);
                }
                subject.next(data);
                subject.complete();
            });
            maybeRequest = subject.asObservable();
            npmPackageJsonCache.set(fullUrl.toString(), maybeRequest);
            return maybeRequest;
        }));
    }));
}
exports.getNpmPackageJson = getNpmPackageJson;
//# sourceMappingURL=data:application/json;base64,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