import { find, fire, ready, findOne, addClass, styles, removeClass, on } from 'domassist';
import attrobj from 'attrobj';
import tinybounce from 'tinybounce';

var Events = {
  In: 'scrolltriggers:inView',
  Out: 'scrolltriggers:outOfView',
  Pause: 'scrolltriggers:pause',
  Resume: 'scrolltriggers:resume',
  Bounds: 'scrolltriggers:bounds'
};

var ScrollTrigger =
/*#__PURE__*/
function () {
  function ScrollTrigger(el, options) {
    var _this = this;

    if (el.hasAttribute('data-scroll-init')) {
      return;
    }

    this.added = false;
    this.el = el;
    this.options = options;
    this.eventHandler = tinybounce(this.onScroll.bind(this), 10, true);
    this.dCalcBounds = tinybounce(this.calcBounds.bind(this), 10);
    this.paused = false;
    this.disabled = false;
    this.calcOffset(); // If images, once by default

    if (this.options.image || this.options.src || this.options.srcset) {
      this.options.once = true;
    }

    el.setAttribute('data-scroll-init', 'true');
    this.calcBounds();
    window.addEventListener('scroll', this.eventHandler);
    window.addEventListener('resize', this.dCalcBounds);
    on(this.el, Events.Pause, function () {
      _this.paused = true;
    });
    on(this.el, Events.Resume, function () {
      _this.paused = false;
    });
    /*
      Prevents a bug on Blink+Webkit in which scroll is always 0 until around
      400 milliseconds due to anchor scrolling features.
     */

    setTimeout(this.eventHandler, 400);
  }

  var _proto = ScrollTrigger.prototype;

  _proto.calcBounds = function calcBounds() {
    var _this2 = this;

    this.calcOffset(); // Element is hidden and not fixed

    var isAllowedToBeFixed = this.options.progress === true || typeof this.options.fixed !== 'undefined';

    if (!this.el.offsetParent && !isAllowedToBeFixed || this.added && this.options.once) {
      // Don't even bother calculating
      this.disabled = true;
      return;
    }

    this.disabled = false;

    if (isAllowedToBeFixed && this.added) {
      this.outOfView();
      return requestAnimationFrame(function () {
        return _this2.calcBounds();
      });
    }

    var position = this.options.position || 'bottom';
    this.startEl = this.options.start ? findOne(this.options.start) : this.el;
    ScrollTrigger.checkElement(this.startEl, 'start', this.options.start);
    var rect = this.startEl.getBoundingClientRect();
    var scrollY = ScrollTrigger.getScrollY();
    var start = rect.top + scrollY + (this.options.offset || 0);
    this.start = ScrollTrigger.processPosition(position, start);

    if (this.options.end) {
      var endEl = findOne(this.options.end);
      var endRect = endEl.getBoundingClientRect();
      var end = endRect.top + scrollY;
      var endPosition = this.options.positionEnd || 'bottom';

      if (endPosition === 'auto') {
        endPosition = 'top';
      }

      this.end = ScrollTrigger.processPosition(endPosition, end);

      if (this.options.positionEnd === 'auto') {
        this.end -= this.el.offsetHeight;
      }

      ScrollTrigger.checkElement(endEl, 'end', this.options.end);
    }

    this.fire(Events.Bounds);
    this.eventHandler();
  };

  _proto.calcOffset = function calcOffset() {
    this.options.offset = this.options.offset ? this.options.offset : this.el.dataset.scrollOffset; // A screen above loading

    if (this.options.image || this.options.srcset || this.options.offset === 'auto') {
      this.options.offset = Math.max(document.documentElement.clientHeight, window.innerHeight, 0) * -1;
    } else {
      this.options.offset = parseInt(this.options.offset || 0, 10);
    }
  };

  _proto.inView = function inView() {
    var _this$options = this.options,
        className = _this$options.className,
        inView = _this$options.inView;

    if (className && this.el.classList) {
      addClass(this.el, className);
    }

    var image = this.options.image;
    var src = this.options.src;
    var srcset = this.options.srcset;

    if (image || src) {
      var source = image || src;

      switch (this.el.tagName) {
        case 'IMG':
        case 'IFRAME':
        case 'VIDEO':
        case 'SCRIPT':
          this.el.setAttribute('src', source);
          break;

        default:
          styles(this.el, {
            backgroundImage: "url(" + source + ")",
            backgroundRepeat: 'no-repeat'
          });
      }
    }

    if (srcset) {
      this.el.setAttribute('srcset', srcset);
    }

    if (typeof inView === 'function') {
      inView(this.el, this.options);
    }

    this.fire(Events.In);

    if (this.options.once) {
      this.disabled = true;
      window.removeEventListener('scroll', this.eventHandler);
      window.removeEventListener('resize', this.dCalcBounds);
    }

    this.added = true;
  };

  _proto.outOfView = function outOfView() {
    var _this$options2 = this.options,
        className = _this$options2.className,
        outOfView = _this$options2.outOfView;

    if (className && this.el.classList) {
      removeClass(this.el, className);
    }

    if (typeof outOfView === 'function') {
      outOfView(this.el, this.options);
    }

    this.fire(Events.Out);
    this.added = false;
  };

  _proto.fire = function fire$1(event) {
    fire(this.el, event, {
      detail: {
        instance: this,
        options: this.options
      }
    });
  };

  _proto.onScroll = function onScroll() {
    var scroll = ScrollTrigger.getScrollY();

    if (this.paused || this.disabled) {
      return;
    }

    if (this.options.progress) {
      var perc = scroll / (document.documentElement.scrollHeight - window.innerHeight);
      this.el.style.width = perc * 100 + "%";
    }

    if (scroll < this.start || this.end && scroll > this.end) {
      if (this.added) {
        this.outOfView();
      }

      return;
    }

    if (this.added) {
      return;
    }

    this.inView();
  };

  ScrollTrigger.checkElement = function checkElement(element, position, selector) {
    if (!element) {
      throw new Error(position + " element doesn't match any element with selector: \"" + selector + "\"");
    }
  };

  ScrollTrigger.getScrollY = function getScrollY() {
    return window.pageYOffset || document.documentElement.scrollTop;
  };

  ScrollTrigger.processPosition = function processPosition(position, currentValue) {
    if (position === 'top') {
      return currentValue;
    }

    if (position === 'middle') {
      currentValue -= window.innerHeight / 2;
    } else if (position === 'bottom') {
      currentValue -= window.innerHeight;
    } else {
      currentValue -= window.innerHeight * (parseInt(position, 10) / 100);
    }

    return currentValue;
  };

  return ScrollTrigger;
}();

var init = function init(items) {
  var instances = [];

  if (items && Array.isArray(items)) {
    items.forEach(function (item) {
      var els = find(item.el);

      if (els === null) {
        throw new Error('unknown element');
      }

      els.forEach(function (el) {
        delete item.el;
        instances.push(new ScrollTrigger(el, item));
      });
    });
  } else if (items) {
    throw new Error('please convert object to array');
  } else {
    var els = find('[data-scroll]');
    els.forEach(function (el) {
      var options = attrobj('scroll', el);

      if (options.progress !== null && typeof options.progress !== 'undefined') {
        options.progress = true;
      }

      options.className = options["class"];

      if (options.offset) {
        options.offset = parseInt(options.offset, 10);
      }

      if (typeof options.once !== 'undefined') {
        options.once = true;
      }

      instances.push(new ScrollTrigger(el, options));
    });
  }

  return instances;
};

if (document.readyState !== 'complete') {
  // Avoid image loading impacting on calculations
  document.addEventListener('readystatechange', function () {
    if (document.readyState === 'complete') {
      fire(window, 'resize');
    }
  });
}

ready(init);
init.Events = Events;
init.ScrollTrigger = ScrollTrigger;

export default init;
//# sourceMappingURL=scroll-triggers.js.map
