import { getCurrentHub } from '@sentry/core/esm';
import { logger } from '@sentry/utils/esm/logger';
import { eventFromStacktrace } from '../parsers';
import { installGlobalHandler, installGlobalUnhandledRejectionHandler, subscribe, } from '../tracekit';
import { shouldIgnoreOnError } from './helpers';
/** Global handlers */
export class GlobalHandlers {
    /** JSDoc */
    constructor(options) {
        /**
         * @inheritDoc
         */
        this.name = GlobalHandlers.id;
        this.options = {
            onerror: true,
            onunhandledrejection: true,
            ...options,
        };
    }
    /**
     * @inheritDoc
     */
    setupOnce() {
        subscribe((stack, _, error) => {
            // TODO: use stack.context to get a valuable information from TraceKit, eg.
            // [
            //   0: "  })"
            //   1: ""
            //   2: "  function foo () {"
            //   3: "    Sentry.captureException('some error')"
            //   4: "    Sentry.captureMessage('some message')"
            //   5: "    throw 'foo'"
            //   6: "  }"
            //   7: ""
            //   8: "  function bar () {"
            //   9: "    foo();"
            //   10: "  }"
            // ]
            if (shouldIgnoreOnError()) {
                return;
            }
            const self = getCurrentHub().getIntegration(GlobalHandlers);
            if (self) {
                getCurrentHub().captureEvent(self.eventFromGlobalHandler(stack), { originalException: error, data: { stack } });
            }
        });
        if (this.options.onerror) {
            logger.log('Global Handler attached: onerror');
            installGlobalHandler();
        }
        if (this.options.onunhandledrejection) {
            logger.log('Global Handler attached: onunhandledrejection');
            installGlobalUnhandledRejectionHandler();
        }
    }
    /** JSDoc */
    eventFromGlobalHandler(stacktrace) {
        const event = eventFromStacktrace(stacktrace);
        return {
            ...event,
            exception: {
                ...event.exception,
                mechanism: {
                    data: {
                        mode: stacktrace.mode,
                    },
                    handled: false,
                    type: stacktrace.mechanism,
                },
            },
        };
    }
}
/**
 * @inheritDoc
 */
GlobalHandlers.id = 'GlobalHandlers';
//# sourceMappingURL=globalhandlers.js.map